/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010. 
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.  
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl> 
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */

#define _XOPEN_SOURCE	600

#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <unistd.h>
#include <sys/types.h>
#include <time.h>
#include <limits.h>
#include <errno.h>

#include <gssapi.h>
#include <globus_gsi_credential.h>

#include <globus_gss_assist.h>
#include <globus_gridmap_callout_error.h>

#include "llgt_config.h"

/* Make sure to test this AFTER including llgt_config.h */
/* Also note: AC_CHECK_DECL works differently from AC_CHECK_LIB in that it
 * actually defines the HAVE_DECL_ to 0. Add this check to make sure that that's
 * the case on non-conforming platforms */
#ifndef HAVE_DECL_GLOBUS_GSI_CRED_READ_CERT_BUFFER
# define HAVE_DECL_GLOBUS_GSI_CRED_READ_CERT_BUFFER	0
#endif

/* Now check with a #if instead of an #ifdef */
#if !HAVE_DECL_GLOBUS_GSI_CRED_READ_CERT_BUFFER
#include <openssl/bio.h>
#endif

#include "llgt_globus_internal.h"

#include "llgt_utils.h"

static int llgt_debugging_mode = 0; /* disabled by default */

static int llgt_log_to_file = -1; /* uninitialized */

static FILE *llgt_logfile=NULL;	    /* FILE used for logging (when llgt_log_to_file == 1) */
static char *llgt_log_ident=NULL;   /* log ident used when logging to file */

static char *llgt_loglevel[]={
    "LOG_EMERG",
    "LOG_ALERT",
    "LOG_CRIT",
    "LOG_ERR",
    "LOG_WARNING",
    "LOG_NOTICE",
    "LOG_INFO",
    "LOG_DEBUG"};

static int llgt_setup_syslog(void);

static int llgt_setup_syslog(void)
{
    int   syslog_facility       = LOG_DAEMON;
    char *syslog_log_prefix     = NULL;
    char *llgt_log_facility     = NULL;

    /* Enable debugging */
    if (getenv("LLGT_ENABLE_DEBUG")) {
        llgt_enable_debugging_mode();
    }


    /* Check for a specific Syslog IDENT */
    if ((syslog_log_prefix = getenv("LLGT_LOG_IDENT"))) {
        /* If the string length is zero, then nothing is really specified and
         * then we just NULLify the string */
        if (strlen(syslog_log_prefix) == 0) {
            syslog_log_prefix = NULL;
        }
    }

    /* Set an alternative SysLog facility */
    if ((llgt_log_facility = getenv("LLGT_LOG_FACILITY"))) {
        if      (strcmp("LOG_DAEMON",   llgt_log_facility) == 0) { syslog_facility = LOG_DAEMON; }
        else if (strcmp("LOG_AUTH",     llgt_log_facility) == 0) { syslog_facility = LOG_AUTH; }
        else if (strcmp("LOG_CRON",     llgt_log_facility) == 0) { syslog_facility = LOG_CRON; }
        else if (strcmp("LOG_KERN",     llgt_log_facility) == 0) { syslog_facility = LOG_KERN; }
        else if (strcmp("LOG_LPR",      llgt_log_facility) == 0) { syslog_facility = LOG_LPR; }
        else if (strcmp("LOG_MAIL",     llgt_log_facility) == 0) { syslog_facility = LOG_MAIL; }
        else if (strcmp("LOG_NEWS",     llgt_log_facility) == 0) { syslog_facility = LOG_NEWS; }
        else if (strcmp("LOG_USER",     llgt_log_facility) == 0) { syslog_facility = LOG_USER; }
        else if (strcmp("LOG_UUCP",     llgt_log_facility) == 0) { syslog_facility = LOG_UUCP; }
        else if (strcmp("LOG_LOCAL0",   llgt_log_facility) == 0) { syslog_facility = LOG_LOCAL0; }
        else if (strcmp("LOG_LOCAL1",   llgt_log_facility) == 0) { syslog_facility = LOG_LOCAL1; }
        else if (strcmp("LOG_LOCAL2",   llgt_log_facility) == 0) { syslog_facility = LOG_LOCAL2; }
        else if (strcmp("LOG_LOCAL3",   llgt_log_facility) == 0) { syslog_facility = LOG_LOCAL3; }
        else if (strcmp("LOG_LOCAL4",   llgt_log_facility) == 0) { syslog_facility = LOG_LOCAL4; }
        else if (strcmp("LOG_LOCAL5",   llgt_log_facility) == 0) { syslog_facility = LOG_LOCAL5; }
        else if (strcmp("LOG_LOCAL6",   llgt_log_facility) == 0) { syslog_facility = LOG_LOCAL6; }
        else if (strcmp("LOG_LOCAL7",   llgt_log_facility) == 0) { syslog_facility = LOG_LOCAL7; }
	/* Now the non-standard ones... */
#ifdef LOG_SYSLOG
        else if (strcmp("LOG_SYSLOG",   llgt_log_facility) == 0) { syslog_facility = LOG_SYSLOG; }
#endif
#ifdef LOG_FTP
        else if (strcmp("LOG_FTP",      llgt_log_facility) == 0) { syslog_facility = LOG_FTP; }
#endif
#ifdef LOG_AUTHPRIV
        else if (strcmp("LOG_AUTHPRIV", llgt_log_facility) == 0) { syslog_facility = LOG_AUTHPRIV; }
#endif
#ifdef LOG_AUDIT
        else if (strcmp("LOG_AUDIT",    llgt_log_facility) == 0) { syslog_facility = LOG_AUDIT; }
#endif
        else {
            syslog(LOG_ERR, "The Syslog facility is configured with $LLGT_LOG_FACILITY and set to the unknown facility: \"%s\". Overriding to LOG_DAEMON. Please fix the setting.", llgt_log_facility);
        }
    } else {
        /* Default - Nothing specified */
        syslog_facility = LOG_DAEMON;
    }

    /* When either LLGT_LOG_FACILITY or LLGT_LOG_IDENT is given, call
     * openlog(), otherwise, skip it */
    if (llgt_log_facility || syslog_log_prefix)	{
	/* initialize Syslog */
	openlog (syslog_log_prefix, LOG_CONS | LOG_PID | LOG_NDELAY,
		 syslog_facility);
    }

    return 0;
}

void llgt_enable_debugging_mode(void) {
    llgt_debugging_mode = 1;
}

int llgt_is_debugmode_enabled(void) {
    return llgt_debugging_mode;
}


/* Opens log, eeither syslog or logfile */
void llgt_open_log(void)	{
    char *llgt_logfilename=getenv("LLGT_LOG_FILE");
    int errno_fopen;

    /* Only continue when we aren't initialized */
    if (llgt_log_to_file>=0)
	return;

    /* Do we have a LLGT_LOG_FILE? */
    if (llgt_logfilename==NULL) {	/* log to syslog */
	llgt_log_to_file=0;
	llgt_setup_syslog();
    } else { /* Log to file */
	llgt_logfile=fopen(llgt_logfilename, "a");
	if (llgt_logfile)  { /* Succesfully opened logfile */
	    llgt_log_to_file=1;

	    /* Enable debugging */
	    if (getenv("LLGT_ENABLE_DEBUG")) {
		llgt_enable_debugging_mode();
	    }

	    if ( (llgt_log_ident=getenv("LLGT_LOG_IDENT")) == NULL)
		/* Default log ident */
		llgt_log_ident="llgt";

	    /* Make sure LCAS and LCMAPS use the same logfile if no default has
	     * been set for them. */
	    if (getenv("LCMAPS_LOG_FILE")==NULL)
		setenv("LCMAPS_LOG_FILE", llgt_logfilename, 1);
#ifndef DISABLE_LCAS
	    if (getenv("LCAS_LOG_FILE")==NULL)
		setenv("LCAS_LOG_FILE", llgt_logfilename, 1);
#endif
	} else {/* Switch to syslog and log error */
	    errno_fopen=errno;
	    llgt_log_to_file=0;
	    llgt_setup_syslog();
	    llgt_logmsg(LOG_ERR,"Cannot open %s, using syslog: %s\n",
		        llgt_logfilename,strerror(errno_fopen));
	}
    }
}

/* Close the logfile when opened */
void llgt_close_log(void)   {
    if (llgt_logfile)	{
	fclose(llgt_logfile);
	llgt_logfile=NULL;
	llgt_log_to_file=-1;
    }
}

/* Log function for lcas-lcmaps-gt4-interface messages, logs to either file or
 * syslog. When needed the log will be opened first via llgt_open_log() */
void llgt_logmsg (int priority, const char* format, ...)
{
    va_list     args;
    char	buf[MAX_ERRORBUF_LEN];
    char	datetime[21];
    int		res,i;
    time_t      mclock;
    struct tm * tmp;


    /* Check if the module was enabled with debugging enabled.
     * If so, pass the LOG_DEBUG level message though */
    if (priority == LOG_DEBUG && !llgt_is_debugmode_enabled()) {
        /* Skip debugging messages */
        return;
    }

    /* check whether log destination is initialized */
    if (llgt_log_to_file<0)
	llgt_open_log();

    /* Put logstring in buffer */
    va_start(args, format);
    res=vsnprintf(buf,MAX_ERRORBUF_LEN,format,args);
    va_end(args);

    /* Handle too large msg */
    if (res>0 && (size_t)res>= MAX_ERRORBUF_LEN)	{
	sprintf(&(buf[MAX_ERRORBUF_LEN-5]),"...\n");
	res=MAX_ERRORBUF_LEN-1;
    }

    /* Replace unprintable characters with ? */
    for (i=0; buf[i]!='\0'; i++)    {
	if (buf[i]!='\n' && !isprint(buf[i]))
	    buf[i]='?';
    }
    /* Guarantee a newline at the end of the logstring */
    if (buf[res-1]!='\n')   {
	/* Need to add one character, check if we can... */
	if ((size_t)res<MAX_ERRORBUF_LEN-1) {
	    /* Increase and make sure last character will be a null character */
	    buf[++res]='\0';
	    buf[res-1]='\n';
	} else	{
	    sprintf(&(buf[MAX_ERRORBUF_LEN-5]),"...\n");
#if 0 /* If we at some point do more with res, reenable this line */
	    res=MAX_ERRORBUF_LEN-1;
#endif
	}
    }

    if (llgt_log_to_file==0)
	/* Log to syslog */
	syslog(priority, "%s", buf);
    else {
	/* Log to file: get time */
	time(&mclock);
	/* Log in GMT, no exceptions */
	if ( (tmp = gmtime(&mclock))==NULL )
	    datetime[0]='\0';
	else
	    snprintf(datetime, (size_t)21, "%04d-%02d-%02d.%02d:%02d:%02dZ",
			      tmp->tm_year + 1900, tmp->tm_mon + 1,
			      tmp->tm_mday,
			       tmp->tm_hour, tmp->tm_min, tmp->tm_sec);
	/* print in logfile */
	fprintf(llgt_logfile,"%s[%ld]: %11s: %s: %s",
		llgt_log_ident, (long)getpid(),
		llgt_loglevel[priority], datetime, buf);
    }
}



globus_result_t llgt_get_client_name(gss_ctx_id_t context, char **client_name)
{
    int initiator;
    unsigned int major_status;
    unsigned int minor_status;
    gss_name_t peer=GSS_C_NO_NAME;
    gss_buffer_desc peer_name_buffer;
    char *subject = NULL;
    globus_result_t result;
    int rc = 0;

    /* start necessary GSS modules */
    rc = globus_module_activate(GLOBUS_GSI_GSS_ASSIST_MODULE);
    if(rc!=0) {
        llgt_logmsg(LOG_ERR, "Error activating Globus GSS ASSIST MODULE.\n");
	return (globus_result_t)GLOBUS_FAILURE;;
    }
    rc = globus_module_activate(GLOBUS_GSI_GSSAPI_MODULE);
    if(rc!=0) {
        llgt_logmsg(LOG_ERR, "Error activating Globus GSSAPI MODULE.\n");
        return (globus_result_t)GLOBUS_FAILURE;
    }

    /* find out who is the initiator of the connection (typically the client) */
    major_status = gss_inquire_context(&minor_status,
				   context,
				   GLOBUS_NULL,
				   GLOBUS_NULL,
				   GLOBUS_NULL,
				   GLOBUS_NULL,
				   GLOBUS_NULL,
				   &initiator,
				   GLOBUS_NULL);
    if(GSS_ERROR(major_status))
    {
        GLOBUS_GRIDMAP_CALLOUT_GSS_ERROR(result, major_status, minor_status);
	llgt_logmsg(LOG_ERR, "Error inquiring context to find the initiator.\n");
    	return result;
    }

    /* get the peer's name */
    major_status = gss_inquire_context(&minor_status,
				   context,
				   initiator ? GLOBUS_NULL : &peer,
				   initiator ? &peer : GLOBUS_NULL,
				   GLOBUS_NULL,
				   GLOBUS_NULL,
				   GLOBUS_NULL,
				   GLOBUS_NULL,
				   GLOBUS_NULL);
    if(GSS_ERROR(major_status))	{
	GLOBUS_GRIDMAP_CALLOUT_GSS_ERROR(result, major_status, minor_status);
        llgt_logmsg(LOG_ERR,
		"Error inquiring context to extract the identity of the peer");
	return result;
    }

    /* Get name */
    major_status = gss_display_name(&minor_status,
                                    peer,
                                    &peer_name_buffer,
                                    GLOBUS_NULL);
    if(GSS_ERROR(major_status))	{
        GLOBUS_GRIDMAP_CALLOUT_GSS_ERROR(result, major_status, minor_status);
	llgt_logmsg(LOG_ERR, "Cannot obtain peername");
        gss_release_name(&minor_status, &peer);
	return result;
    }

    /* Free peer */
    gss_release_name(&minor_status, &peer);

    /* Copy into client_name */
    if((subject = (char*)malloc(peer_name_buffer.length + 1)))
    {
	memcpy(subject, peer_name_buffer.value, peer_name_buffer.length);
	subject[peer_name_buffer.length] = 0;
    } else {
	llgt_logmsg(LOG_ERR, "Out of memory");
	gss_release_buffer(&minor_status, &peer_name_buffer);
	return (globus_result_t)GLOBUS_FAILURE;
    }
    
    gss_release_buffer(&minor_status, &peer_name_buffer);
    *client_name=subject;
    return GLOBUS_SUCCESS;
}



gss_cred_id_t llgt_get_user_cred_handle(gss_ctx_id_t context_handle)
{
    llgt_gss_ctx_id_desc *local_handle=(llgt_gss_ctx_id_desc*)context_handle;
    return local_handle->peer_cred_handle;
}


int llgt_create_jobid(void)
{
    /*
     * Create the unique Job Repository ID and export it as an environment variable
     * the ID is the same one as for the Job Manager with the addition of microseconds.
     * We moved this part here so that the LCMAPS Job Repository plugin
     * can use this environment variable to uniquely identify the job
     * To have a really unique id we also add the microseconds to the time string
     */
    struct tm  * jr_tmp;
    const char * jr_id_var = "JOB_REPOSITORY_ID";
    char         jr_id[71];
    const char * gk_jm_id_var = "GATEKEEPER_JM_ID";
    char         gatekeeper_jm_id[64];
    const pid_t  gatekeeper_pid = getpid();
    static unsigned int   reqnr = 0;

#ifdef HAVE_GETTIMEOFDAY
    {
    struct timeval  jr_clock;

    /* get time in microseconds ! */
    gettimeofday(&jr_clock,NULL);
    jr_tmp = gmtime(&(jr_clock.tv_sec));

    snprintf(jr_id, (size_t)70, "%04d-%02d-%02d.%02d:%02d:%02d.%06d.%010u.%010u",
        jr_tmp->tm_year + 1900, jr_tmp->tm_mon + 1, jr_tmp->tm_mday,
        jr_tmp->tm_hour, jr_tmp->tm_min, jr_tmp->tm_sec, (int)(jr_clock.tv_usec),
        (unsigned)gatekeeper_pid & 0xFFFFFFFF, reqnr & 0xFFFFFFFF);
    }
#else
    {
    time_t       myclock;

    time(&myclock);
    jr_tmp = gmtime(&myclock);
    }

    snprintf(jr_id, (size_t)70, "%04d-%02d-%02d.%02d:%02d:%02d.%010u.%010u",
        jr_tmp->tm_year + 1900, jr_tmp->tm_mon + 1, jr_tmp->tm_mday,
        jr_tmp->tm_hour, jr_tmp->tm_min, jr_tmp->tm_sec,
        (unsigned)gatekeeper_pid & 0xFFFFFFFF, reqnr & 0xFFFFFFFF);
#endif
    snprintf(gatekeeper_jm_id, (size_t)63, "%04d-%02d-%02d.%02d:%02d:%02d.%010u.%010u",
        jr_tmp->tm_year + 1900, jr_tmp->tm_mon + 1, jr_tmp->tm_mday,
        jr_tmp->tm_hour, jr_tmp->tm_min, jr_tmp->tm_sec,
        (unsigned)gatekeeper_pid & 0xFFFFFFFF, reqnr & 0xFFFFFFFF);


    setenv(jr_id_var, jr_id, 1);
    setenv(gk_jm_id_var, gatekeeper_jm_id, 1);

    reqnr++;
    return 0;
}

/*
 * Extracts cert and chain from input pem file and produces an otherwise empty
 * gss_cred_id_t containing them. At the same time, it also extracts the DN from
 * the EEC certificate. When we use GT5.2.5 or later, we can use
 * globus_gsi_cred_read_cert_buffer() to convert the pem string into a
 * globus_gsi_cred_handle_t, which greatly simplifies it. Even then, we still
 * have to handmake our gss_cred_id_t. The resulting gss_cred_id_t can be
 * cleaned up using llgt_cleanup_gss_cred_id_t
 */
globus_result_t llgt_pem_to_gsscred(char *pem_buf,
				    gss_cred_id_t *gss_cred,
				    char **clientname)	{
    globus_gsi_cred_handle_t gsi_handle = NULL;
    char *subject = NULL;
#if !HAVE_DECL_GLOBUS_GSI_CRED_READ_CERT_BUFFER
    BIO *bp = NULL;
#endif
    globus_result_t result;
    llgt_gss_cred_id_desc *mycred;

    /* First get and fill globus_gsi_cred_handle_t from the pem_buf */
#if HAVE_DECL_GLOBUS_GSI_CRED_READ_CERT_BUFFER
    result = globus_gsi_cred_read_cert_buffer(
	    pem_buf, &gsi_handle, NULL, NULL, &subject);
    if (result!= GLOBUS_SUCCESS)	{
	llgt_logmsg(LOG_ERR,
		"Cannot get credential data from PEM string.\n");
	return result;
    }
#else /* HAVE_DECL_GLOBUS_GSI_CRED_READ_CERT_BUFFER */
    /* Copied from GT5.2.5 globus_gsi_cred_read_cert_buffer() */
    if(!pem_buf) {
	llgt_logmsg(LOG_ERR,
		"PEM string is empty, cannot get credentials.\n");
	result=GLOBUS_GSI_CRED_ERROR_READING_CRED;
        goto error;
    }

    /* test pem_buf is not too large for an int */
    if (strlen(pem_buf) > INT_MAX)  {
	llgt_logmsg(LOG_ERR,
		"PEM string is too large.\n");
	result=GLOBUS_GSI_CRED_ERROR_READING_CRED;
        goto error;
    }

    /* Open BIO and write PEM string */
    if ( (bp = BIO_new(BIO_s_mem())) == NULL ||
	 (BIO_write(bp, pem_buf, (int)strlen(pem_buf)))<=0 )	{
	llgt_logmsg(LOG_ERR,
		"Error opening BIO for pemstring.\n");
	result=GLOBUS_GSI_CRED_ERROR_READING_CRED;
	goto error;
    }
   
    /* Create a new globus_gsi_cred_handle_t */
    result = globus_gsi_cred_handle_init(&gsi_handle, NULL);
    if (result!=GLOBUS_SUCCESS)	{
	llgt_logmsg(LOG_ERR,
		"Error initializing new globus_gsi_cred_handle_t.\n");
        goto error;
    }

    /* Fill X509 data from BIO */
    result = globus_gsi_cred_read_cert_bio(gsi_handle, bp);
    if (result!=GLOBUS_SUCCESS)	{
	llgt_logmsg(LOG_ERR,
		"Error getting certificate data from BIO.\n");
        goto error;
    }

    /* Get the user DN from the new handle */
    result = globus_gsi_cred_get_identity_name(gsi_handle, &subject);
    if (result!=GLOBUS_SUCCESS)	{
	llgt_logmsg(LOG_ERR,
		"Error getting DN from globus_gsi_cred_handle_t.\n");
        goto error;
    }

    BIO_free(bp); bp=NULL;
#endif /* HAVE_DECL_GLOBUS_GSI_CRED_READ_CERT_BUFFER */

    /* Now we turn the gsi_handle into a gss_cred_id_t, unfortunately Globus has
     * not (yet?) implemented gss_add_cred(), see
     * source-trees/gsi/gssapi/source/library/gssapi_working.c */
#if 0
    result = gss_add_cred(
                &minor_status,          /* minor_status */
                GSS_C_NO_CREDENTIAL,    /* input_cred_handle */
                GSS_C_NO_NAME,          /* desired_name */
                GSS_C_NO_OID,           /* desired_mech */
                GSS_C_BOTH,             /* cred_usage */
                GSS_C_INDEFINITE,       /* initiator_time_req */
                GSS_C_INDEFINITE,       /* acceptor_time_req */
                gss_cred,               /* output_cred_handle */
                NULL,                   /* actual_mechs */
                NULL,                   /* initiator_time_rec */
                NULL);                  /* acceptor_time_rec */
    if (result != GLOBUS_SUCCESS)   {
	llgt_logmsg(LOG_ERR,
		"Error initializing new gss_cred_id_t: %d.\n",result);
	goto error;
    }
#endif
    /* Create an empty structure by hand */
    mycred=(llgt_gss_cred_id_desc*)calloc((size_t)1,sizeof(llgt_gss_cred_id_desc));
    if (mycred==NULL)	{
	llgt_logmsg(LOG_ERR,"Out of memory.\n");
	result=(globus_result_t)GLOBUS_FAILURE;
	goto error;
    }
    /* Put the new gsi_handle in the new gss_cred_id_t, note that the
     * llgt_gss_cred_id_desc is a local copy of the internal globus definition
     * of the gss_cred_id_desc.
     * Also note using calloc() means that mycred->globusid (type gss_name_t) is
     * initialized with GSS_C_NO_NAME and mycred->cred_usage (type
     * gss_cred_usage_t) is initialized with GSS_C_BOTH. */
    mycred->cred_handle=gsi_handle;
    *gss_cred=(gss_cred_id_t)mycred;

    *clientname=subject;

    return GLOBUS_SUCCESS;
    
error:
    /* Cleanup */
#if !HAVE_DECL_GLOBUS_GSI_CRED_READ_CERT_BUFFER
    if(bp)
        BIO_free(bp);
#endif
    if(subject)
        free(subject);
    if(gsi_handle)
        globus_gsi_cred_handle_destroy(gsi_handle);
    
    return result;
}

/*
 * Cleans up a 'homemade' gss_cred_id_t, as done in llgt_pem_to_gsscred()
 */
void llgt_cleanup_gss_cred_id_t(gss_cred_id_t user_cred_handle)	{
    llgt_gss_cred_id_desc *mycred=(llgt_gss_cred_id_desc*)user_cred_handle;

    if (mycred==NULL)
	return;

    if (mycred->cred_handle)
	globus_gsi_cred_handle_destroy(mycred->cred_handle);

    free(mycred);
}
