"""This module provides a few convenient functions for
the Lid Driven Cavity solutions in the paper:

"High-Re solutions for incompressible flow using the Navier-Stokes
equations and a multigrid method", U Ghia, K.N Ghia, C.T Shin,
JCP, Volume 48, Issue 3, December 1982, Pages 387-411.

"""
import numpy as np
from io import StringIO


RE =  [100, 400, 1000, 3200, 5000, 7500, 10000]

# u velocity along vertical line through center (Table I)
table1 = u"""
1.0000  1.0000    1.00000   1.00000   1.00000   1.00000   1.00000  1.00000
0.9766  0.84123   0.75837   0.65928   0.53236   0.48223   0.47244  0.47221
0.9688  0.78871   0.68439   0.57492   0.48296   0.46120   0.47048  0.47783
0.9609  0.73722   0.61756   0.51117   0.46547   0.45992   0.47323  0.48070
0.9531  0.68717   0.55892   0.46604   0.46101   0.46036   0.47167  0.47804
0.8516  0.23151   0.29093   0.33304   0.34682   0.33556   0.34228  0.34635
0.7344  0.00332   0.16256   0.18719   0.19791   0.20087   0.2059   0.20673
0.6172 -0.13641   0.02135   0.05702   0.07156   0.08183   0.08342  0.08344
0.5000 -0.20581  -0.11477  -0.06080  -0.04272  -0.03039  -0.03800  0.03111
0.4531 -0.21090  -0.17119  -0.10648  -0.86636  -0.07404  -0.07503 -0.07540
0.2813 -0.15662  -0.32726  -0.27805  -0.24427  -0.22855  -0.23176 -0.23186
0.1719 -0.10150  -0.24299  -0.38289  -0.34323  -0.33050  -0.32393 -0.32709
0.1016 -0.06434  -0.14612  -0.29730  -0.41933  -0.40435  -0.38324 -0.38000
0.0703 -0.04775  -0.10338  -0.22220  -0.37827  -0.43643  -0.43025 -0.41657
0.0625 -0.04192  -0.09266  -0.20196  -0.35344  -0.42901  -0.43590 -0.42537
0.0547 -0.03717  -0.08186  -0.18109  -0.32407  -0.41165  -0.43154 -0.42735
0.0000  0.00000   0.00000   0.00000   0.00000   0.00000   0.00000  0.00000
"""

# v velocity along horizontal line through center (Table II)
table2 = u"""
1.0000  0.00000  0.00000   0.00000   0.00000   0.00000   0.00000   0.00000
0.9688  -0.05906 -0.12146  -0.21388  -0.39017  -0.49774  -0.53858  -0.54302
0.9609  -0.07391 -0.15663  -0.27669  -0.47425  -0.55069  -0.55216  -0.52987
0.9531  -0.08864 -0.19254  -0.33714  -0.52357  -0.55408  -0.52347  -0.49099
0.9453  -0.10313 -0.22847  -0.39188  -0.54053  -0.52876  -0.48590  -0.45863
0.9063  -0.16914 -0.23827  -0.51550  -0.44307  -0.41442  -0.41050  -0.41496
0.8594  -0.22445 -0.44993  -0.42665  -0.37401  -0.36214  -0.36213  -0.36737
0.8047  -0.24533 -0.38598  -0.31966  -0.31184  -0.30018  -0.30448  -0.30719
0.5000  0.05454  0.05188   0.02526   0.00999   0.00945   0.00824   0.00831
0.2344  0.17527  0.30174   0.32235   0.28188   0.27280   0.27348   0.27224
0.2266  0.17507  0.30203   0.33075   0.29030   0.28066   0.28117   0.28003
0.1563  0.16077  0.28124   0.37095   0.37119   0.35368   0.35060   0.35070
0.0938  0.12317  0.22965   0.32627   0.42768   0.42951   0.41824   0.41487
0.0781  0.10890  0.20920   0.30353   0.41906   0.43648   0.43564   0.43124
0.0703  0.10091  0.19713   0.29012   0.40917   0.43329   0.44030   0.43733
0.0625  0.09233  0.18360   0.27485   0.39560   0.42447   0.43979   0.43983
0.0000  0.00000  0.00000   0.00000   0.00000   0.00000   0.00000   0.00000
"""

def _get_data(table):
    data = np.loadtxt(StringIO(table))
    y = data[:,0]
    result = {}
    for i, r in enumerate(RE):
        result[r] = data[:,i+1]
    return y, result

def get_u_vs_y():
    """Return the data from table 1, this returns an array y and a
    dictionary, with the keys as the available Reynolds numbers.
    """
    return _get_data(table1)

def get_v_vs_x():
    """Return the data from table 2, this returns an array x and a
    dictionary, with the keys as the available Reynolds numbers.
    """
    return _get_data(table2)
