program QGPV
! QG 系で PV に関する諸解析を行うプログラム.
! 水平は緯度経度座標系, 鉛直は圧力座標で定義されたデータを想定している.
  use Math_Const
  use Phys_Const
  use file_operate
  use derivation
  use Ellip_Slv
  use Statistics
  use PV_analy
  use Thermo_Function
  use sub_mod

  implicit none

  !-- parameter
  integer, parameter :: nzmax=100
  integer, parameter :: fmg_lmax=100

  !-- namelist variables
  integer :: nx, ny, nz, zord, tord, diaqord
  integer :: fmg_level, fmg_levelz(fmg_lmax), fmg_convnum, fmg_additr
  real :: dlon, dlat, lonmin, latmin, undef
  real, dimension(nzmax) :: pres
  character(1000) :: listname
  logical :: fmg_flag

  !-- internal variables
  integer :: i, j, k, l, m, n, nl, ic
  real :: d2r, f0
  real, allocatable, dimension(:) :: lon, lat, lonr, latr
  real, allocatable, dimension(:) :: sig, ref_temp, ref_pt
  real, allocatable, dimension(:,:) :: sx, sy, fcor, temp1d
  real, allocatable, dimension(:,:,:) :: ug, vg, tg, uga, vga, tga, pvm, pva
  real, allocatable, dimension(:,:,:) :: zgphm, zgpha, chi, chia, w, wa
  real, allocatable, dimension(:,:,:) :: chi_vadv, chi_tadv, chi_diaq
  real, allocatable, dimension(:,:,:) :: fmg_omg_err, fmg_omga_err
  real, allocatable, dimension(:,:,:,:) :: zgph, diaq, dzgph
  character(1000) :: ofname
  character(1000), allocatable, dimension(:,:) :: fname

  namelist /input /listname, nx, ny, nz, lonmin, latmin, dlon, dlat,  &
  &                pres, zord, tord, diaqord, undef
  namelist /fmg_opt /fmg_flag, fmg_level, fmg_levelz, fmg_convnum, fmg_additr
  read(5,nml=input)
  read(5,nml=fmg_opt)

  !-- Read all file name from listname
  nl=line_number_counter( trim(adjustl(listname)) )
  allocate(fname(1,nl))
  call read_file_text( trim(adjustl(listname)), 1, nl, fname )
  write(*,*) "*** MESSAGE (main) ***: Read file list."

  !-- Set some parameters
  d2r=pi/180.0

  !-- Allocate all variables
  allocate(lon(nx))
  allocate(lat(ny))
  allocate(lonr(nx))
  allocate(latr(ny))
  allocate(ref_temp(nz))
  allocate(ref_pt(nz))
  allocate(sig(nz))
  allocate(fcor(nx,ny))
  allocate(sx(nx,ny))
  allocate(sy(nx,ny))
  allocate(temp1d(nz,nl))
  allocate(zgph(nx,ny,nz,nl))
  allocate(dzgph(nx,ny,nz,nl))
  allocate(diaq(nx,ny,nz,nl))
  allocate(tg(nx,ny,nz))
  allocate(ug(nx,ny,nz))
  allocate(vg(nx,ny,nz))
  allocate(uga(nx,ny,nz))
  allocate(vga(nx,ny,nz))
  allocate(tga(nx,ny,nz))
  allocate(w(nx,ny,nz))
  allocate(wa(nx,ny,nz))
  allocate(pva(nx,ny,nz))
  allocate(pvm(nx,ny,nz))
  allocate(zgpha(nx,ny,nz))
  allocate(zgphm(nx,ny,nz))
  allocate(chi(nx,ny,nz))
  allocate(chia(nx,ny,nz))
  allocate(chi_vadv(nx,ny,nz))
  allocate(chi_tadv(nx,ny,nz))
  allocate(chi_diaq(nx,ny,nz))
  allocate(fmg_omg_err(nx,ny,nz))
  allocate(fmg_omga_err(nx,ny,nz))
  write(*,*) "*** MESSAGE (main) ***: Allocate variables."

  !-- Set lat-lon coordinate
  lon=(/((lonmin+dlon*real(i-1)),i=1,nx)/)
  lat=(/((latmin+dlat*real(j-1)),j=1,ny)/)
  lonr=lon*d2r
  latr=lat*d2r

  !-- Calculate constant parameters
  do j=1,ny
     fcor(1:nx,j)=2.0*omega*sin(latr(j))
     sx(1:nx,j)=radius*cos(latr(j))
  end do
  call Mean_2d( fcor, f0 )
  sy=radius

  !-- Input geopotential height from a data file
  do l=1,nl
     call read_file_3d( trim(adjustl(fname(1,l))), nx, ny, nz, zord,  &
  &                     zgph(:,:,:,l) )
     call calc_fact_3d( zgph(:,:,:,l), g, undef )  ! Z -> PHI

     call read_file_3d( trim(adjustl(fname(1,l))), nx, ny, nz, tord,  &
  &                     diaq(:,:,:,nl) )
                        ! diaq はここでは一時変数 (nl は最後に更新)
     do k=1,nz
        call Mean_2d( diaq(:,:,k,nl), temp1d(k,l) )
     end do
     if(diaqord>0)then
        call read_file_3d( trim(adjustl(fname(1,l))), nx, ny, nz, diaqord,  &
  &                        diaq(:,:,:,l) )
     else
        diaq(:,:,:,l)=0.0
     end if
     write(*,*) "*** MESSAGE (main) ***: Read "//trim(adjustl(fname(1,l)))
  end do
  write(*,*) "*** MESSAGE (main) ***: Read each file."

  !-- Calculate mean of zgph
!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,k)

  do k=1,nz
     do j=1,ny
        do i=1,nx
           call Mean_1d( zgph(i,j,k,1:nl), zgphm(i,j,k), error=undef )
        end do
     end do
  end do

!$omp end do
!$omp end parallel

  !-- Calculate reference thermodynamic field
  do k=1,nz
     call Mean_1d( temp1d(k,1:nl), ref_temp(k), error=undef )
     ref_pt(k)=theta_dry( ref_temp(k), pres(k) )
  end do

  call grad_1d( pres(1:nz), ref_pt, sig )
  do k=1,nz
     sig(k)=-(Rd*ref_temp(k)/(pres(k)*ref_pt(k)))*sig(k)
  end do

!  !-- Calculate mean geostrophic winds and temperature on the QG framework
!  call calc_zgph2uvt_QG( lonr, latr, pres, zgphm, ugm, vgm, tempgm,  &
!  &                      undef, sx, sy, f0 )

  !-- Calculate mean QGPV on P-coordinate
  call HQGPV( lonr, latr, pres(1:nz), zgphm, ref_temp, fcor, pvm,  &
  &           undef=undef, hx=sx, hy=sy )
  write(*,*) "*** MESSAGE (main) ***: Calculate HQGPV."

  !-- Calculate actual tendency of zph
  call calc_tendency_3d( zgph, dzgph, undef )
  write(*,*) "*** MESSAGE (main) ***: Calculate tendency of Phi."

  do l=1,nl
  !-- Calculate full geostrophic winds and temperature on the QG framework
     call calc_zgph2uvt_QG( lonr, latr, pres(1:nz), zgph(:,:,:,l), ug, vg, tg,  &
  &                         undef, sx, sy, f0 )
     write(*,*) "*** MESSAGE (main) ***: Calculate QG elements."

  !-- Calculate QGPV on P-coordinate
     call HQGPV( lonr, latr, pres(1:nz), zgph(:,:,:,l), ref_temp, fcor, pva,  &
  &              undef=undef, hx=sx, hy=sy )
     write(*,*) "*** MESSAGE (main) ***: Calculate HQGPV."

  !-- Calculate chi (dphi/dt) and omega for total QGPV
     call calc_chiomega_QG( lonr, latr, pres(1:nz), ug, vg, ug, vg, tg,  &
  &                         chi, w, diaq(:,:,:,l), undef, sx, sy, sig, f0,  &
  &                         chi_vadv, chi_tadv, chi_diaq, beta_flag=.true.,  &
  &                         fmg_flag=fmg_flag, fmg_level=fmg_level,  &
  &                         fmg_levelz=fmg_levelz(1:fmg_level),  &
  &                         fmg_convnum=fmg_convnum, fmg_additr=fmg_additr,  &
  &                         fmg_err=fmg_omg_err )
     write(*,*) "*** MESSAGE (main) ***: Calculate chi and omega on QG"

  !-- Calculate QGPV anomaly

     call calc_subtract_3d( pva, pvm, undef )

  !-- Calculate geopotential for the QGPV anomaly
  !-- pva は時間平均からの偏差で定義されているので, QGPV の定義式から
  !-- ポアソン方程式を解いてジオポテンシャルを求める必要がない.
  !-- 単にジオポテンシャルの時間平均からの偏差をとれば, それが pva に
  !-- 対応するジオポテンシャルの偏差に一致する.

     zgpha=zgph(:,:,:,l)
     call calc_subtract_3d( zgpha, zgphm, undef )

  !-- Calculate geostrophic winds and temperature for QGPV anomaly on the QG framework
     call calc_zgph2uvt_QG( lonr, latr, pres(1:nz), zgpha, uga, vga, tga,  &
  &                         undef, sx, sy, f0 )
     write(*,*) "*** MESSAGE (main) ***: Calculate QG elements for temporal anomaly."

  !-- Calculate chi (dphi/dt) and omega for the QGPV anomaly
     call calc_chiomega_QG( lonr, latr, pres(1:nz), ug, vg,  &
  &                         uga, vga, tga, chia, wa,  &
  &                         diaq(:,:,:,l), undef, sx, sy, sig, f0,  &
  &                         fmg_flag=fmg_flag, fmg_level=fmg_level,  &
  &                         fmg_levelz=fmg_levelz(1:fmg_level),  &
  &                         fmg_convnum=fmg_convnum, fmg_additr=fmg_additr,  &
  &                         fmg_err=fmg_omga_err )
!  &                         chia_vadv, chia_tadv, chia_diaq, uga, vga, tga )
     write(*,*) "*** MESSAGE (main) ***: Calculate chi and omega on QG for temporal anomaly."

     ofname=trim(adjustl(fname(1,l)))//'.QGPV'
     ic=1
     call write_file_3d( trim(adjustl(ofname)), nx, ny, nz, ic,  &
  &                      pvm, mode='replace' )
     ic=ic+nz
     call write_file_3d( trim(adjustl(ofname)), nx, ny, nz, ic,  &
  &                      pva, mode='old' )
     ic=ic+nz
     call write_file_3d( trim(adjustl(ofname)), nx, ny, nz, ic,  &
  &                      chi, mode='old' )
     ic=ic+nz
     call write_file_3d( trim(adjustl(ofname)), nx, ny, nz, ic,  &
  &                      w, mode='old' )
     ic=ic+nz
     call write_file_3d( trim(adjustl(ofname)), nx, ny, nz, ic,  &
  &                      ug, mode='old' )
     ic=ic+nz
     call write_file_3d( trim(adjustl(ofname)), nx, ny, nz, ic,  &
  &                      vg, mode='old' )
     ic=ic+nz
     call write_file_3d( trim(adjustl(ofname)), nx, ny, nz, ic,  &
  &                      tg, mode='old' )
     ic=ic+nz
     call write_file_3d( trim(adjustl(ofname)), nx, ny, nz, ic,  &
  &                      chia, mode='old' )
     ic=ic+nz
     call write_file_3d( trim(adjustl(ofname)), nx, ny, nz, ic,  &
  &                      wa, mode='old' )
     ic=ic+nz
     call write_file_3d( trim(adjustl(ofname)), nx, ny, nz, ic,  &
  &                      uga, mode='old' )
     ic=ic+nz
     call write_file_3d( trim(adjustl(ofname)), nx, ny, nz, ic,  &
  &                      vga, mode='old' )
     ic=ic+nz
     call write_file_3d( trim(adjustl(ofname)), nx, ny, nz, ic,  &
  &                      tga, mode='old' )
     ic=ic+nz
     call write_file_3d( trim(adjustl(ofname)), nx, ny, nz, ic,  &
  &                      dzgph(:,:,:,l), mode='old' )
     ic=ic+nz
     call write_file_3d( trim(adjustl(ofname)), nx, ny, nz, ic,  &
  &                      chi_vadv, mode='old' )
     ic=ic+nz
     call write_file_3d( trim(adjustl(ofname)), nx, ny, nz, ic,  &
  &                      chi_tadv, mode='old' )
     ic=ic+nz
     call write_file_3d( trim(adjustl(ofname)), nx, ny, nz, ic,  &
  &                      chi_diaq, mode='old' )
     if(fmg_flag.eqv..true.)then
        ic=ic+nz
        call write_file_3d( trim(adjustl(ofname)), nx, ny, nz, ic,  &
  &                         fmg_omg_err, mode='old' )
        ic=ic+nz
        call write_file_3d( trim(adjustl(ofname)), nx, ny, nz, ic,  &
  &                         fmg_omga_err, mode='old' )
     end if

     write(*,*) "Output "//trim(adjustl(ofname))

  end do

  write(*,*) "Output variables"
  write(*,*) "pvm 3d QGPV mean"
  write(*,*) "pva 3d QGPV anomaly"
  write(*,*) "chi 3d Geopotential tendency on QG"
  write(*,*) "omega 3d Omega-P velocity on QG"
  write(*,*) "ug 3d Zonal component of geostrophic wind speed"
  write(*,*) "vg 3d Meridional component of geostrophic wind speed"
  write(*,*) "tg 3d Air temperature on hydrostatic balance"
  write(*,*) "chia 3d Geopotential tendency for pva on QG"
  write(*,*) "omegaa 3d Omega-P velocity for pva on QG"
  write(*,*) "uga 3d Zonal component of geostrophic wind speed for pva"
  write(*,*) "vga 3d Meridional component of geostrophic wind speed for pva"
  write(*,*) "tga 3d Air temperature on hydrostatic balance for pva"
  write(*,*) "dzdt 3d Actual tendency of geopotential height"
  write(*,*) "chivadv 3d Geopotential tendency for PV advection on QG"
  write(*,*) "chitadv 3d Geopotential tendency for thickness advection on QG"
  write(*,*) "chidiaq 3d Geopotential tendency for diabatic heating on QG"
  write(*,*) "omgerr 3d The final error for FMG omega on QG"
  write(*,*) "omgaerr 3d The final error for FMG omegaa on QG"

end program QGPV
