module sub_mod

  use derivation
  use Algebra
  use Ellip_Slv

contains

subroutine calc_zgph2uvt_QG( lon, lat, pres, zgph, ug, vg, tg, undef, hx, hy, f0 )
!-- QG 系でジオポテンシャルから地衡風, 温度を計算する.
  use Math_Const
  use Phys_Const
  use Thermo_Const

  implicit none

  real, intent(in) :: lon(:)  ! 経度 [rad]
  real, intent(in) :: lat(:)  ! 緯度 [rad]
  real, intent(in) :: pres(:) ! 気圧 [Pa]
  real, intent(in) :: zgph(size(lon),size(lat),size(pres))
                              ! ジオポテンシャル [J/kg]
  real, intent(inout) :: ug(size(lon),size(lat),size(pres))
                              ! 地衡風 lon 成分 [m/s]
  real, intent(inout) :: vg(size(lon),size(lat),size(pres))
                              ! 地衡風 lat 成分 [m/s]
  real, intent(inout) :: tg(size(lon),size(lat),size(pres))
                              ! 静力学平衡温度 [K]
  real, intent(in) :: undef   ! 未定義値
  real, intent(in) :: hx(size(lon),size(lat))  ! lon 方向スケール因子
  real, intent(in) :: hy(size(lon),size(lat))  ! lat 方向スケール因子
  real, intent(in) :: f0      ! 基準緯度でのコリオリパラメータ [1/s]

  integer :: i, j, k, nx, ny, nz

  nx=size(lon)
  ny=size(lat)
  nz=size(pres)

!$omp parallel default(shared)
!$omp do schedule(runtime) private(k)

  do k=1,nz
     call grad_2d( lon, lat, zgph(:,:,k), vg(:,:,k), ug(:,:,k),  &
  &                undeff=undef, hx=hx, hy=hy )
  end do

!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j)

  do j=1,ny
     do i=1,nx
        call grad_1d( pres, zgph(i,j,:), tg(i,j,:), undef=undef )
     end do
  end do

!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j,k)

  do k=1,nz
     do j=1,ny
        do i=1,nx
           ug(i,j,k)=-ug(i,j,k)/f0
           vg(i,j,k)=vg(i,j,k)/f0
           tg(i,j,k)=-(pres(k)/Rd)*tg(i,j,k)
        end do
     end do
  end do

!$omp end do
!$omp end parallel

end subroutine calc_zgph2uvt_QG

!------------------------------
!------------------------------

subroutine calc_chiomega_QG( lon, lat, pres, ug, vg, uga, vga, tga,  &
  &                          chi, w, diaq, undef, hx, hy, sig, f0,  &
  &                          chi1, chi2, chi3, beta_flag,  &
  &                          fmg_flag, fmg_level, fmg_levelz,  &
  &                          fmg_convnum, fmg_additr, fmg_err )
!-- QG 系でジオポテンシャルからジオポテンシャルの傾向を診断する.
  use Phys_Const
  use Thermo_Const
  implicit none
  real, intent(in) :: lon(:)  ! 経度 [rad]
  real, intent(in) :: lat(:)  ! 緯度 [rad]
  real, intent(in) :: pres(:) ! 気圧 [Pa]
  real, intent(in) :: ug(size(lon),size(lat),size(pres))
                              ! 移流地衡風 lon 成分 [m/s]
  real, intent(in) :: vg(size(lon),size(lat),size(pres))
                              ! 移流地衡風 lat 成分 [m/s]
  real, intent(in) :: uga(size(lon),size(lat),size(pres))
                              ! 移流計算時の渦度用地衡風 lon 成分 [m/s]
  real, intent(in) :: vga(size(lon),size(lat),size(pres))
                              ! 移流計算時の渦度用地衡風 lat 成分 [m/s]
  real, intent(in) :: tga(size(lon),size(lat),size(pres))
                              ! 移流計算時の温度用静力学平衡温度 [K]
  real, intent(inout) :: chi(size(lon),size(lat),size(pres))
                              ! ジオポテンシャルの傾向 [gph/s]
  real, intent(inout) :: w(size(lon),size(lat),size(pres))
                              ! 鉛直p速度 [Pa/s]
  real, intent(in) :: diaq(size(lon),size(lat),size(pres))
                              ! 非断熱加熱 [K/s]
  real, intent(in) :: undef   ! 未定義値
  real, intent(in) :: hx(size(lon),size(lat))  ! lon 方向スケール因子
  real, intent(in) :: hy(size(lon),size(lat))  ! lat 方向スケール因子
  real, intent(in) :: sig(size(pres))
                              ! 静的安定度 [m2/Pa2/s2]
  real, intent(in) :: f0      ! 基準緯度でのコリオリパラメータ [1/s]
  real, intent(inout), optional :: chi1(size(lon),size(lat),size(pres))
                              ! 渦位移流によるジオポテンシャルの傾向 [gph/s]
  real, intent(inout), optional :: chi2(size(lon),size(lat),size(pres))
                              ! 層厚移流によるジオポテンシャルの傾向 [gph/s]
  real, intent(inout), optional :: chi3(size(lon),size(lat),size(pres))
                              ! 非断熱加熱によるジオポテンシャルの傾向 [gph/s]
  logical, intent(in), optional :: beta_flag
                              ! ベータ項を考慮する (=true). デフォルト = false
  logical, intent(in), optional :: fmg_flag
                              ! ポアソン方程式の求解に多重格子法を用いる
                              ! デフォルト = false
  integer, intent(in), optional :: fmg_level  ! 多重格子法での水平粗視化レベル
  integer, intent(in), optional :: fmg_levelz(:)
                              ! 多重格子法での鉛直粗視化レベル
  integer, intent(in), optional :: fmg_convnum
                              ! 多重格子法での反復回数
  integer, intent(in), optional :: fmg_additr
                              ! 多重格子法での完全 V サイクルの回数
  real, intent(inout), optional :: fmg_err(size(lon),size(lat),size(pres))
                              ! 多重格子法で得られた最終的な omega の誤差

  integer :: i, j, k, nx, ny, nz
  real, dimension(size(pres)) :: dsigdp
  real, dimension(size(lon),size(lat),size(pres)) :: term1, term2, term3
  real, dimension(size(lon),size(lat),size(pres)) :: termya, termza_chi, termza_omg
  real, dimension(size(lon),size(lat),size(pres)) :: terme, termf, rhov
  real, dimension(size(lon),size(lat),size(pres)) :: advvg, advtg, ds3dx
  real, dimension(size(lon),size(lat),size(pres)) :: boundv
  real, dimension(size(lon),size(lat)) :: betat, ds2dx, ux2, vy2

  real :: eps_chi, eps_omg
  character(6) :: boundary_chi, boundary_omg
  logical :: flfmg

  nx=size(lon)
  ny=size(lat)
  nz=size(pres)

  eps_chi=1.0e-3
  eps_omg=1.0e-2
  boundary_chi='111111'  ! For FMG
!ORG  boundary_chi='111122'
  boundary_omg='111111'
  boundv=0.0
  betat=0.0

  !-- calculate beta term (beta=df/dy)
  if(present(beta_flag))then
     if(beta_flag.eqv..true.)then
        do j=1,ny
           betat(1:nx,j)=(2.0*omega/radius)*cos(lat(j))
        end do
     end if
  end if

  if(present(fmg_flag))then
     flfmg=fmg_flag
     if(size(fmg_levelz)/=fmg_level)then
        write(*,*) "*** ERROR (calc_chiomega_QG) ***: the size of fmg_levelz is not fmg_level. STOP"
        stop
     end if
  else
     flfmg=.false.
  end if

  !-- calculate dsig/dp
  call grad_1d( pres, sig, dsigdp, undef=undef )

  !-- Calculate forcing terms
  !-- For term1 (horizontal advection of vorticity)
  do k=1,nz
     call curl( lon, lat, uga(:,:,k), vga(:,:,k), ds2dx, undeff=undef,  &
  &             hx=hx, hy=hy )
     call grad_2d( lon, lat, ds2dx, ux2, vy2, undeff=undef, hx=hx, hy=hy )
     call dot_prod_2d( ug(:,:,k), vg(:,:,k), ux2, vy2, advvg(:,:,k), undeff=undef )
  end do
  do k=1,nz
     do j=1,ny
        do i=1,nx
           if(advvg(i,j,k)/=undef)then
              advvg(i,j,k)=advvg(i,j,k)+vg(i,j,k)*betat(i,j)
              term1(i,j,k)=-f0*advvg(i,j,k)
           end if
        end do
     end do
  end do

  !-- For term2 (horizontal advection of thickness)
  do k=1,nz
     call grad_2d( lon, lat, tga(:,:,k), ux2, vy2, undeff=undef,  &
  &                hx=hx, hy=hy )
     call dot_prod_2d( ug(:,:,k), vg(:,:,k), ux2, vy2, advtg(:,:,k), undeff=undef )
  end do
  do k=1,nz
     do j=1,ny
        do i=1,nx
           if(advtg(i,j,k)/=undef)then
              ds3dx(i,j,k)=(f0*f0*Rd)*advtg(i,j,k)/(sig(k)*pres(k))
           end if
        end do
     end do
  end do
  do j=1,ny
     do i=1,nx
        call grad_1d( pres, ds3dx(i,j,:), term2(i,j,:), undef=undef )
     end do
  end do

  !-- For term3 (diabatic heating effects)
  do k=1,nz
     do j=1,ny
        do i=1,nx
           if(diaq(i,j,k)/=undef)then
              ! J/cp = diaq * pi
              ds3dx(i,j,k)=-f0*f0*Rd*diaq(i,j,k)*((pres(k)/p0)**(Rd/Cpd))  &
  &                        /(sig(k)*pres(k))
           end if
        end do
     end do
  end do
  do j=1,ny
     do i=1,nx
        call grad_1d( pres, ds3dx(i,j,:), term3(i,j,:), undef=undef )
     end do
  end do

  !-- Calculate chi (dphi/dt) (Invert Poisson equation)
  do k=1,nz
     do j=1,ny
        do i=1,nx
           termya(i,j,k)=cos(lat(j))**2
           termza_chi(i,j,k)=(dsigdp(k)/sig(k))*(radius*cos(lat(j))*f0)**2
           termza_omg(i,j,k)=(1.0/sig(k))*(radius*cos(lat(j))*f0)**2
           terme(i,j,k)=-sin(lat(j))*cos(lat(j))
           termf(i,j,k)=dsigdp(k)*(radius*cos(lat(j))*f0/sig(k))**2
           term1(i,j,k)=((radius*cos(lat(j)))**2)*term1(i,j,k)
           term2(i,j,k)=((radius*cos(lat(j)))**2)*term2(i,j,k)
           term3(i,j,k)=((radius*cos(lat(j)))**2)*term3(i,j,k)
           rhov(i,j,k)=term1(i,j,k)+term2(i,j,k)+term3(i,j,k)
           if(k==1.or.k==nz)then
              boundv(i,j,k)=advtg(i,j,k)*(Rd/pres(k))
           end if
        end do
     end do
  end do

  if(flfmg.eqv..true.)then
     call Full_Multi_Grid_3d( fmg_level, fmg_levelz(1:fmg_level),  &
  &                           lon, lat, pres, rhov, eps_chi, boundary_chi, chi,  &
  &                           ya=termya, za=termza_chi, e=terme, f=termf,  &
  &                           conv_num=fmg_convnum, add_itr=fmg_additr )
  else
     call Ellip_Jacobi_3d( lon, lat, pres, rhov, eps_chi, boundary_chi, chi,  &
  &                        ya=termya, za=termza_chi,  &
  &                        e=terme, f=termf, undef=undef, bound_opt=boundv )
  end if

  !-- Each contribution of term{1,2,3} to chi
  if (present(chi1))then
     if(flfmg.eqv..true.)then
        call Full_Multi_Grid_3d( fmg_level, fmg_levelz(1:fmg_level),  &
  &                              lon, lat, pres, term1, eps_chi,  &
  &                              boundary_chi, chi1,  &
  &                              ya=termya, za=termza_chi, e=terme, f=termf,  &
  &                              conv_num=fmg_convnum, add_itr=fmg_additr )
     else
        call Ellip_Jacobi_3d( lon, lat, pres, term1, eps_chi, boundary_chi,  &
  &                           chi1, ya=termya, za=termza_chi,  &
  &                           e=terme, f=termf, undef=undef, bound_opt=boundv )
     end if
  end if
  if (present(chi2))then
     if(flfmg.eqv..true.)then
        call Full_Multi_Grid_3d( fmg_level, fmg_levelz(1:fmg_level),  &
  &                              lon, lat, pres, term2, eps_chi,  &
  &                              boundary_chi, chi2,  &
  &                              ya=termya, za=termza_chi, e=terme, f=termf,  &
  &                              conv_num=fmg_convnum, add_itr=fmg_additr )
     else
        call Ellip_Jacobi_3d( lon, lat, pres, term2, eps_chi, boundary_chi,  &
  &                           chi2, ya=termya, za=termza_chi,  &
  &                           e=terme, f=termf, undef=undef, bound_opt=boundv )
     end if
  end if
  if (present(chi3))then
     if(flfmg.eqv..true.)then
        call Full_Multi_Grid_3d( fmg_level, fmg_levelz(1:fmg_level),  &
  &                              lon, lat, pres, term3, eps_chi,  &
  &                              boundary_chi, chi3,  &
  &                              ya=termya, za=termza_chi, e=terme, f=termf,  &
  &                              conv_num=fmg_convnum, add_itr=fmg_additr )
     else
        call Ellip_Jacobi_3d( lon, lat, pres, term3, eps_chi, boundary_chi,  &
  &                           chi3, ya=termya, za=termza_chi,  &
  &                           e=terme, f=termf, undef=undef, bound_opt=boundv )
     end if
  end if

  !-- Calculate omega (Invert Poisson equation)
  ds3dx=undef
  do k=1,nz
     do j=1,ny
        do i=1,nx
           if(advtg(i,j,k)/=undef.and.diaq(i,j,k)/=undef)then
              ! J/cp = Q*pi
              ds3dx(i,j,k)=advtg(i,j,k)-diaq(i,j,k)*((pres(k)/p0)**(Rd/Cpd))
           end if
        end do
     end do
  end do
  do k=1,nz
     call laplacian_2d( lon, lat, ds3dx(:,:,k), term2(:,:,k), undef=undef,  &
  &                     hx=hx, hy=hy )
  end do
  do j=1,ny
     do i=1,nx
        call grad_1d( pres, advvg(i,j,:), term1(i,j,:), undef=undef )
     end do
  end do
  do k=1,nz
     do j=1,ny
        do i=1,nx
           rhov(i,j,k)=((radius*cos(lat(j)))**2)*((f0/sig(k))*term1(i,j,k)  &
  &                           +(Rd/(sig(k)*pres(k)))*term2(i,j,k))
        end do
     end do
  end do
  boundv=0.0

  if(flfmg.eqv..true.)then
     call Full_Multi_Grid_3d( fmg_level, fmg_levelz(1:fmg_level),  &
  &                           lon, lat, pres, rhov, eps_omg, boundary_omg, w,  &
  &                           ya=termya, za=termza_omg, e=terme,  &
  &                           conv_num=fmg_convnum, add_itr=fmg_additr,  &
  &                           rlu_err=fmg_err )
  else
     call Ellip_Jacobi_3d( lon, lat, pres, rhov, eps_omg, boundary_omg, w,  &
  &                        ya=termya, za=termza_omg, e=terme,  &
  &                        undef=undef, bound_opt=boundv )
  end if

end subroutine calc_chiomega_QG

!------------------------------
!------------------------------

subroutine calc_fact_3d( val, fact, undef )
! val x fact を 3 次元引数について計算する.
  implicit none
  real, intent(inout) :: val(:,:,:)
  real, intent(in) :: fact
  real, intent(in) :: undef

  integer :: i, j, k

  do k=1,size(val,3)
     do j=1,size(val,2)
        do i=1,size(val,1)
           if(val(i,j,k)/=undef)then
              val(i,j,k)=val(i,j,k)*fact
           end if
        end do
     end do
  end do

end subroutine calc_fact_3d

!------------------------------
!------------------------------

subroutine calc_subtract_3d( a, b, undef )
! 3 次元引数について a-b を計算し, a に返す.
  implicit none
  real, intent(inout) :: a(:,:,:)
  real, intent(in) :: b(size(a,1),size(a,2),size(a,3))
  real, intent(in) :: undef

  integer :: i, j, k

  do k=1,size(a,3)
     do j=1,size(a,2)
        do i=1,size(a,1)
           if(a(i,j,k)/=undef)then
              a(i,j,k)=a(i,j,k)-b(i,j,k)
           end if
        end do
     end do
  end do

end subroutine calc_subtract_3d

!------------------------------
!------------------------------

subroutine calc_tendency_3d( ival, oval, undef )
!-- 任意の 4 次元配列の最後尾要素を用いて, 3 次元の時間変化を計算する.
  implicit none
  real, intent(in) :: ival(:,:,:,:)
  real, intent(inout) :: oval(size(ival,1),size(ival,2),size(ival,3),size(ival,4))
  real, intent(in) :: undef
  integer :: i, j, k, l, nx, ny, nz, nt

  nx=size(ival,1)
  ny=size(ival,2)
  nz=size(ival,3)
  nt=size(ival,4)

  do l=2,nt-1
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(ival(i,j,k,l+1)/=undef.and.ival(i,j,k,l-1)/=undef)then
                 oval(i,j,k,l)=0.5*(ival(i,j,k,l+1)-ival(i,j,k,l-1))
              end if
           end do
        end do
     end do
  end do

  do k=1,nz
     do j=1,ny
        do i=1,nx
           if(ival(i,j,k,1)/=undef.and.ival(i,j,k,2)/=undef)then
              oval(i,j,k,1)=ival(i,j,k,2)-ival(i,j,k,1)
           end if
        end do
     end do
  end do

  do k=1,nz
     do j=1,ny
        do i=1,nx
           if(ival(i,j,k,nt-1)/=undef.and.ival(i,j,k,nt)/=undef)then
              oval(i,j,k,nt)=ival(i,j,k,nt)-ival(i,j,k,nt-1)
           end if
        end do
     end do
  end do

end subroutine calc_tendency_3d

!------------------------------
!------------------------------

end module sub_mod
