program cloud_1d
  use Derivation
  use file_operate
  use gtool_history
  use Thermo_Function
  use Thermo_Const
  use Cloud
  use Basis
  use Math_Const

  implicit none

  integer :: i, j, k, nmax, nt, dmpstp
  real, parameter :: zmax=5000.0   ! vertical wind maximum height.
  real, parameter :: wfact=2.0   ! vertical wind maximum height.
  real :: dt, rho0
  real :: EV_term, CL_term, CN_term
  real, allocatable, dimension(:) :: z, w, qv, qr, qc, rho, pt
  real, allocatable, dimension(:) :: old_qv, old_qr, old_qc, old_pt
  real, allocatable, dimension(:) :: tmp_qv, tmp_qr, tmp_qc, tmp_pt
  real, allocatable, dimension(:) :: temp, pres, fall
  real, allocatable, dimension(:) :: dqvdz, dqrdz, dqcdz, dptdz, dfalldz
  real, allocatable, dimension(:) :: forceqv, forceqc, forceqr, forcept
  real, dimension(2) :: qvbc, qcbc, qrbc, ptbc
  character(100) :: fname, outname
  character(20), allocatable, dimension(:,:) :: cval

  namelist /input /dt, nt, fname, outname, dmpstp
  read(5,nml=input)

  nmax=line_number_counter( trim(fname) )

  allocate(cval(4,nmax))
  allocate(z(nmax))
  allocate(w(nmax))
  allocate(qv(nmax))
  allocate(qc(nmax))
  allocate(qr(nmax))
  allocate(rho(nmax))
  allocate(pt(nmax))
  allocate(fall(nmax))
  allocate(tmp_qv(nmax))
  allocate(tmp_qc(nmax))
  allocate(tmp_qr(nmax))
  allocate(tmp_pt(nmax))
  allocate(old_qv(nmax))
  allocate(old_qc(nmax))
  allocate(old_qr(nmax))
  allocate(old_pt(nmax))
  allocate(temp(nmax))
  allocate(pres(nmax))
  allocate(dqvdz(nmax))
  allocate(dqcdz(nmax))
  allocate(dqrdz(nmax))
  allocate(dptdz(nmax))
  allocate(dfalldz(nmax))
  allocate(forcept(nmax))
  allocate(forceqv(nmax))
  allocate(forceqc(nmax))
  allocate(forceqr(nmax))

!-- reading data from fname

  call read_file_text( trim(fname), 4, nmax, cval )

  do j=1,nmax
     z(j)=c2r_convert( trim(adjustl(cval(1,j))) )
     pres(j)=c2r_convert( trim(adjustl(cval(2,j))) )
     temp(j)=c2r_convert( trim(adjustl(cval(3,j))) )
     qv(j)=c2r_convert( trim(adjustl(cval(4,j))) )
     pt(j)=theta_dry( temp(j), pres(j) )
     rho(j)=TP_2_rho( thetaP_2_T(pt(j), pres(j)), pres(j))
  end do

!-- setting vertical wind

  do i=1,nmax
!     w(i)=exp(-(z(i)-z(nmax/2))**2)
     if(2.0*(zmax-z(1))>=z(i))then
        w(i)=wfact*sin(pi*(z(i)-z(1))/(z(nmax)-z(1)))
     else
        w(i)=0.0
     end if
     qc(i)=0.0
     qr(i)=0.0
  end do
write(*,*) w

  rho0=TP_2_rho( thetaP_2_T(pt(1), pres(1)), pres(1))

  ptbc=(/pt(1),pt(nmax)/)
  qvbc=(/qv(1),qv(nmax)/)
  qcbc=0.0
  qrbc=0.0

!-- output file is initialized.

  call HistoryCreate( file=trim(outname), title='cloud',  &
  &                   source='test', institution='test', dims=(/'z', 't'/),  &
  &                   dimsizes=(/nmax,0/),  &
  &                   longnames=(/'Z-coordinate', 'time        '/),  &
  &                   units=(/'m','s'/), origin=0.0,  &
  &                   interval=dt*real(dmpstp) )

  call HistoryPut( 'z', z )
!  call HistoryPut( 't', t )

  call HistoryAddVariable( varname='pt', dims=(/'z','t'/),  &
  &                        longname='potential temperature',  &
  &                        units='K', xtype='float' )
  call HistoryAddVariable( varname='qv', dims=(/'z','t'/),  &
  &                        longname='mixing ratio of water vapor',  &
  &                        units='kg kg-1', xtype='float' )
  call HistoryAddVariable( varname='qc', dims=(/'z','t'/),  &
  &                        longname='mixing ratio of cloud water',  &
  &                        units='kg kg-1', xtype='float' )
  call HistoryAddVariable( varname='qr', dims=(/'z','t'/),  &
  &                        longname='mixing ratio of rain',  &
  &                        units='kg kg-1', xtype='float' )

!-- starting time integration

  do i=1,nt

  !-- initializing forcing terms

     do j=1,nmax
        forcept(j)=0.0
        forceqv(j)=0.0
        forceqc(j)=0.0
        forceqr(j)=0.0
     end do

  !-- calculating terminal velocity.
  !-- [note] 

     do j=1,nmax
        fall(j)=rho(j)*term_v( rho(j), qr(j), rho0 )*qr(j)
     end do

  !-- calculating microphysics schemes

     do j=2,nmax-1
        if(qv(j)>0.0.or.qc(j)>0.0.or.qr(j)>0.0)then
           EV_term=EVrv( temp(j), qv(j), qr(j), pres(j) )
           CN_term=CNcr( qc(j) )
           CL_term=CLcr( qc(j), qr(j) )
           forcept(j)=forcept(j)  &
  &                   +Src_theta_warm( temp(j), pres(j), qv(j), qr(j) )
           forceqv(j)=forceqv(j)+EV_term
           forceqc(j)=forceqc(j)-CN_term-CL_term
           forceqr(j)=forceqr(j)+CN_term+CL_term-EV_term
        end if
     end do

  !-- adding the advecting terms
  !-- calculating vertical gradient

     call grad_1d( z, qv, dqvdz )
     call grad_1d( z, qc, dqcdz )
     call grad_1d( z, qr, dqrdz )
     call grad_1d( z, pt, dptdz )
     call grad_1d( z, fall, dfalldz )

     do j=2,nmax-1
        forceqv(j)=forceqv(j)-w(j)*dqvdz(j)
        forceqc(j)=forceqc(j)-w(j)*dqcdz(j)
        forceqr(j)=forceqr(j)-w(j)*dqrdz(j)+dfalldz(j)
     end do

     if(i==1)then  ! first step is Euler method.
        do j=2,nmax-1
           tmp_pt(j)=pt(j)+dt*forcept(j)
           tmp_qv(j)=qv(j)+dt*forceqv(j)
           tmp_qc(j)=qc(j)+dt*forceqc(j)
           tmp_qr(j)=qr(j)+dt*forceqr(j)
        end do

        call forced_zero( tmp_qv )
        call forced_zero( tmp_qc )
        call forced_zero( tmp_qr )

        !-- Moist Saturated Adjustment

        do j=2,nmax-1
           if(tmp_qv(j)>0.0.or.tmp_qc(j)>0.0)then
              call Moist_Sature_Adjust( pres(j), tmp_pt(j), tmp_qv(j), tmp_qc(j) )
           end if
        end do

        call forced_zero( tmp_qv )
        call forced_zero( tmp_qc )

        !-- boundary condition

!        tmp_pt(1)=tmp_pt(2)
!        tmp_pt(nmax)=tmp_pt(nmax-1)
!        tmp_qv(1)=tmp_qv(2)
!        tmp_qv(nmax)=tmp_qv(nmax-1)
        tmp_qc(1)=tmp_qc(2)
        tmp_qc(nmax)=tmp_qc(nmax-1)
        tmp_qr(1)=tmp_qr(2)
        tmp_qr(nmax)=tmp_qr(nmax-1)
        tmp_pt(1)=ptbc(1)
        tmp_pt(nmax)=ptbc(2)
        tmp_qv(1)=qvbc(1)
        tmp_qv(nmax)=qvbc(2)
!        tmp_qc(1)=qcbc(1)
!        tmp_qc(nmax)=qcbc(2)
!        tmp_qr(1)=qrbc(1)
!        tmp_qr(nmax)=qrbc(2)

  write(*,*) "*******************************************"
  write(*,*) "File damp (time =", 0.0, "[s])."
  write(*,*) "*******************************************"
  do j=1,nmax
  end do
        call HistoryPut( 'pt', pt )
        call HistoryPut( 'qv', qv )
        call HistoryPut( 'qc', qc )
        call HistoryPut( 'qr', qr )

        do j=1,nmax
           old_pt(j)=pt(j)
           old_qv(j)=qv(j)
           old_qc(j)=qc(j)
           old_qr(j)=qr(j)
           pt(j)=tmp_pt(j)
           qv(j)=tmp_qv(j)
           qc(j)=tmp_qc(j)
           qr(j)=tmp_qr(j)
        end do

     else  ! after step is leap flog method.

        do j=2,nmax-1
           tmp_pt(j)=old_pt(j)+2.0*dt*forcept(j)
           tmp_qv(j)=old_qv(j)+2.0*dt*forceqv(j)
           tmp_qc(j)=old_qc(j)+2.0*dt*forceqc(j)
           tmp_qr(j)=old_qr(j)+2.0*dt*forceqr(j)
        end do

        call forced_zero( tmp_qv )
        call forced_zero( tmp_qc )
        call forced_zero( tmp_qr )

        !-- Moist Saturated Adjustment

        do j=2,nmax-1
           if(tmp_qv(j)>0.0.or.tmp_qc(j)>0.0)then
              call Moist_Sature_Adjust( pres(j), tmp_pt(j), tmp_qv(j), tmp_qc(j) )
           end if
        end do

        call forced_zero( tmp_qv )
        call forced_zero( tmp_qc )

        !-- boundary condition

!        tmp_pt(1)=tmp_pt(2)
!        tmp_pt(nmax)=tmp_pt(nmax-1)
!        tmp_qv(1)=tmp_qv(2)
!        tmp_qv(nmax)=tmp_qv(nmax-1)
        tmp_qc(1)=tmp_qc(2)
        tmp_qc(nmax)=tmp_qc(nmax-1)
        tmp_qr(1)=tmp_qr(2)
        tmp_qr(nmax)=tmp_qr(nmax-1)
        tmp_pt(1)=ptbc(1)
        tmp_pt(nmax)=ptbc(2)
        tmp_qv(1)=qvbc(1)
        tmp_qv(nmax)=qvbc(2)
!        tmp_qc(1)=qcbc(1)
!        tmp_qc(nmax)=qcbc(2)
!        tmp_qr(1)=qrbc(1)
!        tmp_qr(nmax)=qrbc(2)

        if(mod(i-1,dmpstp)==0)then
           call HistoryPut( 'pt', pt )
           call HistoryPut( 'qv', qv )
           call HistoryPut( 'qc', qc )
           call HistoryPut( 'qr', qr )
           write(*,*) "*******************************************"
           write(*,*) "File damp (time =", real(i-1)*dt, "[s])."
           write(*,*) "*******************************************"
        end if

        do j=1,nmax
           old_pt(j)=pt(j)
           old_qv(j)=qv(j)
           old_qc(j)=qc(j)
           old_qr(j)=qr(j)
           pt(j)=tmp_pt(j)
           qv(j)=tmp_qv(j)
           qc(j)=tmp_qc(j)
           qr(j)=tmp_qr(j)
        end do
     end if

     write(*,*) "This step is ", i, "(time =", real(i-1)*dt, "[s])."

  end do

  write(*,*) "solver is normally."

contains

subroutine forced_zero( val )
  implicit none
  real, intent(inout) :: val(:)
  integer :: i, n

  n=size(val)

  do i=1,n
     if(val(i)<0.0)then
        val(i)=0.0
     end if
  end do

end subroutine forced_zero
end program
