!= 物理過程のみの計算のための力学過程
!
!= A dynamics for calculation with physical processes only
!
! Authors::   Yoshiyuki O. Takahashi
! Version::   $Id: dynamics_physicsonly.f90,v 1.7 2015/01/31 06:16:26 yot Exp $ 
! Tag Name::  $Name:  $
! Copyright:: Copyright (C) GFD Dennou Club, 2008-2009. All rights reserved.
! License::   See COPYRIGHT[link:../../../COPYRIGHT]
!

module dynamics_physicsonly
  !
  != 物理過程のみの計算のための力学過程
  !
  != A dynamics for calculation with physical processes only
  !
  ! <b>Note that Japanese and English are described in parallel.</b>
  !
  !== Procedures List
  !
  !== NAMELIST
  !
  !

  ! モジュール引用 ; USE statements
  !

  ! 格子点設定
  ! Grid points settings
  !
  use gridset, only: imax, & ! 経度格子点数. 
                             ! Number of grid points in longitude
    &                jmax, & ! 緯度格子点数. 
                             ! Number of grid points in latitude
    &                kmax    ! 鉛直層数. 
                             ! Number of vertical level
  ! 組成に関わる配列の設定
  ! Settings of array for atmospheric composition
  !
  use composition, only: &
    &                    ncmax
                              ! 成分の数
                              ! Number of composition

  ! 種別型パラメタ
  ! Kind type parameter
  !
  use dc_types, only: DP, &      ! 倍精度実数型. Double precision. 
    &                 STRING     ! 文字列.       Strings. 

  ! メッセージ出力
  ! Message output
  !
  use dc_message, only: MessageNotify

  ! 宣言文 ; Declaration statements
  !
  implicit none
  private

  ! 公開手続き
  ! Public procedure
  !
  public :: DynamicsPhysicsOnly
  public :: DynamicsPhysicsOnlyInit

  ! 公開変数
  ! Public variables
  !

  ! 非公開変数
  ! Private variables
  !
  real(DP), allocatable, save :: xyz_CorPar(:,:,:)
  real(DP)             , save :: UGeo
  real(DP)             , save :: VGeo
  real(DP)             , save :: WMagnitude
  real(DP)             , save :: WHeight
  real(DP)             , save :: WTimeInit

  logical              , save :: FlagDynExp

  logical, save :: dynamics_physicsonly_inited = .false.
                              ! 初期設定フラグ. 
                              ! Initialization flag


  character(*), parameter:: module_name = 'dynamics_physicsonly'
                              ! モジュールの名称. 
                              ! Module name
  character(*), parameter:: version = &
    & '$Name:  $' // &
    & '$Id: dynamics_physicsonly.f90,v 1.7 2015/01/31 06:16:26 yot Exp $'
                              ! モジュールのバージョン
                              ! Module version

  ! INTERFACE 文 ; INTERFACE statements
  !

contains

  subroutine DynamicsPhysicsOnly(                                &
    & xyz_Exner, xy_SurfHeight, xyz_Height,                      & ! (in)
    & xyz_DUDtPhy, xyz_DVDtPhy, xyz_DTempDtPhy, xyzf_DQMixDtPhy, & ! (in)
    & xy_PsB, xyz_UB, xyz_VB, xyz_TempB, xyzf_QMixB,             & ! (in)
    & xy_PsN, xyz_UN, xyz_VN, xyz_TempN, xyzf_QMixN,             & ! (in)
    & xy_PsA, xyz_UA, xyz_VA, xyz_TempA, xyzf_QMixA              & ! (out)
    & )

    ! モジュール引用 ; USE statements
    !

    ! 時刻管理
    ! Time control
    !
    use timeset, only: &
      & DelTime, &            ! $ \Delta t $ [s]
      & TimeN, &              ! ステップ $ t $ の時刻. Time of step $ t $. 
      & TimesetClockStart, TimesetClockStop

    ! デバッグ用ユーティリティ
    ! Utilities for debug
    !
    use dc_trace, only: DbgMessage, BeginSub, EndSub

    ! 組成に関わる配列の設定
    ! Settings of array for atmospheric composition
    !
    use composition, only:                              &
      &                    ncmax,                       &
                               ! 成分の数
                               ! Number of composition
      &                    CompositionInqFlagAdv

    ! 1 次元計算用力学過程ユーティリティモジュール
    ! Utility module for dynamics for 1-D calculation
    !
    use dynamics_1d_utils, only : Dynamics1DUtilsVerAdv

!!$    ! 質量の補正
!!$    ! Mass fixer
!!$    !
!!$    use mass_fixer, only: MassFixerColumn


    ! 宣言文 ; Declaration statements
    !

    real(DP), intent(in):: xyz_Exner      (0:imax-1, 1:jmax, 1:kmax)
    real(DP), intent(in):: xy_SurfHeight  (0:imax-1, 1:jmax)
    real(DP), intent(in):: xyz_Height     (0:imax-1, 1:jmax, 1:kmax)

    real(DP), intent(in):: xyz_DUDtPhy    (0:imax-1, 1:jmax, 1:kmax)
                              ! $ \left(\DP{u}{t}\right)^{phy} $ .
                              ! 外力項 (物理過程) による東西風速変化.
                              ! Eastward wind tendency by external force terms (physical processes)
    real(DP), intent(in):: xyz_DVDtPhy    (0:imax-1, 1:jmax, 1:kmax)
                              ! $ \left(\DP{v}{t}\right)^{phy} $ .
                              ! 外力項 (物理過程) による南北風速変化.
                              ! Northward wind tendency by external force terms (physicalprocesses)
    real(DP), intent(in):: xyz_DTempDtPhy (0:imax-1, 1:jmax, 1:kmax)
                              ! $ \left(\DP{T}{t}\right)^{phy} $ .
                              ! 外力項 (物理過程) による温度変化.
                              ! Temperature tendency by external force terms (physical processes)
    real(DP), intent(in):: xyzf_DQMixDtPhy (0:imax-1, 1:jmax, 1:kmax, 1:ncmax)
                              ! $ \left(\DP{q}{t}\right)^{phy} $ .
                              ! 外力項 (物理過程) による比湿変化.
                              ! Temperature tendency by external force terms (physical processes)
    real(DP), intent(in ):: xy_PsB(0:imax-1, 1:jmax)
                              ! $ p_s $ . 地表面気圧 (半整数レベル). 
                              ! Surface pressure (half level)
    real(DP), intent(in ):: xyz_UB(0:imax-1, 1:jmax, 1:kmax)
                              ! $ u $ . 東西風速. Eastward wind
    real(DP), intent(in ):: xyz_VB(0:imax-1, 1:jmax, 1:kmax)
                              ! $ v $ . 南北風速. Northward wind
    real(DP), intent(in ):: xyz_TempB(0:imax-1, 1:jmax, 1:kmax)
                              ! $ T $ . 温度 (整数レベル). 
                              ! Temperature (full level)
    real(DP), intent(in ):: xyzf_QMixB(0:imax-1, 1:jmax, 1:kmax, 1:ncmax)
                              ! $ q $ .     比湿. Specific humidity
    real(DP), intent(in):: xy_PsN    (0:imax-1, 1:jmax)
                              ! $ p_s (t) $ .   地表面気圧. Surface pressure
    real(DP), intent(in):: xyz_UN   (0:imax-1, 1:jmax, 1:kmax)
                              ! $ u (t) $ .     東西風速. Eastward wind
    real(DP), intent(in):: xyz_VN   (0:imax-1, 1:jmax, 1:kmax)
                              ! $ v (t) $ .     南北風速. Northward wind
    real(DP), intent(in):: xyz_TempN (0:imax-1, 1:jmax, 1:kmax)
                              ! $ T (t) $ .     温度. Temperature
    real(DP), intent(in):: xyzf_QMixN(0:imax-1, 1:jmax, 1:kmax, 1:ncmax)
                              ! $ q (t) $ .     比湿. Specific humidity
    real(DP), intent(out):: xy_PsA(0:imax-1, 1:jmax)
                              ! $ p_s $ . 地表面気圧 (半整数レベル). 
                              ! Surface pressure (half level)
    real(DP), intent(out):: xyz_UA(0:imax-1, 1:jmax, 1:kmax)
                              ! $ u $ . 東西風速. Eastward wind
    real(DP), intent(out):: xyz_VA(0:imax-1, 1:jmax, 1:kmax)
                              ! $ v $ . 南北風速. Northward wind
    real(DP), intent(out):: xyz_TempA(0:imax-1, 1:jmax, 1:kmax)
                              ! $ T $ . 温度 (整数レベル). 
                              ! Temperature (full level)
    real(DP), intent(out):: xyzf_QMixA(0:imax-1, 1:jmax, 1:kmax, 1:ncmax)
                              ! $ q $ .     比湿. Specific humidity


    ! 作業変数
    ! Work variables
    !
    real(DP) :: xyz_PotTempB     (0:imax-1, 1:jmax, 1:kmax)

    real(DP) :: xyz_W            (0:imax-1, 1:jmax, 1:kmax)

    real(DP) :: xyz_DUDtAdv      (0:imax-1, 1:jmax, 1:kmax)
    real(DP) :: xyz_DVDtAdv      (0:imax-1, 1:jmax, 1:kmax)
    real(DP) :: xyz_DPotTempDtAdv(0:imax-1, 1:jmax, 1:kmax)
    real(DP) :: xyzf_DQMixDtAdv  (0:imax-1, 1:jmax, 1:kmax, 1:ncmax)

    real(DP) :: xyz_UT           (0:imax-1, 1:jmax, 1:kmax)
    real(DP) :: xyz_VT           (0:imax-1, 1:jmax, 1:kmax)

    real(DP) :: DelTimeX2

    integer:: k
    integer:: n

    ! 実行文 ; Executable statement
    !

    ! 初期化確認
    ! Initialization check
    !
    if ( .not. dynamics_physicsonly_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    ! 計算時間計測開始
    ! Start measurement of computation time
    !
    call TimesetClockStart( module_name )


    DelTimeX2 = 2.0_DP * DelTime


    xyz_PotTempB = xyz_TempB / xyz_Exner

    if ( TimeN >= WTimeInit ) then
      if ( WHeight > 0.0_DP ) then
        do k = 1, kmax
          xyz_W(:,:,k) =                                                     &
            & WMagnitude * min( xyz_Height(:,:,k) - xy_SurfHeight, WHeight ) &
            &            / WHeight
        end do
      else
        xyz_W = 0.0_DP
      end if
    else
      xyz_W = 0.0_DP
    end if

    ! 1 次元計算用力学過程ユーティリティモジュール
    ! Utility module for dynamics for 1-D calculation
    !
    call Dynamics1DUtilsVerAdv(    &
      & xyz_W, xyz_Height, xyz_UB, &
      & xyz_DUDtAdv                &
      & )
    call Dynamics1DUtilsVerAdv(    &
      & xyz_W, xyz_Height, xyz_VB, &
      & xyz_DVDtAdv                &
      & )
    call Dynamics1DUtilsVerAdv(          &
      & xyz_W, xyz_Height, xyz_PotTempB, &
      & xyz_DPotTempDtAdv                &
      & )
    do n = 1, ncmax
      if ( CompositionInqFlagAdv( n ) ) then
        call Dynamics1DUtilsVerAdv(          &
          & xyz_W, xyz_Height, xyzf_QMixB(:,:,:,n), &
          & xyzf_DQMixDtAdv(:,:,:,n)                &
          & )
      else
        xyzf_DQMixDtAdv(:,:,:,n) = 0.0_DP
      end if
    end do

    xy_PsA = xy_PsB

    if ( FlagDynExp ) then
      xyz_UA = xyz_UB &
        & + ( xyz_DUDtAdv + xyz_CorPar * xyz_VB + xyz_DUDtPhy ) &
        &     * DelTimeX2
      xyz_VA = xyz_VB &
        & + ( xyz_DVDtAdv - xyz_CorPar * xyz_UB + xyz_DVDtPhy ) &
        &     * DelTimeX2
    else
      xyz_UT = xyz_UB + ( xyz_DUDtAdv + xyz_DUDtPhy ) * DelTimeX2
      xyz_VT = xyz_VB + ( xyz_DVDtAdv + xyz_DVDtPhy ) * DelTimeX2
      xyz_UA = &
        &   (   xyz_UT                                                  &
        &     + DelTimeX2 * xyz_CorPar                                  &
        &         * ( xyz_VT - VGeo + DelTimeX2 * xyz_CorPar * UGeo ) ) &
        & / ( 1.0_DP + ( DelTimeX2 * xyz_CorPar )**2 )
      xyz_VA = &
        &   (   xyz_VT                                                  &
        &     - DelTimeX2 * xyz_CorPar                                  &
        &         * ( xyz_UT - UGeo - DelTimeX2 * xyz_CorPar * VGeo ) ) &
        & / ( 1.0_DP + ( DelTimeX2 * xyz_CorPar )**2 )
    end if

!!$    xyz_TempA  = xyz_TempB  + xyz_DTempDtPhy  * 2.0d0 * DelTime
    xyz_TempA = xyz_PotTempB &
      & + ( xyz_DPotTempDtAdv + xyz_DTempDtPhy / xyz_Exner ) * DelTimeX2
    xyz_TempA = xyz_TempA * xyz_Exner

    xyzf_QMixA = xyzf_QMixB &
      & + ( xyzf_DQMixDtAdv + xyzf_DQMixDtPhy ) * DelTimeX2
    xyzf_QMixA = max( xyzf_QMixA, 0.0_DP )


  end subroutine DynamicsPhysicsOnly

  !-------------------------------------------------------------------

  subroutine DynamicsPhysicsOnlyInit

    ! モジュール引用 ; USE statements
    !

    ! 物理定数設定
    ! Physical constants settings
    !
    use constants, only: &
      & Omega
                              ! $ \Omega $ [s-1].
                              ! 回転角速度.
                              ! Angular velocity

    ! 座標データ設定
    ! Axes data settings
    !
    use axesset, only : y_Lat

    ! NAMELIST ファイル入力に関するユーティリティ
    ! Utilities for NAMELIST file input
    !
    use namelist_util, only: namelist_filename, NmlutilMsg, NmlutilAryValid

    ! ファイル入出力補助
    ! File I/O support
    !
    use dc_iounit, only: FileOpen

    ! 1 次元計算用力学過程ユーティリティモジュール
    ! Utility module for dynamics for 1-D calculation
    !
    use dynamics_1d_utils, only : Dynamics1DUtilsInit


    ! 宣言文 ; Declaration statements
    !

    integer:: unit_nml        ! NAMELIST ファイルオープン用装置番号. 
                              ! Unit number for NAMELIST file open
    integer:: iostat_nml      ! NAMELIST 読み込み時の IOSTAT. 
                              ! IOSTAT of NAMELIST read

!!$    integer:: i               ! 経度方向に回る DO ループ用作業変数
!!$                              ! Work variables for DO loop in longitude
    integer:: j               ! 緯度方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in latitude
    integer:: k


    ! NAMELIST 変数群
    ! NAMELIST group name
    !
    namelist /dynamics_physicsonly_nml/ &
      & FlagDynExp,                     &
      & UGeo, VGeo,                     &
      & WMagnitude, WHeight, WTimeInit
          !
          ! デフォルト値については初期化手続 "surface_flux_bulk#SurfFluxInit" 
          ! のソースコードを参照のこと. 
          !
          ! Refer to source codes in the initialization procedure
          ! "surface_flux_bulk#SurfFluxInit" for the default values. 
          !

    ! 実行文 ; Executable statement
    !


    if ( dynamics_physicsonly_inited ) return


    ! デフォルト値の設定
    ! Default values settings
    !
    FlagDynExp = .false.

    UGeo = 0.0_DP
    VGeo = 0.0_DP

    WMagnitude = 0.0_DP
    WHeight    = 0.0_DP
!!$    WMagnitude = - 0.005_DP
!!$    WHeight    =   1000.0_DP
    WTimeInit  = 0.0_DP

    ! NAMELIST の読み込み
    ! NAMELIST is input
    !
    if ( trim(namelist_filename) /= '' ) then
      call FileOpen( unit_nml, &          ! (out)
        & namelist_filename, mode = 'r' ) ! (in)

      rewind( unit_nml )
      read( unit_nml, &                      ! (in)
        & nml = dynamics_physicsonly_nml, &  ! (out)
        & iostat = iostat_nml )              ! (out)
      close( unit_nml )

      call NmlutilMsg( iostat_nml, module_name ) ! (in)
    end if


    allocate( xyz_CorPar(0:imax-1, 1:jmax, 1:kmax) )

    do k = 1, kmax
      do j = 1, jmax
        xyz_CorPar(:,j,k) = 2.0_DP * Omega * sin( y_Lat(j) )
      end do
    end do

    ! 1 次元計算用力学過程ユーティリティモジュール
    ! Utility module for dynamics for 1-D calculation
    !
    call Dynamics1DUtilsInit


!!$    ! ヒストリデータ出力のためのへの変数登録
!!$    ! Register of variables for history data output
!!$    !
!!$    call HistoryAutoAddVariable( 'TauX', &
!!$      & (/ 'lon ', 'lat ', 'time' /), &
!!$      & 'surface stress(x)  ', 'N m-2' )
!!$
!!$    call HistoryAutoAddVariable( 'TauY', &
!!$      & (/ 'lon ', 'lat ', 'time' /), &
!!$      & 'surface stress(y)  ', 'N m-2' )
!!$
!!$    call HistoryAutoAddVariable( 'Sens', &
!!$      & (/ 'lon ', 'lat ', 'time' /), &
!!$      & 'sensible heat flux', 'W m-2' )
!!$
!!$    call HistoryAutoAddVariable( 'EVap', &
!!$      & (/ 'lon ', 'lat ', 'time' /), &
!!$      & 'latent heat flux  ', 'W m-2' )

    ! 印字 ; Print
    !
    call MessageNotify( 'M', module_name, '----- Initialization Messages -----' )
    call MessageNotify( 'M', module_name, '  FlagDynExp = %b', l = (/ FlagDynExp /) )
    call MessageNotify( 'M', module_name, '  UGeo = %f', d = (/ UGeo /) )
    call MessageNotify( 'M', module_name, '  VGeo = %f', d = (/ VGeo /) )
    call MessageNotify( 'M', module_name, '  WMagnitude = %f', d = (/ WMagnitude /) )
    call MessageNotify( 'M', module_name, '  WHeight    = %f', d = (/ WHeight    /) )
    call MessageNotify( 'M', module_name, '  WTimeInit  = %f', d = (/ WTimeInit  /) )
    call MessageNotify( 'M', module_name, '-- version = %c', c1 = trim(version) )

    dynamics_physicsonly_inited = .true.

  end subroutine DynamicsPhysicsOnlyInit

  !--------------------------------------------------------------------------------------

end module dynamics_physicsonly
