!= 雪, 氷の割合
!
!= snow/ice fraction
!
! Authors::   Yoshiyuki O. Takahashi
! Version::   $Id: snowice_frac.f90,v 1.1 2015/01/29 12:12:21 yot Exp $
! Tag Name::  $Name:  $
! Copyright:: Copyright (C) GFD Dennou Club, 2008-2009. All rights reserved.
! License::   See COPYRIGHT[link:../../../COPYRIGHT]
!

module snowice_frac

  ! モジュール引用 ; USE statements
  !

  ! 種別型パラメタ
  ! Kind type parameter
  !
  use dc_types, only: DP, &      ! 倍精度実数型. Double precision.
    &                 STRING     ! 文字列.       Strings.

  ! メッセージ出力
  ! Message output
  !
  use dc_message, only: MessageNotify

  ! 格子点設定
  ! Grid points settings

  ! 宣言文 ; Declaration statements
  !
  implicit none
  private

  ! 公開手続き
  ! Public procedure
  !
  public :: SeaIceAboveThreshold
  public :: CalcSnowFrac
  public :: CalcSfcLiqSolFrac
  public :: CalcSeaIceFrac
  public :: CalcMajCompIceFrac
  public :: SnowIceFracInit

  ! 公開変数
  ! Public variables
  !

  ! 非公開変数
  ! Private variables
  !
  real(DP), save :: NOrd

  logical, save :: snowice_frac_inited = .false.
                              ! 初期設定フラグ.
                              ! Initialization flag

  character(*), parameter:: module_name = 'snowice_frac'
                              ! モジュールの名称.
                              ! Module name
  character(*), parameter:: version = &
    & '$Name:  $' // &
    & '$Id: snowice_frac.f90,v 1.1 2015/01/29 12:12:21 yot Exp $'
                              ! モジュールのバージョン
                              ! Module version

contains

  !--------------------------------------------------------------------------------------

  function SeaIceAboveThreshold( SeaIceConc ) result( FlagExist )

    ! 雪と海氷の定数の設定
    ! Setting constants of snow and sea ice
    !
    use constants_snowseaice, only: &
      & SeaIceThreshold

    real(DP), intent(in) :: SeaIceConc

    logical :: FlagExist


    if ( SeaIceConc > SeaIceThreshold ) then
      FlagExist = .true.
    else
      FlagExist = .false.
    end if

  end function SeaIceAboveThreshold

  !--------------------------------------------------------------------------------------

  subroutine CalcSnowFrac(       &
!!$    & xy_FlagLand, xy_SurfSnow,  & ! (in )
    & xy_SurfSnow,               & ! (in )
    & xy_SnowFrac                & ! (out)
    & )

    ! 格子点設定
    ! Grid points settings
    !
    use gridset, only: imax, & ! 経度格子点数.
                               ! Number of grid points in longitude
      &                jmax, & ! 緯度格子点数.
                               ! Number of grid points in latitude
      &                kmax    ! 鉛直層数.
                               ! Number of vertical level

    ! 雪と海氷の定数の設定
    ! Setting constants of snow and sea ice
    !
    use constants_snowseaice, only:            &
      & SnowThresholdForAlbedo

!!$    logical , intent(in ) :: xy_FlagLand( 0:imax-1, 1:jmax )
    real(DP), intent(in ) :: xy_SurfSnow( 0:imax-1, 1:jmax )
    real(DP), intent(out) :: xy_SnowFrac( 0:imax-1, 1:jmax )


    ! 作業変数
    ! Work variables
    !
    integer:: i               ! 経度方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in longitude
    integer:: j               ! 緯度方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in latitude

    ! 初期化確認
    ! Initialization check
    !
    if ( .not. snowice_frac_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    if ( SnowThresholdForAlbedo > 0.0_DP ) then
      do j = 1, jmax
        do i = 0, imax-1

          xy_SnowFrac(i,j) = &
            & ( ( xy_SurfSnow(i,j) - 0.0_DP ) / ( SnowThresholdForAlbedo - 0.0_DP ) )**NOrd
          xy_SnowFrac(i,j) = max( min( xy_SnowFrac(i,j), 1.0_DP ), 0.0_DP )

!!$        if ( xy_FlagLand(i,j) ) then
!!$          xy_SnowFrac(i,j) = &
!!$            & ( ( xy_SurfSnow(i,j) - 0.0_DP ) / ( SnowThresholdForAlbedo - 0.0_DP ) )**NOrd
!!$          xy_SnowFrac(i,j) = max( min( xy_SnowFrac(i,j), 1.0_DP ), 0.0_DP )
!!$        else
!!$          xy_SnowFrac(i,j) = 0.0_DP
!!$        end if

        end do
      end do
    else
      xy_SnowFrac = 1.0_DP
    end if


  end subroutine CalcSnowFrac

  !--------------------------------------------------------------------------------------

  subroutine CalcSfcLiqSolFrac(       &
!!$    & xy_FlagLand, xy_SurfSnow,  & ! (in )
    & xy_SurfSnow,               & ! (in )
    & xy_SnowFrac                & ! (out)
    & )

    ! 格子点設定
    ! Grid points settings
    !
    use gridset, only: imax, & ! 経度格子点数.
                               ! Number of grid points in longitude
      &                jmax, & ! 緯度格子点数.
                               ! Number of grid points in latitude
      &                kmax    ! 鉛直層数.
                               ! Number of vertical level

    ! 雪と海氷の定数の設定
    ! Setting constants of snow and sea ice
    !
    use constants_snowseaice, only:            &
      & SnowThresholdForFlux

!!$    logical , intent(in ) :: xy_FlagLand( 0:imax-1, 1:jmax )
    real(DP), intent(in ) :: xy_SurfSnow( 0:imax-1, 1:jmax )
    real(DP), intent(out) :: xy_SnowFrac( 0:imax-1, 1:jmax )


    ! 作業変数
    ! Work variables
    !
    integer:: i               ! 経度方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in longitude
    integer:: j               ! 緯度方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in latitude

    ! 初期化確認
    ! Initialization check
    !
    if ( .not. snowice_frac_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    if ( SnowThresholdForFlux > 0.0_DP ) then
      do j = 1, jmax
        do i = 0, imax-1

!!$        if ( xy_FlagLand(i,j) ) then
          xy_SnowFrac(i,j) = &
            & ( ( xy_SurfSnow(i,j) - 0.0_DP ) / ( SnowThresholdForFlux - 0.0_DP ) )**NOrd
          xy_SnowFrac(i,j) = max( min( xy_SnowFrac(i,j), 1.0_DP ), 0.0_DP )
!!$        else
!!$          xy_SnowFrac(i,j) = 0.0_DP
!!$        end if

        end do
      end do

    else
      xy_SnowFrac = 1.0_DP
    end if


  end subroutine CalcSfcLiqSolFrac

  !--------------------------------------------------------------------------------------

  subroutine CalcSeaIceFrac(       &
!!$    & xy_FlagOcean, xy_SeaIceConc, & ! (in )
    & xy_SeaIceConc,               & ! (in )
    & xy_SeaIceFrac                & ! (out)
    & )

    ! 格子点設定
    ! Grid points settings
    !
    use gridset, only: imax, & ! 経度格子点数.
                               ! Number of grid points in longitude
      &                jmax, & ! 緯度格子点数.
                               ! Number of grid points in latitude
      &                kmax    ! 鉛直層数.
                               ! Number of vertical level

    ! 雪と海氷の定数の設定
    ! Setting constants of snow and sea ice
    !
    use constants_snowseaice, only:  &
      & SeaIceThreshold


!!$    logical , intent(in ) :: xy_FlagOcean ( 0:imax-1, 1:jmax )
    real(DP), intent(in ) :: xy_SeaIceConc( 0:imax-1, 1:jmax )
    real(DP), intent(out) :: xy_SeaIceFrac( 0:imax-1, 1:jmax )


    ! 作業変数
    ! Work variables
    !
    integer:: i               ! 経度方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in longitude
    integer:: j               ! 緯度方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in latitude

    ! 初期化確認
    ! Initialization check
    !
    if ( .not. snowice_frac_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    do j = 1, jmax
      do i = 0, imax-1

!!$        if ( ( xy_FlagOcean(i,j) ) .and. &
!!$          &  ( xy_SeaIceConc(i,j) > SeaIceThreshold ) ) then
        if ( xy_SeaIceConc(i,j) > SeaIceThreshold ) then
          xy_SeaIceFrac(i,j) = 1.0_DP
        else
          xy_SeaIceFrac(i,j) = 0.0_DP
        end if

      end do
    end do


  end subroutine CalcSeaIceFrac

  !--------------------------------------------------------------------------------------

  subroutine CalcMajCompIceFrac(      &
!!$    & xy_FlagLand, xy_SurfMajCompIce, & ! (in )
    & xy_SurfMajCompIce,              & ! (in )
    & xy_MajCompIceFrac               & ! (out)
    )

    ! 格子点設定
    ! Grid points settings
    !
    use gridset, only: imax, & ! 経度格子点数.
                               ! Number of grid points in longitude
      &                jmax, & ! 緯度格子点数.
                               ! Number of grid points in latitude
      &                kmax    ! 鉛直層数.
                               ! Number of vertical level

    ! 雪と海氷の定数の設定
    ! Setting constants of snow and sea ice
    !
    use constants_snowseaice, only:            &
      & CO2IceThreshold


!!$    logical , intent(in ) :: xy_FlagLand      ( 0:imax-1, 1:jmax )
    real(DP), intent(in ) :: xy_SurfMajCompIce( 0:imax-1, 1:jmax )
    real(DP), intent(out) :: xy_MajCompIceFrac( 0:imax-1, 1:jmax )


    ! 作業変数
    ! Work variables
    !
    real(DP):: MajCompIceThreshold
    integer:: i               ! 経度方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in longitude
    integer:: j               ! 緯度方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in latitude

    ! 初期化確認
    ! Initialization check
    !
    if ( .not. snowice_frac_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if



    MajCompIceThreshold = CO2IceThreshold
    do j = 1, jmax
      do i = 0, imax-1

!!$        if ( xy_FlagLand(i,j) ) then
          xy_MajCompIceFrac(i,j) =                  &
            &   ( xy_SurfMajCompIce(i,j) - 0.0_DP ) &
            & / ( MajCompIceThreshold    - 0.0_DP )
          xy_MajCompIceFrac(i,j) = max( min( xy_MajCompIceFrac(i,j), 1.0_DP ), 0.0_DP )
!!$        else
!!$          xy_MajCompIceFrac(i,j) = 0.0_DP
!!$        end if

      end do
    end do


  end subroutine CalcMajCompIceFrac

  !--------------------------------------------------------------------------------------

  subroutine SnowIceFracInit

    ! モジュール引用 ; USE statements
    !

    ! 種別型パラメタ
    ! Kind type parameter
    !
    use dc_types, only: STDOUT ! 標準出力の装置番号. Unit number of standard output

    ! ファイル入出力補助
    ! File I/O support
    !
    use dc_iounit, only: FileOpen

    ! NAMELIST ファイル入力に関するユーティリティ
    ! Utilities for NAMELIST file input
    !
    use namelist_util, only: namelist_filename, NmlutilMsg


    ! 作業変数
    ! Work variables
    !
    integer:: unit_nml        ! NAMELIST ファイルオープン用装置番号.
                              ! Unit number for NAMELIST file open
    integer:: iostat_nml      ! NAMELIST 読み込み時の IOSTAT.
                              ! IOSTAT of NAMELIST read


    ! NAMELIST 変数群
    ! NAMELIST group name
    !
    namelist /snowice_frac_nml/ &
      & NOrd

    ! 実行文 ; Executable statement
    !


    if ( snowice_frac_inited ) return


    ! デフォルト値の設定
    ! Default values settings
    !
    NOrd = 2.0_DP


    ! NAMELIST の読み込み
    ! NAMELIST is input
    !
    if ( trim(namelist_filename) /= '' ) then
      call FileOpen( unit_nml, &          ! (out)
        & namelist_filename, mode = 'r' ) ! (in)

      rewind( unit_nml )
      read( unit_nml,                  &  ! (in)
        & nml = snowice_frac_nml,      &  ! (out)
        & iostat = iostat_nml          &  ! (out)
        & )
      close( unit_nml )

      call NmlutilMsg( iostat_nml, module_name ) ! (in)
      if ( iostat_nml == 0 ) write( STDOUT, nml = snowice_frac_nml )
    end if

    ! 印字 ; Print
    !
    call MessageNotify( 'M', module_name, '----- Initialization Messages -----' )
    call MessageNotify( 'M', module_name, 'NOrd = %f', d = (/NOrd/) )
    call MessageNotify( 'M', module_name, '-- version = %c', c1 = trim(version) )


    snowice_frac_inited = .true.

  end subroutine SnowIceFracInit

  !--------------------------------------------------------------------------------------

end module snowice_frac
