!= 積分と平均の操作
!
!= Operation for integral and average
!
! Authors::   Yoshiyuki O. Takahashi, Yasuhiro MORIKAWA
! Version::   $Id: intavr_operate.F90,v 1.6 2014/05/07 09:39:23 murashin Exp $ 
! Tag Name::  $Name:  $
! Copyright:: Copyright (C) GFD Dennou Club, 2008. All rights reserved.
! License::   See COPYRIGHT[link:../../../COPYRIGHT]
!

module intavr_operate
  !
  != 積分と平均の操作
  !
  != Operation for integral and average
  !
  ! <b>Note that Japanese and English are described in parallel.</b>
  !
  ! 積分で用いる座標重みを考慮した積分や平均操作のための関数を提供します. 
  ! {SPMODEL ライブラリ}[http://www.gfd-dennou.org/library/spmodel]
  ! の w_integral_module.f90 を参考に作成しました. 
  !
  ! Functions for integral or average operation with weight for integration
  ! are provided
  ! This program is created referring to 
  ! "w_integral_module.f90" in 
  ! {SPMODEL library}[http://www.gfd-dennou.org/library/spmodel]
  !
  !== Procedures List
  !
  ! IntLonLat_xy           :: 緯度経度積分
!!$  ! y_IntLon_xy, IntLon_x  :: 経度積分
!!$  ! ya_IntLon_xya          :: 経度積分 (多層用)
!!$  ! x_IntLat_xy, IntLat_y  :: 緯度積分
!!$  ! xa_IntLat_xya          :: 緯度積分 (多層用)
!!$  ! AvrLonLat_xy           :: 緯度経度平均
!!$  ! y_AvrLon_xy, AvrLon_x  :: 経度平均
!!$  ! ya_AvrLon_xya          :: 経度平均 (多層用)
!!$  ! x_AvrLat_xy, AvrLat_y  :: 緯度平均
!!$  ! xa_AvrLat_xya          :: 緯度平均 (多層用)
  ! ---------------------  :: ---------------------
  ! y_IntLon_xy, IntLon_x  :: Meridional integral
!!$  ! ya_IntLon_xya          :: Meridional integral (for multi layer)
!!$  ! x_IntLat_xy, IntLat_y  :: Zonal integral
!!$  ! xa_IntLat_xya          :: Zonal integral (for multi layer)
!!$  ! AvrLonLat_xy           :: Zonal and meridional average
!!$  ! y_AvrLon_xy, AvrLon_x  :: Meridional average
!!$  ! ya_AvrLon_xya          :: Meridional average (for multi layer)
!!$  ! x_AvrLat_xy, AvrLat_y  :: Zonal average
!!$  ! xa_AvrLat_xya          :: Zonal average (for multi layer)
  !
  !--
  !== NAMELIST
  !
  ! NAMELIST#intavr_operate_nml
  !++

  ! モジュール引用 ; USE statements
  !

  ! 座標データ設定
  ! Axes data settings
  !
  use axesset, only: &
    & x_Lon_Weight, &
                              ! $ \Delta \lambda $ [rad.] . 
                              ! 経度座標重み. 
                              ! Weight of longitude
    & y_Lat_Weight
                              ! $ \Delta \varphi $ [rad.] . 
                              ! 緯度座標重み. 
                              ! Weight of latitude

  ! 格子点設定
  ! Grid points settings
  !
  use gridset, only: imax, & ! 経度格子点数. 
                             ! Number of grid points in longitude
    &                jmax, & ! 緯度格子点数. 
                             ! Number of grid points in latitude
    &                kmax    ! 鉛直層数. 
                             ! Number of vertical level

  ! 種別型パラメタ
  ! Kind type parameter
  !
  use dc_types, only: DP, &      ! 倍精度実数型. Double precision. 
    &                 STRING     ! 文字列.       Strings. 

  ! メッセージ出力
  ! Message output
  !
  use dc_message, only: MessageNotify

  ! 宣言文 ; Declaration statements
  !
  implicit none
  private

  ! 公開手続き
  ! Public procedure
  !
  public:: IntLonLat_xy
  public:: a_IntLonLat_xya
!!$  public:: y_IntLon_xy, ya_IntLon_xya, IntLon_x
!!$  public:: x_IntLat_xy, xa_IntLat_xya, IntLat_y
!!$  public:: IntLat_y
!!$  public:: AvrLonLat_xy
!!$  public:: y_AvrLon_xy, ya_AvrLon_xya, AvrLon_x
!!$  public:: x_AvrLat_xy, xa_AvrLat_xya, AvrLat_y

  ! 公開変数
  ! Public variables
  !
  logical, save, public:: intavr_operate_inited = .false.
                              ! 初期設定フラグ. 
                              ! Initialization flag


  ! 非公開変数
  ! Private variables
  !

  character(*), parameter:: module_name = 'intavr_operate'
                              ! モジュールの名称. 
                              ! Module name
  character(*), parameter:: version = &
    & '$Name:  $' // &
    & '$Id: intavr_operate.F90,v 1.6 2014/05/07 09:39:23 murashin Exp $'
                              ! モジュールのバージョン
                              ! Module version

  ! INTERFACE 文 ; INTERFACE statements
  !

contains

  !-------------------------------------------------------------------

  function a_IntLonLat_xya( xya_Data )
    !
    ! 2 次元緯度経度格子点データの全領域積分(1 層用). 
    !
    ! 実際には格子点データ各点毎に x_Lon_Weight, y_Lat_Weight を掛けた
    ! 総和を計算している. 
    !
    ! Global integration of 2-dimensional (latitude and longitude)
    ! grid data.
    !
    ! Practically, the sum total of grid data is calculated
    ! by multiplying in each grid "x_Lon_Weight" and "y_Lat_Weight".
    !

    real(DP), intent(in) :: xya_Data       (:,:,:)
    real(DP)             :: a_IntLonLat_xya(size(xya_Data,3))


    ! 実行文 ; Executable statement
    !


    a_IntLonLat_xya = a_IntLat_ya( ya_IntLon_xya( xya_Data ) )


  end function a_IntLonLat_xya

  !-------------------------------------------------------------------

  function IntLonLat_xy( xy_Data )
    !
    ! 2 次元緯度経度格子点データの全領域積分(1 層用). 
    !
    ! 実際には格子点データ各点毎に x_Lon_Weight, y_Lat_Weight を掛けた
    ! 総和を計算している. 
    !
    ! Global integration of 2-dimensional (latitude and longitude)
    ! grid data.
    !
    ! Practically, the sum total of grid data is calculated
    ! by multiplying in each grid "x_Lon_Weight" and "y_Lat_Weight".
    !

    real(DP), intent(in) :: xy_Data (0:imax-1, 1:jmax)
    real(DP)             :: IntLonLat_xy

    ! 実行文 ; Executable statement
    !

    IntLonLat_xy = IntLat_y( y_IntLon_xy( xy_Data ) )

  end function IntLonLat_xy

  !-------------------------------------------------------------------

  function ya_IntLon_xya( xya_Data )
    !
    ! 2 次元緯度経度格子点データの経度方向積分(1 層用). 
    !
    ! 実際には格子点データ各点毎に x_Lon_Weight を掛けた
    ! 総和を計算している. 
    !
    ! Zonal integration of 2-dimensional (latitude and longitude)
    ! grid data.
    !
    ! Practically, the sum total of grid data is calculated
    ! by multiplying in each grid "x_Lon_Weight".
    !

    real(DP), intent(in) :: xya_Data     (:,:,:)
    real(DP)             :: ya_IntLon_xya(size(xya_Data,2), size(xya_Data,3))

    ! 作業変数
    ! Work variables
    !
    integer:: lmax
    integer:: i               ! 経度方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in longitudinal direction
    integer:: j               ! 緯度方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in latitudinal direction
    integer:: l


    ! 実行文 ; Executable statement
    !
    lmax = size( xya_Data, 3 )

    ya_IntLon_xya = 0.0_DP
    do l = 1, lmax
      do j = 1, jmax
        do i = 0, imax - 1
!!$          ya_IntLon_xya(j,l) = ya_IntLon_xya(j,l) + xya_Data(i,j,l) * x_Lon_Weight(i)
          ya_IntLon_xya(j,l) = ya_IntLon_xya(j,l) + xya_Data(i+1,j,l) * x_Lon_Weight(i)
        end do
      end do
    end do


  end function ya_IntLon_xya

  !-------------------------------------------------------------------

  function y_IntLon_xy( xy_Data )
    !
    ! 2 次元緯度経度格子点データの経度方向積分(1 層用). 
    !
    ! 実際には格子点データ各点毎に x_Lon_Weight を掛けた
    ! 総和を計算している. 
    !
    ! Zonal integration of 2-dimensional (latitude and longitude)
    ! grid data.
    !
    ! Practically, the sum total of grid data is calculated
    ! by multiplying in each grid "x_Lon_Weight".
    !

    real(DP), intent(in) :: xy_Data (0:imax-1, 1:jmax)
    real(DP)             :: y_IntLon_xy (1:jmax)

    ! 作業変数
    ! Work variables
    !
    integer:: i               ! 経度方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in longitudinal direction
    integer:: j               ! 緯度方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in latitudinal direction

    ! 実行文 ; Executable statement
    !

    y_IntLon_xy = 0.0_DP
    do j = 1, jmax
      do i = 0, imax - 1
        y_IntLon_xy(j) = y_IntLon_xy(j) + xy_Data (i,j) * x_Lon_Weight(i)
      end do
    end do

  end function y_IntLon_xy

#ifdef LIB_MPI
  !-------------------------------------------------------------------

  function a_IntLat_ya( ya_Data )
    !
    ! 1 次元緯度経度格子点データの全領域平均(1 層用). 
    !
    ! Global mean of 2-dimensional (latitude and longitude)
    ! grid data.
    !

    ! MPI
    !
    use mpi_wrapper, only: nprocs, myrank, &
      & MPIWrapperISend, &
      & MPIWrapperIRecv, &
      & MPIWrapperWait

    ! 格子点数・最大波数設定
    ! Number of grid points and maximum truncated wavenumber settings
    !
    use gridset, only: a_jmax, jmax_max

    real(DP), intent(in) :: ya_Data (:,:)
    real(DP)             :: a_IntLat_ya(size(ya_Data,2))


    ! Local variable
    !
    integer               :: lmax
    real(DP), allocatable :: aa_SendBuf (:,:)
    real(DP), allocatable :: aaa_RecvBuf(:,:,:)
    integer , allocatable :: a_iReqSend (:)
    integer , allocatable :: a_iReqRecv (:)

    integer               :: j
    integer               :: l
    integer               :: n


    ! 実行文 ; Executable statement
    !
    lmax = size(ya_Data,2)


    allocate( aa_SendBuf (1:jmax_max, 1:lmax) )
    allocate( aaa_RecvBuf(1:jmax_max, 1:lmax, 0:nprocs-1) )
    allocate( a_iReqSend(0:nprocs-1) )
    allocate( a_iReqRecv(0:nprocs-1) )

    do l = 1, lmax
      do j = 1, jmax
        aa_SendBuf(j,l) = ya_Data(j,l) * y_Lat_Weight(j)
      end do
      do j = jmax+1, jmax_max
        aa_SendBuf(j,l) = -1.0_DP
      end do
    end do

    do n = 0, nprocs-1
      if ( n == myrank ) then
        aaa_RecvBuf(:,:,n) = aa_SendBuf
      else
        call MPIWrapperISend( n, jmax_max, lmax, aa_SendBuf        , a_iReqSend(n) )
        call MPIWrapperIRecv( n, jmax_max, lmax, aaa_RecvBuf(:,:,n), a_iReqRecv(n) )
      end if
    end do
    do n = 0, nprocs-1
      if ( n == myrank ) cycle
      call MPIWrapperWait( a_iReqSend(n) )
      call MPIWrapperWait( a_iReqRecv(n) )
    end do


    a_IntLat_ya = 0.0_DP
    do n = nprocs-1, 0, -1
      do l = 1, lmax
        do j = 1, a_jmax(n) / 2
          a_IntLat_ya(l) = a_IntLat_ya(l) + aaa_RecvBuf(j,l,n)
        end do
      end do
    end do
    do n = 0, nprocs-1
      do l = 1, lmax
        do j = a_jmax(n) / 2 + 1, a_jmax(n)
          a_IntLat_ya(l) = a_IntLat_ya(l) + aaa_RecvBuf(j,l,n)
        end do
      end do
    end do


    deallocate( aa_SendBuf  )
    deallocate( aaa_RecvBuf )
    deallocate( a_iReqSend  )
    deallocate( a_iReqRecv  )


  end function a_IntLat_ya

  !-------------------------------------------------------------------

  function IntLat_y( y_Data )
    !
    ! 1 次元緯度経度格子点データの全領域平均(1 層用). 
    !
    ! Global mean of 2-dimensional (latitude and longitude)
    ! grid data.
    !

    ! MPI
    !
    use mpi_wrapper, only: nprocs, myrank, &
      & MPIWrapperISend, &
      & MPIWrapperIRecv, &
      & MPIWrapperWait

    ! 格子点数・最大波数設定
    ! Number of grid points and maximum truncated wavenumber settings
    !
    use gridset, only: a_jmax, jmax_max

    real(DP), intent(in) :: y_Data (1:jmax)
    real(DP)             :: IntLat_y


    ! Local variable
    !
    real(DP), allocatable :: a_SendBuf (:)
    real(DP), allocatable :: aa_RecvBuf(:,:)
    integer , allocatable :: a_iReqSend(:)
    integer , allocatable :: a_iReqRecv(:)

    integer               :: j
    integer               :: n


    ! 実行文 ; Executable statement
    !


    allocate( a_SendBuf (1:jmax_max) )
    allocate( aa_RecvBuf(1:jmax_max,0:nprocs-1) )
    allocate( a_iReqSend(0:nprocs-1) )
    allocate( a_iReqRecv(0:nprocs-1) )

    do j = 1, jmax
      a_SendBuf(j) = y_Data(j) * y_Lat_Weight(j)
    end do
    do j = jmax+1, jmax_max
      a_SendBuf(j) = -1.0_DP
    end do

    do n = 0, nprocs-1
      if ( n == myrank ) then
        aa_RecvBuf(:,n) = a_SendBuf
      else
        call MPIWrapperISend( n, jmax_max, a_SendBuf      , a_iReqSend(n) )
        call MPIWrapperIRecv( n, jmax_max, aa_RecvBuf(:,n), a_iReqRecv(n) )
      end if
    end do
    do n = 0, nprocs-1
      if ( n == myrank ) cycle
      call MPIWrapperWait( a_iReqSend(n) )
      call MPIWrapperWait( a_iReqRecv(n) )
    end do


    IntLat_y = 0.0d0
    do n = nprocs-1, 0, -1
      do j = 1, a_jmax(n) / 2
        IntLat_y = IntLat_y + aa_RecvBuf(j,n)
      end do
    end do
    do n = 0, nprocs-1
      do j = a_jmax(n) / 2 + 1, a_jmax(n)
        IntLat_y = IntLat_y + aa_RecvBuf(j,n)
      end do
    end do


    deallocate( a_SendBuf  )
    deallocate( aa_RecvBuf )
    deallocate( a_iReqSend )
    deallocate( a_iReqRecv )


  end function IntLat_y

#else
  !-------------------------------------------------------------------

  function a_IntLat_ya( ya_Data )
    !
    ! 1 次元緯度格子点データの緯度方向積分(1 層用). 
    !
    ! 実際には格子点データ各点毎に y_Lat_Weight を掛けた
    ! 総和を計算している. 
    !
    ! Meridonal integration of 1-dimensional (latitude)
    ! grid data.
    !
    ! Practically, the sum total of grid data is calculated
    ! by multiplying in each grid "y_Lat_Weight".
    !
    real(DP), intent(in) :: ya_Data    (:,:)
    real(DP)             :: a_IntLat_ya(size(ya_Data,2))

    ! 作業変数
    ! Work variables
    !
    integer :: lmax
    integer :: l


    ! 実行文 ; Executable statement
    !
    lmax = size(ya_Data,2)

    do l = 1, lmax
      a_IntLat_ya(l) = sum( ya_Data(:,l) * y_Lat_Weight )
    end do


  end function a_IntLat_ya

  !-------------------------------------------------------------------

  function IntLat_y( y_Data )
    !
    ! 1 次元緯度格子点データの緯度方向積分(1 層用). 
    !
    ! 実際には格子点データ各点毎に y_Lat_Weight を掛けた
    ! 総和を計算している. 
    !
    ! Meridonal integration of 1-dimensional (latitude)
    ! grid data.
    !
    ! Practically, the sum total of grid data is calculated
    ! by multiplying in each grid "y_Lat_Weight".
    !
    real(DP), intent(in) :: y_Data (1:jmax)
    real(DP)             :: IntLat_y

    ! 作業変数
    ! Work variables
    !

    ! 実行文 ; Executable statement
    !
    IntLat_y = sum( y_Data * y_Lat_Weight )

  end function IntLat_y

#endif
  !-------------------------------------------------------------------
!!$
!!$  function AvrLonLat_xy( xy_Data )
!!$    !
!!$    ! 2 次元緯度経度格子点データの全領域平均(1 層用). 
!!$    !
!!$    ! Global mean of 2-dimensional (latitude and longitude)
!!$    ! grid data.
!!$    !
!!$
!!$    real(DP), intent(in):: xy_Data (0:imax-1, 1:jmax)
!!$
!!$
!!$    ! Local variable
!!$    !
!!$    real(DP)              :: AvrLonLat_xy
!!$
!!$
!!$    ! 実行文 ; Executable statement
!!$    !
!!$
!!$    ! Old one to be deleted
!!$!    AvrLonLat_xy = AvrLon_x( x_AvrLat_xy( xy_Data ) )
!!$
!!$    AvrLonLat_xy = AvrLat_y( y_AvrLon_xy( xy_Data ) )
!!$
!!$
!!$  end function AvrLonLat_xy
!!$
!!$  !-------------------------------------------------------------------
!!$
!!$  function x_IntLat_xy( xy_Data )
!!$    !
!!$    ! 2 次元緯度経度格子点データの緯度方向積分(1 層用). 
!!$    !
!!$    ! 実際には格子点データ各点毎に y_Lat_Weight を掛けた
!!$    ! 総和を計算している. 
!!$    !
!!$    ! Meridional integration of 2-dimensional (latitude and longitude)
!!$    ! grid data.
!!$    !
!!$    ! Practically, the sum total of grid data is calculated
!!$    ! by multiplying in each grid "y_Lat_Weight".
!!$    !
!!$    real(DP), intent(in):: xy_Data (0:imax-1, 1:jmax)
!!$    real(DP):: x_IntLat_xy (0:imax-1)
!!$
!!$    ! 作業変数
!!$    ! Work variables
!!$    !
!!$    integer:: j               ! 緯度方向に回る DO ループ用作業変数
!!$                              ! Work variables for DO loop in latitude
!!$
!!$    ! 実行文 ; Executable statement
!!$    !
!!$    x_IntLat_xy = 0.0_DP
!!$    do j = 1, jmax
!!$      x_IntLat_xy = x_IntLat_xy + xy_Data (:,j) * y_Lat_Weight(j)
!!$    enddo
!!$  end function x_IntLat_xy
!!$
!!$  !-------------------------------------------------------------------
!!$
!!$  function xa_IntLat_xya( xya_Data )
!!$    !
!!$    ! 3 次元緯度経度格子点データの緯度方向積分(多層用). 
!!$    !
!!$    ! 実際には格子点データ各点毎に y_Lat_Weight を掛けた
!!$    ! 総和を計算している. 
!!$    !
!!$    ! Meridional integration of 3-dimensional (latitude and longitude)
!!$    ! grid data. (for multi layer)
!!$    !
!!$    ! Practically, the sum total of grid data is calculated
!!$    ! by multiplying in each grid "y_Lat_Weight".
!!$    !
!!$    real(DP), intent(in):: xya_Data (0:imax-1, 1:jmax, 1:kmax)
!!$    real(DP):: xa_IntLat_xya (0:imax-1, 1:kmax)
!!$
!!$    ! 作業変数
!!$    ! Work variables
!!$    !
!!$    integer:: k               ! 鉛直方向に回る DO ループ用作業変数
!!$                              ! Work variables for DO loop in vertical direction
!!$
!!$    ! 実行文 ; Executable statement
!!$    !
!!$    do k = 1, kmax
!!$      xa_IntLat_xya(:,k) = x_IntLat_xy( xya_Data(:,:,k) )
!!$    end do
!!$  end function xa_IntLat_xya
!!$
!!$  !-------------------------------------------------------------------
!!$
!!$  function ya_IntLon_xya( xya_Data )
!!$    !
!!$    ! 3 次元緯度経度格子点データの経度方向積分(多層用). 
!!$    !
!!$    ! 実際には格子点データ各点毎に x_Lon_Weight を掛けた
!!$    ! 総和を計算している. 
!!$    !
!!$    ! Zonal integration of 3-dimensional (latitude and longitude)
!!$    ! grid data. (for multi layer)
!!$    !
!!$    ! Practically, the sum total of grid data is calculated
!!$    ! by multiplying in each grid "x_Lon_Weight".
!!$    !
!!$    real(DP), intent(in):: xya_Data (0:imax-1, 1:jmax, 1:kmax)
!!$    real(DP):: ya_IntLon_xya (1:jmax,1:kmax)
!!$
!!$    ! 作業変数
!!$    ! Work variables
!!$    !
!!$    integer:: k               ! 鉛直方向に回る DO ループ用作業変数
!!$                              ! Work variables for DO loop in vertical direction
!!$
!!$    ! 実行文 ; Executable statement
!!$    !
!!$    do k = 1, kmax
!!$      ya_IntLon_xya(:,k) = y_IntLon_xy( xya_Data(:,:,k) )
!!$    end do
!!$  end function ya_IntLon_xya
!!$
!!$  !-------------------------------------------------------------------
!!$
!!$  function IntLon_x( x_Data )
!!$    !
!!$    ! 1 次元経度格子点データの経度方向積分(1 層用). 
!!$    !
!!$    ! 実際には格子点データ各点毎に x_Lon_Weight を掛けた
!!$    ! 総和を計算している. 
!!$    !
!!$    ! Zonal integration of 1-dimensional (longitude)
!!$    ! grid data.
!!$    !
!!$    ! Practically, the sum total of grid data is calculated
!!$    ! by multiplying in each grid "x_Lon_Weight".
!!$    !
!!$    real(DP), intent(in):: x_Data (0:imax-1)
!!$    real(DP):: IntLon_x
!!$
!!$    ! 作業変数
!!$    ! Work variables
!!$    !
!!$
!!$    ! 実行文 ; Executable statement
!!$    !
!!$    IntLon_x = sum( x_Data * x_Lon_Weight )
!!$  end function IntLon_x
!!$
!!$  !-------------------------------------------------------------------
!!$
!!$  function x_AvrLat_xy( xy_Data )
!!$    !
!!$    ! 2 次元緯度経度格子点データの緯度方向平均(1 層用). 
!!$    !
!!$    ! 実際には格子点データ各点毎に y_Lat_Weight を掛けた
!!$    ! 総和を計算し, y_Lat_Weight の総和で割ることで
!!$    ! 平均している. 
!!$    !
!!$    ! Meridional mean of 2-dimensional (latitude and longitude)
!!$    ! grid data.
!!$    !
!!$    ! Practically, the mean grid data is calculated
!!$    ! by multiplying in each grid "y_Lat_Weight" 
!!$    ! and deviding by the sum total of "y_Lat_Weight".
!!$    !
!!$    real(DP), intent(in):: xy_Data (0:imax-1, 1:jmax)
!!$    real(DP):: x_AvrLat_xy (0:imax-1)
!!$
!!$    ! 作業変数
!!$    ! Work variables
!!$    !
!!$
!!$    ! 実行文 ; Executable statement
!!$    !
!!$    x_AvrLat_xy = x_IntLat_xy ( xy_Data ) / sum( y_Lat_Weight )
!!$  end function x_AvrLat_xy
!!$
!!$  !-------------------------------------------------------------------
!!$
!!$  function xa_AvrLat_xya( xya_Data )
!!$    !
!!$    ! 3 次元緯度経度格子点データの緯度方向平均(多層用). 
!!$    !
!!$    ! 実際には格子点データ各点毎に y_Lat_Weight を掛けた
!!$    ! 総和を計算し, y_Lat_Weight の総和で割ることで
!!$    ! 平均している. 
!!$    !
!!$    ! Meridional mean of 3-dimensional (latitude and longitude)
!!$    ! grid data. (for multi layer)
!!$    !
!!$    ! Practically, the mean grid data is calculated
!!$    ! by multiplying in each grid "y_Lat_Weight" 
!!$    ! and deviding by the sum total of "y_Lat_Weight".
!!$    !
!!$    real(DP), intent(in):: xya_Data (0:imax-1, 1:jmax, 1:kmax)
!!$    real(DP):: xa_AvrLat_xya (0:imax-1, 1:kmax)
!!$
!!$    ! 作業変数
!!$    ! Work variables
!!$    !
!!$    integer:: k               ! 鉛直方向に回る DO ループ用作業変数
!!$                              ! Work variables for DO loop in vertical direction
!!$
!!$    ! 実行文 ; Executable statement
!!$    !
!!$    do k = 1, kmax
!!$      xa_AvrLat_xya(:,k) = x_AvrLat_xy( xya_Data(:,:,k) )
!!$    end do
!!$  end function xa_AvrLat_xya
!!$
!!$  !-------------------------------------------------------------------
!!$
!!$  function y_AvrLon_xy( xy_Data )
!!$    !
!!$    ! 2 次元緯度経度格子点データの経度方向平均(1 層用). 
!!$    !
!!$    ! 実際には格子点データ各点毎に x_Lon_Weight を掛けた
!!$    ! 総和を計算し, x_Lon_Weight の総和で割ることで
!!$    ! 平均している. 
!!$    !
!!$    ! Zonal mean of 2-dimensional (latitude and longitude)
!!$    ! grid data.
!!$    !
!!$    ! Practically, the mean grid data is calculated
!!$    ! by multiplying in each grid "x_Lon_Weight" 
!!$    ! and deviding by the sum total of "x_Lon_Weight".
!!$    !
!!$    real(DP), intent(in):: xy_Data (0:imax-1, 1:jmax)
!!$    real(DP):: y_AvrLon_xy (1:jmax)
!!$
!!$    ! 作業変数
!!$    ! Work variables
!!$    !
!!$
!!$    ! 実行文 ; Executable statement
!!$    !
!!$    y_AvrLon_xy = y_IntLon_xy( xy_Data ) / sum( x_Lon_Weight )
!!$  end function y_AvrLon_xy
!!$
!!$  !-------------------------------------------------------------------
!!$
!!$  function ya_AvrLon_xya( xya_Data )
!!$    !
!!$    ! 3 次元緯度経度格子点データの経度方向平均(多層用). 
!!$    !
!!$    ! 実際には格子点データ各点毎に x_Lon_Weight を掛けた
!!$    ! 総和を計算し, x_Lon_Weight の総和で割ることで
!!$    ! 平均している. 
!!$    !
!!$    ! Zonal mean of 3-dimensional (latitude and longitude)
!!$    ! grid data. (for multi layer)
!!$    !
!!$    ! Practically, the mean grid data is calculated
!!$    ! by multiplying in each grid "x_Lon_Weight" 
!!$    ! and deviding by the sum total of "x_Lon_Weight".
!!$    !
!!$    real(DP), intent(in):: xya_Data (0:imax-1, 1:jmax, 1:kmax)
!!$    real(DP):: ya_AvrLon_xya (1:jmax, 1:kmax)
!!$
!!$    ! 作業変数
!!$    ! Work variables
!!$    !
!!$    integer:: k               ! 鉛直方向に回る DO ループ用作業変数
!!$                              ! Work variables for DO loop in vertical direction
!!$
!!$    ! 実行文 ; Executable statement
!!$    !
!!$    do k = 1, kmax
!!$      ya_AvrLon_xya(:,k) = y_AvrLon_xy( xya_Data(:,:,k) )
!!$    end do
!!$  end function ya_AvrLon_xya
!!$
!!$#ifdef LIB_MPI
!!$  !-------------------------------------------------------------------
!!$
!!$  function AvrLat_y( y_Data )
!!$    !
!!$    ! 1 次元緯度格子点データの緯度方向平均(1 層用). 
!!$    !
!!$    ! 実際には格子点データ各点毎に y_Lat_Weight を掛けた
!!$    ! 総和を計算し, y_Lat_Weight の総和で割ることで
!!$    ! 平均している. 
!!$    !
!!$    ! Meridonal mean of 1-dimensional (latitude)
!!$    ! grid data.
!!$    !
!!$    ! Practically, the sum total of grid data is calculated
!!$    ! by multiplying in each grid "y_Lat_Weight" 
!!$    ! and deviding by the sum total of "y_Lat_Weight".
!!$    !
!!$
!!$    ! MPI
!!$    !
!!$    use mpi_wrapper, only: nprocs, myrank, &
!!$      & MPIWrapperISend, &
!!$      & MPIWrapperIRecv, &
!!$      & MPIWrapperWait
!!$
!!$    ! 格子点数・最大波数設定
!!$    ! Number of grid points and maximum truncated wavenumber settings
!!$    !
!!$    use gridset, only: a_jmax, jmax_max
!!$
!!$    real(DP), intent(in):: y_Data (1:jmax)
!!$    real(DP):: AvrLat_y
!!$
!!$    ! 作業変数
!!$    ! Work variables
!!$    !
!!$
!!$    ! 実行文 ; Executable statement
!!$    !
!!$
!!$    real(DP), allocatable :: a_SendBuf (:)
!!$    real(DP), allocatable :: aa_RecvBuf(:,:)
!!$    integer , allocatable :: a_iReqSend(:)
!!$    integer , allocatable :: a_iReqRecv(:)
!!$
!!$    real(DP)              :: SumWeight
!!$
!!$    integer               :: j
!!$    integer               :: n
!!$
!!$    allocate( a_SendBuf (1:jmax_max) )
!!$    allocate( aa_RecvBuf(1:jmax_max,0:nprocs-1) )
!!$    allocate( a_iReqSend(0:nprocs-1) )
!!$    allocate( a_iReqRecv(0:nprocs-1) )
!!$
!!$    do j = 1, jmax
!!$      a_SendBuf(j) = y_Lat_Weight(j)
!!$    end do
!!$    do j = jmax+1, jmax_max
!!$      a_SendBuf(j) = -1.0d0
!!$    end do
!!$
!!$    do n = 0, nprocs-1
!!$      if ( n == myrank ) then
!!$        do j = 1, jmax
!!$          aa_RecvBuf(:,n) = a_SendBuf
!!$        end do
!!$      else
!!$        call MPIWrapperISend( n, jmax_max, a_SendBuf      , a_iReqSend(n) )
!!$        call MPIWrapperIRecv( n, jmax_max, aa_RecvBuf(:,n), a_iReqRecv(n) )
!!$      end if
!!$    end do
!!$    do n = 0, nprocs-1
!!$      if ( n == myrank ) cycle
!!$      call MPIWrapperWait( a_iReqSend(n) )
!!$      call MPIWrapperWait( a_iReqRecv(n) )
!!$    end do
!!$
!!$
!!$    if ( nprocs == 1 ) then
!!$      SumWeight = 0.0d0
!!$      do n = 0, nprocs-1
!!$        do j = 1, a_jmax(n)
!!$          SumWeight = SumWeight + aa_RecvBuf(j,n)
!!$        end do
!!$      end do
!!$    else
!!$
!!$      write( 6, * ) 'THIS WOULD BE WRONG.'
!!$
!!$      SumWeight = 0.0d0
!!$      do n = nprocs-1, 0, -1
!!$        do j = a_jmax(n), a_jmax(n) / 2 + 1, -1
!!$          SumWeight = SumWeight + aa_RecvBuf(j,n)
!!$        end do
!!$      end do
!!$      do n = 0, nprocs-1
!!$        do j = 1, a_jmax(n) / 2
!!$          SumWeight = SumWeight + aa_RecvBuf(j,n)
!!$        end do
!!$      end do
!!$    end if
!!$
!!$    deallocate( a_SendBuf  )
!!$    deallocate( aa_RecvBuf )
!!$    deallocate( a_iReqSend )
!!$    deallocate( a_iReqRecv )
!!$
!!$
!!$    AvrLat_y = IntLat_y( y_Data ) / SumWeight
!!$
!!$  end function AvrLat_y
!!$
!!$#else
!!$  !-------------------------------------------------------------------
!!$
!!$  function AvrLat_y( y_Data )
!!$    !
!!$    ! 1 次元緯度格子点データの緯度方向平均(1 層用). 
!!$    !
!!$    ! 実際には格子点データ各点毎に y_Lat_Weight を掛けた
!!$    ! 総和を計算し, y_Lat_Weight の総和で割ることで
!!$    ! 平均している. 
!!$    !
!!$    ! Meridonal mean of 1-dimensional (latitude)
!!$    ! grid data.
!!$    !
!!$    ! Practically, the sum total of grid data is calculated
!!$    ! by multiplying in each grid "y_Lat_Weight" 
!!$    ! and deviding by the sum total of "y_Lat_Weight".
!!$    !
!!$    real(DP), intent(in):: y_Data (1:jmax)
!!$    real(DP):: AvrLat_y
!!$
!!$    ! 作業変数
!!$    ! Work variables
!!$    !
!!$
!!$    ! 実行文 ; Executable statement
!!$    !
!!$
!!$    AvrLat_y = IntLat_y( y_Data ) / sum( y_Lat_Weight )
!!$
!!$  end function AvrLat_y
!!$
!!$#endif
!!$  !-------------------------------------------------------------------
!!$
!!$  function AvrLon_x( x_Data )
!!$    !
!!$    ! 1 次元経度格子点データの経度方向平均(1 層用). 
!!$    !
!!$    ! 実際には格子点データ各点毎に x_Lon_Weight を掛けた
!!$    ! 総和を計算し, x_Lon_Weight の総和で割ることで
!!$    ! 平均している. 
!!$    !
!!$    ! Zonal mean of 1-dimensional (longitude)
!!$    ! grid data.
!!$    !
!!$    ! Practically, the sum total of grid data is calculated
!!$    ! by multiplying in each grid "x_Lon_Weight" 
!!$    ! and deviding by the sum total of "x_Lon_Weight".
!!$    !
!!$    real(DP), intent(in):: x_Data (0:imax-1)
!!$    real(DP):: AvrLon_x
!!$
!!$    ! 作業変数
!!$    ! Work variables
!!$    !
!!$
!!$    ! 実行文 ; Executable statement
!!$    !
!!$    AvrLon_x = IntLon_x( x_Data ) / sum( x_Lon_Weight )
!!$  end function AvrLon_x
!!$

  !--------------------------------------------------------------------------------------

  subroutine IntAvrOprInit
    !
    ! intavr_operate モジュールの初期化を行います. 
    ! NAMELIST#intavr_operate_nml の読み込みはこの手続きで行われます. 
    !
    ! "intavr_operate" module is initialized. 
    ! "NAMELIST#intavr_operate_nml" is loaded in this procedure. 
    !

    ! モジュール引用 ; USE statements
    !

    ! 宣言文 ; Declaration statements
    !

!!$    integer:: unit_nml        ! NAMELIST ファイルオープン用装置番号. 
!!$                              ! Unit number for NAMELIST file open
!!$    integer:: iostat_nml      ! NAMELIST 読み込み時の IOSTAT. 
!!$                              ! IOSTAT of NAMELIST read

    ! NAMELIST 変数群
    ! NAMELIST group name
    !
!!$    namelist /intavr_operate_nml/
          !
          ! デフォルト値については初期化手続 "intavr_operate#IntAvrOprInit" 
          ! のソースコードを参照のこと. 
          !
          ! Refer to source codes in the initialization procedure
          ! "intavr_operate#IntAvrOprInit" for the default values. 
          !

    ! 実行文 ; Executable statement
    !

    if ( intavr_operate_inited ) return


    ! デフォルト値の設定
    ! Default values settings
    !

!!$    ! NAMELIST の読み込み
!!$    ! NAMELIST is input
!!$    !
!!$    if ( trim(namelist_filename) /= '' ) then
!!$      call FileOpen( unit_nml, &          ! (out)
!!$        & namelist_filename, mode = 'r' ) ! (in)
!!$
!!$      rewind( unit_nml )
!!$      read( unit_nml, &           ! (in)
!!$        & nml = intavr_operate_nml, &  ! (out)
!!$        & iostat = iostat_nml )   ! (out)
!!$      close( unit_nml )
!!$
!!$      call NmlutilMsg( iostat_nml, module_name ) ! (in)
!!$    end if

    ! 印字 ; Print
    !
    call MessageNotify( 'M', module_name, '----- Initialization Messages -----' )
    call MessageNotify( 'M', module_name, '-- version = %c', c1 = trim(version) )
    intavr_operate_inited = .true.
  end subroutine IntAvrOprInit

  !--------------------------------------------------------------------------------------

end module intavr_operate
