!= 短波入射 (太陽入射)
!
!= Short wave (insolation) incoming
!
! Authors::   Satoshi NODA, Yasuhiro MORIKAWA, Yukiko YAMADA, Yoshiyuki O. TAKAHASHI
! Version::   $Id: rad_short_income.f90,v 1.6 2013/05/25 06:33:57 yot Exp $
! Tag Name::  $Name:  $
! Copyright:: Copyright (C) GFD Dennou Club, 2008. All rights reserved.
! License::   See COPYRIGHT[link:../../../COPYRIGHT]
!

module rad_short_income
  !
  != 短波入射 (太陽入射)
  !
  != Short wave (insolation) incoming
  !
  ! <b>Note that Japanese and English are described in parallel.</b>
  !
  ! 短波入射 (太陽入射) を計算します.
  !
  ! Calculate short wave (insolation) incoming radiation. 
  !
  !== Procedures List
  !
  ! ShortIncoming      :: 短波入射 (太陽入射) の計算
  ! ------------       :: ------------
  ! ShortIncoming      :: Calculate short wave (insolation) incoming radiation. 
  !
  !== NAMELIST
  !
  ! NAMELIST#rad_short_income_nml
  !

  ! モジュール引用 ; USE statements
  !

  ! 種別型パラメタ
  ! Kind type parameter
  !
  use dc_types, only: DP, &      ! 倍精度実数型. Double precision. 
    &                 STRING, &  ! 文字列.       Strings. 
    &                 TOKEN      ! キーワード.   Keywords.

  ! メッセージ出力
  ! Message output
  !
  use dc_message, only: MessageNotify

  ! 物理・数学定数設定
  ! Physical and mathematical constants settings
  !
  use constants0, only: &
    & PI                    ! $ \pi $.
                            ! 円周率. Circular constant

  ! 格子点設定
  ! Grid points settings
  !
  use gridset, only: imax, & ! 経度格子点数. 
                             ! Number of grid points in longitude
    &                jmax, & ! 緯度格子点数. 
                             ! Number of grid points in latitude
    &                kmax    ! 鉛直層数. 
                             ! Number of vertical level

  ! 時刻管理
  ! Time control
  !
  use timeset, only: &
    & TimeN,                & !
    & InitialDate             ! 計算開始日時.
                              ! Start date of calculation

  ! 座標データ設定
  ! Axes data settings
  !
  use axesset, only: &
    & y_Lat,   & ! $ \varphi $ [rad.] . 緯度. Latitude
    & x_Lon      ! $ \lambda $ [rad.] . 経度. Longitude


  ! 宣言文 ; Declaration statements
  !
  implicit none
  private

  ! 公開手続き
  ! Public procedure
  !
  public:: RadShortIncome
  public:: RadShortIncomeInit

  ! 公開変数
  ! Public variables
  !
  logical, save :: rad_short_income_inited = .false.
                              ! 初期設定フラグ. 
                              ! Initialization flag.


  ! 非公開変数
  ! Private variables
  !

  integer,  save      :: IDRadDist
  integer,  parameter :: IDRadDistAnnuallyDiurnallyVarying = 11
  integer,  parameter :: IDRadDistAnnualDiurnalMean        = 12
  integer,  parameter :: IDRadDistPerpetual                = 13
  integer,  parameter :: IDRadDistSynchronous              = 14
  integer,  parameter :: IDRadDistUniform                  = 15

  real(DP), save :: PerpSinDel
                              ! sine of declination angle for perpetual experiments
  real(DP), save :: PerpDistFromStarScld
                              ! distance between a planet and a central star for 
                              ! perpetual experiment
  logical,  save:: FlagSpecifySolDay
                              ! 
                              ! Flag for specifying solar day
  real(DP), save :: SolDay
                              ! length of solar day


  ! NOTE:
  !   Distance between the central star and the planet, and the declination can be
  !   determined by specifying following three parameters: 
  !     PerLonFromVE     : the longitude of the perihelion measured from the vernal 
  !                        equinox, 
  !     LonFromVEAtEpoch : the longitude of the planet at epoch, 
  !     epoch            : date or time. 
  !
  real(DP), save:: PerLonFromVE
                              ! 春分から測った近日点の経度 (近日点黄経) (degree)
                              ! Longitude of the perihelion from vernal equinox (degree)
  real(DP), save:: LonFromVEAtEpoch
                              ! 元期における惑星の経度 (黄経) (degree)
                              ! Longitude of the planet at epoch (degree)
  real(DP), save:: TimeAtEpoch
                              ! 元期における時刻 (sec)
                              ! Time at epoch (sec)

  real(DP), save:: EpsOrb
                              ! 赤道傾斜角.
                              ! Inclination of equator to orbit.
  real(DP), save:: Eccentricity
                              ! 離心率.
                              ! Eccentricity.

  integer,  save:: MaxItrEccAnomaly
                              ! 離心近点角を計算する時の最大繰り返し回数.
                              ! Maximum iteration number of times
                              ! to calculate eccentric anomaly.
  real(DP), save:: ThreEccAnomalyError
                              ! 離心近点角を計算する時の誤差の許容しきい値.
                              ! Threshold of error
                              ! to calculate eccentric anomaly.

  real(DP), save:: IncomAIns
                              ! $ A_{ins} $ . 年平均入射の係数. 
                              ! Coefficient of annual mean incoming radiation. 
  real(DP), save:: IncomBIns
                              ! $ B_{ins} $ . 年平均入射の係数. AIns に同じ. 
                              ! Coefficient of annual mean incoming radiation. 
                              ! Same as "AIns". 
  real(DP), save:: IncomAZet
                              ! $ A_{\zeta} $ . 年平均入射角の係数. AIns に同じ. 
                              ! Coefficient of annual mean incoming radiation. 
                              ! Same as "AIns". 
  real(DP), save:: IncomBZet
                              ! $ B_{\zeta} $ . 年平均入射角の係数. AIns に同じ. 
                              ! Coefficient of annual mean incoming radiation. 
                              ! Same as "AIns". 


  character(*), parameter:: module_name = 'rad_short_income'
                              ! モジュールの名称. 
                              ! Module name
  character(*), parameter:: version = &
    & '$Name:  $' // &
    & '$Id: rad_short_income.f90,v 1.6 2013/05/25 06:33:57 yot Exp $'
                              ! モジュールのバージョン
                              ! Module version

contains

  !--------------------------------------------------------------------------------------

  subroutine RadShortIncome(        &
    & xy_CosSZA, FluxFactor,        & ! (out)
    & DistFromStarScld,             & ! (out) optional
    & PlanetLonFromVE,              & ! (out) optional
    & SinDel, xy_DCosSZADt,         & ! (out) optional
    & FlagOutput                    & ! (in)  optional
    & )
    !
    ! 短波入射 (太陽入射) を計算します.
    !
    ! Calculate short wave (insolation) incoming radiation. 
    !

    ! モジュール引用 ; USE statements
    !

    ! 日付および時刻の取り扱い
    ! Date and time handler
    !
    use dc_calendar, only: DCCalInquire, DCCalDateEvalSecOfDay

    ! ヒストリデータ出力
    ! History data output
    !
    use gtool_historyauto, only: HistoryAutoPut


    ! 宣言文 ; Declaration statements
    !
    implicit none

    real(DP), intent(out)           :: xy_CosSZA(0:imax-1, 1:jmax)
                              ! cos( 入射角 )
                              ! cos( Incident angle )
    real(DP), intent(out)           :: FluxFactor
    real(DP), intent(out), optional :: DistFromStarScld
                              ! 軌道長半径でスケーリングした恒星からの距離
                              ! Distance from the star scaled
                              ! by semimajor axis of the planet's orbit
    real(DP), intent(out), optional :: PlanetLonFromVE
                              ! 中心星を中心とする座標における春分点から測った惑星の経度
                              ! Longitude of the planet measured from the vernal equinox
                              ! in the coordinate that the central star is located on 
                              ! the origin.
    real(DP), intent(out), optional :: SinDel          ! 赤緯
                               ! Declination
    real(DP), intent(out), optional :: xy_DCosSZADt(0:imax-1, 1:jmax)
    logical , intent(in ), optional :: FlagOutput


    ! 作業変数
    ! Work variables
    !
    integer:: i                ! 経度方向に回る DO ループ用作業変数
                               ! Work variables for DO loop in longitude
    integer:: j                ! 緯度方向に回る DO ループ用作業変数
                               ! Work variables for DO loop in latitude
    real(DP):: SinDelLV        ! 赤緯
                               ! Declination
    real(DP):: AngleMaxLon     ! 入射が最大となる緯度

    real(DP):: HourAngle       ! 時角
                               ! Hour angle
    real(DP):: ClockByDay      ! 時刻を日で表記したもの (0.0 - 1.0)
                               ! Clock expressed by day (0.0 - 1.0)

    real(DP):: DCosSZADt       !
    real(DP) :: xy_DCosSZADtLV     (0:imax-1, 1:jmax)

    real(DP) :: DistFromStarScldLV
                               ! Distance between the central star and the planet
                               ! (local variable)
    real(DP) :: PlanetLonFromVELV

    integer         :: hour_in_a_day, min_in_a_hour
    real(DP)        :: sec_in_a_min, sec_in_a_day

    logical :: FlagOutputLocal


    ! 実行文 ; Executable statement
    !

    ! 初期化確認
    ! Initialization check
    !
    if ( .not. rad_short_income_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    select case ( IDRadDist )
    case ( IDRadDistAnnuallyDiurnallyVarying, IDRadDistPerpetual )

      ! Set sine of declination and distance between a planet and a central star 
      ! scaled with semi-major axis
      !
      select case ( IDRadDist )
      case ( IDRadDistAnnuallyDiurnallyVarying )
        call ShortIncomCalcOrbParam(                         &
          & SinDelLV, DistFromStarScldLV, PlanetLonFromVELV  & ! (out) optional
          & )
      case ( IDRadDistPerpetual )
        SinDelLV           = PerpSinDel
        DistFromStarScldLV = PerpDistFromStarScld
        PlanetLonFromVELV  = 0.0_DP
        xy_DCosSZADtLV     = 0.0_DP
      end select

      call DCCalInquire( &
        & hour_in_day      = hour_in_a_day,     & ! (out)
        & min_in_hour      = min_in_a_hour,     & ! (out)
        & sec_in_min       = sec_in_a_min       & ! (out)
        & )
      sec_in_a_day  = hour_in_a_day * min_in_a_hour * sec_in_a_min

      if ( FlagSpecifySolDay ) then
        ! case with solar day which is different from sec_in_a_day (rotation
        ! period)
        ClockByDay = mod( TimeN / SolDay, 1.0_DP )
      else
        ! case with solar day which is the same as sec_in_a_day (rotation
        ! period)
        ClockByDay = DCCalDateEvalSecOfDay( TimeN, date = InitialDate )
        ClockByDay = ClockByDay / sec_in_a_day
      end if

      do i = 0, imax - 1
        do j = 1, jmax
          HourAngle = ClockByDay * 2.0_DP * PI - PI + x_Lon(i)
          !
          xy_CosSZA(i,j) = &
            &    sin( y_Lat(j) ) * SinDelLV &
            &  + cos( y_Lat(j) ) * sqrt( 1.0_DP - SinDelLV**2 ) * cos( HourAngle )
          !
          DCosSZADt = - cos( y_Lat(j) ) * sqrt( 1.0_DP - SinDelLV**2 ) * sin( HourAngle ) * 2.0_DP * PI
          if ( FlagSpecifySolDay ) then
            DCosSZADt = DCosSZADt / SolDay
          else
            DCosSZADt = DCosSZADt / sec_in_a_day
          end if
          xy_DCosSZADtLV(i,j) = DCosSZADt
        end do
      end do

      FluxFactor = 1.0_DP

    case ( IDRadDistAnnualDiurnalMean )

      do i = 0, imax - 1
        do j = 1, jmax
          ! FluxFactor is set to (IncomAIns+IncomBIns)/(IncomAZet+IncomBZet)
          ! above. So,
          !   FluxFactor * ( IncomAZet + IncomBZet * cos( y_Lat(j) )**2 )
          !    = ( IncomAIns + IncomBIns * cos( y_Lat(j) )**2 )
!!$            xy_CosSZA(i,j) = IncomAIns + IncomBIns * cos( y_Lat(j) )**2
          xy_CosSZA(i,j) = IncomAZet + IncomBZet * cos( y_Lat(j) )**2
        end do
      end do

      FluxFactor =  ( IncomAIns + IncomBIns ) / ( IncomAZet + IncomBZet )

      DistFromStarScldLV = 1.0_DP
      SinDelLV           = 0.0_DP
      PlanetLonFromVELV  = 0.0_DP
      xy_DCosSZADtLV     = 0.0_DP

    case ( IDRadDistSynchronous )

      ! 同期回転日射
      ! incoming radiation for synchronous rotation

      ! 同期回転惑星を想定しており,
      ! 常に経度 90 度で最大入射, 経度 180-360 度で入射ゼロとなっています.
      !
      ! Calculate short wave (insolation) incoming radiation.
      ! A planet with synchronized rotation are assumed.
      ! Incoming is max at latitude 90 deg., and zero between 180 and 360 deg. constantly.
      do i = 0, imax - 1
         AngleMaxLon  = - PI / 2.0_DP + x_Lon( i )
         do j = 1, jmax
           xy_CosSZA(i,j) = cos( y_Lat(j) ) * cos( AngleMaxLon )
         end do
      end do

      DistFromStarScldLV = 1.0_DP
      SinDelLV           = 0.0_DP

      FluxFactor         = 1.0_DP
      xy_DCosSZADtLV     = 0.0_DP

    case ( IDRadDistUniform )

      xy_CosSZA = cos( 60.0_DP * PI / 180.0_DP )

      DistFromStarScldLV = 1.0_DP
      SinDelLV           = 0.0_DP

      FluxFactor         = 0.5_DP
      xy_DCosSZADtLV     = 0.0_DP

    end select


    if ( present( DistFromStarScld ) ) then
      DistFromStarScld = DistFromStarScldLV
    end if
    if ( present( PlanetLonFromVE ) ) then
      PlanetLonFromVE = PlanetLonFromVELV
    end if
    if ( present( SinDel ) ) then
      SinDel = SinDelLV
    end if
    if ( present( xy_DCosSZADt ) ) then
      xy_DCosSZADt = xy_DCosSZADtLV
    end if


    ! ヒストリデータ出力
    ! History data output
    !
    FlagOutputLocal = .false.
    if ( present( FlagOutput ) ) then
      if ( FlagOutput ) then
        FlagOutputLocal = .true.
      end if
    else
      FlagOutputLocal = .true.
    end if
    if ( FlagOutputLocal ) then
      call HistoryAutoPut( TimeN, 'Decl'            , asin(SinDelLV)*180.0_DP/PI      )
      call HistoryAutoPut( TimeN, 'DistFromStarScld', DistFromStarScldLV            )
      call HistoryAutoPut( TimeN, 'PlanetLonFromVE' , PlanetLonFromVELV*180.0_DP/PI )
    end if


  end subroutine RadShortIncome

  !--------------------------------------------------------------------------------------

  subroutine ShortIncomCalcOrbParam(             &
    & SinDel, DistFromStarScld, PlanetLonFromVE  & ! (out)
    & )
    !
    ! 短波入射 (太陽入射) を計算します.
    !
    ! Calculate short wave (insolation) incoming radiation. 
    !

    ! モジュール引用 ; USE statements
    !

    ! 日付および時刻の取り扱い
    ! Date and time handler
    !
    use dc_calendar, only: DCCalInquire, DCCalDateChkLeapYear

    ! 宣言文 ; Declaration statements
    !
    implicit none
    real(DP), intent(out) :: SinDel
                              ! 赤緯
                              ! Declination
    real(DP), intent(out) :: DistFromStarScld
                              ! 軌道長半径でスケーリングした恒星からの距離
                              ! Distance from the star scaled
                              ! by semimajor axis of the planet's orbit
    real(DP), intent(out) :: PlanetLonFromVE
                              ! 中心星を中心とする座標における春分点から測った惑星の経度
                              ! Longitude of the planet measured from the vernal equinox
                              ! in the coordinate that the central star is located on 
                              ! the origin.

    ! 作業変数
    ! Work variables
    !
    integer:: itr              ! イテレーション方向に回る DO ループ用作業変数
                               ! Work variables for DO loop in iteration direction

    real(DP):: MeanAnomaly     ! 平均近点角
                               ! Mean anomaly
    real(DP):: EccAnomaly      ! 離心近点角
                               ! eccentric anomaly
    real(DP):: EccAnomalyError ! ニュートン法における離心近点角の誤差
                               ! error of eccentric anomaly in Newton method
    real(DP):: TrueAnomaly     ! 真点離角
                               ! true anomaly

    integer         :: hour_in_a_day, min_in_a_hour, day_in_a_year
    integer, pointer:: day_in_month_ptr(:) => null()
    real(DP)        :: sec_in_a_min, sec_in_a_day, sec_in_a_year



    ! 実行文 ; Executable statement
    !

    ! 初期化確認
    ! Initialization check
    !
    if ( .not. rad_short_income_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    ! 季節変化, 日変化がある場合の計算
    ! Calculate with seasonal change and diurnal change
    !
    call DCCalInquire( &
      & day_in_month_ptr = day_in_month_ptr , & ! (out)
      & hour_in_day      = hour_in_a_day,     & ! (out)
      & min_in_hour      = min_in_a_hour,     & ! (out)
      & sec_in_min       = sec_in_a_min       & ! (out)
      & )
    ! Add 1 to day_in_month_ptr(2) if it is leap year.
    !
    if ( DCCalDateChkLeapYear( TimeN, InitialDate ) ) then
      day_in_month_ptr(2) = day_in_month_ptr(2) + 1
    end if

    day_in_a_year = sum( day_in_month_ptr )
    deallocate( day_in_month_ptr )
    sec_in_a_day  = hour_in_a_day * min_in_a_hour * sec_in_a_min
    sec_in_a_year = day_in_a_year * sec_in_a_day

    MeanAnomaly  =                                               &
      &   2.0_DP * PI * ( TimeN - TimeAtEpoch ) / sec_in_a_year  &
      & + ( LonFromVEAtEpoch - PerLonFromVE ) * PI / 180.0_DP
    MeanAnomaly  = mod( MeanAnomaly, 2.0_DP * PI )


    ! ニュートン法を用いて平均近点角から離心近点角を求める.
    ! Calculate eccentric anomaly from mean anomaly by using Newton method.

    EccAnomaly = MeanAnomaly
    do itr = 1, MaxItrEccAnomaly
      EccAnomalyError = EccAnomaly                                        &
        & - Eccentricity * sin(EccAnomaly) - MeanAnomaly
      if ( abs(EccAnomalyError) <= ThreEccAnomalyError ) exit
      EccAnomaly      = EccAnomaly                                        &
        & - EccAnomalyError / ( 1.0_DP - Eccentricity * cos(EccAnomaly) )
      EccAnomaly      = mod( EccAnomaly, 2.0 * PI )
    end do
    if ( itr > MaxItrEccAnomaly ) then
      call MessageNotify( 'E', module_name,                        &
        & 'Calculation for eccentric anomaly does not converge.' )
    end if

    DistFromStarScld = 1.0_DP - Eccentricity * cos( EccAnomaly )

    TrueAnomaly = 2.0_DP                                                    &
      & * atan(                                                             &
      &         sqrt( ( 1.0d0 + Eccentricity ) / ( 1.0d0 - Eccentricity ) ) &
      &           * tan( EccAnomaly / 2.0_DP )                              &
      &       )

    PlanetLonFromVE = TrueAnomaly + PerLonFromVE * PI / 180.0_DP
    PlanetLonFromVE = mod( PlanetLonFromVE, 2.0_DP * PI )

    SinDel = sin( EpsOrb * PI / 180.0_DP ) * sin( PlanetLonFromVE )


        ! code for debug
!!$        write( 60, * ) TimeN/sec_in_a_day, DCCalDateChkLeapYear(TimeN,date=InitialDate), day_in_a_year
!!$        write(  6, * ) TimeN/sec_in_a_day, DCCalDateChkLeapYear(TimeN,date=InitialDate), day_in_a_year
!!$        call flush( 60 )


!!$        write( 60, * ) TimeN/sec_in_a_day, asin(SinDel)*180.0/PI, DistFromStarScld, PlanetLonFromVE*180.0_DP/PI
!!$        write(  6, * ) TimeN/sec_in_a_day, asin(SinDel)*180.0/PI, DistFromStarScld, PlanetLonFromVE*180.0_DP/PI
!!$        call flush( 60 )


  end subroutine ShortIncomCalcOrbParam

  !--------------------------------------------------------------------------------------

  subroutine RadShortIncomeInit
    !
    ! rad_short_income モジュールの初期化を行います. 
    ! NAMELIST#rad_short_income_nml の読み込みはこの手続きで行われます. 
    !
    ! "rad_short_income" module is initialized. 
    ! "NAMELIST#rad_short_income_nml" is loaded in this procedure. 
    !

    ! モジュール引用 ; USE statements
    !

    ! 種別型パラメタ
    ! Kind type parameter
    !
    use dc_types, only: STDOUT ! 標準出力の装置番号. Unit number of standard output

    ! 文字列操作
    ! Character handling
    !
    use dc_string, only: LChar

    ! ファイル入出力補助
    ! File I/O support
    !
    use dc_iounit, only: FileOpen

    ! ヒストリデータ出力
    ! History data output
    !
    use gtool_historyauto, only: HistoryAutoAddVariable

    ! 暦と日時の取り扱い
    ! Calendar and Date handler
    !
    use dc_calendar, only: &
      & DC_CAL_DATE, &          ! 日時を表現するデータ型.
                                ! Data type for date and time
      & DCCalDateInquire, DCCalDateCreate, DCCalDateDifference, &
      & DCCalConvertByUnit

    ! NAMELIST ファイル入力に関するユーティリティ
    ! Utilities for NAMELIST file input
    !
    use namelist_util, only: namelist_filename, NmlutilMsg, NmlutilAryValid

    ! 宣言文 ; Declaration statements
    !
    implicit none

    character(STRING) :: RadDist

    ! --- Flags below are obsolete and will be deleted. ---
    logical,  save:: FlagAnnualMean
                              ! 年平均入射フラグ.
                              ! Flag for annual mean incoming radiation. 
    logical,  save:: FlagDiurnalMean
                              ! 日平均入射フラグ.
                              ! Flag for diurnal mean incoming radiation. 
    logical,  save:: FlagPerpetual
                              ! 季節変化なし入射フラグ.
                              ! Flag for perpetual incoming radiation. 
    logical,  save:: FlagRadSynchronous
                              ! 同期回転日射のフラグ
                              ! Flag for synchronous rotation
    ! -----------------------------------------------------

    integer:: EpochYear, EpochMonth, EpochDay, EpochHour, EpochMin
                              ! 元期日時 (年月日時分).
                              ! "TimeAtEpoch" が負の場合にこちらが使用される.
                              !
                              ! Date at epoch (year, month, day, hour, minute)
                              ! These are used when "TimeAtEpoch" is negative.
    real(DP):: EpochSec
                              ! 元期日時 (秒).
                              ! "TimeAtEpoch" が負の場合にこちらが使用される.
                              !
                              ! Date at epoch (second)
                              ! These are used when "TimeAtEpoch" is negative.

    type(DC_CAL_DATE):: EpochDate
                              ! 元期の日時
                              ! Date at epoch

    real(DP)         :: PerpDelDeg
                              ! Declination angle in degree used for perpetual experiment

    real(DP)         :: SolDayValue
    character(TOKEN) :: SolDayUnit

    integer:: unit_nml        ! NAMELIST ファイルオープン用装置番号. 
                              ! Unit number for NAMELIST file open
    integer:: iostat_nml      ! NAMELIST 読み込み時の IOSTAT. 
                              ! IOSTAT of NAMELIST read

    logical         :: FlagUseOfEpochDate
    character(TOKEN):: date_print


    ! NAMELIST 変数群
    ! NAMELIST group name
    !
    namelist /rad_short_income_nml/                                     &
      & RadDist,                                                        &
      !
      ! --- Flags below are obsolete and will be deleted. ---
      & FlagRadSynchronous,                                             &
      & FlagAnnualMean, FlagDiurnalMean, FlagPerpetual,                 &
      ! -----------------------------------------------------
      !
      & PerpDelDeg, PerpDistFromStarScld,                               &
      & EpsOrb,                                                         &
      & PerLonFromVE,                                                   &
      & LonFromVEAtEpoch,                                               &
      & Eccentricity,                                                   &
      & TimeAtEpoch,                                                    &
      & EpochYear, EpochMonth, EpochDay, EpochHour, EpochMin, EpochSec, &
      & MaxItrEccAnomaly, ThreEccAnomalyError,                          &
      & IncomAIns, IncomBIns, IncomAZet, IncomBZet,                     &
      & FlagSpecifySolDay, SolDayValue, SolDayUnit
          !
          ! デフォルト値については初期化手続 "rad_short_income#RadShortIncomeInit" 
          ! のソースコードを参照のこと. 
          !
          ! Refer to source codes in the initialization procedure
          ! "rad_short_income#RadShortIncomeInit" for the default values. 
          !

    ! 実行文 ; Executable statement
    !

    if ( rad_short_income_inited ) return


    ! デフォルト値の設定
    ! Default values settings
    !
!!$    RadDist = 'annual diurnal mean'
    RadDist = 'annually diurnally varying'

    ! --- Flags below are obsolete and will be deleted. ---
!    FlagRadSynchronous       = .false.
!    FlagAnnualMean           = .true.
!    FlagDiurnalMean          = .true.
!    FlagPerpetual            = .false.
    FlagRadSynchronous       = .false.
    FlagAnnualMean           = .false.
    FlagDiurnalMean          = .false.
    FlagPerpetual            = .false.
    ! -----------------------------------------------------

    !---

    PerpDelDeg               = 0.0_DP
    PerpDistFromStarScld     = 1.0_DP

    !---

    EpsOrb               =   23.5_DP    ! Earth-like value
    PerLonFromVE         =    0.0_DP
    LonFromVEAtEpoch     =  280.0_DP    ! This value results in the fact that the planet
                                        ! is located at the position of vernal equinox 
                                        ! at 80 days after calculation with the use of 
                                        ! "360day" calendar.
    Eccentricity         =    0.0_DP
    TimeAtEpoch          =    0.0_DP
    EpochYear            =   -1
    EpochMonth           =   -1
    EpochDay             =   -1
    EpochHour            =   -1
    EpochMin             =   -1
    EpochSec             =   -1.0_DP

    !---

    ! Sample values for the Earth
    !  References: 
    !    Duffett-Smith, P., Practical astronomy with your calculator Third Edition, 
    !    Cambridge University Press, pp.185, 1988.
    !
!!$    EpsOrb               =   23.44_DP                ! Rika nenpyo (Chronological 
!!$                                                     ! Scientific Tables 2010)
!!$    PerLonFromVE         =  102.768413_DP + 180.0_DP ! Duffett-Smith (1988), p.105
!!$                                                     ! modified (plus 180 degrees)
!!$    LonFromVEAtEpoch     =   99.403308_DP + 180.0_DP ! Duffett-Smith (1988), p.105
!!$                                                     ! modified (plus 180 degrees)
!!$    Eccentricity         =    0.016713_DP            ! Duffett-Smith (1988), p.105
!!$    TimeAtEpoch          =   -1.0_DP                 ! EpochXXX written below are used 
!!$                                                     ! because this is negative.
!!$    EpochYear            = 1990                      ! Duffett-Smith (1988), p.105
!!$    EpochMonth           =    1
!!$    EpochDay             =    1
!!$    EpochHour            =    0
!!$    EpochMin             =    0
!!$    EpochSec             =    0.0_DP
    !---

    ! Sample values for Mars
    !  References: 
    !    Allison, M., Geophys. Res. Lett., 24, 1967-1970, 1997.
    !
!!$    EpsOrb               =   25.19_DP              ! Allison (1997)
!!$    PerLonFromVE         =  250.98_DP              ! Allison (1997) (modified)
!!$    LonFromVEAtEpoch     =  -10.342_DP             ! Arbitrarily set for clarity
!!$                                                   ! This results in Ls ~ 0 at Time = 0
!!$    Eccentricity         =    0.0934_DP            ! Allison (1997), value at epoch J2000
!!$    TimeAtEpoch          =    0.0_DP               ! Arbitrarily set for clarity
!!$    EpochYear            =   -1                    ! not used
!!$    EpochMonth           =   -1
!!$    EpochDay             =   -1
!!$    EpochHour            =   -1
!!$    EpochMin             =   -1
!!$    EpochSec             =   -1.0_DP

    !---

    MaxItrEccAnomaly     = 20
    ThreEccAnomalyError  = 1e-6_DP

    IncomAIns            = 0.127_DP   ! see dcpam document for reference
    IncomBIns            = 0.183_DP   ! see dcpam document for reference
    IncomAZet            = 0.410_DP   ! see dcpam document for reference
    IncomBZet            = 0.590_DP   ! see dcpam document for reference

    FlagSpecifySolDay = .false.
    SolDayValue       = 0.0_DP
    SolDayUnit        = 'sec'


    ! NAMELIST の読み込み
    ! NAMELIST is input
    !
    if ( trim(namelist_filename) /= '' ) then
      call FileOpen( unit_nml, &          ! (out)
        & namelist_filename, mode = 'r' ) ! (in)

      rewind( unit_nml )
      read( unit_nml, &                ! (in)
        & nml = rad_short_income_nml, &  ! (out)
        & iostat = iostat_nml )        ! (out)
      close( unit_nml )

      call NmlutilMsg( iostat_nml, module_name ) ! (in)
    end if


    ! set insolation distribution
    !
    select case ( LChar( trim(RadDist) ) )
    case ( 'annually diurnally varying' )
      IDRadDist = IDRadDistAnnuallyDiurnallyVarying
    case ( 'annual diurnal mean' )
      IDRadDist = IDRadDistAnnualDiurnalMean
    case ( 'perpetual' )
      IDRadDist = IDRadDistPerpetual
    case ( 'synchronous' )
      IDRadDist = IDRadDistSynchronous
    case ( 'uniform' )
      IDRadDist = IDRadDistUniform
    case default
      call MessageNotify( 'E', module_name, &
        & 'RadDist = %c is not supported.', &
        & c1 = trim( RadDist ) )
    end select


    ! Lines below are obsolete and will be deleted.
    if ( FlagRadSynchronous ) then
      call MessageNotify( 'M', module_name, &
        & 'IDRadDist is overwritten and is set to IDRadDistSynchronous.' )
      IDRadDist = IDRadDistSynchronous
    else

      if ( ( .not. FlagAnnualMean ) .and. FlagDiurnalMean ) then
        call MessageNotify( 'E', module_name, &
          & 'FlagDiurnalMean cannot be true if FlagAnnualMean is false.' )
      end if

      if ( FlagAnnualMean .and. ( .not. FlagDiurnalMean ) ) then
        call MessageNotify( 'E', module_name, &
          & 'FlagAnnualMean cannot be true if FlagDiurnalMean is false.' )
      end if

      if ( FlagAnnualMean .and. FlagDiurnalMean ) then
        call MessageNotify( 'M', module_name, &
          & 'IDRadDist is overwritten and is set to IDRadDistAnnualDiurnalMean.' )
        IDRadDist = IDRadDistAnnualDiurnalMean
      end if

      if ( FlagPerpetual ) then
        call MessageNotify( 'M', module_name, &
          & 'IDRadDist is overwritten and is set to IDRadDistPerpetual.' )
        IDRadDist = IDRadDistPerpetual
      end if

    end if


    PerpSinDel = sin( PerpDelDeg * PI / 180.0_DP )


    FlagUseOfEpochDate = .false.

    if ( TimeAtEpoch < 0.0_DP ) then
      call DCCalDateCreate(    &
        & year  = EpochYear,   & ! (in)
        & month = EpochMonth,  & ! (in)
        & day   = EpochDay,    & ! (in)
        & hour  = EpochHour,   & ! (in)
        & min   = EpochMin,    & ! (in)
        & sec   = EpochSec,    & ! (in)
        & date  = EpochDate )    ! (out) optional

      TimeAtEpoch = DCCalDateDifference( &
        &                                 start_date = InitialDate, &  ! (in)
        &                                 end_date   = EpochDate    &  ! (in)
        &                              )

      FlagUseOfEpochDate = .true.
    end if

    SolDay = DCCalConvertByUnit( SolDayValue, SolDayUnit, 'sec' )

    ! 保存用の変数の割り付け
    ! Allocate variables for saving
    !

    ! ヒストリデータ出力のためのへの変数登録
    ! Register of variables for history data output
    !
!!$    call HistoryAutoAddVariable( 'xxxxx' , &
!!$      & (/ 'lon ', 'lat ', 'sig', 'time'/), &
!!$      & 'xxxx', 'W m-2' )

    call HistoryAutoAddVariable( 'Decl' , &
      & (/ 'time'/),                      &
      & 'declination of the central star', 'degree' )

    call HistoryAutoAddVariable( 'DistFromStarScld' , &
      & (/ 'time'/),                                  &
      & 'distance between the central star and the planet', '1' )

    call HistoryAutoAddVariable( 'PlanetLonFromVE' , &
      & (/ 'time'/),                                 &
      & 'planetary longitude from the vernal equinox', 'degree' )

    ! 印字 ; Print
    !
    call MessageNotify( 'M', module_name, '----- Initialization Messages -----' )
    call MessageNotify( 'M', module_name, 'ShortIncomming:' )
    call MessageNotify( 'M', module_name, '  IDRadDist                = %d', i = (/ IDRadDist /) )
    call MessageNotify( 'M', module_name, '    RadDist                = %c', c1 = trim( RadDist ) )
    call MessageNotify( 'M', module_name, '    FlagRadSynchronous     = %b', l = (/ FlagRadSynchronous /) )
    if ( FlagRadSynchronous ) then
      call MessageNotify( 'M', module_name, '    FlagRadSynchronous is obsolete, use RadDist.' )
    end if
    call MessageNotify( 'M', module_name, '    FlagAnnualMean         = %b', l = (/ FlagAnnualMean            /) )
    if ( FlagAnnualMean ) then
      call MessageNotify( 'M', module_name, '    FlagAnnualMean is obsolete, use RadDist.' )
    end if
    call MessageNotify( 'M', module_name, '    FlagDiurnalMean        = %b', l = (/ FlagDiurnalMean           /) )
    if ( FlagDiurnalMean ) then
      call MessageNotify( 'M', module_name, '    FlagDiurnalMean is obsolete, use RadDist.' )
    end if
    call MessageNotify( 'M', module_name, '    FlagPerpetual          = %b', l = (/ FlagPerpetual             /) )
    if ( FlagPerpetual ) then
      call MessageNotify( 'M', module_name, '    FlagPerpetual is obsolete, use RadDist.' )
    end if
    call MessageNotify( 'M', module_name, '  PerpDelDeg               = %f', d = (/ PerpDelDeg                /) )
    call MessageNotify( 'M', module_name, '  PerpDistFromStarScld     = %f', d = (/ PerpDistFromStarScld      /) )
    call MessageNotify( 'M', module_name, '  EpsOrb                   = %f', d = (/ EpsOrb                   /) )
    call MessageNotify( 'M', module_name, '  PerLonFromVE             = %f', d = (/ PerLonFromVE             /) )
    call MessageNotify( 'M', module_name, '  Eccentricity             = %f', d = (/ Eccentricity             /) )

    if ( FlagUseOfEpochDate ) then
      call DCCalDateInquire( date_print, date = EpochDate )
      call MessageNotify( 'M', module_name, '  EpochDate  = %c', &
        & c1 = trim(date_print) )
    end if
    call MessageNotify( 'M', module_name, '  TimeAtEpoch              = %f', d = (/ TimeAtEpoch              /) )
    call MessageNotify( 'M', module_name, '  LonFromVEAtEpoch         = %f', d = (/ LonFromVEAtEpoch         /) )

    call MessageNotify( 'M', module_name, '  MaxItrEccAnomaly         = %d', i = (/ MaxItrEccAnomaly         /) )
    call MessageNotify( 'M', module_name, '  ThreEccAnomalyError      = %f', d = (/ ThreEccAnomalyError      /) )
    call MessageNotify( 'M', module_name, '  IncomAIns                = %f', d = (/ IncomAIns                /) )
    call MessageNotify( 'M', module_name, '  IncomBIns                = %f', d = (/ IncomBIns                /) )
    call MessageNotify( 'M', module_name, '  IncomAZet                = %f', d = (/ IncomAZet                /) )
    call MessageNotify( 'M', module_name, '  IncomBZet                = %f', d = (/ IncomBZet                /) )

    call MessageNotify( 'M', module_name, '  FlagSpecifySolDay        = %b', l = (/ FlagSpecifySolDay        /) )
    call MessageNotify( 'M', module_name, '  SolDay                   = %f', d = (/ SolDay                   /) )


    call MessageNotify( 'M', module_name, '-- version = %c', c1 = trim(version) )

    rad_short_income_inited = .true.

  end subroutine RadShortIncomeInit

  !-------------------------------------------------------------------

end module rad_short_income
