!= 全球一定混合比の設定
!
!= Set globally constant mixing ratio
!
! Authors::   Yoshiyuki O. Takahashi
! Version::   $Id: set_gcmr.f90,v 1.4 2015/01/29 12:06:43 yot Exp $
! Tag Name::  $Name:  $
! Copyright:: Copyright (C) GFD Dennou Club, 2008. All rights reserved.
! License::   See COPYRIGHT[link:../../../COPYRIGHT]
!

module set_gcmr
  !
  != 全球一定混合比の設定
  !
  != Set globally constant mixing ratio
  !
  ! <b>Note that Japanese and English are described in parallel.</b>
  !
  ! 全球一定混合比.
  !
  ! Set globally constant mixing ratio.
  !
  !== References
  !
!!$  !  Roewe, D., and K.-N. Liou, Influence of cirrus clouds on the infrared cooling 
!!$  !    rate in the troposphere and lower stratosphere, J. Appl. Met., 17, 92-106, 1978.
  !
  !== Procedures List
  !
!!$  ! RadiationFluxDennouAGCM :: 放射フラックスの計算
!!$  ! RadiationDTempDt        :: 放射フラックスによる温度変化の計算
!!$  ! RadiationFluxOutput     :: 放射フラックスの出力
!!$  ! RadiationFinalize       :: 終了処理 (モジュール内部の変数の割り付け解除)
!!$  ! ------------            :: ------------
!!$  ! RadiationFluxDennouAGCM :: Calculate radiation flux
!!$  ! RadiationDTempDt        :: Calculate temperature tendency with radiation flux
!!$  ! RadiationFluxOutput     :: Output radiation fluxes
!!$  ! RadiationFinalize       :: Termination (deallocate variables in this module)
  !
  !== NAMELIST
  !
  ! NAMELIST#set_gcmr_nml
  !

  ! USE statements
  !

  ! 
  ! Kind type parameter
  !
  use dc_types, only: DP, &      ! Double precision.
    &                 STRING, &  ! Strings.
    &                 TOKEN      ! Keywords.

  ! メッセージ出力
  ! Message output
  !
  use dc_message, only: MessageNotify


  ! Declaration statements
  !
  implicit none
  private

  ! 
  ! Public procedure
  !
  public:: SetGCMR
  public:: SetGCMRInit


  character(*), parameter:: module_name = 'set_gcmr'
                              ! モジュールの名称.
                              ! Module name
  character(*), parameter:: version = &
    & '$Name:  $' // &
    & '$Id: set_gcmr.f90,v 1.4 2015/01/29 12:06:43 yot Exp $'
                              ! モジュールのバージョン
                              ! Module version

  real(DP), save         :: MRCO2
  character(STRING), save:: CO2File
  character(STRING), save:: CO2Name
  real(DP), save         :: MRN2O
  character(STRING), save:: N2OFile
  character(STRING), save:: N2OName
  real(DP), save         :: MRCH4
  character(STRING), save:: CH4File
  character(STRING), save:: CH4Name

  logical, save :: set_gcmr_inited = .false.
                              ! 初期設定フラグ.
                              ! Initialization flag

contains

  !--------------------------------------------------------------------------------------

  subroutine SetGCMR( &
    & Spec,           & ! (in)
    & QGC             & ! (out)
    & )


    ! USE statements
    !

    ! ヒストリデータ出力
    ! History data output
    !
    use gtool_historyauto, only: HistoryAutoAddVariable, HistoryAutoPut

    ! 時刻管理
    ! Time control
    !
    use timeset, only: &
      & TimeN, &              ! ステップ $ t $ の時刻.
                              ! Time of step $ t $.
      & TimesetClockStart, TimesetClockStop

    ! 時系列データの読み込み
    ! Reading time series
    !
    use read_time_series, only: SetValuesFromTimeSeriesWrapper


    character(*), intent(in ):: Spec
    real(DP)    , intent(out):: QGC


    !
    ! Work variables
    !


    ! 初期化確認
    ! Initialization check
    !
    if ( .not. set_gcmr_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    ! 計算時間計測開始
    ! Start measurement of computation time
    !
    call TimesetClockStart( module_name )


    if ( Spec == 'CO2' ) then

      if ( CO2File == '' ) then
        QGC = MRCO2
      else
        call SetValuesFromTimeSeriesWrapper( &
          & "CO2",            &
          & CO2File, CO2Name, &
          & QGC               &               ! (inout)
          & )
      end if

      call HistoryAutoPut( TimeN, "QCO2", QGC )

    else if ( Spec == 'N2O' ) then

      if ( N2OFile == '' ) then
        QGC = MRN2O
      else
        call SetValuesFromTimeSeriesWrapper( &
          & "N2O",            &
          & N2OFile, N2OName, &
          & QGC               &               ! (inout)
          & )
      end if

      call HistoryAutoPut( TimeN, "QN2O", QGC )

    else if ( Spec == 'CH4' ) then

      if ( CH4File == '' ) then
        QGC = MRCH4
      else
        call SetValuesFromTimeSeriesWrapper( &
          & "CH4",            &
          & CH4File, CH4Name, &
          & QGC               &               ! (inout)
          & )
      end if

      call HistoryAutoPut( TimeN, "QCH4", QGC )

    end if


    ! 計算時間計測一時停止
    ! Pause measurement of computation time
    !
    call TimesetClockStop( module_name )


  end subroutine SetGCMR

  !--------------------------------------------------------------------------------------

  subroutine SetGCMRInit


    ! ファイル入出力補助
    ! File I/O support
    !
    use dc_iounit, only: FileOpen

    ! ヒストリデータ出力
    ! History data output
    !
    use gtool_historyauto, only: HistoryAutoAddVariable

    ! NAMELIST ファイル入力に関するユーティリティ
    ! Utilities for NAMELIST file input
    !
    use namelist_util, only: namelist_filename, NmlutilMsg, NmlutilAryValid


    !
    ! Work variables
    !
    integer:: unit_nml        ! NAMELIST ファイルオープン用装置番号.
                              ! Unit number for NAMELIST file open
    integer:: iostat_nml      ! NAMELIST 読み込み時の IOSTAT.
                              ! IOSTAT of NAMELIST read

    ! NAMELIST 変数群
    ! NAMELIST group name
    !
    namelist /set_gcmr_nml/ &
      & MRCO2,   &
      & CO2File, &
      & CO2Name, &
      & MRN2O,   &
      & N2OFile, &
      & N2OName, &
      & MRCH4,   &
      & CH4File, &
      & CH4Name
          !
          ! デフォルト値については初期化手続 "set_gcmr#SetGCMRInit"
          ! のソースコードを参照のこと.
          !
          ! Refer to source codes in the initialization procedure
          ! "set_gcmr#SetGCMRInit" for the default values.
          !

    if ( set_gcmr_inited ) return


    ! デフォルト値の設定
    ! Default values settings
    !

!!$    MRCO2   = 382.0d-6     ! old default value
    MRCO2   =  369.0d-6 ! Value at 2000 in CMIP5 recommendation, PRE2005_MIDYR_CONC.DAT
    CO2File = ''
    CO2Name = ''
    MRN2O   =  316.0d-9 ! Value at 2000 in CMIP5 recommendation, PRE2005_MIDYR_CONC.DAT
    N2OFile = ''
    N2OName = ''
    MRCH4   = 1751.0d-9 ! Value at 2000 in CMIP5 recommendation, PRE2005_MIDYR_CONC.DAT
    CH4File = ''
    CH4Name = ''


    ! NAMELIST の読み込み
    ! NAMELIST is input
    !
    if ( trim(namelist_filename) /= '' ) then
      call FileOpen( unit_nml, &          ! (out)
        & namelist_filename, mode = 'r' ) ! (in)

      rewind( unit_nml )
      read( unit_nml,                     & ! (in)
        & nml = set_gcmr_nml,              & ! (out)
        & iostat = iostat_nml )             ! (out)
      close( unit_nml )

      call NmlutilMsg( iostat_nml, module_name ) ! (in)
    end if


    call HistoryAutoAddVariable( "QCO2",       & ! (in)
      & (/ 'time' /),                          & ! (in)
      & "carbon dioxide", '1' )                  ! (in)
    call HistoryAutoAddVariable( "QN2O",       & ! (in)
      & (/ 'time' /),                          & ! (in)
      & "nitrous oxide", '1' )                   ! (in)
    call HistoryAutoAddVariable( "QCH4",       & ! (in)
      & (/ 'time' /),                          & ! (in)
      & "methane", '1' )                         ! (in)

    ! 印字 ; Print
    !
    call MessageNotify( 'M', module_name, '----- Initialization Messages -----' )
    call MessageNotify( 'M', module_name, '  MRCO2   = %f', d = (/ MRCO2 /) )
    call MessageNotify( 'M', module_name, '  CO2File = %c', c1 = trim( CO2File ) )
    call MessageNotify( 'M', module_name, '  CO2Name = %c', c1 = trim( CO2Name ) )
    call MessageNotify( 'M', module_name, '  MRN2O   = %f', d = (/ MRN2O /) )
    call MessageNotify( 'M', module_name, '  N2OFile = %c', c1 = trim( N2OFile ) )
    call MessageNotify( 'M', module_name, '  N2OName = %c', c1 = trim( N2OName ) )
    call MessageNotify( 'M', module_name, '  MRCH4   = %f', d = (/ MRCH4 /) )
    call MessageNotify( 'M', module_name, '  CH4File = %c', c1 = trim( CH4File ) )
    call MessageNotify( 'M', module_name, '  CH4Name = %c', c1 = trim( CH4Name ) )
    call MessageNotify( 'M', module_name, '-- version = %c', c1 = trim(version) )

    set_gcmr_inited = .true.

  end subroutine SetGCMRInit

  !--------------------------------------------------------------------------------------

end module set_gcmr
