!= 地球放射モデル Ver. 3 (汎惑星大気放射モデル, 地球 Ver. 1)
!
!= Earth radiation model Ver. 3 (general-purpose radiation model for planetary atmosphere, Earth Ver. 1)
!
module rad_Earth_V3
  !
  != 地球放射モデル Ver. 3 (汎惑星大気放射モデル Earth Ver. 1)
  !
  != Earth radiation model Ver. 3 (general-purpose radiation model for planetary atmosphere, Earth Ver. 1)
  !
  ! <b>Note that Japanese and English are described in parallel.</b>
  !
  ! 
  !
  ! This is a general-purpose radiation model for planetary atmospheres
  !
  !== References
  !
  !
  !== Procedures List
  !
  ! RadEarthV3Flux :: 放射フラックスの計算
  ! -------------- :: ------------
  ! RadEarthV3Flux :: Calculate radiation flux
  !
  !== NAMELIST
  !
  ! NAMELIST#rad_Earth_V3_nml
  !

  ! USE statements
  !

  !
  ! Kind type parameter
  !
  use dc_types, only: DP, &      ! Double precision.
    &                 STRING, &  ! Strings.
    &                 TOKEN      ! Keywords.

  ! メッセージ出力
  ! Message output
  !
  use dc_message, only: MessageNotify

  !
  ! Physical constants settings
  !
  use constants, only: Grav     ! $ g $ [m s-2].
                                !
                                ! Gravitational acceleration

  ! 格子点設定
  ! Grid points settings
  !
  use gridset, only: imax, & ! 経度格子点数.
                             ! Number of grid points in longitude
    &                jmax, & ! 緯度格子点数.
                             ! Number of grid points in latitude
    &                kmax    ! 鉛直層数.
                             ! Number of vertical level

  implicit none

  private


  integer                       , save :: NMolNum
  integer                       , save :: NPtcl
  integer          , allocatable, save :: m_MolNum(:)
  character(64)    , allocatable, save :: a_PtclName(:)

  real(DP)         , allocatable, save :: a_PrescPtclDens   (:)

  integer                       , save :: iLoop
  integer                       , save :: NCalcInterval

  logical                       , save :: FlagDiurnalMean
  integer                       , save :: NSZA
  real(DP)         , allocatable, save :: a_DiurnalMeanHA(:)
  real(DP)         , allocatable, save :: a_SZAGW        (:)
  real(DP)         , allocatable, save :: xya_SZADeg     (:,:,:)

  real(DP), save :: CloudWatREff
  real(DP), save :: CloudIceREff


  ! 公開変数
  ! Public variables
  !
  logical, save :: rad_Earth_V3_inited = .false.
                              ! 初期設定フラグ.
                              ! Initialization flag

  public :: RadEarthV3Flux
  public :: RadEarthV3Init
  public :: RadEarthV3Finalize

  character(*), parameter:: module_name = 'rad_Earth_V3'
                              ! モジュールの名称.
                              ! Module name
  character(*), parameter:: version = &
    & '$Name:  $' // &
    & '$Id: rad_Earth_V2.f90,v 1.9 2015/01/29 12:06:43 yot Exp $'
                              ! モジュールのバージョン
                              ! Module version

  !--------------------------------------------------------------------------------------

contains

  !--------------------------------------------------------------------------------------

  subroutine RadEarthV3Flux(                       &
    & xy_SurfAlbedo,                               & ! (in)
    & xyz_Press, xyr_Press, xyz_Temp,              & ! (in)
    & xy_SurfTemp,                                 & ! (in)
    & xyz_QH2OVap, xyz_QH2OLiq, xyz_QH2OSol,       & ! (in)
    & xyr_RadSUwFlux, xyr_RadSDwFlux,              & ! (inout)
    & xyr_RadLUwFlux, xyr_RadLDwFlux,              & ! (inout)
    & xyra_DelRadLUwFlux, xyra_DelRadLDwFlux       & ! (inout)
    & )


    ! USE statements
    !

    ! ヒストリデータ出力
    ! History data output
    !
    use gtool_historyauto, only: HistoryAutoPut

    ! 時刻管理
    ! Time control
    !
    use timeset, only: &
      & TimeN, &              ! ステップ $ t $ の時刻.
                              ! Time of step $ t $.
      & TimesetClockStart, TimesetClockStop

    ! 物理・数学定数設定
    ! Physical and mathematical constants settings
    !
    use constants0, only: &
      & PI

    ! 座標データ設定
    ! Axes data settings
    !
    use axesset, only: &
      & y_Lat

    ! 物理定数設定
    ! Physical constants settings
    !
    use constants, only : &
      & MolWtDry, &
      & MolWtWet

    ! 太陽放射フラックスの設定
    ! Set solar constant
    !
    use set_solarconst, only : SetSolarConst

    use rad_flux_wrapper, only : RadFluxWrapper
!!$    use rad_flux_wrapper, only : RadFluxWrapperInterfaceFor2DRoutine

    ! 短波入射 (太陽入射)
    ! Short wave (insolation) incoming
    !
    use rad_short_income, only : RadShortIncome

    ! 全球一定体積混合比の設定
    ! Set globally constant volume mixing ratio
    !
    use set_gcmr, only : SetGCMR

    ! O3 分布の設定
    ! Set O3 distribution
    !
    use set_o3, only: SetO3


    real(DP), intent(in   ) :: xy_SurfAlbedo   (0:imax-1, 1:jmax)
    real(DP), intent(in   ) :: xyz_Press       (0:imax-1, 1:jmax, 1:kmax)
    real(DP), intent(in   ) :: xyr_Press       (0:imax-1, 1:jmax, 0:kmax)
    real(DP), intent(in   ) :: xyz_Temp        (0:imax-1, 1:jmax, 1:kmax)
    real(DP), intent(in   ) :: xy_SurfTemp     (0:imax-1, 1:jmax)
    real(DP), intent(in   ) :: xyz_QH2OVap     (0:imax-1, 1:jmax, 1:kmax)
    real(DP), intent(in   ) :: xyz_QH2OLiq     (0:imax-1, 1:jmax, 1:kmax)
    real(DP), intent(in   ) :: xyz_QH2OSol     (0:imax-1, 1:jmax, 1:kmax)
    real(DP), intent(inout) :: xyr_RadSUwFlux   (0:imax-1, 1:jmax, 0:kmax)
    real(DP), intent(inout) :: xyr_RadSDwFlux   (0:imax-1, 1:jmax, 0:kmax)
    real(DP), intent(inout) :: xyr_RadLUwFlux    (0:imax-1, 1:jmax, 0:kmax)
    real(DP), intent(inout) :: xyr_RadLDwFlux    (0:imax-1, 1:jmax, 0:kmax)
    real(DP), intent(inout) :: xyra_DelRadLUwFlux(0:imax-1, 1:jmax, 0:kmax, 0:1)
    real(DP), intent(inout) :: xyra_DelRadLDwFlux(0:imax-1, 1:jmax, 0:kmax, 0:1)


    ! Work variables
    !
    real(DP) :: xyz_DelAtmMass   (0:imax-1, 1:jmax, 1:kmax)

    real(DP) :: xyr_Temp      (0:imax-1, 1:jmax, 0:kmax)
    real(DP) :: xyrm_VMR      (0:imax-1, 1:jmax, 0:kmax, NMolNum)

    real(DP) :: PlanetLonFromVE
    real(DP) :: xyr_DOD067  (0:imax-1, 1:jmax, 0:kmax)
    real(DP) :: xyr_DOD     (0:imax-1, 1:jmax, 0:kmax)

    real(DP) :: StrFluxTOA0
    real(DP) :: StrFluxTOA
    real(DP) :: xy_CosSZA     (0:imax-1, 1:jmax)
    real(DP) :: FluxFactor

    real(DP) :: xy_AlbedoSW   (0:imax-1, 1:jmax)
    real(DP) :: xy_AlbedoLW   (0:imax-1, 1:jmax)
    real(DP) :: xy_EmisSW     (0:imax-1, 1:jmax)
    real(DP) :: xy_EmisLW     (0:imax-1, 1:jmax)

    real(DP) :: MajCompIceThreshold
    real(DP) :: MajCompIceEmis
    real(DP) :: xy_SurfEmis   (0:imax-1, 1:jmax)

    real(DP) :: xyr_RadUwFlux      (0:imax-1, 1:jmax, 0:kmax)
    real(DP) :: xyr_RadDwFlux      (0:imax-1, 1:jmax, 0:kmax)
    real(DP) :: xyra_DelRadUwFlux  (0:imax-1, 1:jmax, 0:kmax, -1:1)
    real(DP) :: xyra_DelRadDwFlux  (0:imax-1, 1:jmax, 0:kmax, -1:1)
    real(DP) :: xyr_RadUwFluxSW    (0:imax-1, 1:jmax, 0:kmax)
    real(DP) :: xyr_RadDwFluxSW    (0:imax-1, 1:jmax, 0:kmax)
    real(DP) :: xyr_RadUwFluxLW    (0:imax-1, 1:jmax, 0:kmax)
    real(DP) :: xyr_RadDwFluxLW    (0:imax-1, 1:jmax, 0:kmax)
    real(DP) :: xyra_DelRadUwFluxSW(0:imax-1, 1:jmax, 0:kmax, -1:1)
    real(DP) :: xyra_DelRadDwFluxSW(0:imax-1, 1:jmax, 0:kmax, -1:1)
    real(DP) :: xyra_DelRadUwFluxLW(0:imax-1, 1:jmax, 0:kmax, -1:1)
    real(DP) :: xyra_DelRadDwFluxLW(0:imax-1, 1:jmax, 0:kmax, -1:1)

    real(DP)          :: xyza_PtclDens   (0:imax-1, 1:jmax, 1:kmax, 1:NPtcl)
    real(DP)          :: xyza_PtclEffRad (0:imax-1, 1:jmax, 1:kmax, 1:NPtcl)
    real(DP)          :: xyza_PtclMMR    (0:imax-1, 1:jmax, 1:kmax, 1:NPtcl)
    real(DP)          :: xyza_DelPtclMass(0:imax-1, 1:jmax, 1:kmax, 1:NPtcl)
    real(DP)          :: xyr_MeanMolWt   (0:imax-1, 1:jmax, 0:kmax)
    real(DP)          :: H2OMolWt


    real(DP) :: DistFromStarScld
    real(DP) :: SinDel

    real(DP) :: VMRCO2
    real(DP) :: xyz_QO3(0:imax-1, 1:jmax, 1:kmax)

    integer  :: i
    integer  :: j
    integer  :: k
    integer  :: l
    integer  :: m
    integer  :: n


    ! 初期化確認
    ! Initialization check
    !
    if ( .not. rad_Earth_V3_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    ! 計算時間計測開始
    ! Start measurement of computation time
    !
    call TimesetClockStart( module_name )


    if ( ( iLoop > 0 ) .and. ( mod( iLoop, NCalcInterval ) /= 0 ) ) then
      iLoop = iLoop + 1
      return
    end if
    iLoop = iLoop + 1


    k = 0
    xyr_Temp(:,:,k) =                                    &
      &      ( xyz_Temp (:,:,k+2) - xyz_Temp (:,:,k+1) ) &
      & / log( xyz_Press(:,:,k+2) / xyz_Press(:,:,k+1) ) &
      & * log( xyr_Press(:,:,k  ) / xyz_Press(:,:,k+1) ) &
      & + xyz_Temp(:,:,k+1)
    do k = 1, kmax-1
      xyr_Temp(:,:,k) =                                &
        &      ( xyz_Temp (:,:,k+1) - xyz_Temp (:,:,k) ) &
        & / log( xyz_Press(:,:,k+1) / xyz_Press(:,:,k) ) &
        & * log( xyr_Press(:,:,k  ) / xyz_Press(:,:,k) ) &
        & + xyz_Temp(:,:,k)
    end do
    k = kmax
    xyr_Temp(:,:,k) = xyz_Temp(:,:,k)



!!$    xy_AlbedoSW = xy_SurfAlbedo
!!$    xy_AlbedoLW = 0.0_DP

    !  Surface emissivity
    !
    xy_SurfEmis = 1.0_DP
    !
    xy_AlbedoSW = xy_SurfAlbedo
    xy_AlbedoLW = 1.0_DP - xy_SurfEmis
    xy_EmisSW   = 1.0_DP - xy_SurfAlbedo
    xy_EmisLW   = xy_SurfEmis


    ! 太陽放射フラックスの設定
    ! Set solar constant
    !
    call SetSolarConst( &
      & StrFluxTOA0 & ! (out)
      & )

    ! 短波入射の計算
    ! Calculate short wave (insolation) incoming radiation
    !
    call RadShortIncome(                       &
      & xy_CosSZA, FluxFactor,                 & ! (out) optional
      & DistFromStarScld  = DistFromStarScld,  & ! (out) optional
      & SinDel            = SinDel             & ! (out) optional
      & )
    !
    if ( FlagDiurnalMean ) then
      do n = 1, NSZA
        do j = 1, jmax
          xy_CosSZA(:,j) = &
            &  + sin( y_Lat(j) ) * SinDel &
            &  + cos( y_Lat(j) ) * sqrt( 1.0_DP - SinDel**2 ) * cos( a_DiurnalMeanHA(n) )
        end do
        xya_SZADeg(:,:,n) = acos( xy_CosSZA ) * 180.0_DP / PI
      end do
    else
      xya_SZADeg(:,:,1) = acos( xy_CosSZA ) * 180.0_DP / PI
    end if


    StrFluxTOA = StrFluxTOA0 / ( DistFromStarScld * DistFromStarScld ) * FluxFactor

    xyr_MeanMolWt = MolWtDry
    H2OMolWt      = MolWtWet

    do k = 1, kmax
      xyz_DelAtmMass(:,:,k) = ( xyr_Press(:,:,k-1) - xyr_Press(:,:,k) ) / Grav
    end do

    do m = 1, NMolNum
      select case ( m_MolNum(m) )
      case ( 1 )
!!$        k = 0
!!$        xyrm_VMR(:,:,k,m) = xyz_QH2OVap(:,:,1)
!!$        do k = 0+1, kmax-1
!!$          xyrm_VMR(:,:,k,m) = ( xyz_QH2OVap(:,:,k) + xyz_QH2OVap(:,:,k+1) ) / 2.0d0
!!$        end do
!!$        k = kmax
!!$        xyrm_VMR(:,:,k,m) = xyz_QH2OVap(:,:,kmax)
        !
        k = 0
        xyrm_VMR(:,:,k,m) =                                      &
          &      ( xyz_QH2OVap(:,:,k+2) - xyz_QH2OVap(:,:,k+1) ) &
          & / log( xyz_Press  (:,:,k+2) / xyz_Press  (:,:,k+1) ) &
          & * log( xyr_Press  (:,:,k  ) / xyz_Press  (:,:,k+1) ) &
          & + xyz_QH2OVap(:,:,k+1)
        do k = 1, kmax-1
          xyrm_VMR(:,:,k,m) =                                    &
            &      ( xyz_QH2OVap(:,:,k+1) - xyz_QH2OVap(:,:,k) ) &
            & / log( xyz_Press  (:,:,k+1) / xyz_Press  (:,:,k) ) &
            & * log( xyr_Press  (:,:,k  ) / xyz_Press  (:,:,k) ) &
            & + xyz_QH2OVap(:,:,k)
        end do
        k = kmax
        xyrm_VMR(:,:,k,m) = xyz_QH2OVap(:,:,k)
        !
        xyrm_VMR(:,:,:,m) = max( xyrm_VMR(:,:,:,m), 0.0d0 )
        ! vmr = n_h2o / n_all
        ! mmr = m_h2o * n_h2o / ( m_mean * n_all )
        !     = m_h2o / m_mean * n_h2o / n_all
        !     = m_h2o / m_mean * vmr
        ! vmr = m_mean / m_h2o * mmr
        xyrm_VMR(:,:,:,m) = xyrm_VMR(:,:,:,m) * xyr_MeanMolWt / H2OMolWt
      case ( 2 )
        ! 全球一定体積混合比の設定
        ! Set globally constant volume mixing ratio
        !
        call SetGCMR( &
          & 'CO2',    & ! (in)
          & VMRCO2    & ! (out)
          & )
        xyrm_VMR(:,:,:,m) = VMRCO2
      case ( 3 )
        ! O3 分布の設定
        ! Setting of O3 distribution
        !
        call SetO3( &
          & xyz_Press, & ! (in )
          & xyz_QO3    & ! (out)
          & )
        k = 0
        xyrm_VMR(:,:,k,m) =                                  &
          &   log( xyz_QO3  (:,:,k+2) / xyz_QO3  (:,:,k+1) ) &
          & / log( xyz_Press(:,:,k+2) / xyz_Press(:,:,k+1) ) &
          & * log( xyr_Press(:,:,k  ) / xyz_Press(:,:,k+1) ) &
          & + log( xyz_QO3(:,:,k+1) )
        do k = 1, kmax-1
          xyrm_VMR(:,:,k,m) =                                &
            &   log( xyz_QO3  (:,:,k+1) / xyz_QO3  (:,:,k) ) &
            & / log( xyz_Press(:,:,k+1) / xyz_Press(:,:,k) ) &
            & * log( xyr_Press(:,:,k  ) / xyz_Press(:,:,k) ) &
            & + log( xyz_QO3(:,:,k) )
        end do
        k = kmax
        xyrm_VMR(:,:,k,m) = log( xyz_QO3(:,:,k) )
        !
        xyrm_VMR(:,:,:,m) = exp( xyrm_VMR(:,:,:,m) )
      case default
        xyrm_VMR(:,:,:,m) = 0.0d0


!!$        ! 全球一定体積混合比の設定
!!$        ! Set globally constant volume mixing ratio
!!$        !
!!$        call SetGCMR( &
!!$          & 'N2O',    & ! (in)
!!$          & VMRN2O    & ! (out)
!!$          & )
!!$
!!$        ! 全球一定体積混合比の設定
!!$        ! Set globally constant volume mixing ratio
!!$        !
!!$        call SetGCMR( &
!!$          & 'CH4',    & ! (in)
!!$          & VMRCH4    & ! (out)
!!$          & )

      end select
    end do


    do m = 1, NPtcl
      select case ( a_PtclName(m) )
      case ( 'water' )
        xyza_PtclEffRad(:,:,:,m) = CloudWatREff
        xyza_PtclMMR   (:,:,:,m) = xyz_QH2OLiq
      case ( 'ice' )
        xyza_PtclEffRad(:,:,:,m) = CloudIceREff
        xyza_PtclMMR   (:,:,:,m) = xyz_QH2OSol
      case default
        xyza_PtclEffRad(:,:,:,m) = 1.0d-6
        xyza_PtclMMR   (:,:,:,m) = 0.0d0
      end select
    end do
    do m = 1, NPtcl
      xyza_PtclDens(:,:,:,m) = a_PrescPtclDens(m)
    end do

    do m = 1, NPtcl
      xyza_DelPtclMass(:,:,:,m) = &
        &   xyz_DelAtmMass * xyza_PtclMMR(:,:,:,m)
    end do

    call RadFluxWrapper(                      &
!!$    call RadFluxWrapperInterfaceFor2DRoutine( &
      & imax, jmax, kmax,                           & ! (in)
      & NMolNum, m_MolNum,                          & ! (in)
      & xy_AlbedoSW, xy_AlbedoLW,                   & ! (in)
      & xy_EmisSW, xy_EmisLW,                       & ! (in)
      & StrFluxTOA,                                 & ! (in)
      & NSZA, xya_SZADeg, a_SZAGW,                  & ! (in)
      & xyr_Press, xyr_Temp, xy_SurfTemp,           & ! (in)
      & xyrm_VMR, xyr_MeanMolWt,                    & ! (in)
      & NPtcl, a_PtclName,                          & ! (in)
      & xyza_PtclDens,                              & ! (in)
      & xyza_PtclEffRad, xyza_DelPtclMass,          & ! (in)
      & xyr_RadUwFlux, xyr_RadDwFlux,               & ! (out)
      & xyra_DelRadUwFlux, xyra_DelRadDwFlux,       & ! (out)
      & xyr_RadUwFluxLW, xyr_RadDwFluxLW,           & ! (out)
      & xyr_RadUwFluxSW, xyr_RadDwFluxSW,           & ! (out)
      & xyra_DelRadUwFluxLW, xyra_DelRadDwFluxLW,   & ! (out)
      & xyra_DelRadUwFluxSW, xyra_DelRadDwFluxSW    & ! (out)
      & )

    xyr_RadLUwFlux     = xyr_RadUwFluxLW
    xyr_RadLDwFlux     = xyr_RadDwFluxLW
    xyr_RadSUwFlux     = xyr_RadUwFluxSW
    xyr_RadSDwFlux     = xyr_RadDwFluxSW

    xyra_DelRadLUwFlux(:,:,:,0) = xyra_DelRadUwFluxLW(:,:,:,-1)
    xyra_DelRadLDwFlux(:,:,:,0) = xyra_DelRadDwFluxLW(:,:,:,-1)
    !
!!$    xyra_DelRadLUwFlux(:,:,:,1) = 0.0_DP
!!$    xyra_DelRadLDwFlux(:,:,:,1) = 0.0_DP
    do l = 1, 1
      do k = 0, kmax
        xyra_DelRadLUwFlux(:,:,k,l) = &
          &   ( xyra_DelRadUwFluxLW(:,:,k,l) - xyra_DelRadUwFluxLW(:,:,k,l-1) ) &
          & / log( xyr_Press(:,:,l) / xyr_Press(:,:,l-1) ) &
          & * log( xyz_Press(:,:,l) / xyr_Press(:,:,l-1) ) &
          & + xyra_DelRadUwFluxLW(:,:,k,l-1)
        xyra_DelRadLDwFlux(:,:,k,l) = &
          &   ( xyra_DelRadDwFluxLW(:,:,k,l) - xyra_DelRadDwFluxLW(:,:,k,l-1) ) &
          & / log( xyr_Press(:,:,l) / xyr_Press(:,:,l-1) ) &
          & * log( xyz_Press(:,:,l) / xyr_Press(:,:,l-1) ) &
          & + xyra_DelRadDwFluxLW(:,:,k,l-1)
      end do
    end do


    ! Output variables
    !
!!$    call HistoryAutoPut( TimeN, 'DOD067', xyr_DOD067 )


    ! 計算時間計測一時停止
    ! Pause measurement of computation time
    !
    call TimesetClockStop( module_name )

  end subroutine RadEarthV3Flux

  !--------------------------------------------------------------------------------------

  subroutine RadEarthV3Init

    ! ファイル入出力補助
    ! File I/O support
    !
    use dc_iounit, only: FileOpen

    ! ヒストリデータ出力
    ! History data output
    !
    use gtool_historyauto, only: HistoryAutoAddVariable

    ! NAMELIST ファイル入力に関するユーティリティ
    ! Utilities for NAMELIST file input
    !
    use namelist_util, only: namelist_filename, NmlutilMsg, NmlutilAryValid

    ! 物理・数学定数設定
    ! Physical and mathematical constants settings
    !
    use constants0, only: &
      & PI

    !
    ! Physical constants settings
    !
    use constants, only: Grav     ! $ g $ [m s-2].
                                  !
                                  ! Gravitational acceleration

    ! 座標データ設定
    ! Axes data settings
    !
    use axesset, only: &
      & AxnameX, &
      & AxnameY, &
      & AxnameZ, &
      & AxnameR, &
      & AxnameT

    ! 太陽放射フラックスの設定
    ! Set solar constant
    !
    use set_solarconst, only : SetSolarConstInit

    ! 全球一定体積混合比の設定
    ! Set globally constant volume mixing ratio
    !
    use set_gcmr, only : SetGCMRInit

    ! O3 分布の設定
    ! Set O3 distribution
    !
    use set_o3, only : SetO3Init

    ! 短波入射 (太陽入射)
    ! Short wave (insolation) incoming
    !
    use rad_short_income, only : RadShortIncomeInit

    use rad_flux_wrapper, only : RadFluxWrapperInit

    ! ガウス重み, 分点の計算
    ! Calculate Gauss node and Gaussian weight
    !
    use gauss_quad, only : GauLeg

    ! gtool データ入力
    ! Gtool data input
    !
    use gtool_history, only: HistoryGet

    use ni3_module


    ! 宣言文 ; Declaration statements
    !


    ! Local variables
    !
    real(DP)          :: WaveNumAlbedoSwitch
    character(STRING) :: KDOptPropNcFN

    integer :: NGQDiurnalMean

    integer           :: NcID

    integer:: unit_nml        ! NAMELIST ファイルオープン用装置番号.
                              ! Unit number for NAMELIST file open
    integer:: iostat_nml      ! NAMELIST 読み込み時の IOSTAT.
                              ! IOSTAT of NAMELIST read


    ! NAMELIST 変数群
    ! NAMELIST group name
    !
    namelist /rad_Earth_V3_nml/ &
      & WaveNumAlbedoSwitch, &
      & KDOptPropNcFN, &
      & NCalcInterval, &
      & FlagDiurnalMean, &
      & NGQDiurnalMean, &
!!$      & CloudIceREffMethod,     &
      & CloudWatREff,           &
      & CloudIceREff

      !
      ! デフォルト値については初期化手続 "rad_Earth_V2#RadEarthV2Init"
      ! のソースコードを参照のこと.
      !
      ! Refer to source codes in the initialization procedure
      ! "rad_Earth_V2#RadEarthV2Init" for the default values.
      !

    if ( rad_Earth_V3_inited ) return



    ! デフォルト値の設定
    ! Default values settings
    !
    KDOptPropNcFN = ""

    NCalcInterval = 1

    FlagDiurnalMean = .false.
    NGQDiurnalMean  = 0

!!$    CloudIceREffMethod = 'Const'
!!!$    CloudIceREffMethod = 'Lin'

    CloudWatREff    = 10.0d-6
    CloudIceREff    = 50.0d-6


    ! NAMELIST の読み込み
    ! NAMELIST is input
    !
    if ( trim(namelist_filename) /= '' ) then
      call FileOpen( unit_nml, &          ! (out)
        & namelist_filename, mode = 'r' ) ! (in)

      rewind( unit_nml )
      read( unit_nml,                     & ! (in)
        & nml = rad_Earth_V3_nml,         & ! (out)
        & iostat = iostat_nml )             ! (out)
      close( unit_nml )

      call NmlutilMsg( iostat_nml, module_name ) ! (in)
    end if


    iLoop = 0


!!$    ! Identification of calculation method of cloud particle effective radius
!!$    !
!!$    call MessageNotify( 'M', module_name, &
!!$      & 'CloudIceREffMethod=<%c>.', &
!!$      & c1 = trim(CloudIceREffMethod) )
!!$    !
!!$    select case ( CloudIceREffMethod )
!!$    case ( 'Const' )
!!$      IDCloudIceREffMethod = IDCloudIceREffMethodConst
!!$    case ( 'Lin' )
!!$      IDCloudIceREffMethod = IDCloudIceREffMethodLin
!!$    case default
!!$      call MessageNotify( 'E', module_name, &
!!$        & 'CloudIceREffMethod=<%c> is not supported.', &
!!$        & c1 = trim(CloudIceREffMethod) )
!!$    end select


    call ni3_open( KDOptPropNcFN, 'read', NcID )
    call ni3_inq_dimlen( NcID, 'MolNum' , NMolNum )
    call ni3_inq_dimlen( NcID, 'PtclNum', NPtcl   )
    !
    allocate( m_MolNum          (NMolNum)                  )
    allocate( a_PtclName        (NPtcl)                    )
    !
    call ni3_get_var( NcID, 'MolNum'  , m_MolNum   )
    call ni3_get_var( NcID, 'PtclName', a_PtclName )
    call ni3_close( NcID )


    allocate( a_PrescPtclDens   (NPtcl) )

    a_PrescPtclDens(1) = 1.0d3
    a_PrescPtclDens(2) = 1.0d3


    if ( FlagDiurnalMean ) then
      if ( NGQDiurnalMean < 1 ) then
        write( 6, * ) 'NGQDiurnalMean has to be greater than 0, when FlagDiurnalMean is .true.: ', &
          & FlagDiurnalMean, NGQDiurnalMean
        stop
      end if
      NSZA = NGQDiurnalMean
      allocate( a_DiurnalMeanHA( NSZA ) )
      allocate( a_SZAGW        ( NSZA ) )
      call GauLeg( 0.0_DP, PI, NSZA, a_DiurnalMeanHA, a_SZAGW )
      a_SZAGW = a_SZAGW / PI
    else
      NSZA = 1
      allocate( a_SZAGW( NSZA ) )
      a_SZAGW = 1.0_DP
    end if
    allocate( xya_SZADeg(0:imax-1, 1:jmax, NSZA) )


    ! Initialization of modules used in this module
    !
    call RadFluxWrapperInit( &
      & imax, jmax,                              &
      & Grav, WaveNumAlbedoSwitch, KDOptPropNcFN &
      & )


    ! 太陽放射フラックスの設定
    ! Set solar constant
    !
    call SetSolarConstInit

    ! 短波入射 (太陽入射)
    ! Short wave (insolation) incoming
    !
    call RadShortIncomeInit

    ! 全球一定体積混合比の設定
    ! Set globally constant volume mixing ratio
    !
    call SetGCMRInit

    ! O3 分布の設定
    ! Set O3 distribution
    !
    call SetO3Init


    ! ヒストリデータ出力のためのへの変数登録
    ! Register of variables for history data output
    !
!!$    call HistoryAutoAddVariable( 'DustDensScledOptDep',    &
!!$      & (/ AxnameX, AxnameY, AxnameZ, AxnameT /),         &
!!$      & 'dust density-scaled optical depth at 0.67 micron meter', 'm2 kg-1' )


    ! 印字 ; Print
    !
    call MessageNotify( 'M', module_name, '----- Initialization Messages -----' )
    call MessageNotify( 'M', module_name, '  WaveNumAlbedoSwitch = %f', &
      & d = (/ WaveNumAlbedoSwitch /) )
    call MessageNotify( 'M', module_name, '  KDOptPropNcFN       = %c', &
      & c1 = trim(KDOptPropNcFN) )
    call MessageNotify( 'M', module_name, '  NCalcInterval       = %d', &
      & i = (/ NCalcInterval /) )
    call MessageNotify( 'M', module_name, '  FlagDiurnalMean     = %b', &
      & l = (/ FlagDiurnalMean /) )
    call MessageNotify( 'M', module_name, '  NGQDiurnalMean      = %d', &
      & i = (/ NGQDiurnalMean /) )
    call MessageNotify( 'M', module_name, '  CloudWatREff        = %f', &
      & d = (/ CloudWatREff /) )
    call MessageNotify( 'M', module_name, '  CloudIceREff        = %f', &
      & d = (/ CloudIceREff /) )
    call MessageNotify( 'M', module_name, '-- version = %c', c1 = trim(version) )


    rad_Earth_V3_inited = .true.

  end subroutine RadEarthV3Init

  !--------------------------------------------------------------------------------------

  subroutine RadEarthV3Finalize


    use rad_flux_wrapper, only : RadFluxWrapperFinalize



    ! 宣言文 ; Declaration statements
    !


    ! Local variables
    !


    if ( rad_Earth_V3_inited ) return


    call RadFluxWrapperFinalize


    deallocate( m_MolNum           )
    deallocate( a_PtclName         )
    deallocate( a_PrescPtclDens    )

    if ( FlagDiurnalMean ) then
      deallocate( a_DiurnalMeanHA )
      deallocate( a_SZAGW         )
    end if
    deallocate( xya_SZADeg )

    rad_Earth_V3_inited = .false.

  end subroutine RadEarthV3Finalize

  !--------------------------------------------------------------------------------------

end module rad_Earth_V3
