!= Taguchi and Yoden (2002a) による強制と散逸
!
!= Forcing and dissipation suggested by Taguchi and Yoden (2002a)
!
! Authors::   Satoshi NODA
! Version::   $Id: ty2002a_forcing.f90,v 1.14 2012-04-27 11:24:45 noda Exp $ 
! Tag Name::  $Name: dcpam5-20121128 $
! Copyright:: Copyright (C) GFD Dennou Club, 2013. All rights reserved.
! License::   See COPYRIGHT[link:../../../COPYRIGHT]
!

module ty2002a_forcing
  !
  != Taguchi and Yoden (2002a) による強制と散逸
  !
  != Forcing and dissipation suggested by Taguchi and Yoden (2002a)
  !
  ! <b>Note that Japanese and English are described in parallel.</b>
  !
  ! Taguchi and Yoden (2002a) で提案される乾燥大気 GCM 用の
  ! 強制と散逸を計算します. 
  ! 与える強制と散逸として, 温度場の帯状対称場への簡単なニュートン冷却と, 
  ! 境界層摩擦を表現する下層風のレイリー摩擦を用います. 
  !
  ! Forcing and dissipation for dry air GCM
  ! suggested by Taguchi and Yoden (2002a) are caluclate.
  ! We use simple Newtonian relaxation of the temperature field to a
  ! zonally symmetric state and Rayleigh damping of low-level winds to
  ! represent boundary-layer friction.
  !
  !
  !== Procedures List
  ! 
  ! TY2002aForcingInit     :: 初期化
  ! TY2002aForcingInit2    :: 初期化その 2 (地形補正)
  ! TY2002aForcing         :: 強制と散逸の計算
  ! TY2002aForcingFinalize :: 終了処理 (モジュール内部の変数の割り付け解除)
  ! ------------  :: ------------
  ! TY2002aForcingInit     :: Initialization
  ! TY2002aForcingInit2    :: Initialization part 2 (correction by topography)
  ! TY2002aForcing         :: Calculate forcing and dissipation
  ! TY2002aForcingFinalize :: Termination (deallocate variables in this module)
  !
  !--
  !== NAMELIST
  !
  ! NAMELIST#ty2002a_forcing_nml
  !++
  !== References
  !
  ! * Taguchi, M., S. Yoden, 2002:
  !   Internal Interannual Variability of the Troposphere-Stratosphere
  !   Coupled System in a Simple Global Circulation Model.
  !   Part I: Parameter Sweep Experiment.
  !   <i>J. Atmos. Sci.</i>, <b>59</b>, 3021--3036.
  !

  ! モジュール引用 ; USE statements
  !

  ! 格子点設定
  ! Grid points settings
  !
  use gridset, only: imax, & ! 経度格子点数. 
                             ! Number of grid points in longitude
    &                jmax, & ! 緯度格子点数. 
                             ! Number of grid points in latitude
    &                kmax    ! 鉛直層数. 
                             ! Number of vertical level

  ! 種別型パラメタ
  ! Kind type parameter
  !
  use dc_types, only: DP, &      ! 倍精度実数型. Double precision. 
    &                 STRING, &  ! 文字列.       Strings. 
    &                 TOKEN      ! キーワード.   Keywords. 


  ! メッセージ出力
  ! Message output
  !
  use dc_message, only: MessageNotify

  ! 物理・数学定数設定
  ! Physical and mathematical constants settings
  !
  use constants0, only: PI
  use constants, only: Grav, &
                              ! $ g $ [m s-2]. 
                              ! 重力加速度. 
                              ! Gravitational acceleration
    & GasRDry, CpDry, RPlanet, Omega


  ! 宣言文 ; Declaration statements
  !
  implicit none
  private

  ! 公開手続き
  ! Public procedure
  !
  public :: TY2002aForcing
  public :: TY2002aForcingInit
!  public :: TY2002aForcingInit2    ! 外部からは参照されない
  public :: TY2002aForcingFinalize

  ! 公開変数
  ! Public variables
  !

  ! 非公開変数
  ! Private variables
  !
  real(DP), parameter :: sec_in_a_day = 86400.0_DP
                              ! 1 日の秒数. 
                              ! Seconds in day. 

  logical, save :: ty2002a_forcing_inited = .false.
                              ! 初期設定フラグ. 
                              ! Initialization flag
  logical, save :: ty2002a_forcing_inited2 = .false.
                              ! 初期設定フラグその 2.
                              ! Initialization flag part 2

  real(DP), save:: Kappa            ! $ \kappa = R/C_p $ .
                              ! 気体定数の定圧比熱に対する比. Ratio of gas constant to specific heat

  real(DP), save:: InitDumpPeriod     ! 計算初期に強制の強さを変える期間 [日] (廃止予定)
  real(DP), save:: InitDumpMagForWind ! 計算初期のレイリー摩擦の強度 [倍] (廃止予定)
  real(DP), save:: InitDumpMagForTemp ! 計算初期のニュートン冷却の強度 [倍] (廃止予定)
  real(DP), save:: DayInAYear         ! 1 年の長さ [日]

  character(STRING), save:: TempEQSetting
                              ! 平衡温度の設定方法
                              ! Setting of equilibrium temperature
  character(STRING), save:: TempEQFile
                              ! 平衡温度のファイル名. 
                              ! File name of equilibrium temperature
  character(TOKEN) , save:: TempEQName
                              ! 平衡温度の変数名. 
                              ! Variable name of equilibrium temperature

  real(DP), save:: ScaleHeight      ! 平衡温度を決めるのに用いるスケールハイト

  logical, save:: CorrectProgVarByTopo    ! 計算初回に予報変数の地形補正を行うか (廃止予定)

  real(DP), allocatable, save:: z_HeightRef (:)
                              ! 幾何学的高度. スケールハイトと sigma から求める.

  real(DP), allocatable, save:: xyz_TempEQNHWin (:,:,:)
                              ! 北半球冬至の平衡温度場 (地形補正入り)
                              ! Equilibrium temperature in north hemisphere of the winter solstice
  real(DP), allocatable, save:: xyz_TempEQNHSum (:,:,:)
                              ! 北半球夏至の平衡温度場 (地形補正入り)
                              ! Equilibrium temperature in north hemisphere of the summer solstice
  real(DP), allocatable, save:: xyz_TempEQNHWinWoTopo (:,:,:)
                              ! 北半球冬至の平衡温度場 (地形補正なし)
                              ! Equilibrium temperature in north hemisphere of the winter solstice without topography
  real(DP), allocatable, save:: xyz_TempEQNHSumWoTopo (:,:,:)
                              ! 北半球夏至の平衡温度場 (地形補正なし)
                              ! Equilibrium temperature in north hemisphere of the summer solstice without topography

  real(DP), allocatable, save:: yz_URef (:,:)
                              ! 成層圏の平衡温度を求めるための風の場.
                              ! Scott and Haynes (1998) に準拠.
  real(DP), allocatable, save:: yz_TempEQStr (:,:)
                              ! 成層圏の平衡温度場.
                              ! Scott and Haynes (1998) に準拠.
  real(DP), allocatable, save:: yz_TempEQTro (:,:)
                              ! 成層圏の平衡温度場.
                              ! Akahori and Yoden (1997) に準拠.

  real(DP), allocatable, save:: xyz_alpha (:,:,:)
                              ! ニュートン冷却の係数
                              ! 
  real(DP), allocatable, save:: xyz_kv (:,:,:)
                              ! レイリー摩擦の係数
                              ! 

  real(DP), allocatable, save:: xyz_HeightRef (:,:,:)
                              ! 基本場計算用高度 (地形入り) [m].
                              ! ジオポテンシャル高度ではない.

  character(*), parameter:: module_name = 'ty2002a_forcing'
                              ! モジュールの名称. 
                              ! Module name
  character(*), parameter:: version = &
    & '$Name: dcpam5-20121128 $' // &
    & '$Id: ty2002a_forcing.f90,v 1.14 2012-04-27 11:24:45 noda Exp $'
                              ! モジュールのバージョン
                              ! Module version

contains

  !--------------------------------------------------------------------------------------

  subroutine TY2002aForcing( &
    & xyz_U,    xyz_V,    xyz_Temp, & ! (in)
    & xyz_Press, xy_Ps, &                  ! (in)
    & xy_SurfHeight, &                       ! (in)
    & xyz_DUDt, xyz_DVDt, xyz_DTempDt &      ! (out)
    & )
    !
    ! 引数として与えられた東西風速 xyz_U, 南北風速 xyz_V, 
    ! 温度 xyz_Temp から, 
    ! 温度場の帯状対称場への簡単なニュートン冷却と
    ! 境界層摩擦を表現する下層風のレイリー摩擦による
    ! 風速と温度の変化率を求め, 
    ! xyz_DUDt, xyz_DVDt, xyz_DTempDt に返します. 
    !
    ! Tendencies by simple Newtonian relaxation of the temperature field to a
    ! zonally symmetric state and Rayleigh damping of low-level winds to
    ! represent boundary-layer friction are calculated 
    ! from eastward wind "xyz_U", northward wind "xyz_V", 
    ! temperature "xyz_Temp".
    ! And the tencencies are returned as 
    ! "xyz_DUDt", "xyz_DVDt", "xyz_DTempDt". 
    !
    !

    ! モジュール引用 ; USE statements
    !

    ! 座標データ設定
    ! Axes data settings
    !
    use axesset, only: &
      & y_Lat, &              ! $ \varphi $ [rad.] . 緯度. Latitude
      & z_Sigma               ! $ \sigma $ レベル (整数). 
                              ! Full $ \sigma $ level

    ! 時刻管理
    ! Time control
    !
    use timeset, only: &
      & DelTime, &            ! $ \Delta t $
      & TimeN, &              ! ステップ $ t $ の時刻. Time of step $ t $. 
      & TimesetClockStart, TimesetClockStop

    ! ヒストリデータ出力
    ! History data output
    !
    use gtool_historyauto, only: HistoryAutoPut

    ! 時系列データの読み込み
    ! Reading time series
    !
    use read_time_series, only: SetValuesFromTimeSeriesWrapper

    ! 宣言文 ; Declaration statements
    !
    implicit none

    real(DP), intent(inout):: xyz_U (0:imax-1, 1:jmax, 1:kmax)
                              ! $ u $ . 東西風速. 
                              ! Eastward wind
    real(DP), intent(inout):: xyz_V (0:imax-1, 1:jmax, 1:kmax)
                              ! $ v $ . 南北風速. 
                              ! Northward wind
    real(DP), intent(inout):: xyz_Temp (0:imax-1, 1:jmax, 1:kmax)
                              ! $ T $ . 温度. 
                              ! Temperature
    real(DP), intent(inout):: xyz_Press(0:imax-1, 1:jmax, 1:kmax)
                              ! $ p $ . 気圧.
                              ! Pressure
    real(DP), intent(inout):: xy_Ps (0:imax-1, 1:jmax)
                              ! $ p_s $ . 地表面気圧. Surface pressure (Pa)
    real(DP), intent(in):: xy_SurfHeight (0:imax-1, 1:jmax)
                              ! $ z $ . 地表面高度.
                              ! Surface Height
    real(DP), intent(out):: xyz_DUDt (0:imax-1, 1:jmax, 1:kmax)
                              ! $ \DP{u}{t} $ . 東西風速変化. 
                              ! Eastward wind tendency
    real(DP), intent(out):: xyz_DVDt (0:imax-1, 1:jmax, 1:kmax)
                              ! $ \DP{v}{t} $ . 南北風速変化. 
                              ! Northward wind tendency
    real(DP), intent(out):: xyz_DTempDt (0:imax-1, 1:jmax, 1:kmax)
                              ! $ \DP{T}{t} $ . 温度変化. 
                              ! Temperature tendency

    ! 作業変数
    ! Work variables
    !
    real(DP):: xyz_TempEQ (0:imax-1, 1:jmax, 1:kmax)
                              ! $ T_{eq} $ . 平衡温度. 
                              ! Equilibrium temperature
!    real(DP):: xyz_TempEQWoTopo (0:imax-1, 1:jmax, 1:kmax)
!                              ! ファイルから読み込んだ平衡温度 (地形補正なし)

    real(DP):: splA(1:max(jmax,kmax)-1)
    real(DP):: splB(1:max(jmax,kmax)-1)
    real(DP):: splC(1:max(jmax,kmax)-1)
    real(DP):: splD(1:max(jmax,kmax)-1)
    real(DP):: splX(1:max(jmax,kmax))
    real(DP):: splY(1:max(jmax,kmax))

    real(DP):: sec_in_a_year
    real(DP):: season
    real(DP):: r
    real(DP):: mag

    integer:: i               ! 経度方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in longitude
    integer:: j               ! 緯度方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in latitude
    integer:: k               ! 鉛直方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in vertical direction
    integer:: m


    ! 実行文 ; Executable statement
    !

    ! 初期化確認
    ! Initialization check
    !
    if ( .not. ty2002a_forcing_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if

    ! 地形補正は別途行う
    if ( .not. ty2002a_forcing_inited2 ) then
      if ( TempEQSetting == 'Taguchi and Yoden (2002a)' ) then
        call TY2002aForcingInit2(xy_SurfHeight, &
          & xyz_U, xyz_V, xyz_Temp, xyz_Press, xy_Ps)
      end if
      ty2002a_forcing_inited2 = .true.
    end if

    ! 計算時間計測開始
    ! Start measurement of computation time
    !
    call TimesetClockStart( module_name )

    ! 平衡温度
    ! Equilibrium temperature
    !

    ! 現在の季節の平衡温度
    if ( TempEQSetting == 'Taguchi and Yoden (2002a)' ) then

!!! debug
!    print *, "DayInAYear", DayInAYear
    
      if (DayInAYear > 0.0_DP) then
        sec_in_a_year = sec_in_a_day * DayInAYear
        season = TimeN / sec_in_a_year
        r = 0.5 * (1.0_DP + cos(2.0_DP * PI * season)) 
        xyz_TempEQ = &
          & r * xyz_TempEQNHWin + (1.0_DP - r) * xyz_TempEQNHSum
      else

!!! debug
!      print *, "perpetual winter"
      
        ! perpetual winter
        xyz_TempEQ = xyz_TempEQNHWin
      end if

    ! ファイルで与える場合は季節変化, 地形補正していることを前提
    else if ( TempEQSetting == 'file' ) then

      ! 引数の中の "O3" はデータのタイプに相当.
      ! 現在は O3 の読み込み方を真似ている.
      ! 本当なら read_time_series.f90 の SetValFromTimeSeriesWrapper3D に
      ! "TempEQ" を追加すべき
      call SetValuesFromTimeSeriesWrapper( &
        & TempEQFile, "TempEQ",          & ! (in)
        & xyz_Press,                     & ! (in)
        & xyz_TempEQ,                    & ! (inout)
        & "O3"                           &
        & )

    end if

    ! 地形と初期値の不一致に起因する計算破綻を抑えるため,
    ! 初期に強制の強さを変える場合の処理 (実験的に導入したもの. 廃止予定)
    ! 本来は init_data.f90 で, 地形に合わせた初期値を作成する必要があるだろう.
    if (TimeN < InitDumpPeriod) then

      ! 全層に強い Dumping を適用
      ! 係数はとりあえず埋め込み
      xyz_U = 0.1 * xyz_U
      xyz_V = 0.1 * xyz_V

      xyz_Temp = 250.0

!      mag = 1.0_DP + (InitDumpMagForTemp - 1.0) * (InitDumpPeriod - TimeN) / InitDumpPeriod
!      xyz_alpha = mag * xyz_alpha

!      mag = 1.0_DP + (InitDumpMagForWind - 1.0) * (InitDumpPeriod - TimeN) / InitDumpPeriod
!      xyz_kv = mag * xyz_kv
    endif

    ! 温度 $ T $ へニュートン冷却を適用
    ! Apply Newtonian relaxation to temperature $ T $
    !

    xyz_DTempDt =  - xyz_alpha * (xyz_Temp - xyz_TempEQ)

    ! 東西風速 $ u $ と南北風速 $ v $ へレイリー摩擦を適用
    ! Apply Rayleigh damping to eastward wind $ u $ and northward wind $ v $
    !

    xyz_DUDt = - xyz_kv * xyz_U
    xyz_DVDt = - xyz_kv * xyz_V

    ! ヒストリデータ出力
    ! History data output
    !
    call HistoryAutoPut( TimeN, 'DUDtTY2002a',    xyz_DUDt )
    call HistoryAutoPut( TimeN, 'DVDtTY2002a',    xyz_DVDt )
    call HistoryAutoPut( TimeN, 'DTempDtTY2002a', xyz_DTempDt )
    call HistoryAutoPut( TimeN, 'TempEQ',         xyz_TempEQ )
    if (TempEQSetting == 'Taguchi and Yoden (2002a)') then
      call HistoryAutoPut( TimeN, 'TempEQStr',      yz_TempEQStr )
      call HistoryAutoPut( TimeN, 'TempEQTro',      yz_TempEQTro )
      call HistoryAutoPut( TimeN, 'URef',           yz_URef )
      call HistoryAutoPut( TimeN, 'TempEQNHWin',    xyz_TempEQNHWin )
      call HistoryAutoPut( TimeN, 'TempEQNHWinWoTopo', xyz_TempEQNHWinWoTopo )
      if (DayInAYear > 0.0_DP) then
        call HistoryAutoPut( TimeN, 'TempEQNHSum',    xyz_TempEQNHSum )
        call HistoryAutoPut( TimeN, 'TempEQNHSumWoTopo', xyz_TempEQNHSumWoTopo )
      end if
!    else if (TempEQSetting == 'file') then
!      call HistoryAutoPut( TimeN, 'TempEQWoTopo',   xyz_TempEQWoTopo )
    end if
    call HistoryAutoPut( TimeN, 'HeightRef',      xyz_HeightRef )
    call HistoryAutoPut( TimeN, 'SurfHeight',     xy_SurfHeight )
    call HistoryAutoPut( TimeN, 'RadRelaxRate',   xyz_alpha )
    call HistoryAutoPut( TimeN, 'RayleighRelaxRate', xyz_kv )

    ! 計算時間計測一時停止
    ! Pause measurement of computation time
    !
    call TimesetClockStop( module_name )

  end subroutine TY2002aForcing

  !--------------------------------------------------------------------------------------

  ! solve coefficients of spline function
  subroutine solve_cubic_spline_intpl(a,b,c,d,x,y,n)
    integer:: n
    real(DP), intent(out):: a(1:n-1)
    real(DP), intent(out):: b(1:n-1)
    real(DP), intent(out):: c(1:n-1)
    real(DP), intent(out):: d(1:n-1)
    real(DP), intent(in)::  x(1:n)
    real(DP), intent(in)::  y(1:n)
    real(DP):: h(1:n-1)
    real(DP):: v(1:n-1)
    real(DP):: a1(1:n-2)
    real(DP):: b1(1:n-2)
    real(DP):: c1(1:n-2)
    real(DP):: d1(1:n-2)
    real(DP):: u1(1:n-2)
    real(DP):: u(1:n)
    integer:: i

    h(1) = x(2) - x(1)
    v(1) = 0.0_DP
    do i = 2, n-1
      h(i) = x(i+1) - x(i)        
      v(i) = 6.0_DP * ( (y(i+1)-y(i))/h(i) - (y(i)-y(i-1))/h(i-1) )
    end do

    a1(1) = 0.0_DP
    do i = 1, n-3
      a1(i+1) = h(i)
      c1(i)   = h(i)
    end do
    c1(n-2) = 0.0_DP
    do i = 1, n-2
      b1(i) = 2.0_DP * (h(i) + h(i+1))
      d1(i) = v(i+1)
    end do

    call solve_tridiag(a1,b1,c1,d1,u1,n-2)

    u(1) = 0.0_DP
    u(2:n-1) = u1
    u(n) = 0.0_DP

    b = 0.5_DP * u
    do i = 1, n-1
      a(i) = (u(i+1) - u(i)) / (6.0_DP * h(i))
      c(i) = (y(i+1) - y(i)) / h(i) &
        & - h(i) * (2.0_DP * u(i) + u(i+1)) / 6.0_DP
      d(i) = y(i)
    end do
  end subroutine solve_cubic_spline_intpl

  !--------------------------------------------------------------------------------------

  ! solve tridiagonal matrix by using Thomas algorithm
  subroutine  solve_tridiag(a,b,c,d,x,n)
    integer:: n
    real(DP), intent(in):: a(1:n)
    real(DP), intent(in):: b(1:n)
    real(DP), intent(in):: c(1:n)
    real(DP), intent(in):: d(1:n)
    real(DP), intent(out):: x(1:n)
    real(DP):: c1(1:n)
    real(DP):: d1(1:n)
    integer:: i
    real(DP):: w

    c1(1) = c(1) / b(1)
    d1(1) = d(1) / b(1)
    do i = 2, n-1
      w = 1.0_DP / (b(i) - c1(i-1)*a(i))
      c1(i) = c(i) * w
      d1(i) = (d(i) - d1(i-1)*a(i)) * w
    end do
    d1(n) = (d(n) - d1(n-1)*a(n)) / (b(n) - c1(n-1)*a(n))

    x(n) = d1(n)
    do i = n-1, 1, -1
      x(i) = d1(i) - c1(i)*x(i+1)
    end do
  end subroutine solve_tridiag

  !--------------------------------------------------------------------------------------

  ! スプライン関数の値を返す
  ! 予め係数 a,b,c,d を求めておくこと
  function cubic_spline_intpl(a,b,c,d,n,x,xx)
    integer:: n
    real(DP), intent(in):: a(1:n-1)
    real(DP), intent(in):: b(1:n-1)
    real(DP), intent(in):: c(1:n-1)
    real(DP), intent(in):: d(1:n-1)
    real(DP), intent(in):: x(1:n)
    real(DP):: xx, dx, cubic_spline_intpl
    integer:: i, i1, i2

    i = search_grid(x,xx,n-1)

    dx = xx-x(i)
    cubic_spline_intpl = a(i)*dx**3 + b(i)*dx**2 + c(i)*dx + d(i)
  end function cubic_spline_intpl

  !--------------------------------------------------------------------------------------

  ! スプライン関数の微分
  function cubic_spline_intpl_deriv(a,b,c,d,n,x,xx)
    integer:: n
    real(DP), intent(in):: a(1:n-1)
    real(DP), intent(in):: b(1:n-1)
    real(DP), intent(in):: c(1:n-1)
    real(DP), intent(in):: d(1:n-1)
    real(DP), intent(in):: x(1:n)
    real(DP):: xx, dx, cubic_spline_intpl_deriv
    integer:: i

    i = search_grid(x,xx,n-1)

    dx = xx-x(i)
    cubic_spline_intpl_deriv = 3.0_DP * a(i)*dx**2 + 2.0_DP * b(i)*dx + c(i)
  end function cubic_spline_intpl_deriv

  !--------------------------------------------------------------------------------------

  ! スプライン関数の積分
  function cubic_spline_intpl_integ(a,b,c,d,n,x,x1,x2,yy)
    integer:: n
    real(DP), intent(in):: a(1:n-1)
    real(DP), intent(in):: b(1:n-1)
    real(DP), intent(in):: c(1:n-1)
    real(DP), intent(in):: d(1:n-1)
    real(DP), intent(in):: x(1:n)
    real(DP):: x1, x2, yy
    real(DP):: xx, xmin, xmax, dx
    real(DP):: cubic_spline_intpl_integ
    integer:: i, i1, i2
    
    ! x1 > x2 の場合は, 小さい方から積分して最後にマイナスを掛ける
    xmin = min(x1, x2)
    xmax = max(x1, x2)
    i1 = search_grid(x,xmin,n-1)
    i2 = search_grid(x,xmax,n-1)
    i = i1
    xx = xmin

    cubic_spline_intpl_integ = 0.0_DP
    if (xmin < x(1)) then
      if (xmax < x(1)) then
        cubic_spline_intpl_integ = cubic_spline_intpl_integ &
          & + f(a(1),b(1),c(1),d(1),xmax-x(1)) &
          & - f(a(1),b(1),c(1),d(1),xmin-x(1))
      else
        cubic_spline_intpl_integ = cubic_spline_intpl_integ &
          & - f(a(1),b(1),c(1),d(1),xmin)
      end if
      i = 1
      xx = x(1)
    end if
    do while (i <= i2 .and. i <= n-1)
      cubic_spline_intpl_integ = cubic_spline_intpl_integ &
        & + f(a(i),b(i),c(i),d(i),min(xmax,x(i+1))-x(i)) &
        & - f(a(i),b(i),c(i),d(i),xx-x(i))
      i = i+1
      xx = x(i)
    end do
    if (xmax > x(n)) then
      if (xmin > x(n)) then
        cubic_spline_intpl_integ = cubic_spline_intpl_integ &
          & + f(a(n-1),b(n-1),c(n-1),d(n-1),xmax-x(n)) &
          & - f(a(n-1),b(n-1),c(n-1),d(n-1),xmin-x(n))
      else
        cubic_spline_intpl_integ = cubic_spline_intpl_integ &
          & + f(a(n-1),b(n-1),c(n-1),d(n-1),xmax-x(n))
      end if
    end if

    if (x1 > x2) then
      cubic_spline_intpl_integ = - cubic_spline_intpl_integ
    end if
    cubic_spline_intpl_integ = cubic_spline_intpl_integ + yy

  contains
    ! スプライン関数を積分した関数
    function f(a,b,c,d,x)
      real(DP):: a,b,c,d,x,f
      f = 0.25*a*x**4 + (1.0_DP/3.0_DP)*b*x**3 + 0.5*c*x**2 + d*x
    end function f
  end function cubic_spline_intpl_integ

  !--------------------------------------------------------------------------------------

  ! 二分法で値の近い格子点を探索
  ! x の中は昇順であることを仮定
  function search_grid(x,xx,n)
    integer:: n
    real(DP):: x(n), xx
    integer:: search_grid
    integer:: i, i1, i2

    if (xx <= x(1)) then
      search_grid = 1
    elseif (xx >= x(n)) then
      search_grid = n
    else
      i1 = 1
      i2 = n
      i = n / 2
      do while ((i2-i1) > 1)
        if (xx > x(i)) then
          i1 = i
          i = (i1 + i2) / 2
        else
          i2 = i
          i = (i1 + i2) / 2
        end if
      end do
      search_grid = i
    end if
  end function search_grid

  !--------------------------------------------------------------------------------------

  subroutine TY2002aForcingInit
    !
    ! ty2002a_forcing モジュールの初期化を行います. 
    ! NAMELIST#ty2002a_forcing_nml の読み込みはこの手続きで行われます. 
    !
    ! "ty2002a_forcing" module is initialized. 
    ! "NAMELIST#ty2002a_forcing_nml" is loaded in this procedure. 
    !

    ! モジュール引用 ; USE statements
    !

    ! 物理定数設定
    ! Physical constants settings
    !
    use constants, only: &
      & GasRDry, &
                              ! $ R $ [J kg-1 K-1]. 
                              ! 乾燥大気の気体定数. 
                              ! Gas constant of air
      & CpDry
                              ! $ C_p $ [J kg-1 K-1]. 
                              ! 乾燥大気の定圧比熱. 
                              ! Specific heat of air at constant pressure

    ! 座標データ設定
    ! Axes data settings
    !
    use axesset, only: &
      & x_Lon, &              ! $ \lambda $ [rad.] . 経度. Longitude
      & y_Lat, &              ! $ \varphi $ [rad.] . 緯度. Latitude
      & z_Sigma               ! $ \sigma $ レベル (整数). 
                              ! Full $ \sigma $ level

    ! NAMELIST ファイル入力に関するユーティリティ
    ! Utilities for NAMELIST file input
    !
    use namelist_util, only: namelist_filename, NmlutilMsg, NmlutilAryValid

    ! ファイル入出力補助
    ! File I/O support
    !
    use dc_iounit, only: FileOpen

    ! 種別型パラメタ
    ! Kind type parameter
    !
    use dc_types, only: STDOUT ! 標準出力の装置番号. Unit number of standard output

    ! 文字列操作
    ! Character handling
    !
    use dc_string, only: StoA

    ! ヒストリデータ出力
    ! History data output
    !
    use gtool_historyauto, only: HistoryAutoAddVariable

    ! 宣言文 ; Declaration statements
    !
    implicit none

                              ! 初回を示すフラグ.
                              ! Flag that indicates first loop
                              !
    logical, save:: flag_first_TempEQ = .true.

    integer:: i               ! 経度方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in longitude
    integer:: j               ! 緯度方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in latitude
    integer:: k               ! 鉛直方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in vertical direction

    integer:: unit_nml        ! NAMELIST ファイルオープン用装置番号. 
                              ! Unit number for NAMELIST file open
    integer:: iostat_nml      ! NAMELIST 読み込み時の IOSTAT. 
                              ! IOSTAT of NAMELIST read

    ! NAMELIST 変数群
    ! NAMELIST group name
    !
    namelist /ty2002a_forcing_nml/ &
      & InitDumpPeriod,        &   ! 計算初期に強制の強さを変える期間 [日] (廃止予定)
      & InitDumpMagForWind,    &   ! 計算初期のレイリー摩擦の強度 [倍] (廃止予定)
      & InitDumpMagForTemp,    &   ! 計算初期のニュートン冷却の強度 [倍] (廃止予定)
      & DayInAYear, &              ! 1 年の長さ [日]. ゼロまたは負の値を指定すると永続的な冬至になる.
      & TempEQSetting, &           ! 平衡温度の設定方法.
      & TempEQFile,   &            ! 平衡温度のファイル名. 
      & TempEQName, &              ! 平衡温度の変数名. 
      & ScaleHeight, &             ! スケールハイト [m]
      & CorrectProgVarByTopo       ! 計算初回に予報変数の地形補正を行うか (廃止予定)
          !
          ! デフォルト値については初期化手続 "ty2002a_forcing#TY2002aForcingInit" 
          ! のソースコードを参照のこと. 
          !
          ! Refer to source codes in the initialization procedure
          ! "ty2002a_forcing#TY2002aForcingInit" for the default values. 
          !

    ! 実行文 ; Executable statement
    !

    if ( ty2002a_forcing_inited ) return


    ! デフォルト値の設定
    ! Default values settings
    !
    InitDumpPeriod = 10.0_DP     ! day
    InitDumpMagForWind = 1.0_DP
    InitDumpMagForTemp = 1.0_DP
    DayInAYear     = 360.0_DP   ! Taguchi and Yoden (2002a)
    TempEQSetting = 'Taguchi and Yoden (2002a)'
    TempEQFile = ''
    TempEQName = ''
    ! Scott and Haynes (1998), 240 K 等温大気の場合
    ScaleHeight = 7000.0_DP
    CorrectProgVarByTopo = .false.    ! 計算初回に予報変数の地形補正を行うか (廃止予定)

    ! NAMELIST の読み込み
    ! NAMELIST is input
    !
    if ( trim(namelist_filename) /= '' ) then
      call FileOpen( unit_nml, &          ! (out)
        & namelist_filename, mode = 'r' ) ! (in)

      rewind( unit_nml )
      read( unit_nml, &                ! (in)
        & nml = ty2002a_forcing_nml, &  ! (out)
        & iostat = iostat_nml )        ! (out)
      close( unit_nml )

      call NmlutilMsg( iostat_nml, module_name ) ! (in)
!$      if ( iostat_nml == 0 ) write( STDOUT, nml = ty2002a_forcing_nml )
    end if

    ! 係数の設定
    ! Configure coefficients
    !
    Kappa = GasRDry / CpDry
    InitDumpPeriod = InitDumpPeriod * sec_in_a_day

    ! sigma から幾何学的高度を計算
    allocate( z_HeightRef (1:kmax) )
    z_HeightRef = - ScaleHeight * log(z_Sigma)

    allocate( xyz_alpha     (0:imax-1, 1:jmax, 1:kmax))
    allocate( xyz_kv        (0:imax-1, 1:jmax, 1:kmax))
    allocate( xyz_HeightRef (0:imax-1, 1:jmax, 1:kmax))

    ! 平衡温度
    ! Equilibrium temperature
    !
    if ( TempEQSetting == 'Taguchi and Yoden (2002a)' ) then
      if ( flag_first_TempEQ ) then

        allocate( yz_URef      (1:jmax, 1:kmax) )
        allocate( yz_TempEQStr (1:jmax, 1:kmax) )
        allocate( yz_TempEQTro (1:jmax, 1:kmax) )
        allocate( xyz_TempEQNHWin       (0:imax-1, 1:jmax, 1:kmax) )
        allocate( xyz_TempEQNHWinWoTopo (0:imax-1, 1:jmax, 1:kmax) )

        ! 北半球冬至の平衡温度. AGCM5 における tbasic.F の処理に相当.
        call SetTempEQNHWinTY02a( &
          & z_HeightRef, &    ! (in)
          & xyz_TempEQNHWinWoTopo )  ! (out)

        ! 北半球夏至の平衡温度
        if (DayInAYear > 0.0_DP) then
          allocate( xyz_TempEQNHSum       (0:imax-1, 1:jmax, 1:kmax))
          allocate( xyz_TempEQNHSumWoTopo (0:imax-1, 1:jmax, 1:kmax))
          do j = 1, jmax
            xyz_TempEQNHSumWoTopo(:,j,:) = xyz_TempEQNHWinWoTopo(:,jmax-j+1,:)
          enddo
        end if

      end if
    else if ( TempEQSetting == 'file' ) then
      ! データをファイルから取得
      ! Data is input from files
      !
      if ( flag_first_TempEQ ) then

        ! 何もしない

      end if
    else
      call MessageNotify( 'E', module_name, &
        & ' TempEQSetting = %c is not appropriate.', &
        & c1 = trim(TempEQSetting) )
    end if
    
    flag_first_TempEQ = .false.

    ! ヒストリデータ出力のためのへの変数登録
    ! Register of variables for history data output
    !
    call HistoryAutoAddVariable( 'DUDtTY2002a', &
      & (/ 'lon ', 'lat ', 'sig ', 'time' /), &
      & 'eastward wind tendency', 'm s-2' )
    call HistoryAutoAddVariable( 'DVDtTY2002a', &
      & (/ 'lon ', 'lat ', 'sig ', 'time' /), &
      & 'northward wind tendency', 'm s-2' )
    call HistoryAutoAddVariable( 'DTempDtTY2002a', &
      & (/ 'lon ', 'lat ', 'sig ', 'time' /), &
      & 'temperature tendency', 'K s-1' )
    call HistoryAutoAddVariable( 'TempEQ', &
      & (/ 'lon ', 'lat ', 'sig ', 'time' /), &
      & 'equilibrium temperature', 'K' )
    if (TempEQSetting == 'file') then
      call HistoryAutoAddVariable( 'TempEQWoTopo', &
        & (/ 'lon ', 'lat ', 'sig ', 'time' /), &
        & 'equilibrium temperature without topography', 'K' )
    else if (TempEQSetting == 'Taguchi and Yoden (2002a)') then
      call HistoryAutoAddVariable( 'URef', &
        & (/ 'lat ', 'sig ', 'time' /), &
        & 'reference eastward wind for stratosphere', 'm s-1' )
      call HistoryAutoAddVariable( 'TempEQStr', &
        & (/ 'lat ', 'sig ', 'time' /), &
        & 'equilibrium temperature for stratosphere', 'K' )
      call HistoryAutoAddVariable( 'TempEQTro', &
        & (/ 'lat ', 'sig ', 'time' /), &
        & 'equilibrium temperature for troposphere', 'K' )
      call HistoryAutoAddVariable( 'TempEQNHWin', &
        & (/ 'lon ', 'lat ', 'sig ', 'time' /), &
        & 'equilibrium temperature of NH winter', 'K' )
      call HistoryAutoAddVariable( 'TempEQNHWinWoTopo', &
        & (/ 'lon ', 'lat ', 'sig ', 'time' /), &
        & 'equilibrium temperature of NH winter without topography', 'K' )
      if (DayInAYear > 0.0_DP) then
        call HistoryAutoAddVariable( 'TempEQNHSum', &
          & (/ 'lon ', 'lat ', 'sig ', 'time' /), &
          & 'equilibrium temperature of NH summer', 'K' )
        call HistoryAutoAddVariable( 'TempEQNHSumWoTopo', &
          & (/ 'lon ', 'lat ', 'sig ', 'time' /), &
          & 'equilibrium temperature of NH summer without topography', 'K' )
      end if
    end if
    call HistoryAutoAddVariable( 'HeightRef', &
      & (/ 'lon ', 'lat ', 'sig ', 'time' /), &
      & 'reference height for equilibrium temperature etc.', 'm' )
    call HistoryAutoAddVariable( 'SurfHeight', &
      & (/ 'lon ', 'lat ', 'time' /), &
      & 'surface height', 'm' )
    call HistoryAutoAddVariable( 'RadRelaxRate', &
      & (/ 'lon ', 'lat ', 'sig ', 'time' /), &
      & 'radiative relaxation rate', 's-1' )
    call HistoryAutoAddVariable( 'RayleighRelaxRate', &
      & (/ 'lon ', 'lat ', 'sig ', 'time' /), &
      & 'Rayleigh friction relaxation rate', 's-1' )

    ! 印字 ; Print
    !
    call MessageNotify( 'M', module_name, '----- Initialization Messages -----' )
    call MessageNotify( 'M', module_name, 'InitDumpPeriod       = %f', d = (/ InitDumpPeriod       /) )
    call MessageNotify( 'M', module_name, 'InitDumpMagForWind   = %f', d = (/ InitDumpMagForWind   /) )
    call MessageNotify( 'M', module_name, 'InitDumpMagForTemp   = %f', d = (/ InitDumpMagForTemp   /) )
    call MessageNotify( 'M', module_name, 'DayInAYear           = %f', d = (/ DayInAYear           /) )
    call MessageNotify( 'M', module_name, 'TempEQSetting        = %c', c1 = trim(TempEQSetting) )
    call MessageNotify( 'M', module_name, 'TempEQFile           = %c', c1 = trim(TempEQFile) )
    call MessageNotify( 'M', module_name, 'TempEQName           = %c', c1 = trim(TempEQName) )
    call MessageNotify( 'M', module_name, 'ScaleHeight          = %f', d = (/ ScaleHeight          /) )
    call MessageNotify( 'M', module_name, 'CorrectProgVarByTopo = %b', l = (/ CorrectProgVarByTopo /) )
    call MessageNotify( 'M', module_name, '-- version = %c', c1 = trim(version) )

    ty2002a_forcing_inited = .true.

  end subroutine TY2002aForcingInit

  !--------------------------------------------------------------------------------------

  subroutine TY2002aForcingInit2(xy_SurfHeight, &    ! in
    & xyz_U, xyz_V, xyz_Temp, xyz_Press, xy_Ps)      ! inout
    !
    ! 平衡温度, Newton 冷却やレイリー摩擦の係数の地形補正.
    ! 地表面高度の設定との依存関係を考慮し,
    ! 各種モジュールの初期化が終わった, メインループ内で行うようにした.
    !
    ! ユーザの指定により予報変数の地形補正も行う (廃止予定).
    ! 

    real(DP), intent(in):: xy_SurfHeight (0:imax-1, 1:jmax)
                              ! $ z $ . 地表面高度.
                              ! Surface Height
    real(DP), intent(inout):: xyz_U (0:imax-1, 1:jmax, 1:kmax)
                              ! $ u $ . 東西風速. 
                              ! Eastward wind
    real(DP), intent(inout):: xyz_V (0:imax-1, 1:jmax, 1:kmax)
                              ! $ v $ . 南北風速. 
                              ! Northward wind
    real(DP), intent(inout):: xyz_Temp (0:imax-1, 1:jmax, 1:kmax)
                              ! $ T $ . 温度. 
                              ! Temperature
    real(DP), intent(inout):: xyz_Press(0:imax-1, 1:jmax, 1:kmax)
                              ! $ p $ . 気圧.
                              ! Pressure
    real(DP), intent(inout):: xy_Ps (0:imax-1, 1:jmax)
                              ! $ p_s $ . 地表面気圧. Surface pressure (Pa)

    real(DP):: z_alphaRef (1:kmax)
    real(DP):: z_kvRef (1:kmax)
    real(DP):: z_tmp (1:kmax)
    real(DP):: xyz_tmp (0:imax-1, 1:jmax, 1:kmax)
    integer:: i,j,k

    if ( ty2002a_forcing_inited2 ) return

    do k = 1, kmax
      do j = 1, jmax
        do i = 0, imax-1
          xyz_HeightRef(i,j,k) = z_HeightRef(k) + xy_SurfHeight(i,j)
        end do
      end do
    end do

    ! 平衡温度の地形補正
    if ( TempEQSetting == 'Taguchi and Yoden (2002a)' ) then
      call CorrectbyTopo3D(xyz_TempEQNHWinWoTopo, &
        & z_HeightRef, xyz_HeightRef, xyz_TempEQNHWin)
      if (DayInAYear > 0.0_DP) then
        call CorrectbyTopo3D(xyz_TempEQNHSumWoTopo, &
          & z_HeightRef, xyz_HeightRef, xyz_TempEQNHSum)
      end if
    end if

    ! ニュートン冷却の係数
    ! 地形補正にスプライン補間を使う
    z_alphaRef(:) = &
      & ( 1.5_DP &
      &   + tanh( (z_HeightRef(:) - 35000.0_DP)/7000.0_DP ) ) &
      & * 10.0_DP ** (-6)
    do j = 1, jmax
      do i = 0, imax-1
!        call CorrectByTopo1D(z_alphaRef, &
!          & z_HeightRef, xyz_HeightRef(i,j,:), xyz_alpha(i,j,:))
        call CorrectByTopo1D(z_alphaRef, &
          & z_HeightRef, xyz_HeightRef(i,j,:), z_tmp)
        xyz_alpha(i,j,:) = z_tmp
      enddo
    enddo

    ! レイリー摩擦の係数
    ! 地形補正にスプライン補間を使う
    z_kvRef(:) = max( 0.0_DP, &
      &           5.0_DP * 10.0**(-6) &
      &           * ( 1.02_DP - exp((50000.0_DP - z_HeightRef(:)) / 40000.0_DP) ) )
    do j = 1, jmax
      do i = 0, imax-1
!        call CorrectByTopo1D(z_kvRef, &
!          & z_HeightRef, xyz_HeightRef(i,j,:), xyz_kv(i,j,:))
        call CorrectByTopo1D(z_kvRef, &
          & z_HeightRef, xyz_HeightRef(i,j,:), z_tmp)
        xyz_kv(i,j,:) = z_tmp
      enddo
    enddo

    ! 最下層のみ時定数 0.5 日
    ! この時定数は 1 地球日の長さに依存することは
    ! 期待されていないと考え, sec_in_a_day を使わず 86400 sec を埋め込み
    xyz_kv(:,:,1) = 1.0_DP / (0.5_DP * 86400.0_DP)

    ! 予報変数の地形補正
    if (CorrectProgVarByTopo) then
      call CorrectbyTopo3D(xyz_U, &
        & z_HeightRef, xyz_HeightRef, xyz_tmp)
      xyz_U = xyz_tmp
      call CorrectbyTopo3D(xyz_V, &
        & z_HeightRef, xyz_HeightRef, xyz_tmp)
      xyz_V = xyz_tmp
      call CorrectbyTopo3D(xyz_Temp, &
        & z_HeightRef, xyz_HeightRef, xyz_tmp)
      xyz_Temp = xyz_tmp
      call CorrectbyTopo3D(xyz_Press, &
        & z_HeightRef, xyz_HeightRef, xyz_tmp)
      xyz_Press = xyz_tmp
      do j=1, jmax
        do i=0, imax-1
          xy_Ps(i,j) = xy_Ps(i,j) &
            & * exp( - Kappa * xy_SurfHeight(i,j) / ScaleHeight )
         ! debug
!          print *, i, j, xy_Ps(i,j)
        end do
      end do
    end if

    ty2002a_forcing_inited2 = .true.

  end subroutine TY2002aForcingInit2

  !--------------------------------------------------------------------------------------

  ! 北半球冬至の基本温度場を求める.
  ! AGCM5 における tbasic.F およびそこから呼ばれていたサブルーチンの処理に相当.
  subroutine SetTempEQNHWinTY02a( &
    & z_HeightRef, &    ! (in)
    & xyz_TempEQNHWin )  ! (out)

    ! モジュール引用 ; USE statements
    !

    ! 座標データ設定
    ! Axes data settings
    !
    use axesset, only: &
      & y_Lat, &              ! $ \varphi $ [rad.] . 緯度. Latitude
      & z_Sigma               ! $ \sigma $ レベル (整数). 
                              ! Full $ \sigma $ level

    ! 宣言文 ; Declaration statements
    !
    implicit none

    real(DP), intent(in):: z_HeightRef (1:kmax)
    real(DP), intent(out):: xyz_TempEQNHWin (0:imax-1, 1:jmax, 1:kmax)
                              ! 北半球冬至の平衡温度場
                              ! Equilibrium temperature in north hemisphere
                              ! of the winter solstice

    ! 作業変数
    ! Work variables
    !
    integer, parameter:: EDGE=5   ! yz_DUDz の端を求めるときのスプライン補間に使う格子点数
    integer, parameter:: KMAX_AY97=20   ! Akahori and Yoden (1997) の鉛直層数

    real(DP):: yz_TempEQNHWin (1:jmax, 1:kmax)

    ! AY97における諸情報
    Real(DP):: z_SigmaAY97(KMAX_AY97)
    DATA   z_SigmaAY97    / 0.963, 0.881, 0.782, 0.675, 0.572, 0.480,   &
      &                       0.403, 0.340, 0.288, 0.241, 0.196, 0.151, &
      &                       0.105, 0.0655,0.0371,0.0191,0.00908,      &
      &                       0.00374, 0.00148, 0.000415/
  
    Real(DP):: z_TempRefAY97(KMAX_AY97)
    DATA  z_TempRefAY97       / 283.1, 278.7, 272.8, 265.6, 257.6, 249.4, &
      &                       241.2, 233.4, 225.9, 219.3, 214.4, 211.3,   &
      &                       211.4, 214.7, 219.5, 226.4, 236.7,          &
      &                       254.1, 282.2, 356.5 /
    
    Real(DP):: z_delTempAY97(KMAX_AY97)
    DATA   z_delTempAY97   / 59.9, 59.6, 59.0, 57.7, 54.6, 49.6,  &
      &                       43.3, 36.7, 30.5, 22.4, 13.2,  4.6, &
      &                        0.0,  0.0,  0.0,  0.0,  0.0,       &
      &                        0.0,  0.0,  0.0 /

    real(DP):: a_u (1:3)
    real(DP):: a_b (1:3)
    real(DP):: a_phi (1:3)
    real(DP):: a_a (1:3)
    real(DP):: a_z (1:3)
    real(DP):: u0, b0, phi0, phi
    real(DP):: zB, zU
    real(DP):: yz_J (1:jmax, 1:kmax)   !  J1 + J2 + J3

    real(DP):: dz, w1, t1
    real(DP):: z_StaticStab (1:kmax)   ! 静的安定度 $N^2$

    real(DP):: TempRefSurfForStr
    real(DP):: StaticStabForStr1
    real(DP):: StaticStabForStr2
    real(DP):: StaticStabForStr3

    real(DP):: TempRefForStr(1:3)  ! 厳密解を求める際の積分定数

    real(DP):: TempRefSurfForTro
    real(DP):: StaticStabForTro1
    real(DP):: StaticStabForTro2

    real(DP):: z_TempRefStr (1:kmax)
    real(DP):: z_TempRefTro (1:kmax)

    real(DP):: z_delTemp(1:kmax)

    real(DP):: splA(1:max(jmax,max(kmax,KMAX_AY97))-1)
    real(DP):: splB(1:max(jmax,max(kmax,KMAX_AY97))-1)
    real(DP):: splC(1:max(jmax,max(kmax,KMAX_AY97))-1)
    real(DP):: splD(1:max(jmax,max(kmax,KMAX_AY97))-1)
    real(DP):: splX(1:max(jmax,max(kmax,KMAX_AY97)))
    real(DP):: splY(1:max(jmax,max(kmax,KMAX_AY97)))

    real(DP):: yz_DUDz (1:jmax, 1:kmax)
    real(DP):: yz_DTempDLat (1:jmax, 1:kmax)
    real(DP):: CoriolisParam(1:jmax)

    real(DP):: h1
    real(DP):: z1, z2
    real(DP):: w, w1, dx

    integer:: i               ! 経度方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in longitude
    integer:: j               ! 緯度方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in latitude
    integer:: k               ! 鉛直方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in vertical direction
    integer:: m


    ! 実行文 ; Executable statement
    !

    ! 平衡温度分布の計算は以下の順序で行う.
    ! 最初に北半球の冬の温度を成層圏と対流圏に分けて計算し,
    ! 両者を滑らかにつなぐことで北半球の冬の温度とする.

! --------------- ここから AGCM5 における sub_sh98.F の処理 ---------------

    ! 北半球の冬の成層圏の温度.
    ! Scott and Haynes (1998) の式による.
    ! まず速度分布を求めてから温度風バランスを用いて温度分布を出す.

! Scott and Haynes (1998) 論文
    a_u(1) = 340.0_DP   ; a_u(2) = -20.0_DP   ; a_u(3) = -220.0_DP
! TY02a オリジナルソース
!    a_u(1) = 340.0_DP   ; a_u(2) = 0.0_DP   ; a_u(3) = -220.0_DP

    a_b(1) = 0.04_DP    ; a_b(2) = 0.1_DP     ; a_b(3) = 0.03_DP
    a_phi(1) = 60.0_DP  ; a_phi(2) = 15.0_DP  ; a_phi(3) = -55.0_DP
    a_a(1) = 0.05_DP    ; a_a(2) = 0.3_DP     ; a_a(3) = 0.08_DP
    a_z(1) = 65.0_DP    ; a_z(2) = 30.0_DP    ; a_z(3) = 70.0_DP   ! 単位は km

    yz_J = 0.0_DP
    do m = 1, 3
      do k = 1, kmax
        do j = 1, jmax
          phi = 180.0_DP * y_Lat(j) / PI   ! radian -> degree
          yz_J(j,k) = yz_J(j,k) &
            & + a_u(m) * sech( a_b(m) * ( phi - a_phi(m) ) ) &
            & * sech( a_a(m) * ( 0.001_DP * z_HeightRef(k) - a_z(m) ) )
        enddo
      enddo
    enddo

    u0 = 20.0_DP
    b0 = 0.1_DP
    phi0 = 20.0_DP

    zB = 11400.0_DP   ! Scott and Haynes (1998)
    zU = 90000.0_DP   ! Scott and Haynes (1998)
    do k = 1, kmax
      do j = 1, jmax
        phi = 180.0_DP * y_Lat(j) / PI   ! radian -> degree
        yz_URef(j,k) = cos(y_Lat(j)) &
          & * cos( 0.5 * PI * (z_HeightRef(k) - zB)/(zU - zB) ) &
          & * ( u0 * tanh(b0 * (phi - phi0)) + yz_J(j,k) )
      enddo
    enddo

    TempRefSurfForStr = 250.0_DP

    ! 各高度における静的安定度
    StaticStabForStr1 = 1.3_DP * 10.0**(-4)
    ! TYY01 論文 (田口さん曰く, 論文のほうが間違っているらしい)
    StaticStabForStr2 = 5.0_DP * 10.0**(-4)
    ! TY02a オリジナルソース
!    StaticStabForStr2 = 4.5_DP * 10.0**(-4)
    StaticStabForStr3 = 2.5_DP * 10.0**(-4)

    ! 静的安定度より温位を計算
    z1 = 12000.0_DP
    z2 = 50000.0_DP

    ! 温度の式の積分定数を求める
    w = ScaleHeight**2 / (GasRDry * Kappa)

    ! 0 < z < z1 のときに用いる積分定数
    TempRefForStr(1) = TempRefSurfForStr - StaticStabForStr1 * w

    ! 高度 z1 における温度
    t1 = TempRefForStr(1) * exp(- Kappa * z1 / ScaleHeight) + StaticStabForStr1 * w
    ! z1 < z < z2 のときに用いる積分定数
    TempRefForStr(2) = (t1 - StaticStabForStr2 * w) * exp(Kappa * z1 / ScaleHeight)

    ! 高度 z2 における温度
    t1 = TempRefForStr(2) * exp(- Kappa * z2 / ScaleHeight) + StaticStabForStr2 * w
    ! z2 < z のときに用いる積分定数
    TempRefForStr(3) = (t1 - StaticStabForStr3 * w) * exp(Kappa * z2 / ScaleHeight)

    ! 厳密解
    do k = 1, kmax
      h1 = z_HeightRef(k)
      w1 = exp(- Kappa * h1 / ScaleHeight)
      if (h1 <= z1) then
        z_TempRefStr(k) = TempRefForStr(1) * w1 + StaticStabForStr1 * w
      elseif (h1 <= z2) then
        z_TempRefStr(k) = TempRefForStr(2) * w1 + StaticStabForStr2 * w
      else
        z_TempRefStr(k) = TempRefForStr(3) * w1 + StaticStabForStr3 * w
      end if
    enddo

!! --------------- ここから AGCM5 における sub_keidofuu.F の処理 ---------------
!
!    ! ----- 静的安定度を指定 -----
!
!    ! TY02a オリジナルソース
!    do k = 1, kmax
!      h1 = z_HeightRef(k)
!      if (h1 <= z1) then
!        z_StaticStab(k) = StaticStabForStr1
!      elseif (h1 <= z2) then
!        z_StaticStab(k) = StaticStabForStr2
!      else
!        z_StaticStab(k) = StaticStabForStr3
!      end if
!    enddo
!
!    ! ----- 静的安定度 (N^2) から phi_0 上での温度プロファイルを求める -----
!
!    ! 何のアルゴリズムを使っているのか不明.
!    ! 田口さんに伺ったが, 山家さん作成のものをそのまま使用されたので
!    ! 分からないとのこと.
!
!!! 論文通り (例えば田口 D 論 A.3 式) に実装すればこうなる
!!    k = 1
!!    dz = z_HeightRef(1)
!!    w1 = ( 2.0*ScaleHeight - Kappa*dz ) / ( 2.0*ScaleHeight + Kappa*dz)
!!    z_TempRefStr(k) = w1 * TempRefSurfForStr  &
!!    & + ( ScaleHeight/(2.0*GasRDry) )*( z_StaticStab(k)+z_StaticStab(k-1) )*dz
!
!! TY02a オリジナルソース (sub_keidofuu.F) ではこうなっていた
!    k = 1
!    z_TempRefStr(k) = TempRefSurfForStr
!
!    do k = 2, kmax
!      dz = z_HeightRef(k) - z_HeightRef(k-1)
!      w1 = ( 2.0*ScaleHeight - Kappa*dz ) / ( 2.0*ScaleHeight + Kappa*dz)
!      z_TempRefStr(k) = w1 * z_TempRefStr(k-1)  &
!      & + ( ScaleHeight/(2.0*GasRDry) )*( z_StaticStab(k)+z_StaticStab(k-1) )*dz
!    end do

    ! ----- U から DUDz を求める -----

    ! 2 次精度. 端の微分はスプライン補間で求める
    CoriolisParam = 2.0_DP * Omega * sin(y_Lat)
    do j = 1, jmax
      ! k = 1 の処理
      splX(1:EDGE) = z_HeightRef(1:EDGE)
      splY(1:EDGE) = yz_URef(j,1:EDGE)
      call solve_cubic_spline_intpl(splA,splB,splC,splD,splX,splY,EDGE)
      ! 端の微分
!!$      dx = splX(1) - splX(1)
!!$      yz_DUDz(j,1) = 3.0_DP * splA(1) * dx**2 &
!!$        & + 2.0_DP * splB(1) * dx + splC(1)
      yz_DUDz(j,1) = splC(1)  ! 上記はこの式と等価

      ! k = kmax の処理
      splX(1:EDGE) = z_HeightRef(kmax-EDGE+1:kmax)
      splY(1:EDGE) = yz_URef(j,kmax-EDGE+1:kmax)
      call solve_cubic_spline_intpl(splA,splB,splC,splD,splX,splY,EDGE)
      ! 端の微分
      dx = splX(EDGE) - splX(EDGE-1)
      yz_DUDz(j,kmax) = 3.0_DP * splA(EDGE-1) * dx**2 &
        & + 2.0_DP * splB(EDGE-1) * dx + splC(EDGE-1)

! debug
!      print *, (splX(i), splY(i), cubic_spline_intpl(splA,splB,splC,splD,EDGE,splX,splX(i)), i=1,EDGE)
!      print *, (splX(i), splY(i), cubic_spline_intpl(splA,splB,splC,splD,EDGE,splX,splX(i)-0.0001), i=1,EDGE)

      do k = 2, kmax-1
        yz_DUDz(j,k) = 0.5_DP * &
          & ( & 
          &   ( yz_URef(j,k+1)-yz_URef(j,k) ) &
          &     / (z_HeightRef(k+1) - z_HeightRef(k)) &
          &   + ( yz_URef(j,k)-yz_URef(j,k-1) ) &
          &     / (z_HeightRef(k) - z_HeightRef(k-1)) &
          & )
      end do
    end do

    ! ----- U と DUDz から DTempDtLat を求める -----

    do k = 1, kmax
      yz_DTempDLat(:,k) = &
        & - RPlanet * ScaleHeight / GasRDry &
        & * ( CoriolisParam &
        &    + 2.0_DP * yz_URef(:,k) * tan(y_Lat) / RPlanet ) &
        & * yz_DUDz(:,k)
    end do

    ! ----- 緯度 phi_0 における温度と DTempDLat から T を求める -----

    do k=1,kmax
      ! DTempDLat のスプライン関数の係数を求める
      splX(1:jmax) = y_Lat
      splY(1:jmax) = yz_DTempDLat(:,k)
      call solve_cubic_spline_intpl(splA,splB,splC,splD,splX,splY,jmax)

      ! 赤道に最も近い格子点の値を先に求める
      do j=jmax/2, jmax/2+1
        yz_TempEQStr(j,k) = & 
          & cubic_spline_intpl_integ(splA,splB,splC,splD,jmax,splX,0.0_DP,y_Lat(j),z_TempRefStr(k))
      end do

      ! 赤道から極に向かって積分
      ! 南半球
      do j = jmax/2 -1, 1, -1
        yz_TempEQStr(j,k) = & 
          & cubic_spline_intpl_integ(splA,splB,splC,splD,jmax,splX,y_Lat(j+1),y_Lat(j),yz_TempEQStr(j+1,k))
      enddo

      ! 北半球
      do j = jmax/2 +2, jmax
        yz_TempEQStr(j,k) = & 
          & cubic_spline_intpl_integ(splA,splB,splC,splD,jmax,splX,y_Lat(j-1),y_Lat(j),yz_TempEQStr(j-1,k))
      enddo
    enddo

! --------------- ここから AGCM5 における sub_ay97.F の処理 ---------------

    ! 対流圏の基本温度場 (Akahori and Yoden, 1997)

    ! z_TempRefTro を Akahori and Yoden (1997) の値からスプライン補間で求める
    splX(1:KMAX_AY97) = - ScaleHeight * log(z_SigmaAY97)
    splY(1:KMAX_AY97) = z_TempRefAY97
    call solve_cubic_spline_intpl(splA,splB,splC,splD,splX,splY,KMAX_AY97)

    ! 値を求める
    do k = 1, kmax
      z_TempRefTro(k) = cubic_spline_intpl(splA,splB,splC,splD,KMAX_AY97,splX,z_HeightRef(k))
! debug
!      print *, 'debug', z_HeightRef(k), z_TempRefTro(k)
    end do


!!$! debug: オリジナルの温度と合っているか確認
!!$    do k = 1, KMAX_AY97
!!$      h1 = - ScaleHeight * log(z_SigmaAY97(k))
!!$      if (h1 <= z1) then
!!$        print *, z_SigmaAY97(k), z_TempRefAY97(k), exp(h1 * StaticStabForTro1 / Grav + log(TempRefSurfForTro)) * z_SigmaAY97(k)**Kappa
!!$      else
!!$        print *, z_SigmaAY97(k), z_TempRefAY97(k), exp((z1 * StaticStabForTro1 + (h1-z1) * StaticStabForTro2) / Grav + log(TempRefSurfForTro)) * z_SigmaAY97(k)**Kappa
!!$      end if
!!$    enddo

    ! z_DelTemp を Akahori and Yoden (1997) の値からスプライン補間で求める
    splX(1:KMAX_AY97) = - ScaleHeight * log(z_SigmaAY97)

! 論文通りの式
    splY(1:KMAX_AY97) = z_delTempAY97

! TY02a オリジナルソースの sub_ay97.F に相当する式.
! おそらくバグ. 厳密には未定義変数 (z_delTemp に相当) を与えていた.
!    splY(1:KMAX_AY97) = 0.0_DP

    call solve_cubic_spline_intpl(splA,splB,splC,splD,splX,splY,KMAX_AY97)

    ! 値を求める
    do k = 1, kmax

! 論文通りの式
        z_delTemp(k) = cubic_spline_intpl(splA,splB,splC,splD,KMAX_AY97,splX,z_HeightRef(k))

! TY02a オリジナルソースの sub_ay97.F に相当する式.
! splD (定数) の部分が z_delTempAY97 になっている
!
! 注意: TY02a オリジナルソースでは
! スプライン関数の定数項 (本プログラムの splD) は
! スプライン補間のルーチンの出力ではなく,
! 入力元の変数 z_delTempAY97 を使いまわすことになっている.
! 本プログラムでは上記のバグの場合 splD はゼロになり,
! そのままだと結果が再現できないので, 下記のようにして再現している.
!        z_delTemp(k) = cubic_spline_intpl(splA,splB,splC,z_delTempAY97,KMAX_AY97,splX,z_HeightRef(k))

! debug
!      print *, 'foo', z_HeightRef(k), z_delTemp(k)
    end do

! debug
!!$    do k = 1, KMAX_AY97
!!$      print *, ' '
!!$      print *, 'bar', k, z_delTempAY97(k), cubic_spline_intpl(splA,splB,splC,splD,KMAX_AY97,splX,splX(k))
!!$      print *, ' '
!!$    end do

    do k = 1, kmax
      do j = 1, jmax 
        yz_TempEQTro(j,k) = z_TempRefTro(k) &
          & + 0.5 * z_DelTemp(k) * ( cos(2.0*y_Lat(j)) - 1.0_DP/3.0_DP )
      enddo
    enddo

! --------------- ここから AGCM5 における tbasic.F の処理 ---------------

    ! 成層圏と対流圏の温度を組み合わせ, 冬の平衡温度を求める
    z1 =  9000.0_DP
    z2 = 15000.0_DP
    do k = 1, kmax
      h1 = z_HeightRef(k)
      do j = 1, jmax
        if (h1 <= z1) then
          yz_TempEQNHWin(j,k) = yz_TempEQTro(j,k)
        elseif (h1 > z1 .and. h1 <= z2) then
          ! 田口 D 論 式 A.1 より
          w = sin( 0.5_DP * PI * (h1 - z1) / (z2 - z1) )
          yz_TempEQNHWin(j,k) = &
            & (1.0_DP - w) * yz_TempEQTro(j,k) &
            & + w * yz_TempEQStr(j,k)
        else
          yz_TempEQNHWin(j,k) = yz_TempEQStr(j,k)
        endif
      enddo
    enddo

    do i=0,imax-1
      xyz_TempEQNHWin(i,:,:) = yz_TempEQNHWin
    end do

  contains 

    function sech(x)
      real(DP) :: x, sech
      sech = 1.0_DP / cosh(x)
    end function sech

  end subroutine SetTempEQNHWinTY02a

  !--------------------------------------------------------------------------------------

  ! 3 次元物理量の地形補正
  ! 変数名に xyz_TempEQ を使っているが他の変数にも使える
  subroutine CorrectByTopo3D(xyz_TempEQWoTopo, &  ! in
    & z_HeightRef, xyz_HeightRef, &                   ! in
    & xyz_TempEQ)                                     ! out

    implicit none
    real(DP), intent(in):: xyz_TempEQWoTopo (0:imax-1, 1:jmax, 1:kmax) ! 平衡温度 (地形補正なし)
    real(DP), intent(in):: z_HeightRef (1:kmax)  ! スケールハイトと sigma から求めた高度
    real(DP), intent(in):: xyz_HeightRef (0:imax-1, 1:jmax, 1:kmax)
    real(DP):: xyz_TempEQ (0:imax-1, 1:jmax, 1:kmax)
                              ! $ T_{eq} $ . 平衡温度. 
                              ! Equilibrium temperature

    real(DP):: splA(1:kmax-1)
    real(DP):: splB(1:kmax-1)
    real(DP):: splC(1:kmax-1)
    real(DP):: splD(1:kmax-1)
    real(DP):: splX(1:kmax)
    real(DP):: splY(1:kmax)

    integer:: i,j,k

    do j = 1, jmax
      do i = 0, imax-1
        ! スプライン補間
        splX(1:kmax) = z_HeightRef     ! 地形なしの場合の高度
        splY(1:kmax) = xyz_TempEQWoTopo(i,j,:)
        call solve_cubic_spline_intpl(splA,splB,splC,splD,splX,splY,kmax)

        do k = 1, kmax
          xyz_TempEQ(i,j,k) = cubic_spline_intpl(splA,splB,splC,splD,kmax,splX,xyz_HeightRef(i,j,k))
        end do
      end do
    end do
  end subroutine CorrectByTopo3D

  !--------------------------------------------------------------------------------------

  ! 鉛直 1 次元物理量の地形補正
  ! 変数名に xyz_TempEQ を使っているが他の変数にも使える
  subroutine CorrectByTopo1D(z_TempEQWoTopo, &  ! in
    & z_HeightRef, z_Height, &                  ! in
    & z_TempEQ)                                 ! out

    implicit none
    real(DP), intent(in):: z_TempEQWoTopo (1:kmax) ! 平衡温度 (地形補正なし)
    real(DP), intent(in):: z_HeightRef (1:kmax)  ! スケールハイトと sigma から求めた高度
    real(DP), intent(in):: z_Height (1:kmax)
    real(DP):: z_TempEQ (1:kmax)
                              ! $ T_{eq} $ . 平衡温度. 
                              ! Equilibrium temperature

    real(DP):: splA(1:kmax-1)
    real(DP):: splB(1:kmax-1)
    real(DP):: splC(1:kmax-1)
    real(DP):: splD(1:kmax-1)
    real(DP):: splX(1:kmax)
    real(DP):: splY(1:kmax)

    integer:: i,j,k

    ! スプライン補間
    splX = z_HeightRef     ! 地形なしの場合の高度
    splY = z_TempEQWoTopo
    call solve_cubic_spline_intpl(splA,splB,splC,splD,splX,splY,kmax)

    do k = 1, kmax
      z_TempEQ(k) = cubic_spline_intpl(splA,splB,splC,splD,kmax,splX,z_Height(k))
    end do
  end subroutine CorrectByTopo1D

  !--------------------------------------------------------------------------------------

  subroutine TY2002aForcingFinalize
    !
    ! モジュール内部の変数の割り付け解除を行います. 
    !
    ! Deallocate variables in this module. 
    !

    ! 宣言文 ; Declaration statements
    !
    implicit none

    ! 実行文 ; Executable statement
    !

    if ( .not. ty2002a_forcing_inited ) return

    ! 割り付け解除
    ! Deallocation
    !
    if ( allocated( z_HeightRef     ) ) deallocate( z_HeightRef     )
    if ( allocated( yz_URef         ) ) deallocate( yz_URef         )
    if ( allocated( yz_TempEQStr    ) ) deallocate( yz_TempEQStr    )
    if ( allocated( yz_TempEQTro    ) ) deallocate( yz_TempEQTro    )
    if ( allocated( xyz_alpha       ) ) deallocate( xyz_alpha       )
    if ( allocated( xyz_kv          ) ) deallocate( xyz_kv          )
    if ( allocated( xyz_HeightRef   ) ) deallocate( xyz_HeightRef   )
    if ( allocated( xyz_TempEQNHWin ) ) deallocate( xyz_TempEQNHWin )
    if ( allocated( xyz_TempEQNHSum ) ) deallocate( xyz_TempEQNHSum )
    if ( allocated( xyz_TempEQNHWinWoTopo ) ) deallocate( xyz_TempEQNHWinWoTopo )
    if ( allocated( xyz_TempEQNHSumWoTopo ) ) deallocate( xyz_TempEQNHSumWoTopo )

    ty2002a_forcing_inited  = .false.
    ty2002a_forcing_inited2 = .false.

  end subroutine TY2002aForcingFinalize

  !--------------------------------------------------------------------------------------

end module ty2002a_forcing
