!= Taguchi and Yoden (2002a) による強制と散逸
!
!= Forcing and dissipation suggested by Taguchi and Yoden (2002a)
!
! Authors::   Satoshi NODA
! Version::   $Id: held_suarez_1994.f90,v 1.14 2012-04-27 11:24:45 noda Exp $ 
! Tag Name::  $Name: dcpam5-20121128 $
! Copyright:: Copyright (C) GFD Dennou Club, 2013. All rights reserved.
! License::   See COPYRIGHT[link:../../../COPYRIGHT]
!

!!! 注意: これは田口さんの版を真似した版。入力ファイルが必要。

module ty2002a_forcing
  !
  != Taguchi and Yoden (2002a) による強制と散逸
  !
  != Forcing and dissipation suggested by Taguchi and Yoden (2002a)
  !
  ! <b>Note that Japanese and English are described in parallel.</b>
  !
  ! Taguchi and Yoden (2002a) で提案される乾燥大気 GCM ベンチマーク用の
  ! 強制と散逸を計算します. 
  ! 与える強制と散逸として, 温度場の帯状対称場への簡単なニュートン冷却と, 
  ! 境界層摩擦を表現する下層風のレイリー摩擦を用います. 
  !
  ! Forcing and dissipation for dry air GCM benchmark
  ! suggested by Taguchi and Yoden (2002a) are caluclate.
  ! We use simple Newtonian relaxation of the temperature field to a
  ! zonally symmetric state and Rayleigh damping of low-level winds to
  ! represent boundary-layer friction.
  !
  !
  !== Procedures List
  ! 
  ! TY2002aForcingInit     :: 初期化
  ! TY2002aForcing         :: 強制と散逸の計算
  ! TY2002aForcingFinalize :: 終了処理 (モジュール内部の変数の割り付け解除)
  ! ------------  :: ------------
  ! TY2002aForcingInit     :: Initialization
  ! TY2002aForcing         :: Calculate forcing and dissipation
  ! TY2002aForcingFinalize :: Termination (deallocate variables in this module)
  !
  !--
  !== NAMELIST
  !
  ! NAMELIST#ty2002a_forcing_nml
  !++
  !== References
  !
  ! * Taguchi, M., Yoden, S., 2002:
  !   Internal Interannual Variability of the Troposphere-Stratosphere
  !   Coupled System in a Simple Global Circulation Model.
  !   Part I: Parameter Sweep Experiment.
  !   <i>J. Atmos. Sci.</i>, <b>59</b>, 3021--3036.
  !

  ! モジュール引用 ; USE statements
  !

  ! 格子点設定
  ! Grid points settings
  !
  use gridset, only: imax, & ! 経度格子点数. 
                             ! Number of grid points in longitude
    &                jmax, & ! 緯度格子点数. 
                             ! Number of grid points in latitude
    &                kmax    ! 鉛直層数. 
                             ! Number of vertical level

  ! 種別型パラメタ
  ! Kind type parameter
  !
  use dc_types, only: DP, &      ! 倍精度実数型. Double precision. 
    &                 STRING, &  ! 文字列.       Strings. 
    &                 TOKEN      ! キーワード.   Keywords. 


  ! メッセージ出力
  ! Message output
  !
  use dc_message, only: MessageNotify

  ! 物理・数学定数設定
  ! Physical and mathematical constants settings
  !
  use constants0, only: PI
  use constants, only: Grav, &
                              ! $ g $ [m s-2]. 
                              ! 重力加速度. 
                              ! Gravitational acceleration
    & GasRDry, CpDry, RPlanet, Omega


  ! 宣言文 ; Declaration statements
  !
  implicit none
  private

  ! 公開手続き
  ! Public procedure
  !
  public :: TY2002aForcing
  public :: TY2002aForcingInit
  public :: TY2002aForcingFinalize

  ! 公開変数
  ! Public variables
  !

  ! 非公開変数
  ! Private variables
  !
  real(DP), parameter :: sec_in_a_day = 86400.0_DP
                              ! 1 日の秒数. 
                              ! Seconds in day. 

  logical, save :: ty2002a_forcing_inited = .false.
                              ! 初期設定フラグ. 
                              ! Initialization flag

  real(DP), save:: Kappa            ! $ \kappa = R/C_p $ .
                              ! 気体定数の定圧比熱に対する比. Ratio of gas constant to specific heat

  real(DP), save:: DayInAYear         ! 1 年の長さ [日]

  character(STRING), save:: TempEQSetting
                              ! 平衡温度の設定方法
                              ! Setting of equilibrium temperature
  character(STRING), save:: TempEQFile
                              ! 平衡温度のファイル名. 
                              ! File name of equilibrium temperature
  character(TOKEN) , save:: TempEQName
                              ! 平衡温度の変数名. 
                              ! Variable name of equilibrium temperature

  real(DP), save:: ScaleHeight      ! 平衡温度を決めるのに用いるスケールハイト
  real(DP), save:: SurfPress0       ! 平均表面気圧 [Pa]
!  real(DP), allocatable::   ! スケールハイトと sigma から求めた高度
  real(DP), allocatable, save:: z_HeightRef (:)  ! スケールハイトと sigma から求めた高度

!  real(DP), allocatable, save:: yz_TempEQNHWin (:,:)
                              ! 北半球冬至の平衡温度場
                              ! Equilibrium temperature in north hemisphere of the winter solstice
!  real(DP), allocatable, save:: yz_TempEQNHSum (:,:)
                              ! 北半球夏至の平衡温度場
                              ! Equilibrium temperature in north hemisphere of the winter solstice

  real(DP), allocatable, save:: xyz_alpha (:,:,:)
  real(DP), allocatable, save:: xyz_kv (:,:,:)
  real(DP), allocatable, save:: xy_SurfGeopot (:,:)
  real(DP), allocatable, save:: xyz_TempEQNHWin (:,:,:)
  real(DP), allocatable, save:: xyz_TempEQNHSum (:,:,:)

  character(*), parameter:: module_name = 'ty2002a_forcing'
                              ! モジュールの名称. 
                              ! Module name
  character(*), parameter:: version = &
    & '$Name: dcpam5-20121128 $' // &
    & '$Id: held_suarez_1994.f90,v 1.14 2012-04-27 11:24:45 noda Exp $'
                              ! モジュールのバージョン
                              ! Module version

contains

  !--------------------------------------------------------------------------------------

  subroutine TY2002aForcing( &
    & xyz_U,    xyz_V,    xyz_Temp, & ! (in)
    & xyz_Press, &                       ! (in)
    & xy_SurfHeight, &                       ! (in) -> (inout)
    & xyz_DUDt, xyz_DVDt, xyz_DTempDt &      ! (out)
    & )
    !
    ! 引数として与えられた東西風速 xyz_U, 南北風速 xyz_V, 
    ! 温度 xyz_Temp から, 
    ! 温度場の帯状対称場への簡単なニュートン冷却と
    ! 境界層摩擦を表現する下層風のレイリー摩擦による
    ! 風速と温度の変化率を求め, 
    ! xyz_DUDt, xyz_DVDt, xyz_DTempDt に返します. 
    !
    ! Tendencies by simple Newtonian relaxation of the temperature field to a
    ! zonally symmetric state and Rayleigh damping of low-level winds to
    ! represent boundary-layer friction are calculated 
    ! from eastward wind "xyz_U", northward wind "xyz_V", 
    ! temperature "xyz_Temp".
    ! And the tencencies are returned as 
    ! "xyz_DUDt", "xyz_DVDt", "xyz_DTempDt". 
    !
    !

    ! モジュール引用 ; USE statements
    !

    ! 座標データ設定
    ! Axes data settings
    !
    use axesset, only: &
      & y_Lat, &              ! $ \varphi $ [rad.] . 緯度. Latitude
      & z_Sigma               ! $ \sigma $ レベル (整数). 
                              ! Full $ \sigma $ level

    ! 時刻管理
    ! Time control
    !
    use timeset, only: &
      & DelTime, &            ! $ \Delta t $
      & TimeN, &              ! ステップ $ t $ の時刻. Time of step $ t $. 
      & TimesetClockStart, TimesetClockStop

    ! ヒストリデータ出力
    ! History data output
    !
    use gtool_historyauto, only: HistoryAutoPut

    ! 時系列データの読み込み
    ! Reading time series
    !
    use read_time_series, only: SetValuesFromTimeSeriesWrapper

    ! 宣言文 ; Declaration statements
    !
    implicit none

    real(DP), intent(in):: xyz_U (0:imax-1, 1:jmax, 1:kmax)
                              ! $ u $ . 東西風速. 
                              ! Eastward wind
    real(DP), intent(in):: xyz_V (0:imax-1, 1:jmax, 1:kmax)
                              ! $ v $ . 南北風速. 
                              ! Northward wind
    real(DP), intent(in):: xyz_Temp (0:imax-1, 1:jmax, 1:kmax)
                              ! $ T $ . 温度. 
                              ! Temperature
    real(DP), intent(in):: xyz_Press(0:imax-1, 1:jmax, 1:kmax)
                              ! $ p $ . 気圧.
                              ! Pressure
    real(DP), intent(in):: xy_SurfHeight (0:imax-1, 1:jmax)
                              ! $ z $ . 地表面高度.
                              ! Surface Height
    real(DP), intent(out):: xyz_DUDt (0:imax-1, 1:jmax, 1:kmax)
                              ! $ \DP{u}{t} $ . 東西風速変化. 
                              ! Eastward wind tendency
    real(DP), intent(out):: xyz_DVDt (0:imax-1, 1:jmax, 1:kmax)
                              ! $ \DP{v}{t} $ . 南北風速変化. 
                              ! Northward wind tendency
    real(DP), intent(out):: xyz_DTempDt (0:imax-1, 1:jmax, 1:kmax)
                              ! $ \DP{T}{t} $ . 温度変化. 
                              ! Temperature tendency

    ! 作業変数
    ! Work variables
    !
!    real(DP):: xyz_kv (0:imax-1, 1:jmax, 1:kmax)
!    real(DP):: z_kvRef (1:kmax)
    real(DP):: xyz_TempEQ (0:imax-1, 1:jmax, 1:kmax)
                              ! $ T_{eq} $ . 平衡温度. 
                              ! Equilibrium temperature
!    real(DP):: xyz_TempEQWoTopo (0:imax-1, 1:jmax, 1:kmax) ! 平衡温度 (地形補正なし)

!    real(DP):: h1
!    real(DP):: z1, z2
!    real(DP):: w, dx

    real(DP):: splA(1:max(jmax,kmax)-1)
    real(DP):: splB(1:max(jmax,kmax)-1)
    real(DP):: splC(1:max(jmax,kmax)-1)
    real(DP):: splD(1:max(jmax,kmax)-1)
    real(DP):: splX(1:max(jmax,kmax))
    real(DP):: splY(1:max(jmax,kmax))

    real(DP):: xyz_HeightRef (0:imax-1, 1:jmax, 1:kmax)
                               ! 基本場計算用高度 (地形入り) [m].
                               ! 実際の高度ではない.

    real(DP):: sec_in_a_year
    real(DP):: season
    real(DP):: r

    real(DP):: z_alphaRef (1:kmax)
    real(DP):: mag


    integer:: i               ! 経度方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in longitude
    integer:: j               ! 緯度方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in latitude
    integer:: k               ! 鉛直方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in vertical direction
    integer:: m


    ! 実行文 ; Executable statement
    !

    ! 初期化確認
    ! Initialization check
    !
    if ( .not. ty2002a_forcing_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if

    ! 計算時間計測開始
    ! Start measurement of computation time
    !
    call TimesetClockStart( module_name )

    ! 地形の上書き
    xy_SurfHeight = xy_SurfGeopot / Grav

! debug
!    write(*,*) maxval(xy_SurfHeight(:,:))

    ! 平衡温度
    ! Equilibrium temperature
    !
    ! ファイル (地形補正なしの温度場) を読む場合
    ! 現在の季節の平衡温度
!    if (DayInAYear > 0.0_DP) then
!      sec_in_a_year = sec_in_a_day * DayInAYear
!      season = TimeN / sec_in_a_year
!      r = 0.5 * (1.0_DP + cos(2.0_DP * PI * season)) 
!      xyz_TempEQ = &
!        & r * xyz_TempEQNHWin + (1.0_DP - r) * xyz_TempEQNHSum
!    else
      ! perpetual winter
!      do i = 0, imax-1
!        xyz_TempEQWoTopo(i,:,:) = yz_TempEQNHWin
!      end do
      xyz_TempEQ = xyz_TempEQNHWin
!    end if


    ! 温度 $ T $ へニュートン冷却を適用
    ! Apply Newtonian relaxation to temperature $ T $
    !
    xyz_DTempDt =  - xyz_alpha * (xyz_Temp - xyz_TempEQ)

    ! 東西風速 $ u $ と南北風速 $ v $ へレイリー摩擦を適用
    ! Apply Rayleigh damping to eastward wind $ u $ and northward wind $ v $
    !
    xyz_DUDt = - xyz_kv * xyz_U
    xyz_DVDt = - xyz_kv * xyz_V

    ! ヒストリデータ出力
    ! History data output
    !
    call HistoryAutoPut( TimeN, 'DUDtTY2002a',    xyz_DUDt )
    call HistoryAutoPut( TimeN, 'DVDtTY2002a',    xyz_DVDt )
    call HistoryAutoPut( TimeN, 'DTempDtTY2002a', xyz_DTempDt )
    call HistoryAutoPut( TimeN, 'TempEQ',         xyz_TempEQ )
!    call HistoryAutoPut( TimeN, 'TempEQWoTopo',   xyz_TempEQWoTopo )
!    if (TempEQSetting == 'Taguchi and Yoden (2002a)') then
!      call HistoryAutoPut( TimeN, 'TempEQNHWin',    yz_TempEQNHWin )
      call HistoryAutoPut( TimeN, 'TempEQNHWin',    xyz_TempEQNHWin )
!      call HistoryAutoPut( TimeN, 'TempEQStr',      yz_TempEQStr )
!      call HistoryAutoPut( TimeN, 'TempEQTro',      yz_TempEQTro )
!      call HistoryAutoPut( TimeN, 'URef',           yz_URef )
!    end if
    call HistoryAutoPut( TimeN, 'HeightRef',      xyz_HeightRef )
    call HistoryAutoPut( TimeN, 'SurfHeight',     xy_SurfHeight )
    call HistoryAutoPut( TimeN, 'RadRelaxRate',   xyz_alpha )
    call HistoryAutoPut( TimeN, 'RayleighRelaxRate', xyz_kv )

    ! 計算時間計測一時停止
    ! Pause measurement of computation time
    !
    call TimesetClockStop( module_name )


  end subroutine TY2002aForcing

  !--------------------------------------------------------------------------------------
  subroutine TY2002aForcingInit
    !
    ! held_suarez_1994 モジュールの初期化を行います. 
    ! NAMELIST#held_suarez_1994_nml の読み込みはこの手続きで行われます. 
    !
    ! "held_suarez_1994" module is initialized. 
    ! "NAMELIST#held_suarez_1994_nml" is loaded in this procedure. 
    !

    ! モジュール引用 ; USE statements
    !

    ! 物理定数設定
    ! Physical constants settings
    !
    use constants, only: &
      & GasRDry, &
                              ! $ R $ [J kg-1 K-1]. 
                              ! 乾燥大気の気体定数. 
                              ! Gas constant of air
      & CpDry
                              ! $ C_p $ [J kg-1 K-1]. 
                              ! 乾燥大気の定圧比熱. 
                              ! Specific heat of air at constant pressure

    ! 座標データ設定
    ! Axes data settings
    !
    use axesset, only: &
      & x_Lon, &              ! $ \lambda $ [rad.] . 経度. Longitude
      & y_Lat, &              ! $ \varphi $ [rad.] . 緯度. Latitude
      & z_Sigma               ! $ \sigma $ レベル (整数). 
                              ! Full $ \sigma $ level

    ! NAMELIST ファイル入力に関するユーティリティ
    ! Utilities for NAMELIST file input
    !
    use namelist_util, only: namelist_filename, NmlutilMsg, NmlutilAryValid

    ! ファイル入出力補助
    ! File I/O support
    !
    use dc_iounit, only: FileOpen

    ! 種別型パラメタ
    ! Kind type parameter
    !
    use dc_types, only: STDOUT ! 標準出力の装置番号. Unit number of standard output

    ! 文字列操作
    ! Character handling
    !
    use dc_string, only: StoA

    ! ヒストリデータ出力
    ! History data output
    !
    use gtool_history, only: HistoryGet
    use gtool_historyauto, only: HistoryAutoAddVariable

    ! 宣言文 ; Declaration statements
    !
    implicit none

                              ! 初回を示すフラグ.
                              ! Flag that indicates first loop
                              !
    logical, save:: flag_first_TempEQ = .true.

    integer:: i               ! 経度方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in longitude
    integer:: j               ! 緯度方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in latitude
    integer:: k               ! 鉛直方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in vertical direction

    integer:: unit_nml        ! NAMELIST ファイルオープン用装置番号. 
                              ! Unit number for NAMELIST file open
    integer:: iostat_nml      ! NAMELIST 読み込み時の IOSTAT. 
                              ! IOSTAT of NAMELIST read
    character(STRING):: AlphaFile
    character(STRING):: BetaFile
    character(STRING):: SurfGeopotFile
    character(STRING):: TempEQWinFile
    character(TOKEN)::  AlphaName
    character(TOKEN)::  BetaName
    character(TOKEN)::  SurfGeopotName
    character(TOKEN)::  TempEQWinName

    real(DP):: xyz_alpha0 (0:imax-1, 1:jmax, 1:kmax)
    real(DP):: xyz_kv0 (0:imax-1, 1:jmax, 1:kmax)
    real(DP):: xy_SurfGeopot0 (0:imax-1, 1:jmax)
    real(DP):: xyz_TempEQNHWin0 (0:imax-1, 1:jmax, 1:kmax)
    real(DP):: xyz_TempEQNHSum0 (0:imax-1, 1:jmax, 1:kmax)


    ! NAMELIST 変数群
    ! NAMELIST group name
    !
    namelist /ty2002a_forcing_nml/ &
      & AlphaFile, AlphaName, &
      & BetaFile,  BetaName, &
      & SurfGeopotFile, SurfGeopotName, &
      & TempEQWinFile, TempEQWinName
          !
          ! デフォルト値については初期化手続 "ty2002a_forcing#TY2002aForcingInit" 
          ! のソースコードを参照のこと. 
          !
          ! Refer to source codes in the initialization procedure
          ! "ty2002a_forcing#TY2002aForcingInit" for the default values. 
          !

    ! 実行文 ; Executable statement
    !

    if ( ty2002a_forcing_inited ) return


    ! デフォルト値の設定
    ! Default values settings
    !
    AlphaFile = "A-w1h1000.nc"
    AlphaName = "ALPHA"
    BetaFile  = "B-w1h1000.nc"
    BetaName  = "BETA"
    SurfGeopotFile = "G-w1h1000.nc"
    SurfGeopotName = "GPHIS"
    TempEQWinFile = "TW-w1h1000.nc"
    TempEQWinName = "T"

    ! NAMELIST の読み込み
    ! NAMELIST is input
    !
    if ( trim(namelist_filename) /= '' ) then
      call FileOpen( unit_nml, &          ! (out)
        & namelist_filename, mode = 'r' ) ! (in)

      rewind( unit_nml )
      read( unit_nml, &                ! (in)
        & nml = ty2002a_forcing_nml, &  ! (out)
        & iostat = iostat_nml )        ! (out)
      close( unit_nml )

      call NmlutilMsg( iostat_nml, module_name ) ! (in)
!$      if ( iostat_nml == 0 ) write( STDOUT, nml = held_suarez_1994_nml )
    end if


    allocate( xyz_alpha (0:imax-1, 1:jmax, 1:kmax) )
    allocate( xyz_kv (0:imax-1, 1:jmax, 1:kmax) )
    allocate( xy_SurfGeopot (0:imax-1, 1:jmax) )
    allocate( xyz_TempEQNHWin (0:imax-1, 1:jmax, 1:kmax) )
    allocate( xyz_TempEQNHSum (0:imax-1, 1:jmax, 1:kmax) )

    ! ファイルの読み込み
    call HistoryGet(trim(AlphaFile), trim(AlphaName), xyz_alpha0)
    call HistoryGet(trim(BetaFile),  trim(BetaName),  xyz_kv0)
    call HistoryGet(trim(SurfGeopotFile), trim(SurfGeopotName),  xy_SurfGeopot0)
    call HistoryGet(trim(TempEQWinFile),  trim(TempEQWinName),  xyz_TempEQNHWin0)

    ! AGCM 用ファイルは南北が逆なので詰め直し
    do j = 1, jmax
      xyz_TempEQNHSum(:,j,:) = xyz_TempEQNHWin0(:,j,:)
      xyz_TempEQNHWin(:,j,:) = xyz_TempEQNHWin0(:,jmax-j+1,:)
      xyz_alpha(:,j,:) = xyz_alpha0(:,jmax-j+1,:)
      xyz_kv(:,j,:) = xyz_kv0(:,jmax-j+1,:)
      xy_SurfGeopot(:,j) = xy_SurfGeopot0(:,jmax-j+1)
    enddo
    
    flag_first_TempEQ = .false.


    ! ヒストリデータ出力のためのへの変数登録
    ! Register of variables for history data output
    !
    call HistoryAutoAddVariable( 'DUDtTY2002a', &
      & (/ 'lon ', 'lat ', 'sig ', 'time' /), &
      & 'eastward wind tendency', 'm s-2' )
    call HistoryAutoAddVariable( 'DVDtTY2002a', &
      & (/ 'lon ', 'lat ', 'sig ', 'time' /), &
      & 'northward wind tendency', 'm s-2' )
    call HistoryAutoAddVariable( 'DTempDtTY2002a', &
      & (/ 'lon ', 'lat ', 'sig ', 'time' /), &
      & 'temperature tendency', 'K s-1' )
    call HistoryAutoAddVariable( 'TempEQ', &
      & (/ 'lon ', 'lat ', 'sig ', 'time' /), &
      & 'equilibrium temperature', 'K' )
    call HistoryAutoAddVariable( 'TempEQWoTopo', &
      & (/ 'lon ', 'lat ', 'sig ', 'time' /), &
      & 'equilibrium temperature without topography', 'K' )
!    if (TempEQSetting == 'Taguchi and Yoden (2002a)') then
!      call HistoryAutoAddVariable( 'TempEQNHWin', &
!        & (/ 'lat ', 'sig ', 'time' /), &
!        & 'equilibrium temperature of NH Winter', 'K' )
      call HistoryAutoAddVariable( 'TempEQNHWin', &
        & (/ 'lon ', 'lat ', 'sig ', 'time' /), &
        & 'equilibrium temperature of NH Winter', 'K' )
!!$      call HistoryAutoAddVariable( 'URef', &
!!$        & (/ 'lat ', 'sig ', 'time' /), &
!!$        & 'reference eastward wind for stratosphere', 'm s-1' )
!!$      call HistoryAutoAddVariable( 'TempEQStr', &
!!$        & (/ 'lat ', 'sig ', 'time' /), &
!!$        & 'equilibrium temperature for stratosphere', 'K' )
!!$      call HistoryAutoAddVariable( 'TempEQTro', &
!!$        & (/ 'lat ', 'sig ', 'time' /), &
!!$        & 'equilibrium temperature for troposphere', 'K' )
!    end if
    call HistoryAutoAddVariable( 'HeightRef', &
      & (/ 'lon ', 'lat ', 'sig ', 'time' /), &
      & 'reference height for equilibrium temperature etc.', 'm' )
    call HistoryAutoAddVariable( 'SurfHeight', &
      & (/ 'lon ', 'lat ', 'time' /), &
      & 'surface height', 'm' )
    call HistoryAutoAddVariable( 'RadRelaxRate', &
      & (/ 'lon ', 'lat ', 'sig ', 'time' /), &
      & 'radiative relaxation rate', 's-1' )
    call HistoryAutoAddVariable( 'RayleighRelaxRate', &
      & (/ 'lon ', 'lat ', 'sig ', 'time' /), &
      & 'Rayleigh friction relaxation rate', 's-1' )

    ! 印字 ; Print
    !
    call MessageNotify( 'M', module_name, '----- Initialization Messages -----' )
!    call MessageNotify( 'M', module_name, 'DayInAYear         = %f', d = (/ DayInAYear         /) )
!    call MessageNotify( 'M', module_name, 'TempEQSetting      = %c', c1 = trim(TempEQSetting) )
!    call MessageNotify( 'M', module_name, 'TempEQFile         = %c', c1 = trim(TempEQFile) )
!    call MessageNotify( 'M', module_name, 'TempEQName         = %c', c1 = trim(TempEQName) )
!    call MessageNotify( 'M', module_name, 'ScaleHeight        = %f', d = (/ ScaleHeight        /) )
    call MessageNotify( 'M', module_name, '-- version = %c', c1 = trim(version) )

    ty2002a_forcing_inited = .true.

  end subroutine TY2002aForcingInit

  !--------------------------------------------------------------------------------------

  subroutine TY2002aForcingFinalize
    !
    ! モジュール内部の変数の割り付け解除を行います. 
    !
    ! Deallocate variables in this module. 
    !

    ! 宣言文 ; Declaration statements
    !
    implicit none

    ! 実行文 ; Executable statement
    !

    if ( .not. ty2002a_forcing_inited ) return

    ! 割り付け解除
    ! Deallocation
    !
!    if ( allocated( z_HeightRef    ) ) deallocate( z_HeightRef    )
    if ( allocated( xyz_TempEQNHWin ) ) deallocate( xyz_TempEQNHWin )
    if ( allocated( xyz_TempEQNHSum ) ) deallocate( xyz_TempEQNHSum )

    ty2002a_forcing_inited = .false.

  end subroutine TY2002aForcingFinalize

  !--------------------------------------------------------------------------------------

end module ty2002a_forcing
