
// -*- mode: c++; c-basic-offset:4 -*-

// This file is part of libdap, A C++ implementation of the OPeNDAP Data
// Access Protocol.

// Copyright (c) 2002,2003 OPeNDAP, Inc.
// Author: James Gallagher <jgallagher@opendap.org>
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
// You can contact OPeNDAP, Inc. at PO Box 112, Saunderstown, RI. 02874-0112.
 
// (c) COPYRIGHT URI/MIT 1994-1999
// Please read the full copyright statement in the file COPYRIGHT_URI.
//
// Authors:
//      jhrg,jimg       James Gallagher <jgallagher@gso.uri.edu>

// Implementation for Array.
//
// jhrg 9/13/94


#include "config.h"

#include "Array.h"
#include "util.h"
#include "debug.h"
#include "InternalErr.h"
#include "escaping.h"
#include "ArrayIterAdapter.h"

#include <algorithm>
#include <functional>


using namespace std;

void
Array::_duplicate(const Array &a)
{
    _shape = a._shape;
}

// The first method of calculating length works when only one dimension is
// constrained and you want the others to appear in total. This is important
// when selecting from grids since users may not select from all dimensions
// in which case that means they want the whole thing. Array projection
// should probably work this way too, but it doesn't. 9/21/2001 jhrg
    
/** @deprecated Calling this method should never be necessary. It is called
    whenever the size of the Array is changed.

    Changes the size property of the array.  If the array
    exists, it is augmented by a factor of <tt>size</tt>. This does
    not change the actual size of the array.
*/
void
Array::update_length(int)
{
    int length = 1;
    for (Dim_citer i = _shape.begin(); i != _shape.end(); i++)
    {
	length *= (*i).c_size > 0 ? (*i).c_size : 1;
    }

    set_length(length);
}

// Construct an instance of Array. The (BaseType *) is assumed to be
// allocated using new - The dtor for Vector will delete this object.

/** The Array constructor requires the name of the variable to be
    created, and the type of data the Array is to hold.  The name
    may be omitted, which will create a nameless variable.  The
    template pointer may also be omitted.  Note that if the template
    pointer is omitted when the Array is created, it <i>must</i> be
    added (with <tt>add_var()</tt>) before <tt>read()</tt> or
    <tt>deserialize()</tt> is called. 

    @note Force the Array::add_var() method to be used to add \e v.
    This version of add_var() calls Vector::add_var().

    @param n A string containing the name of the variable to be
    created. 
    @param v A pointer to a variable of the type to be included 
    in the Array. 
    @brief Array constructor
*/
Array::Array(const string &n, BaseType *v) : Vector(n, 0, dods_array_c)
{
    add_var(v);
}

/** @brief The Array copy constructor. */
Array::Array(const Array &rhs) : Vector(rhs)
{
    _duplicate(rhs);
}

/** @brief The Array destructor. */
Array::~Array()
{
    DBG(cerr << "Entering ~Array (" << this << ")" << endl);
    DBG(cerr << "Exiting ~Array" << endl);
}

BaseType *
Array::ptr_duplicate()
{
    return new Array(*this);
}

Array &
Array::operator=(const Array &rhs)
{
    if (this == &rhs)
	return *this;

    dynamic_cast<Vector &>(*this) = rhs;

    _duplicate(rhs);

    return *this;
}

/** @brief Add the BaseType pointer to this constructor type
    instance. 

    Propagate the name of the BaseType instance to this instance. This
    ensures that variables at any given level of the DDS table have
    unique names (i.e., that Arrays do not have their default name ""). If
    <tt>v</tt>'s name is null, then assume that the array \e is named and
    don't overwrite it with <tt>v</tt>'s null name.
    
    @note This version checks to see if \e v is an array. If so, it calls
    Vector::add_var() using the template variable of \e v and then appends
    the dimensions of \e v to this array. This somewhat obscure behavior
    simplifies 'translating' Sequences to arrays when the actual variable
    being translated is not a regular Sequence but an array of Sequences.

    @param v The template variable for the array
    @param p The Part parameter defaults to nil and is ignored by this method.
*/

void
Array::add_var(BaseType *v, Part)
{
    if (v && v->type() == dods_array_c) {
        Array &a = dynamic_cast<Array&>(*v);
        Vector::add_var(a.var());
        Dim_iter i =a.dim_begin();
        Dim_iter i_end = a.dim_end();
        while (i != i_end) {
            append_dim(a.dimension_size(i), a.dimension_name(i));
            ++i;
        }
    }
    else {
        Vector::add_var(v);
    }
}

/** Given a size and a name, this function adds a dimension to the
    array.  For example, if the Array is already 10 elements long,
    calling <tt>append_dim</tt> with a size of 5 will transform the array
    into a 10x5 matrix.  Calling it again with a size of 2 will
    create a 10x5x2 array, and so on.  This sets Vector's length
    member as a side effect. 

    @param size The size of the desired new row.
    @param name The name of the new dimension.  This defaults to
    an empty string. 
    @brief Add a dimension of a given size. */
void 
Array::append_dim(int size, string name)
{ 
    dimension d;

    // This is invariant
    d.size = size;
    d.name = www2id(name);

    // this information changes with each constraint expression
    d.start = 0; 
    d.stop = size -1;
    d.stride = 1;
    d.c_size = size;
    d.selected = true;		// assume all dims selected.

    _shape.push_back(d); 

    update_length(size);
}

/** Resets the dimension constraint information so that the entire
    array is selected. 

    @brief Reset constraint to select entire array.
*/

void
Array::reset_constraint()
{
    set_length(-1);

    for (Dim_iter i = _shape.begin(); i != _shape.end(); i++)
    {
	(*i).start = 0;
	(*i).stop = (*i).size - 1;
	(*i).stride = 1;
	(*i).c_size = (*i).size;

	(*i).selected = true;

	update_length((*i).size);
    }
}


/** Tell the Array object to clear the constraint information about
    dimensions. Do this <i>once</i> before calling
    <tt>add_constraint()</tt> for each new constraint expression. Only
    the dimensions explicitly selected using <tt>add_constraint()</tt>
    will be sent.

    @brief Clears the projection; add each projected dimension explicitly using
    <tt>add_constraint</tt>. 
*/
void
Array::clear_constraint()
{
    for (Dim_iter i = _shape.begin(); i != _shape.end(); i++)
    {
	(*i).start = 0;
	(*i).stop = 0;
	(*i).stride = 0;
	(*i).c_size = 0;
	(*i).selected = false;
    }

    set_length(-1);
}

// the start and stop indices are inclusive.

// Note: MS VC++ won't tolerate embedded newlines in strings, hence the \n
// is explicit.  
static const char *array_sss = \
"Invalid constraint parameters: At least one of the start, stride or stop \n\
specified do not match the array variable.";

/** Once a dimension has been created (see <tt>append_dim()</tt>), it can
    be ``constrained''.  This will make the array appear to the rest
    of the world to be smaller than it is.  This functions sets the
    projection for a dimension, and marks that dimension as part of the
    current projection.
    
    A stride value <= 0 or > the array size is an error and causes
    <tt>add_constraint</tt> to return FALSE. Similarly, start or
    stop values greater than the dimension size will cause a FALSE
    return value.

    @brief Adds a dimension constraint to an Array.
    @deprecated Methods which use the Pix objects are deprecated in favor of
    the methods which use iterators.

    @param p An index (of type Pix) pointing to the dimension in the
    list of dimensions.
    @param start The start index of the constraint.
    @param stride The stride value of the constraint.
    @param stop The stop index of the constraint.
    @return void; in case of failure it throws an exception. 
*/

void
Array::add_constraint(Pix p, int start, int stride, int stop)
{
    ArrayIterAdapter *iter = (ArrayIterAdapter *)p.getIterator() ;
    Dim_iter &i = iter->getIterator() ;
    add_constraint( i, start, stride, stop ) ;
    return ;
}

/** Once a dimension has been created (see append_dim()), it can
    be constrained.  This will make the array appear to the rest
    of the world to be smaller than it is.  This functions sets the
    projection for a dimension, and marks that dimension as part of the
    current projection.

    @note A stride value <= 0 or > the array size is an error and causes
    add_constraint to throw an Error. Similarly, start or stop values >
    size also cause an Error exception to be thrown.

    @brief Adds a constraint to an Array dimension.  

    @param i An iterator pointing to the dimension in the list of 
    dimensions.
    @param start The start index of the constraint.
    @param stride The stride value of the constraint.
    @param stop The stop index of the constraint.
    @exception Error Thrown if the any of values of start, stop or stride
    cannot be applied to this array. */
void
Array::add_constraint(Dim_iter i, int start, int stride, int stop)
{
    dimension &d = *i ;

    // Check for bad constraints.
    // Jose Garcia
    // Usually invalid data for a constraint is the user's mistake
    // because they build a wrong URL in the client side.
    if (start >= d.size || stop >= d.size || stride > d.size || stride <= 0)
	throw Error(malformed_expr, array_sss);

    if (((stop - start) / stride + 1) > d.size)
	throw Error(malformed_expr, array_sss);

    d.start = start;
    d.stop = stop;
    d.stride = stride;

    d.c_size = (stop - start) / stride + 1;
    
    DBG(cerr << "add_constraint: c_size = " << d.c_size << endl);

    d.selected = true;

    update_length(d.c_size);
}

/** Returns a pointer to the first dimension of the array.  Use 
    <tt>next_dim()</tt> to return successive dimensions.

    @deprecated Methods which use the Pix objects are deprecated in favor of
    the methods which use iterators.

    @brief Return first dimension of array.
    @return A <b>Pix</b> object indicating the first array dimension.
*/    
Pix 
Array::first_dim() 
{ 
    ArrayIterAdapter *i = new ArrayIterAdapter( _shape ) ;
    i->first() ;
    return i ;
}

/** Returns an iterator to the first dimension of the Array. */
Array::Dim_iter
Array::dim_begin()
{
    return _shape.begin() ;
}

/** Returns an iterator past the last dimension of the Array. */
Array::Dim_iter
Array::dim_end()
{
    return _shape.end() ;
}

/** Given a dimension index, increments it to point to the next
    dimension.   Use <tt>first_dim()</tt> to return the first dimensions.

    @brief Return next array dimension.
    @deprecated Methods which use the Pix objects are deprecated. Use the
    methods which use iterators.

    @param p A <b>Pix</b> object indicating the array dimension immediately
           before the desired one.
    @return A <b>Pix</b> object indicating the array dimension immediately 
            after the one specified by <i>p</i>.
*/
void 
Array::next_dim(Pix p) 
{ 
    p.next() ;
}

/** Return the total number of dimensions contained in the array.
    When <i>constrained</i> is TRUE, return the number of dimensions
    given the most recently evaluated constraint expression. 

    @brief Return the total number of dimensions in the array.
    @param constrained A boolean flag to indicate whether the array is
    constrained or not.  By default, constrained is FALSE.
*/

unsigned int
Array::dimensions(bool constrained)
{
    unsigned int dim = 0;
    for(Dim_citer i = _shape.begin(); i != _shape.end(); i++) 
    {
	if (constrained) {
	    if ((*i).selected)
		dim++;
	}
	else
	{
	    dim++;
	}
    }

    return dim;
}

/** Return the size of the array dimension referred to by <i>p</i>. 
    If the dimension is constrained (indicated with the
    <i>constrained</i> argument), the constrained size is returned.

    @brief Returns the size of the dimension.  
    @deprecated Methods which use the Pix objects are deprecated in favor of
    the methods which use iterators.

    @param p The Pix index of the dimension.
    @param constrained If this parameter is TRUE, the function
    returns the constrained size of the array.  If the dimension is
    not selected, the function returns zero.  If it is FALSE, the
    function returns the dimension size whether or not the dimension
    is constrained.  The default value is FALSE.
    @return An integer containing the size of the specified dimension.
*/

int 
Array::dimension_size(Pix p, bool constrained) 
{ 
    ArrayIterAdapter *iter = (ArrayIterAdapter *)p.getIterator() ;
    Dim_iter i = iter->getIterator() ;
    return dimension_size( i, constrained ) ;
}

/** Return the size of the array dimension referred to by <i>i</i>. 
    If the dimension is constrained the constrained size is returned if 
    <i>constrained</i> is \c true. 

    @brief Returns the size of the dimension.  

    @param i The dimension.

    @param constrained If this parameter is TRUE, the method returns the
    constrained size of the array so long as a constraint has been applied to
    this dimension. If TRUE and no constraint has been applied, this method
    returns zero. If it is FALSE, the method ignores any constraint that
    has been applied to this dimension and returns the full size of the
    dimension. The default value is FALSE.

    @return An integer containing the size of the specified dimension.
*/
int 
Array::dimension_size(Dim_iter i, bool constrained) 
{ 
    int size = 0;

    if (!_shape.empty())
	if (constrained) {
	    if ((*i).selected)
		size = (*i).c_size;
	    else
		size = 0;
	}
	else
	    size = (*i).size; 

    return size;
}

/** Use this function to return the start index of an array
    dimension.  If the array is constrained (indicated with the
    <i>constrained</i> argument), the start index of the constrained
    array is returned (or zero if the dimension in question is not
    selected at all).  See also <tt>dimension_stop()</tt> and
    <tt>dimension_stride()</tt>.

    @brief Return the start index of a dimension.
    @deprecated Methods which use the Pix objects are deprecated in favor of
    the methods which use iterators.

    @param p The Pix index of the dimension.
    @param constrained If this parameter is TRUE, the function
    returns the start index only if the dimension is constrained
    (subject to a start, stop, or stride constraint).  If
    the dimension is not constrained, the function returns zero.  If it
    is FALSE, the function returns the start index whether or not
    the dimension is constrained. 
    @return The desired start index.
*/

int 
Array::dimension_start(Pix p, bool constrained) 
{ 
    ArrayIterAdapter *iter = (ArrayIterAdapter *)p.getIterator() ;
    Dim_iter i = iter->getIterator() ;
    return dimension_start( i, constrained ) ;
}

/** Use this function to return the start index of an array
    dimension.  If the array is constrained (indicated with the
    <i>constrained</i> argument), the start index of the constrained
    array is returned (or zero if the dimension in question is not
    selected at all).  See also <tt>dimension_stop()</tt> and
    <tt>dimension_stride()</tt>.

    @brief Return the start index of a dimension.

    @param i The dimension.
    @param constrained If this parameter is TRUE, the function
    returns the start index only if the dimension is constrained
    (subject to a start, stop, or stride constraint).  If
    the dimension is not constrained, the function returns zero.  If it
    is FALSE, the function returns the start index whether or not
    the dimension is constrained. 
    @return The desired start index.
*/
int 
Array::dimension_start(Dim_iter i, bool constrained) 
{ 
    int start = 0;

    if (!_shape.empty())
	if (constrained) {
	    if ((*i).selected)
		start = (*i).start;
	    else
		start= 0;
	}
	else
	    start = (*i).start; 

    return start;
}

/** Use this function to return the stop index of an array
    dimension.  If the array is constrained (indicated with the
    <i>constrained</i> argument), the stop index of the constrained
    array is returned (or zero if the dimension in question is not
    selected at all).  See also <tt>dimension_start()</tt> and
    <tt>dimension_stride()</tt>.

    @brief Return the stop index of the constraint.
    @deprecated Methods which use the Pix objects are deprecated in favor of
    the methods which use iterators.

    @param p The Pix index of the dimension.
    @param constrained If this parameter is TRUE, the function
    returns the stop index only if the dimension is  constrained
    (subject to a start, stop, or stride constraint).  If
    the dimension is not constrained, the function returns zero.  If it
    is FALSE, the function returns the stop index whether or not
    the dimension is constrained. 
    @return The desired stop index.
*/
int 
Array::dimension_stop(Pix p, bool constrained) 
{ 
    ArrayIterAdapter *iter = (ArrayIterAdapter *)p.getIterator() ;
    Dim_iter i = iter->getIterator() ;
    return dimension_stop( i, constrained ) ;
}

/** Use this function to return the stop index of an array
    dimension.  If the array is constrained (indicated with the
    <i>constrained</i> argument), the stop index of the constrained
    array is returned (or zero if the dimension in question is not
    selected at all).  See also <tt>dimension_start()</tt> and
    <tt>dimension_stride()</tt>.

    @brief Return the stop index of the constraint.

    @param i The dimension.
    @param constrained If this parameter is TRUE, the function
    returns the stop index only if the dimension is  constrained
    (subject to a start, stop, or stride constraint).  If
    the dimension is not constrained, the function returns zero.  If it
    is FALSE, the function returns the stop index whether or not
    the dimension is constrained. 
    @return The desired stop index.
*/
int 
Array::dimension_stop(Dim_iter i, bool constrained) 
{ 
    int stop = 0;

    if (!_shape.empty())
	if (constrained) {
	    if ((*i).selected)
		stop = (*i).stop;
	    else
		stop= 0;
	}
	else
	    stop = (*i).stop; 

    return stop;
}

/** Use this function to return the stride value of an array
    dimension.  If the array is constrained (indicated with the
    <i>constrained</i> argument), the stride value of the constrained
    array is returned (or zero if the dimension in question is not
    selected at all).  See also <tt>dimension_stop()</tt> and
    <tt>dimension_start()</tt>.

    @brief Returns the stride value of the constraint.
    @deprecated Methods which use the Pix objects are deprecated in favor of
    the methods which use iterators.

    @param p The Pix index of the dimension.
    @param constrained If this parameter is TRUE, the function
    returns the stride value only if the dimension is constrained
    (subject to a start, stop, or stride constraint).  If
    the dimension is not constrained, the function returns zero.  If it
    is FALSE, the function returns the stride value whether or not
    the dimension is constrained. 
    @return The stride value requested, or zero, if <i>constrained</i>
    is TRUE and the dimension is not selected.
*/
int 
Array::dimension_stride(Pix p, bool constrained) 
{ 
    ArrayIterAdapter *iter = (ArrayIterAdapter *)p.getIterator() ;
    Dim_iter i = iter->getIterator() ;
    return dimension_stride( i, constrained ) ;
}

/** Use this function to return the stride value of an array
    dimension.  If the array is constrained (indicated with the
    <i>constrained</i> argument), the stride value of the constrained
    array is returned (or zero if the dimension in question is not
    selected at all).  See also <tt>dimension_stop()</tt> and
    <tt>dimension_start()</tt>.

    @brief Returns the stride value of the constraint.

    @param i The dimension.
    @param constrained If this parameter is TRUE, the function
    returns the stride value only if the dimension is constrained
    (subject to a start, stop, or stride constraint).  If
    the dimension is not constrained, the function returns zero.  If it
    is FALSE, the function returns the stride value whether or not
    the dimension is constrained. 
    @return The stride value requested, or zero, if <i>constrained</i>
    is TRUE and the dimension is not selected.
*/
int 
Array::dimension_stride(Dim_iter i, bool constrained) 
{ 
    int stride = 0;

    if (!_shape.empty())
	if (constrained) {
	    if ((*i).selected)
		stride = (*i).stride;
	    else
		stride= 0;
	}
	else
	    stride = (*i).stride; 

    return stride;
}

/** This function returns the name of the dimension indicated with
    <i>p</i>.  Since this method is public, it is possible to call it
    before the Array object has been properly initialized.  This will
    cause an exception.  So don't do that.

    @brief Returns the name of the specified dimension. 
    @deprecated Methods which use the Pix objects are deprecated in favor of
    the methods which use iterators.

    @param p The Pix index of the dimension.
    @return A pointer to a string containing the dimension name.
*/
string
Array::dimension_name(Pix p) 
{ 
    ArrayIterAdapter *iter = (ArrayIterAdapter *)p.getIterator() ;
    Dim_iter i = iter->getIterator() ;
    return dimension_name( i ) ;
}

/** This function returns the name of the dimension indicated with
    <i>p</i>.  Since this method is public, it is possible to call it
    before the Array object has been properly initialized.  This will
    cause an exception.  So don't do that.

    @brief Returns the name of the specified dimension. 

    @param i The dimension.
    @return A pointer to a string containing the dimension name.
*/
string
Array::dimension_name(Dim_iter i) 
{ 
  // Jose Garcia
  // Since this method is public, it is possible for a user
  // to call it before the Array object has been properly set
  // this will cause an exception which is the user's fault.
  // (User in this context is the developer of the surrogate library.)
  if (_shape.empty())
      throw  InternalErr(__FILE__, __LINE__, 
			 "*This* array has no dimensions.");
  return (*i).name; 
}

/** Prints a declaration for the Array.  This is what appears in a
    DDS.  If the Array is constrained, the declaration will allocate
    only enough space for the constrained values.

    @deprecated Use the FILE * interface.

    @brief Prints a DDS entry for the Array.
    @param os An output stream to print on.
    @param space A string containing spaces to precede the
    declaration.
    @param print_semi A boolean indicating whether to print a
    semi-colon after the declaration.  (TRUE means ``print a
    semi-colon.'') 
    @param constraint_info A boolean value.  See
    <tt>BaseType::print_decl()</tt>.
    @param constrained This argument should be TRUE if the Array is
    constrained, and FALSE otherwise.
*/
void
Array::print_decl(ostream &os, string space, bool print_semi,
		  bool constraint_info, bool constrained)
{
    if (constrained && !send_p())
	return;

    // print it, but w/o semicolon
    var()->print_decl(os, space, false, constraint_info, constrained); 

    for (Dim_citer i = _shape.begin(); i != _shape.end(); i++) {
	if (constrained && !((*i).selected))
	    continue;
	os << "[";
	if ((*i).name != "")
	    os << id2www((*i).name) << " = ";
	if (constrained)
	    os << (*i).c_size << "]";
	else
	    os << (*i).size << "]";
    }

    if (print_semi)
	os << ";" << endl;
}

/** Prints a declaration for the Array.  This is what appears in a
    DDS.  If the Array is constrained, the declaration will reflect
    the size of the Array once the constraint is applied.

    @brief Prints a DDS entry for the Array.

    @param out Write the output to this FILE *.
    @param space A string containing spaces to precede the
    declaration.
    @param print_semi A boolean indicating whether to print a
    semi-colon after the declaration.  (TRUE means ``print a
    semi-colon.'') 
    @param constraint_info A boolean value.  See
    <tt>BaseType::print_decl()</tt>.
    @param constrained This argument should be TRUE if the Array is
    constrained, and FALSE otherwise.
*/
void
Array::print_decl(FILE *out, string space, bool print_semi,
		  bool constraint_info, bool constrained)
{
    if (constrained && !send_p())
	return;

    // print it, but w/o semicolon
    var()->print_decl(out, space, false, constraint_info, constrained); 

    for (Dim_citer i = _shape.begin(); i != _shape.end(); i++)
    {
	if (constrained && !((*i).selected))
	    continue;
	fprintf( out, "[" ) ;
	if ((*i).name != "")
	{
	    fprintf( out, "%s = ", id2www((*i).name).c_str() ) ;
	}
	if (constrained)
	{
	    fprintf( out, "%d]", (*i).c_size ) ;
	}
	else
	{
	    fprintf( out, "%d]", (*i).size ) ;
	}
    }

    if( print_semi )
    {
	fprintf( out, ";\n" ) ;
    }
}

void
Array::print_xml(FILE *out, string space, bool constrained) 
{
    print_xml_core(out, space, constrained, "Array");
}

void
Array::print_as_map_xml(FILE *out, string space, bool constrained)
{
    print_xml_core(out, space, constrained, "Map");
}

class PrintArrayDim : public unary_function<Array::dimension&, void> {
    FILE *d_out;
    string d_space;
    bool d_constrained;
public:
    PrintArrayDim(FILE *o, string s, bool c) 
	: d_out(o), d_space(s), d_constrained(c) {}

    void operator()(Array::dimension &d) {
	int size = d_constrained ? d.c_size: d.size;
	if (d.name.empty())
	    fprintf(d_out, "%s<dimension size=\"%d\"/>\n", d_space.c_str(), 
		    size);
	else
	    fprintf(d_out, "%s<dimension name=\"%s\" size=\"%d\"/>\n", 
		    d_space.c_str(), id2xml(d.name).c_str(), size);
    }
};
	
void
Array::print_xml_core(FILE *out, string space, bool constrained, string tag)
{
    if (constrained && !send_p())
	return;

    fprintf(out, "%s<%s", space.c_str(), tag.c_str());
    if (!name().empty())
	fprintf(out, " name=\"%s\"", id2xml(name()).c_str());
    fprintf(out , ">\n");

    get_attr_table().print_xml(out, space + "    ", constrained);

    BaseType *btp = var();
    string tmp_name = btp->name();
    btp->set_name("");
    btp->print_xml(out, space + "    ", constrained);
    btp->set_name(tmp_name);

    for_each(dim_begin(), dim_end(), 
	     PrintArrayDim(out, space + "    ", constrained));

    fprintf(out, "%s</%s>\n", space.c_str(), tag.c_str());
}

/** Prints the value of the entire (constrained) array.

    @deprecated Use the FILE * interface.
    @param os The output stream to print on.
    @param space The space to use in printing.
    @param print_decl_p A boolean value indicating whether you want
    the Array declaration to precede the Array value.
    @brief Print the value given the current constraint.
*/
unsigned int
Array::print_array(ostream &os, unsigned int index, unsigned int dims, 
		   unsigned int shape[])
{
    if (dims == 1) {
	os << "{";
	for (unsigned i = 0; i < shape[0]-1; ++i) {
	    var(index++)->print_val(os, "", false);
	    os << ", ";
	}
	var(index++)->print_val(os, "", false);
	os << "}";

	return index;
    }
    else {
	os << "{";
	// Fixed an off-by-one error in the following loop. Since the array
	// length is shape[dims-1]-1 *and* since we want one less dimension
	// than that, the correct limit on this loop is shape[dims-2]-1. From
	// Todd Karakasian.
	// The saga continues; the loop test should be `i < shape[0]-1'. jhrg
	// 9/12/96.
	for (unsigned i = 0; i < shape[0]-1; ++i) {
	    index = print_array(os, index, dims - 1, shape + 1);
	    os << ",";		// Removed the extra `}'. Also from Todd
	}
	index = print_array(os, index, dims - 1, shape + 1);
	os << "}";

	return index;
    }
}

/** Prints the values in ASCII of the entire (constrained) array. This method
    Attempts to make an aesthetically pleasing display. However, it is
    primarily intended for debugging purposes.

    @param out Write the output to this FILE *.
    @param space The space to use in printing.
    @param print_decl_p A boolean value indicating whether you want
    the Array declaration to precede the Array value.
    @brief Print the value given the current constraint.
*/
unsigned int
Array::print_array(FILE *out, unsigned int index, unsigned int dims, 
		   unsigned int shape[])
{
    if (dims == 1) {
	fprintf( out, "{" ) ;
	for (unsigned i = 0; i < shape[0]-1; ++i) {
	    var(index++)->print_val(out, "", false);
	    fprintf( out, ", " ) ;
	}
	var(index++)->print_val(out, "", false);
	fprintf( out, "}" ) ;

	return index;
    }
    else {
	fprintf( out, "{" ) ;
	// Fixed an off-by-one error in the following loop. Since the array
	// length is shape[dims-1]-1 *and* since we want one less dimension
	// than that, the correct limit on this loop is shape[dims-2]-1. From
	// Todd Karakasian.
	// The saga continues; the loop test should be `i < shape[0]-1'. jhrg
	// 9/12/96.
	for (unsigned i = 0; i < shape[0]-1; ++i) {
	    index = print_array(out, index, dims - 1, shape + 1);
	    fprintf( out, "," ) ;	// Removed the extra `}'. Also from Todd
	}
	index = print_array(out, index, dims - 1, shape + 1);
	fprintf( out, "}" ) ;

	return index;
    }
}

// print the value given the current constraint.

void 
Array::print_val(ostream &os, string space, bool print_decl_p)
{
    // print the declaration if print decl is true.
    // for each dimension,
    //   for each element, 
    //     print the array given its shape, number of dimensions.
    // Add the `;'
    
    if (print_decl_p) {
	print_decl(os, space, false, false, false);
	os << " = ";
    }

    unsigned int dims = dimensions(true);
    unsigned int *shape = new unsigned int[dims];
    unsigned int index = 0;
    for (Dim_iter i = _shape.begin(); i != _shape.end(); i++)
	shape[index++] = dimension_size(i, true);

    print_array(os, 0, dims, shape);
    delete [] shape; shape = 0;

    if (print_decl_p) {
	os << ";" << endl;
    }
}

void 
Array::print_val(FILE *out, string space, bool print_decl_p)
{
    // print the declaration if print decl is true.
    // for each dimension,
    //   for each element, 
    //     print the array given its shape, number of dimensions.
    // Add the `;'
    
    if (print_decl_p) {
	print_decl(out, space, false, false, false);
	fprintf( out, " = " ) ;
    }

    unsigned int dims = dimensions(true);
    unsigned int *shape = new unsigned int[dims];
    unsigned int index = 0;
    for (Dim_iter i = _shape.begin(); i != _shape.end(); i++)
	shape[index++] = dimension_size(i, true);

    print_array(out, 0, dims, shape);
    delete [] shape; shape = 0;

    if (print_decl_p) {
	fprintf( out, ";\n" ) ;
    }
}

/** This function checks semantic features of the Array.  Currently,
    the only check specific to the Array is that there must be
    dimensions.  The rest is inherited from
    <tt>BaseType::check_semantics()</tt>.

    @brief Check semantic features of the Array.
    @return A boolean value.  FALSE means there was a problem.
*/

bool
Array::check_semantics(string &msg, bool)
{
    bool sem = BaseType::check_semantics(msg) && !_shape.empty();

    if (!sem)
	msg = "An array variable must have dimensions";

    return sem;
}

// $Log: Array.cc,v $
// Revision 1.69  2005/04/07 22:32:47  jimg
// Updated doxygen comments: fixed errors; updated comments about set_read_p.
// Removed the VirtualCtor classes. Added a README about the factory
// classes.
//
// Revision 1.68  2005/03/23 23:11:19  pwest
// algorithm and functional needed on sun and can be included on Linux, so no need to wrap in ifdef WIN32
//
// Revision 1.67  2004/11/16 22:50:20  jimg
// Fixed tests. Also fixed a bug intorduced in Vector where a template
// with no name caused some software (any code which depends on the
// template having the same name as the array) to fail.
//
// Revision 1.66  2004/11/16 17:53:14  jimg
// Added subclass version of add_var(). This version looks at the variable
// being added and, if it's an Array adds the template variable. It calls
// Vector::add_var() to actually add the variable.
//
// Revision 1.65  2004/07/07 21:08:46  jimg
// Merged with release-3-4-8FCS
//
// Revision 1.59.2.4  2004/07/02 20:41:51  jimg
// Removed (commented) the pragma interface/implementation lines. See
// the ChangeLog for more details. This fixes a build problem on HP/UX.
//
// Revision 1.64  2004/06/27 11:28:01  rmorris
// Get it to compile under win32 by including <functional> and <algorithm>
//
// Revision 1.63  2004/03/10 16:29:18  jimg
// Repairs to the methods which provide access using iterators. These
// were using '*_iter &' type params and that made newer versions of g++
// gag. I'm not absolutely sure what the problem was, but making the
// parameters regular value params and not references fixed it.
//
// Revision 1.62  2004/02/19 19:42:52  jimg
// Merged with release-3-4-2FCS and resolved conflicts.
//
// Revision 1.59.2.3  2004/02/11 22:26:45  jimg
// Changed all calls to delete so that whenever we use 'delete x' or
// 'delete[] x' the code also sets 'x' to null. This ensures that if a
// pointer is deleted more than once (e.g., when an exception is thrown,
// the method that throws may clean up and then the catching method may
// also clean up) the second, ..., call to delete gets a null pointer
// instead of one that points to already deleted memory.
//
// Revision 1.59.2.2  2004/01/29 06:51:49  jimg
// Fixed up the doxygen comments.
//
// Revision 1.61  2003/12/08 18:02:29  edavis
// Merge release-3-4 into trunk
//
// Revision 1.59.2.1  2003/09/06 22:37:50  jimg
// Updated the documentation.
//
// Revision 1.60  2003/05/23 03:24:56  jimg
// Changes that add support for the DDX response. I've based this on Nathan
// Potter's work in the Java DAP software. At this point the code can
// produce a DDX from a DDS and it can merge attributes from a DAS into a
// DDS to produce a DDX fully loaded with attributes. Attribute aliases
// are not supported yet. I've also removed all traces of strstream in
// favor of stringstream. This code should no longer generate warnings
// about the use of deprecated headers.
//
// Revision 1.59  2003/04/22 19:40:27  jimg
// Merged with 3.3.1.
//
// Revision 1.58  2003/02/21 00:14:24  jimg
// Repaired copyright.
//
// Revision 1.57.2.1  2003/02/21 00:10:06  jimg
// Repaired copyright.
//
// Revision 1.57  2003/01/23 00:22:23  jimg
// Updated the copyright notice; this implementation of the DAP is
// copyrighted by OPeNDAP, Inc.
//
// Revision 1.56  2003/01/15 19:24:39  pwest
// Removing IteratorAdapterT and replacing with non-templated versions.
//
// Revision 1.55  2003/01/10 19:46:39  jimg
// Merged with code tagged release-3-2-10 on the release-3-2 branch. In many
// cases files were added on that branch (so they appear on the trunk for
// the first time).
//
// Revision 1.48.4.13  2002/12/31 16:43:20  rmorris
// Patches to handle some of the fancier template code under VC++ 6.0.
//
// Revision 1.48.4.12  2002/12/17 22:35:02  pwest
// Added and updated methods using stdio. Deprecated methods using iostream.
//
// Revision 1.48.4.11  2002/12/01 14:37:52  rmorris
// Smalling changes for the win32 porting and maintenance work.
//
// Revision 1.48.4.10  2002/10/28 21:17:43  pwest
// Converted all return values and method parameters to use non-const iterator.
// Added operator== and operator!= methods to IteratorAdapter to handle Pix
// problems.
//
// Revision 1.48.4.9  2002/09/22 14:24:58  rmorris
// Changed the 1st arg in add_constraint(Dim_iter &i,...).  The l-value
// designation isn't used within and VC++ won't swallow passing a non-lvalue
// in - so "&" was removed and all is well.
//
// Revision 1.48.4.8  2002/09/12 22:49:57  pwest
// Corrected signature changes made with Pix to IteratorAdapter changes. Rather
// than taking a reference to a Pix, taking a Pix value.
//
// Revision 1.48.4.7  2002/09/05 22:52:54  pwest
// Replaced the GNU data structures SLList and DLList with the STL container
// class vector<>. To maintain use of Pix, changed the Pix.h header file to
// redefine Pix to be an IteratorAdapter. Usage remains the same and all code
// outside of the DAP should compile and link with no problems. Added methods
// to the different classes where Pix is used to include methods to use STL
// iterators. Replaced the use of Pix within the DAP to use iterators instead.
// Updated comments for documentation, updated the test suites, and added some
// unit tests. Updated the Makefile to remove GNU/SLList and GNU/DLList.
//
// Revision 1.48.4.6  2002/08/08 06:54:56  jimg
// Changes for thread-safety. In many cases I found ugly places at the
// tops of files while looking for globals, et c., and I fixed them up
// (hopefully making them easier to read, ...). Only the files RCReader.cc
// and usage.cc actually use pthreads synchronization functions. In other
// cases I removed static objects where they were used for supposed
// improvements in efficiency which had never actually been verified (and
// which looked dubious).
//
// Revision 1.54  2002/06/18 15:36:24  tom
// Moved comments and edited to accommodate doxygen documentation-generator.
//
// Revision 1.53  2002/06/03 22:21:15  jimg
// Merged with release-3-2-9
//
// Revision 1.52  2002/05/23 15:22:39  tom
// modified for doxygen
//
// Revision 1.48.4.5  2002/05/22 16:57:51  jimg
// I modified the `data type classes' so that they do not need to be
// subclassed for clients. It might be the case that, for a complex client,
// subclassing is still the best way to go, but you're not required to do
// it anymore.
//
// Revision 1.48.4.4  2001/10/30 06:51:32  rmorris
// Omit use of unescaped newline in string constant.  MS VC++ won't tolerate.
//
// Revision 1.51  2001/09/28 17:50:07  jimg
// Merged with 3.2.7.
//
// Revision 1.48.4.3  2001/09/25 20:39:15  jimg
// Changed update_length() so that it's no longer necessary to call
// clear_constraint before setting dimension constraints.
//
// Revision 1.50  2001/08/24 17:46:22  jimg
// Resolved conflicts from the merge of release 3.2.6
//
// Revision 1.48.4.2  2001/07/28 01:10:41  jimg
// Some of the numeric type classes did not have copy ctors or operator=.
// I added those where they were needed.
// In every place where delete (or delete []) was called, I set the pointer
// just deleted to zero. Thus if for some reason delete is called again
// before new memory is allocated there won't be a mysterious crash. This is
// just good form when using delete.
// I added calls to www2id and id2www where appropriate. The DAP now handles
// making sure that names are escaped and unescaped as needed. Connect is
// set to handle CEs that contain names as they are in the dataset (see the
// comments/Log there). Servers should not handle escaping or unescaping
// characters on their own.
//
// Revision 1.49  2001/06/15 23:49:01  jimg
// Merged with release-3-2-4.
//
// Revision 1.48.4.1  2001/06/05 06:49:19  jimg
// Added the Constructor class which is to Structures, Sequences and Grids
// what Vector is to Arrays and Lists. This should be used in future
// refactorings (I thought it was going to be used for the back pointers).
// Introduced back pointers so children can refer to their parents in
// hierarchies of variables.
// Added to Sequence methods to tell if a child sequence is done
// deserializing its data.
// Fixed the operator=() and copy ctors; removed redundency from
// _duplicate().
// Changed the way serialize and deserialize work for sequences. Now SOI and
// EOS markers are written for every `level' of a nested Sequence. This
// should fixed nested Sequences. There is still considerable work to do
// for these to work in all cases.
//
// Revision 1.48  2000/09/22 02:17:18  jimg
// Rearranged source files so that the CVS logs appear at the end rather than
// the start. Also made the ifdef guard symbols use the same naming scheme and
// wrapped headers included in other headers in those guard symbols (to cut
// down on extraneous file processing - See Lakos).
//
// Revision 1.47  2000/09/21 16:22:06  jimg
// Merged changes from Jose Garcia that add exceptions to the software.
// Many methods that returned error codes now throw exectptions. There are
// two classes which are thrown by the software, Error and InternalErr.
// InternalErr is used to report errors within the library or errors using
// the library. Error is used to reprot all other errors. Since InternalErr
// is a subclass of Error, programs need only to catch Error.
//
// Revision 1.46  2000/07/09 22:05:35  rmorris
// Changes to increase portability, minimize ifdef's for win32 and account
// for differences in the iostreams implementations.
//
// Revision 1.45  2000/06/16 18:14:59  jimg
// Merged with 3.1.7
//
// Revision 1.43.6.1  2000/06/14 16:59:00  jimg
// Added instrumentation for the dtor.
//
// Revision 1.44  2000/06/07 18:06:57  jimg
// Merged the pc port branch
//
// Revision 1.43.20.1  2000/06/02 18:11:19  rmorris
// Mod's for Port to Win32.
//
// Revision 1.43.14.2  2000/02/17 05:03:12  jimg
// Added file and line number information to calls to InternalErr.
// Resolved compile-time problems with read due to a change in its
// parameter list given that errors are now reported using exceptions.
//
// Revision 1.43.14.1  2000/01/28 22:14:03  jgarcia
// Added exception handling and modify add_var to get a copy of the object
//
// Revision 1.43  1999/04/29 02:29:26  jimg
// Merge of no-gnu branch
//
// Revision 1.42  1998/12/15 20:50:23  jimg
// Added fix for strides <= 0 (which results in division by zero).
//
// Revision 1.41.4.1  1999/02/02 21:56:54  jimg
// String to string version
//
// Revision 1.41  1997/12/15 18:10:19  jimg
// Changed check_semantics() so that it returns an error message instead of
// printing one (thus it now works like all the other implementations of
// check_semantics().
//
// Revision 1.40  1997/03/08 19:01:52  jimg
// Changed default param to check_semantics() from  to String()
// and removed the default from the argument list in the mfunc definition
//
// Revision 1.39  1997/02/28 01:28:01  jimg
// Changed check_semantics() so that it now returns error messages in a String
// object (passed by reference).
//
// Revision 1.38  1996/09/12 21:02:35  jimg
// Fixed a nasty bug in print_array (private member function) where recursive
// calls were made in the wrong order causing 3+ dimension arrays to print many
// more values than actually exist in the array.
//
// Revision 1.37  1996/08/26 21:12:48  jimg
// Changes for version 2.07
//
// Revision 1.36  1996/08/13 16:46:14  jimg
// Added bounds checking to the add_constraint member function.
// add_constraint() now returns false when a bogus constraint is used.
//
// Revision 1.35  1996/06/11 17:21:31  jimg
// Fixed a bug in clear_constraint(); the dimension variables in the list
// _SHAPE were not correctly `cleared', now they are.
//
// Revision 1.34  1996/05/31 23:29:17  jimg
// Updated copyright notice.
//
// Revision 1.33  1996/05/29 22:08:32  jimg
// Made changes necessary to support CEs that return the value of a function
// instead of the value of a variable. This was done so that it would be
// possible to translate Sequences into Arrays without first reading the
// entire sequence over the network.
//
// Revision 1.32  1996/05/14 06:41:17  jimg
// Update for
//
// Revision 1.32  1996/05/10 21:18:16  jimg
// Fixed off-by-one error in add_constraint().
//
// Revision 1.31  1996/05/06 21:15:58  jimg
// Added the member functions dimension_start, _stop and _stride to this class.
// Changed the first argument of add_constraint from Pix &p to Pix p (the
// member function does not change its argument).
//
// Revision 1.30  1996/04/05 00:21:17  jimg
// Compiled with g++ -Wall and fixed various warnings.
//
// Revision 1.29  1996/03/05 18:48:24  jimg
// Fixed bugs in the print_array software.
// Fixed the ctor and related CE member functions (so that projections work).
//
// Revision 1.28  1996/02/01 22:22:42  jimg
// Merged changes between DODS-1.1 and DODS 2.x.
//
// Revision 1.27  1995/12/09  01:06:29  jimg
// Added changes so that relational operators will work properly for all the
// datatypes (including Sequences). The relational ops are evaluated in
// DDS::eval_constraint() after being parsed by DDS::parse_constraint().
//
// Revision 1.26  1995/12/06  21:40:58  jimg
// Added reset_constraint(), clear_constraint() and add_constraint() to manage
// the new constraint-related members (in struct dimension).
// Fixed variour member functions to return information about sizes,
// ... correctly both before and after constraints are set.
//
// Revision 1.25  1995/11/22  22:31:02  jimg
// Modified so that the Vector class is now the parent class.
//
// Revision 1.24  1995/10/23  23:20:44  jimg
// Added _send_p and _read_p fields (and their accessors) along with the
// virtual mfuncs set_send_p() and set_read_p().
//
// Revision 1.23  1995/08/26  00:31:21  jimg
// Removed code enclosed in #ifdef NEVER #endif.
//
// Revision 1.22  1995/08/22  23:45:33  jimg
// Removed DBMALLOC code.
// Added set_vec/vec mfuncs so that non-Array software can access the
// BaseType * vector.
// Changed names read_val and store_val to buf2val and val2buf. The old names
// remain.
// removed the card member function: used the new _type enum with a switch in
// its place.
//
// Revision 1.21.2.4  1995/09/27 21:49:03  jimg
// Fixed casts.
//
// Revision 1.21.2.3  1995/09/27  19:06:56  jimg
// Add casts to `cast away' const and unsigned in places where we call various
// xdr functions (which don't know about, or use, const or unsigned.
//
// Revision 1.21.2.2  1995/09/14  20:59:50  jimg
// Fixed declaration of, and calls to, _duplicate() by changing the formal
// param from a pointer to a reference.
//
// Revision 1.21.2.1  1995/07/11  18:17:09  jimg
// Added cast of xdr_array (used in BaseType's constructor) to xdrproc_t.
//
// Revision 1.21  1995/07/09  21:28:48  jimg
// Added copyright notice.
//
// Revision 1.20  1995/05/10  15:33:52  jimg
// Failed to change `config.h' to `config.h' in these files.
//
// Revision 1.19  1995/05/10  13:45:01  jimg
// Changed the name of the configuration header file from `config.h' to
// `config.h' so that other libraries could have header files which were
// installed in the DODS include directory without overwriting this one. Each
// config header should follow the convention config_<name>.h.
//
// Revision 1.18  1995/04/28  19:53:46  reza
// First try at adding constraints capability.
// Enforce a new size calculated from constraint expression.
//
// Revision 1.17  1995/03/16  17:22:58  jimg
// Added include of config.h before all other includes.
// Fixed deletes of buffers in read_val().
// Added initialization of _buf in ctor.
//
// Revision 1.16  1995/03/04  14:34:40  jimg
// Major modifications to the transmission and representation of values:
// Added card() virtual function which is true for classes that
// contain cardinal types (byte, int float, string).
// Changed the representation of Str from the C rep to a C++
// class represenation.
// Chnaged read_val and store_val so that they take and return
// types that are stored by the object (e.g., inthe case of Str
// an URL, read_val returns a C++ String object).
// Modified Array representations so that arrays of card()
// objects are just that - no more storing strings, ... as
// C would store them.
// Arrays of non cardinal types are arrays of the DODS objects (e.g.,
// an array of a structure is represented as an array of Structure
// objects).
//
// Revision 1.15  1995/02/10  02:22:54  jimg
// Added DBMALLOC includes and switch to code which uses malloc/free.
// Private and protected symbols now start with `_'.
// Added new accessors for name and type fields of BaseType; the old ones
// will be removed in a future release.
// Added the store_val() mfunc. It stores the given value in the object's
// internal buffer.
// Made both List and Str handle their values via pointers to memory.
// Fixed read_val().
// Made serialize/deserialize handle all malloc/free calls (even in those
// cases where xdr initiates the allocation).
// Fixed print_val().
//
// Revision 1.14  1995/01/19  20:05:21  jimg
// ptr_duplicate() mfunc is now abstract virtual.
// Array, ... Grid duplicate mfuncs were modified to take pointers, not
// referenves.
//
// Revision 1.13  1995/01/11  15:54:39  jimg
// Added modifications necessary for BaseType's static XDR pointers. This
// was mostly a name change from xdrin/out to _xdrin/out.
// Removed the two FILE pointers from ctors, since those are now set with
// functions which are friends of BaseType.
//
// Revision 1.12  1994/12/19  20:52:45  jimg
// Minor modifications to the print_val mfunc.
//
// Revision 1.11  1994/12/16  20:13:31  dan
// Fixed serialize() and deserialize() for arrays of strings.
//
// Revision 1.10  1994/12/14  20:35:36  dan
// Added dimensions() member function to return number of dimensions
// contained in the array.
// Removed alloc_buf() and free_buf() member functions and placed them
// in util.cc.
//
// Revision 1.9  1994/12/14  17:50:34  dan
// Modified serialize() and deserialize() member functions to special
// case BaseTypes 'Str' and 'Url'.  These special cases do not call
// xdr_array, but iterate through the arrays using calls to XDR_STR.
// Modified print_val() member function to handle arrays of different
// BaseTypes.
// Modified append_dim() member function for initializing its dimension
// components.
// Removed dim() member function.
//
// Revision 1.7  1994/12/09  21:36:33  jimg
// Added support for named array dimensions.
//
// Revision 1.6  1994/12/08  15:51:41  dan
// Modified size() member to return cumulative size of all dimensions
// given the variable basetype.
// Modified serialize() and deserialize() member functions for data
// transmission using XDR.
//
// Revision 1.5  1994/11/22  20:47:45  dan
// Modified size() to return total number of elements.
// Fixed errors in deserialize (multiple returns).
//
// Revision 1.4  1994/11/22  14:05:19  jimg
// Added code for data transmission to parts of the type hierarchy. Not
// complete yet.
// Fixed erros in type hierarchy headers (typos, incorrect comments, ...).
//
// Revision 1.3  1994/10/17  23:34:42  jimg
// Added code to print_decl so that variable declarations are pretty
// printed.
// Added private mfunc duplicate().
// Added ptr_duplicate().
// Added Copy ctor, dtor and operator=.
//
// Revision 1.2  1994/09/23  14:31:36  jimg
// Added check_semantics mfunc.
// Added sanity checking for access to shape list (is it empty?).
// Added cvs log listing to Array.cc.
//

