
// -*- mode: c++; c-basic-offset:4 -*-

// This file is part of libdap, A C++ implementation of the OPeNDAP Data
// Access Protocol.

// Copyright (c) 2002,2003,2005 OPeNDAP, Inc.
// Author: James Gallagher <jgallagher@opendap.org>
//         Reza Nekovei <reza@intcomm.net>
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
// You can contact OPeNDAP, Inc. at PO Box 112, Saunderstown, RI. 02874-0112.

// (c) COPYRIGHT URI/MIT 1994-1996
// Please read the full copyright statement in the file COPYRIGHT.
//
// Authors:
//      reza            Reza Nekovei (reza@intcomm.net)

// Interface for subclass of Connect: subclassed to add ctor for netcdf.
//
// jhrg 9/29/94

#ifndef _ncconnect_h
#define _ncconnect_h 1

#include "AISConnect.h"		// Changed to AISConnect. 03/11/03 jhrg
#include "NCTypeFactory.h"
#include "ClientParams.h"
#include "nc_util.h"

/** NCConnect assumes that the DAS and DDS are flat. This class is the basis
    for the netCDF client library. 

    @todo Design change: To support hierarchies, add \c id fields to the
    variables and attribtues. This will support looking up the
    variable/attribute id without counting every time. Also, create a data
    structure which can map those id numbers directly to the variables and
    attributes.

    @note Document how this code works here. 

    @todo Update; remove deprecated Connect methods.

    @author Reza Nekovei, James Gallagher */
class NCConnect: public AISConnect {
private:
    const ClientParams &d_client_params;
    AttrTable *d_global_attributes;
    DDS d_constrained_dds;     // Current (Constrained) dds
    DDS d_translated_dds;      // Current DDS after translation
    NCTypeFactory *d_factory;   // Instantiate the NC types.
    
    int d_ncid;			// used only if LOCAL is true, otherwise == -1
    int d_nvars;		// Number of variables in this file
    int d_recdim;		// Dimension ID for unlimited dim.
    int d_ndims;		// Number of dimensions found
    string d_dim_name[MAX_NC_DIMS];
    int d_dim_size[MAX_NC_DIMS];

    string d_proj_ce;
    string d_sel_ce;
    
    void parse_array_dims(DDS &dds);
    void parse_grid_dims(DDS &dds);
    void parse_string_dims(DDS &dds);
    void set_recdim(DAS &das);
    void set_global_attributes();
    AttrTable *flatten_attributes(AttrTable *at);
    
    // Suppres these.
    NCConnect() : d_client_params(ClientParams("")) {
        throw InternalErr(__FILE__, __LINE__, "Unimplemented.");
    }
    NCConnect &operator=(const NCConnect &rhs) {
        throw InternalErr(__FILE__, __LINE__, "Unimplemented.");
    }
    
    friend class NCConnectTest;

public:
    NCConnect(const string &name, const ClientParams &cp);
    virtual ~NCConnect();
    
    void init_remote_source(const string &ce) throw(Error);

    int get_ncid();
    void set_ncid(int id);
    int get_ndims();
    int get_nvars();
    void set_nvars(int n);
    int recdim();
    int dim_size(const int dimid);
    const string &dim_name(const int dimid);

    DDS &get_constrained_dds();
    DDS &get_translated_dds();

    void translate_dds();

    /// Store the projection and selection parts of the initial CE
    void store_ce(const string &ce);
    
    /// Get the selection part of the stored CE.
    string get_sel_ce() { return d_sel_ce; }
    /// Get the projection part of the stored CE.
    string get_proj_ce() { return d_proj_ce; }
    
    /// Get a variable
    BaseType *get_variable(int varid) throw(Error);
    /// Get the number of attribtues
    int get_num_attr(int varid);
    /// Get an attribute
    AttrTable::Attr_iter get_attribute_iter(int varid, int attnum) 
    	throw(Error);
    /// Get the global attribute table
    AttrTable &get_global_attributes();
    /// Get an attribute table
    AttrTable &get_attribute_table(int varid) throw(Error);
    /// Get the values of an attribute
    char *get_raw_values(int varid, const char *name, size_t *count,
        nc_type *datatype) throw(Error);
    /// Get information about a variable
    void var_info(BaseType *bt, nc_type *typep, int *ndimsp, int dims[]) 
        throw(InternalErr);
};

/* 
 * $Log: NCConnect.h,v $
 * Revision 1.27  2005/04/11 18:38:20  jimg
 * Fixed a problem with NCSequence where nested sequences were not flagged
 * but instead were translated. The extract_values software cannot process a
 * nested sequence yet. Now the code inserts an attribute that notes that a
 * nested sequence has been elided.
 *
 * Revision 1.26  2005/03/31 00:04:51  jimg
 * Modified to use the factory class in libdap++ 3.5.
 *
 * Revision 1.25  2005/03/23 19:30:21  jimg
 * Changed prototype for flatten_attributes() to match the new code.
 *
 * Revision 1.24  2005/02/26 00:43:20  jimg
 * Check point: This version of the CL can now translate strings from the
 * server into char arrays. This is controlled by two things: First a
 * compile-time directive STRING_AS_ARRAY can be used to remove/include
 * this feature. When included in the code, only Strings associated with
 * variables created by the translation process will be turned into char
 * arrays. Other String variables are assumed to be single character strings
 * (although there may be a bug with the way these are handled, see
 * NCAccess::extract_values()).
 *
 * Revision 1.23  2005/02/17 23:44:13  jimg
 * Modifications for processing of command line projections combined
 * with the limit stuff and projection info passed in from the API. I also
 * consolodated some of the code by moving d_source from various
 * classes to NCAccess. This may it so that DODvario() could be simplified
 * as could build_constraint() and store_projection() in NCArray.
 *
 * Revision 1.22  2005/01/26 23:25:51  jimg
 * Implemented a fix for Sequence access by row number when talking to a
 * 3.4 or earlier server (which contains a bug in is_end_of_rows()).
 *
 * Revision 1.21  2004/12/06 17:11:12  jimg
 * Added store_ce() and fields to store the projection and selection parts of a CE.
 *
 * Revision 1.20  2004/11/30 22:11:35  jimg
 * I replaced the flatten_*() functions with a flatten() method in
 * NCAccess. The default version of this method is in NCAccess and works
 * for the atomic types; constructors must provide a specialization.
 * Then I removed the code that copied the variables from vectors to
 * lists. The translation code in NCConnect was modified to use the
 * new method.
 *
 * Revision 1.19  2004/11/05 17:13:57  jimg
 * Added code to copy the BaseType pointers from the vector container into
 * a list. This will enable more efficient translation software to be
 * written.
 *
 * Revision 1.18  2004/10/28 16:38:19  jimg
 * Added support for error handling to ClientParams. Added use of
 * ClientParams to NCConnect, although that's not complete yet. NCConnect
 * now has an instance of ClientParams. The instance is first built and
 * then passed into NCConnect's ctor which stores a const reference to the CP
 * object.
 *
 * Revision 1.17  2004/09/08 22:08:21  jimg
 * More Massive changes: Code moved from the files that clone the netCDF
 * function calls into NCConnect, NCAccess or nc_util.cc. Much of the
 * translation functions are now methods. The netCDF type classes now
 * inherit from NCAccess in addition to the DAP type classes.
 *
 * Revision 1.16  2004/08/03 23:23:13  jimg
 * Vast changes to get attribute support working for variables from
 * data sources with Structures. The Structure variables are 'translated'
 * to sets of variables with 'dots in their names.' These new changes
 * correctly find the attribtues for the old variables and add them to
 * the built in attribute table objects for the new variables. Global
 * attributes are now handled as well. This software relies heavily on the
 * heuristic code in DDS::transfer_attributes(). During the course of the
 * work, I moved all of the software that interacts with the OPeNDAP servers
 * to NCConnect or nc_util. The files Dvar.cc, Dattr.cc, ..., are now
 * just the clones of the netCDF api functions.
 *
 * Revision 1.15  2004/07/28 18:10:05  jimg
 * Fixed handling of global attributes. I used the new code in
 * DDS::transfer_attributes().
 *
 * Revision 1.14  2004/07/26 19:10:44  jimg
 * Moved netCDF CL <--> OPeNDAP server interface code to nc_util and
 * NCConnect.
 *
 * Revision 1.13  2004/03/09 22:56:32  jimg
 * Refactored so that Pix is no longer used. Some common code (in the
 * glue routines) was also factored out to functions. The netCDF 2
 * interface is now supplied by the lnetcdf/lv2i.c file (which is a mostly
 * unaltered copy of the original v2i.c source file). See lnetcdf/README.
 *
 * Revision 1.12  2004/03/08 19:08:33  jimg
 * This version of the code uses the Unidata netCDF 3.5.1 version of the
 * netCDF 2 API emulation. This functions call our netCDF 3 API functions
 * which may either interact with a DAP server r call the local netCDF 3
 * functions.
 *
 * Revision 1.11  2004/03/01 22:30:58  jimg
 * Update; more fixes for translation of Structure types. This code can
 * be built with ncdump and that client can read from a host of HDF4
 * datasets. The changes that enable this are filtering of Sequences.
 *
 * Revision 1.10  2004/02/25 00:47:52  jimg
 * This code will translate Structures, including ones that are nested.
 * Not tested much; needs work.
 *
 * Revision 1.9  2003/12/08 18:06:37  edavis
 * Merge release-3-4 into trunk
 *
 * Revision 1.8  2003/09/30 22:28:39  jimg
 * I used #if 0 ... #endif to remove some declarations for methods that are
 * never used (and which I removed from the compile in the implementation file).
 *
 * Revision 1.7  2003/09/29 12:11:13  reza
 * Merged v3.2 translation code.
 *
 * Revision 1.6  2003/09/25 23:09:36  jimg
 * Meerged from 3.4.1.
 *
 * Revision 1.5.2.1  2003/06/24 11:36:32  rmorris
 * Removed #pragma interface directives for the OS X.
 *
 * Revision 1.5  2003/05/02 16:30:29  jimg
 * Fixes for the builds
 *
 * Revision 1.4  2003/01/28 07:08:24  jimg
 * Merged with release-3-2-8.
 *
 * Revision 1.3.4.2  2002/12/27 00:35:24  jimg
 * Added set_ncid(). See NCConnect for an explaination.
 *
 * Revision 1.3.4.1  2002/06/21 00:31:40  jimg
 * I changed many files throughout the source so that the 'make World' build
 * works with the new versions of Connect and libdap++ that use libcurl.
 * Most of these changes are either to Makefiles, configure scripts or to
 * the headers included by various C++ files. In a few places the oddities
 * of libwww forced us to hack up code and I've undone those and some of the
 * clients had code that supported libwww's generous tracing capabilities
 * (that's one part of libwww I'll miss); I had to remove support for that.
 * Once this code compiles and more work is done on Connect, I'll return to
 * each of these changes and polish them.
 *
 * Revision 1.3  2000/10/06 01:22:02  jimg
 * Moved the CVS Log entries to the ends of files.
 * Modified the read() methods to match the new definition in the dap library.
 * Added exception handlers in various places to catch exceptions thrown
 * by the dap library.
 *
 * Revision 1.2  1999/11/05 05:15:05  jimg
 * Result of merge woth 3-1-0
 *
 * Revision 1.1.2.1  1999/10/29 05:05:21  jimg
 * Reza's fixes plus the configure & Makefile update
 *
 * Revision 1.1  1999/07/28 00:22:42  jimg
 * Added
 *
 * Revision 1.10  1999/05/07 23:45:31  jimg
 * String --> string fixes
 *
 * Revision 1.9  1996/09/19 16:58:17  jimg
 * Made the NCConnect dtor virtual.
 *
 * Revision 1.8  1996/09/17 17:06:23  jimg
 * Merge the release-2-0 tagged files (which were off on a branch) back into
 * the trunk revision.
 *
 * Revision 1.6.4.2  1996/07/10 21:44:01  jimg
 * Changes for version 2.06. These fixed lingering problems from the migration
 * from version 1.x to version 2.x.
 * Removed some (but not all) warning generated with gcc's -Wall option.
 *
 * Revision 1.6.4.1  1996/06/25 22:04:24  jimg
 * Version 2.0 from Reza.
 *
 * Revision 1.6  1994/11/23  20:59:54  reza
 * Added heuristic dimension matching for Grid and Array classes.
 * Added Dimension handling routines (ndims(), dim_size(), dim_name(),
 * parse_dim(),and etc.)
 *
 * Revision 1.5  1994/11/18  21:33:32  reza
 * Added _nvars and nvars() for number of variables in the file.
 * Added _das_loc[] and parse_das_loc() to make a look-up table for variables
 * in dds vs. das.
 *
 * Revision 1.4  1994/11/03  05:45:08  reza
 * Moved error, request_das, and request _dds to the surrogate library for more
 * flexibility and simpler error handling.
 *
 * Revision 1.3  1994/10/28  19:09:04  jimg
 * Added _error, error() and fixed ncid() so that it retruns a reference.
 *
 * Revision 1.2  1994/10/05  20:23:31  jimg
 * Fixed errors in *.h files comments - CVS bites again.
 * Changed request_{das,dds} so that they use the field `_api_name'
 * instead of requiring callers to pass the api name.
 *
 * Revision 1.1  1994/10/05  18:02:15  jimg
 * First version of the connection management classes.
 * This commit also includes early versions of the test code.
 */

#endif

