
// -*- mode: c++; c-basic-offset:4 -*-

// This file is part of libdap, A C++ implementation of the OPeNDAP Data
// Access Protocol.

// Copyright (c) 2002,2003,2005 OPeNDAP, Inc.
// Author: James Gallagher <jgallagher@opendap.org>
//         Reza Nekovei <reza@intcomm.net>
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
// You can contact OPeNDAP, Inc. at PO Box 112, Saunderstown, RI. 02874-0112.
//
// A data exchange interface for DODS-netCDF V3.x API library.
//
//  Reza 5/18/99
// 

// $Id: Dncx.cc 11906 2005-08-08 19:51:43Z root $ 

#include "Dncx.h"
#include <string.h>
#include <limits.h>
/* alias poorly named limits.h macros */
#define  SHORT_MAX  SHRT_MAX
#define  SHORT_MIN  SHRT_MIN
#define USHORT_MAX USHRT_MAX
#include <float.h>
#ifndef FLT_MAX /* This POSIX macro missing on some systems */
# ifndef NO_IEEE_FLOAT
# define FLT_MAX 3.40282347e+38f
# else
# error "You will need to define FLT_MAX"
# endif
#endif

int
putn_into_uchar(void **xpp, size_t nelems, unsigned char *tp, nc_type type)
{
  unsigned char *bp;
  short *sp;
  int *ip; 
  float *fp;
  double *dp;
  int stat = NC_NOERR;

  switch (type) {
    case NC_CHAR:
      return NC_ECHAR;
    case NC_BYTE:
#if 0
      (void) memcpy(tp, *xpp, nelems);
#else
      bp = (unsigned char *) *xpp; 
      for( ; nelems != 0; nelems--, bp++, tp++)
	*tp = (unsigned char) *bp;
#endif
      break;
    case NC_SHORT:
      sp = (short *) *xpp; 
      for( ; nelems != 0; nelems--, sp++, tp++){
        if(*sp > UCHAR_MAX || *sp < 0)
	  stat = NC_ERANGE;
	*tp = (unsigned char) *sp;
      }     
      break;
    case NC_INT:
      ip = (int *) *xpp; 
      for( ; nelems != 0; nelems--, ip++, tp++){
       if(*ip > UCHAR_MAX || *ip < 0)
	  stat = NC_ERANGE;
	*tp = (unsigned char) *ip;
      }  
      break;
    case NC_FLOAT: 
      fp = (float *) *xpp; 
      for( ; nelems != 0; nelems--, fp++, tp++){
	if(*fp > UCHAR_MAX || *fp < 0)
	  stat = NC_ERANGE;      
	*tp = (unsigned char) *fp; 
      }  
      break;
    case NC_DOUBLE:
      dp = (double *) *xpp; 
      for( ; nelems != 0; nelems--, dp++, tp++){
	if(*dp > UCHAR_MAX || *dp < 0)
	  stat = NC_ERANGE;      
	*tp = (unsigned char) *dp;
      }  
      break;
    default:
      return -1; // enumeration value `NC_NAT' not handled in switch. FIX ME
  }

  return stat;
}

int
putn_into_schar(void **xpp, size_t nelems, signed char *tp, nc_type type)
{
  signed char *bp;
  short *sp;
  int *ip; 
  float *fp;
  double *dp;
  int stat = NC_NOERR;

  switch (type) {
    case NC_CHAR:
      return NC_ECHAR;
    case NC_BYTE:
#if 0
      (void) memcpy(tp, *xpp, nelems);
#else
      bp = (signed char *) *xpp; 
      for( ; nelems != 0; nelems--, bp++, tp++)
	*tp = (signed char) *bp;
#endif
      break;
    case NC_SHORT:
      sp = (short *) *xpp; 
      for( ; nelems != 0; nelems--, sp++, tp++){
        if(*sp > SCHAR_MAX || *sp < SCHAR_MIN)
	  stat = NC_ERANGE;
	*tp = (signed char) *sp;
      }     
      break;
    case NC_INT:
      ip = (int *) *xpp; 
      for( ; nelems != 0; nelems--, ip++, tp++){
       if(*ip > SCHAR_MAX || *ip < SCHAR_MIN)
	  stat = NC_ERANGE;
	*tp = (signed char) *ip;
      }  
      break;
    case NC_FLOAT: 
      fp = (float *) *xpp; 
      for( ; nelems != 0; nelems--, fp++, tp++){
	if(*fp > SCHAR_MAX || *fp < SCHAR_MIN)
	  stat = NC_ERANGE;      
	*tp = (signed char) *fp; 
      }  
      break;
    case NC_DOUBLE:
      dp = (double *) *xpp; 
      for( ; nelems != 0; nelems--, dp++, tp++){
	if(*dp > SCHAR_MAX || *dp < SCHAR_MIN)
	  stat = NC_ERANGE;      
	*tp = (signed char) *dp;
      }  
      break;
    default:
      return -1; // enumeration value `NC_NAT' not handled in switch. FIX ME
  }

  return stat;
}

int
putn_into_short(void **xpp, size_t nelems, short *tp, nc_type type)
{
  signed char *bp;
  short *sp;
  int *ip; 
  float *fp;
  double *dp;
  int stat = NC_NOERR;

  switch (type) {
    case NC_CHAR:
      return NC_ECHAR;
    case NC_BYTE:
      bp = (signed char *) *xpp; 
      for( ; nelems != 0; nelems--, bp++, tp++)
	*tp = (short) *bp;
      break;
    case NC_SHORT:
#if 0
     (void) memcpy(tp, *xpp, nelems * sizeof(short));
#else
      sp = (short *) *xpp; 
      for( ; nelems != 0; nelems--, sp++, tp++)
	*tp = (short) *sp;
#endif
      break;
    case NC_INT:
      ip = (int *) *xpp; 
      for( ; nelems != 0; nelems--, ip++, tp++){
	if(*ip > SHORT_MAX || *ip < SHORT_MIN)
	  stat = NC_ERANGE;      
	*tp = (short) *ip;
      }  
      break;
    case NC_FLOAT: 
      fp = (float *) *xpp; 
      for( ; nelems != 0; nelems--, fp++, tp++){
	if(*fp > SHORT_MAX || *fp < SHORT_MIN)
	  stat = NC_ERANGE;      
	*tp = (short) *fp; 
      }  
      break;
    case NC_DOUBLE:
      dp = (double *) *xpp; 
      for( ; nelems != 0; nelems--, dp++, tp++){
	if(*dp > SHORT_MAX || *dp < SHORT_MIN)
	  stat = NC_ERANGE;      
	*tp = (short) *dp;
      }  
      break;
    default:
      return -1; // enumeration value `NC_NAT' not handled in switch. FIX ME
  }

  return stat;
}

int
putn_into_int(void **xpp, size_t nelems, int *tp, nc_type type)
{
  signed char *bp;
  short *sp;
  int *ip; 
  float *fp;
  double *dp;
  int stat = NC_NOERR;

  switch (type) {
    case NC_CHAR:
      return NC_ECHAR;
    case NC_BYTE:
      bp = (signed char *) *xpp; 
      for( ; nelems != 0; nelems--, bp++, tp++)
	*tp = (int) *bp;
      break;
    case NC_SHORT:
      sp = (short *) *xpp; 
      for( ; nelems != 0; nelems--, sp++, tp++)
	*tp = (int) *sp;
      break;
    case NC_INT:
#if 0
      (void) memcpy(tp, *xpp, nelems * sizeof(int)); 
#else
      ip = (int *) *xpp; 
      for( ; nelems != 0; nelems--, ip++, tp++)
	*tp = (int) *ip; 
#endif
      break;
    case NC_FLOAT:
      fp = (float *) *xpp; 
      for( ; nelems != 0; nelems--, fp++, tp++){
       if(*fp > (double)INT_MAX || *fp < (double)INT_MIN)
	 stat = NC_ERANGE;
       *tp = (int) *fp; 
      }  
      break;
    case NC_DOUBLE:
      dp = (double *) *xpp; 
      for( ; nelems != 0; nelems--, dp++, tp++){
	if(*dp > (double)INT_MAX || *dp < (double)INT_MIN)
	  stat = NC_ERANGE;
	*tp = (int) *dp;
      }  
      break;
    default:
      return -1; // enumeration value `NC_NAT' not handled in switch. FIX ME
  }

  return stat;
}

int
putn_into_long(void **xpp, size_t nelems, long *tp, nc_type type)
{ 
  signed char *bp;
  short *sp;
  int *ip; 
  float *fp;
  double *dp;  
  int stat = NC_NOERR;


  switch (type) {
    case NC_CHAR:
      return NC_ECHAR;
    case NC_BYTE:
      bp = (signed char *) *xpp; 
      for( ; nelems != 0; nelems--, bp++, tp++)
	*tp = (long) *bp;
      break;
    case NC_SHORT:
      sp = (short *) *xpp; 
      for( ; nelems != 0; nelems--, sp++, tp++)
	*tp = (long) *sp;
      break;
    case NC_INT:
      ip = (int *) *xpp; 
      for( ; nelems != 0; nelems--, ip++, tp++)
	*tp = (long) *ip;
      break;
    case NC_FLOAT: 
      fp = (float *) *xpp; 
      for( ; nelems != 0; nelems--, fp++, tp++){
	if(*fp > LONG_MAX || *fp < LONG_MIN)
	  stat = NC_ERANGE;
	*tp = (long) *fp; 
      }  
      break;
    case NC_DOUBLE:
      dp = (double *) *xpp; 
      for( ; nelems != 0; nelems--, dp++, tp++){
	if(*dp > LONG_MAX || *dp < LONG_MIN)
	  stat = NC_ERANGE;
	*tp = (long) *dp;
      }  
      break;
    default:
      return -1; // enumeration value `NC_NAT' not handled in switch. FIX ME
  }
  return stat;
}


int
putn_into_float(void **xpp, size_t nelems, float *tp, nc_type type)
{
  signed char *bp;
  short *sp;
  int *ip; 
  float *fp;
  double *dp;
  int stat = NC_NOERR;

  switch (type) {
    case NC_CHAR: 
      return NC_ECHAR;
    case NC_BYTE:
      bp = (signed char *) *xpp; 
      for( ; nelems != 0; nelems--, bp++, tp++)
	*tp = (float) *bp;
      break;
    case NC_SHORT:
      sp = (short *) *xpp; 
      for( ; nelems != 0; nelems--, sp++, tp++)
	*tp = (float) *sp;
      break;
    case NC_INT:
      ip = (int *) *xpp; 
      for( ; nelems != 0; nelems--, ip++, tp++){
        // Not sure why this is commented out... jhrg 3/3/05
	//	if(*ip > FLT_MAX || *ip < (-FLT_MAX))
	//	  stat = NC_ERANGE;      
	*tp = (float) *ip;
      }  
      break;
    case NC_FLOAT: 
#if 0
      (void) memcpy(tp, *xpp, nelems * sizeof(float));
#else
      fp = (float *) *xpp; 
      for( ; nelems != 0; nelems--, fp++, tp++)
	*tp = *fp; 
#endif
      break;
    case NC_DOUBLE:
      dp = (double *) *xpp; 
      for( ; nelems != 0; nelems--, dp++, tp++){
	if(*dp > FLT_MAX || *dp < (-FLT_MAX))
	  stat= NC_ERANGE;      
	*tp = (float) *dp;
      }  
      break;
    default:
      return -1; // enumeration value `NC_NAT' not handled in switch. FIX ME
  }

  return stat;
}

int
putn_into_double(void **xpp, size_t nelems, double *tp, nc_type type)
{
  signed char *bp;
  short *sp;
  int *ip; 
  float *fp;
  double *dp;

  switch (type) {
    case NC_CHAR:
      return NC_ECHAR;
    case NC_BYTE:
      bp = (signed char *) *xpp; 
      for( ; nelems != 0; nelems--, bp++, tp++)
	*tp = (double) *bp;
      break;
    case NC_SHORT:
      sp = (short *) *xpp; 
      for( ; nelems != 0; nelems--, sp++, tp++)
	*tp = (double) *sp;
      break;
    case NC_INT:
      ip = (int *) *xpp; 
      for( ; nelems != 0; nelems--, ip++, tp++)
	*tp = (double) *ip;
      break;
    case NC_FLOAT: 
      fp = (float *) *xpp; 
      for( ; nelems != 0; nelems--, fp++, tp++)
	*tp = (double) *fp; 
      break;
    case NC_DOUBLE:
#if 0
      (void) memcpy(tp, *xpp, nelems * sizeof(double));
#else
      dp = (double *) *xpp; 
      for( ; nelems != 0; nelems--, dp++, tp++)
	*tp =  *dp;
#endif
      break;
    default:
      return -1; // enumeration value `NC_NAT' not handled in switch. FIX ME
  }

  return NC_NOERR;
  
}

int
putn_into_text(void **xpp, size_t nelems, char *tp)
{
      (void) memcpy(tp, *xpp, nelems);
      //      *xpp = (void *)((char *)(*xpp) + nelems);
      return NC_NOERR;
}

// $Log: Dncx.cc,v $
// Revision 1.7  2005/03/04 18:10:49  jimg
// At this point valgrind runs the Unidata tests for both local and remote access
// and shows no errors or leaks. There are 8 bytes still reachable from an
// exception, but that's it.
//
// Revision 1.6  2004/02/25 00:47:52  jimg
// This code will translate Structures, including ones that are nested.
// Not tested much; needs work.
//
// Revision 1.5  2003/12/08 18:06:37  edavis
// Merge release-3-4 into trunk
//
// Revision 1.4  2000/10/06 01:22:02  jimg
// Moved the CVS Log entries to the ends of files.
// Modified the read() methods to match the new definition in the dap library.
// Added exception handlers in various places to catch exceptions thrown
// by the dap library.
//
