
// -*- mode: c++; c-basic-offset:4 -*-

// This file is part of libdap, A C++ implementation of the OPeNDAP Data
// Access Protocol.

// Copyright (c) 2002,2003,2005 OPeNDAP, Inc.
// Author: James Gallagher <jgallagher@opendap.org>
//         Reza Nekovei <reza@intcomm.net>
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
// You can contact OPeNDAP, Inc. at PO Box 112, Saunderstown, RI. 02874-0112.

// (c) COPYRIGHT URI/MIT 1994-1996
// Please read the full copyright statement in the file COPYRIGHT.
//
// Authors:
//      reza            Reza Nekovei (reza@intcomm.net)

// DODS-netCDF surrogate library for data retrieval. 
// 
// ReZa 10/14/94
//
// A rewrite to support netCDF 3.x interface. This file 
// replaces old (version 2.x) Dputget.cc and Dputgetg.cc 
// files. ReZa 5/7/99 

#include "config_nc.h"

static char rcsid[] not_used ={"$Id: Dputget.cc 11906 2005-08-08 19:51:43Z root $"};

#include <sstream>

#include "Str.h"
#include "Error.h"

//#define DODS_DEBUG
#include "debug.h"

#include "Dnetcdf.h"
// #include "Dncx.h"
// Undefine SIZEOF_SIZE_T to force macros in Dv2i.h to expand a certain way.
// This was actually always happening, but switching to a newer autoconf
// makes the problem explict. The real fix is to upgrade to netcdf 3.6.1.
// 5/11/05 jhrg 
#undef SIZEOF_SIZE_T
#include "Dv2i.h"
#include "nc_util.h"


// Used for the netCDF 2 interface emulation. See lv2i.c. 
extern "C" int
nc_get_rec(int cdfid, size_t recnum, void **datap)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) {
	rcode = lnc_get_rec((*conns)[cdfid]->get_ncid(), recnum, datap);
	return rcode;
    }
#endif
    cdf_routine_name = "nc_get_rec" ;

    NCadvise(NC_EPERM, "Record access is not supported yet, DODS") ;
    return NC_EPERM;
}

// Used for the netCDF 2 interface emulation. See lv2i.c. 
extern "C" int
nc_inq_rec(int cdfid, size_t *nrecvars, int *recvarids, size_t *recsizes)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) {
	rcode=lnc_inq_rec((*conns)[cdfid]->get_ncid(),nrecvars,recvarids,recsizes);
	return rcode;
    }
#endif
    cdf_routine_name = "nc_inq_rec" ;
    NCadvise(NC_EPERM, "Record access is not supported yet, DODS") ;
    return NC_EPERM;
}

// Used for the netCDF 2 interface emulation. See lv2i.c. 
extern "C" int
nc_put_rec(int cdfid, size_t recnum, void * const *datap)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) {
	rcode = lnc_put_rec((*conns)[cdfid]->get_ncid(), recnum, datap);
	return rcode;
    }
#endif

    return NC_EPERM;
}

int
nc_put_var1_text(int cdfid, int varid, const size_t *coord,
		 const char *value)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) 
	return lnc_put_var1_text((*conns)[cdfid]->get_ncid(), varid, coord, value);
#endif

    return NC_EPERM;
}
int
nc_put_var1_uchar(int cdfid, int varid, const size_t *coord,
		  const unsigned char *value)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) 
	return lnc_put_var1_uchar((*conns)[cdfid]->get_ncid(), varid, coord, value);
#endif

    return NC_EPERM;
}
int
nc_put_var1_schar(int cdfid, int varid, const size_t *coord,
		  const signed char *value)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) 
	return lnc_put_var1_schar((*conns)[cdfid]->get_ncid(), varid, coord, value);
#endif

    return NC_EPERM;
}

int
nc_put_var1_short(int cdfid, int varid, const size_t *coord,
		  const short *value)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) 
	return lnc_put_var1_short((*conns)[cdfid]->get_ncid(), varid, coord, value);
#endif

    return NC_EPERM;
}
int
nc_put_var1_int(int cdfid, int varid, const size_t *coord,
		const int *value)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) 
	return lnc_put_var1_int((*conns)[cdfid]->get_ncid(), varid, coord, value);
#endif

    return NC_EPERM;
}
int
nc_put_var1_long(int cdfid, int varid, const size_t *coord,
		 const long *value)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) 
	return lnc_put_var1_long((*conns)[cdfid]->get_ncid(), varid, coord, value);
#endif

    return NC_EPERM;
}
int
nc_put_var1_float(int cdfid, int varid, const size_t *coord,
		  const float *value)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) 
	return lnc_put_var1_float((*conns)[cdfid]->get_ncid(), varid, coord, value);
#endif

    return NC_EPERM;
}
int
nc_put_var1_double(int cdfid, int varid, const size_t *coord,
		   const double *value)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) 
	return lnc_put_var1_double((*conns)[cdfid]->get_ncid(), varid, coord, value);
#endif

    return NC_EPERM;
}

// Used for the netCDF 2 interface emulation. See lv2i.c. This is not
// advertized in netcdf.h so we declare it extern "C" here so lv2i.c will see
// it.
extern "C" int
nc_put_var1(int cdfid, int varid, const size_t *coord, const void *value)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) 
	return lnc_put_var1((*conns)[cdfid]->get_ncid(), varid, coord, value);
#endif

    return NC_EPERM;
}

int
nc_put_vara_text(int cdfid, int varid,
		 const size_t *start, const size_t *edges, const char *value)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) 
	return lnc_put_vara_text((*conns)[cdfid]->get_ncid(), varid, start,edges, value);
#endif

    return NC_EPERM;
}
int
nc_put_vara_uchar(int cdfid, int varid,
		  const size_t *start, const size_t *edges, const unsigned char *value)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) 
	return lnc_put_vara_uchar((*conns)[cdfid]->get_ncid(), varid, start,edges, value);
#endif

    return NC_EPERM;
}
int
nc_put_vara_schar(int cdfid, int varid,
		  const size_t *start, const size_t *edges, const signed char *value)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) 
	return lnc_put_vara_schar((*conns)[cdfid]->get_ncid(), varid, start,edges, value);
#endif

    return NC_EPERM;
}
int
nc_put_vara_short(int cdfid, int varid,
		  const size_t *start, const size_t *edges, const short *value)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) 
	return lnc_put_vara_short((*conns)[cdfid]->get_ncid(), varid, start,edges, value);
#endif

    return NC_EPERM;
}
int
nc_put_vara_int(int cdfid, int varid,
		const size_t *start, const size_t *edges, const int *value)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) 
	return lnc_put_vara_int((*conns)[cdfid]->get_ncid(), varid, start,edges, value);
#endif

    return NC_EPERM;
}
int
nc_put_vara_long(int cdfid, int varid,
		 const size_t *start, const size_t *edges, const long *value)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) 
	return lnc_put_vara_long((*conns)[cdfid]->get_ncid(), varid, start,edges, value);
#endif

    return NC_EPERM;
}
 
int
nc_put_vara_float(int cdfid, int varid,
		  const size_t *start, const size_t *edges, const float *value)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) 
	return lnc_put_vara_float((*conns)[cdfid]->get_ncid(), varid, start,edges, value);
#endif

    return NC_EPERM;
}
int
nc_put_vara_double(int cdfid, int varid,
		   const size_t *start, const size_t *edges, const double *value)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) 
	return lnc_put_vara_double((*conns)[cdfid]->get_ncid(), varid, start,edges, value);
#endif

    return NC_EPERM;
}
int
nc_put_vara(int cdfid, int varid,
	    const size_t *start, const size_t *edges, const void *value)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) 
	return lnc_put_vara((*conns)[cdfid]->get_ncid(), varid, start,edges, value);
#endif

    return NC_EPERM;
}
int
nc_put_var_text(int cdfid, int varid, const char *value)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) 
	return lnc_put_var_text((*conns)[cdfid]->get_ncid(), varid, value);
#endif

    return NC_EPERM;
}
int
nc_put_var_uchar(int cdfid, int varid, const unsigned char *value)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) 
	return lnc_put_var_uchar((*conns)[cdfid]->get_ncid(), varid, value);
#endif

    return NC_EPERM;
}
int
nc_put_var_schar(int cdfid, int varid, const signed char *value)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) 
	return lnc_put_var_schar((*conns)[cdfid]->get_ncid(), varid, value);
#endif

    return NC_EPERM;
}
int
nc_put_var_short(int cdfid, int varid, const short *value)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) 
	return lnc_put_var_short((*conns)[cdfid]->get_ncid(), varid, value);
#endif

    return NC_EPERM;
}
int
nc_put_var_int(int cdfid, int varid, const int *value)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) 
	return lnc_put_var_int((*conns)[cdfid]->get_ncid(), varid, value);
#endif

    return NC_EPERM;
}
int
nc_put_var_long(int cdfid, int varid, const long *value)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) 
	return lnc_put_var_long((*conns)[cdfid]->get_ncid(), varid, value);
#endif

    return NC_EPERM;
}
int
nc_put_var_float(int cdfid, int varid, const float *value)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) 
	return lnc_put_var_float((*conns)[cdfid]->get_ncid(), varid, value);
#endif

    return NC_EPERM;
}
int
nc_put_var_double(int cdfid, int varid, const double *value)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) 
	return lnc_put_var_double((*conns)[cdfid]->get_ncid(), varid, value);
#endif

    return NC_EPERM;
}

int
nc_put_varm_text(int cdfid, int varid,
		 const size_t *start, const size_t *edges,
		 const ptrdiff_t *stride, const ptrdiff_t *imap,
		 const char *value)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) 
	return lnc_put_varm_text((*conns)[cdfid]->get_ncid(), varid,
				 start, edges, stride, imap, value);
#endif

    return NC_EPERM;
}
int
nc_put_varm_uchar(int cdfid, int varid,
		  const size_t *start, const size_t *edges,
		  const ptrdiff_t *stride, const ptrdiff_t *imap,
		  const unsigned char *value)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) 
	return lnc_put_varm_uchar((*conns)[cdfid]->get_ncid(), varid,
				  start, edges, stride, imap, value);
#endif

    return NC_EPERM;
}
int
nc_put_varm_schar(int cdfid, int varid,
		  const size_t *start, const size_t *edges,
		  const ptrdiff_t *stride, const ptrdiff_t *imap,
		  const signed char *value)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) 
	return lnc_put_varm_schar((*conns)[cdfid]->get_ncid(), varid,
				  start, edges, stride, imap, value);
#endif

    return NC_EPERM;
} 
int
nc_put_varm_short(int cdfid, int varid,
		  const size_t *start, const size_t *edges,
		  const ptrdiff_t *stride, const ptrdiff_t *imap,
		  const short *value)
{ 
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) 
	return lnc_put_varm_short((*conns)[cdfid]->get_ncid(), varid,
				  start, edges, stride, imap, value);
#endif

    return NC_EPERM;
}
int
nc_put_varm_int(int cdfid, int varid,
		const size_t *start, const size_t *edges,
		const ptrdiff_t *stride, const ptrdiff_t *imap,
		const int *value)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) 
	return lnc_put_varm_int((*conns)[cdfid]->get_ncid(), varid,
				start, edges, stride, imap, value);
#endif

    return NC_EPERM;
}
int
nc_put_varm_long(int cdfid, int varid,
		 const size_t *start, const size_t *edges,
		 const ptrdiff_t *stride, const ptrdiff_t *imap,
		 const long *value)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) 
	return lnc_put_varm_long((*conns)[cdfid]->get_ncid(), varid,
				 start, edges, stride, imap, value);
#endif

    return NC_EPERM;
}
int
nc_put_varm_float(int cdfid, int varid,
		  const size_t *start, const size_t *edges,
		  const ptrdiff_t *stride, const ptrdiff_t *imap,
		  const float *value)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) 
	return lnc_put_varm_float((*conns)[cdfid]->get_ncid(), varid,
				  start, edges, stride, imap, value);
#endif

    return NC_EPERM;
}

int
nc_put_varm_double(int cdfid, int varid,
		   const size_t *start, const size_t *edges,
		   const ptrdiff_t *stride, const ptrdiff_t *imap,
		   const double *value)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) 
	return lnc_put_varm_double((*conns)[cdfid]->get_ncid(), varid,
				   start, edges, stride, imap, value);
#endif

    return NC_EPERM;
}

// Used for the netCDF 2 interface emulation. See lv2i.c. 
extern "C" int
nc_put_varm (
	     int cdfid, int varid,
	     const size_t * start,
	     const size_t * edges,
	     const ptrdiff_t * stride,
	     const ptrdiff_t * imap,
	     const void *value)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) 
	return lnc_put_varm((*conns)[cdfid]->get_ncid(), varid,
			    start, edges, stride, imap, value);
#endif
    return NC_EPERM;
}
 
int
nc_put_vars_text (
		  int ncid,
		  int varid,
		  const size_t * start,
		  const size_t * edges,
		  const ptrdiff_t * stride,
		  const char *value)
{
    return nc_put_varm_text (ncid, varid, start, edges,
			     stride, 0, value);
}

int
nc_put_vars_uchar (
		   int ncid,
		   int varid,
		   const size_t * start,
		   const size_t * edges,
		   const ptrdiff_t * stride,
		   const unsigned char *value)
{
    return nc_put_varm_uchar (ncid, varid, start, edges,
			      stride, 0, value);
}

int
nc_put_vars_schar (
		   int ncid,
		   int varid,
		   const size_t * start,
		   const size_t * edges,
		   const ptrdiff_t * stride,
		   const signed char *value)
{
    return nc_put_varm_schar (ncid, varid, start, edges,
			      stride, 0, value);
}

int
nc_put_vars_short (
		   int ncid,
		   int varid,
		   const size_t * start,
		   const size_t * edges,
		   const ptrdiff_t * stride,
		   const short *value)
{
    return nc_put_varm_short (ncid, varid, start, edges,
			      stride, 0, value);
}
int
nc_put_vars_int (
		 int ncid,
		 int varid,
		 const size_t * start,
		 const size_t * edges,
		 const ptrdiff_t * stride,
		 const int *value)
{
    return nc_put_varm_int (ncid, varid, start, edges,
			    stride, 0, value);
}

int
nc_put_vars_long (
		  int ncid,
		  int varid,
		  const size_t * start,
		  const size_t * edges,
		  const ptrdiff_t * stride,
		  const long *value)
{
    return nc_put_varm_long (ncid, varid, start, edges,
			     stride, 0, value);
}

int
nc_put_vars_float (
		   int ncid,
		   int varid,
		   const size_t * start,
		   const size_t * edges,
		   const ptrdiff_t * stride,
		   const float *value)
{
    return nc_put_varm_float (ncid, varid, start, edges,
			      stride, 0, value);
}

int
nc_put_vars_double (
		    int ncid,
		    int varid,
		    const size_t * start,
		    const size_t * edges,
		    const ptrdiff_t * stride,
		    const double *value)
{
    return nc_put_varm_double (ncid, varid, start, edges,
			       stride, 0, value);
}

// Used for the netCDF 2 interface emulation. See lv2i.c. 
extern "C" int
nc_put_vars(
	     int ncid,
	     int varid,
	     const size_t * start,
	     const size_t * edges,
	     const ptrdiff_t * stride,
	     const void *value)
{
    return nc_put_varm (ncid, varid, start, edges,
			stride, 0, value);
}


int
nc_get_var1_text(int cdfid, int varid, const size_t *coord, char *value)
{  
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) {
	rcode = lnc_get_var1_text((*conns)[cdfid]->get_ncid(), varid, coord, value);
	return rcode;
    }
#endif

    NDIMS_DECL
	A_DECL(count, size_t, ndims, index);

    for (unsigned int i = 0; i < ndims ; i++)
	count[i] = 1;     // set count to one
  
    rcode= DODvario(cdfid, varid, coord, count, 0, value, Ttext);

    A_FREE(count);
    return rcode;
}
 
int
nc_get_var1_uchar(int cdfid, int varid, const size_t *coord, unsigned char *value)
{  
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) {
	rcode = lnc_get_var1_uchar((*conns)[cdfid]->get_ncid(), varid, coord, value);
	return rcode;
    }
#endif

    NDIMS_DECL
	A_DECL(count, size_t, ndims, index);

    for (unsigned int i =0; i < ndims ; i++)
	count[i] = 1;     // set count to one
  
    rcode= DODvario(cdfid, varid, coord, count, 0, value, Tuchar);

    A_FREE(count);
    return rcode;
}

int
nc_get_var1_schar(int cdfid, int varid, const size_t *coord, signed char *value)
{  
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) {
	rcode = lnc_get_var1_schar((*conns)[cdfid]->get_ncid(), varid, coord, value);
	return rcode;
    }
#endif

    NDIMS_DECL
	A_DECL(count, size_t, ndims, index);

    for (unsigned int i =0; i < ndims ; i++)
	count[i] = 1;     // set count to one
  
    rcode= DODvario(cdfid, varid, coord, count, 0, value, Tschar);

    A_FREE(count);
    return rcode;
}
 
int
nc_get_var1_short(int cdfid, int varid, const size_t *coord, short *value)
{  
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) {
	rcode = lnc_get_var1_short((*conns)[cdfid]->get_ncid(), varid, coord, value);
	return rcode;
    }
#endif

    NDIMS_DECL
	A_DECL(count, size_t, ndims, index);

    for (unsigned int i =0; i < ndims ; i++)
	count[i] = 1;     // set count to one
  
    rcode= DODvario(cdfid, varid, coord, count, 0, value, Tshort);

    A_FREE(count);
    return rcode;
}
 
int
nc_get_var1_int(int cdfid, int varid, const size_t *coord, int *value)
{  
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) {
	rcode = lnc_get_var1_int((*conns)[cdfid]->get_ncid(), varid, coord, value);
	return rcode;
    }
#endif

    NDIMS_DECL
	A_DECL(count, size_t, ndims, index);

    for (unsigned int i =0; i < ndims ; i++)
	count[i] = 1;     // set count to one
  
    rcode= DODvario(cdfid, varid, coord, count, 0, value, Tint);

    A_FREE(count);
    return rcode;
}
 
int
nc_get_var1_long(int cdfid, int varid, const size_t *coord, long *value)
{  
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) {
	rcode = lnc_get_var1_long((*conns)[cdfid]->get_ncid(), varid, coord, value);
	return rcode;
    }
#endif

    NDIMS_DECL
	A_DECL(count, size_t, ndims, index);

    for (unsigned int i =0; i < ndims ; i++)
	count[i] = 1;     // set count to one
  
    rcode= DODvario(cdfid, varid, coord, count, 0, value, Tlong);

    A_FREE(count);
    return rcode;
}
 
int
nc_get_var1_float(int cdfid, int varid, const size_t *coord, float *value)
{  
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) {
	rcode = lnc_get_var1_float((*conns)[cdfid]->get_ncid(), varid, coord, value);
	return rcode;
    }
#endif

    NDIMS_DECL
	A_DECL(count, size_t, ndims, index);

    for (unsigned int i =0; i < ndims ; i++)
	count[i] = 1;     // set count to one
  
    rcode= DODvario(cdfid, varid, coord, count, 0, value, Tfloat);

    A_FREE(count);
    return rcode;
}

int
nc_get_var1_double(int cdfid, int varid, const size_t *coord, double *value)
{  
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) {
	rcode = lnc_get_var1_double((*conns)[cdfid]->get_ncid(), varid, coord, value);
	return rcode;
    }
#endif

    NDIMS_DECL
	A_DECL(count, size_t, ndims, index);

    for (unsigned int i =0; i < ndims ; i++)
	count[i] = 1;     // set count to one
  
    rcode= DODvario(cdfid, varid, coord, count, 0, value, Tdouble);

    A_FREE(count);
    return rcode;
}

// Used for the netCDF 2 interface emulation. See lv2i.c.
extern "C" int
nc_get_var1(int cdfid, int varid, const size_t *coord, void *value)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) {
	rcode = lnc_get_var1((*conns)[cdfid]->get_ncid(), varid, coord, value);
	return rcode;
    }
#endif

    NDIMS_DECL
	A_DECL(count, size_t, ndims, index);

    for (unsigned int i =0; i < ndims ; i++)
	count[i] = 1;     // set count to one
  
    rcode= DODvario(cdfid, varid, coord, count, 0, value, Tvoid);

    A_FREE(count);
    return rcode;
}
 

int
nc_get_vara_text(int cdfid, int varid,
		 const size_t *start, const size_t *edges, char *value)
{    
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) 
	return lnc_get_vara_text((*conns)[cdfid]->get_ncid(), varid,
				 start, edges, value);
#endif

    return DODvario(cdfid, varid, start, edges, 0, value, Ttext);
}
int
nc_get_vara_uchar(int cdfid, int varid,
		  const size_t *start, const size_t *edges, unsigned char *value)
{  
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) 
	return lnc_get_vara_uchar((*conns)[cdfid]->get_ncid(), varid,
				  start, edges, value);
#endif

    return DODvario(cdfid, varid, start, edges, 0, value, Tuchar);
}
int
nc_get_vara_schar(int cdfid, int varid,
		  const size_t *start, const size_t *edges, signed char *value)
{  
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) 
	return lnc_get_vara_schar((*conns)[cdfid]->get_ncid(), varid,
				  start, edges, value);
#endif

    return DODvario(cdfid, varid, start, edges, 0, value, Tschar);
}
int
nc_get_vara_short(int cdfid, int varid,
		  const size_t *start, const size_t *edges, short *value)
{  
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) 
	return lnc_get_vara_short((*conns)[cdfid]->get_ncid(), varid,
				  start, edges, value);
#endif

    return DODvario(cdfid, varid, start, edges, 0, value, Tshort);
}
int
nc_get_vara_int(int cdfid, int varid,
		const size_t *start, const size_t *edges, int *value)
{  
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) 
	return lnc_get_vara_int((*conns)[cdfid]->get_ncid(), varid,
				start, edges, value);
#endif

    return DODvario(cdfid, varid, start, edges, 0, value, Tint);
}
int
nc_get_vara_long(int cdfid, int varid,
		 const size_t *start, const size_t *edges, long *value)
{  
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) 
	return lnc_get_vara_long((*conns)[cdfid]->get_ncid(), varid,
				 start, edges, value);
#endif

    return DODvario(cdfid, varid, start, edges, 0, value, Tlong);
}
int
nc_get_vara_float(int cdfid, int varid,
		  const size_t *start, const size_t *edges, float *value)
{  
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) 
	return lnc_get_vara_float((*conns)[cdfid]->get_ncid(), varid,
				  start, edges, value);
#endif

    return DODvario(cdfid, varid, start, edges, 0, value, Tfloat);
}
 
int
nc_get_vara_double(int cdfid, int varid,
		   const size_t *start, const size_t *edges, double *value)
{  
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) 
	return lnc_get_vara_double((*conns)[cdfid]->get_ncid(), varid,
				   start, edges, value);
#endif

    return DODvario(cdfid, varid, start, edges, 0, value, Tdouble);
}

int
nc_get_vara(int cdfid, int varid,
	    const size_t *start, const size_t *edges, void *value)
{  
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    NCConnect *c = (*conns)[cdfid];
    if(c->is_local()) 
	return lnc_get_vara(c->get_ncid(), varid, start, edges, value);
#endif

    return DODvario(cdfid, varid, start, edges, 0, value, Tvoid);
}

int
nc_get_var_text(int cdfid, int varid, char *value)
{  
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) 
	return lnc_get_var_text((*conns)[cdfid]->get_ncid(), varid, value);
#endif

    return DODvario(cdfid, varid, 0, 0, 0, value, Ttext);
}

int
nc_get_var_uchar(int cdfid, int varid, unsigned char *value)
{  
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) 
	return lnc_get_var_uchar((*conns)[cdfid]->get_ncid(), varid, value);
#endif

    return DODvario(cdfid, varid, 0, 0, 0, value, Tuchar);
}
int
nc_get_var_schar(int cdfid, int varid, signed char *value)
{  
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) 
	return lnc_get_var_schar((*conns)[cdfid]->get_ncid(), varid, value);
#endif

    return DODvario(cdfid, varid, 0, 0, 0, value, Tschar);
}
int
nc_get_var_short(int cdfid, int varid, short *value)
{  
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) 
	return lnc_get_var_short((*conns)[cdfid]->get_ncid(), varid, value);
#endif

    return DODvario(cdfid, varid, 0, 0, 0, value, Tshort);
}
int
nc_get_var_int(int cdfid, int varid, int *value)
{  
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) 
	return lnc_get_var_int((*conns)[cdfid]->get_ncid(), varid, value);
#endif

    return DODvario(cdfid, varid, 0, 0, 0, value, Tint);
}
int
nc_get_var_long(int cdfid, int varid, long *value)
{  
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) 
	return lnc_get_var_long((*conns)[cdfid]->get_ncid(), varid, value);
#endif

    return DODvario(cdfid, varid, 0, 0, 0, value, Tlong);
}
int
nc_get_var_float(int cdfid, int varid, float *value)
{  
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) 
	return lnc_get_var_float((*conns)[cdfid]->get_ncid(), varid, value);
#endif

    return DODvario(cdfid, varid, 0, 0, 0, value, Tfloat);
}
int
nc_get_var_double(int cdfid, int varid, double *value)
{   
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) 
	return lnc_get_var_double((*conns)[cdfid]->get_ncid(), varid, value);
#endif

    return DODvario(cdfid, varid, 0, 0, 0, value, Tdouble);
} 

int
nc_get_varm_text(int cdfid, int varid,
		 const size_t *start, const size_t *edges,
		 const ptrdiff_t *stride,
		 const ptrdiff_t *imap,
		 char *value)
{   
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) 
	return lnc_get_varm_text((*conns)[cdfid]->get_ncid(), varid, start,
				 edges, stride, imap, value);
#endif

    return GenRead(cdfid, varid, start, edges, 
		   stride, imap, (void *)value, Ttext);
}
int
nc_get_varm_uchar(int cdfid, int varid,
		  const size_t *start, const size_t *edges,
		  const ptrdiff_t *stride,
		  const ptrdiff_t *imap,
		  unsigned char *value)
{   
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) 
	return lnc_get_varm_uchar((*conns)[cdfid]->get_ncid(), varid, start,
				  edges, stride, imap, value);
#endif

    return GenRead(cdfid, varid, start, edges, 
		   stride, imap, (void *)value, Tuchar);
}
int
nc_get_varm_schar(int cdfid, int varid,
		  const size_t *start, const size_t *edges,
		  const ptrdiff_t *stride,
		  const ptrdiff_t *imap,
		  signed char *value)
{   
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) 
	return lnc_get_varm_schar((*conns)[cdfid]->get_ncid(), varid, start,
				  edges, stride, imap, value);
#endif

    return GenRead(cdfid, varid, start, edges, 
		   stride, imap, (void *)value, Tschar);
}
int
nc_get_varm_short(int cdfid, int varid,
		  const size_t *start, const size_t *edges,
		  const ptrdiff_t *stride,
		  const ptrdiff_t *imap,
		  short *value)
{   
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) 
	return lnc_get_varm_short((*conns)[cdfid]->get_ncid(), varid, start,
				  edges, stride, imap, value);
#endif

    return GenRead(cdfid, varid, start, edges, 
		   stride, imap, (void *)value, Tshort);
} 
int
nc_get_varm_int(int cdfid, int varid,
		const size_t *start, const size_t *edges,
		const ptrdiff_t *stride,
		const ptrdiff_t *imap,
		int *value)
{   
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) 
	return lnc_get_varm_int((*conns)[cdfid]->get_ncid(), varid, start,
				edges, stride, imap, value);
#endif

    return GenRead(cdfid, varid, start, edges, 
		   stride, imap, (void *)value, Tint);
} 
int
nc_get_varm_long(int cdfid, int varid,
		 const size_t *start, const size_t *edges,
		 const ptrdiff_t *stride,
		 const ptrdiff_t *imap,
		 long *value)
{   
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) 
	return lnc_get_varm_long((*conns)[cdfid]->get_ncid(), varid, start,
				 edges, stride, imap, value);
#endif

    return GenRead(cdfid, varid, start, edges, 
		   stride, imap, (void *)value, Tlong);
} 
int
nc_get_varm_float(int cdfid, int varid,
		  const size_t *start, const size_t *edges,
		  const ptrdiff_t *stride,
		  const ptrdiff_t *imap,
		  float *value)
{   
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) 
	return lnc_get_varm_float((*conns)[cdfid]->get_ncid(), varid, start,
				  edges, stride, imap, value);
#endif

    return GenRead(cdfid, varid, start, edges, 
		   stride, imap, (void *)value, Tfloat);
}
int
nc_get_varm_double(int cdfid, int varid,
		   const size_t *start, const size_t *edges,
		   const ptrdiff_t *stride,
		   const ptrdiff_t *imap,
		   double *value)
{   
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) 
	return lnc_get_varm_double((*conns)[cdfid]->get_ncid(), varid, start,
				   edges, stride, imap, value);
#endif

    return GenRead(cdfid, varid, start, edges, 
		   stride, imap, (void *)value, Tdouble);
}

// Used for the netCDF 2 interface emulation. See lv2i.c. 
extern "C" int
nc_get_varm (
	     int cdfid,
	     int varid,
	     const size_t * start,
	     const size_t * edges,
	     const ptrdiff_t * stride,
	     const ptrdiff_t * imap,
	     void *value)
{   
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) 
	return lnc_get_varm((*conns)[cdfid]->get_ncid(), varid, start,
			    edges, stride, imap, value);
#endif

    return GenRead(cdfid, varid, start, edges, 
		   stride, imap, (void *)value, Tvoid);
}
int
nc_get_vars_text (
		  int ncid,
		  int varid,
		  const size_t * start,
		  const size_t * edges,
		  const ptrdiff_t * stride,
		  char *value)
{
    return nc_get_varm_text (ncid, varid, start, edges,
			     stride, 0, value);
}


int
nc_get_vars_uchar (
		   int ncid,
		   int varid,
		   const size_t * start,
		   const size_t * edges,
		   const ptrdiff_t * stride,
		   unsigned char *value)
{
    return nc_get_varm_uchar (ncid, varid, start, edges,
			      stride, 0, value);
}

int
nc_get_vars_schar (
		   int ncid,
		   int varid,
		   const size_t * start,
		   const size_t * edges,
		   const ptrdiff_t * stride,
		   signed char *value)
{
    return nc_get_varm_schar (ncid, varid, start, edges,
			      stride, 0, value);
}

int
nc_get_vars_short (
		   int ncid,
		   int varid,
		   const size_t * start,
		   const size_t * edges,
		   const ptrdiff_t * stride,
		   short *value)
{
    return nc_get_varm_short (ncid, varid, start, edges,
			      stride, 0, value);
}
int
nc_get_vars_int (
		 int ncid,
		 int varid,
		 const size_t * start,
		 const size_t * edges,
		 const ptrdiff_t * stride,
		 int *value)
{
    return nc_get_varm_int (ncid, varid, start, edges,
			    stride, 0, value);
}

int
nc_get_vars_long (
		  int ncid,
		  int varid,
		  const size_t * start,
		  const size_t * edges,
		  const ptrdiff_t * stride,
		  long *value)
{
    return nc_get_varm_long (ncid, varid, start, edges,
			     stride, 0, value);
}

int
nc_get_vars_float (
		   int ncid,
		   int varid,
		   const size_t * start,
		   const size_t * edges,
		   const ptrdiff_t * stride,
		   float *value)
{
    return nc_get_varm_float (ncid, varid, start, edges,
			      stride, 0, value);
}

int
nc_get_vars_double (
		    int ncid,
		    int varid,
		    const size_t * start,
		    const size_t * edges,
		    const ptrdiff_t * stride,
		    double *value)
{
    return nc_get_varm_double (ncid, varid, start, edges,
			       stride, 0, value);
}

// Used for the netCDF 2 interface emulation. See lv2i.c. 
extern "C" int
nc_get_vars (
	     int ncid,
	     int varid,
	     const size_t * start,
	     const size_t * edges,
	     const ptrdiff_t * stride,
	     void *value)
{
    return nc_get_varm (ncid, varid, start, edges,
			stride, 0, value);
}

// $Log: Dputget.cc,v $
// Revision 1.17  2005/05/11 20:17:35  jimg
// Changes for the new build: libnc-dap is now in it's own CVS module and
// builds using autoconf 2.5.7 scripts.
//
// Revision 1.16  2004/07/26 19:10:44  jimg
// Moved netCDF CL <--> OPeNDAP server interface code to nc_util and
// NCConnect.
//
// Revision 1.15  2004/06/16 20:42:20  jimg
// Added better error reporting; DODvario now catches the exception var_info()
// throws and returns NC_ENOTVAR along with the exception's error message.
//
// Revision 1.14  2004/06/16 19:19:12  jimg
// Fixed up the code in var_info. Added a test for Structure, Sequence and
// unknown which throws InternalErr.
//
// Revision 1.13  2004/06/15 22:57:58  jimg
// Fixed bug 713: Infinite loop for any translated data source with a Structure
// when accessed using one of the 'varm' functions.
//
// Revision 1.12  2004/03/09 22:56:32  jimg
// Refactored so that Pix is no longer used. Some common code (in the
// glue routines) was also factored out to functions. The netCDF 2
// interface is now supplied by the lnetcdf/lv2i.c file (which is a mostly
// unaltered copy of the original v2i.c source file). See lnetcdf/README.
//
// Revision 1.11  2004/03/08 19:08:33  jimg
// This version of the code uses the Unidata netCDF 3.5.1 version of the
// netCDF 2 API emulation. This functions call our netCDF 3 API functions
// which may either interact with a DAP server r call the local netCDF 3
// functions.
//
// Revision 1.10  2004/02/26 16:41:11  jimg
// Removed use of Pix.
//
// Revision 1.9  2004/02/26 01:04:13  jimg
// Broke out the building of a constraint from DODvario(). Removed Pix from
// DODvario() and the new build_constraint() function.
//
// Revision 1.8  2004/02/25 00:47:52  jimg
// This code will translate Structures, including ones that are nested.
// Not tested much; needs work.
//
// Revision 1.7  2003/12/08 18:06:37  edavis
// Merge release-3-4 into trunk
//
// Revision 1.6  2003/09/25 23:09:36  jimg
// Merged from 3.4.1.
//
// Revision 1.5.4.1  2003/06/06 08:23:41  reza
// Updated the servers to netCDF-3 and fixed error handling between client and server.
//
// Revision 1.5  2003/01/28 07:08:24  jimg
// Merged with release-3-2-8.
//
// Revision 1.4.4.4  2002/12/27 00:36:34  jimg
// Made several loop variables unsigned to quiet warnings about mixed compares.
//
// Revision 1.4.4.3  2002/12/05 20:32:39  pwest
// Corrected problems with IteratorAdapter code and cleaned up file descriptors
// and memory management problems.
//
// Revision 1.4.4.2  2002/11/21 23:02:19  pwest
// fixed some memory leaks and memory read errors
//
// Revision 1.4.4.1  2002/11/18 01:11:37  rmorris
// VC++ required an explicit cast to Pix in certain cases.
//
// Revision 1.4  2000/10/06 01:22:02  jimg
// Moved the CVS Log entries to the ends of files.
// Modified the read() methods to match the new definition in the dap library.
// Added exception handlers in various places to catch exceptions thrown
// by the dap library.
//
// Revision 1.3  1999/11/05 05:15:04  jimg
// Result of merge with 3-1-0
//
// Revision 1.1.2.3  1999/10/31 16:09:21  reza
// Removed incompatibilities in error handling and added support for scalar types.
//
// Revision 1.1.2.2  1999/10/29 05:05:19  jimg
// Reza's fixes plus the configure & Makefile update
//
// Revision 1.2  1999/10/21 13:19:06  reza
// IMAP and other bugs fixed for version3.
//
// Revision 1.1.2.1  1999/10/15 19:50:56  jimg
// Changed return values and conditions for NC API entry points
//
// Revision 1.1  1999/07/28 00:22:41  jimg
// Added
//
// Revision 1.21.2.2  1999/05/27 18:35:33  reza
// Added a fix for null strings.
//
// Revision 1.21.2.1  1999/05/27 17:43:22  reza
// Fixed bugs in string changes
//
// Revision 1.21  1999/05/07 23:45:31  jimg
// String --> string fixes
//
// Revision 1.20  1999/04/01 20:42:03  jimg
// Fixed a bug in the handling of arrays of String variables.
//
// Revision 1.19  1999/03/30 05:20:55  reza
// Added support for the new data types (Int16, UInt16, and Float32).
//
// Revision 1.18  1998/02/05 20:14:44  jimg
// DODS now compiles with gcc 2.8.x
//
// Revision 1.17  1997/03/04 17:18:18  jimg
// Added `delete Rdds' to DODvario so that DDS objects allocated on the heap
// are freed once they are no longer needed.
// Changed the call `free tmpbufout' to `delete tmpbufout' since tmpbufout is
// allocated using new (see type_conv).
//
// Revision 1.16  1997/02/12 06:37:47  reza
// Changed DDS returned by request_data to a pointer.
// Add check for Null DDS and turned the GUI on.
//
// Revision 1.15  1996/09/17 17:06:09  jimg
// Merge the release-2-0 tagged files (which were off on a branch) back into
// the trunk revision.
//
// Revision 1.14.2.4  1996/09/17 00:26:10  jimg
// Merged changes from a side branch which contained various changes from
// Reza and Charles.
// Removed ncdump and netexec since ncdump is now in its own directory and
// netexec is no longer used.
//
// Revision 1.14.2.3  1996/08/13 21:18:57  jimg
// *** empty log message ***
//
// Revision 1.14.2.2  1996/07/10 21:43:49  jimg
// Changes for version 2.06. These fixed lingering problems from the migration
// from version 1.x to version 2.x.
// Removed some (but not all) warning generated with gcc's -Wall option.
//
// Revision 1.14.2.1  1996/06/25 22:04:12  jimg
// Version 2.0 from Reza.
//
// Revision 1.14  1995/11/18  11:33:22  reza
// Updated member function names for DAP-1.1.1.
//
// Revision 1.13  1995/07/09  21:33:34  jimg
// Added copyright notice.
//
// Revision 1.12  1995/06/28  20:22:36  jimg
// Replaced malloc calls with calls to new (and calls to free with calls to
// delete).
//
// Revision 1.11  1995/06/23  20:14:08  reza
// Updated DODvario() call.
//
// Revision 1.10  1995/06/23  19:53:32  jimg
// Added break.
// Reformatted.
//
// Revision 1.9  1995/06/23  15:27:00  jimg
// Changed DODvario() so that it makes use of the new netio Connect class
// methods.
//
// Revision 1.8  1995/04/28  20:39:57  reza
// Moved hyperslab access to the server side (passed as a constraint argument).
//
// Revision 1.7  1995/03/23  15:03:09  reza
// Fixed a bug in HyperGet recursion.
//
// Revision 1.6  1995/03/16  16:56:24  reza
// Updated for the new DAP. All the read_val mfunc. and their memory management
// are now moved to their parent class.
// Data transfers are now in binary while DAS and DDS are still sent in ASCII.
//
// Revision 1.5  1995/02/10  04:51:58  reza
// Finished data transmission code and added Hyperslab and type conversion func.
//
// Revision 1.4  1994/12/22  04:45:34  reza
// Rearranged for simpler DODS data transmission code interface.
//
// Revision 1.3  1994/12/08  19:01:30  reza
// Added support for local netCDF file access through the original netCDF code.
//
// Revision 1.2  1994/11/23  21:06:00  reza
// First working version.
//
// Revision 1.1  1994/11/03  04:35:47  reza
// Preliminary version of netCDF -> DODS library.
