module io_profile

  use vtype_module
  use ni3_module

  implicit none

  private

  public :: IOProfileInqDimLen
  public :: IOProfileRead
  public :: IOProfileReadPtcl
  public :: IOProfileReadFlag
  public :: IOProfileOutput
  public :: IOProfileInqDimLenASCII
  public :: IOProfileReadASCII
  public :: IOProfileOutputASCII

  !----------------------------------------------------------------------------

contains

  !----------------------------------------------------------------------------

  subroutine IOProfileInqDimLen(      &
    & InFileName, DimName, &
    & NLev &
    & )

    character(*), intent(in ) :: InFileName
    character(*), intent(in ) :: DimName

    integer     , intent(out) :: NLev


    !
    ! local variables
    !
    character(stdstr) :: Mode
    integer           :: NcID


    Mode = "read"
    call ni3_open( InFileName, Mode, NcID )

    call ni3_inq_dimlen( NcID, DimName, NLev )

    call ni3_close( NcID )


  end subroutine IOProfileInqDimLen

  !----------------------------------------------------------------------------

  subroutine IOProfileRead(      &
    & InFileName, PressName, MolNumName, TempName, SurfTempName, VMRName, &
    & NLev, NMolNum, &
    & z_Press, m_MolNum, z_Temp, SurfTemp, zm_VMR     &
    & )

    character(*), intent(in) :: InFileName
    character(*), intent(in) :: PressName
    character(*), intent(in) :: MolNumName
    character(*), intent(in) :: TempName
    character(*), intent(in) :: SurfTempName
    character(*), intent(in) :: VMRName
    integer     , intent(in ) :: NLev
    integer     , intent(in ) :: NMolNum
    real(DP)    , intent(out) :: z_Press  (1:NLev)
    integer     , intent(out) :: m_MolNum (1:NMolNum)
    real(DP)    , intent(out) :: z_Temp   (1:NLev)
    real(DP)    , intent(out) :: SurfTemp
    real(DP)    , intent(out) :: zm_VMR   (1:NLev,1:NMolNum)


    !
    ! local variables
    !
    character(stdstr) :: Mode
    integer           :: NcID


    Mode = "read"
    call ni3_open( InFileName, Mode, NcID )

    call ni3_get_var( NcID, PressName   , z_Press  )
    call ni3_get_var( NcID, MolNumName  , m_MolNum )
    call ni3_get_var( NcID, TempName    , z_Temp   )
    call ni3_get_var( NcID, VMRName     , zm_VMR   )
    call ni3_get_var( NcID, SurfTempName, SurfTemp )

    call ni3_close( NcID )


  end subroutine IOProfileRead

  !----------------------------------------------------------------------------

  subroutine IOProfileReadPtcl(      &
    & InFileName, &
    & NLev, NPtclNum, &
    & z_Press, a_PtclNum, a_PtclName, a_PtclDens, za_PtclEffRad, za_PtclMMR  &
    & )

    character(*), intent(in ) :: InFileName
    integer     , intent(in ) :: NLev
    integer     , intent(in ) :: NPtclNum
    real(DP)    , intent(out) :: z_Press      (1:NLev)
    integer     , intent(out) :: a_PtclNum    (1:NPtclNum)
    character(*), intent(out) :: a_PtclName   (1:NPtclNum)
    real(DP)    , intent(out) :: a_PtclDens   (1:NPtclNum)
    real(DP)    , intent(out) :: za_PtclEffRad(1:NLev,1:NPtclNum)
    real(DP)    , intent(out) :: za_PtclMMR   (1:NLev,1:NPtclNum)


    !
    ! local variables
    !
    character(stdstr) :: Mode
    integer           :: NcID
    character(stdstr) :: Name


    Mode = "read"
    call ni3_open( InFileName, Mode, NcID )

    !-------------------

    Name     = "PressM"
    call ni3_get_var( NcID, Name, z_Press )
!!$    Name     = "CharNum"
!!$    call ni3_get_var( NcID, Name, a_CharNum )

    !-------------------

    if ( NPtclNum > 0 ) then

      !-------------------

      Name     = "PtclNum"
      call ni3_get_var( NcID, Name, a_PtclNum )

      !-------------------

      Name     = "PtclDens"
      call ni3_get_var( NcID, Name, a_PtclDens )
      !
      Name     = "PtclName"
      call ni3_get_var( NcID, Name, a_PtclName )
      !
      Name     = "PtclEffRad"
      call ni3_get_var( NcID, Name, za_PtclEffRad )
      !
      Name     = "PtclMMR"
      call ni3_get_var( NcID, Name, za_PtclMMR )

      !-------------------

    end if

    !-------------------

    call ni3_close( NcID )


  end subroutine IOProfileReadPtcl

  !----------------------------------------------------------------------------

  subroutine IOProfileReadFlag( &
    & InFileName,               &
    & NLev,                     &
    & z_FlagCalc                &
    & )

    character(*), intent(in ) :: InFileName
    integer     , intent(in ) :: NLev
    logical     , intent(out) :: z_FlagCalc   (1:NLev)


    !
    ! local variables
    !
    character(stdstr) :: Mode
    integer           :: NcID
    character(stdstr) :: Name
    integer           :: z_iFlagCalc(1:NLev)
    integer           :: k


    Mode = "read"
    call ni3_open( InFileName, Mode, NcID )

    !-------------------

    Name = "FlagCalc"
    if ( ni3_chk_var( NcID, Name ) ) then

      call ni3_get_var( NcID, Name, z_iFlagCalc )

      do k = 1, NLev
        if ( z_iFlagCalc(k) == 1 ) then
          z_FlagCalc(k) = .true.
        else
          z_FlagCalc(k) = .false.
        end if
      end do

    end if

    !-------------------

    call ni3_close( NcID )


  end subroutine IOProfileReadFlag

  !----------------------------------------------------------------------------

  subroutine IOProfileOutput( &
    & OutNcFn, &
    & NLev, NMolNum, r_Press, m_MolNum, &
    & r_Temp, rm_VMR, Ps, Ts, &
    & z_Press, &
    & NChar, NPtcl, a_PtclNum, a_PtclName, &
    & a_PtclDens, za_PtclEffRad, za_PtclMMR, &
    & r_Dens, r_AdLapRate, r_Cp &
    & )

    character(*), intent(in) :: OutNcFn
    integer     , intent(in) :: NLev
    integer     , intent(in) :: NMolNum
    real(dp)    , intent(in) :: r_Press (NLev)
    integer     , intent(in) :: m_MolNum(NMolNum)
    real(dp)    , intent(in) :: r_Temp  (NLev)
    real(dp)    , intent(in) :: rm_VMR  (NLev,NMolNum)
    real(dp)    , intent(in) :: Ps
    real(dp)    , intent(in) :: Ts
    real(dp)    , intent(in) :: z_Press      (NLev-1)
    integer     , intent(in) :: NChar
    integer     , intent(in) :: NPtcl
    integer     , intent(in) :: a_PtclNum    (NPtcl)
    character(*), intent(in) :: a_PtclName   (NPtcl)
    real(dp)    , intent(in) :: a_PtclDens   (NPtcl)
    real(dp)    , intent(in) :: za_PtclEffRad(NLev-1,NPtcl)
    real(dp)    , intent(in) :: za_PtclMMR   (NLev-1,NPtcl)
    real(dp)    , intent(in), optional :: r_Dens     (NLev)
    real(dp)    , intent(in), optional :: r_AdLapRate(NLev)
    real(dp)    , intent(in), optional :: r_Cp       (NLev)


    ! Local variables
    !
    integer                 :: NcID
    character(extstr)       :: Mode
    character(extstr)       :: Name
    character(extstr)       :: StdName
    character(extstr)       :: LongName
    character(extstr)       :: Units
    integer                 :: NDims
    character(extstr), allocatable :: a_DimNames(:)

    integer                 :: a_CharNum( NChar )

    integer :: l


    do l = 1, NChar
      a_CharNum(l) = l
    end do


    ! Output

    Mode = "new"
    call ni3_open( OutNcFN, Mode, NcID )

    Name     = "Press"
    StdName  = Name
    LongName = "Pressure"
    Units    = "Pa"
    call ni3_set_dim( NcID, Name, NI3_DOUBLE, r_Press, &
      & StdName, LongName, Units )
    Name     = "MolNum"
    StdName  = Name
    LongName = "Molecular number"
    Units    = "1"
    call ni3_set_dim( NcID, Name, NI3_INT, m_MolNum, &
      & StdName, LongName, Units )
    Name     = "PtclNum"
    StdName  = Name
    LongName = "Particle number"
    Units    = "1"
    call ni3_set_dim( NcID, Name, NI3_INT, a_PtclNum, &
      & StdName, LongName, Units )
    Name     = "PressM"
    StdName  = Name
    LongName = "Pressure"
    Units    = "Pa"
    call ni3_set_dim( NcID, Name, NI3_DOUBLE, z_Press, &
      & StdName, LongName, Units )
    Name     = "CharNum"
    StdName  = Name
    LongName = "character number"
    Units    = "1"
    call ni3_set_dim( NcID, Name, NI3_INT, a_CharNum, &
      & StdName, LongName, Units )


    NDims = 1
    allocate( a_DimNames( NDims ) )
    !
    Name     = "Temp"
    StdName  = Name
    LongName = "Temperature"
    Units    = "K"
    a_DimNames( 1 ) = "Press"
    call ni3_def_var( NcID, Name, NI3_DOUBLE, NDims, a_DimNames, &
      & StdName, LongName, Units )
    call ni3_put_var( NcID, Name, r_Temp )
!!$  !
!!$  Name     = "MolWt"
!!$  StdName  = Name
!!$  LongName = "Molecular weight"
!!$  Units    = "kg mol-1"
!!$  a_DimNames( 1 ) = "MolNum"
!!$  call ni3_def_var( NcID, Name, NI3_DOUBLE, NDims, a_DimNames, &
!!$    & StdName, LongName, Units )
!!$  call ni3_put_var( NcID, Name, m_MolWt )
    !
    deallocate( a_DimNames )


    NDims = 2
    allocate( a_DimNames( NDims ) )
    !
    Name     = "VMR"
    StdName  = Name
    LongName = "Volume mixing ratio"
    Units    = "1"
    a_DimNames( 1 ) = "Press"
    a_DimNames( 2 ) = "MolNum"
    call ni3_def_var( NcID, Name, NI3_DOUBLE, NDims, a_DimNames, &
      & StdName, LongName, Units )
    call ni3_put_var( NcID, Name, rm_VMR )
    !
    deallocate( a_DimNames )

    !-------------------

    NDims = 0
    allocate( a_DimNames( NDims ) )
    !
    Name     = "Ts"
    StdName  = Name
    LongName = "Surface temperature"
    Units    = "K"
    call ni3_def_var( NcID, Name, NI3_DOUBLE, NDims, a_DimNames, &
      & StdName, LongName, Units )
    call ni3_put_var( NcID, Name, Ts )
    !
    Name     = "Ps"
    StdName  = Name
    LongName = "Surface pressure"
    Units    = "Pa"
    call ni3_def_var( NcID, Name, NI3_DOUBLE, NDims, a_DimNames, &
      & StdName, LongName, Units )
    call ni3_put_var( NcID, Name, Ps )
    !
    deallocate( a_DimNames )

    !-------------------

    NDims = 1
    allocate( a_DimNames( NDims ) )
    !
    Name     = "PtclDens"
    StdName  = Name
    LongName = "particle density"
    Units    = "kg m-3"
    a_DimNames( 1 ) = "PtclNum"
    call ni3_def_var( NcID, Name, NI3_DOUBLE, NDims, a_DimNames, &
      & StdName, LongName, Units )
    call ni3_put_var( NcID, Name, a_PtclDens )
    !
    deallocate( a_DimNames )

    NDims = 2
    allocate( a_DimNames( NDims ) )
    !
    Name     = "PtclName"
    StdName  = Name
    LongName = "particle name"
    Units    = "m"
    a_DimNames( 1 ) = "CharNum"
    a_DimNames( 2 ) = "PtclNum"
    call ni3_def_var( NcID, Name, NI3_CHAR, NDims, a_DimNames, &
      & StdName, LongName, Units )
    call ni3_put_var( NcID, Name, a_PtclName )
    !
    Name     = "PtclEffRad"
    StdName  = Name
    LongName = "particle effective radius"
    Units    = "m"
    a_DimNames( 1 ) = "PressM"
    a_DimNames( 2 ) = "PtclNum"
    call ni3_def_var( NcID, Name, NI3_DOUBLE, NDims, a_DimNames, &
      & StdName, LongName, Units )
    call ni3_put_var( NcID, Name, za_PtclEffRad )
    !
    Name     = "PtclMMR"
    StdName  = Name
    LongName = "particle mass mixing ratio"
    Units    = "1"
    a_DimNames( 1 ) = "PressM"
    a_DimNames( 2 ) = "PtclNum"
    call ni3_def_var( NcID, Name, NI3_DOUBLE, NDims, a_DimNames, &
      & StdName, LongName, Units )
    call ni3_put_var( NcID, Name, za_PtclMMR )
    !
    deallocate( a_DimNames )


    if ( present( r_Dens ) ) then
      NDims = 1
      allocate( a_DimNames( NDims ) )
      !
      Name     = "Dens"
      StdName  = Name
      LongName = "Density"
      Units    = "kg m-3"
      a_DimNames( 1 ) = "Press"
      call ni3_def_var( NcID, Name, NI3_DOUBLE, NDims, a_DimNames, &
        & StdName, LongName, Units )
      call ni3_put_var( NcID, Name, r_Dens )
      !
      deallocate( a_DimNames )
    end if

    if ( present( r_AdLapRate ) ) then
      NDims = 1
      allocate( a_DimNames( NDims ) )
      !
      Name     = "AdLapRate"
      StdName  = Name
      LongName = "Adiabatic lapse rate"
      Units    = "K m-1"
      a_DimNames( 1 ) = "Press"
      call ni3_def_var( NcID, Name, NI3_DOUBLE, NDims, a_DimNames, &
        & StdName, LongName, Units )
      call ni3_put_var( NcID, Name, r_AdLapRate )
      !
      deallocate( a_DimNames )
    end if

    if ( present( r_Cp ) ) then
      NDims = 1
      allocate( a_DimNames( NDims ) )
      !
      Name     = "Cp"
      StdName  = Name
      LongName = "Specific heat at constant pressure"
      Units    = "J K-1 kg-1"
      a_DimNames( 1 ) = "Press"
      call ni3_def_var( NcID, Name, NI3_DOUBLE, NDims, a_DimNames, &
        & StdName, LongName, Units )
      call ni3_put_var( NcID, Name, r_Cp )
      !
      deallocate( a_DimNames )
    end if
    !-------------------

    call ni3_close( NcID )


  end subroutine IOProfileOutput

  !----------------------------------------------------------------------------

  subroutine IOProfileInqDimLenASCII(      &
    & InFileName, NLev, NMol, NPtcl &
    & )

    use fi_module, only : fi_open

    character(*), intent(in ) :: InFileName
    integer     , intent(out) :: NLev
    integer     , intent(out) :: NMol
    integer     , intent(out) :: NPtcl


    !
    ! local variables
    !
    integer :: FileUnit

    integer :: k


    call fi_open( InFileName, "read", FileUnit )
    read( FileUnit, * )
    read( FileUnit, * ) NLev
    read( FileUnit, * ) NMol
    read( FileUnit, * ) !( m_MolNum(m), m=1,NMol )
    read( FileUnit, * ) !Ts
    do k = 1, NLev
      read( FileUnit, * ) !r_Press(k), r_Temp(k), ( rm_VMR(k,m), m=1,NMol )
    end do
    read( FileUnit, * ) NPtcl
    close( FileUnit )


  end subroutine IOProfileInqDimLenASCII

  !----------------------------------------------------------------------------

  subroutine IOProfileReadASCII( &
    & InFileName, &
    & PressName, MolNumName, TempName, SurfTempName, VMRName, &
    & NLev, &
    & NMol, &
    & r_Press, m_MolNum, r_Temp, rm_VMR, SurfTemp, &
    & NPtcl, &
    & z_Press, a_PtclNum, a_PtclName, a_PtclDens, za_PtclEffRad, za_PtclMMR  &
    & )

    use fi_module, only : fi_open

    character(*), intent(in) :: InFileName
    character(*), intent(in) :: PressName
    character(*), intent(in) :: MolNumName
    character(*), intent(in) :: TempName
    character(*), intent(in) :: SurfTempName
    character(*), intent(in) :: VMRName
    integer     , intent(in ) :: NLev
    integer     , intent(in ) :: NMol
    real(DP)    , intent(out) :: r_Press  (1:NLev)
    integer     , intent(out) :: m_MolNum (1:NMol)
    real(DP)    , intent(out) :: r_Temp   (1:NLev)
    real(DP)    , intent(out) :: rm_VMR   (1:NLev,1:NMol)
    real(DP)    , intent(out) :: SurfTemp
    !
    integer     , intent(in ) :: NPtcl
    real(DP)    , intent(out) :: z_Press      (1:NLev-1)
    integer     , intent(out) :: a_PtclNum    (1:NPtcl)
    character(*), intent(out) :: a_PtclName   (1:NPtcl)
    real(DP)    , intent(out) :: a_PtclDens   (1:NPtcl)
    real(DP)    , intent(out) :: za_PtclEffRad(1:NLev-1,1:NPtcl)
    real(DP)    , intent(out) :: za_PtclMMR   (1:NLev-1,1:NPtcl)


    ! Local variables
    !
    integer :: FileUnit

    integer :: NLevTmp
    integer :: NMolTmp
    integer :: NPtclTmp

    integer :: k
    integer :: m


    !
    ! Input
    !

    ! ==========  P(NLev)  , T(NLev)  , VMR(NLev)    (NLev)-th level (upper boundary)
    ! -  -  -  -  Rad(NLev-1), MMR(NLev-1)           (NLev-1)-th layer
    ! ==========  P(NLev-1), T(NLev-1), VMR(NLev-1)  (NLev-1)-th level
    ! -  -  -  -  Rad(NLev-2), MMR(NLev-2)           (NLev-2)-th layer
    !
    ! ...
    !
    ! -  -  -  -  Rad(3), MMR(3)      2nd layer
    ! ==========  P(3), T(3), VMR(3)  3rd level
    ! -  -  -  -  Rad(2), MMR(2)      2nd layer
    ! ==========  P(2), T(2), VMR(2)  2nd level
    ! -  -  -  -  Rad(1), MMR(1)      1st layer
    ! ==========  P(1), T(1), VMR(1)  1st level
    ! ==========  P(1), Ts            ground    (lower boundary)
    !
    !
    ! NLev        : Number of levels (layer interfaces)
    ! NMol        : Number of molecules
    ! MolNumX     : Xth molecular number
    ! Ts          : surface temperature
    ! P(k)        : pressure at k'th level
    ! T(k)        : temperature at k'th level
    ! MolX_VMR(k) : volume mixing ratio of Xth molecule at k'th level
    ! NPtcl       : Number of particles
    ! PtclYName   : name of Yth particle
    ! PtclYDens   : density of Yth particle
    ! PtclYRad(l) : effective radius of of Yth particle at l'th layer
    ! PtclYMMR(l) : mass mixing ratio of of Yth particle at l'th layer


    ! l.1                        : Comment (This line is not read, actually.)
    ! l.2                        : NLev
    ! l.3                        : NMol
    ! l.4                        : MolNum1, MolNum2, ...
    !
    ! l.5                        : Ps
    ! l.6                        : P(1)   , T(1)   , Mol1_VMR(1)   , Mol2_VMR(1)   , ...
    ! l.7                        : P(2)   , T(2)   , Mol1_VMR(2)   , Mol2_VMR(2)   , ...
    ! ...
    ! l.(5+NLev)                 : P(NLev), T(NLev), Mol1_VMR(NLev), Mol2_VMR(NLev), ...
    !
    ! l.(5+NLev+1)               : NPtcl
    ! l.(5+NLev+2)               : Ptcl1Name, Ptcl2Name, ...
    ! l.(5+NLev+3)               : Ptcl1Dens, Ptcl2Dens, ...
    !
    ! l.(5+NLev+3+1)             : Ptcl1Rad(1)     , Ptcl2Rad(1)     , ...
    ! l.(5+NLev+3+2)             : Ptcl1Rad(2)     , Ptcl2Rad(2)     , ...
    ! ...
    ! l.(5+NLev+3+NLev-1)        : Ptcl1Rad(NLev-1), Ptcl2Rad(NLev-1), ...
    !
    ! l.(5+NLev+3+NLev-1+1)      : Ptcl1MMR(1)     , Ptcl2MMR(1) , ...
    ! l.(5+NLev+3+NLev-1+2)      : Ptcl1MMR(2)     , Ptcl2MMR(2) , ...
    ! ...
    ! l.(5+NLev+3+NLev-1+NLev-1) : Ptcl1MMR(NLev-1), Ptcl2MMR(NLev-1) , ...
    !
    call fi_open( InFileName, "read", FileUnit )
    read( FileUnit, * )
    read( FileUnit, * ) NLevTmp
    read( FileUnit, * ) NMolTmp

    if ( NLevTmp .ne. size(r_Press) ) then
      write( 6, * ) 'Error : Number of level is inappropriate.'
      write( 6, * ) '        ', NLevTmp, ' != ', size(r_Press)
      stop
    end if
    if ( NMolTmp .ne. size(m_MolNum) ) then
      write( 6, * ) 'Error : Number of molecular number is inappropriate.'
      write( 6, * ) '        ', NMolTmp, ' != ', size(m_MolNum)
      stop
    end if

    read( FileUnit, * ) ( m_MolNum(m), m=1,NMol )
    read( FileUnit, * ) SurfTemp
    do k = 1, NLev
      read( FileUnit, * ) r_Press(k), r_Temp(k), ( rm_VMR(k,m), m=1,NMol )
    end do
    read( FileUnit, * ) NPtclTmp

    if ( NPtclTmp .ne. NPtcl ) then
      write( 6, * ) 'Error : Number of particles is inappropriate.'
      write( 6, * ) '        ', NPtcl, ' != ', size(a_PtclNum)
      stop
    end if

    if ( NPtcl > 0 ) then
      do m = 1, NPtcl
        a_PtclNum(m) = m
      end do
      read( FileUnit, * ) ( a_PtclName  (m), m=1,NPtcl )
      read( FileUnit, * ) ( a_PtclDens  (m), m=1,NPtcl )
      do k = 1, NLev-1
        read( FileUnit, * ) ( za_PtclEffRad(k,m), m=1,NPtcl )
      end do
      do k = 1, NLev-1
        read( FileUnit, * ) ( za_PtclMMR(k,m), m=1,NPtcl )
      end do
    end if
    close( FileUnit )


  end subroutine IOProfileReadASCII

  !----------------------------------------------------------------------------

  subroutine IOProfileOutputASCII( &
    & OutFileName, Comment, &
    & NLev, NMol, &
    & r_Press, m_MolNum, r_Temp, rm_VMR, SurfTemp,     &
    & NPtcl, &
    & z_Press, a_PtclNum, &
    & a_PtclName, a_PtclDens, za_PtclEffRad, za_PtclMMR  &
    & )

    use fi_module, only : fi_open

    character(*), intent(in) :: OutFileName
    character(*), intent(in) :: Comment
    integer     , intent(in) :: NLev
    integer     , intent(in) :: NMol
    real(DP)    , intent(in) :: r_Press  (1:NLev)
    integer     , intent(in) :: m_MolNum (1:NMol)
    real(DP)    , intent(in) :: r_Temp   (1:NLev)
    real(DP)    , intent(in) :: rm_VMR   (1:NLev,1:NMol)
    real(DP)    , intent(in) :: SurfTemp
    !
    integer     , intent(in) :: NPtcl
    real(DP)    , intent(in) :: z_Press      (1:NLev-1)
    integer     , intent(in) :: a_PtclNum    (1:NPtcl)
    character(*), intent(in) :: a_PtclName   (1:NPtcl)
    real(DP)    , intent(in) :: a_PtclDens   (1:NPtcl)
    real(DP)    , intent(in) :: za_PtclEffRad(1:NLev-1,1:NPtcl)
    real(DP)    , intent(in) :: za_PtclMMR   (1:NLev-1,1:NPtcl)


    ! Local variables
    !
    integer :: FileUnit

    integer :: k
    integer :: m


    !
    ! Output
    !

    call fi_open( OutFileName, "write", FileUnit )
    write( FileUnit, * ) trim( Comment )
    write( FileUnit, * ) NLev
    write( FileUnit, * ) NMol
    write( FileUnit, * ) ( m_MolNum(m), m=1,NMol )
    write( FileUnit, * ) SurfTemp
    do k = 1, NLev
      write( FileUnit, * ) r_Press(k), r_Temp(k), ( rm_VMR(k,m), m=1,NMol )
    end do
    write( FileUnit, * ) NPtcl
    if ( NPtcl > 0 ) then
!!$      do m = 1, NPtcl
!!$        a_PtclNum(m) = m
!!$      end do
      write( FileUnit, * ) ( a_PtclName  (m), m=1,NPtcl )
      write( FileUnit, * ) ( a_PtclDens  (m), m=1,NPtcl )
      do k = 1, NLev-1
        write( FileUnit, * ) ( za_PtclEffRad(k,m), m=1,NPtcl )
      end do
      do k = 1, NLev-1
        write( FileUnit, * ) ( za_PtclMMR(k,m), m=1,NPtcl )
      end do
    end if
    close( FileUnit )


  end subroutine IOProfileOutputASCII

  !----------------------------------------------------------------------------

end module io_profile
