!= 
!
!= CO2 condensation temperature (Kasting, 1991)
!
! Authors::   Yoshiyuki O. Takahashi
! Version::   $Id: saturate_co2_k91.f90,v 1.2 2014/05/07 09:39:21 murashin Exp $ 
! Tag Name::  $Name:  $
! Copyright:: Copyright (C) GFD Dennou Club, 2008. All rights reserved.
! License::   See COPYRIGHT[link:../../../COPYRIGHT]
!

module saturate_co2_k91
  !
  != 
  !
  != CO2 condensation temperature (Kasting, 1981)
  !
  ! <b>Note that Japanese and English are described in parallel.</b>
  !
  !== References
  !
  ! * Kasting, J. F., 1991:
  !   CO2 condensation and the climate of early Mars,
  !   <i>Icarus</i>, <b>94</b>, 1--13.
  !
  !== Procedures List
  ! 
!!$  ! DryConvAdjust :: 乾燥対流調節
!!$  ! ------------  :: ------------
!!$  ! DryConvAdjust :: Dry convective adjustment
  !
  !== NAMELIST
  !
  ! NAMELIST#major_comp_saturate_nml
  !

  ! モジュール引用 ; USE statements
  !

  ! 種別型パラメタ
  ! Kind type parameter
  !
  use dc_types, only: DP, &      ! 倍精度実数型. Double precision. 
    &                 STRING     ! 文字列.       Strings. 

  ! NAMELIST ファイル入力に関するユーティリティ
  ! Utilities for NAMELIST file input
  !
  use namelist_util, only: MaxNmlArySize
                              ! NAMELIST から読み込む配列の最大サイズ. 
                              ! Maximum size of arrays loaded from NAMELIST

  ! メッセージ出力
  ! Message output
  !
  use dc_message, only: MessageNotify

  ! 宣言文 ; Declaration statements
  !
  implicit none
  private


  ! 公開手続き
  ! Public procedure
  !
  public :: xyz_SaturateCO2K91TempCond
  public :: xyz_SaturateCO2K91PressSat
  public :: xyz_SaturateCO2K91DPressSatDT
  public :: SaturateCO2K91Init


  ! 公開変数
  ! Public variables
  !
  logical, save, public:: saturate_co2_k91_inited = .false.
                              ! 初期設定フラグ. 
                              ! Initialization flag

  ! 非公開変数
  ! Private variables
  !


  character(*), parameter:: module_name = 'saturate_co2_k91'
                              ! モジュールの名称. 
                              ! Module name
  character(*), parameter:: version = &
    & '$Name:  $' // &
    & '$Id: saturate_co2_p81.f90,v 1.2 2014/05/07 09:39:21 murashin Exp $'
                              ! モジュールのバージョン
                              ! Module version


contains

  !--------------------------------------------------------------------------------------

  function xyz_SaturateCO2K91TempCond( xyz_Press ) result( xyz_TempCond )
    !
    ! CO2 相変化
    !
    ! CO2 phase change
    !

    ! モジュール引用 ; USE statements
    !


    ! 宣言文 ; Declaration statements
    !
    implicit none

    real(DP), intent(in ):: xyz_Press   (:,:,:)
                              ! $ p $ . 気圧 (整数レベル). 
                              ! Air pressure (full level)
    real(DP) :: xyz_TempCond(size(xyz_Press,1), size(xyz_Press,2), size(xyz_Press,3))
                              !
                              ! Condensation temperature

    ! 作業変数
    ! Work variables
    !
    real(DP) :: Temp1
    real(DP) :: Temp2
    real(DP) :: TempMP
    real(DP) :: DelTemp
    real(DP) :: PressCO2VapSat1
    real(DP) :: PressCO2VapSat2
    real(DP) :: PressCO2VapSatMP
    integer :: imax
    integer :: jmax
    integer :: kmax
    integer :: i
    integer :: j
    integer :: k


    ! 実行文 ; Executable statement
    !


    ! 初期化
    ! Initialization
    !
    if ( .not. saturate_co2_k91_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    imax = size( xyz_TempCond, 1 )
    jmax = size( xyz_TempCond, 2 )
    kmax = size( xyz_TempCond, 3 )

    do k = 1, kmax
      do j = 1, jmax
        do i = 1, imax

          ! Bisection method
          Temp1 = 200.0d0
          PressCO2VapSat1 = SaturateCO2K91PressSat( Temp1 )
          if ( PressCO2VapSat1 >= xyz_Press(i,j,k) ) then
            DelTemp = -20.0d0
            do
              Temp1 = Temp1 + DelTemp
              PressCO2VapSat1 = SaturateCO2K91PressSat( Temp1 )
              if ( PressCO2VapSat1 < xyz_Press(i,j,k) ) exit
            end do
          end if
          DelTemp = 20.0d0
          do
            Temp2 = Temp1 + DelTemp
            PressCO2VapSat2 = SaturateCO2K91PressSat( Temp2 )
            if ( PressCO2VapSat2 >= xyz_Press(i,j,k) ) exit
            Temp1 = Temp2
          end do
          do
            TempMP = ( Temp1 + Temp2 ) / 2.0d0
            PressCO2VapSatMP = SaturateCO2K91PressSat( TempMP )
            if ( PressCO2VapSatMP >= xyz_Press(i,j,k) ) then
              Temp2 = TempMP
            else
              Temp1 = TempMP
            end if
            if ( abs( Temp2 - Temp1 ) < 1.0d-5 ) then
              exit
            end if
          end do
          xyz_TempCond(i,j,k) = Temp1
        end do
      end do
    end do


  end function xyz_SaturateCO2K91TempCond

  !--------------------------------------------------------------------------------------

  function xyz_SaturateCO2K91PressSat( xyz_Temp ) result( xyz_Press )
    !
    ! CO2 相変化
    !
    ! CO2 phase change
    !

    ! モジュール引用 ; USE statements
    !


    ! 宣言文 ; Declaration statements
    !
    implicit none

    real(DP), intent(in ):: xyz_Temp (:,:,:)
                              ! $ T $ . 
                              ! Air temperature
    real(DP) :: xyz_Press(size(xyz_Temp,1), size(xyz_Temp,2), size(xyz_Temp,3))
                              !
                              ! Saturation pressure

    ! 作業変数
    ! Work variables
    !
    integer :: imax
    integer :: jmax
    integer :: kmax
    integer :: i
    integer :: j
    integer :: k


    ! 実行文 ; Executable statement
    !


    ! 初期化
    ! Initialization
    !
    if ( .not. saturate_co2_k91_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    imax = size( xyz_Temp, 1 )
    jmax = size( xyz_Temp, 2 )
    kmax = size( xyz_Temp, 3 )

    do k = 1, kmax
      do j = 1, jmax
        do i = 1, imax

          if ( xyz_Temp(i,j,k) > 216.56d0 ) then
            xyz_Press(i,j,k) = &
              &   3.128082d0 &
              & - 867.2124 / xyz_Temp(i,j,k) &
              & + 18.65612d-3 * xyz_Temp(i,j,k) &
              & - 72.48820d-6 * xyz_Temp(i,j,k) * xyz_Temp(i,j,k) &
              & + 93.0d-9 * xyz_Temp(i,j,k) * xyz_Temp(i,j,k) * xyz_Temp(i,j,k)
          else
            xyz_Press(i,j,k) = &
              &   6.760956d0 &
              & - 1284.07d0 / ( xyz_Temp(i,j,k) - 4.718d0 ) &
              & + 1.256d-4 * ( xyz_Temp(i,j,k) - 143.15d0 )
          end if

        end do
      end do
    end do
    xyz_Press = 10.0d0**xyz_Press
    xyz_Press = xyz_Press * 101325.0d0


  end function xyz_SaturateCO2K91PressSat

  !--------------------------------------------------------------------------------------

  function SaturateCO2K91PressSat( Temp ) result( Press )
    !
    ! CO2 相変化
    !
    ! CO2 phase change
    !

    ! モジュール引用 ; USE statements
    !


    ! 宣言文 ; Declaration statements
    !
    implicit none

    real(DP), intent(in ):: Temp
                              ! $ T $ . 
                              ! Air temperature
    real(DP) :: Press
                              !
                              ! Saturation pressure

    ! 作業変数
    ! Work variables
    !
    real(DP) :: xyz_Temp (1,1,1)
    real(DP) :: xyz_Press(1,1,1)


    ! 実行文 ; Executable statement
    !


    ! 初期化
    ! Initialization
    !
    if ( .not. saturate_co2_k91_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    xyz_Temp = Temp

    xyz_Press = xyz_SaturateCO2K91PressSat( xyz_Temp )

    Press = xyz_Press(1,1,1)


  end function SaturateCO2K91PressSat

  !--------------------------------------------------------------------------------------

  function xyz_SaturateCO2K91DPressSatDT( xyz_Temp ) result( xyz_DPressSatDT )
    !
    ! CO2 相変化
    !
    ! CO2 phase change
    !

    ! モジュール引用 ; USE statements
    !


    ! 宣言文 ; Declaration statements
    !
    implicit none

    real(DP), intent(in ):: xyz_Temp (:,:,:)
                              ! $ T $ . 
                              ! Air temperature
    real(DP) :: xyz_DPressSatDT(size(xyz_Temp,1), size(xyz_Temp,2), size(xyz_Temp,3))
                              !
                              ! Derivative of Condensation temperature with temperature

    ! 作業変数
    ! Work variables
    !
    integer :: imax
    integer :: jmax
    integer :: kmax
    integer :: i
    integer :: j
    integer :: k


    ! 実行文 ; Executable statement
    !


    ! 初期化
    ! Initialization
    !
    if ( .not. saturate_co2_k91_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    imax = size( xyz_Temp, 1 )
    jmax = size( xyz_Temp, 2 )
    kmax = size( xyz_Temp, 3 )

    do k = 1, kmax
      do j = 1, jmax
        do i = 1, imax

          if ( xyz_Temp(i,j,k) > 216.56d0 ) then
            xyz_DPressSatDT(i,j,k) = &
              & + 867.2124 / ( xyz_Temp(i,j,k) * xyz_Temp(i,j,k) ) &
              & + 18.65612d-3 &
              & - 72.48820d-6 * 2.0d0 * xyz_Temp(i,j,k) &
              & + 93.0d-9 * 3.0d0 * xyz_Temp(i,j,k) * xyz_Temp(i,j,k)
          else
            xyz_DPressSatDT(i,j,k) = &
              & + 1284.07 / ( ( xyz_Temp(i,j,k) - 4.718d0 ) * ( xyz_Temp(i,j,k) - 4.718d0 ) ) &
              & + 1.256d-4
          end if

        end do
      end do
    end do

    xyz_DPressSatDT = xyz_DPressSatDT &
      & * xyz_SaturateCO2K91PressSat( xyz_Temp )


  end function xyz_SaturateCO2K91DPressSatDT

  !--------------------------------------------------------------------------------------

  subroutine SaturateCO2K91Init
    !
    ! saturate_co2_k91 モジュールの初期化を行います. 
    ! NAMELIST#saturate_co2_k91_nml の読み込みはこの手続きで行われます. 
    !
    ! "saturate_co2_k91" module is initialized. 
    ! "NAMELIST#saturate_co2_k91_nml" is loaded in this procedure. 
    !

    ! モジュール引用 ; USE statements
    !

!!$    ! NAMELIST ファイル入力に関するユーティリティ
!!$    ! Utilities for NAMELIST file input
!!$    !
!!$    use namelist_util, only: namelist_filename, NmlutilMsg, NmlutilAryValid
!!$
!!$    ! ファイル入出力補助
!!$    ! File I/O support
!!$    !
!!$    use dc_iounit, only: FileOpen
!!$
!!$    ! 種別型パラメタ
!!$    ! Kind type parameter
!!$    !
!!$    use dc_types, only: STDOUT ! 標準出力の装置番号. Unit number of standard output

    ! 文字列操作
    ! Character handling
    !
    use dc_string, only: StoA


    ! 宣言文 ; Declaration statements
    !
    implicit none


!!$    integer:: unit_nml        ! NAMELIST ファイルオープン用装置番号. 
!!$                              ! Unit number for NAMELIST file open
!!$    integer:: iostat_nml      ! NAMELIST 読み込み時の IOSTAT. 
!!$                              ! IOSTAT of NAMELIST read

    ! NAMELIST 変数群
    ! NAMELIST group name
    !
!!$    namelist /major_comp_saturate_nml/ &
!!$      & FlagUse

          ! デフォルト値については初期化手続 "major_comp_saturate#MajorCompSaturateInit" 
          ! のソースコードを参照のこと. 
          !
          ! Refer to source codes in the initialization procedure
          ! "major_comp_saturate#MajorCompSaturateInit" for the default values. 
          !

    ! 実行文 ; Executable statement
    !

    if ( saturate_co2_k91_inited ) return


    ! デフォルト値の設定
    ! Default values settings
    !


    ! NAMELIST の読み込み
    ! NAMELIST is input
    !
!!$    if ( trim(namelist_filename) /= '' ) then
!!$      call FileOpen( unit_nml, &          ! (out)
!!$        & namelist_filename, mode = 'r' ) ! (in)
!!$
!!$      rewind( unit_nml )
!!$      read( unit_nml,                    &  ! (in)
!!$        & nml = major_comp_saturate_nml, &  ! (out)
!!$        & iostat = iostat_nml )             ! (out)
!!$      close( unit_nml )
!!$
!!$      call NmlutilMsg( iostat_nml, module_name ) ! (in)
!!$!      if ( iostat_nml == 0 ) write( STDOUT, nml = cumulus_adjust_nml )
!!$    end if


    ! 印字 ; Print
    !
    call MessageNotify( 'M', module_name, '----- Initialization Messages -----' )
    call MessageNotify( 'M', module_name, '-- version = %c', c1 = trim(version) )


    saturate_co2_k91_inited = .true.

  end subroutine SaturateCO2K91Init

  !-------------------------------------------------------------------

end module saturate_co2_k91
