#!/bin/sh
# https://unlicense.org

#
# param:
#       mo  - create mo files, `rm *.mo` to update
#       pot - update pot file
#       po  - update pot file + pot files
#
# Utilities:
# - msgfmt   (required - create .mo files)
# - xgettext (create .pot)
# - msgmerge (update .po files)

cd $(dirname "$0")

if [ -z "$VERSION" ] ; then
	if [ -f config.sh ] ; then
		. ./config.sh
	elif [ -f ../config.sh ] ; then
		. ../config.sh
	elif [ -f ../../config.sh ] ; then
		. ../../config.sh
	fi
fi

#==========================================================================

get_linguas()
{
	if [ -z "$LINGUAS" ] ; then
		LINGUAS="$(ls *.po | sed 's/\.po$//')"
	fi
	if [ -z "$LINGUAS" ] ; then
		echo "-- There no .po files --"
		exit 0
	fi
}


get_pot_source_files()
{
	if [ -z "$POT_SOURCE_FILES" ] ; then
		filez=$(find .. -type f -name '*.h' -or -name '*.c' -or -name '*.cc' -or -name '*.cpp' -or -name '*.hh')
		if [ -z "$filez" ] ; then
			echo "-- No source files have been translated --"
			exit 0
		fi
	fi
	#--
	POT_SOURCE_FILES="$(grep '_(' $filez | sed -e 's%^\./%%' -e 's%:.*%%' | sort -u)"
	if [ -z "$POT_SOURCE_FILES" ] ; then
		echo "-- No source files have been translated --"
		rm -f POTFILES
		exit 0
	fi
	echo "# generated by zzpo.sh" >  POTFILES
	echo "# this file is only used as a reference to notice if something has changed" >>  POTFILES
	echo "$POT_SOURCE_FILES" >> POTFILES
}


get_gettext_package()
{
	if [ -z "$GETTEXT_PACKAGE" ] ; then
		GETTEXT_PACKAGE=$(ls *.pot | head -n 1 | sed 's%.pot%%')
		if [ -z "$GETTEXT_PACKAGE" ] ; then
			echo ".pot file is missing"
			exit 1
		fi
	fi
}

#==========================================================================

update_pot()
{
	get_pot_source_files
	get_gettext_package
	#--
	echo "GETTEXT_PACKAGE: ${GETTEXT_PACKAGE}.pot"
	xgettext \
		--default-domain=${GETTEXT_PACKAGE} \
		--add-comments \
		--no-wrap \
		--keyword=_ \
		--keyword=N_ \
		--from-code=UTF-8 \
		--package-name=${GETTEXT_PACKAGE} \
		-o ${GETTEXT_PACKAGE}.pot2 ${POT_SOURCE_FILES}
	if [ $? -ne 0 ] ; then
		rm -f ${GETTEXT_PACKAGE}.pot2
		exit 1
	fi
	(
	echo '# This file is put in the public domain.'
	# fix charset and remove some unnecesary info
	sed \
		-e 's%charset=CHARSET%charset=UTF-8%' \
		-e '/^# /d' \
		-e '/Project-Id-Version/d' \
		-e '/Report-Msgid-Bugs-To/d' \
		-e '/POT-Creation-Date/d' \
		-e '/PO-Revision-Date/d' \
		-e '/Last-Translator/d' \
		-e '/Language-Team/d' \
		${GETTEXT_PACKAGE}.pot2
	) > ${GETTEXT_PACKAGE}.pot
	rm -f ${GETTEXT_PACKAGE}.pot2
}


update_po()
{
	get_linguas
	get_gettext_package
	#--
	for lang in ${LINGUAS}
	do
		printf " %s " "${lang}";
		sed -i -e 's%charset=CHARSET%charset=UTF-8%' ${lang}.po;
		if msgmerge ${lang}.po ${GETTEXT_PACKAGE}.pot -o ${lang}.new.po; then
			mv -f ${lang}.new.po ${lang}.po || exit 1;
		else
			echo "msgmerge for $lang failed!";
			rm -f ${lang}.new.po;
		fi;
		if grep -q 'Language: \\\n' ${lang}.po ; then
			# fix language:
			sed -i 's%Language: \\n%Language: '${lang}'\\n%' ${lang}.po
		fi
	done
    #sed -i '/#~ /d' po/*.po
	exit
}


build_mo()
{
	get_linguas
	MSGFMT=msgfmt # gmsgfmt
	# --
	for i in ${LINGUAS} ; do
		if ! test -f ${i}.mo ; then
			echo "${MSGFMT} -o ${i}.mo ${i}.po"
			${MSGFMT} -o ${i}.mo ${i}.po || exit 1
		fi
	done
}

#==========================================================================

case "$1" in
	pot)
		update_pot
		exit $?
		;;
	po)
		update_pot
		update_po
		exit $?
		;;
	mo)
		build_mo
		exit $?
		;;

	# this is only for Makefile
	install)
		DESTDIR="$2"
		if [ -z "$DESTDIR" ] ; then
			echo "$0 install: pleasy specify installation directory"
			exit 1
		fi
		#--
		build_mo
		get_gettext_package
		for cat in ${LINGUAS}
		do
			dir=${DESTDIR}/${cat}/LC_MESSAGES
			mkdir -p ${dir}
			install -c -m 644 ${cat}.mo ${dir}/${GETTEXT_PACKAGE}.mo
			echo "installing ${cat}.mo as ${dir}/${GETTEXT_PACKAGE}.mo"
		done
		exit 0
		;;
	uninstall)
		DESTDIR="$2"
		if [ -z "$DESTDIR" ] ; then
			echo "$0 install: pleasy specify installation directory"
			exit 1
		fi
		#--
		get_linguas
		get_gettext_package
		for cat in ${LINGUAS}
		do
			echo "rm -f ${DESTDIR}/${cat}/LC_MESSAGES/${GETTEXT_PACKAGE}.mo"
			rm -f ${DESTDIR}/${cat}/LC_MESSAGES/${GETTEXT_PACKAGE}.mo
		done
		exit 0
		;;
	*)
		echo "unrecognized command: $1"
		exit 1
		;;
esac
