/*
 * Copyright 2000-2017 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.jetbrains.uast.test.env

import com.intellij.openapi.components.ServiceManager
import com.intellij.openapi.vfs.VirtualFile
import com.intellij.psi.PsiElement
import com.intellij.psi.PsiFile
import com.intellij.testFramework.fixtures.LightCodeInsightFixtureTestCase
import org.jetbrains.uast.UElement
import org.jetbrains.uast.UFile
import org.jetbrains.uast.UastContext
import org.jetbrains.uast.toUElementOfType
import org.jetbrains.uast.visitor.UastVisitor

abstract class AbstractUastFixtureTest : LightCodeInsightFixtureTestCase() {
  protected val uastContext: UastContext by lazy {
    ServiceManager.getService(project, UastContext::class.java)
  }

  abstract fun getVirtualFile(testName: String): VirtualFile
  abstract fun check(testName: String, file: UFile)

  fun doTest(testName: String, checkCallback: (String, UFile) -> Unit = { testName, file -> check(testName, file) }) {
    val virtualFile = getVirtualFile(testName)

    val psiFile = psiManager.findFile(virtualFile) ?: error("Can't get psi file for $testName")
    val uFile = uastContext.convertElementWithParent(psiFile, null) ?: error("Can't get UFile for $testName")
    checkCallback(testName, uFile as UFile)
  }
}


fun <T> UElement.findElementByText(refText: String, cls: Class<T>): T {
  val matchingElements = mutableListOf<T>()
  accept(object : UastVisitor {
    override fun visitElement(node: UElement): Boolean {
      if (cls.isInstance(node) && node.psi?.text == refText) {
        matchingElements.add(node as T)
      }
      return false
    }
  })

  if (matchingElements.isEmpty()) {
    throw IllegalArgumentException("Reference '$refText' not found")
  }
  if (matchingElements.size != 1) {
    throw IllegalArgumentException("Reference '$refText' is ambiguous")
  }
  return matchingElements.single()
}

inline fun <reified T : Any> UElement.findElementByText(refText: String): T = findElementByText(refText, T::class.java)

inline fun <reified T : UElement> UElement.findElementByTextFromPsi(refText: String, strict: Boolean = true): T =
  (this.psi ?: throw AssertionError("no psi for $this")).findUElementByTextFromPsi(refText, strict)

inline fun <reified T : UElement> PsiElement.findUElementByTextFromPsi(refText: String, strict: Boolean = true): T {
  val elementAtStart = this.findElementAt(this.text.indexOf(refText))
                       ?: throw AssertionError("requested text '$refText' was not found in $this")
  val uElementContainingText = generateSequence(elementAtStart) { if (it is PsiFile) null else it.parent }
    .let {
      if (strict) it.dropWhile { !it.text.contains(refText) } else it
    }.mapNotNull { it.toUElementOfType<T>() }.firstOrNull() ?: throw AssertionError("requested text '$refText' not found as ${T::class.java}")
  if (strict && uElementContainingText.psi != null && uElementContainingText.psi?.text != refText) {
    throw AssertionError("requested text '$refText' found as '${uElementContainingText.psi?.text}' in $uElementContainingText")
  }
  return uElementContainingText;
}
