//******************************************************************************
//
// File:    Test03.java
// Package: edu.rit.mri.test
// Unit:    Class edu.rit.mri.test.Test03
//
// This Java source file is copyright (C) 2008 by Alan Kaminsky. All rights
// reserved. For further information, contact the author, Alan Kaminsky, at
// ark@cs.rit.edu.
//
// This Java source file is part of the Parallel Java Library ("PJ"). PJ is free
// software; you can redistribute it and/or modify it under the terms of the GNU
// General Public License as published by the Free Software Foundation; either
// version 3 of the License, or (at your option) any later version.
//
// PJ is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
// A PARTICULAR PURPOSE. See the GNU General Public License for more details.
//
// Linking this library statically or dynamically with other modules is making a
// combined work based on this library. Thus, the terms and conditions of the
// GNU General Public License cover the whole combination.
//
// As a special exception, the copyright holders of this library give you
// permission to link this library with independent modules to produce an
// executable, regardless of the license terms of these independent modules, and
// to copy and distribute the resulting executable under terms of your choice,
// provided that you also meet, for each linked independent module, the terms
// and conditions of the license of that module. An independent module is a
// module which is not derived from or based on this library. If you modify this
// library, you may extend this exception to your version of the library, but
// you are not obligated to do so. If you do not wish to do so, delete this
// exception statement from your version.
//
// A copy of the GNU General Public License is provided in the file gpl.txt. You
// may also obtain a copy of the GNU General Public License on the World Wide
// Web at http://www.gnu.org/licenses/gpl.html.
//
//******************************************************************************

package edu.rit.mri.test;

import edu.rit.mri.SpinSignal;

import edu.rit.numeric.ListXYSeries;
import edu.rit.numeric.NonNegativeLeastSquares;

import edu.rit.numeric.plot.Dots;
import edu.rit.numeric.plot.Plot;
import edu.rit.numeric.plot.Strokes;

import java.awt.Color;

import java.io.File;

import java.util.Scanner;

/**
 * Class Test03 is a main program that does a spin relaxometry analysis on a
 * spin signal data set. The data set is generated by the {@linkplain GenData}
 * program. The model spin signal is the sum of a series of spin signals with
 * fixed spin relaxation rates. The length of the series is specified on the
 * command line. The largest spin relaxation rate is specified on the command
 * line. The individual spin relaxation rates are spaced linearly up to the
 * largest. The program does a nonnegative linear least squares curve fit of the
 * model to the data to find the spin density for each spin signal in the
 * series. The program prints out the final fitted parameters (spin densities)
 * for the model spin signal. The program displays a plot of the data set and
 * the final model spin signal.
 * <P>
 * Usage: java edu.rit.mri.test.Test03 <I>datafile</I> <I>N</I> <I>Rmax</I>
 * <BR><I>datafile</I> = File containing <I>t</I> and <I>S</I>(<I>t</I>) values
 * <BR><I>N</I> = Number of model spin signals
 * <BR><I>Rmax</I> = Maximum spin relaxation rate
 *
 * @author  Alan Kaminsky
 * @version 11-Jun-2008
 */
public class Test03
	{

// Prevent construction.

	private Test03()
		{
		}

// Main program.

	/**
	 * Main program.
	 */
	public static void main
		(String[] args)
		throws Exception
		{
		// Parse command line arguments.
		if (args.length != 3) usage();
		File datafile = new File (args[0]);
		int N = Integer.parseInt (args[1]);
		double Rmax = Double.parseDouble (args[2]);

		// Get data X-Y series.
		ListXYSeries dataseries = new ListXYSeries();
		dataseries.add (new Scanner (datafile));
		int M = dataseries.length();

		// Compute spin relaxation rates.
		double[] R = new double [N];
		for (int j = 0; j < N; ++ j)
			{
			R[j] = ((double)(j+1))/((double) N)*Rmax;
			}

		// Create nonnegative linear least squares solver.
		NonNegativeLeastSquares solver = new NonNegativeLeastSquares (M, N);

		// Find the solution.
		for (int i = 0; i < M; ++ i)
			{
			double[] a_i = solver.a[i];
			double t_i = dataseries.x(i);
			for (int j = 0; j < N; ++ j)
				{
				a_i[j] = SpinSignal.S (R[j], t_i);
				}
			solver.b[i] = dataseries.y(i);
			}
		solver.solve();

		// Print the solution.
		for (int j = 0; j < N; ++ j)
			{
			System.out.print (R[j]);
			System.out.print ('\t');
			System.out.print (solver.x[j]);
			System.out.println();
			}

		// Create plot with data X-Y series.
		Plot plot = new Plot();
		plot.xAxisLength (1200)
			.xAxisMajorDivisions (20)
			.yAxisLength (600)
			.yAxisMajorDivisions (10)
			.seriesDots (Dots.circle (Color.BLACK, null, null, 5))
			.seriesColor (Color.BLACK)
			.seriesStroke (Strokes.solid (2))
			.xySeries (dataseries);

		// Generate model spin signal X-Y series and add to plot.
		double chisqr = 0.0;
		ListXYSeries signalseries = new ListXYSeries();
		for (int i = 0; i < dataseries.length(); ++ i)
			{
			double t = dataseries.x(i);
			double s = 0.0;
			for (int j = 0; j < N; ++ j)
				{
				s += SpinSignal.S (solver.x[j], R[j], t);
				}
			double d = s - dataseries.y(i);
			chisqr += d*d;
			signalseries.add (t, s);
			}
		System.out.println ("chi^2 = "+chisqr);
		plot.seriesDots (null)
			.seriesColor (Color.RED)
			.seriesStroke (Strokes.solid (1))
			.xySeries (signalseries);

		// Display plot.
		plot.getFrame().setVisible (true);
		}

// Hidden operations.

	/**
	 * Print a usage message and exit.
	 */
	private static void usage()
		{
		System.err.println ("Usage: java edu.rit.mri.test.Test03 <datafile> <N> <Rmax>");
		System.err.println ("<datafile> = File containing t and S(t) values");
		System.err.println ("<N> = Number of model spin signals");
		System.err.println ("<Rmax> = Maximum spin relaxation rate");
		System.exit (1);
		}

	}
