;;; diclookup-mule.el --- a client of dserver

;; Copyright (C) 1991 Shingo NISHIOKA <nishioka@sanken.osaka-u.ac.jp>

;; Author: Shingo NISHIOKA <nishioka@sanken.osaka-u.ac.jp>
;;	SAITO Yutaka <yutaka@vsp.cpg.sony.co.jp>
;;	YAMAMOTO Mitsuharu <mituharu@is.s.u-tokyo.ac.jp>
;; Maintainer: YAMAMOTO Mitsuharu <mituharu@is.s.u-tokyo.ac.jp>
;; Version: 2.1.1

;; This file is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY.  No author or distributor
;; accepts responsibility to anyone for the consequences of using it
;; or for whether it serves any particular purpose or works at all,
;; unless he says so in writing.  Refer to the GNU Emacs General Public
;; License for full details.

;; Everyone is granted permission to copy, modify and redistribute
;; this file, but only under the conditions described in the
;; GNU Emacs General Public License.   A copy of this license is
;; supposed to have been given to you along with GNU Emacs so you
;; can know your rights and responsibilities.  It should be in a
;; file named COPYING.	Among other things, the copyright notice
;; and this notice must be preserved on all copies.

;;; Commentary:

;; $B%$%s%9%H!<%k$N;EJ}(B:
;; $BJQ?t(B dserver-server-list $B$r(B .emacs, site-init.el, default.el $B$N$$$:(B
;; $B$l$+$G<!$N$h$&$K@_Dj$7$^$9!#(B
;;
;;	(setq dserver-server-list '("dserver $B$,F0$$$F$$$k%[%9%H(B" ...))
;;
;; $B$"$k$$$O$3$N%U%!%$%k$N(B (defvar dserver-server-list ...) $B$NItJ,$rD>(B
;; $B@\=q$-49$($^$9!#(B
;; $B$b$78fMxMQ$N1QOBCf<-E5$,(B eiwa $B$H(B waei $B$KJ,$+$l$F$*$i$:!"(Bchujiten $B$H(B
;; $B$$$&L>A0$K$J$C$F$$$k>l9g$O!"F1MM$K!"(B
;;
;;	(setq od-chujiten-flag t)
;;
;; $B$H$7$F2<$5$$!#(B($BCm(B: chujiten $BMQ$N30;z%F!<%V%k$O$^$@%5%]!<%H$5$l$F$$(B
;; $B$^$;$s!#:n$i$l$?J}$O(B Maintainer $B$^$G8fO"Mm$r!#(B)
;; $B<!$K$3$N%U%!%$%k$r(B load-path $B$NDL$C$?%G%#%l%/%H%j$K0\$7$F!"%P%$%H%3(B
;; $B%s%Q%$%k$7$^$9(B($B>/$7;~4V$,$+$+$j$^$9(B)$B!#(B

;; $B@_Dj$N%R%s%H(B:($BI,MW$K1~$8$F(B .emacs $B$K=q$$$F2<$5$$(B)
;;
;; $B!&:G>.8B$N@_Dj$O!)(B
;; (autoload 'online-dictionary "diclookup-mule" "Online dictionary." t nil)
;; (autoload 'diclookup "diclookup-mule"
;;   "Online dictionary. Display diclookup buffer in the current frame." t nil)
;; (autoload 'diclookup-other-frame "diclookup-mule"
;;   "Like `diclookup' command, but display diclookup buffer in another frame."
;;   t nil)
;; M-x online-dictionary $B$G5/F0$7$^$9!#(B"?" $B$G4JC1$J%X%k%W$,=P$^$9!#(B
;;
;; $B!&F|K\8lF~NO$N;~$K(B Canna $B$d$+$JF~NO$r;H$$$?$$!#(B
;; $BNc(B1: (setq od-fep-type 'canna)	; egg, canna, t-code or no-fep
;; $BNc(B2: (setq od-fep-type "kanainput")	; its-mode name
;;
;; $B!&8+$?L\$O5$$K$7$J$$$+$iB.$/I=<($7$FM_$7$$!#(B
;; $B<!$N$$$:$l$+$r=q$$$F2<$5$$!#(B
;; (setq od-fill-function nil) ; fill $B$r9T$o$J$$!#7k9=B.$$!#(B
;; (setq od-fill-function 'od-fill-region) ; $B6XB'=hM}$J$7!#$$$/$i$+B.$$!#(B
;; $B$?$@$7!"(Bod-fill-function $B$r(B nil $B$K$9$k$H!"1QOB<-E5$NNcJ8I=<($N@ZBX(B
;; $B$(5!G=$,F/$+$J$/$J$j$^$9!#(B
;;
;; $B!&%G%U%)%k%H$G<-=q$r5/F0$9$k$H$-$K%U%l!<%`$r3+$/$h$&$K$7$?$$!#(B
;; $BJQ?t(B od-frame-geometry $B$K(B X $B$N(B geometory $B$r;XDj$7$F2<$5$$!#(B
;; M-x online-dictionary $B$G<-=qMQ$N%U%l!<%`$,3+$-$^$9!#(B
;; $BNc(B1: (setq od-frame-geometry "80x50+20-0")
;; $BNc(B2: (setq od-frame-geometry "80x50")
;; $BNc(B3: (setq od-frame-geometry "")
;;
;; $B!&<-=qMQ$N%U%l!<%`$N%U%)%s%H$r;XDj$7$?$$!#(B
;; $BJQ?t(B od-frame-font $B$K(B fontset $B$r;XDj$7$F2<$5$$!#%j%=!<%9$rE,Ev$K@_(B
;; $BDj$9$l$P!"IaCJ$O(B 12x24 $B$r;H$C$F!"<-=q$NI=<($N;~$K$OH/2;5-9f$K9g$o$;(B
;; $B$F(B ETL $B%U%)%s%H$r;H$&$H$$$C$?$3$H$b$G$-$^$9!#(B
;; $BNc(B: (setq od-frame-font "ETL24")
;;
;; $B!&(B"q" $B$G<-=q$r=*$k;~$K<-=qMQ$N%U%l!<%`$r1#$7$?$$!#(B
;; $BJQ?t(B od-hide-frame $B$K<!$N$$$:$l$+$r;XDj$7$F2<$5$$!#(B
;;	'iconify	--- $B%"%$%3%s2=$5$l$k!#(B
;;	'make-invisible	--- $B8+$($J$/$J$k!#(B
;;	'lower		--- $B2<$K1#$l$k!#(B
;;	$B$=$NB>(B nil $B0J30(B	--- $BA0$N%U%l!<%`$K0\$k!#(B
;; $B$h$j:Y$+$$@_Dj$r$7$?$$>l9g$O!"(Bod-quit-hook,
;; od-pre-pop-to-buffer-hook, od-post-pop-to-buffer-hook $B$J$I$r;H$C$F(B
;; $B2<$5$$!#(B
;;
;; $B!&(B"Q" $B$G<-=q$r=*$k$H$-$K<-=qMQ$N%U%l!<%`$r:o=|$7$?$$!#(B
;; (add-hook 'od-really-quit-hook 'od-delete-frame)
;;
;; $B!&(Bmenu-bar $B$+$i5/F0$7$?$$!#(B
;; (define-key-after menu-bar-file-menu [dictionary]
;;   '("Online Dictionary" . online-dictionary) 'calendar)
;; $B$H$9$k$H!"(Bmenu-bar $B$N(B File $B$H$$$&$H$3$m$+$i5/F0$G$-$^$9!#(B
;;
;; $B!&%+!<%=%k$N2<$NC18l$rD4$Y$?$$!#(B
;; (define-key help-map "o" 'od-lookup-pattern-edit)
;; (define-key menu-bar-help-menu [diclookup]
;;   '("Diclookup word" . od-lookup-pattern-edit))
;; (autoload 'od-lookup-pattern-edit "diclookup-mule" "Look up a word." t nil)
;; $B$H$9$k$H!"(BC-h o $B$d(B menu-bar Help -> Diclookup word $B$G%+!<%=%k$N2<$N(B
;; $BC18l$rD4$Y$^$9!#(B
;;
;; $B!&1QC18l$r4JC1$KD4$Y$i$l$k%b!<%I$,M_$7$$!#(B
;; lup-mode $B$H$$$&I{%b!<%I$,$"$j$^$9!#(B
;; (autoload 'lup-mode "diclookup-mule"
;;   "Minor mode for online-dictionary." t nil)
;; $B$H=q$/$H!"(BM-x lup-mode $B$G(B mouse-1 $B$r%@%V%k%/%j%C%/$7$F%^%&%9%+!<%=(B
;; $B%k$N<~JU$NC18l$rD4$Y$?$j!"(Bmouse-1 $B$r%I%i%C%0$7$F!";XDj$7$?HO0O$rD4(B
;; $B$Y$k$3$H$,$G$-$^$9!#$b$&0lEY(B M-x lup-mode $B$GH4$1$^$9!#(B
;;
;; $B!&$$$A$$$A(B lup-mode $B$KF~$k$N$OLLE]!#$^$?$O%^%&%9$K$h$k%+%C%H!u%Z!<(B
;;   $B%9%H$,;H$($J$/$J$k$N$O$$$d!#(B
;; Shift + mouse-1 $B$O%f!<%6!<$,;XDj$G$-$k$H$3$m$J$N$G!"$3$3$K;XDj$7$F(B
;; $B$7$^$($P$$$D$G$b(B Shift $B$r2!$7$J$,$i%@%V%k%/%j%C%/$7$?$j$9$k$3$H$G1Q(B
;; $BC18l$r0z$/$3$H$,$G$-$^$9!#(B
;; (global-set-key [S-down-mouse-1] 'lup-mouse-drag-region)
;; (global-set-key [S-mouse-1] 'mouse-set-point)
;; (global-set-key [S-drag-mouse-1] 'lup-mouse-set-region)
;; (global-set-key [S-double-mouse-1] 'lup-mouse-set-point)
;; (global-set-key [S-triple-mouse-1] 'lup-mouse-set-point)
;; (autoload 'lup-mouse-drag-region "diclookup-mule"
;;   "Lookup a double-clicked(or dragged) word." t nil)

;;; ChangeLog:

;; 92.11.18 modified for Mule Ver.0.9.7
;;			by M.Higashida <manabu@sigmath.osaka-u.ac.jp>
;;	$BK^Nc$d1|IU$rI=<($5$;$k$HJ8;z2=$1$9$k%P%0$N%U%#%/%9(B
;;	read-hiragana-string $B$K4X$9$kJQ99$bF~$C$F$$$^$9!#(B
;; 93.2.11  modified for Mule Ver.0.9.7.1 by T.Ito <toshi@his.cpl.melco.co.jp>
;;	In od:really-change-dictionary, avoid multiple connection.
;; 94.3.29 modified for Mule Ver.1.0 by M.Yamamoto <mituharu@is.s.u-tokyo.ac.jp>
;;     Rename the program diclookup-mule to avoid a conflict.
;;	Reference search and picture search are supported.
;; 94.5.20 modified by M.Yamamoto <mituharu@is.s.u-tokyo.ac.jp>
;;	Multiple pictures are supported.
;; 94.8.31 Release Version 2.0 (for Mule Ver.2.0).
;; 94.11.5 Release Version 2.1 (for Mule Ver.2.1).

;;; Code:

(require 'outline)

;; Please customize the following variables for your local site.

(defvar dserver-server-list '("dserver" "localhost")
  "*List of dservers.
Each element must be in the form of either

	\"hostname\"
or
	'(\"hostname\" . \"protocol\").

If an element is in the first form, the value of
`dserver-protocol' is used for dserver protocol name.")
(defvar dserver-protocol "ndtp" "*Dserver protocol name.")
(defvar od-chujiten-flag nil
  "*Non-nil means the gaiji-table for chujiten is used.")

;; These are user variables. Customize them in your startup file.

(defvar od-window-config 6
  "*Height of `*Dict-Header*' buffer.")
(defvar od-body-fillcolumn nil
  "*Non-nil value defines fill-column of `*Dict-Body*' buffer.")
(defvar od-select-unique-entry-immediately t
  "*Non-nil means the unique entry is displayed automatically.")
(defvar od-scroll-step 'full
  "*Define scroll step used by functions od-scroll-entry-*.
See also the documentation of odic-mode.")
(defvar od-eiwa-summary-mode nil
  "*Non-nil means example sentences are hidden in eiwa dictionary.")
(defvar od-default-jisyo "eiwa"
  "*Define the dictionary selected when invoking online-dictionary.")
(defvar od-fep-type (cond ((boundp 'EGG) 'egg)
			  ((boundp 'CANNA) 'canna)
			  (t 'no-fep))
  "*Define how to input a pattern in Japanese.
This value must be either an fep symbol (egg, canna, t-code, or no-fep)
or an its-mode name string.")
(defvar od-ispell-ver-3 t
  "*Define whether you use the ispell3 or not.")
(defvar od-show-help-automatically nil
  "*Non-nil means show help messages when invoking online-dictionary.")
(defvar od-tcode-toggle-events '(?\034)
  "*List of events used for toggling tcode mode.")
(defvar od-fill-function 'fill-region
  "*Function used for filling text.")
(defvar od-history-length 16
  "*Length of history.")
(defvar od-use-only-ascii-and-jisx0208 (not window-system)
  "*Non-nil means only ASCII and JISX0208 are used.")
(defvar od-view-picture-program-and-args '("xv" "-")
  "*List of a program and arguments for displaying picture data.
Picture data are piped to this program in PBM format.")
(defvar od-frame-name "Online-Dictionary"
  "*Name of dictionary frame.")
(defvar od-frame-geometry nil
  "*Define the geometry of the frame for online-dictionary.
This value must be nil or a standard X windows geometry
string. Nil means that the current frame is used for
online-dictionary. If this value is \"\", the geometry is
left to your window manager.")
(defvar od-frame-font nil
  "*Font used in frames for online-dictionary")
(defvar od-frame-alist nil
  "*Alist of frame parameters used when creating frames for dictionary.")
(defvar od-hide-frame nil
  "*Non-nil means the selected frame used for dictionary is hidden when quitting.
Non-nil value defines how to hide the frame. 

	'iconify	--- iconify the frame
	'make-invisible	--- make the frame invisible
	'lower		--- lower the frame")
(defvar od-default-jisyo-japanese "waei"
  "*Dictionary used when od-lookup-pattern-edit reads japanese.")
(defvar od-quit-hook
  (and od-hide-frame
       (or (cdr (assq od-hide-frame
		      '((iconify . od-iconify)
			(make-invisible . od-make-invisible)
			(lower . od-lower-and-select-previous-frame))))
	   'od-select-and-raise-previous-frame)))
(defvar od-really-quit-hook nil)
(defvar od-pre-pop-to-buffer-hook
  (cdr (assq od-hide-frame
	     '((lower . od-raise-dictionary-frames)
	       (make-invisible . od-make-dictionary-frames-visible)))))
(defvar od-post-pop-to-buffer-hook nil)
(defvar od-dict-filter-func
  '(("eiwa" . (od-filter-buffer))
    ("waei" . (od-filter-buffer))
    ("kojien" . (od-filter-buffer))))
(defvar od-dict-filter
  '(("eiwa" . (("\n" . "\n\n")
	       ("\\([.!?]\\) \\([1-9][0-9]?a?[[ (]\\)" . "\\1\n\n    \\2")
	       ("\\([.!?]\\) \\(([1-9][0-9]?)\\)" . "\\1\n    \\2")
	       ("\\([.!?]\\) \\([b-z][[ (]\\)" . "\\1\n     \\2")))
    ("waei" . (("/ " . "\n    / ")
	       ("$B"y(B" . "\n    $B"y(B")
	       ("$B!%(B/" . "$B!%(B\n    /")))))

;; Set up other variables and macros.

(defconst od-version "2.1.1"
  "version of current diclookup-mule.el")
(defun od-version () (interactive)
  (message (concat "diclookup-mule.el Version " od-version)))
(defvar dserver-current-server nil)
(defmacro od-server-name (server)
  (` (if (consp (, server)) (car (, server)) (, server))))
(defmacro od-server-protocol (server)
  (` (if (consp (, server)) (cdr (, server)) dserver-protocol)))
(defvar od-dict-list nil)
(defvar dserver-buffer " *ndtp*" "Buffer associated with dserver process.")
(defconst od-dict-header-buffer-name
  "*Dict-Header*" "Buffer name for Dict Header.")
(defvar od-dict-header-buffer nil "Buffer for Dict Header.")
(defconst od-dict-body-buffer-name
  "*Dict-Body*" "Buffer name for Dict Body.")
(defvar od-dict-body-buffer nil "Buffer for Dict Body.")
(defconst od-dict-select-buffer-name
  "*Dict-Select*" "Buffer name for Dict Selection.")
(defvar od-dict-select-buffer nil "Buffer for Dict Selection.")
(defvar od-previous-config nil)
(defvar od-dict nil)
(defvar od-headers)
(defvar od-references nil)
(defvar od-current-entry -1)
(defvar od-history-alist nil)
(defvar od-history (make-vector od-history-length nil))
(defvar od-history-current-point 0)
(defvar od-history-drifting-point 0)
(defvar od-current-entry-in-body-flag nil)
(defvar od-input-by-kanji nil)
(defvar od-current-jisyo-name nil)
(defvar od-current-jisyo-pretty-name "")
(defvar od-current-jisyo-ccl)
(defvar od-index-types)
(defvar od-eiwa-current-display-mode "")
(defvar od-ccl-status (make-vector 9 nil))
(defvar od-current-dict-filter nil)
(defvar od-current-dict-filter-func nil)
(defvar od-overlay-list nil)

(defconst od-jisyo-table
  '(("eiwa"	"$B?71QOBCf<-E5(B"	nil	od-ccl-eiwa)
    ("waei"	"$B?7OB1QCf<-E5(B"	t	od-ccl-eiwa)
    ("kojien"	"$B9-<-1q(B"	t	od-ccl-kojien)))
(defconst od-index-type-table
  '(("OK" ?O "$BCx:n8"I=<((B         ... \"\\[od-show-okuduke]\" $B$GI=<($9$k(B."
     od-show-okuduke)
    ("HA" ?H "$B%a%K%e!<8!:w(B       ... \"\\[od-show-hanrei]\" $B$H(B \
\"\\[od-make-entry-table]\" $B$G8!:w$G$-$k(B."
     od-show-hanrei)
    ("BA" ?A "$B%"%k%U%!%Y%C%H8eJ}(B ... $B8lF,$K(B * $B$rIU$1$k$3$H$,$G$-$k(B."
     od-lookup-pattern)
    ("BK" ?K "$B$+$J8eJ}(B           ... $B8lF,$K(B * $B$rIU$1$k$3$H$,$G$-$k(B."
     od-lookup-pattern)
    ("IA" ?a "$B%"%k%U%!%Y%C%HA0J}(B ... $B8lHx$K(B * $B$rIU$1$k$3$H$,$G$-$k(B."
     od-lookup-pattern)
    ("IK" ?k "$B$+$JA0J}(B           ... $B8lHx$K(B * $B$rIU$1$k$3$H$,$G$-$k(B."
     od-lookup-pattern)))


;;; OD-Log
(defvar od-log-buffer "*OD-Log*");; "For debuging only"
(defvar od-debug-flag nil)

;;;
(defun od-japanese-p (word)
  (string-match "\\cj" word))

;;; LOG
(defun od-quote-null (str)
  (mapconcat (function (lambda (elt) (if (= elt 0) "^@"
				       (char-to-string elt))))
	     str ""))

(defun od-log (debug &rest args)
  (and debug
       (save-excursion
	 (set-buffer (get-buffer-create od-log-buffer))
	 (goto-char (point-max))
	 (insert (apply (function format) args)))))

(defun od-filter-buffer ()
  (let ((tmp od-current-dict-filter))
    (while tmp
      (goto-char 1)
      (while (re-search-forward (car (car tmp)) nil t nil)
	(replace-match (cdr (car tmp)) t nil))
      (setq tmp (cdr tmp)))))


(defvar odic-lookup-map nil "")
(if odic-lookup-map
    ()
  (setq odic-lookup-map (make-sparse-keymap "Lookup"))
  (define-key odic-lookup-map [show-okuduke]
    '("Show Copyright" . od-show-okuduke))
  (define-key odic-lookup-map [show-hanrei]
    '("Menu Search" . od-show-hanrei))
  (define-key odic-lookup-map [make-entry-table]
    '("Make Entry Table" . od-make-entry-table))
  (define-key odic-lookup-map [lookup-pattern-with-ispell]
    '("Lookup Pattern with Ispell" . od-lookup-pattern-with-ispell))
  (define-key odic-lookup-map [lookup-pattern]
    '("Lookup Pattern" . od-lookup-pattern)))

(put 'od-show-okuduke 'menu-enable '(memq ?O od-index-types))
(put 'od-show-hanrei 'menu-enable '(memq ?H od-index-types))
(put 'od-make-entry-table 'menu-enable 'od-references)
(let ((lookup-pattern-enable
       '(or (memq ?a od-index-types) (memq ?A od-index-types)
	    (memq ?k od-index-types) (memq ?K od-index-types))))
  (put 'od-lookup-pattern-with-ispell 'menu-enable
       lookup-pattern-enable)
  (put 'od-lookup-pattern 'menu-enable
       lookup-pattern-enable))

(defvar odic-dict-map nil "")
(if odic-dict-map
    ()
  (setq odic-dict-map (make-sparse-keymap "Dict"))
  (define-key odic-dict-map [really-quit]
    '("Quit and Disconnect" . od-really-quit))
  (define-key odic-dict-map [quit]
    '("Quit" . od-quit))
  (define-key odic-dict-map [toggle-character-sets]
    '("Toggle Character Sets" . od-toggle-character-sets))
  (define-key odic-dict-map [toggle-eiwa-summary-mode]
    '("Toggle Summary" . od-toggle-eiwa-summary-mode))
  (define-key odic-dict-map [down-history]
    '("Down History" . od-down-history))
  (define-key odic-dict-map [up-history]
    '("Up History" . od-up-history))
  (let ((dics (reverse od-jisyo-table))
	(change-dictionary-map '("Select Dictionary")))
    (while dics
      (setq change-dictionary-map
	    (cons (cons (car (car dics))
			(cons (car (car dics))
			      (cons '(nil)
				    'od-menu-bar-change-dictionary)))
		  change-dictionary-map))
      (setq dics (cdr dics)))
    (define-key odic-dict-map [change-dictionary]
      (cons "Change Dictionary"
	    (cons 'keymap change-dictionary-map)))))

(put 'od-up-history 'menu-enable
     '(let ((previous-point
	     (% (+ od-history-drifting-point od-history-length -1)
		od-history-length)))
	(and (/= od-history-current-point previous-point)
	     (aref od-history previous-point))))
(put 'od-down-history 'menu-enable
     '(let ((next-point
	     (% (1+ od-history-drifting-point) od-history-length)))
	(and (/= od-history-current-point od-history-drifting-point)
	     (aref od-history next-point))))

(put 'od-toggle-eiwa-summary-mode 'menu-enable
     '(string= od-current-jisyo-name "eiwa"))

(defvar odic-fundamental-map nil)
(if odic-fundamental-map
    nil
  (setq odic-fundamental-map (make-sparse-keymap))
  (define-key odic-fundamental-map "f" 'od-lookup-pattern)
  (define-key odic-fundamental-map "i" 'od-lookup-pattern-with-ispell)
  (define-key odic-fundamental-map "o" 'other-window)
  (define-key odic-fundamental-map "h" 'backward-char)
  (define-key odic-fundamental-map "j" 'next-line)
  (define-key odic-fundamental-map "k" 'previous-line)
  (define-key odic-fundamental-map "l" 'forward-char)
  (define-key odic-fundamental-map "w" 'forward-word)
  (define-key odic-fundamental-map "b" 'backward-word)
  (define-key odic-fundamental-map "e" 'od-lookup-pattern-edit)
  (define-key odic-fundamental-map "q" 'od-quit)
  (define-key odic-fundamental-map "Q" 'od-really-quit)
  (define-key odic-fundamental-map "v" 'od-version)
  (define-key odic-fundamental-map "C" 'od-change-dictionary)
  (define-key odic-fundamental-map "E" 'od-toggle-eiwa-summary-mode)
  (define-key odic-fundamental-map "S" 'od-toggle-character-sets)
  (define-key odic-fundamental-map "H" 'od-show-hanrei)
  (define-key odic-fundamental-map "m" 'od-make-entry-table)
  (define-key odic-fundamental-map "O" 'od-show-okuduke)
  (define-key odic-fundamental-map "<" 'beginning-of-buffer)
  (define-key odic-fundamental-map ">" 'end-of-buffer)
  (define-key odic-fundamental-map [down-mouse-2] nil)
  (define-key odic-fundamental-map [menu-bar dict]
    (cons "Dict" odic-dict-map))
  (define-key odic-fundamental-map [menu-bar lookup]
    (cons "Lookup" odic-lookup-map)))

(defvar odic-mode-syntax-table nil
  "Syntax table used while in odic mode.")

(defvar odic-mode-abbrev-table nil
  "Abbrev table used while in odic mode.")
(define-abbrev-table 'odic-mode-abbrev-table ())

(if odic-mode-syntax-table
    ()
  (setq odic-mode-syntax-table (make-syntax-table)))

(defvar odic-mode-map nil "")
(if odic-mode-map
    ()
  (setq odic-mode-map (cons 'keymap odic-fundamental-map))
  (define-key odic-mode-map "." 'od-show-current-entry)
  (define-key odic-mode-map " " 'od-scroll-entry-up)
  (define-key odic-mode-map "\177" 'od-scroll-entry-down)
  (define-key odic-mode-map "d" 'od-scroll-entry-up-half)
  (define-key odic-mode-map "u" 'od-scroll-entry-down-half)
  (define-key odic-mode-map "\C-m" 'od-scroll-entry-up-line)
  (define-key odic-mode-map [return] 'od-scroll-entry-up-line)
  (define-key odic-mode-map "\ep" 'od-up-history)
  (define-key odic-mode-map "\en" 'od-down-history)
  (define-key odic-mode-map "p" 'od-previous-entry)
  (define-key odic-mode-map "n" 'od-next-entry)
  (define-key odic-mode-map "s" 'od-select-entry)
  (define-key odic-mode-map "1" 'od-direct-select-entry)
  (define-key odic-mode-map "2" 'od-direct-select-entry)
  (define-key odic-mode-map "3" 'od-direct-select-entry)
  (define-key odic-mode-map "4" 'od-direct-select-entry)
  (define-key odic-mode-map "5" 'od-direct-select-entry)
  (define-key odic-mode-map "6" 'od-direct-select-entry)
  (define-key odic-mode-map "7" 'od-direct-select-entry)
  (define-key odic-mode-map "8" 'od-direct-select-entry)
  (define-key odic-mode-map "9" 'od-direct-select-entry)
  (define-key odic-mode-map [mouse-2] 'od-mouse-select-entry)
  (define-key odic-mode-map "?" 'describe-mode))

(defun od-set-mode-line-format ()
  (setq mode-line-buffer-identification
	'("Dict: " od-current-jisyo-pretty-name
	  od-eiwa-current-display-mode)))

(defun odic-mode ()
  "Major mode for editing odic intended for humans to read.

Commands:($B6&DL(B)\\<odic-fundamental-map>
\\[od-lookup-pattern]\
	$B8!:w$r9T$J$$$^$9(B.
	$B%Q%?!<%s$OH>3Q%"%k%U%!%Y%C%H(B($B%"%k%U%!%Y%C%H8!:w(B),
	$BJ?2>L>(B, $BJR2>L>(B($B$+$J8!:w(B) $B$GF~NO$7$^$9(B.
	$B%Q%?!<%s$O8lF,$^$?$O8lHx$K(B1$B$D$N(B \"*\" $B$rIU$1$k$3$H$,$G$-$^$9(B.
	$B$=$N:](B, \"*\" $B$O(B0$BJ8;z0J>e$NG$0U$NJ8;zNs$r0UL#$7$^$9(B.
	$B%Q%?!<%s$OG$0U$N0LCV$K(B \"?\" $B$r$$$/$D$G$b4^$`$3$H$,$G$-$^$9(B.
	$B$=$N:](B, \"?\" $B$OG$0U$N(B1$BJ8;z$r0UL#$7$^$9(B.
\\[od-lookup-pattern-with-ispell]\
	ispell$B$rMQ$$$F$"$d$U$d$JDV$+$iF~NO%Q%?!<%s$rA*$S$^$9(B. 
	($B1QOB<-=q$N;~JXMx(B)
\\[od-quit]\
	$B%*%s%i%$%s<-=q5/F0A0$N%&%$%s%I%&%3%s%U%#%0%l!<%7%g%s$KLa$j$^$9(B.
\\[od-really-quit]\
	$B%*%s%i%$%s<-=q$N;}$D%P%C%U%!$rA4$F:o=|$7(B,
	$B%*%s%i%$%s<-=q5/F0A0$N%&%$%s%I%&%3%s%U%#%0%l!<%7%g%s$KLa$j$^$9(B.
\\[od-change-dictionary]\
	$B<-=q$r@ZBX$($^$9(B.
	$B8=:_(B eiwa($B1QOB<-E5(B), waei($BOB1Q<-E5(B), kojien($B9-<-1q(B) $B$,;H$($^$9(B.
	$B%3%s%W%j!<%7%g%sF~NO$,2DG=$G$9(B.
\\[od-toggle-eiwa-summary-mode]\
	$B1QOB<-E5$G$NNcJ8$NI=<($N(B ON/OFF $B$r@ZBX$($^$9(B.
\\[od-toggle-character-sets]\
	$BK\J8$NI=<($KMQ$$$i$l$kJ8;z=89g$N=8$^$j(B($BC<KvMQ$H(B X $BMQ(B)$B$r@ZBX$($^$9(B.
\\[od-show-hanrei]\
	$B$b$7$G$-$l$P!"%a%K%e!<8!:w$r9T$$$^$9(B.
\\[od-make-entry-table]\
	$BK\J8Cf$KB>$NItJ,$r;2>H$7$F$$$k$H$3$m(B($B2<@~It(B)$B$,$"$l$P!"(B
	$B$=$N;2>H$7$F$$$kItJ,$r=8$a$F?7$7$$8+=P$7$H$7$^$9(B.
\\[od-show-okuduke]\
	$B$b$7$"$l$P(B, $B1|IU$1$rI=<($7$^$9(B.

Commands:($B8uJd$rI=<($7$F$$$k%P%C%U%!(B)\\<odic-mode-map>
\\[od-show-current-entry]\
	$B8=:_$N8uJd$N@bL@J8$rI=<($7$^$9(B.
\\[od-previous-entry]\
	$BD>A0$N8uJd$N@bL@J8$rI=<($7$^$9(B.
\\[od-next-entry]\
	$BD>8e$N8uJd$N@bL@J8$rI=<($7$^$9(B.
\\[od-select-entry]\
	$B8uJd$rHV9f$GA*$S(B, $B$=$N@bL@J8$rI=<($7$^$9(B.
\\[od-mouse-select-entry]\
	$B%^%&%9%+!<%=%k$N2<$K$"$k8uJd$N@bL@J8$rI=<($7$^$9(B.
	(X $B$G;HMQ$7$F$$$k$H$-$N$_(B)
1-9	1-9$BHV$N8uJd$rD>@\A*$S(B, $B$=$N@bL@J8$rI=<($7$^$9(B.
	10$BHV0J9_$O(B \"\\[od-select-entry]\" $B$^$?$O(B \"\\[od-next-entry]\" $B%3%^%s%I$r;H$C$F2<$5$$(B.
\\[od-scroll-entry-up]\
	$B@bL@J8$r(B od-scroll-step $B$G<($5$l$?CM$@$1>e$K%9%/%m!<%k$7$^$9(B.
\\[od-scroll-entry-down]\
	$B@bL@J8$r(B od-scroll-step $B$G<($5$l$?CM$@$12<$K%9%/%m!<%k$7$^$9(B.
\\[od-scroll-entry-up-half]
	$B@bL@J8$r(B od-scroll-step $B$G<($5$l$?CM$NH>J,$@$1>e$K%9%/%m!<%k$7$^$9(B.
\\[od-scroll-entry-down-half]\
	$B@bL@J8$r(B od-scroll-step $B$G<($5$l$?CM$NH>J,$@$12<$K%9%/%m!<%k$7$^$9(B.
\\[od-scroll-entry-up-line]\
	$B@bL@J8$r0l9T$@$1>e$K%9%/%m!<%k$7$^$9(B.
	$B$3$l$i$N%3%^%s%I$O@bL@J8$,I=<($5$l$F$$$J$1$l$P(B \"\\[od-show-current-entry]\" $B$HF1$8F0:n$r$7$^$9(B.
	od-scroll-step $B$K$O(B, $B@0?t(B, half, full $B$N$$$:$l$+$,;XDj$G$-$^$9(B.
	$B%G%U%)%k%H$O(B full $B$G$9(B.
\\[od-previous-entry]\
	$B%R%9%H%j!<$rAL$j$^$9(B.
\\[od-next-entry]\
	$BAL$C$?%R%9%H%j!<$rLa$j$^$9(B.

Commands:($BK\J8$rI=<($7$F$$$k%P%C%U%!(B)\\<odic-body-mode-map>
\\[od-mouse-lookup-reference]\
	$B%^%&%9%+!<%=%k$N2<$KB>$NItJ,$r;2>H$7$F$$$k$H$3$m(B($B2<@~It(B)$B$,$"$k>l9g(B,
	$B$=$N@bL@J8$rI=<($7$^$9(B.(X $B$G;HMQ$7$F$$$k$H$-$N$_(B)
\\[od-lookup-reference]\
	$B%+!<%=%k$N2<$KB>$NItJ,$r;2>H$7$F$$$k$H$3$m(B($B2<@~It(B)$B$,$"$k>l9g(B,
	$B$=$N@bL@J8$rI=<($7$^$9(B.

Special commands:\\{odic-mode-map}
Turning on odic-mode calls the value of the variable odic-mode-hook,
if that value is non-nil."
  (interactive)
  (kill-all-local-variables)
  (use-local-map odic-mode-map)
  (setq mode-name "Dict")
  (setq major-mode 'odic-mode)
  (setq local-abbrev-table odic-mode-abbrev-table)
  (set-syntax-table odic-mode-syntax-table)
  (setq buffer-read-only t)
  (make-local-variable 'goal-column)
  (setq goal-column 4)
  (od-set-mode-line-format)
  (run-hooks 'odic-mode-hook))

(defvar odic-body-mode-syntax-table nil
  "Syntax table used while in odic body mode.")

(defvar odic-body-mode-abbrev-table nil
  "Abbrev table used while in odic body mode.")
(define-abbrev-table 'odic-body-mode-abbrev-table ())

(if odic-body-mode-syntax-table
    ()
  (setq odic-body-mode-syntax-table (make-syntax-table)))

(defvar odic-body-mode-map nil "")
(if odic-body-mode-map
    ()
  (setq odic-body-mode-map (cons 'keymap odic-fundamental-map))
  (define-key odic-body-mode-map "p" 'previous-line)
  (define-key odic-body-mode-map "n" 'next-line)
  (define-key odic-body-mode-map " " 'scroll-up)
  (define-key odic-body-mode-map "\177" 'scroll-down)
  (define-key odic-body-mode-map "0" 'beginning-of-line)
  (define-key odic-body-mode-map "$" 'end-of-line)
  (define-key odic-body-mode-map [mouse-2] 'od-mouse-lookup-reference)
  (define-key odic-body-mode-map "\C-m" 'od-lookup-reference)
  (define-key odic-body-mode-map [return] 'od-lookup-reference))

(defun odic-body-mode ()
  "Major mode for editing odic intended for humans to read.
Special commands:\\{odic-body-mode-map}
Turning on odic-body-mode calls the value of the variable odic-body-mode-hook,
if that value is non-nil.
See the documentation of `odic-mode' for more information on this mode."
  (interactive)
  (kill-all-local-variables)
  (use-local-map odic-body-mode-map)
  (setq mode-name "DictBody")
  (setq major-mode 'odic-body-mode)
  (setq local-abbrev-table odic-body-mode-abbrev-table)
  (set-syntax-table odic-body-mode-syntax-table)
  (setq buffer-read-only t)
  (make-local-variable 'fill-column)
  (setq fill-column (or od-body-fillcolumn
			(let ((fillcol (- (window-width) 10)))
			  (if (< 0 fillcol)
			      fillcol))
			fill-column))
  (od-set-mode-line-format)
  (make-local-variable 'outline-regexp)
  (setq outline-regexp " +")
  (make-local-variable 'outline-heading-end-regexp)
  (setq outline-heading-end-regexp "[$B!'(B\^M]")
  (outline-minor-mode 1)
  (run-hooks 'odic-body-mode-hook))

(defvar odic-select-mode-syntax-table nil
  "Syntax table used while in odic select mode.")

(defvar odic-select-mode-abbrev-table nil
  "Abbrev table used while in odic select mode.")
(define-abbrev-table 'odic-select-mode-abbrev-table ())

(if odic-select-mode-syntax-table
    ()
  (setq odic-select-mode-syntax-table (make-syntax-table)))

(defvar odic-select-mode-map nil "")
(if odic-select-mode-map
    ()
  (setq odic-select-mode-map (cons 'keymap odic-fundamental-map))
  (define-key odic-select-mode-map [mouse-2] 'od-mouse-choose-candidate)
  (define-key odic-select-mode-map "\C-m" 'od-choose-candidate)
  (define-key odic-select-mode-map [return] 'od-choose-candidate))

(defun odic-select-mode ()
  "Major mode for selecting candidate.
Special commands:\\{odic-select-mode-map}
Turning on odic-select-mode calls the value of the variable
odic-select-mode-hook, if that value is non-nil.
See the documentation of `odic-mode' for more information on this mode."
  (interactive)
  (kill-all-local-variables)
  (use-local-map odic-select-mode-map)
  (setq mode-name "DictSelect")
  (setq major-mode 'odic-select-mode)
  (setq local-abbrev-table odic-select-mode-abbrev-table)
  (set-syntax-table odic-select-mode-syntax-table)
  (setq buffer-read-only t)
  (run-hooks 'odic-select-mode-hook))


(defun od-open-dictionary ()
  (if (not (eq (process-status od-dict) 'open))
      (od-really-change-dictionary od-current-jisyo-name)))

(defun od-connect-to-servers ()
  (let ((tmp dserver-server-list)
	new-dict-list dict)
    (while tmp
      (let ((dserver-server (car tmp)))
	(setq dict (car od-dict-list))
	(cond ((and od-dict-list
		    (or (not dict)
			(eq (process-status dict) 'open))))
	      ((progn
		 (message "Connecting to %s server on %s..."
			  (od-server-protocol dserver-server)
			  (od-server-name dserver-server))
		 (condition-case err
		     (setq dict
			   (open-network-stream
			    "*ndtp*" dserver-buffer
			    (od-server-name dserver-server)
			    (od-server-protocol dserver-server)))
		   (file-error (setq dict nil))))
	       (process-kill-without-query dict)
	       (set-process-coding-system dict *internal* *euc-japan*)
	       (set-process-sentinel dict 'od-watchdog)
	       (set-buffer dserver-buffer)
	       (erase-buffer)
	       (process-send-string dict
				    (format "A%s@%s\n" (getenv "USER")
					    (system-name)))
	       (od-wait-until '("$A\n" "$?\n" "$N\n" "$<\n") dict)))
	(message "")
	(setq od-dict-list (cdr od-dict-list))
	(setq new-dict-list (cons dict new-dict-list))
	(if (equal (od-server-name dserver-current-server) dserver-server)
	    (setq od-dict dict)))
      (setq tmp (cdr tmp)))
    (setq od-dict-list (nreverse new-dict-list))))

(defun od-wait-until (list proc)
  (catch 'od-wait-until
    (while t
      (let* ((str (buffer-string))
	     (len (length str))
	     (tmp list))
	(while tmp
	  (let* ((item (car tmp))
		 (ilen (length item)))
	    (cond ((< len ilen) nil)
		  ((string= item (substring str (- len ilen)))
		   (throw 'od-wait-until item))))
	  (setq tmp (cdr tmp)))
	(accept-process-output proc)))))

(defun od-setup-windows ()
  (delete-other-windows)
  (split-window (selected-window) od-window-config)
  (if (not (eq (window-buffer) od-dict-header-buffer))
      (switch-to-buffer od-dict-header-buffer-name))
  (other-window 1)
  (switch-to-buffer od-dict-body-buffer-name)
  (other-window 1))

(defun od-kill-active-buffers ()
  (kill-buffer od-dict-header-buffer-name)
  (kill-buffer od-dict-body-buffer-name)
  (kill-buffer od-dict-select-buffer-name)
  (kill-buffer dserver-buffer)
  (if (get-buffer " *od-temp*") (kill-buffer " *od-temp*")))

(defun od-set-header ()
  (od-setup-windows)
  (set-buffer od-dict-header-buffer-name))

(defun od-set-body ()
  (od-setup-windows)
  (set-buffer od-dict-body-buffer-name))

(defun od-set-select ()
  (od-setup-windows)
  (set-buffer od-dict-select-buffer-name))

(defun od-watchdog (process event)
  (if (string= "finished\n" event)
      (setq od-dict nil)))

;;;###autoload
(defun online-dictionary ()
  (interactive)
  (if (and (eq window-system 'x) od-frame-geometry)
      (diclookup-other-frame)
    (diclookup)))

;;;###autoload
(defun diclookup ()
  (interactive)
  (let ((is-first (not od-dict)))
    (if (or is-first (not dserver-current-server))
	(setq dserver-current-server (car dserver-server-list)))
    (if (not (or (od-dictionary-frame-p)
		 (equal major-mode 'odic-mode)
		 (equal major-mode 'odic-body-mode)))
	(setq od-previous-config (current-window-configuration)))
    (make-face 'od-bold-inverse)
    (copy-face 'bold 'od-bold-inverse)
    (invert-face 'od-bold-inverse)
    (get-buffer-create dserver-buffer)
    (get-buffer-create " *od-temp*")
    (setq od-dict-header-buffer
	  (get-buffer-create od-dict-header-buffer-name))
    (setq od-dict-body-buffer
	  (get-buffer-create od-dict-body-buffer-name))
    (setq od-dict-select-buffer
	  (get-buffer-create od-dict-select-buffer-name))
    (od-set-header)
    (odic-mode)
    (od-set-body)
    (odic-body-mode)
    (od-set-select)
    (odic-select-mode)
    (od-set-header)
    (od-connect-to-servers)
    (if is-first
	(if od-default-jisyo
	    (od-really-change-dictionary od-default-jisyo)
	  (delete-other-windows)
	  (od-show-title)
	  (od-change-dictionary)))
    (if od-show-help-automatically
	(describe-mode))))

(defun od-make-dictionary-frames-visible ()
  (let ((frames (frame-list)))
    (while frames
      (if (od-dictionary-frame-p (car frames))
	  (make-frame-visible (car frames)))
      (setq frames (cdr frames)))))

(defun od-raise-dictionary-frames ()
  (let ((frames (frame-list)))
    (while frames
      (if (od-dictionary-frame-p (car frames))
	  (raise-frame (car frames)))
      (setq frames (cdr frames)))))

;;;###autoload
(defun diclookup-other-frame ()
  "Like `diclookup' command, but display diclookup buffer in another frame."
  (interactive)
  (run-hooks 'od-pre-pop-to-buffer-hook)
  (let* ((pop-up-frames t)
	 (frame-geometry (or od-frame-geometry ""))
	 (pop-up-frame-alist
	  (append
	   (list (cons 'name od-frame-name))
	   (and od-frame-font
		(list (cons 'font od-frame-font)))
	   (x-parse-geometry frame-geometry)
	   od-frame-alist))
	 (special-display-frame-alist pop-up-frame-alist))
    (pop-to-buffer od-dict-header-buffer-name))
  (run-hooks 'od-post-pop-to-buffer-hook)
  (diclookup))

(defun od-show-title ()
  (let (buffer-read-only)
    (erase-buffer)
    (insert "

		     $BEE;R%V%C%/HG<-=q8!:w%7%9%F%`(B

		      $B4dGH=qE9(B $B9-<-1q(B($BBh;0!&;MHG(B)
	   $B8&5f<R(B $B?71QOBCf<-E5(B($BBh(B5$BHG(B)$B!&?7OB1QCf<-E5(B($BBh(B3$BHG(B)

		 Copyright (C) Shingo NISHIOKA, 1991
		    nishioka@sanken.osaka-u.ac.jp

		 Modified by SAITO Yutaka, 1993
		    yutaka@sys1.cpg.sony.co.jp

		 Modified by YAMAMOTO Mitsuharu, 1994
		    mituharu@is.s.u-tokyo.ac.jp

* $B!V9-<-1q!W$O3t<02q<R4dGH=qE9$NEPO?>&I8$G$9(B.
"))
  (goto-char 1))

(defun od-menu-bar-change-dictionary ()
  (interactive)
  (od-really-change-dictionary last-command-event))

(defun od-change-dictionary ()
  (interactive)
  (cond ((eql od-fep-type 'canna)
	 (if canna:*japanese-mode* (canna-toggle-japanese-mode))))
  (let ((new-dic (completing-read "Select jisyo: " od-jisyo-table nil t)))
    (od-really-change-dictionary new-dic)))

(defun od-really-change-dictionary (new-dic)
  (od-connect-to-servers)
  (set-buffer dserver-buffer)
  (let ((dict-list od-dict-list)
	(dserver-list dserver-server-list)
	new-dict dict)
    (while (and dict-list (null new-dict))
      (setq dict (car dict-list))
      (if (not dict)
	  nil
	(erase-buffer)
	(process-send-string dict (format "L%s\n" new-dic))
	(let ((ans (od-wait-until '("$*\n" "$&\n" "$?\n" "$N\n" "$<\n") dict)))
	  (if (string= ans "$*\n")
	      (setq new-dict dict
		    dserver-current-server (car dserver-list)))))
      (setq dict-list (cdr dict-list))
      (setq dserver-list (cdr dserver-list)))
    (if (null new-dict)
	(error "Jisyo set failed")
      (setq od-dict new-dict)))
  (if (string= od-current-jisyo-name new-dic)
      nil
    (setq od-headers nil)
    (setq od-references nil)
    (od-reset-history od-current-jisyo-name new-dic)
    (od-set-body)
    (let ((buffer-read-only nil))
      (erase-buffer))
    (setq od-current-entry-in-body-flag nil))
  (setq od-current-jisyo-name new-dic)
  (setq od-current-jisyo-pretty-name
	(nth 1 (assoc new-dic od-jisyo-table)))
  (setq od-input-by-kanji
	(nth 2 (assoc new-dic od-jisyo-table)))
  (setq od-current-jisyo-ccl
	(symbol-value (nth 3 (assoc new-dic od-jisyo-table))))
  (setq od-current-dict-filter
	(cdr (assoc new-dic od-dict-filter)))
  (setq od-current-dict-filter-func
	(cdr (assoc new-dic od-dict-filter-func)))
  (message "$B<-=q$,!V(B%s(%s)$B!W$K%;%C%H$5$l$^$7$?(B."
	   od-current-jisyo-pretty-name
	   (od-server-name dserver-current-server))
  (setq od-eiwa-current-display-mode
	(if (and (string= od-current-jisyo-name "eiwa")
		 od-eiwa-summary-mode)
	    "[Summary]"
	  ""))
  (od-set-header)
  (delete-other-windows)
  (if (not od-headers)
      (od-show-title)
    (od-show-headers)
    (setq od-current-entry -1))
  (od-change-index-type))

(defun od-change-index-type ()
  (od-open-dictionary)
  (set-buffer dserver-buffer)
  (erase-buffer)
  (process-send-string od-dict "I\n")
  (let ((ans (od-wait-until '("$$\n" "$?\n" "$N\n" "$<\n") od-dict)))
    (cond ((string= "$$\n" ans)
	   (let ((tmp od-index-type-table)
		 lst doc str)
	     (while tmp
	       (goto-char 1)
	       (if (re-search-forward (car (car tmp)) nil t)
		   (progn
		     (if (car (cdr (car tmp)))
			 (setq lst (cons (car (cdr (car tmp))) lst)))
		     (save-excursion
		       (set-buffer od-dict-header-buffer)
		       (setq str (substitute-command-keys (nth 2 (car tmp)))))
		     (put-text-property
		      0 1 'od-func (nth 3 (car tmp)) str)
		     (setq doc (cons str doc))))
	       (setq tmp (cdr tmp)))
	     (setq od-index-types lst)
	     (switch-to-buffer-other-window od-dict-select-buffer)
	     (let (buffer-read-only)
	       (erase-buffer)
	       (if window-system
		   (insert (substitute-command-keys
			    "Click \\[od-mouse-choose-candidate] \
on a method to select it.\n")))
	       (insert (substitute-command-keys
			"In this buffer, type \\[od-choose-candidate] \
to select the method near point.\n\n"))
	       (insert (format "\n$B$3$N<-=q$O0J2<$N(B%d$B<oN`$N8!:w$,2DG=$G$9(B.\n\n"
			       (length doc)))
	       (while doc
		 (put-text-property
		  (point) (progn (insert (car doc) ?\n) (1- (point)))
		  'mouse-face 'highlight)
		 (setq doc (cdr doc))))
	     (other-window 1))))))

;;;
;;; modified by yutaka@sys1.cpg.sony.co.jp
;;;
(defun od-lookup-pattern ()
  (interactive)
  (let ((pat (od-read-string-with-fep (if od-input-by-kanji
					  "Pattern(in kana): "
					"Pattern: ")
				      nil)))
    (od-really-lookup-pattern pat)))

(defun od-really-lookup-pattern (orig-pat)
  (od-open-dictionary)
  (set-buffer dserver-buffer)
  (erase-buffer)
  (let* ((index (od-pattern-consistency-check orig-pat))
	 (pat (if (or (eql index ?A) (eql index ?K))
		  (od-string-reverse orig-pat)
		orig-pat)))
    (od-log od-debug-flag "pat [%s]\n" (od-quote-null orig-pat))
    (od-log od-debug-flag (format "P%c%s\n" index pat))
    (set-process-coding-system od-dict *euc-japan* *euc-japan*)
    (process-send-string od-dict (format "P%c%s\n" index pat))
    (let ((ans (od-wait-until '("$$\n" "$?\n" "$N\n" "$<\n") od-dict)))
      (set-process-coding-system od-dict *internal* *euc-japan*)
      (cond ((string= "$$\n" ans)
	     (let ((str (buffer-string)))
	       (od-log od-debug-flag "buffer-string[%s]\n"
		       (od-quote-null str))
	       (od-find-entries orig-pat str)))
	    (t (od-no-entries-found orig-pat))))))

(defun od-pattern-consistency-check (pat)
  (let ((len (length pat))
	(i 0)
	kana alfa other first last)
    (od-log od-debug-flag "consistency pat[%s]\n" (od-quote-null pat))
    (while (< i len)
      (let* ((c (sref pat i))
	     (bytes (char-bytes c)))
	(setq i (+ bytes i))
	(cond ((or (= ?* c) (= ?$B!v(B c))
	       (cond ((= i bytes)
		      (setq first t))	;at beginning
		     ((= i len)
		      (setq last t))	;at end
		     (t (error "\"*\"$B$O%Q%?!<%s$NC<0J30$G$O;H$($^$;$s(B."))))
	      ((= ?? c))		;do nothing
	      ((or (and (<= ?a c) (<= c ?z))
		   (and (<= ?A c) (<= c ?Z))
		   (and (<= ?$B#a(B c) (<= c ?$B#z(B))
		   (and (<= ?$B#A(B c) (<= c ?$B#Z(B)))
	       (setq alfa t))
	      ((string-match "\\(\\cH\\|\\cK\\)" (char-to-string c))
	       (setq kana t))
	      (t
	       (setq other t)))))
    (if (and kana alfa)
	(message "$B2>L>$H1Q;z$,:.$6$C$F$$$^$9$,(B, $B2>L>$H$7$F8!:w$rB3$1$^$9(B."))
    (if (and first last)
	(error "\"*\"$B$O(B1$B8D$7$+;H$($^$;$s(B."))
    (let ((idx (cond (kana (if first ?K ?k))
		     (alfa (if first ?A ?a)))))
      (if (not (memq idx od-index-types))
	  (error "$B$=$N8!:wJ}K!$O;H$($^$;$s(B."))
      idx)))

(defun od-find-entries (pat str)
  (set-buffer " *od-temp*")
  (erase-buffer)
  (insert str)
  (let (new-headers)
    (goto-char 1)
    (if (looking-at "$0\n")		;this is $0 line
	(forward-char 3))		;just after $0
    (while (< (point) (- (point-max) 3))
      (let (entry body)
	(od-decode-header)
	(setq entry (buffer-substring
		     (progn (beginning-of-line) (point))
		     (progn (end-of-line) (point))))
	(forward-line 1)
	(setq body (buffer-substring
		    (progn (beginning-of-line) (point))
		    (progn (end-of-line) (point))))
	(if (not (od-findp-equal (cons entry body) new-headers))
	    (setq new-headers
		  (cons (cons entry body)
			new-headers)))
	(forward-line 1)))
    (if (null new-headers)
	(od-no-entries-found pat)
      (setq od-current-entry -1)
      (setq od-headers (nreverse new-headers))
      (od-show-headers)
      (od-add-to-history)
      (delete-other-windows)
      (if (and od-select-unique-entry-immediately
	       (= 1 (length od-headers)))
	  (od-select-entry 1)))))

(defun od-decode-header ()
  (narrow-to-region (progn (beginning-of-line) (point))
		    (progn (end-of-line) (point)))
  (let ((tmp '(("(C)" . "#xa1#x42")
	       ("(U)" . "#xa1#x43")
	       ("$B!](B" . "-"))))
    (while tmp
      (goto-char (point-min))
      (let ((pair (car tmp)))
	(while (search-forward (car pair) nil t)
	  (replace-match (cdr pair) nil t)))
      (setq tmp (cdr tmp))))
  (goto-char (point-min))
  (while
      (re-search-forward "\\(#x[0-9a-f][0-9a-f]\\)+" nil t nil)
    (let* ((hexstr (buffer-substring (match-beginning 0) (match-end 0)))
	   (len (/ (length hexstr) 4))
	   (str (make-string (+ len 2) 0))
	   (i 0))
      (delete-region (match-beginning 0) (match-end 0))
      (while (< i len)
	(aset str i
	      (od-xtoi (substring hexstr (+ (* i 4) 2) (+ (* i 4) 4))))
	(setq i (1+ i)))
      (fillarray od-ccl-status 0)
      (if (equal od-current-jisyo-name "kojien")
	  (aset od-ccl-status 3 1))
      (insert (exec-ccl-string
	       (if window-system 
		   (car od-current-jisyo-ccl)
		 (cdr od-current-jisyo-ccl))
	       od-ccl-status str))))
  (widen))

(defun od-show-headers ()
  (interactive)
  (od-set-header)
  (let ((buffer-read-only nil))
    (erase-buffer)
    (insert (format "%d entries.\n"
		    (length od-headers)))
    (let ((tmp od-headers)
	  (i 1))
      (while tmp
	(insert (format "%3d: " i))
	(put-text-property
	 (point) (progn (insert (car (car tmp))) (point))
	 'mouse-face 'highlight)
	(insert ?\n)
	(setq i (1+ i))
	(setq tmp (cdr tmp)))))
  (if (> od-current-entry 0)
      (goto-line (1+ od-current-entry))
    (goto-line 2))
  (forward-char 4))

(defmacro od-select-entry-nth-pat (i)
  (` (od-xtoi (substring (cdr pat)
			 (match-beginning (, i)) (match-end (, i))))))

(defun od-mouse-select-entry (click)
  (interactive "e")
  (let* ((start (event-start click))
	 (window (posn-window start))
	 (pos (posn-point start)))
    (select-window window)
    (goto-char pos))
  (let ((curr (od-current-header-line)))
    (od-select-entry curr)))

(defun od-select-entry (n)
  (interactive "nNumber:")
  (let ((len (length od-headers)))
    (if (and (<= 1 n) (<= n len))
	(progn
	  (let ((case-fold-search t)
		(pat (nth (1- n) od-headers))
		fno ofs)
	    (cond ((string-match "\\`\\([0-9a-f]*\\):\\([0-9a-f]*\\)"
				 (cdr pat))
		   (setq fno (od-select-entry-nth-pat 1))
		   (setq ofs (or (od-select-entry-nth-pat 2) 0))
		   (setq od-current-entry n)
		   (od-show-entry fno ofs)
		   (setq od-current-entry-in-body-flag t))
		  ((string-match "\\`P\\([0-9a-f]*\\):\\([0-9a-f]*\\)" 
				 (cdr pat))
		   (setq fno (od-select-entry-nth-pat 1))
		   (setq ofs (or (od-select-entry-nth-pat 2) 0))
		   (od-show-picture fno ofs))
		  ((string-match "\\`T\\([0-9a-f]*\\):\\([0-9a-f]*\\):\
\\([0-9a-f]*\\):\\([0-9a-f]*\\)"
				 (cdr pat))
		   (od-transfer-picture
		    (od-select-entry-nth-pat 1) (od-select-entry-nth-pat 2)
		    (od-select-entry-nth-pat 3) (od-select-entry-nth-pat 4)))
		  (t
		   (message "sorry, this type of entry is not supported."))))
	  (od-set-header)
	  (od-add-to-history)
	  (goto-line (1+ n))
	  (forward-char 4))
      (message "%d out of range" n))))

(defun od-direct-select-entry ()
  (interactive)
  (let ((n (string-to-int (char-to-string last-command-char))))
    (od-select-entry n)))

(defun od-current-header-line ()
  (save-excursion
    (od-set-header)
    (1- (count-lines 1 (1+ (point))))))

(defun od-previous-entry ()
  (interactive)
  (let ((curr (od-current-header-line)))
    (if (< 1 curr)
	(od-select-entry (1- curr))
	(message "No previous entry"))))

(defun od-show-current-entry ()
  (interactive)
  (let ((curr (od-current-header-line)))
    (od-select-entry curr)))

(defun od-next-entry ()
  (interactive)
  (let ((curr (od-current-header-line)))
    (if (< curr (length od-headers))
	(od-select-entry (1+ curr))
      (message "No following entry"))))

(defun od-scroll-step ()
  (cond ((integerp od-scroll-step)
	 od-scroll-step)
	((eql od-scroll-step 'full)
	 (- (window-height (next-window (selected-window))) 2))
	((eql od-scroll-step 'half)
	 (- (/ (window-height (next-window (selected-window))) 2) 1))))

(defun od-scroll-entry (step)
  (let ((curr (od-current-header-line)))
    (if (or (not od-current-entry-in-body-flag)
	    (/= od-current-entry curr)
	    (/= od-history-drifting-point od-history-current-point))
	(od-select-entry curr)
      (scroll-other-window step))))

(defun od-scroll-entry-up ()
  (interactive)
  (od-scroll-entry (od-scroll-step)))

(defun od-scroll-entry-down ()
  (interactive)
  (od-scroll-entry (- (od-scroll-step))))

(defun od-scroll-entry-up-half ()
  (interactive)
  (od-scroll-entry (/ (od-scroll-step) 2)))

(defun od-scroll-entry-down-half ()
  (interactive)
  (od-scroll-entry (- (/ (od-scroll-step) 2))))

(defun od-scroll-entry-up-line ()
  (interactive)
  (od-scroll-entry 1))

(defun od-up-history ()
  (interactive)
  (let ((previous-point
	 (% (+ od-history-drifting-point od-history-length -1)
	    od-history-length)))
    (if (or (= od-history-current-point previous-point)
	    (not (aref od-history previous-point)))
	(message "no more history.")
      (setq od-history-drifting-point previous-point)
      (setq od-headers (car (aref od-history previous-point)))
      (setq od-current-entry (cdr (aref od-history previous-point)))
      (od-show-headers))))

(defun od-down-history ()
  (interactive)
  (let ((next-point
	 (% (1+ od-history-drifting-point) od-history-length)))
    (if (or (= od-history-current-point od-history-drifting-point)
	    (not (aref od-history next-point)))
	(message "no more history.")
      (setq od-history-drifting-point next-point)
      (setq od-headers (car (aref od-history next-point)))
      (setq od-current-entry (cdr (aref od-history next-point)))
      (od-show-headers))))

(defun od-reset-history (old-dic new-dic)
  (let ((old-history (assoc old-dic od-history-alist))
	(new-history (assoc new-dic od-history-alist)))
    (if old-history
	(setcdr old-history (cons od-history od-history-current-point))
      (setq od-history-alist
	    (cons (cons old-dic (cons od-history od-history-current-point))
		  od-history-alist)))
    (if (not new-history)
	(setq od-history (make-vector od-history-length nil)
	      od-history-current-point 0)
      (setq od-history (car (cdr new-history))
	    od-history-current-point (cdr (cdr new-history)))
      (let ((header-and-entry (aref od-history od-history-current-point)))
	(if header-and-entry
	    (progn
	      (setq od-headers (car header-and-entry)
		    od-current-entry (cdr header-and-entry))))))
    (setq od-history-drifting-point od-history-current-point)))

(defun od-add-to-history ()
  (if (not (equal (car (aref od-history od-history-current-point))
		  od-headers))
      (setq od-history-current-point
	    (% (1+ od-history-current-point) od-history-length)))
  (setq od-history-drifting-point od-history-current-point)
  (aset od-history od-history-current-point
	(cons od-headers od-current-entry)))

(defun od-make-fno-ofs (point)
  (format "%x:%x"
	  (string-to-int (buffer-substring point
					   (+ 8 point)))
	  (string-to-int (buffer-substring (+ 8 point)
					   (+ 12 point)))))

(defun od-collect-references ()
  (goto-char 1)
  (let ((overlay-list od-overlay-list)
	(i 1)
	mc-flag references)
    (while (re-search-forward "\037\\([BC]\\|E\037[23]\\)" nil t nil)
      (let ((head (string-to-char (buffer-substring (match-beginning 1)
						    (match-end 1))))
	    start caption caption-start ref)
	(setq start (match-beginning 0))
	(setq caption-start (point))
	(re-search-forward "\037[bcRS]")
	(setq caption (buffer-substring caption-start (- (point) 2)))
	(cond ((or (= head ?B) (= head ?C))
	       (setq ref (cons caption
			       (od-make-fno-ofs (point))))
	       (forward-char 12))
	      (t
	       (cond ((= (preceding-char) ?R)
		      (setq ref
			    (cons (concat "$B?^(B:" caption)
				  (concat "P" (od-make-fno-ofs (point)))))
		      (forward-char 12))
		     ((= (preceding-char) ?S)
		      (setq ref
			    (cons (concat "$B2;(B:" caption)
				  (concat "S" (buffer-substring
					       (1+ (point))
					       (+ 16 (point))))))
		      (forward-char 16)))))
	(delete-region start (point))
	(let ((start (point))
	      overlay)
	  (insert caption)
	  (cond (overlay-list
		 (setq overlay (car overlay-list))
		 (setq overlay-list (cdr overlay-list))
		 (move-overlay overlay start (point)))
		(t
		 (setq overlay (make-overlay start (point)))
		 (overlay-put overlay 'face 'underline)
		 (overlay-put overlay 'mouse-face 'highlight)
		 (setq od-overlay-list (cons overlay od-overlay-list))))
	  (overlay-put overlay 'od-eno i)
	  (setq i (1+ i))
	  (setq references (cons ref references)))))
    (nreverse references)))

(defun od-show-entry (fno &optional ofs)
  (save-excursion
    (od-set-body)
    (let (buffer-read-only)
      (erase-buffer)
      (message "inserting frames...")
      (od-insert-frames fno ofs)
      (indent-rigidly 1 (point-max) 1)
      (message "collecting references...")
      (setq od-references (od-collect-references))
      (message "filtering...")
      (let ((tmp od-current-dict-filter-func)
	    (case-fold-search nil))
	(while tmp
	  (goto-char 1)
	  (funcall (car tmp))
	  (setq tmp (cdr tmp))))
      (if od-fill-function
	  (progn
	    (message "filling ...")
	    (funcall od-fill-function 1 (point-max))
	    (if (and (string= od-current-jisyo-name "eiwa")
		     od-eiwa-summary-mode)
		(hide-body)))))
    (if od-references
	(message
	 (substitute-command-keys
	  "\\<odic-mode-map>\
Type \"\\[od-make-entry-table]\" to see references."))
      (message ""))))

(defun od-fill-region (start end)
  (let ((end-marker (make-marker)))
    (set-marker end-marker end)
    (goto-char start)
    (while (< (point) end-marker)
      (if (< (move-to-column fill-column) fill-column)
	  (if (< (point) (point-max))
	      (forward-char))
	(while (and (not (= (following-char) ?\n))
		    (looking-at "[A-Za-z]"))
	  (backward-char))
	(forward-char)
	(insert ?\n)
	(delete-horizontal-space)))))

(defun od-insert-frames (fno &optional ofs)
  (let ((start-point (point))
	(end-marker (make-marker))
	(previous-pos (point))
	(previous-face 'default)
	(str (od-get-first-frame fno ofs)))
    (fillarray od-ccl-status 0)
    (aset od-ccl-status 3 1)
    (while
	(progn
	  (insert (exec-ccl-string
		   (if od-use-only-ascii-and-jisx0208
		       (cdr od-current-jisyo-ccl)
		     (car od-current-jisyo-ccl))
		   od-ccl-status str))
	  (set-marker end-marker (point))
	  (goto-char start-point)
	  (while (search-forward "\e[" end-marker t)
	    (let ((face
		   (cond ((looking-at "0m")
			  (delete-region (- (point) 2) (+ (point) 2))
			  'default)
			 ((looking-at "1m")
			  (delete-region (- (point) 2) (+ (point) 2))
			  'bold)
			 ((looking-at "7m")
			  (delete-region (- (point) 2) (+ (point) 2))
			  'modeline)
			 ((looking-at "1;7m")
			  (delete-region (- (point) 2) (+ (point) 4))
			  'od-bold-inverse)
			 (t
			  nil))))
	      (if face
		  (progn
		    (if (not (eq previous-face 'default))
			(add-text-properties previous-pos (point)
					     (list 'face previous-face
						   'rear-nonsticky '(face))))
		    (setq previous-pos (point)
			  previous-face face)))))
	  (= (logand (aref od-ccl-status 3) 2) 0))
      (goto-char end-marker)
      (setq fno (1+ fno))
      (setq str (od-get-next-frame fno))
      (setq start-point (marker-position end-marker)))
    (goto-char start-point)
    (if (not (search-forward "\C-c" end-marker t))
	(error "diclookup-mule internal error: can't find endmark."))
    (if (not (eq previous-face 'default))
	(add-text-properties previous-pos (1- (point))
			     (list 'face previous-face
				   'rear-nonsticky '(face))))
    (delete-region (1- (point)) end-marker)
    (od-get-last-frame)))

(defun od-no-entries-found (pat)
  (message "\"%s\" Not found" pat)
  (ding))



(defun od-toggle-eiwa-summary-mode ()
  (interactive)
  (if (string= od-current-jisyo-name "eiwa")
      (progn
	(setq od-eiwa-summary-mode (not od-eiwa-summary-mode))
	(if od-eiwa-summary-mode
	    (setq od-eiwa-current-display-mode "[Summary]")
	  (setq od-eiwa-current-display-mode ""))
	(save-excursion
	  (set-buffer od-dict-body-buffer)
	  (if od-eiwa-summary-mode
	      (hide-body)
	    (show-all))))))

(defun od-toggle-character-sets ()
  (interactive)
  (setq od-use-only-ascii-and-jisx0208
	(not od-use-only-ascii-and-jisx0208))
  (od-show-current-entry))


(defun od-bury-buffers ()
  (bury-buffer od-dict-header-buffer)
  (bury-buffer od-dict-body-buffer)
  (bury-buffer od-dict-select-buffer))

(defun od-dictionary-frame-p (&optional frame)
  (string= od-frame-name
	   (cdr (assq 'name (frame-parameters frame)))))

(defun od-iconify ()
  (if (od-dictionary-frame-p)
      (progn
	(iconify-frame)
	(if (>= emacs-minor-version 27)
	    (select-frame (previous-frame)))
	(set-mouse-position (selected-frame) (1- (frame-width)) 0)
	(unfocus-frame))))

(defun od-make-invisible ()
  (if (od-dictionary-frame-p)
      (progn
	(make-frame-invisible)
	(if (>= emacs-minor-version 27)
	    (select-frame (previous-frame)))
	(set-mouse-position (selected-frame) (1- (frame-width)) 0)
	(unfocus-frame))))

(defun od-lower-and-select-previous-frame ()
  (if (od-dictionary-frame-p)
      (progn
	(lower-frame (selected-frame))
	(select-frame (previous-frame))
	(set-mouse-position (selected-frame) (1- (frame-width)) 0)
	(unfocus-frame))))

(defun od-select-and-raise-previous-frame ()
  (if (od-dictionary-frame-p)
      (progn
	(select-frame (previous-frame))
	(raise-frame (selected-frame))
	(set-mouse-position (selected-frame) (1- (frame-width)) 0)
	(unfocus-frame))))

(defun od-quit ()
  (interactive)
  (od-bury-buffers)
  (run-hooks 'od-quit-hook)
  (if (and (not (od-dictionary-frame-p)) od-previous-config)
	(progn
	  (set-window-configuration od-previous-config)
	  (setq od-previous-config nil))))

(defun od-delete-frame ()
  (if (od-dictionary-frame-p)
      (delete-frame)))

(defun od-really-quit ()
  (interactive)
  (od-bury-buffers)
  (let ((tmp od-dict-list))
    (while tmp
      (let ((dict (car tmp)))
	(process-send-string dict (format "Q\n"))
	(delete-process dict))
      (setq tmp (cdr tmp))))
  (od-kill-active-buffers)
  (setq od-dict nil)
  (setq od-dict-list nil)
  (run-hooks 'od-really-quit-hook)
  (if (and (not (od-dictionary-frame-p)) od-previous-config)
      (progn
	(set-window-configuration od-previous-config)
	(setq od-previous-config nil))))


(defvar od-ispell-program-name "ispell"
  "Program invoked by ispell-word and ispell-region commands.")

(defconst od-ispell-out-name " *ispell*"
  "Name of the buffer that is associated with the 'ispell' process")

(defvar od-ispell-process nil
  "Holds the process object for 'ispell'")

(defun od-ispell-init-process ()
  (if (and od-ispell-process
	   (eq (process-status od-ispell-process) 'run))
      (save-excursion
	(set-buffer od-ispell-out-name)
	(erase-buffer))
      (message "Starting new ispell process...")
      (and (get-buffer od-ispell-out-name) (kill-buffer od-ispell-out-name))
      (setq od-ispell-process (apply 'start-process "ispell"
				   od-ispell-out-name od-ispell-program-name
				   (list "-a")))
      (process-kill-without-query od-ispell-process)
      (sleep-for 3)))

(defun od-lookup-pattern-with-ispell ()
  (interactive)
  (let ((orig-pat (read-string "Pattern: ")))
    (od-lookup-pattern-with-ispell-sub orig-pat)))

(defun od-lookup-pattern-with-ispell-sub (orig-pat)
  (od-ispell-init-process)
  (if od-ispell-ver-3
      (progn
	(set-buffer od-ispell-out-name)
	(erase-buffer)))
  (send-string od-ispell-process orig-pat)
  (send-string od-ispell-process "\n")
  (sleep-for 1)
  (set-buffer od-ispell-out-name)
  (goto-char 1)
  (cond ((looking-at "\\*")
	 ;; correct!
	 (od-really-lookup-pattern orig-pat))
	((looking-at "+")
	 ;; a word follows
	 (od-really-lookup-pattern (od-select-candidate)))
	((looking-at "&")
	 ;; some words follows
	 (od-really-lookup-pattern (od-select-candidate)))
	((looking-at "#")
	 ;; cannot find any candidates
	 (message "No candidate."))))

(defun od-select-candidate ()
  (cond ((and od-ispell-ver-3
	      (looking-at "&"))
	 (goto-char 3)
	 (zap-to-char 1 ?:)
	 (delete-char 1)
	 (while (search-forward "," nil t)
	   (replace-match "" nil t))))
  (goto-char 1)
  (forward-char 2)
  (let ((list nil))
    (while (looking-at "[^ ]")
      (setq list
	    (cons (buffer-substring (point)
				    (progn (forward-word 1) (point)))
		  list))
      (skip-chars-forward " \t\n"))
    (switch-to-buffer-other-window od-dict-select-buffer)
    (let (buffer-read-only)
      (erase-buffer)
      (if window-system
	  (insert (substitute-command-keys
		   "Click \\[od-mouse-choose-candidate] \
on a candidate to select it.\n")))
      (insert (substitute-command-keys
	       "In this buffer, type \\[od-choose-candidate] \
to select the candidate near point.\n\n"))
      (let ((i 1)
	    (col 0)
	    (l list))
	(while l
	  (let* ((str (format "(%d)  %s  " i (car l)))
		 (len (length str)))
	    (if (< (- (frame-width) 2) (+ col len))
		(progn
		  (insert "\n")
		  (setq col 0)))
	    (setq col (+ col len))
	    (put-text-property
	     (point) (progn (insert str) (- (point) 2))
	     'mouse-face 'highlight))
	  (setq l (cdr l))
	  (setq i (1+ i)))))
    (other-window 1)
    (let ((sel (or (and od-ispell-ver-3 (= (length list) 1) 1)
		   (od-read-integer))))
      (while (or (< sel 1) (< (length list) sel))
	(setq sel (od-read-integer)))
      (nth (1- sel) list))))

(defvar od-read-integer-buffer nil)

(defun od-read-integer ()
  (let ((od-read-integer-buffer (window-buffer (minibuffer-window))))
    (let ((n (string-to-int (read-from-minibuffer "Choose:"))))
      (while (not (integerp n))
	(setq n (string-to-int (read-from-minibuffer "Choose:"))))
      n)))

(defun od-mouse-choose-candidate (click)
  (interactive "e")
  (let* ((start (event-start click))
	 (window (posn-window start))
	 (pos (posn-point start)))
    (save-excursion
      (set-buffer (window-buffer window))
      (goto-char pos)
      (od-choose-candidate))))
	    
(defun od-choose-candidate ()
  (interactive)
  (if (eq (get-text-property (point) 'mouse-face) 'highlight)
      (let* ((start
	      (or (previous-single-property-change (1+ (point)) 'mouse-face)
		  1))
	     (func (get-text-property start 'od-func)))
	(cond (func
	       (funcall func))
	      ((and (eq (char-after start) ?\()
		    od-read-integer-buffer)
	       (goto-char start)
	       (if (looking-at "([1-9][0-9]*)")
		   (let ((str (buffer-substring
			       (1+ (match-beginning 0))
			       (1- (match-end 0)))))
		     (set-buffer od-read-integer-buffer)
		     (insert str)
		     (exit-minibuffer))))))))

(defun od-read-string-with-fep (prompt initial)
  (cond ((or (not od-input-by-kanji)
	     (eql od-fep-type 'no-fep))
	 (read-string prompt (or initial "")))
	((eql od-fep-type 'egg)
	 (read-hiragana-string prompt (or initial "")))
	((eql od-fep-type 'canna)
	 (if (not canna:*japanese-mode*)
	     (canna-toggle-japanese-mode))
	 (read-string prompt (or initial "")))
	((eql od-fep-type 't-code)
	 (setq tcode-on-in-minibuffer nil)
	 (setq unread-command-events od-tcode-toggle-events)
	 (read-string prompt (or initial "")))
	(t
	 (let ((its:*current-map* (its:get-mode-map od-fep-type)))
	   (if its:*current-map*
	       (read-current-its-string prompt (or initial ""))
	     (error "Unknown fep-type %s" od-fep-type))))))

(defun od-findp-equal (item sequence)
  (let ((is-true nil))
    (while (and sequence
		(not is-true))
      (if (equal item (car sequence))
	  (setq is-true t)
	  (setq sequence (cdr sequence))))
    is-true))

(defun od-string-reverse (str)
  (let* ((len (length str))
	 (rev (make-string len ? ))
	 (i 0))
    (while (< i len)
      (let* ((char (sref str i))
	     (bytes (char-bytes char)))
	(sset rev (- len i bytes) char)
	(setq i (+ i bytes))))
    rev))


(defun od-get-first-frame (fno &optional ofs)
  (save-excursion
    (od-open-dictionary)
    (set-buffer dserver-buffer)
    (let (str mc-flag)
      (erase-buffer)
      (process-send-string od-dict (format "F%04x\n" fno))
      (while (< (length (buffer-string)) 2051)
	(accept-process-output od-dict))
      (setq str (buffer-substring (+ 3 (or ofs 0)) (+ 2048 3)))
      (erase-buffer)
      (process-send-string od-dict (format "F%04x\n" (1+ fno)))
      str)))

(defun od-get-next-frame (fno)
  (save-excursion
    (set-buffer dserver-buffer)
    (let (str mc-flag)
      (while (< (buffer-size) 2051)
	(accept-process-output od-dict))
      (setq str (buffer-substring 3 (+ 2048 3)))
      (erase-buffer)
      (process-send-string od-dict (format "F%04x\n" (1+ fno)))
      str)))

(defun od-get-last-frame ()
  (save-excursion
    (set-buffer dserver-buffer)
    (let (mc-flag)
      (while (< (buffer-size) 2051)
	(accept-process-output od-dict))
      (buffer-substring 3 (+ 2048 3)))))

(defun od-get-frame (fno &optional ofs min)
  (save-excursion
    (od-open-dictionary)
    (set-buffer dserver-buffer)
    (erase-buffer)
    (let (mc-flag
	  (str ""))
      (while (< (length str) (or min 1))
	(erase-buffer)
	(process-send-string od-dict (format "F%04x\n" fno))
	(while (< (buffer-size) 2051)
	  (accept-process-output od-dict))
	(setq str (concat str (buffer-substring (+ 3 (or ofs 0)) (+ 2048 3))))
	(setq ofs nil)
	(setq fno (1+ fno)))
      str)))


;;; hanrei, okuduke
(defun od-show-hanrei ()
  (interactive)
  (od-open-dictionary)
  (set-buffer dserver-buffer)
  (erase-buffer)
  (process-send-string od-dict "I\n")
  (let ((ans (od-wait-until '("$$\n" "$?\n" "$N\n" "$<\n") od-dict)))
    (cond ((string= "$$\n" ans)
	   (goto-char 1)
	   (if (re-search-forward "HA " nil t nil)
	       (if (re-search-forward "[0-9a-f][0-9a-f]*")
		   (let ((frm (buffer-substring
			       (match-beginning 0) (match-end 0))))
		     (od-show-entry (od-xtoi frm))
		     (setq od-current-entry-in-body-flag nil))))))))

(defun od-show-okuduke ()
  (interactive)
  (od-open-dictionary)
  (set-buffer dserver-buffer)
  (erase-buffer)
  (process-send-string od-dict "I\n")
  (let ((ans (od-wait-until '("$$\n" "$?\n" "$N\n" "$<\n") od-dict)))
    (cond ((string= "$$\n" ans)
	   (goto-char 1)
	   (if (re-search-forward "OK " nil t nil)
	       (if (re-search-forward "[0-9a-f][0-9a-f]*")
		   (let ((frm (buffer-substring
			       (match-beginning 0) (match-end 0))))
		     (od-show-entry (od-xtoi frm))
		     (setq od-current-entry-in-body-flag nil))))))))

(defun od-lookup-reference ()
  (interactive)
  (od-really-lookup-reference (overlays-at (point))))
    
(defun od-mouse-lookup-reference (click)
  (interactive "e")
  (let* ((start (event-start click))
	 (window (posn-window start))
	 (pos (posn-point start)))
    (od-really-lookup-reference 
     (save-excursion
       (set-buffer (window-buffer window))
       (overlays-at pos)))))

(defun od-really-lookup-reference (overlays)
  (if (not overlays)
      (message "no reference at this point.")
    (od-make-entry-table)
    (od-select-entry (overlay-get (car overlays) 'od-eno))))

(defun od-make-entry-table ()
  (interactive)
  (if (not od-references)
      (message "there is no reference.")
    (setq od-headers od-references)
    (setq od-current-entry -1)
    (od-show-headers)
    (od-add-to-history)))

(defun od-xtoi (s)
  (let ((r 0)
	(i 0)
	(l (length s)))
    (while (< i l)
      (let ((c (aref s i)))
	(setq r (+ (* r 16)
		   (if (and (<= ?0 c) (<= c ?9))
		       (- c ?0)
		       (+ (- c ?a) 10))))
	(setq i (1+ i))))
    r))

(defun od-bcd-to-int (str)
  (let ((len (length str))
	(result 0)
	(i 0))
    (while (< i len)
      (setq result (+ (* 100 result)
		      (* 10 (/ (aref str i) 16))
		      (% (aref str i) 16)))
      (setq i (1+ i)))
    result))

(defun od-show-picture (fno ofs)
  (let ((page 1)
	profile sx sy pf po references)
    (while (progn
	     (setq profile (substring (od-get-frame fno ofs 52) 0 52))
	     (setq sx (od-bcd-to-int (substring profile 8 10)))
	     (setq sy (od-bcd-to-int (substring profile 10 12)))
	     (setq pf (od-bcd-to-int (substring profile 14 18)))
	     (setq po (od-bcd-to-int (substring profile 18 20)))
	     (setq references
		   (cons (cons (format "Page%d" page)
			       (format "T%x:%x:%x:%x" sx sy pf po))
			 references))
	     (/= (aref profile 3) 1))
      (setq fno (od-bcd-to-int (substring profile 44 48)))
      (setq ofs (od-bcd-to-int (substring profile 48 50)))
      (setq page (1+ page)))
    (if (= page 1)
	(od-transfer-picture sx sy pf po)
      (setq od-references (nreverse references))
      (od-make-entry-table))))

(defun od-transfer-picture (sx sy pf po)
  (if (not (or (eq window-system 'x) (getenv "DISPLAY")))
      (error "X window is not running.")
    (let* ((inhibit-quit t)
	   (process-connection-type nil)
	   (process (apply 'start-process "dict-view-picture" nil
			   od-view-picture-program-and-args)))
      (set-process-coding-system process *noconv* *noconv*)
      (message "transferring picture data...")
      (process-send-string process (format "P4\n%d %d\n" sx sy))
      (let* ((size (* sx sy))
	     (rest (+ size po -2048))
	     str)
	(if (<= rest 0)
	    (process-send-string process (substring (od-get-frame pf po) size))
	  (setq str (od-get-first-frame pf po))
	  (while (> rest 2048)
	    (if quit-flag
		(progn (delete-process process)
		       (setq inhibit-quit nil)))
	    (process-send-string process str)
	    (message "transferring picture data... %d%%"
		     (/ (* 100 rest) size))
	    (setq pf (1+ pf))
	    (setq str (od-get-next-frame pf))
	    (setq rest (- rest 2048)))
	  (process-send-string process str)
	  (setq str (od-get-last-frame))
	  (process-send-string process (substring str 0 rest)))
	(process-send-eof process)
	(message "transferring picture data... done")))))

;;;
;;; Additional new interface [lookup-current-word].
;;;  look up a current point word.
;;;
;;;  15 Nov 91	Tetsuya Nishimaki   (t-nishim@dn.softbank.co.jp)
;;;
(defun od-word-at-point ()
  (interactive)
  (condition-case ()
      (save-excursion
	(if (looking-at "\\([a-zA-Z]\\|\\cH\\|\\cK\\)")
	    (forward-word 1))
	(re-search-backward
	 "\\(\\<[a-zA-Z]+\\|\\CH\\cH+\\|\\CK\\cK+\\)\\>" nil t)
	(goto-char (match-beginning 0))
	(let ((end (match-end 0)))
	  (if (not (looking-at "[a-zA-Z]")) (forward-char 1))
	  (buffer-substring (point) end)))
   (error nil)))

;;;
;;; modified by yutaka@sys1.cpg.sony.co.jp
;;;
;;;###autoload
(defun od-lookup-pattern-edit ()
  (interactive)
  (let* ((v (od-word-at-point))
	 (enable-recursive-minibuffers t)
	 (pat (read-string (if od-input-by-kanji
			       "Pattern(in kana): "
			     "Pattern: ")
			   v)))
    (online-dictionary)
    (if (od-japanese-p pat)
	;; search japanese
	(or (string= od-current-jisyo-name od-default-jisyo-japanese)
	    (od-really-change-dictionary od-default-jisyo-japanese))
      (or (string= od-current-jisyo-name "eiwa")
	  (od-really-change-dictionary "eiwa")))
    (od-really-lookup-pattern pat)))

;;;###autoload
(defun od-lookup-pattern-ispell ()
  (interactive)
  (let* ((c (current-window-configuration))
	 (v (od-word-at-point))
	 (enable-recursive-minibuffers t)
	 (pat v))
    (set-window-configuration c)
    (online-dictionary)
    (or (string= od-current-jisyo-name "eiwa")
	(od-really-change-dictionary "eiwa"))
    (od-lookup-pattern-with-ispell-sub pat)))


(defvar lup-mode nil)
(make-variable-buffer-local 'lup-mode)

(or (assq 'lup-mode minor-mode-alist)
    (setq minor-mode-alist (cons '(lup-mode " LUP") minor-mode-alist)))

(defvar lup-mode-prefix "\C-c\C-l"
  "*Prefix key to use for Lup commands in Lup minor mode.")
(defvar lup-mode-map nil)
(if lup-mode-map
    nil
  (setq lup-mode-map (make-sparse-keymap))
  (if lup-mode-prefix
      (let ((map (make-sparse-keymap)))
	(define-key map "i" 'od-lookup-pattern-ispell)
	(define-key map "l" 'od-lookup-pattern-edit)
	(define-key map "q" 'lup-quit)
	(define-key lup-mode-map lup-mode-prefix map)))
  (define-key lup-mode-map [down-mouse-1] 'lup-mouse-drag-region)
  (define-key lup-mode-map [drag-mouse-1] 'lup-mouse-set-region)
  (define-key lup-mode-map [double-mouse-1] 'lup-mouse-set-point)
  (define-key lup-mode-map [triple-mouse-1] 'lup-mouse-set-point)
  (define-key lup-mode-map [menu-bar lup]
    (cons "Lup" (make-sparse-keymap "Lup")))
  (define-key lup-mode-map [menu-bar lup quit]
    '("Quit" . lup-quit))
  (define-key lup-mode-map [menu-bar lup lookup-pattern-ispell]
    '("Lookup Word with Ispell" . od-lookup-pattern-ispell))
  (define-key lup-mode-map [menu-bar lup lookup-pattern-edit]
    '("Lookup Word" . od-lookup-pattern-edit)))

(or (assq 'lup-mode minor-mode-map-alist)
    (setq minor-mode-map-alist
	  (cons (cons 'lup-mode lup-mode-map)
		minor-mode-map-alist)))

;;;###autoload
(defun lup-mode (arg)
  "Toggle Lup minor mode.
With arg, turn Lup minor mode on if arg is positive, off otherwise.

Commands:\\<lup-mode-map>
\\[lup-mode]\
	lup$B%b!<%I$K$J$j$^$9(B.
\\[od-lookup-pattern-edit]\
	$B%+!<%=%k>e$N8l$r@Z$j=P$78!:w$7$^$9(B. 
\\[od-lookup-pattern-ispell]\
	$B%+!<%=%k>e$N8l$r@Z$j=P$7(Bispell$B$G867?$KD>$7$?8e8!:w$7$^$9(B.
\\[lup-quit]\
	lup$B%b!<%I$r$L$1$^$9(B."
  (interactive "P")
  (setq lup-mode
	(if (null arg) (not lup-mode)
	    (> (prefix-numeric-value arg) 0)))
  (if lup-mode
      (run-hooks 'lup-mode-hook))
  (set-buffer-modified-p (buffer-modified-p)))

(defun lup-quit ()
  (interactive)
  (lup-mode -1))

(defun lup-mouse-set-point (event)
  (interactive "e")
  (if (not (= (overlay-start mouse-drag-overlay)
	      (overlay-end mouse-drag-overlay)))
      (let (this-command)
	(push-mark (overlay-start mouse-drag-overlay) t t)
	(goto-char (overlay-end mouse-drag-overlay))
	(copy-region-as-kill (point) (mark t))
	(online-dictionary)
	(or (string= od-current-jisyo-name "eiwa")
	    (od-really-change-dictionary "eiwa"))
	(od-lookup-pattern-with-ispell-sub (current-kill 0)))
    (goto-char (overlay-end mouse-drag-overlay))))

(defun lup-mouse-set-region (click)
  (interactive "e")
  (mouse-set-region click)
  (online-dictionary)
  (or (string= od-current-jisyo-name "eiwa")
      (od-really-change-dictionary "eiwa"))
  (od-lookup-pattern-with-ispell-sub (current-kill 0)))

;;;###autoload
(defun lup-mouse-drag-region (start-event)
  "Set the region to the text that the mouse is dragged over.
Highlight the drag area as you move the mouse.
This must be bound to a button-down mouse event.
In Transient Mark mode, the highlighting remains once you
release the mouse button.  Otherwise, it does not."
  (interactive "e")
  (mouse-minibuffer-check start-event)
  (let* ((start-posn (event-start start-event))
	 (start-point (posn-point start-posn))
	 (start-window (posn-window start-posn))
	 (start-frame (window-frame start-window))
	 (bounds (window-edges start-window))
	 (top (nth 1 bounds))
	 (bottom (if (window-minibuffer-p start-window)
		     (nth 3 bounds)
		   ;; Don't count the mode line.
		   (1- (nth 3 bounds))))
	 (click-count (1- (event-click-count start-event))))
    (setq mouse-selection-click-count click-count)
    (mouse-set-point start-event)
    (let ((range (mouse-start-end start-point start-point click-count)))
      (move-overlay mouse-drag-overlay (car range) (nth 1 range)
		    (window-buffer start-window)))
    (deactivate-mark)
    (let (event end end-point)
      (track-mouse
	(while (progn
		 (setq event (read-event))
		 (or (mouse-movement-p event)
		     (eq (car-safe event) 'switch-frame)))
	  (if (eq (car-safe event) 'switch-frame)
	      nil
	    (setq end (event-end event)
		  end-point (posn-point end))

	    (cond
	     ;; Are we moving within the original window?
	     ((and (eq (posn-window end) start-window)
		   (integer-or-marker-p end-point))
	      (goto-char end-point)
	      (let ((range (mouse-start-end start-point (point) click-count)))
		(move-overlay mouse-drag-overlay (car range) (nth 1 range))))

	     (t
	      (let ((mouse-row (cdr (cdr (mouse-position)))))
		(cond
		 ((null mouse-row))
		 ((< mouse-row top)
		  (mouse-scroll-subr start-window (- mouse-row top)
				     mouse-drag-overlay start-point))
		 ((>= mouse-row bottom)
		  (mouse-scroll-subr start-window (1+ (- mouse-row bottom))
				     mouse-drag-overlay start-point)))))))))
      (if (consp event)
;;; When we scroll into the mode line or menu bar, or out of the window,
;;; we get events that don't fit these criteria.
;;;	       (eq (get (event-basic-type event) 'event-kind) 'mouse-click)
;;;	       (eq (posn-window (event-end event)) start-window)
;;;	       (numberp (posn-point (event-end event)))
	  (let ((fun (key-binding (vector (car event)))))
	    (if (not (memq fun '(mouse-set-region mouse-set-point)))
		(if (fboundp fun)
		    (funcall fun event))
	      (if (not (= (overlay-start mouse-drag-overlay)
			  (overlay-end mouse-drag-overlay)))
		  (let (last-command this-command)
		    (push-mark (overlay-start mouse-drag-overlay) t t)
		    (goto-char (overlay-end mouse-drag-overlay))
		    (copy-region-as-kill (point) (mark t)))
		(goto-char (overlay-end mouse-drag-overlay))
		(setq this-command 'mouse-set-point)))))
      (delete-overlay mouse-drag-overlay))))

(if (< emacs-minor-version 27)
    (fset 'lup-mouse-drag-region (symbol-function 'mouse-drag-region)))


(defun od-dump-frame (fno)
  (insert "\n")
  (let ((frame (od-get-frame fno 0))
	(count1 0)
	(count2 0))
    (while (< count1 2048)
      (while (progn
	       (insert (format "%02x" (aref frame count1)))
	       (setq count1 (1+ count1))
	       (/= (% count1 16) 0)))
      (insert "\t")
      (while (progn
	       (let ((hi (aref frame count2))
		     (lo (aref frame (1+ count2))))
		 (if (and (> hi 32) (> lo 32)
			  (< hi 128) (< lo 128))
		     (insert ?\[ (make-character lc-jp hi lo) ?\])
		   (insert (format "%02x%02x" hi lo))))
	       (setq count2 (+ 2 count2))
	       (/= (% count2 16) 0)))
      (insert "\n"))))

(defun od-dump-frame-ascii (fno)
  (insert "\n")
  (let ((frame (od-get-frame fno 0))
	(count1 0)
	(count2 0))
    (while (< count1 2048)
      (while (progn
	       (insert (format "%02x" (aref frame count1)))
	       (setq count1 (1+ count1))
	       (/= (% count1 16) 0)))
      (insert "\t")
      (while (progn
	       (let ((c (aref frame count2)))
		 (if (and (>= c 32) (< c 128))
		     (insert ?_ c)
		   (insert (format "%02x" c ))))
	       (setq count2 (1+ count2))
	       (/= (% count2 16) 0)))
      (insert "\n"))))


;;; ccl programs

;;; register assignment
;;
;; r0 temporary
;; r1 high
;; r2 low
;; r3 flags
;; r4 indent level

(eval-when-compile

(require 'ccl)

(defconst odccl-self
  '((write r1 r2)))

(defconst odccl-flags
  '((col . 1) (eop . 2)(bold . 4) (inv . 8)
    (item . 16) (ret . 32)))

(defun odccl-set (key)
  (let ((bit (cdr (assq key odccl-flags))))
    (if bit
	(` (r3 |= (, bit)))
      (error "odccl-set: %s not found" key))))

(defun odccl-reset (key)
  (let ((bit (cdr (assq key odccl-flags))))
    (if bit
	(` (r3 &= (, (- 32767 bit))))
      (error "odccl-reset: %s not found" key))))

(defun odccl-check (key)
  (let ((bit (cdr (assq key odccl-flags))))
    (if bit
	(` (r3 & (, bit)))
      (error "odccl-check: %s not found" key))))

(defconst odccl-prtbcd
  '((read r2)
    (r0 = (r2 >> 4))
    (r0 += ?0)
    (write r0)
    (r0 = (r2 & 15))
    (r0 += ?0)
    (write r0)))

(defconst odccl-jis
  (` ((read r2)
      (write (, lc-jp))
      (r0 = (r1 | 128))
      (write r0)
      (r0 = (r2 | 128))
      (write-repeat r0))))

(defconst odccl-start-bold
  (` ((, (odccl-set 'bold))
      (if (, (odccl-check 'inv))
	  "\e[1;7m"
	"\e[1m"))))

(defconst odccl-end-bold
  (` ((, (odccl-reset 'bold))
      (if (, (odccl-check 'inv))
	  "\e[7m"
	"\e[0m"))))

(defconst odccl-start-inv
  (` ((, (odccl-set 'inv))
      (if (, (odccl-check 'bold))
	  "\e[1;7m"
	"\e[7m"))))

(defconst odccl-end-inv
  (` ((, (odccl-reset 'inv))
      (if (, (odccl-check 'bold))
	  "\e[1m"
	"\e[0m"))))

(defconst odccl-indent
  '((r0 = (r4 >> 3))
    (loop
     (if (r0 == 0)
	 (break)
       ((r0 -= 1)
	(write-repeat "        "))))
    (branch (r4 & 7)
	    ""
	    " "
	    "  "
	    "   "
	    "    "
	    "     "
	    "      "
	    "       ")))

(defun odccl-select (table &optional key)
  (mapcar
   (function
    (lambda (element)
      (cond ((not (vectorp element)))
	    ((or (and (not (eq key 'sc)) (not (eq key 'x)))
		 (= (length element) 1))
	     (setq element (aref element 0)))
	    ((or (not (eq key 'x)) (= (length element) 2))
	     (setq element (aref element 1)))
	    (t
	     (setq element (aref element 2))))
      (if (stringp element)
	  (if (string= element (char-to-string (string-to-char element)))
	      (list 'write-repeat (string-to-char element))
	    (list 'write-repeat element))
	element)))
   table))

(defun odccl-make-low-tables (low-alist)
  (let ((low-table (make-vector 39 nil))
	(low-table1 (make-vector 129 nil)))
    (while low-alist
      (let ((pair (car low-alist)))
	(cond ((< (car pair) 38)
	       (aset low-table (car pair) (cdr pair)))
	      ((>= (car pair) 127)
	       (aset low-table1 (- (car pair) 127) (cdr pair)))))
      (setq low-alist (cdr low-alist)))
    (cons (append low-table nil) (append low-table1 nil))))

(defun odccl-make-ccl (low-tables &optional single-column)
  (` (2					; buffer-mag
      ((r4 = 0)
       (loop
	(read r1)
	(if (r1 == 31)
	    ((read r2)
	     (loop
	      (branch
	       r2
	       nil nil nil		; 0x1f00-0x1f02
	       ("\C-c"			; 0x1f03 stop
		(, (odccl-set 'eop)))
	       ((, (odccl-reset 'col))) ; 0x1f04 single-column
	       ((, (odccl-set 'col)))	; 0x1f05 double-column
	       ("_{"			; 0x1f06 subscript start
		(, (odccl-reset 'col)))
	       ("}"			; 0x1f07 subscript end
		(, (odccl-set 'col)))
	       nil			; 0x1f08
	       ((read r4 r0)		; 0x1f09 set indent level
		(r4 <8= r0)
		(,@ odccl-indent))
	       ((, (odccl-set 'ret))	; 0x1f0a \n
		(write ?\n)
		(read-if (r1 != 31)
			 ((,@ odccl-indent)
			  (break))
			 (read-if (r2 == 9)
				  (repeat)
				  ((,@ odccl-indent)
				   (repeat)))))
	       nil nil nil		; 0x1f0b-0d
	       ("^{"			; 0x1f0e superscript start
		(, (odccl-reset 'col)))
	       ("}"			; 0x1f0f superscript end
		(, (odccl-set 'col)))
	       nil nil			; 0x1f10,11 prohibit \n (ignored)
	       nil nil nil nil nil nil	; 0x1f12-
	       nil nil nil nil nil nil nil nil
	       nil nil nil nil nil nil nil nil ; 0x1f20-
	       nil nil nil nil nil nil nil nil
	       nil nil			; 0x1f30-0x1f31
	       ((,@ odccl-self))	; 0x1f32 caption of figure
	       ((,@ odccl-self))	; 0x1f33 caption of sound
	       nil nil nil nil		; 0x1f34-
	       nil nil nil nil nil nil nil nil
	       nil			; 0x1f40
	       ((if (, (odccl-check 'item)) ; 0x1f41
		    (if (, (odccl-check 'ret))
			("\C-c"
			 (, (odccl-set 'eop)))
		      nil)
		  ((, (odccl-set 'item))))
		(read r0 r0))
	       ((,@ odccl-self))	; 0x1f42 caption of reference
	       ((,@ odccl-self))	; 0x1f43 caption of reference
	       nil
	       ((,@ odccl-self)		; 0x1f45 figure or sound
		(read r0 r0))
	       nil nil			; 0x1f46-47
	       nil nil nil nil nil nil nil nil ; 0x1f48-4f
	       nil nil			; 0x1f50-51
	       ((,@ odccl-self)		; 0x1f52 figure data
		(,@ odccl-prtbcd) (,@ odccl-prtbcd)
		(,@ odccl-prtbcd) (,@ odccl-prtbcd)
		(,@ odccl-prtbcd) (,@ odccl-prtbcd))
	       ((,@ odccl-self)		; 0x1f53 sound data
		(,@ odccl-prtbcd) (,@ odccl-prtbcd)
		(,@ odccl-prtbcd) (,@ odccl-prtbcd)
		(,@ odccl-prtbcd) (,@ odccl-prtbcd)
		(,@ odccl-prtbcd) (,@ odccl-prtbcd))
	       nil nil nil nil		; 0x1f54-
	       nil nil nil nil nil nil nil nil
	       nil nil			; 0x1f60-0x1f61
	       ((,@ odccl-self)		; 0x1f62 reference honmon
		(,@ odccl-prtbcd) (,@ odccl-prtbcd)
		(,@ odccl-prtbcd) (,@ odccl-prtbcd)
		(,@ odccl-prtbcd) (,@ odccl-prtbcd))
	       ((,@ odccl-self)		; 0x1f63 reference menu
		(,@ odccl-prtbcd) (,@ odccl-prtbcd)
		(,@ odccl-prtbcd) (,@ odccl-prtbcd)
		(,@ odccl-prtbcd) (,@ odccl-prtbcd)))
	      (read-if (r1 != 31)
		       (break)
		       ((read r2)
			(repeat)))))
	  nil)
	;; r1 != ESC 
	(, (if (not single-column)
	       (` (if (r1 < 38)
		      (branch r1 (,@ (car low-tables)))
		    (if (r1 >= 127)
			(branch (r1 - 127) (,@ (cdr low-tables)))
		      (, odccl-jis))))
	     (` (if (r1 < 32)
		    (branch r1 (,@ (car low-tables)))
		  (if (r1 < 127)
		      (write-repeat r1)
		    (if (r1 < 160)
			(branch (r1 - 127) (,@ (cdr low-tables)))
		      ((write (, lc-ltn1))
		       (write-repeat r1))))))))
	(repeat))))))

(defconst odccl-jis-kana-table
  (mapcar
   (function (lambda (arg)
	       (cond ((null arg) nil)
		     ((= (chars-in-string arg) 1)
		      (list 'write-repeat (string-to-char arg)))
		     (t
		      (list 'write-repeat arg)))))
   '( 	nil	nil	nil	nil	nil	nil	nil	nil
	nil	nil	nil	nil	nil	nil	nil	nil
	nil	nil	nil	nil	nil	nil	nil	nil
	nil	nil	nil	nil	nil	nil	nil	nil
	nil	"(I'(B"	"(I1(B"	"(I((B"	"(I2(B"	"(I)(B"	"(I3(B"	"(I*(B"
	"(I4(B"	"(I+(B"	"(I5(B"	"(I6(B"	"(I6^(B"	"(I7(B"	"(I7^(B"	"(I8(B"
	"(I8^(B"	"(I9(B"	"(I9^(B"	"(I:(B"	"(I:^(B"	"(I;(B"	"(I;^(B"	"(I<(B"
	"(I<^(B"	"(I=(B"	"(I=^(B"	"(I>(B"	"(I>^(B"	"(I?(B"	"(I?^(B"	"(I@(B"
	"(I@^(B"	"(IA(B"	"(IA^(B"	"(I/(B"	"(IB(B"	"(IB^(B"	"(IC(B"	"(IC^(B"
	"(ID(B"	"(ID^(B"	"(IE(B"	"(IF(B"	"(IG(B"	"(IH(B"	"(II(B"	"(IJ(B"
	"(IJ^(B"	"(IJ_(B"	"(IK(B"	"(IK^(B"	"(IK_(B"	"(IL(B"	"(IL^(B"	"(IL_(B"
	"(IM(B"	"(IM^(B"	"(IM_(B"	"(IN(B"	"(IN^(B"	"(IN_(B"	"(IO(B"	"(IP(B"
	"(IQ(B"	"(IR(B"	"(IS(B"	"(I,(B"	"(IT(B"	"(I-(B"	"(IU(B"	"(I.(B"
	"(IV(B"	"(IW(B"	"(IX(B"	"(IY(B"	"(IZ(B"	"(I[(B"	"$B%n(B"	"(I\(B"
	"$B%p(B"	"$B%q(B"	"(I&(B"	"(I](B"	"(I3^(B"	"$B%u(B"	"$B%v(B"	nil)))

(defconst odccl-jis-single-col1
  (list	nil	nil	nil	nil	nil	nil	nil	nil
	nil	nil	nil	nil	nil	nil	nil	nil
	nil	nil	nil	nil	nil	nil	nil	nil
	nil	nil	nil	nil	nil	nil	nil	nil
	nil		" "		["$B!"(B" "(I$(B"]	["$B!#(B" "(I!(B"]
	","		"."		["$B!&(B" ",A7(B"]	"$B!'(B"
	";"		"?"		"!"		["$B!+(B" "(I^(B"]
	["$B!,(B" "(I_(B"]	"$B!-(B"		"$B!.(B"		["$B!/(B" ",A((B"]
	"$B!0(B"	"~"	"$B!2(B"	"$B!3(B"	"$B!4(B"	"$B!5(B"	"$B!6(B"	"$B!7(B"
	"$B!8(B"		"$B!9(B"		"$B!:(B"		"$B!;(B"
	["$B!<(B" "(I0(B"]	"$B!=(B"		["$B!>(B" ",A-(B"]	"/"
	"\\"	"$B!A(B"	"$B!B(B"	"|"	"$B!D(B"	"$B!E(B"	"`"	"'"
	"$B!H(B"	"$B!I(B"	"("	")"	"$B!L(B"	"$B!M(B"	"["	"]"
	"{"		"}"		"$B!R(B"		"$B!S(B"
	"$B!T(B"		"$B!U(B"		["$B!V(B " "(I"(B"]	["$B!W(B" "(I#(B"]
	"$B!X(B"		"$B!Y(B"		"$B!Z(B"		"$B![(B"
	"+"		"-"		["$B!^(B" ",A1(B"]	["$B!_(B" ",AW(B"]
	["$B!`(B" ",Aw(B"]	"="		"$B!b(B"		"<"
	">"		"$B!e(B"		"$B!f(B"		"$B!g(B"
	"$B!h(B"		"$B!i(B"		"$B!j(B"		["$B!k(B" ",A0(B"]
	"$B!l(B"		"$B!m(B"		"$B!n(B"		["$B!o(B" ",A%(B"]
	"$"		["$B!q(B" ",A"(B"] 	["$B!r(B" ",A#(B"]	"%"
	"$B!t(B"		"&"		"*"		"@"
	["$B!x(B" ",A'(B"]	"$B!y(B"		"$B!z(B"		"$B!{(B"
	"$B!|(B"		"$B!}(B"		"$B!~(B"		nil))

(defconst odccl-jis-single-col2
  (list	nil	nil	nil	nil	nil	nil	nil	nil
	nil	nil	nil	nil	nil	nil	nil	nil
	nil	nil	nil	nil	nil	nil	nil	nil
	nil	nil	nil	nil	nil	nil	nil	nil
	nil	"$B"!(B"	"$B""(B"	"$B"#(B"	"$B"$(B"	"$B"%(B"	"$B"&(B"	"$B"'(B"
	"$B"((B"	"$B")(B"	"$B"*(B"	"$B"+(B"	"$B",(B"	"$B"-(B"	"$B".(B"	nil
	nil	nil	nil	nil	nil	nil	nil	nil
	nil	nil	"$B":(B"	"$B";(B"	"$B"<(B"	"$B"=(B"	"$B">(B"	"$B"?(B"
	"$B"@(B"	"$B"A(B"	nil	nil	nil	nil	nil	nil
	nil		nil		"$B"J(B"		"$B"K(B"
	["$B"L(B" ",A,(B"]	"$B"M(B"		"$B"N(B"		"$B"O(B"
	"$B"P(B"	nil	nil	nil	nil	nil	nil	nil
	nil	nil	nil	nil	"$B"\(B"	"$B"](B"	"$B"^(B"	"$B"_(B"
	"$B"`(B"		"$B"a(B"		"$B"b(B"		["$B"c(B" ",A+(B"]
	["$B"d(B" ",A;(B"]	"$B"e(B"		"$B"f(B"		"$B"g(B"
	"$B"h(B"	"$B"i(B"	"$B"j(B"	nil	nil	nil	nil	nil
	nil	nil	"$B"r(B"	"$B"s(B"	"$B"t(B"	"$B"u(B"	"$B"v(B"	"$B"w(B"
	"$B"x(B"		["$B"y(B" ",A6(B"]	nil		nil
	nil		nil		"$B"~(B"		nil))

(defconst odccl-gaiji-eiwa1
  (list	nil	nil	nil	nil	nil	nil	nil	nil
	nil	nil	nil	nil	nil	nil	nil	nil
	nil	nil	nil	nil	nil	nil	nil	nil
	nil	nil	nil	nil	nil	nil	nil	nil
	nil	"$B!B(B"
	(append odccl-start-bold '("[") odccl-end-bold '((repeat)))
	(append odccl-start-bold '("]") odccl-end-bold '((repeat)))
	")"		["#e" ",0'(B"]	[":" ",0r(B"]	["#o" ",07(B"]
	["#/er/" ",0:(B"]	["#/ae/" ",0$(B"]	["#a" ",0,(B"]	["#s" ",0L(B"]
	["#v" ",0+(B"]	["#g" ",0M(B"]	["#/ng/" ",0U(B"]	["#t" ",0H(B"]
	["#E" ",0#(B"]	["#d" ",0I(B"]	["$B&5(B" ",FV(B"]	["#/oe/" ",00(B"]
	["$B'k(B" ",Li(B"]	["#'" ",0p(B"]	["#," ",0q(B"]	["$B!/(B" ",A((B"]
	"#^"		["#=" ",A/(B"]	"~"		["#u" ",B"(B"]
	["#," ",A8(B"]	["#$B!&(B" "#(I%(B"]	["$B!l(B" ",B4(B"]	["$B!m(B" ",B=(B"]
	"#/om/"		"#/t/"		["$B"r(B" ",AE(B"]	["#'a" ",Aa(B"]
	["#'c" ",Bf(B"]	["#'e" ",Ai(B"]	["#'i" ",Am(B"]	["#'o" ",As(B"]
	["#'u" ",Az(B"]	["#`a" ",A`(B"]	["#`e" ",Ah(B"]	["#`i" ",Al(B"]
	["#`o" ",Ar(B"]	["#`u" ",Ay(B"]	["#'A" ",AA(B"]	["#'E" ",AI(B"]
	["#'I" ",AM(B"]	["#'U" ",AZ(B"]	["#^a" ",Ab(B"]	["#^e" ",Aj(B"]
	["#^i" ",An(B"]	["#^o" ",At(B"]	["#\"a" ",Ad(B"]	["#\"e" ",Ak(B"]
	["#\"i" ",Ao(B"]	["#\"o" ",Av(B"]	["#\"u" ",A|(B"]	["#~a" ",Ac(B"]
	["#~n" ",Aq(B"]	["#~o" ",Au(B"]	["#=a" ",D`(B"]	["#=e" ",D:(B"]
	["#=o" ",Dr(B"]			["#/ua/" ",Bc(B"]
	(vector "#/ue/" (compose-string ",B"(Be"))
	(vector "#/uo/" (compose-string ",B"(Bo"))
	["#/vC/" ",BH(B"]	["#/vc/" ",Bh(B"]	["#/vr/" ",Bx(B"]	["#,c" ",Ag(B"]
	(vector "#$B!&(B1" (compose-string ",B(B1"))
	(vector "#$B!&(B3" (compose-string ",B(B3"))
	"^0"				["^1" ",A9(B"]
	["^2" ",A2(B"]	["^3" ",A3(B"]	"^4"		"^5"
	"^6"	"^7"	"^8"	"^9"	"^$B"+(B"	"^'"	"^+"	"^*"
	"^c"	"^h"	"^m"	"^n"	"^o"	"^r"	"^s"	nil))

(defconst odccl-gaiji-eiwa2
  (list	nil	nil	nil	nil	nil	nil	nil	nil
	nil	nil	nil	nil	nil	nil	nil	nil
	nil	nil	nil	nil	nil	nil	nil	nil
	nil	nil	nil	nil	nil	nil	nil	nil
	nil	"_*"	"_1"	"_2"	"_3"	"_4"	"_n"	["_I" ",0!(B"]
	["#~#a" ",0<(B"]	["#~#E" ",0;(B"]	["#~#o" ",0>(B"]	["#~#/oe/" ",0=(B"]
	(append odccl-start-bold '("(") '((repeat)))
	(append '(")") odccl-end-bold '((repeat)))
	["|" ",A&(B"]
	(append odccl-start-inv '("[") '((repeat)))
	(append '("]") odccl-end-inv '((repeat)))
	nil))

(defconst odccl-gaiji-eiwa3
  (list	nil	nil	nil	nil	nil	nil	nil	nil
	nil	nil	nil	nil	nil	nil	nil	nil
	nil	nil	nil	nil	nil	nil	nil	nil
	nil	nil	nil	nil	nil	nil	nil	nil
	nil		["#/$BLZH}(B/" "$(DDH(B"]["#/$B9C1)(B/" "$(DU8(B"]["#/$BQRVx3-3-=w(B/" "$(DN0(B"]
	["#/$B=.;3(B/" "$(DVv(B"]["#/$B%5:s(B/" "$(DXt(B"]["#/$B;e6g(B/" "$(DS`(B"]["#/$BB-IU(B/" "$(D_h(B"]
	["#/$BGOTo(B/" "$(Di%(B"]["#/$BGO@'(B/" "$(Di@(B"]"#/$B2P@P(B/"	["#/$B:MVu(B/" "$(D?Y(B"]
	["#/$BLZ>e2<(B/" "$(DC_(B"]		"#L"
	(vector "$B"+"*(B" (compose-string "$B"+"*(B"))	["#$B"^(B" "$(D"/(B"]
	"#/(IEA-WY(B/"	"#&"
	(vector "($B#C(B)" (compose-string "$B"~#c(B"))
	["_$B",(B" "$(C"V(B"]
	(vector "$B(@(B" (compose-string "$B!B(!(B"))
	(vector "$B(@(B" (compose-string "$B!B(!(B"))
	"#$BRD(B"				"#/$B"$"&(B/"
	(vector "#$B!b(B" (compose-string "$B("!a(B"))
	"#$B"v(B"
	["#/$B;X$5$7(B/" "$(C"Q(B"]		"#/$B>e(IL*YO0@(B/"
	"#/$B2<(IL*YO0@(B/"	"#S"		"#$B#X(B"		"#/fi/"
	"#/$B!B!{!B(B/"	"#$B"e(B"
	(vector "(C)" (compose-string "$B#c""(B"))
	(vector "(U)" (compose-string "$B#u""(B"))
	nil))

(defconst odccl-gaiji-chujiten1
  (list	nil	nil	nil	nil	nil	nil	nil	nil
	nil	nil	nil	nil	nil	nil	nil	nil
	nil	nil	nil	nil	nil	nil	nil	nil
	nil	nil	nil	nil	nil	nil	nil	nil
	nil		["#/ae/" ",0$(B"]	"#xa1#x22"	"#xa1#x23"
	["#/er/" ",0:(B"]	["#e" ",0'(B"]	"#xa1#x26"	["#t" ",0H(B"]
	["#d" ",0I(B"]	["#s" ",0L(B"]	"#xa1#x2a"	["#/ng/" ",0U(B"]
	[":" ",0r(B"]	["#a" ",0,(B"]	"#xa1#x2e"	"#xa1#x2f"
	"#xa1#x30"	"#xa1#x31"	"#xa1#x32"	"#xa1#x33"
	"#xa1#x34"	"#xa1#x35"	"#xa1#x36"	"#xa1#x37"
	"#xa1#x38"	"#xa1#x39"	"#xa1#x3a"	"#xa1#x3b"
	"#xa1#x3c"	"#xa1#x3d"	"#xa1#x3e"	"#xa1#x3f"
	"#xa1#x40"	"#xa1#x41"	"#xa1#x42"	"#xa1#x43"
	"#xa1#x44"	["#'#/ae/" ",0p$(B"]"#xa1#x46"	["#'#v" ",0p+(B"]
	["#'#e" ",0p'(B"]	"#xa1#x49"	["#'#E" ",0p#(B"]	["#'#a" ",0p,(B"]
	"#xa1#x4c"	"#xa1#x4d"	"#xa1#x4e"	"#xa1#x4f"
	"#xa1#x50"	"#xa1#x51"	"#xa1#x52"	"#xa1#x53"
	"#xa1#x54"	"#xa1#x55"	"#xa1#x56"	"#xa1#x57"
	"#xa1#x58"	"#xa1#x59"	"#xa1#x5a"	"#xa1#x5b"
	"#xa1#x5c"	"#xa1#x5d"	"#xa1#x5e"	"#xa1#x5f"
	"#xa1#x60"	"#xa1#x61"	"#xa1#x62"	"#xa1#x63"
	"#xa1#x64"	"#xa1#x65"	"#xa1#x66"	"#xa1#x67"
	"#xa1#x68"	"#xa1#x69"	"#xa1#x6a"	"#xa1#x6b"
	"#xa1#x6c"	"#xa1#x6d"	"#xa1#x6e"	"#xa1#x6f"
	"#xa1#x70"	"#xa1#x71"	"#xa1#x72"	"#xa1#x73"
	"#xa1#x74"	"#xa1#x75"	"#xa1#x76"	"#xa1#x77"
	"#xa1#x78"	"#xa1#x79"	"#xa1#x7a"	"#xa1#x7b"
	"#xa1#x7c"	"#xa1#x7d"	"#xa1#x7e"	"#xa1#x7f"))

(defconst odccl-gaiji-chujiten2
  (list	nil	nil	nil	nil	nil	nil	nil	nil
	nil	nil	nil	nil	nil	nil	nil	nil
	nil	nil	nil	nil	nil	nil	nil	nil
	nil	nil	nil	nil	nil	nil	nil	nil
	nil		"#xa2#x21"	"#xa2#x22"	"#xa2#x23"
	"#xa2#x24"	"#xa2#x25"	"#xa2#x26"	"#xa2#x27"
	"#xa2#x28"	"#xa2#x29"	"#xa2#x2a"	"#xa2#x2b"
	"#xa2#x2c"	"#xa2#x2d"	"#xa2#x2e"	"#xa2#x2f"
	"#xa2#x30"	"#xa2#x31"	"#xa2#x32"	"#xa2#x33"
	"#xa2#x34"	"#xa2#x35"	"#xa2#x36"	"#xa2#x37"
	"#xa2#x38"	"#xa2#x39"	"#xa2#x3a"	"#xa2#x3b"
	"#xa2#x3c"	"#xa2#x3d"	"#xa2#x3e"	"#xa2#x3f"
	"#xa2#x40"	"#xa2#x41"	"#xa2#x42"	"#xa2#x43"
	"#xa2#x44"	"#xa2#x45"	"#xa2#x46"	"#xa2#x47"
	"#xa2#x48"	"#xa2#x49"	"#xa2#x4a"	"#xa2#x4b"
	"#xa2#x4c"	"#xa2#x4d"	"#xa2#x4e"	"#xa2#x4f"
	"#xa2#x50"	"#xa2#x51"	"#xa2#x52"	"#xa2#x53"
	"#xa2#x54"	"#xa2#x55"	"#xa2#x56"	"#xa2#x57"
	"#xa2#x58"	"#xa2#x59"	"#xa2#x5a"	"#xa2#x5b"
	"#xa2#x5c"	"#xa2#x5d"	"#xa2#x5e"	"#xa2#x5f"
	"#xa2#x60"	"#xa2#x61"	"#xa2#x62"	"#xa2#x63"
	"#xa2#x64"	"#xa2#x65"	"#xa2#x66"	"#xa2#x67"
	"#xa2#x68"	"#xa2#x69"	"#xa2#x6a"	"#xa2#x6b"
	"#xa2#x6c"	"#xa2#x6d"	"#xa2#x6e"	"#xa2#x6f"
	"#xa2#x70"	"#xa2#x71"	"#xa2#x72"	"#xa2#x73"
	"#xa2#x74"	"#xa2#x75"	"#xa2#x76"	"#xa2#x77"
	"#xa2#x78"	"#xa2#x79"	"#xa2#x7a"	"#xa2#x7b"
	"#xa2#x7c"	"#xa2#x7d"	"#xa2#x7e"	"#xa2#x7f"))

(defconst odccl-gaiji-chujiten3
  (list	nil	nil	nil	nil	nil	nil	nil	nil
	nil	nil	nil	nil	nil	nil	nil	nil
	nil	nil	nil	nil	nil	nil	nil	nil
	nil	nil	nil	nil	nil	nil	nil	nil
	nil		"#xA1#x21"	"#xA1#x22"	"#xA1#x23"
	"#xA1#x24"	"#xA1#x25"	"#xA1#x26"	"#xA1#x27"
	"#xA1#x28"	"#xA1#x29"	"#xA1#x2A"	"#xA1#x2B"
	"#xA1#x2C"	"#xA1#x2D"	"#xA1#x2E"	"#xA1#x2F"
	"#xA1#x30"	"#xA1#x31"	"#xA1#x32"	"#xA1#x33"
	"#xA1#x34"	"#xA1#x35"	"#xA1#x36"	"#xA1#x37"
	"#xA1#x38"	"#xA1#x39"	"#xA1#x3A"	"#xA1#x3B"
	"#xA1#x3C"	"#xA1#x3D"	"#xA1#x3E"	"#xA1#x3F"
	"#xA1#x40"	"#xA1#x41"	"#xA1#x42"	"#xA1#x43"
	"#xA1#x44"	"#xA1#x45"	"#xA1#x46"	"#xA1#x47"
	"#xA1#x48"	"#xA1#x49"	"#xA1#x4A"	"#xA1#x4B"
	"#xA1#x4C"	"#xA1#x4D"	"#xA1#x4E"	"#xA1#x4F"
	"#xA1#x50"	"#xA1#x51"	"#xA1#x52"	"#xA1#x53"
	"#xA1#x54"	"#xA1#x55"	"#xA1#x56"	"#xA1#x57"
	"#xA1#x58"	"#xA1#x59"	"#xA1#x5A"	"#xA1#x5B"
	"#xA1#x5C"	"#xA1#x5D"	"#xA1#x5E"	"#xA1#x5F"
	"#xA1#x60"	"#xA1#x61"	"#xA1#x62"	"#xA1#x63"
	"#xA1#x64"	"#xA1#x65"	"#xA1#x66"	"#xA1#x67"
	"#xA1#x68"	"#xA1#x69"	"#xA1#x6A"	"#xA1#x6B"
	"#xA1#x6C"	"#xA1#x6D"	"#xA1#x6E"	"#xA1#x6F"
	"#xA1#x70"	"#xA1#x71"	"#xA1#x72"	"#xA1#x73"
	"#xA1#x74"	"#xA1#x75"	"#xA1#x76"	"#xA1#x77"
	"#xA1#x78"	"#xA1#x79"	"#xA1#x7A"	"#xA1#x7B"
	"#xA1#x7C"	"#xA1#x7D"	"#xA1#x7E"	"#xA1#x7F"))

(defconst odccl-gaiji-kojien1
  (list	nil	nil	nil	nil	nil	nil	nil	nil
	nil	nil	nil	nil	nil	nil	nil	nil
	nil	nil	nil	nil	nil	nil	nil	nil
	nil	nil	nil	nil	nil	nil	nil	nil
	nil		["#'a" ",Aa(B"]	["#`a" ",A`(B"]	["#^a" ",Ab(B"]
	["#\"a" ",Ad(B"]	["#~a" ",Ac(B"]	["#=a" ",D`(B"]	"#.a"
	["#'c" ",Bf(B"]	["#/vc/" ",Bh(B"]	"#.d"		["#'e" ",Ai(B"]
	["#`e" ",Ah(B"]	["#^e" ",Aj(B"]	["#\"e" ",Ak(B"]	["#=e" ",D:(B"]
	"#`g"		"#.h"		["#'i" ",Am(B"]	["#^i" ",An(B"]
	["#\"i" ",Ao(B"]	["#=i" ",Do(B"]	["#.l" ",D6(B"]	"#.m"
	["#'n" ",Bq(B"]	["#~n" ",Aq(B"]	["#=n" "#,A/(Bn"]	["#.n" ",Dq(B"]
	["#'o" ",As(B"]	["#`o" ",Ar(B"]	["#^o" ",At(B"]	["#\"o" ",Av(B"]
	["#~o" ",Au(B"]	["#=o" ",Dr(B"]	"#/vo/"		["#.r" ",D3(B"]
	"#./vr/"	["#'s" ",B6(B"]	"#.s"		["#/vs/" ",B9(B"]
	["#.t" ",B~(B"]	["#'u" ",Az(B"]	["#/uu/" ",C}(B"]	["#\"u" ",A|(B"]
	["#=u" ",D~(B"]	"#/vu/"		["#\"y" ",M(B"]	"#'z"
	"#.z"		["#=" ",A/(B"]	"#."		["#i" ",C9(B"]
	"~"		["#,c" ",Ag(B"]	["#/u/" ",B"(B"]	"#/`/"
	"#/g/"		["'" ",A4(B"]	"#/'/"		["#/ng/" ",D?(B"]
	["#/v/" ",B7(B"]	"#g"		":"		["#/l/" ",B3(B"]
	"#s"		"#o"		["#0" ",Ax(B"]	"#/..t/"
	"#f"		["#t" ",Fh(B"]	["#d" ",Ap(B"]	"??"
	"$B%p(B"		"$B%q(B"		"#F"		"#'"
	"#`"		["#\"" ",A((B"]	"#^^"		"$B"e(B"
	["#/3$B"e(B/" ",A3$B"e(B"]["#'A" ",AA(B"]	"#$B&U(B"		["#\"O" ",AV(B"]
	["((" ",A+(B"]	["))" ",A;(B"]	["#\"U" ",A\(B"]	["#=A" ",D@(B"]
	"#/5$B"e(B/"	"#/n$B"e(B/"	["#^I" ",AN(B"]	["#/L/" ",B#(B"]
	["#/AE/" ",AF(B"]	["#h" ",C1(B"]	["#'E" ",AI(B"]	nil))

(defconst odccl-gaiji-kojien2
  (list	nil	nil	nil	nil	nil	nil	nil	nil
	nil	nil	nil	nil	nil	nil	nil	nil
	nil	nil	nil	nil	nil	nil	nil	nil
	nil	nil	nil	nil	nil	nil	nil	nil
	nil		["#\"A" ",AD(B"]	["#'S" ",B&(B"]	["#=I" ",DO(B"]
	["$B"r(B" ",AE(B"]
	(append odccl-start-inv '("[") '((repeat)))
	(append '("]") odccl-end-inv '((repeat)))
	(append odccl-start-bold '("[") '((repeat)))
	(append '("]") odccl-end-bold '((repeat)))
	(append odccl-start-bold '("[") '((repeat)))
	(append '("]") odccl-end-bold '((repeat)))
	["#=O" ",DR(B"]
	"^$B"*(B"		["#,S" ",B:(B"]	["#,e" ",Bj(B"]	["#vC" ",BH(B"]
	["#~N" ",CQ(B"]	"#.Z"		"#.H"		"#.R"
	"#.S"		"#.D"
	(append odccl-start-bold '("'") odccl-end-bold '((repeat)))
	["#vn" ",Br(B"]
	["#'y" ",A}(B"]	["#`u" ",Ay(B"]	"#Y"
	(append odccl-start-bold '("<") '((repeat)))
	(append '(">") odccl-end-bold '((repeat)))
	["#oa" ",Ae(B"]	"#_b"	nil))

(defconst odccl-gaiji-kojien3
  (list	nil	nil	nil	nil	nil	nil	nil	nil
	nil	nil	nil	nil	nil	nil	nil	nil
	nil	nil	nil	nil	nil	nil	nil	nil
	nil	nil	nil	nil	nil	nil	nil	nil
	nil
	(vector "$B"+"*(B" (compose-string "$B"+"*(B"))
	(append odccl-start-inv '("$BJ8(B") odccl-end-inv '((repeat)))
	"#$B!?(B"
	"#$B!@(B"		["#/$B%5:s(B/" "$(DXt(B"]"#$B!9(B"		"#/$BLZ9'(B/"
	["#/$BLZFb(B/" "$(DC.(B"]["#/$BLZ?b(B/" "$(DD.(B"]["#/$BLZ?'(B/" "$(DCP(B"]["#/$BEZ[4(B/" "$(D7[(B"]
	["#/$BLZ>e2<(B/" "$(DC_(B"]		["#/$BJFRT(B/" "$(DSF(B"]
	["#/$B%$H,7n(B/" "$(D0{(B"]		["#/$BEZ?b(B/" "$(D7t(B"]
	["#/$B:MO.(B/" "$(D?g(B"]		["#/$B7JJG(B/" "$(Dh6(B"]
	["#/$BQRVx3-3-=w(B/" "$(DN0(B"]		["#/$B;e7=(B/" "$(DSd(B"]
	["#/$BLgCn(B/" "$(Df,(B"]["#/$BC]WP(B/" "$(DR\(B"]["#/$B7j2L(B/" "$(DQ\(B"]["III" "$(C%2(B"]
	["#/$B6b8r(B/" "$(Dcr(B"]["#/$B8}:P(B/" "$(D6M(B"]"#/$B!@(B\"/"
	(append odccl-start-bold '("$B!L(B") '((repeat)))
	(append '("$B!M(B") odccl-end-bold '((repeat)))
	(vector "($B#C(B)" (compose-string "$B"~#c(B") '((repeat)))
	(append odccl-start-bold '("$B"*(B") odccl-end-bold '((repeat)))
	["#/$B6bAc(B/" "$(De<(B"]
	["#/$BG7sU(B/" "$(Daj(B"]		["IV" "$(C%3(B"]
	["#/$B%$>.7n(B/" "$(D1?(B"]		["#/$B6uD;(B/" "$(Dl!(B"]
	["#/$B2SO%(B/" "$(DQN(B"]["#/$B:M08(B/" "$(D?x(B"]["#/$B</>O(B/" "$(Dlg(B"]["#/$BLZL5(B/" "$(DE&(B"]
	["#/$BEP&B(B/" "$(DbG(B"]"->"		["#/$B8}0&(B/" "$(D6P(B"]["$B#I(B" "$(C%0(B"]
	["II" "$(C%1(B"]	"??"		"#/$B%5W0(B/"	["VI" "$(C%5(B"]
	"#$BRD(B"		"$B!!(B"		"$B2s(B"
	(append odccl-start-bold '("--") odccl-end-bold '((repeat)))
	(append odccl-start-bold '("$B!](B") odccl-end-bold '((repeat)))
	"??"
	(append odccl-start-inv '("$B!Z(BG$B![(B") odccl-end-inv '((repeat)))
	"#/JAS(IO08(B/"
	"$B(1(/(B"		"($B8x(B)"		"#/JIS(IO08(B/"	"$B(-(B"
	"#/$B<V\F(B/"	"??"		"#/$B%5W0P&(B/"
	(append odccl-start-bold odccl-start-inv
		'("$B"*(B") odccl-end-inv odccl-end-bold '((repeat)))
	(append odccl-start-inv '("$B"*(B") odccl-end-inv '((repeat)))
	nil))

(defun odccl-low-alist-japanese (key)
  (list (cons 33
	      (` (if (, (odccl-check 'col))
		     (, odccl-jis)
		   ((read-branch r2
				 (,@ (odccl-select
				      odccl-jis-single-col1 key)))
		    (repeat)))))
	(cons 34
	      (` (if (, (odccl-check 'col))
		     (, odccl-jis)
		   ((read-branch r2
				 (,@ (odccl-select
				      odccl-jis-single-col2 key)))
		    (repeat)))))
	(cons 35
	      (` (if (, (odccl-check 'col))
		     (, odccl-jis)
		   ((read r2)
		    (write-repeat r2)))))
	(cons 36 odccl-jis)
	(cons 37
	      (if (not key)
		  odccl-jis
		(` (if (, (odccl-check 'col))
		       (, odccl-jis)
		     ((read-branch r2
				   (,@ odccl-jis-kana-table))
		      (repeat))))))))

(defun odccl-make-ccl-eiwa (&optional key)
  (odccl-make-ccl
   (odccl-make-low-tables
    (append
     (odccl-low-alist-japanese key)
     (list (cons 161
		 (` ((if (, (odccl-check 'col))
			 (read-branch
			  r2
			  (,@ (odccl-select odccl-gaiji-eiwa3 key)))
		       (read-branch
			r2
			(,@ (odccl-select odccl-gaiji-eiwa1 key))))
		     (repeat))))
	   (cons 162 
		 (` ((read-branch
		 r2
		 (,@ (odccl-select odccl-gaiji-eiwa2 key)))
		     (repeat)))))))))

(defun odccl-make-ccl-chujiten (&optional key)
  (odccl-make-ccl
   (odccl-make-low-tables
    (append
     (odccl-low-alist-japanese key)
     (list (cons 161
		 (` ((if (, (odccl-check 'col))
			 (read-branch
			  r2
			  (,@ (odccl-select odccl-gaiji-chujiten3 key)))
		       (read-branch
			r2
			(,@ (odccl-select odccl-gaiji-chujiten1 key))))
		     (repeat))))
	   (cons 162 
		 (` ((read-branch
		 r2
		 (,@ (odccl-select odccl-gaiji-chujiten2 key)))
		     (repeat)))))))))

(defun odccl-make-ccl-kojien (&optional key)
  (odccl-make-ccl
   (odccl-make-low-tables
    (append
     (odccl-low-alist-japanese key)
     (list (cons 161
		 (` ((read-branch
		      r2
		      (,@ (odccl-select odccl-gaiji-kojien1 key)))
		     (repeat))))
	   (cons 162
		 (` ((read-branch
		      r2 
		      (,@ (odccl-select odccl-gaiji-kojien2 key)))
		     (repeat))))
	   (cons 164
		 (` ((read-branch
		      r2 
		      (,@ (odccl-select odccl-gaiji-kojien3 key)))
		     (repeat)))))))))

(defmacro odccl-compile (func)
  (list 'cons
	(ccl-compile (funcall func 'x))
	(ccl-compile (funcall func))))

)					; end of eval-when-compile

(defconst od-ccl-eiwa 
  (if od-chujiten-flag
      (odccl-compile odccl-make-ccl-chujiten)
    (odccl-compile odccl-make-ccl-eiwa)))

(defconst od-ccl-kojien
  (odccl-compile odccl-make-ccl-kojien))

(provide 'diclookup-mule)

;;; diclookup-mule.el ends here
