;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; $Header: diclookup.el,v 1.3 92/07/08 11:42:08 ojo Exp $
;;;
;;;                    Copyright (C) Shingo NISHIOKA, 1991
;;;                       nishioka@sanken.osaka-u.ac.jp
;;;
;;;
;;; Modified by SAITO Yutaka (yutaka@sys1.cpg.sony.co.jp)
;;;
;;; Comments and/or bug reports about this interface should be directed to:
;;; 
;;;     SAITO Yutaka           <yutaka@sys1.cpg.sony.co.jp>
;;;     4-14-1 Asahi-cho       +81 462 30 5646
;;;     Atsugi-shi Kanagawa    Generally available on IRC as "Yutaka"
;;;
;; This file is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY.  No author or distributor
;; accepts responsibility to anyone for the consequences of using it
;; or for whether it serves any particular purpose or works at all,
;; unless he says so in writing.  Refer to the GNU Emacs General Public
;; License for full details.

;; Everyone is granted permission to copy, modify and redistribute
;; this file, but only under the conditions described in the
;; GNU Emacs General Public License.   A copy of this license is
;; supposed to have been given to you along with GNU Emacs so you
;; can know your rights and responsibilities.  It should be in a
;; file named COPYING.  Among other things, the copyright notice
;; and this notice must be preserved on all copies.

;;; 92.11.18 modified for Mule Ver.0.9.7
;;;			by M.Higashida <manabu@sigmath.osaka-u.ac.jp>
;;;	$BK^Nc$d1|IU$rI=<($5$;$k$HJ8;z2=$1$9$k%P%0$N%U%#%/%9(B
;;;	read-hiragana-string $B$K4X$9$kJQ99$bF~$C$F$$$^$9!#(B
;;; 93.2.11  modified for Mule Ver.0.9.7.1 by T.Ito <toshi@his.cpl.melco.co.jp>
;;;	In od:really-change-dictionary, avoid multiple connection.


(provide 'diclookup)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;; Set up for Diclookup ;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; setup in your startup file:
;(setq dserver-host-name "bcdserv")
;(setq od:*window-config* 6)
;(setq od:*body-fillcolumn* nil)
;(setq od:*select-unique-entry-immediately* t)
;(setq od:*scroll-step* 'half)
;(setq od:*default-jisyo* "eiwa")
;(setq od:*fep-type* 'egg)
;(autoload 'online-dictionary "diclookup" "Online dictionary." t nil)
;(autoload 'od:lookup-pattern-edit "diclookup" "Look up a word." t nil)
;
;(global-set-key "\C-ho" 'od:lookup-pattern-edit)
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;; End of Set up ;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defconst od:version
  "$Revision: 1.3.1 $"
  "version of current diclookup.el")
(defun od:version () (interactive) 
  (message (concat "diclookup.el Version " od:version)))
(defvar dserver-current-host-name nil "dserver name current selected.")
(defvar dserver-host-name nil "dserver name where you want to connect.")
(defvar dserver-host-name-kojien nil "dserver name for kojien.")
(defvar dserver-protocol "ndtp" "dserver protocol name.")
(defvar dserver-buffer " *ndtp*" "Buffer associated with dserver process.")
(defvar od:*dict-header-buffer-name* 
  "*Dict-Header*" "Buffer name for Dict Header.")
(defvar od:*dict-header-buffer* nil "Buffer for Dict Header.")
(defvar od:*dict-body-buffer-name* 
  "*Dict-Body*" "Buffer name for Dict Body.")
(defvar od:*dict-body-buffer* nil "Buffer for Dict Body.")

(defvar od:*window-config* nil)
(defvar od:*body-fillcolumn* nil)
(defvar od:*select-unique-entry-immediately* t)
(defvar od:*scroll-step* 'full)
(defvar od:*eiwa-summary-mode* nil)
(defvar od:*default-jisyo* nil)
(defvar od:*fep-type* 'egg)		;egg, iroha, t-code or no-fep
(defvar od:*ispell-ver-3* t)    ;whether you use the ispell3 (beta) or not
(defvar od:*show-help-automatically* nil)
(defvar od:*tcode-toggle-char* ?\034)

;;; OD-Log
(defvar od:*log-buffer* "*OD-Log*");; "For debuging only"
(defvar od:*debug-flag* nil)
;;;

(defvar mode-line-iroha-mode-in-minibuffer "    " "global variable")

;;; Caution: No variables allowed to be buffer-local
(defvar od:*previous-config* nil)
(defvar od:*dict* nil)
(defvar od:*headers*)

(defvar od:*current-entry* -1)

(defvar od:*input-by-kanji* nil)
(defvar od:*current-jisyo* nil)
(defvar od:*current-jisyo-pretty-name* "")
(defconst od:*jisyo* '(("eiwa" "eiwa") ("waei" "waei") ("kojien" "kojien")))
(defconst od:*dict-name*
    '(("eiwa" . "$B?71QOBCf<-E5(B")
      ("waei" . "$B?7OB1QCf<-E5(B")
      ("kojien" . "$B9-<-1q(B")))
(defconst od:*is-use-kanji* '((?a . nil) (?A . nil) (?k . t) (?K . t)))
(defconst od:*default-index-type-dict*
    '(("eiwa" . ?a) ("waei" . ?k) ("kojien" . ?k)))
(defvar od:*index-types*)
(defvar od:*default-index-type*)


;;;
(cond ((and (boundp 'NEMACS) NEMACS)
       (defun od:*japanese-p* (word) (string-match "[\200-\377]" word))
       )
      ((and (boundp 'MULE) MULE)
       (defun od:*japanese-p* (word) 
	 (let (mc-flag)
	   (string-match "[\200-\377]" word)))
       ))

;;; LOG
(defun od:*quote-null* (str)
  (mapconcat (function (lambda (elt) (if (= elt 0) "^@"
				       (char-to-string elt))))
	     str ""))

(defun od:*log* (debug &rest args)
  (and debug
       (save-excursion
	 (set-buffer (get-buffer-create od:*log-buffer*))
	 (goto-char (point-max))
	 (insert (apply (function format) args)))))


;;; dict filter

(defvar od:*current-dict-filter-func* nil)

(defconst od:*dict-filter-func*
    '(("eiwa" . (eiwa-reformatter od:filter-buffer eiwa-make-summary))
      ("waei" . (od:filter-buffer))
      ("kojien" . (od:filter-buffer))))

;; eiwa
;;
;; "$B!%(B [1|2|3|4|5|6|7|8|9] " 
;; "\($B!%(B\) \([1-9]a? \)" -> "\1\n\n\2"
;;       $BA43Q%T%j%*%I(B, $BH>3Q%9%Z!<%9(B, $BH>3Q?t;z(B, $BH>3Q%9%Z!<%9$N%7!<(B
;;       $B%1%s%9$,$"$C$?$i?t;z$NA0$G2~9T#28D(B. 
;;       $B?t;z$N8e$K(B"a"$B$,F~$k$3$H$,$"$k(B. 
;; $B@hF,$NJ8;z$,!"4'!"4V!"F0!"L>!"I{!"7A!"Be!"@\Hx!"@\F,!"A0!"=uF0(B $B$N(B
;; $B$$$:$l$+$G!"$=$ND>8e$,(B($B$G$J$$$J$i!"(B
;; $B$=$3$G2~9T!#(B ($B!](B $B$,$=$NA0$K$"$k$+$b$7$l$J$$!#(B)
;; "^\($B!](B \)?\($B@\(B\|$B4'(B\|$B4V(B\|$BF0(B\|$BL>(B\|$BI{(B\|$B7A(B\|$BBe(B\|$B@\Hx(B\|$B@\F,(B\|$BA0(B\|$B=uF0(B\)\( [^(]\)"
;; -> "\1\2\n\3"
;;
;; "^\($B!](B \)?\($B@\(B\|$B4'(B\|$B4V(B\|$BF0(B\|$BL>(B\|$BI{(B\|$B7A(B\|$BBe(B\|$B@\Hx(B\|$B@\F,(B\|$BA0(B\|$B=uF0(B\)\( ([^)]*[ $B<+B>(B]*)\)\([1A]\)"
;; -> "\1\2\3\n\4"
;; 
;; "$B!%(B [b|c|d|e|f|...] "
;; "\($B!%(B\) \([b-z] \)" -> "\1\n\2"
;;       $BA43Q%T%j%*%I(B, $BH>3Q%9%Z!<%9(B, a$B0J30$N<c$$H>3Q1Q;z(B, $BH>3Q(B
;;       $B%9%Z!<%9$N%7!<%1%s%9$,$"$C$?$iH>3Q1Q;z$NA0$G2~9T#18D(B. 

;; " $B!](B " -> "\n $B!](B "

(defvar od:*dict-filter*
    '(("eiwa" . (("\\($B!%(B\\) \\([1-9][0-9]?[a(]? \\)" . "\\1\n\n  \\2")
		 ("\\($B!%(B\\) \\([bcdefg] \\)" . "\\1\n  \\2")
		 ("\015\005" . "\n  ")
		 (" \\($B!](B[\\[ ]\\)" . "\n\n\n \\1")))
      ("waei" . (("\015\005" . "\n\n  ")
		 ("$B!?(B " . "\n $B!?(B ")
		 ("$B"y(B" . "\n$B"y(B")
		 ("$B!%!?(B" . "$B!%(B\n $B!?(B")))
      ("kojien" . (("\015\005" . "\n\n  ")
		   ("\\(([1-9][0-9]*)\\)" . "\n \\1")))))

(defvar od:*current-dict-filter* nil)

(defun od:filter-buffer ()
  (let ((tmp od:*current-dict-filter*))
    (while tmp
      (goto-char 1)
      (while (re-search-forward (car (car tmp)) nil t nil)
	(replace-match (cdr (car tmp)) t nil))
      (setq tmp (cdr tmp)))))

(defun eiwa-reformatter ()
  (goto-char 1)
  (if (re-search-forward "\\[[^]]*\\]" nil t nil)
      (insert "\n\n")))

(defun eiwa-make-summary ()
  (goto-char 1)
  (if od:*eiwa-summary-mode*
      (while (re-search-forward "$B!'(B" nil t nil)
	(backward-char 1)
	(kill-line))))



(defvar odic-mode-syntax-table nil
  "Syntax table used while in odic mode.")

(defvar odic-mode-abbrev-table nil
  "Abbrev table used while in odic mode.")
(define-abbrev-table 'odic-mode-abbrev-table ())

(if odic-mode-syntax-table
    ()
  (setq odic-mode-syntax-table (make-syntax-table)))

(defvar odic-mode-map nil "")
(if odic-mode-map
    ()
  (setq odic-mode-map (make-sparse-keymap))
  (define-key odic-mode-map "f" 'od:lookup-pattern)
  (define-key odic-mode-map "i" 'od:lookup-pattern-with-ispell)
  (define-key odic-mode-map "." 'od:current-entry)
  (define-key odic-mode-map " " 'od:scroll-entry-up)
  (define-key odic-mode-map "\177" 'od:scroll-entry-down)
  (define-key odic-mode-map "\C-d" 'od:scroll-entry-up-half)
  (define-key odic-mode-map "\C-u" 'od:scroll-entry-down-half)
;  (define-key odic-mode-map "h" 'od:show-headers)
  (define-key odic-mode-map "h" 'backward-char)
  (define-key odic-mode-map "j" 'next-line)
  (define-key odic-mode-map "k" 'previous-line)
  (define-key odic-mode-map "l" 'forward-char)
  (define-key odic-mode-map "w" 'forward-word)
  (define-key odic-mode-map "b" 'backward-word)
  (define-key odic-mode-map "p" 'od:previous-entry)
  (define-key odic-mode-map "n" 'od:next-entry)
  (define-key odic-mode-map "e" 'od:lookup-pattern-edit)
  (define-key odic-mode-map "?" 'od:help)
  (define-key odic-mode-map "q" 'od:quit)
  (define-key odic-mode-map "Q" 'od:really-quit)
  (define-key odic-mode-map "R" 'od:really-restart)
  (define-key odic-mode-map "s" 'od:select-entry)
  (define-key odic-mode-map "v" 'od:version)
  (define-key odic-mode-map "C" 'od:change-dictionary)
  (define-key odic-mode-map "E" 'od:toggle-eiwa-summary-mode)
  (define-key odic-mode-map "H" 'od:show-hanrei)
  (define-key odic-mode-map "O" 'od:show-okuduke)
  (define-key odic-mode-map "1" 'od:direct-select-entry)
  (define-key odic-mode-map "2" 'od:direct-select-entry)
  (define-key odic-mode-map "3" 'od:direct-select-entry)
  (define-key odic-mode-map "4" 'od:direct-select-entry)
  (define-key odic-mode-map "5" 'od:direct-select-entry)
  (define-key odic-mode-map "6" 'od:direct-select-entry)
  (define-key odic-mode-map "7" 'od:direct-select-entry)
  (define-key odic-mode-map "8" 'od:direct-select-entry)
  (define-key odic-mode-map "9" 'od:direct-select-entry))

(defun od:set-mode-line-format ()
  (make-variable-buffer-local 'mode-line-format)
  (let ((dic-mode-line-format
	 '(""
	   mode-line-modified
	   mode-line-nemacs-header
	   " " od:*current-jisyo-pretty-name*
	   "   " global-mode-string "  %[("
	   (if (boundp 'NEMACS)
	       (kanji-flag ("%c" ":"))
	     (if (boundp 'MULE)
		 (mc-flag ("%c" ":"))))
	   mode-name minor-mode-alist "%n"
	   mode-line-process ")%]--" (-3 . "%p") "-%-")))
    (cond ((eql od:*fep-type* 'egg)
	   (setq mode-line-format
		 (cons '(
			 (if (boundp 'NEMACS)
			     kanji-flag
			   (if (boundp 'MULE)
			       mc-flag))
			 ((minibuffer-window-selected
			   (display-minibuffer-mode "m" " ")
			   " ")
			  "["
			  (minibuffer-window-selected
			   (display-minibuffer-mode
			    mode-line-egg-mode-in-minibuffer
			    mode-line-egg-mode)
			   mode-line-egg-mode)
			  "]"))
		       dic-mode-line-format)))
          ((eql od:*fep-type* 'iroha)
	   (setq mode-line-format
		 (cons '(
			 (if (boundp 'NEMACS)
			     kanji-flag
			   (if (boundp 'MULE)
			       mc-flag))
			 ((minibuffer-window-selected
			   (display-minibuffer-mode "m" " ")
			   " ")
			  "["
			  (minibuffer-window-selected
			   (display-minibuffer-mode
			    mode-line-iroha-mode-in-minibuffer
			    mode-line-iroha-mode)
			   mode-line-iroha-mode)
			  "]"))
		       dic-mode-line-format)))
	  ((eql od:*fep-type* 't-code)
	   (if (not (eq 
		     (if (boundp 'NEMACS)
			 'kanji-flag
		       (if (boundp 'MULE)
			   'mc-flag))
			(car (cdr (default-value 'mode-line-format)))))
	       (setq mode-line-format
		     (cons
		      (list 
		       (if (boundp 'NEMACS)
			   'kanji-flag
			 (if (boundp 'MULE)
			     mc-flag))
			    '("["
			      (tcode-on "T" "-")
			      (tcode-on-in-minibuffer "T" "-")
			      "]"))
		      dic-mode-line-format)))))))

(defun odic-mode ()
  "Major mode for editing odic intended for humans to read.  
Special commands:\\{odic-mode-map}
Turning on odic-mode calls the value of the variable odic-mode-hook,
if that value is non-nil."
  (interactive)
  (kill-all-local-variables)
  (use-local-map odic-mode-map)
  (setq mode-name "Dict")
  (setq major-mode 'odic-mode)
  (setq local-abbrev-table odic-mode-abbrev-table)
  (set-syntax-table odic-mode-syntax-table)
  (setq buffer-read-only t)
  (make-variable-buffer-local 'goal-column)
  (setq goal-column 4)
  (od:set-mode-line-format)
  (run-hooks 'odic-mode-hook))

(defvar odic-body-mode-syntax-table nil
  "Syntax table used while in odic body mode.")

(defvar odic-body-mode-abbrev-table nil
  "Abbrev table used while in odic body mode.")
(define-abbrev-table 'odic-body-mode-abbrev-table ())

(if odic-body-mode-syntax-table
    ()
  (setq odic-body-mode-syntax-table (make-syntax-table)))

(defvar odic-body-mode-map nil "")
(if odic-body-mode-map
    ()
  (setq odic-body-mode-map (make-sparse-keymap))
  (define-key odic-body-mode-map "h" 'backward-char)
  (define-key odic-body-mode-map "j" 'next-line)
  (define-key odic-body-mode-map "k" 'previous-line)
  (define-key odic-body-mode-map "l" 'forward-char)
  (define-key odic-body-mode-map "w" 'forward-word)
  (define-key odic-body-mode-map "b" 'backward-word)
  (define-key odic-body-mode-map " " 'scroll-up)
  (define-key odic-body-mode-map "\177" 'scroll-down)
  (define-key odic-body-mode-map "e" 'od:lookup-pattern-edit)
  (define-key odic-body-mode-map "q" 'od:quit)
  (define-key odic-body-mode-map "Q" 'od:really-quit)
  (define-key odic-body-mode-map "R" 'od:really-restart)
  (define-key odic-body-mode-map "0" 'beginning-of-line)
  (define-key odic-body-mode-map "$" 'end-of-line))

(defun odic-body-mode ()
  "Major mode for editing odic intended for humans to read.  
Special commands:\\{odic-body-mode-map}
Turning on odic-body-mode calls the value of the variable odic-body-mode-hook,
if that value is non-nil."
  (interactive)
  (kill-all-local-variables)
  (use-local-map odic-body-mode-map)
  (setq mode-name "DictBody")
  (setq major-mode 'odic-body-mode)
  (setq local-abbrev-table odic-body-mode-abbrev-table)
  (set-syntax-table odic-body-mode-syntax-table)
  (setq buffer-read-only t)
  (make-variable-buffer-local 'fill-column)
  (setq fill-column (or od:*body-fillcolumn* 
			(let ((fillcol (- (window-width) 10)))
			  (if (< 0 fillcol)
			      fillcol))
			fill-column))
  (od:set-mode-line-format)
  (run-hooks 'odic-body-mode-hook))


(defun od:server-active-p ()
  (interactive)
  (and (processp od:*dict*)
       (memq (process-status od:*dict*) '(open run))
       ;; if someone killed buffer
       (let* ((buffer (process-buffer od:*dict*)))
	 (if (and buffer (buffer-name buffer))
	     t
	   (delete-process od:*dict*)
	   nil))))

(defun od:kill-process ()
  (interactive)
  (and (od:server-active-p)
       (delete-process od:*dict*)))

(defun od:open-dictionary ()
  (if od:*dict*
      nil
    (setq od:*dict* (open-network-stream "*ndtp*" dserver-buffer
					 dserver-current-host-name 
					 dserver-protocol))
    (if (boundp 'NEMACS)
	(set-process-kanji-code od:*dict* 3)
      (if (boundp 'MULE)
	  (set-process-coding-system od:*dict* *internal* *euc-japan*)))
    (set-process-sentinel od:*dict* 'od:watchdog)
    (set-buffer dserver-buffer)
    (erase-buffer)
    (process-send-string od:*dict*
			 (format "A%s@%s\n" (getenv "USER") (od:hostname)))
    (let ((ans (od:wait-until '("$A\n" "$?\n" "$N\n" "$<\n") od:*dict*)))
      (if (not (string= "$A\n" ans))
	  (error "Cannot connect to jisyo server")))))

(defun od:wait-until (list proc)
  (catch 'od:wait-until
    (while t
      (let* ((str (buffer-string))
	     (len (length str))
	     (tmp list))
	(while tmp
	  (let* ((item (car tmp))
		 (ilen (length item)))
	    (cond ((< len ilen) nil)
		  ((string= item (substring str (- len ilen)))
		   (throw 'od:wait-until item))))
	  (setq tmp (cdr tmp)))
	(accept-process-output proc)))))

(defun od:setup-windows ()
  (delete-other-windows)
  (switch-to-buffer od:*dict-header-buffer-name*)
  (switch-to-buffer od:*dict-body-buffer-name*)
  (split-window (selected-window) od:*window-config*)
  (switch-to-buffer od:*dict-header-buffer-name*))

(defun od:kill-active-buffers ()
  (kill-buffer od:*dict-header-buffer-name*)
  (kill-buffer od:*dict-body-buffer-name*)
  (kill-buffer dserver-buffer)
  (if (get-buffer " *od:hostname*") (kill-buffer " *od:hostname*"))
  (if (get-buffer " *od:temp*") (kill-buffer " *od:temp*"))
  (if (get-buffer " *od:Frame*") (kill-buffer " *od:Frame*"))
  (if (get-buffer " *od:hostname*") (kill-buffer " *od:hostname*"))
  (if (get-buffer " *od:Selection*") (kill-buffer " *od:Selection*"))
  (if (get-buffer " *od:Help*") (kill-buffer " *od:Help*")))

(defun od:set-header ()
  (od:setup-windows)
  (set-buffer od:*dict-header-buffer-name*))

(defun od:set-body ()
  (od:setup-windows)
  (set-buffer od:*dict-body-buffer-name*))

(defun od:switch-to-header ()
  (od:setup-windows)
  (switch-to-buffer od:*dict-header-buffer-name*))

(defun od:switch-to-body ()
  (od:setup-windows)
  (switch-to-buffer od:*dict-body-buffer-name*))

(defun od:watchdog (process event)
  (if (string= "finished\n" event)
      (setq od:*dict* nil)))



(defun online-dictionary ()
  (interactive)
  (let ((is-first (not od:*dict*)))
    (setq dserver-current-host-name dserver-host-name)
    (setq od:*eiwa-current-display-mode*
	  (if od:*eiwa-summary-mode* "[Summary]" 
	      ""))
    (if (not (or (equal major-mode 'odic-mode) 
		 (equal major-mode 'odic-body-mode)))
	    (setq od:*previous-config* (current-window-configuration)))
    (get-buffer-create dserver-buffer)
    (get-buffer-create " *od:hostname*")
    (get-buffer-create " *od:temp*")
    (setq od:*dict-header-buffer*
	(get-buffer-create od:*dict-header-buffer-name*))
    (setq od:*dict-body-buffer* 
	(get-buffer-create od:*dict-body-buffer-name*))
    (od:set-header)
    (odic-mode)
    (od:set-body)
    (odic-body-mode)
    (od:set-header)
    (delete-other-windows)
    (let ((buffer-read-only nil))
      (erase-buffer)
      (od:show-title))
    (od:open-dictionary)
    (if is-first
	(if od:*default-jisyo*
	    (od:really-change-dictionary od:*default-jisyo*)
	    (od:change-dictionary)))
    (if od:*show-help-automatically*
	(od:help))))

(defun od:help ()
  (interactive)
  (with-output-to-temp-buffer " *od:Help*"
    (princ "
                   $B%*%s%i%$%s<-=q%3%^%s%I%j%U%!%l%s%9(B

$B%-!<(B  $BF0:n(B

f     $B8!:w$r9T$J$$$^$9(B. 
      $B%Q%?!<%s$OH>3Q%"%k%U%!%Y%C%H(B($B%"%k%U%!%Y%C%H8!:w(B), 
      $BJ?2>L>(B, $BJR2>L>(B($B$+$J8!:w(B) $B$GF~NO$7$^$9(B.
      $B%Q%?!<%s$O8lF,$^$?$O8lHx$K(B1$B$D$N(B \"*\" $B$rIU$1$k$3$H$,$G$-$^$9(B.
      $B$=$N:](B, \"*\" $B$O(B0$BJ8;z0J>e$NG$0U$NJ8;zNs$r0UL#$7$^$9(B.
      $B%Q%?!<%s$OG$0U$N0LCV$K(B \"?\" $B$r$$$/$D$G$b4^$`$3$H$,$G$-$^$9(B.
      $B$=$N:](B, \"?\" $B$OG$0U$N(B1$BJ8;z$r0UL#$7$^$9(B.

i     ispell$B$rMQ$$$F$"$d$U$d$JDV$+$iF~NO%Q%?!<%s$rA*$S$^$9(B. ($B1QOB<-=q$N;~JXMx(B)

.     $B8=:_$N8uJd$N@bL@J8$rI=<($7$^$9(B.
p     $BD>A0$N8uJd$N@bL@J8$rI=<($7$^$9(B.
n     $BD>8e$N8uJd$N@bL@J8$rI=<($7$^$9(B.
s     $B8uJd$rHV9f$GA*$S(B, $B$=$N@bL@J8$rI=<($7$^$9(B.
1-9   1-9$BHV$N8uJd$rD>@\A*$S(B, $B$=$N@bL@J8$rI=<($7$^$9(B.
      10$BHV0J9_$O(B \"s\" $B$^$?$O(B \"n\" $B%3%^%s%I$r;H$C$F2<$5$$(B.

SPC   $B@bL@J8$r(B od:*scroll-step* $B$G<($5$l$?CM$@$1>e$K%9%/%m!<%k$7$^$9(B. 
DEL   $B@bL@J8$r(B od:*scroll-step* $B$G<($5$l$?CM$@$12<$K%9%/%m!<%k$7$^$9(B. 
C-d   $B@bL@J8$r(B od:*scroll-step* $B$G<($5$l$?CM$NH>J,$@$1>e$K%9%/%m!<%k$7$^$9(B. 
C-u   $B@bL@J8$r(B od:*scroll-step* $B$G<($5$l$?CM$NH>J,$@$12<$K%9%/%m!<%k$7$^$9(B. 
      $B$3$l$i$N%3%^%s%I$O@bL@J8$,I=<($5$l$F$$$J$1$l$P(B \".\" $B$HF1$8F0:n$r$7$^$9(B.
      od:*scroll-step* $B$K$O(B, $B@0?t(B, half, full $B$N$$$:$l$+$,;XDj$G$-$^$9(B.
      $B%G%U%)%k%H$O(B full $B$G$9(B.

q     $B%*%s%i%$%s<-=q5/F0A0$N%&%$%s%I%&%3%s%U%#%0%l!<%7%g%s$KLa$j$^$9(B.
      lup-mode $B$N>l9g$K$O(B C-c [li] $B$r$7$?;~$N>uBV$KLa$j$^$9(B. 
Q     $B%*%s%i%$%s<-=q$N;}$D%P%C%U%!$rA4$F:o=|$7(B, 
      $B%*%s%i%$%s<-=q5/F0A0$N%&%$%s%I%&%3%s%U%#%0%l!<%7%g%s$KLa$j$^$9(B.
R     $B%*%s%i%$%s<-=q$r:F5/F0$7$^$9(B. 
      $B<-=q%5!<%PEy$,%j%9%?!<%H$7$?;~$K%3%M%/%7%g%s$N:F3+$K;H$$$^$9(B.
      $B%P%C%U%!Ey$N%3%s%U%#%0%l!<%7%g%s$OA4$F<:$o$l$^$9(B.

C     $B<-=q$r@ZBX$($^$9(B. 
      $B8=:_(B eiwa($B1QOB<-E5(B), waei($BOB1Q<-E5(B), kojien($B9-<-1q(B) $B$,;H$($^$9(B.
      $B%3%s%W%j!<%7%g%sF~NO$,2DG=$G$9(B.
E     $B1QOB<-E5$G$NNcJ8$NI=<($N(B ON/OFF $B$r@ZBX$($^$9(B.

H     $B$b$7$"$l$P(B, $BK^Nc$rI=<($7$^$9(B. p, n, SPC, DEL, q $B$G$=$l$>$lA0%Z!<%8(B, 
      $B<!%Z!<%8(B, $B%9%/%m!<%k%"%C%W(B, $B%9%/%m!<%k%@%&%s(B, $B=*N;$H$J$j$^$9(B.
O     $B$b$7$"$l$P(B, $B1|IU$1$rI=<($7$^$9(B. q$B$G=*N;$7$^$9(B.

M-x lup-mode
      lup$B%b!<%I$K$J$j$^$9(B
C-c l $B%+!<%=%k>e$N8l$r@Z$j=P$78!:w$7$^$9(B. (lup-mode)
C-c l $B%+!<%=%k>e$N8l$r@Z$j=P$7(Bispell$B$G867?$KD>$7$?8e8!:w$7$^$9(B.  (lup-mode)
C-c q lup$B%b!<%I$r$L$1$^$9(B. 

?     $B$3$N%a%C%;!<%8$rI=<($7$^$9(B.
")))

(defun od:show-title ()
  (insert "

		     $BEE;R%V%C%/HG<-=q8!:w%7%9%F%`(B


		       $B4dGH=qE9(B $B9-<-1q(B($BBh;0HG(B)
	   $B8&5f<R(B $B?71QOBCf<-E5(B($BBh(B5$BHG(B)$B!&?7OB1QCf<-E5(B($BBh(B3$BHG(B)

		 Copyright (C) Shingo NISHIOKA, 1991
		    nishioka@sanken.osaka-u.ac.jp

                 Modified by SAITO Yutaka, 1992
                    yutaka@sys1.cpg.sony.co.jp

* $B!V9-<-1q!W$O3t<02q<R4dGH=qE9$NEPO?>&I8$G$9(B.
")
  (goto-char 1))

(defun od:change-dictionary ()
  (interactive)
  (cond ((eql od:*fep-type* 'iroha)
	 (if iroha:*japanese-mode* (iroha-toggle-japanese-mode))))
  (let ((new-dic (completing-read "Select jisyo: " od:*jisyo*)))
    (od:really-change-dictionary new-dic)))

(defun od:really-change-dictionary (new-dic)
  ;; 93.2.11 by T.Ito
  (od:kill-process)
  (setq od:*dict* nil)
  (if (string= new-dic "kojien")
      (if dserver-host-name-kojien
	  (setq dserver-current-host-name dserver-host-name-kojien))
    (setq dserver-current-host-name dserver-host-name))
  ;; end of patch
  (od:open-dictionary)
  (set-buffer dserver-buffer)
  (erase-buffer)
  (process-send-string od:*dict* (format "L%s\n" new-dic))
  (let ((ans (od:wait-until '("$*\n" "$&\n" "$?\n" "$N\n" "$<\n") od:*dict*)))
    (if (not (string= ans "$*\n"))
	(error "Jisyo set failed")
	(setq od:*current-jisyo* new-dic)
	(setq od:*current-jisyo-pretty-name*
	      (cdr (assoc new-dic od:*dict-name*)))
	(setq od:*current-dict-filter*
	      (cdr (assoc new-dic od:*dict-filter*)))
	(setq od:*current-dict-filter-func*
	      (cdr (assoc new-dic od:*dict-filter-func*)))
	(message "$B<-=q$,!V(B%s$B!W$K%;%C%H$5$l$^$7$?(B."
		 od:*current-jisyo-pretty-name*)))
  (od:change-index-type)
  (setq od:*default-index-type*
	(cdr (assoc new-dic od:*default-index-type-dict*)))
  (setq od:*input-by-kanji*
	(cdr (assoc od:*default-index-type* od:*is-use-kanji*))))

(defun od:change-index-type ()
  (od:open-dictionary)
  (set-buffer dserver-buffer)
  (erase-buffer)
  (process-send-string od:*dict* "I\n")
  (let ((ans (od:wait-until '("$$\n" "$?\n" "$N\n" "$<\n") od:*dict*)))
    (cond ((string= "$$\n" ans)
	   (let ((lst))
	     (let ((tmp '(("BA" . ?A) ("BK" . ?K) ("IA" . ?a) ("IK" . ?k))))
	       (while tmp
		 (goto-char 1)
		 (if (re-search-forward (car (car tmp)) nil t)
		     (setq lst (cons (cdr (car tmp)) lst)))
		 (setq tmp (cdr tmp))))
	     (setq od:*index-types* lst)
	     (with-output-to-temp-buffer " *od:Selection*"
	       (princ (format "\n$B$3$N<-=q$O0J2<$N(B%d$B<oN`$N8!:w$,2DG=$G$9(B.\n"
			      (length lst)))
	       (while lst
		 (princ (cdr 
			 (assoc (car lst)
				'((?A . "$B%"%k%U%!%Y%C%H8eJ}(B ... $B8lF,$K(B * $B$rIU$1$k$3$H$,$G$-$k(B.")
				  (?K . "$B$+$J8eJ}(B           ... $B8lF,$K(B * $B$rIU$1$k$3$H$,$G$-$k(B.")
				  (?a . "$B%"%k%U%!%Y%C%HA0J}(B ... $B8lHx$K(B * $B$rIU$1$k$3$H$,$G$-$k(B.")
				  (?k . "$B$+$JA0J}(B           ... $B8lHx$K(B * $B$rIU$1$k$3$H$,$G$-$k(B.")))))
		 (princ "\n")
		 (setq lst (cdr lst)))))))))

;;;
;;; modified by yutaka@sys1.cpg.sony.co.jp
;;;
(defun od:lookup-pattern ()
  (interactive)
  (let ((pat (od:read-string-with-fep (if od:*input-by-kanji* 
					  "Pattern(in kana): " 
					"Pattern: ") 
				      nil)))
;    (if (od:*japanese-p* pat)
;	;; search japanese
;	(if (string= od:*current-jisyo* "waei")
;	    nil
;	  (od:really-change-dictionary "waei"))
;      (if (string= od:*current-jisyo* "eiwa")
;	  nil
;	(od:really-change-dictionary "eiwa")))
    (od:really-lookup-pattern pat)))

(defun od:really-lookup-pattern (pat)
  (od:open-dictionary)
  (set-buffer dserver-buffer)
  (erase-buffer)
  (let ((index (pattern-consistency-check pat))
	(eucpat pat))
    (if (or (eql index ?A)
	    (eql index ?K))
	(setq eucpat (od:string-reverse eucpat)))
    (od:*log* od:*debug-flag* "eucpat [%s]\n" (od:*quote-null* eucpat))
    (od:*log* od:*debug-flag* (concat "P" (char-to-string index) eucpat "\n"))
    (process-send-string od:*dict* 
			 (concat "P" (char-to-string index) eucpat "\n"))
    (let ((ans (od:wait-until '("$$\n" "$?\n" "$N\n" "$<\n") od:*dict*)))
      (cond ((string= "$$\n" ans)
	     (let ((str (buffer-string)))
	       (od:*log* od:*debug-flag* "buffer-string[%s]\n" (od:*quote-null* str))
	       (od:find-entries (buffer-string))))
	    (t (od:no-entries-found pat))))))

(defun pattern-consistency-check (pat)
  (let ((len (length pat))
	(i 0)
	(kana nil)
	(alfa nil)
	(other nil)
	(first nil)
	(last nil))
    (if (boundp 'MULE)
	(progn
	  (setq pat (code-convert-string pat *internal* *euc-japan*))
	  (setq len (length pat))))
    (od:*log* od:*debug-flag* "consistency pat[%s]\n" (od:*quote-null* pat))
    (while (< i len)
      (let ((c (aref pat i)))
	(cond ((= ?* c)
	       (cond ((= i 0)
		      (setq first t))	;at beginning
		     ((= i (1- len))
		      (setq last t))	;at end
		     (t (error "\"*\"$B$O%Q%?!<%s$NC<0J30$G$O;H$($^$;$s(B."))))
	      ((= ?? c))		;do nothing
	      ((< 127 c)
	       (setq i (1+ i))
	       (let* ((cc (aref pat i))
		      (k (+ (* 256 c) cc)))
	       (cond ((or (= 164 c)	;hirakana
			  (= 165 c))	;katakana
		      (setq kana t))
		     ((= 163 c)		;eizi
		      (setq alfa t))
		     ((= k 41462)	;zenkaku *
		      (cond ((= i 1)
			     (setq first t))	;at beginning
			    ((= i (1- len))
			     (setq last t))	;at end
			    (t (error "\"*\"$B$O%Q%?!<%s$NC<0J30$G$O;H$($^$;$s(B."))))
		     (t)		;do nothing
		     )))
	      ((or (and (<= ?a c) (<= c ?z))
		   (and (<= ?A c) (<= c ?Z)))
	       (setq alfa t))
	      (t (setq other t))))
	(setq i (1+ i)))
    (if (and kana alfa)
	(message "$B2>L>$H1Q;z$,:.$6$C$F$$$^$9$,(B, $B2>L>$H$7$F8!:w$rB3$1$^$9(B."))
    (if (and first last)
	(error "\"*\"$B$O(B1$B8D$7$+;H$($^$;$s(B."))
    (let ((idx (cond (kana (if first ?K ?k))
		     (alfa (if first ?A ?a)))))
      (if (not (memq idx od:*index-types*))
	  (error "$B$=$N8!:wJ}K!$O;H$($^$;$s(B."))
      idx)))

(defun od:find-entries (str)
  (set-buffer " *od:temp*")
  (erase-buffer)
  (insert str)
  (setq od:*headers* nil)
  (setq od:*current-entry* -1)
  (goto-char 1)
  (if (looking-at "$0\n")		;this is $0 line
      (forward-char 3))			;just after $0
  (while (< (point) (- (point-max) 3))
    (let (entry body)
      (beginning-of-line)
      (let ((begin (point)))
	(end-of-line)
	(let ((end (point)))
	  (setq entry (buffer-substring begin end))))
      (next-line 1)
      (beginning-of-line)
      (let ((begin (point)))
	(end-of-line)
	(let ((end (point)))
	  (setq body (buffer-substring begin end))))
      (if (not (od:findp-equal (cons entry body) od:*headers*))
	  (setq od:*headers*
		(cons (cons entry body)
		      od:*headers*)))
      (next-line 1)))
  (if (null od:*headers*)
      (od:no-entries-found pat)
      (setq od:*headers* (reverse od:*headers*))
      (od:show-headers)
      (delete-other-windows)
      (if (and od:*select-unique-entry-immediately*
	       (= 1 (length od:*headers*)))
	  (od:select-entry 1))))

(defun od:show-headers ()
  (interactive)
  (od:set-header)
  (let ((buffer-read-only nil))
    (erase-buffer)
    (insert (format "%d entries.\n"
		    (length od:*headers*)))
    (let ((tmp od:*headers*)
	  (i 1))
      (while tmp
	(let ((ent (car tmp)))
	  (if (boundp 'MULE)
	      (rplaca ent 
		      (mapconcat (function 
				  (lambda (elt) (if (= elt 10) ""
						  (char-to-string elt))))
		 (code-convert-string (car ent) *euc-japan* *internal*) "")))
	  (insert (format "%3d: %s\n" i (car ent))))
	(setq i (1+ i))
	(setq tmp (cdr tmp)))))
  (goto-line 2)
  (forward-char 4))

(defun od:select-entry (n)
  (interactive "nNumber:")
  (let ((len (length od:*headers*)))
    (if (and (<= 1 n) (<= n len))
	(progn
	  (setq od:*current-entry* n)
	  (let ((ent (od:get-entry (nth (1- n) od:*headers*))))
	    (od:show-entry ent))
	  (od:set-header)
	  (goto-line (1+ n))
	  (forward-char 4))
	(message (format "%d out of range" n)))))

(defun od:get-entry (pat)
  (od:open-dictionary)
  (set-buffer dserver-buffer)
  (erase-buffer)
  (process-send-string od:*dict* (format "S%s\n" (cdr pat)))
  (let ((ans (od:wait-until '("$$\n" "$?\n" "$N\n" "$<\n") od:*dict*)))
    (cond ((string= "$$\n" ans)
	   (buffer-substring 4 (- (point-max) 4)))
	  (t ""))))

(defun od:direct-select-entry ()
  (interactive)
  (let ((n (string-to-int (char-to-string last-command-char))))
    (od:select-entry n)))

(defun od:current-header-line ()
  (save-excursion
    (od:set-header)
    (1- (count-lines 1 (1+ (point))))))

(defun od:previous-entry ()
  (interactive)
  (let ((curr (od:current-header-line)))
    (if (< 1 curr)
	(od:select-entry (1- curr))
	(message "No previous entry"))))

(defun od:current-entry ()
  (interactive)
  (let ((curr (od:current-header-line)))
    (od:select-entry curr)))

(defun od:next-entry ()
  (interactive)
  (let ((curr (od:current-header-line)))
    (if (< curr (length od:*headers*))
	(od:select-entry (1+ curr))
	(message "No following entry"))))

(defun od:scroll-step ()
  (cond ((integerp od:*scroll-step*)
	 od:*scroll-step*)
	((eql od:*scroll-step* 'full)
	 (- (window-height (next-window (selected-window))) 2))
	((eql od:*scroll-step* 'half)
	 (- (/ (window-height (next-window (selected-window))) 2) 1))))

(defun od:scroll-entry-up ()
  (interactive)
  (let ((curr (od:current-header-line)))
    (if (/= od:*current-entry* curr)
	(od:select-entry curr)
	(scroll-other-window (od:scroll-step)))))

(defun od:scroll-entry-down ()
  (interactive)
  (let ((curr (od:current-header-line)))
    (if (/= od:*current-entry* curr)
	(od:select-entry curr)
	(scroll-other-window (- (od:scroll-step))))))

(defun od:scroll-entry-up-half ()
  (interactive)
  (let ((curr (od:current-header-line)))
    (if (/= od:*current-entry* curr)
	(od:select-entry curr)
	(scroll-other-window (/ (od:scroll-step) 2)))))

(defun od:scroll-entry-down-half ()
  (interactive)
  (let ((curr (od:current-header-line)))
    (if (/= od:*current-entry* curr)
	(od:select-entry curr)
	(scroll-other-window (- (/ (od:scroll-step) 2))))))

(defun od:show-entry (entry)
  (save-excursion
    (od:set-body)
    (let ((buffer-read-only nil))
      (erase-buffer)      
      (if (boundp 'MULE)
	  (setq entry (code-convert-string entry *euc-japan* *internal*)))
      (insert entry)
      (let ((tmp od:*current-dict-filter-func*))
	(while tmp
	  (goto-char 1)
	  (funcall (car tmp))
	  (setq tmp (cdr tmp))))
      (fill-region 1 (point-max))
      (hang-indent-buffer))
    (goto-char 1)))

(defun od:no-entries-found (pat)
  (od:set-body)
  (let ((buffer-read-only nil))
    (erase-buffer)
    (insert (format "\"%s\" Not found\n" pat)))
  (od:set-header))



(defun od:toggle-eiwa-summary-mode ()
  (interactive)
  (setq od:*eiwa-summary-mode* (not od:*eiwa-summary-mode*))
  (if od:*eiwa-summary-mode*
      (setq od:*eiwa-current-display-mode* "[Summary]")
      (setq od:*eiwa-current-display-mode* ""))
  (od:current-entry))



(defun od:quit ()
  (interactive)
  (if od:*previous-config*
      (set-window-configuration od:*previous-config*))
  (setq od:*previous-config* nil))

(defun od:really-quit ()
  (interactive)
  (process-send-string od:*dict* (format "Q\n"))
  (od:kill-process)
  (setq od:*dict* nil)
  (od:kill-active-buffers)
  (if od:*previous-config*
      (set-window-configuration od:*previous-config*))
  (setq od:*previous-config* nil))

(defun od:really-restart ()
  (interactive)
  (if (yes-or-no-p "Really restart? ")
      (progn
	(setq od:*dict* nil)
	(setq od:*headers* nil)
	(setq od:*current-entry* -1)
	(kill-buffer dserver-buffer)
	(kill-buffer " *od:hostname*")
	(kill-buffer " *od:temp*")
	(kill-buffer od:*dict-header-buffer*)
	(kill-buffer od:*dict-body-buffer*)
	(online-dictionary))))



(defun od:hostname ()
  (save-excursion
    (set-buffer " *od:hostname*")
    (erase-buffer)
    (call-process "/bin/hostname" nil " *od:hostname*")
    (let ((str (buffer-string)))
      (let ((i 0)
	    (len (length str)))
	(while (and (< i len)
		    (/= (aref str i) ?\n))
	  (setq i (1+ i)))
	(substring str 0 i)))))


(defvar od:ispell-program-name "ispell"
  "Program invoked by ispell-word and ispell-region commands.")

(defconst od:ispell-out-name " *ispell*"
  "Name of the buffer that is associated with the 'ispell' process")

(defvar od:ispell-process nil
  "Holds the process object for 'ispell'")

(defun od:ispell-init-process ()
  (if (and od:ispell-process
	   (eq (process-status od:ispell-process) 'run))
      (save-excursion
	(set-buffer od:ispell-out-name)
	(erase-buffer))
      (message "Starting new ispell process...")
      (and (get-buffer od:ispell-out-name) (kill-buffer od:ispell-out-name))
      (setq od:ispell-process (apply 'start-process "ispell"
                                   od:ispell-out-name od:ispell-program-name
				   (list "-A")))
      (process-kill-without-query od:ispell-process)
      (sleep-for 3)))

(defun od:lookup-pattern-with-ispell ()
  (interactive)
  (let ((orig-pat (read-string "Pattern: ")))
    (od:lookup-pattern-with-ispell-sub orig-pat)))

(defun od:lookup-pattern-with-ispell-sub (orig-pat)
  (od:ispell-init-process)
  (if od:*ispell-ver-3*
      (progn
	(set-buffer od:ispell-out-name)
	(erase-buffer)))
  (send-string od:ispell-process orig-pat)
  (send-string od:ispell-process "\n")
  (sleep-for 1)
  (set-buffer od:ispell-out-name)
  (goto-char 1)
  (cond ((looking-at "\\*")
	 ;; correct!
	 (od:really-lookup-pattern orig-pat))
	((looking-at "+")
	 ;; a word follows
	 (od:really-lookup-pattern (od:select-candidate)))
	((looking-at "&")
	 ;; some words follows
	 (od:really-lookup-pattern (od:select-candidate)))
	((looking-at "#")
	 ;; cannot find any candidates
	 (message "No candidate."))))

(defun od:select-candidate ()
  (cond ((and od:*ispell-ver-3*
	      (looking-at "&"))
	 (goto-char 2)
	 (zap-to-char 1 ?:)
	 (delete-char 1)
	 (replace-string "," "")))
  (goto-char 1)
  (forward-char 2)
  (let ((list nil))
    (while (looking-at "[^ ]")
      (let ((begin (point)))
	(forward-word 1)
	(setq list (cons (buffer-substring begin (point)) list)))
      (while (looking-at "[ \t\n]") (forward-char 1)))
    (with-output-to-temp-buffer " *od:Selection*"
      (let ((i 1)
	    (col 0)
	    (l list))
	(while l
	  (let ((str (format "(%d)  %s  " i (car l))))
	    (let ((len (length str)))
	      (if (< (- (screen-width) 2) (+ col len))
		  (progn
		    (princ "\n")
		    (setq col 0)))
	      (setq col (+ col len))
	      (princ str)))
	  (setq l (cdr l))
	  (setq i (1+ i)))))
    (let ((sel (or (and od:*ispell-ver-3* (= (length list) 1) 1)
		   (od:read-integer))))
      (while (or (< sel 1) (< (length list) sel))
	(setq sel (od:read-integer)))
      (nth (1- sel) list))))

(defun od:read-integer ()
  (let ((n (string-to-int (read-from-minibuffer "Choose:"))))
    (while (not (integerp n))
      (setq n (string-to-int (read-from-minibuffer "Choose:"))))
    n))

(defun od:read-string-with-fep (prompt initial)
  (set-buffer (format " *Minibuf-%d*" (minibuffer-depth)))
  (cond ((or (not od:*input-by-kanji*)
	     (eql od:*fep-type* 'no-fep))
	 (format "%s" (read-minibuffer prompt (or initial ""))))
	((eql od:*fep-type* 'egg)
	 (format "%s" (read-hiragana-string prompt (or initial ""))))
	((eql od:*fep-type* 'iroha)
	 (if (not iroha:*japanese-mode*)(iroha-toggle-japanese-mode))
	 (format "%s" (read-minibuffer prompt (or initial ""))))
	((eql od:*fep-type* 't-code)
	 (setq tcode-on-in-minibuffer nil)
	 (setq unread-command-char od:*tcode-toggle-char*)
	 (format "%s" (read-minibuffer prompt (or initial ""))))
	(t (error "Unknown fep-type %s" od:*fep-type*))))

(defun od:findp-equal (item sequence)
  (let ((is-true nil))
    (while (and sequence
		(not is-true))
      (if (equal item (car sequence))
	  (setq is-true t)
	  (setq sequence (cdr sequence))))
    is-true))

(defun hang-indent-buffer ()
  (goto-char 1)
  (beginning-of-line)
  (while (< (point) (point-max))
    (cond ((looking-at "[ \t]")
	   (while (looking-at "[ \t]")
	     (delete-char 1)))
	  ((looking-at "\n")
	   ;; do nothing
	   )
	  (t
	   (insert "  ")))
    (next-line 1)
    (beginning-of-line)))

(defun od:string-reverse (s)
  (let ((n (make-string (length s) ? ))
	(j (1- (length s)))
	(i 0))
    (while (<= 0 j)
      (let ((c (aref s i)))
	(if (<= c 127)
	    (progn
	      (aset n j c)
	      (setq i (1+ i)
		    j (1- j)))
	    (progn
	      (aset n j  (aref s (1+ i)))
	      (aset n (1- j) c)
	      (setq i (+ i 2)
		    j (- j 2))))))
    n))

;;; hanrei, okuduke

(defvar od:*previous-config-frm* nil)

(defun od:save-config-frm ()
  (if (null od:*previous-config-frm*)
      (setq od:*previous-config-frm*
	    (current-window-configuration))))

(defun od:restore-config-frm ()
  (interactive)
  (if od:*previous-config-frm*
      (set-window-configuration od:*previous-config-frm*))
  (setq od:*previous-config-frm* nil))

(defun od:show-okuduke ()
  (interactive)
  (od:save-config-frm)
  (od:open-dictionary)
  (set-buffer dserver-buffer)
  (erase-buffer)
  (process-send-string od:*dict* "I\n")
  (let ((ans (od:wait-until '("$$\n" "$?\n" "$N\n" "$<\n") od:*dict*)))
    (cond ((string= "$$\n" ans)
	   (goto-char 1)
	   (if (re-search-forward "OK " nil t nil)
	       (if (re-search-forward "[0-9a-f][0-9a-f]*")
		   (let ((frm (buffer-substring
			       (match-beginning 0) (match-end 0))))
		     (od:show-frame (od:xtoi frm)))))))))

(defun od:show-hanrei ()
  (interactive)
  (od:save-config-frm)
  (od:open-dictionary)
  (set-buffer dserver-buffer)
  (erase-buffer)
  (process-send-string od:*dict* "I\n")
  (let ((ans (od:wait-until '("$$\n" "$?\n" "$N\n" "$<\n") od:*dict*)))
    (cond ((string= "$$\n" ans)
	   (goto-char 1)
	   (if (re-search-forward "HA " nil t nil)
	       (if (re-search-forward "[0-9a-f][0-9a-f]*")
		   (let ((frm (buffer-substring
			       (match-beginning 0) (match-end 0))))
		     (od:show-frame (od:xtoi frm)))))))))

(defvar od:*current-frame* nil)

(defun od:next-frame ()
  (interactive)
  (if od:*current-frame*
      (progn (setq od:*current-frame* (1+ od:*current-frame*))
	     (od:show-frame od:*current-frame*))))

(defun od:previous-frame ()
  (interactive)
  (if od:*current-frame*
      (progn (setq od:*current-frame* (1- od:*current-frame*))
	     (od:show-frame od:*current-frame*))))

(defun od:show-frame (frm)
  (setq od:*current-frame* frm)
  (set-buffer dserver-buffer)
  (erase-buffer)
  (process-send-string od:*dict* (format "F%x\n" frm))
  (sit-for 1)
  (let ((s (buffer-substring 3 (+ 2048 3))))
    (get-buffer-create " *od:Frame*")
    (set-buffer " *od:Frame*")
    (erase-buffer)
    (local-set-key "p" 'od:previous-frame)
    (local-set-key "n" 'od:next-frame)
    (local-set-key " " 'scroll-up)
    (local-set-key "\177" 'scroll-down)
    (local-set-key "q" 'od:restore-config-frm)
    (insert (format "frame=%x\n" frm))
    (od:format-frame s)
    (switch-to-buffer " *od:Frame*")
    (delete-other-windows)
    (goto-char 1)))

(defun od:format-frame (s)
  (let ((i 0))
    (while (< i 2048)
      (let ((hi (aref s i))
	    (lo (aref s (1+ i))))
	(cond ((= hi 31)
	       (cond ((= lo 10)
		      (insert "\n"))
		     ((= lo ?c)
		      (setq i (+ i 2))
		      (let ((num (+ (* (aref s (+ i 0)) (* 256))
				    (* (aref s (+ i 1)) (*))))
			    (frm (+ (* (aref s (+ i 2)) (* 256))
				    (* (aref s (+ i 3)) (*))))
			    (ofs (+ (* (aref s (+ i 4)) (* 256))
				    (* (aref s (+ i 5)) (*)))))
			(setq i (+ i 4))))))
	      ((or (<= 128 hi) (<= 128 lo)))
	      ((or (< hi 32) (< lo 32)))
	      (t (if (boundp 'MULE) (insert (make-character lc-jp hi lo))
		   (insert (format "%c%c" (+ hi 128) (+ lo 128))))))
	(setq i (+ i 2))))))

(defun od:xtoi (s)
  (let ((r 0)
	(i 0)
	(l (length s)))
    (while (< i l)
      (let ((c (aref s i)))
	(setq r (+ (* r 16)
		   (if (and (<= ?0 c) (<= c ?9))
		       (- c ?0)
		       (+ (- c ?a) 10))))
	(setq i (1+ i))))
    r))

;;;
;;; Additional new interface [lookup-current-word].
;;;  look up a current point word.
;;;
;;;  15 Nov 91  Tetsuya Nishimaki   (t-nishim@dn.softbank.co.jp)
;;;
(defun od:word-at-point ()
  (interactive)
  (condition-case ()
      (save-excursion
        (if (looking-at "\\([a-zA-Z]\\|[$B$"(B-$B$s(B]\\|[$B%"(B-$B%s(B]\\)")
	    (forward-word 1))
	(re-search-backward "\\(\\<[a-zA-Z]+\\|[^$B$"(B-$B$s(B][$B$"(B-$B$s(B]+\\|[^$B%"(B-$B%s(B][$B%"(B-$B%s(B
]+\\)\\>" nil t)
	(goto-char (match-beginning 0))
	(let ((end (match-end 0)))
	  (if (not (looking-at "[a-zA-Z]")) (forward-char 1))
	  (buffer-substring (point) end)))
   (error nil)))


;;;
;;; modified by yutaka@sys1.cpg.sony.co.jp
;;;
(defun od:lookup-pattern-edit ()
  (interactive)
  (let* ((v (od:word-at-point))
	 (enable-recursive-minibuffers t)
;	 (pat (od:read-string-with-fep (if od:*input-by-kanji*
	 (pat (read-string (if od:*input-by-kanji*
			       "Pattern(in kana): "
			     "Pattern: ")
			   v)))
    (online-dictionary)
    (if (od:*japanese-p* pat)
	;; search japanese
	(if (string= od:*current-jisyo* "waei")
	    nil
	  (od:really-change-dictionary "waei"))
      (if (string= od:*current-jisyo* "eiwa")
	  nil
	(od:really-change-dictionary "eiwa")))
    (od:really-lookup-pattern pat)))

(defun od:lookup-pattern-ispell ()
  (interactive)
  (let* ((c (current-window-configuration))
	 (v (od:word-at-point))
	 (enable-recursive-minibuffers t)
	 (pat v))
    (set-window-configuration c)
    (online-dictionary)
    (od:lookup-pattern-with-ispell-sub pat)))

(make-variable-buffer-local 'lup-mode)
(make-variable-buffer-local 'original-bindings)

(defun lup-mode (arg)
  "toggle lup mode"
  (interactive "P")
  (setq lup-mode
	(if (null arg) (not lup-mode)
	    (> (prefix-numeric-value arg) 0)))
  (if lup-mode
      (progn
	(setq original-bindings
	      (mapcar '(lambda (key) (cons key (local-key-binding key)))
		      '("\C-ci" "\C-cl" "\C-cq")))
	(local-set-key "\C-ci" 'od:lookup-pattern-ispell)
	(local-set-key "\C-cl" 'od:lookup-pattern-edit))
      (progn
	(mapcar '(lambda (key&binding)
		  (local-set-key (car key&binding) (cdr key&binding)))
		original-bindings)))
  (set-buffer-modified-p (buffer-modified-p)))

(if (not (assq 'lup-mode minor-mode-alist))
    (setq minor-mode-alist (cons '(lup-mode " LUP") minor-mode-alist)))
;;;
;;; diclookup.el end
;;;
