;; -*-Emacs-Lisp-*-
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; File:         jam-zcat.el
;; RCS:          $Id: jam-zcat.el,v 1.63 1994/12/05 04:13:20 kazusi-m Exp $
;; Description:  simple file access through SOME PROGRAMS from GNU Emacs
;; Author:       Kazushi (Jam) Marukawa, kazusi-m@is.aist-nara.ac.jp
;; Created:      Fri Jan  4 12:29:21 JST 1991
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Copyright (C) 1991, 1992, 1994 Kazushi Marukawa.
;;;
;;; Author: Kazushi (Jam) Marukawa (kazusi-m@is.aist-nara.ac.jp)
;;;
;;; This program is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 2 of the License, or
;;; (at your option) any later version.
;;;
;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with this program; if not, write to the Free Software
;;; Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

;;; Commentary:
;;;
;;; $B$3$N%U%!%$%k$r%m!<%I$7$F$*$/$H!"%3%s%W%l%9$5$l$?%U%!%$%k(B
;;; $B$r!"(B``Mule''$B$d(B``NEmacs''$B!"(B``Emacs''$B$+$i!"IaDL$N%U%!%$%k(B
;;; $B$H$^$C$?$/0c$$$J$/%"%/%;%9$G$-$k$h$&$K$J$j$^$9!#(B
;;;
;;; $B$^$?!"FCDj$N%U%!%$%k$K$D$$$F$@$1!"FCDj$N%W%m%0%i%`$r2p$7(B
;;; $B$FF~=PNO$9$k$3$H$,$G$-$k$h$&$K$J$j$^$9!#Nc$($P0E9f2=$7$?(B
;;; $B%U%!%$%k$NFI$_=q$-$b$G$-$k$o$1$G$9!#(B
;;;
;;; $BF|K\8l$r%5%]!<%H$7$?(BMule$B!"(BNEmacs$B$K$bBP1~$9$k$?$a$K!"%P%C(B
;;; $B%U%!Cf$GE,@Z$J;~4|$K%3!<%IJQ49$r9T$&$?$a$N5!9=$r;}$C$F$$(B
;;; $B$^$9!#(B

;;; Instruction:
;;;
;;; $BMxMQJ}K!$O!"(B
;;;   (require 'jam-zcat)
;;; $B$H$9$k$@$1$G$9!#(B
;;;
;;; $BJQ?t$O!"(Bjam-zcat-filename-list$B!"(Bjam-zcat-hack-ange-ftp$B!"(B
;;; jam-zcat-hack-loadablep$B!"(Bjam-zcat-si-mode$B$,$"$j$^$9!#(B
;;;
;;; jam-zcat-filename-list:
;;; $B$3$N%j%9%H$NMWAG$O!"(B
;;;   (((REGEXP . STRRPL) (REGEXP . STRRPL)...)
;;;    COMPRESSPROG UNCOMPRESSPROG UNCOMPRESSERRORSTR
;;;    [COMPRESSARG [UNCOMPRESSARG]])
;;; $B$H$$$C$?7A$r$7$F$$$^$9!#(B
;;;
;;; REGEXP$B$K%^%C%A$7$?%U%!%$%k$KBP$7$F(Bjam-zcat$B$,A`:n$r9T$$$^(B
;;; $B$9!#$^$:%^%C%A$7$?ItJ,$r(BSTRRPL$B$GCV$-49$($?%U%!%$%kL>$,@5(B
;;; $B<0$JL>A0$H$7$F07$o$l$^$9!#$3$N!X@5<0$JL>A0!Y$O!"<B:]$NJT(B
;;; $B=8;~$N%P%C%U%!$N%b!<%I$J$I$r7hDj$9$k$?$a$KMQ$$$i$l$^$9!#(B
;;;
;;; $B$^$?(B2$BHVL\0J9_$N(BREGEXP$B$H(BSTRRPL$B$K$*$$$F$O!"%U%!%$%k$,8+IU(B
;;; $B$+$i$J$+$C$?:]$K!"(BREGEXP$B$r(BSTRRPL$B$GCV$-49$($?%U%!%$%k$r8!(B
;;; $B:w$9$k$h$&$K$J$C$F$$$^$9!#(B
;;;
;;; COMPRESSPROG$B$O=PNO;~$K!"(BUNCOMPRESSPROG$B$OF~NO;~$KMxMQ$9$k(B
;;; $B%W%m%0%i%`L>$r;XDj$7$^$9!#(BUNCOMPRESSERRORSTR$B$G$O!"%W%m%0(B
;;; $B%i%`$N%(%i!<%a%C%;!<%8$r;XDj$7$^$9!#(B
;;;
;;; $B$^$?!"(BCOMPRESSARG$B$H(BUNCOMPRESSARG$B$rMxMQ$7$F!"$=$l$>$l$N%W(B
;;; $B%m%0%i%`$GMxMQ$9$k0z?t$r;XDj$9$k$3$H$b$G$-$^$9!#$3$l$i$O(B
;;; eval$B$K$h$C$FI>2A$5$l!"$=$N7k2L$r0z?t$N%j%9%H$H$7$FMQ$$$^(B
;;; $B$9!#$b$7(BUNCOMPRESSARG$B$,;XDj$5$l$F$$$J$1$l$P!"F~=PNO$N$I(B
;;; $B$A$i$K$G$b(BCOMPRESSARG$B$rMxMQ$7$^$9!#(B
;;;
;;; $BNc$($P!"(B
;;;   ((("\\.taz$" . ".tar") ("\\.tar$" . ".taz")
;;;     ("$" . ".taz"))
;;;    "compress" "uncompress"
;;;    "stdin: not in compressed format\n")
;;; $B$H$$$&%j%9%H$K$h$C$F!"(Btaz$B%U%!%$%k$r05=L$5$l$?(Btar$B%U%!%$%k(B
;;; $B$H$7$F<h$j07$($k$h$&$K$J$j$^$9!#$^$?!"(Bfind-file$B$K<:GT$7(B
;;; $B$?>l9g$K$O!"%U%!%$%kL>$K(B".tar"$B$H$$$&ItJ,$,$"$l$P$=$l$r(B
;;; ".taz"$B$GCV$-49$($?%U%!%$%k$H$7$FC5$7!"<!$K(B".taz"$B$rIU$12C(B
;;; $B$($?%U%!%$%k$H$7$FC5$9$H$$$C$?F0:n$r9T$$$^$9!#(B
;;;
;;; $B$^$?!"(B
;;;   ((("\\.cry$" . "") ("" . ".cry"))
;;;    "crypt" "crypt" nil (jam-zcat-get-crypt-key))
;;; $B$H$$$C$?%j%9%H$rMxMQ$9$k$H!"(Bcrypt$B$rMxMQ$7$?F~=PNO$r!"(B
;;;   ((("\\.des$" . "") ("" . ".des"))
;;;    "des" "des" nil
;;;    (list "-e" "-k" (car (jam-zcat-get-crypt-key)))
;;;    (list "-d" "-k" (car (jam-zcat-get-crypt-key))))
;;; $B$H$$$C$?%j%9%H$rMxMQ$9$k$H!"(Bdes$B$rMxMQ$7$?F~=PNO$r9T$($^(B
;;; $B$9!#(B
;;;
;;; jam-zcat-hack-ange-ftp:
;;; $B$3$N%U%i%0$rN)$F$F$*$/$H!"$3$N%Q%C%1!<%8$r%m!<%I$7$?8e$K!"(B
;;; ange-ftp$B$r%m!<%I$7$F$$$F$b!"@5$7$/F0:n$9$k$h$&$K$J$j$^$9!#(B
;;; $B$A$J$_$K!"(Bange-ftp$B$r%m!<%I$7$?8e$K$3$N%Q%C%1!<%8$r%m!<%I(B
;;; $B$9$kJ,$K$O2?$NLdBj$b$"$j$^$;$s!#(B
;;;
;;; jam-zcat-hack-loadablep:
;;; jam-zcat-hack-loadablep-suffixes:
;;; ``Mule''$B$K$*$$$F!"$3$N%U%i%0$rN)$F$F$*$/$H!"(Bload$B4X?t$K$*(B
;;; $B$$$F!"(Bjam-zcat-hack-loadablep-suffixes$B%j%9%H$KEPO?$7$?3H(B
;;; $BD%;R$r;}$C$?%U%!%$%k$b%m!<%I$G$-$k$h$&$K$J$j$^$9!#(B
;;; $B$^$?!"(B'quick$B$H$$$&CM$K$7$F$*$/$H!"(Bload$B4X?t$,<:GT$7$?:]$K(B
;;; $B$N$_!">e5-$N%A%'%C%/$r9T$$$^$9!#(B
;;;
;;; jam-zcat-si-mode:
;;; ``Mule''$B$K$*$$$F!"$3$N%U%i%0$rN)$F$F$*$/$H!"(B``si:''$B$,IU(B
;;; $B$$$?4X?t$rCV$-49$($k$h$&$K$J$j$^$9!#$3$l$K$h$C$FF~=PNO;~(B
;;; $B$N(Bpre-hook$B$d(Bpost-hook$B$,@_Dj$5$l$F$$$?>l9g$K$b$&$^$/F0:n(B
;;; $B$9$k$h$&$K$J$k$O$:$G$9!#(B
;;;
;;; **$BCm0UE@(B**
;;; $B$7$+$7$3$l$i$N5!G=$r<B8=$9$k$K$"$?$C$F!"0J2<$K5s$2$k4X?t(B
;;; $B$rCV$-49$($F$$$^$9!#(B
;;;
;;; NEmacs$B$d!"(BEmacs$B$N>l9g(B:
;;;   write-region
;;;   insert-file-contents
;;;   normal-mode
;;;   get-file-buffer
;;;
;;; Mule$B$N>l9g(B:
;;;   si:write-region$B$+(Bwrite-region
;;;   si:insert-file-contents$B$+(Binsert-file-contents
;;;   normal-mode
;;;   get-file-buffer
;;;   loadablep
;;;
;;; $B$G$9$+$i!"$3$3$K5s$2$?4X?t$K%Q%C%A$rEv$F$k7A<0$N%W%m%0%i(B
;;; $B%`$rMxMQ$9$k>l9g$O!"$3$l$i$N4X?t$,8F$P$l$k=gHV$r9M$($F%m!<(B
;;; $B%I$9$kI,MW$,$"$j$^$9!#$?$@$7(Bange-ftp$B$K$D$$$F$OBP1~:Q$_$G(B
;;; $B$9$+$i!"$I$A$i$r@h$K%m!<%I$7$F$bLdBj$J$/F0:n$7$^$9!#(B
;;; ange-ftp$B$r(Bautoload$B$9$k$J$I$H$$$C$?$3$H$b2DG=$G$7$g$&!#(B
;;;
;;; **$B8=:_$NLdBjE@(B**
;;; NEMACS$B$N>l9g(B:
;;; callproc.c$BCf$N(Bcall-process-region$B4X?t$G(Bkanji-flag$B$N%A%'%C(B
;;; $B%/$r$7$F$$$J$$$?$a!"$b$7%f!<%6$,(Bfind-kanji-process-code
;;; $B4X?t$J$I$r<+J,$G:n$C$F$$$k>l9g$J$I$K$O!"%P%$%J%j%G!<%?$r(B
;;; process$B$HF~=PNO$7$F$$$k$K$b$+$+$o$i$:4A;zJQ49$,9T$o$l$k(B
;;; $B4m81$,$"$j$^$9!#I8=`$N(BNemacs$B$G;H$o$l$F$$$k$J$iBg>fIW$G$9(B
;;; $B$,!#(B
;;;
;;; $BI8=`$N(Bfind-kanji-file-output-code$BEy$G$O%"%Z%s%I;~$N4A;z(B
;;; $B%3!<%I%A%'%C%/$rFC$K9T$J$C$F$$$^$;$s!#$=$N$?$a%"%Z%s%I$9(B
;;; $B$k$?$a$K(Bwrite-region$B$rMxMQ$9$k$H!"$=$N%U%!%$%k$N4A;z%3!<(B
;;; $B%I$,JQ$o$C$F$7$^$&$3$H$,$"$j$^$9!#(B
;;;
;;; MULE$B$N>l9g(B:
;;; $BFC$K$"$j$^$;$s!#(B
;;;
;;; **$B:G8e$K(B***
;;; $B$5$F:G8e$K$J$j$^$7$?$,!"4X?t$rCV$-49$($k$H$$$&%"%$%G%#%"(B
;;; $B$H!"$=$N4J7i$JJ}K!$,5-=R$5$l$F$$$k!"(B`ange-ftp.el(by
;;; ange@hplb.hpl.hp.com)$B$,BgJQ;29M$K$J$C$F$$$^$9!#46<U$7$^(B
;;; $B$9!#(B
;;;
;;; $B$^$?0J2<$N%P%0$N;XE&$d=u8@$rM?$($F$/$l$?J}!9$K$b46<U$7$^(B
;;; $B$9!#(B
;;;  $B8ENS5*:H(B Noriya KOBAYASHI :<nk@ics.osaka-u.ac.jp>
;;;  $BKYFbJ]=((B Horiuchi Yasuhide:<homy@cs.titech.ac.jp>
;;;  $BFj:j=$Fs(B NARAZAKI Shuji   :<narazaki@nttslb.ntt.jp>
;;;  $B9-@n0lIW(B Kazuo Hirokawa   :<hirokawa@rics.co.jp>
;;;  $B550f?.5A(B Kamei, Nobuyoshi :<zic@tci.toshiba.co.jp>
;;;  $BA0ED70(B   Kaoru Maeda      :<maeda@astaire.src.ricoh.co.jp>
;;;
;;; $B2?$+LdBj$,$"$j$^$7$?$i!"(Bkazusi-m@is.aist-nara.ac.jp$B$^$G(B
;;; $B$*CN$i$;2<$5$$!#(B
;;;

(require 'backquote)

;;; Variable which can be set by USER.
;;;
(defvar jam-zcat-filename-list
  '(((("\\.Z$" . "") ("$" . ".Z")) "compress" "uncompress"
     "stdin: not in compressed format\n")
    ((("\\.z$" . "") ("$" . ".z")) "gzip" "gunzip"
     "gunzip: stdin is not in gzip format\n")
    ((("\\.gz$" . "") ("$" . ".gz")) "gzip" "gunzip"
     "gunzip: stdin is not in gzip format\n")
    ((("\\.taz$" . ".tar") ("\\.tar$" . ".taz") ("$" . ".taz"))
     "compress" "uncompress"
     "stdin: not in compressed format\n")
    ((("\\.tgz$" . ".tar") ("\\.tar$" . ".tgz") ("$" . ".tgz"))
     "gzip" "gunzip"
     "gunzip: stdin is not in gzip format\n")
    ((("\\.Y$" . "") ("$" . ".Y")) "yabba" "unyabba"
     "unyabba: fatal: input not in right format\n"))
  "*Each element looks like (((REGEXP . STRRPL) (REGEXP . STRRPL)...)
COMPRESSPROG UNCOMPRESSPROG UNCOMPRESSERRORSTR [COMPRESSARG [UNCOMPRESSARG]]).

 Reading a file whose name matches first REGEXP cause uncompress it and
choose major mode from real-filename that is created replacing matched area
to first STRRPL.  If file not found, search compressed file with
substituted file name by rest (REGEXP . STRRPL)s.

 COMPRESSPROG is compressing program name, UNCOMPRESSPROG is uncompressing
program name.  UNCOMPRESSERRORSTR is error string when uncompressing.
Each of these 3 argument must be a string.

 When compressing, COMPRESSARG is evaluated and use result as a argument
list for compressing.  UNCOMPRESSARG is evaluated when uncompressing, but
if there is no UNCOMPRESSARG, COMPRESSARG is used as UNCOMPRESSARG.

 Note for old version:
 Each element of old version looks like ((REGEXP . STRRPL) COMPRESSPROG
UNCOMPRESSPROG UNCOMPRESSERRORSTR).  And it supported.")

(defvar jam-zcat-hack-ange-ftp t
  "*Non nil means hack to get real filename when using the ange-ftp.")

(defvar jam-zcat-hack-loadablep 'quick
  "*On the Mule, non nil means hack to load compressed file.
  T means change loadablep with lisp function like C's.  Mule look each
directory for the filename with .elc .el .elc.Z .el.Z <no-suffix>.
  'quick means if C's function cannot found the file, lisp function look
each directory for the compressed file.")

(defvar jam-zcat-hack-loadablep-suffixes
  '(".el.z" ".el.gz" "el.Z")
  "*If a file which has any item in this SUFFIX list, it will be read from
load function.")

(defvar jam-zcat-si-mode t
  "*On the Mule, non nil means that this package patch to
si:insert-file-contents and si:write-region.")

;;; Internal variables.
;;;
(defvar jam-zcat-how-to-list nil
  "Current one of jam-zcat-filename-list.")

;;; Internal routines.
;;;
(defun jam-zcat-error-p ()
  "Check a uncompress program's error message."
  (let ((sexp (nth 3 jam-zcat-how-to-list)))
    (cond ((stringp sexp)
	   (string= (buffer-substring
		     (point-min)
		     (min (point-max) (+ (point-min) (length sexp))))
		    sexp))
	  (sexp (eval sexp)))))

(defun jam-zcat-substitute-string (str slist)
  "Return substituted string for STRING.  Replaces matched text by regular
expression of (car SLIST) with (cdr SLIST)."
  (if (string-match (car slist) str)
      (concat (substring str 0 (match-beginning 0))
	      (cdr slist)
	      (substring str (match-end 0) nil))))

(defun jam-zcat-filename-to-realname (fname)
  "Convert FILENAME to real filename, if it was compressed."
  (and (stringp fname)
       (let ((case-fold-search (eq system-type 'vax-vms)))
	 (catch 'exit
	   (mapcar (function
		    (lambda (how-to)
		      (let* ((name-conv (if (stringp (car (car how-to)))
					    (car how-to)
					  (car (car how-to))))
			     (realname (jam-zcat-substitute-string
					fname name-conv)))
			(if realname
			    (progn
			      (setq jam-zcat-how-to-list how-to)
			      (throw 'exit realname))))))
		   jam-zcat-filename-list)
	   fname))))

(defmacro jam-zcat-localize-code (&optional MULE-CODE NEMACS-CODE)
  "If this called on Mule, eval MULE-CODE.  Otherwise eval NEMACS-CODE."
  (` (cond ((boundp 'MULE) (, MULE-CODE))
	   (t (, NEMACS-CODE)))))

(defmacro jam-zcat-check-ange-ftp ()
  "Check about this is called through ange-ftp."
  (` (and (fboundp 'ange-ftp-insert-file-contents) jam-zcat-hack-ange-ftp
	  (boundp 'parsed) parsed
	  (boundp 'path) (stringp path)
	  (not (boundp 'lscmd)))))

(defun jam-zcat-read-string-no-echo (prompt &optional default)
  "Read a string from the user. Echos a . for each character typed.
End with RET, LFD, or ESC. DEL or C-h rubs out.  ^U kills line.
Optional DEFAULT is string to start with."
  (let ((str (if default default ""))
	(c 0)
	(echo-keystrokes 0)
	(cursor-in-echo-area t))
    (while (and (/= c ?\r) (/= c ?\n) (/= c ?\e))
      (message "%s%s"
	       prompt
	       (make-string (length str) ?.))
      (setq c (read-char))
      (if (= c ?\C-u)
	  (setq str "")
	(if (and (/= c ?\b) (/= c ?\177))
	    (setq str (concat str (char-to-string c)))
	  (if (> (length str) 0)
	      (setq str (substring str 0 -1))))))
    (message "")
    (substring str 0 -1)))

(defun jam-zcat-get-crypt-key ()
  (if (and (boundp 'crypt-key) crypt-key)
      crypt-key
    (make-variable-buffer-local 'crypt-key)
    (setq crypt-key (list (jam-zcat-read-string-no-echo
			   "Set key for cryptogram: ")))
    crypt-key))

(defun jam-zcat-detect-code-category (start end)
  "Detect kanji code of buffer string with algorithm like original
insert-file-contents function."
  (if (string-match "^1.0 " mule-version)
      (detect-code-category start end 1)
    (let ((code (code-detect-region start end)))
      (cond ((listp code) (car code))
	    (t code)))))

;;; Routines will replease original one.
;;;
(defun jam-zcat-insert-file-contents (filename &optional visit &rest code)
  "Documented as original."
  (barf-if-buffer-read-only)
  (setq filename (expand-file-name filename))
  (let ((realname (jam-zcat-filename-to-realname filename))
	(realfilename filename)
	(modp (buffer-modified-p))
	result result-code
	local-file-coding-system)
    ;; Support Ange-ftp
    (if (jam-zcat-check-ange-ftp)
	;; now be called through ange-ftp, hack it!
	(progn
	  (setq realname (jam-zcat-filename-to-realname path))
	  (setq realfilename path)))
    (if (or (string= realname realfilename)
	    (featurep 'crypt++)
	    (featurep 'jka-compr))
	(apply 'jam-zcat-real-insert-file-contents filename visit code)
      (setq result			; READ file without any conversion
	    (jam-zcat-localize-code
	     (cond ((and code (listp code) (eq (length code) 3))
		    ;; for Mule-2
		    (setq local-file-coding-system
			  (local-file-coding-system-p))
		    (let ((input-coding-system *noconv*))
		      (apply 'jam-zcat-real-insert-file-contents
			     filename visit code)))
		   (code
		    ;; for Mule-1
		    (cdr (jam-zcat-real-insert-file-contents filename
							     visit *noconv*)))
		   (t
		    (let ((file-coding-system-for-read *noconv*)
			  file-coding-system)
		      (jam-zcat-real-insert-file-contents filename visit))))
	     (let (kanji-flag)
	       (jam-zcat-real-insert-file-contents filename visit))))
      (save-excursion
	(save-restriction
	  (narrow-to-region (point) (+ (point) (nth 1 result)))
					; UNCOMPRESS without kanji code conv.
	  (message "Uncompressing %s ..." realfilename)
	  (condition-case err
	      (progn
		(let ((args (eval (or (nth 5 jam-zcat-how-to-list)
				      (nth 4 jam-zcat-how-to-list)))))
		  (jam-zcat-localize-code
		   (let ((default-process-coding-system
			   (cons *noconv* *noconv*))
			 (file-coding-system *noconv*)
			 (kill-it
			  (not (local-file-coding-system-p))) ; for Mule BUG
			 process-connection-type
			 mc-flag)
		     (apply 'call-process-region (point) (point-max)
			    (nth 2 jam-zcat-how-to-list) t t nil args)
		     (if kill-it
			 (kill-local-variable 'file-coding-system)))
		   (let (kanji-flag
			 default-kanji-process-code
			 service-kanji-code-alist
			 program-kanji-code-alist
			 process-connection-type)
		     (apply 'call-process-region (point) (point-max)
			    (nth 2 jam-zcat-how-to-list) t t nil args))))
		(if (jam-zcat-error-p)
		    (signal 'file-error
			    (list
			     "Uncompressing input file"
			     (format "Unable to %s input file"
				     (upcase (nth 2 jam-zcat-how-to-list)))
			     realfilename))))
	    (file-error
	     (cond ((not visit)
		    (delete-region (point-min) (point-max))
		    (set-buffer-modified-p modp))
		   (t
		    (set-buffer-modified-p modp)
		    (kill-buffer (current-buffer))))
	     (apply 'error "%s: %s, %s" (cdr err))))
	  (message "Uncompressing %s ... done" realfilename)
	  (jam-zcat-localize-code	; CONVERT kanji code
	   (if mc-flag
	       (let ((code (cond ((or (null (nth 0 code))
				      (equal (nth 0 code) *autoconv*)
				      (and code (listp code)
					   (eq (length code) 3))
				      )
				  (jam-zcat-detect-code-category (point-min)
								 (point-max)))
				 (t (nth 0 code)))))
		 (setq result-code (cond ((eq t code) nil)
					 (code)))
		 (if result-code (code-convert-region (point-min) (point-max)
						      code *internal*))))
	   (if (and (boundp 'kanji-flag) kanji-flag)
	       (let ((code (invoke-find-kanji-file-input-code
			    realname visit (point-min) (point-max))))
		 (if (or (eq code 1) (eq code 2))
		     (progn
		       (convert-region-kanji-code (point-min) (point-max)
						  code 3))))))
	  (if visit
	      (set-buffer-modified-p modp))))
      (cond ((and code (listp code) (eq (length code) 3))
	     ;; Mule-2
	     (if (not (eq local-file-coding-system
			  (local-file-coding-system-p)))
		 (kill-local-variable 'file-coding-system))
	     (setq used-coding-system result-code)
	     (list (car result) (- (point-max) (point-min))))
	    (code
	     ;; Mule-1 and jam-zcat-si-mode is t
	     ;; Return coding-system and others
	     (list result-code (car result) (- (point-max) (point-min))))
	    (t
	     (jam-zcat-localize-code
	      ;; On Mule-1, now CHANGE buffer kanji code
	      (if (not file-coding-system)
		  (set-file-coding-system result-code))
	      ;; On NEmacs, CHANGED buffer kanji code
	      ;; at invoke-find-kanji-file-input-code
	      )
	     (list (car result) (- (point-max) (point-min))))))))

(defun jam-zcat-normal-mode (&optional find-file)
  "Documented as original."
  (interactive)
  (let ((buffer-file-name (jam-zcat-filename-to-realname buffer-file-name)))
    (jam-zcat-real-normal-mode find-file)))

(defun jam-zcat-write-region (start end filename &optional append visit
				    &rest code)
  "Documented as original."
  (interactive "r\nFWrite region to file: ")
  (setq filename (expand-file-name filename))
  (let ((realname (jam-zcat-filename-to-realname filename))
	(realname2 (and (stringp visit)
			(jam-zcat-filename-to-realname visit)))
	(realfilename filename)
	(realfilename2 (and (stringp visit) visit)))
    ;; Support Ange-ftp
    (if (jam-zcat-check-ange-ftp)
	;; now be called through ange-ftp, hack it!
	(progn
	  (setq realname (jam-zcat-filename-to-realname path))
	  (setq realfilename path)))
    (if (or (and (string= realname realfilename)
		 (string= realname2 realfilename2))
	    (featurep 'crypt++)
	    (featurep 'jka-compr))
	(apply 'jam-zcat-real-write-region start end filename append visit
	       code)
      (let ((temp (get-buffer-create "*compress*"))
	    (cbuf (current-buffer))
	    (save-start (make-marker))
	    kcode)
	(save-restriction
	  (narrow-to-region start end)
	  (cond ((not append)
		 (setq kcode		; GET kanji code for conv.
		       (jam-zcat-localize-code
			(if mc-flag
			    (or (nth 0 code)
				(and (boundp 'output-coding-system)
				     output-coding-system)
				(if (and current-prefix-arg (interactive-p))
				    (read-coding-system "Coding-system: ")
				  file-coding-system)))
			(if (and (boundp 'kanji-flag) kanji-flag)
			    (invoke-find-kanji-file-output-code
			     start end realname append visit))))
		 (set-buffer temp)
		 (erase-buffer))
		(t
		 (set-buffer temp)
		 (erase-buffer)
					; READ target file
		 (condition-case err
		     (insert-file-contents filename nil)
		   (file-error
		    nil))
		 (setq kcode		; GET kanji code of target file
		       (jam-zcat-localize-code
			(if mc-flag
			    (or file-coding-system kcode))
			(if (and (boundp 'kanji-flag) kanji-flag)
			    (or (invoke-find-kanji-file-output-code
				 start end realname append visit)
				kcode))))))
	  (goto-char (point-max))
	  (insert-buffer cbuf)
	  (jam-zcat-localize-code
	   nil				; On Mule, will CONVERT it at
					; call-process-region
					; On NEmacs, CONVERT kanji code
	   (if (or (eq kcode 1) (eq kcode 2))
	       (convert-region-kanji-code (point-min) (point-max)
					  3 kcode)))
	  (unwind-protect
	      (progn
		(condition-case err
		    (progn		; COMPRESS without/with kanji code
					; conv.
		      (message "Compressing %s ..." realfilename)
		      (let ((args (prog2
				   (set-buffer cbuf)
				   (eval (nth 4 jam-zcat-how-to-list))
				   (set-buffer temp))))
			(jam-zcat-localize-code
			 (let ((default-process-coding-system
				 (cons *noconv* kcode))
			       process-connection-type)
			   (apply 'call-process-region (point-min) (point-max)
				  (nth 1 jam-zcat-how-to-list) t t nil args))
			 (let (kanji-flag
			       default-kanji-process-code
			       service-kanji-code-alist
			       program-kanji-code-alist
			       process-connection-type)
			   (apply 'call-process-region (point-min) (point-max)
				  (nth 1 jam-zcat-how-to-list) t t nil args))))
		      (message "Compressing %s ...done" realfilename))
		  (file-error
		   (apply 'error "%s: %s, %s" (cdr err))))
		(if (eq visit t)
		    (progn
		      (set-buffer cbuf)
		      (let (buffer-read-only)
			(set-marker save-start (point))
			(insert-buffer-substring temp)))
		  (set-marker save-start (point-min)))
		(jam-zcat-localize-code	; WRITE file without any conversion
		 (if code
		     (let (mc-flag selective-display)
		       (jam-zcat-real-write-region save-start (point) filename
						   nil visit *noconv*))
		   (let ((file-coding-system *noconv*)
			 (output-coding-system *noconv*)
			 mc-flag
			 selective-display)
		     (jam-zcat-real-write-region save-start (point)
						 filename nil visit)))
		 (let (kanji-flag selective-display)
		   (jam-zcat-real-write-region save-start (point)
					       filename nil visit)))
		(if (eq visit t)
		    (let (buffer-read-only)
		      (delete-region save-start (point))
		      (set-buffer-modified-p nil))
		  (set-buffer cbuf))
		nil)
	    (kill-buffer temp)))
	nil))))

(defun jam-zcat-get-file-buffer (filename)
  "Documented as original."
  (setq filename (expand-file-name filename))
  (or (jam-zcat-real-get-file-buffer filename)
      (if (file-exists-p filename)
	  nil
	(catch 'exit
	  (mapcar (function
		   (lambda (buf)
		     (if (string= (jam-zcat-filename-to-realname
				   (buffer-file-name buf)) filename)
			 (throw 'exit buf))))
		  (buffer-list))
	  nil))))

(defun jam-zcat-loadablep-body (str suffixes)
  (catch 'exit
    (mapcar
     '(lambda (dir)
	(let ((file (expand-file-name str dir)))
	  (mapcar
	   '(lambda (suffix)
	      (if (and (not (file-directory-p (concat file suffix)))
		       (file-readable-p (concat file suffix)))
		  (throw 'exit (concat file suffix))))
	   suffixes)))
     load-path)
    nil))

(defun jam-zcat-loadablep (str &optional nosuffix)
  "Documented as original."
  (cond ((or (not jam-zcat-hack-loadablep) nosuffix)
	 (jam-zcat-real-loadablep str nosuffix))
	((eq jam-zcat-hack-loadablep 'quick)
	 (or (jam-zcat-real-loadablep str nosuffix)
	     (jam-zcat-loadablep-body str jam-zcat-hack-loadablep-suffixes)))
	(t
	 (jam-zcat-loadablep-body str (append '(".elc" ".el")
					      jam-zcat-hack-loadablep-suffixes
					      '(nil))))))

;;; Routines to replace.
;;;   Original cames from ange-ftp v4.20
;;;
(defvar jam-zcat-overwrite-msg
  "Note: This function has been extended to deal with compressed file.")

(defun jam-zcat-safe-documentation (fun)
  "A documentation function that isn't quite as fragile."
  (condition-case ()
      (documentation fun)
    (error nil)))

(defun jam-zcat-overwrite-fn (fun)
  "Replace FUN's function definition with jam-zcat-FUN's, saving the
original definition as jam-zcat-real-FUN.  The original documentation is
placed on the new definition suitably augmented."
  (let* ((name (symbol-name fun))
	 (saved (intern (concat "jam-zcat-real-" name)))
	 (new (intern (concat "jam-zcat-" name)))
	 (nfun (symbol-function new))
	 (exec-directory (if (or (equal (nth 3 command-line-args) "dump")
				 (equal (nth 4 command-line-args) "dump"))
			     "../etc/"
			   exec-directory)))			 
    
    ;; *** This is unnecessary for any jam-zcat function (I think):
    (while (symbolp nfun)
      (setq nfun (symbol-function nfun)))
    
    ;; Interpose the jam-zcat function between the function symbol and the
    ;; original definition of the function symbol AT TIME OF FIRST LOAD.
    ;; We must only redefine the symbol-function of FUN the very first
    ;; time, to avoid blowing away stuff that overloads FUN after this.
    
    ;; We direct the function symbol to the jam-zcat's function symbol
    ;; rather than function definition to allow reloading of this file or
    ;; redefining of the individual function (e.g., during debugging)
    ;; later after some other code has been loaded on top of our stuff.
    
    (or (fboundp saved)
	(progn
	  (fset saved (symbol-function fun))
	  (fset fun new)))
    
    ;; Rewrite the doc string on the new jam-zcat function.  This should
    ;; be done every time the file is loaded (or a function is redefined),
    ;; because the underlying overloaded function may have changed its doc
    ;; string.
    
    (let* ((doc-str (jam-zcat-safe-documentation saved))
	   (ndoc-str (concat doc-str (and doc-str "\n")
			     jam-zcat-overwrite-msg)))
      
      (cond ((listp nfun)
	     ;; Probe to test whether function is in preloaded read-only
	     ;; memory, and if so make writable copy:
	     (condition-case nil
		 (setcar nfun (car nfun))
	       (error
		(setq nfun (copy-sequence nfun)) ; shallow copy only
		(fset new nfun)))
	     (let ((ndoc-cdr (nthcdr 2 nfun)))
	       (if (stringp (car ndoc-cdr))
		   ;; Replace the existing docstring.
		   (setcar ndoc-cdr ndoc-str)
		 ;; There is no docstring.  Insert the overwrite msg.
		 (setcdr ndoc-cdr (cons (car ndoc-cdr) (cdr ndoc-cdr)))
		 (setcar ndoc-cdr jam-zcat-overwrite-msg))))
	    (t
	     ;; it's an emacs19 compiled-code object
	     (let ((new-code (append nfun nil))) ; turn it into a list
	       (if (nthcdr 4 new-code)
		   (setcar (nthcdr 4 new-code) ndoc-str)
		 (setcdr (nthcdr 3 new-code) (cons ndoc-str nil)))
	       (fset new (apply 'make-byte-code new-code))))))))

;;; V19's handler
(defun jam-zcat-handler (operation &rest args)
  (let ((op (get operation 'jam-zcat)))
      (if op
	  (apply op args)
	(jam-zcat-run-real-handler operation args))))
(defun jam-zcat-run-real-handler (operation args)
  (let ((inhibit-file-name-handlers
	 (cons 'jam-zcat-handler
	       (and (eq inhibit-file-name-operation operation)
		    inhibit-file-name-handlers)))
	(inhibit-file-name-operation operation))
    (apply operation args)))

(jam-zcat-localize-code
 (cond ((string< mule-version "2.1")
	(cond (jam-zcat-si-mode
	       (fset 'jam-zcat-si:insert-file-contents
		     (symbol-function 'jam-zcat-insert-file-contents))
	       (jam-zcat-overwrite-fn 'si:insert-file-contents)
	       (fset 'jam-zcat-real-insert-file-contents
		     (symbol-function 'jam-zcat-real-si:insert-file-contents))
	       (fset 'jam-zcat-si:write-region
		     (symbol-function 'jam-zcat-write-region))
	       (jam-zcat-overwrite-fn 'si:write-region)
	       (fset 'jam-zcat-real-write-region
		     (symbol-function 'jam-zcat-real-si:write-region))
	       (jam-zcat-overwrite-fn 'loadablep))
	      (t
	       (jam-zcat-overwrite-fn 'insert-file-contents)
	       (jam-zcat-overwrite-fn 'write-region))))
       (t
	(setq file-name-handler-alist
	      (cons '("" . jam-zcat-handler)
		    file-name-handler-alist))
	(put 'insert-file-contents
	     'jam-zcat 'jam-zcat-insert-file-contents)
	(put 'write-region
	     'jam-zcat 'jam-zcat-write-region)
	(defun jam-zcat-real-insert-file-contents (&rest args)
	  (jam-zcat-run-real-handler 'insert-file-contents args))
	(defun jam-zcat-real-write-region (&rest args)
	  (jam-zcat-run-real-handler 'write-region args))))
 (progn
   (jam-zcat-overwrite-fn 'insert-file-contents)
   (jam-zcat-overwrite-fn 'write-region)))
(jam-zcat-overwrite-fn 'normal-mode)
(jam-zcat-overwrite-fn 'get-file-buffer)

;;; Routines for hook.
;;;
(defun jam-zcat-search-compressed-file (name)
  (catch 'exit
    (mapcar (function
	     (lambda (how-to)
	       (if (consp (cdr (car how-to)))
		   (mapcar (function
			    (lambda (rev-name-conv)
			      (let ((fname (jam-zcat-substitute-string
					    name rev-name-conv)))
				(if (and fname
					 (file-exists-p fname))
				    (throw 'exit fname)))))
			   (cdr (car how-to))))
	       nil))
	    jam-zcat-filename-list)
      nil))

(defun jam-zcat-find-file-not-found-hook ()
  " Called when a find-file command has not been able to find the specfied
file. Read and uncompress when a compressed file exists."
  (if (string= (jam-zcat-filename-to-realname buffer-file-name)
	       buffer-file-name)
      (let ((compressed-file (jam-zcat-search-compressed-file
			      buffer-file-name)))
	(if compressed-file
	    (progn
	      (setq buffer-file-name compressed-file)
	      (insert-file-contents compressed-file t)
	      (setq error nil)
	      t)))))

(or (memq 'jam-zcat-find-file-not-found-hook find-file-not-found-hooks)
    (setq find-file-not-found-hooks
	  (cons 'jam-zcat-find-file-not-found-hook find-file-not-found-hooks)))

;;; Other stuff
;;;
(provide 'jam-zcat)
(run-hooks 'jam-zcat-load-hook)
