;; New 'load' function.
;; Gives more useful error information than original.

;;; 92.12.17 written for Mule Ver.0.9.7 by T.Enami <enami@sys.ptg.sony.co.jp>

(defun load (str &optional noerror nomessage nosuffix)
  "Execute a file of Lisp code named FILE.
First tries FILE with .elc appended, then tries with .el,
 then tries FILE unmodified.  Searches directories in load-path.
If optional second arg NOERROR is non-nil,
 report no error if FILE doesn't exist.
Print messages at start and end of loading unless
 optional third arg NOMESSAGE is non-nil.
If optional fourth arg NOSUFFIX is non-nil, don't try adding
 suffixes .elc or .el to the specified name FILE.
Return t if file exists."
  (let ((file (substitute-in-file-name str)))
    (if (or (<= (length file) 0)
	    ;; 92.3.18 by T.Enami
	    ;; (not (setq file (loadablep file)))
	    (not (setq file (loadablep file nosuffix)))
	    )
	(if noerror nil
	  (error "Cannot open load file %s" str))
      (let* ((default-major-mode 'fundamental-mode)
	     (buffer (generate-new-buffer " *load-temp*"))
	     marker			; point to read sexp.
	     lastpos			; last point began to read.
	     (load-in-progress t)
	     (elc (string= ".elc" (substring file -4)))) ; 92.7.13 by K.Handa
	(or nomessage (message (if elc "Loading %s..." "LOADING %s...") str))
	(unwind-protect
	    (condition-case error
		(catch 'parsing-done
		  (save-excursion
		    (set-buffer buffer)
		    (insert "\n(throw 'parsing-done nil)")
		    (goto-char (point-min))
		    (si:insert-file-contents file nil ; 92.7.13 by K.Handa
					     (if elc *internal* *autoconv*))
		    ;; no need to goto point-min.  already stay.
		    (setq marker (point-marker)))
		  (while t
		    (setq lastpos (marker-position marker))
		    (eval (read marker)) ))
	      ;; signal the caught error.
	      (error (signal (car error)
			     (append (cdr error)
				     ;; with information about failure pos.
				     (list (format "in region %d - %s, %s"
						   lastpos
						   (save-excursion
						     (set-buffer buffer)
						     (goto-char marker)
						     (if (eobp)
							 "point-max"
						       (int-to-string (point))
						       ))
						   file))))))
	  (kill-buffer buffer) )
	(or nomessage noninteractive	; 92.7.13 by K.Handa
	    (message (if elc "Loading %s...done" "LOADING %s...done") str)))
      t)))
