/*
 * Author:	William Chia-Wei Cheng (william@cs.ucla.edu)
 *
 * Copyright (C) 1990, 1991, 1992, William Cheng.
 * 
 * Permission limited to the use, copy, modify, and distribute this software
 * and its documentation for any purpose is hereby granted by the Author without
 * fee, provided that the above copyright notice appear in all copies and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation, and that the name of the Author not be used
 * in advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.  The Author makes no
 * representations about the suitability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.  All other
 * rights are reserved by the Author.
 *
 * THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF
 * USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
 * OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */
#ifndef lint
static char RCSid[] =
      "@(#)$Header: /amnt/kona/tangram/u/william/X11/TGIF2/RCS/arc.c,v 2.67 1993/04/28 05:06:11 william Exp $";
#endif

#include <math.h>
#include <stdio.h>
#include <X11/Xlib.h>
#include "const.h"
#include "types.h"

#include "cmd.e"
#include "color.e"
#include "cursor.e"
#include "dialog.e"
#include "dup.e"
#include "file.e"
#include "grid.e"
#include "mainloop.e"
#include "obj.e"
#include "pattern.e"
#include "poly.e"
#include "raster.e"
#include "ruler.e"
#include "select.e"
#include "setup.e"
#include "special.e"
#include "xpixmap.e"

#define EXPAND_BBOX(bbox,x,y) \
   if ((x)<(bbox)->ltx) (bbox)->ltx=(x); if ((y)<(bbox)->lty) (bbox)->lty=(y); \
   if ((x)>(bbox)->rbx) (bbox)->rbx=(x); if ((y)>(bbox)->rby) (bbox)->rby=(y)

int	arcDrawn = FALSE;

/*
 * 0 degree is horizontal in the direction of the X axis.
 * Positive angles measures counter-clockwise from 0 degree.
 * Negative angles measures clockwise from 0 degree.
 * Angle1 means the start angle.
 * Angle2 means the amount between angle1 and the real angle2.
 */

static
int ArcDirection (xc, yc, x1, y1, x2, y2)
   int	xc, yc, x1, y1, x2, y2;
{
   register int		dx, dy;
   register double	theta1, theta2;

   dx = x1-xc; dy = y1-yc;
   theta1 = (dx==0) ? ((dy>=0) ? M_PI/2.0 : -M_PI/2.0) :
         atan2 ((double)(dy), (double)(dx));
   theta2 = (x2==xc) ? ((y2>=yc) ? M_PI/2.0 : -M_PI/2.0) :
         atan2 ((double)(y2-yc), (double)(x2-xc));
   if (theta1 < 0) theta1 += 2*M_PI;
   if (theta2 < 0) theta2 += 2*M_PI;

   if (theta2 > theta1)
   {
      if (theta2-theta1 >= 2*M_PI-theta2+theta1)
         return (ARC_CCW);
      else
         return (ARC_CW);
   }
   else if (theta1 > theta2)
   {
      if (theta1-theta2 >= 2*M_PI-theta1+theta2)
         return (ARC_CW);
      else
         return (ARC_CCW);
   }
   else
      return (ARC_CCW);
}

void PointsToArc (xc, yc, x1, y1, x2, y2, dir, ltx, lty, w, h, angle1, angle2)
   int	xc, yc, x1, y1, x2, y2, dir;
   int	* ltx, * lty, * w, * h, * angle1, * angle2;
   /* Only good is the points are part of a circle, not an oval */
{
   register int	dx, dy, radius, theta1, theta2, d_theta;
   double	tmp_theta;

   dx = x1-xc; dy = y1-yc;
   radius = (int)sqrt((double)(dx*dx+dy*dy));
   *ltx = xc-radius; *lty = yc-radius;
   *w = *h = 2*radius;
   tmp_theta = (dx==0) ? ((dy>=0) ? M_PI/2.0 : -M_PI/2.0) :
         atan2 ((double)(dy),(double)(dx));
   theta1 = (int)(tmp_theta/M_PI*(-180));
   tmp_theta = (x2==xc) ? ((y2>=yc) ? M_PI/2.0 : -M_PI/2.0) :
         atan2 ((double)(y2-yc),(double)(x2-xc));
   theta2 = (int)(tmp_theta/M_PI*(-180));
   /* NOTE:  *angle1 must be between -180 degrees and +180 degrees */
   *angle1 = theta1*64;
   d_theta = theta2-theta1;
   switch (dir)
   {
      case ARC_CCW: if (d_theta < 0) d_theta = 360 + d_theta; break;
      case ARC_CW:  if (d_theta > 0) d_theta = d_theta - 360; break;
   }
   if (d_theta == 0) d_theta = 360;
   *angle2 = d_theta * 64;
}

void ArcRealX2Y2 (ArcPtr, RealX2, RealY2)
   register struct ArcRec	* ArcPtr;
   int				* RealX2, * RealY2;
{
   register double	angle_in_radian;
   int			w = ArcPtr->w, h = ArcPtr->h;

   angle_in_radian = (ArcPtr->angle1+ArcPtr->angle2)*M_PI/180/64;
   *RealX2 = ArcPtr->xc + round((w/2)*cos(angle_in_radian));
   *RealY2 = ArcPtr->yc - round((h/2)*sin(angle_in_radian));
}

static
void CalcArcOBBox (ArcPtr, bbox)
   struct ArcRec	* ArcPtr;
   struct BBRec		* bbox;
{
   register int	theta1, theta2;
   int		real_x2, real_y2, dir = ArcPtr->dir;
   int		ltx = ArcPtr->ltx, lty = ArcPtr->lty;
   int		w = ArcPtr->w, h = ArcPtr->h;
   int		pass_theta1 = FALSE, coverage = 0, angle;

   theta1 = (ArcPtr->angle1)/64;
   theta2 = theta1 + (ArcPtr->angle2)/64;

   ArcRealX2Y2 (ArcPtr, &real_x2, &real_y2);

   if (ArcPtr->fill == NONEPAT)
   {  /* don't counter the center of the arc */
      bbox->ltx = min(ArcPtr->x1,real_x2);
      bbox->lty = min(ArcPtr->y1,real_y2);
      bbox->rbx = max(ArcPtr->x1,real_x2);
      bbox->rby = max(ArcPtr->y1,real_y2);
   }
   else
   {
      bbox->ltx = min(ArcPtr->xc,min(ArcPtr->x1,real_x2));
      bbox->lty = min(ArcPtr->yc,min(ArcPtr->y1,real_y2));
      bbox->rbx = max(ArcPtr->xc,max(ArcPtr->x1,real_x2));
      bbox->rby = max(ArcPtr->yc,max(ArcPtr->y1,real_y2));
   }

   if (theta2 < -180) theta2 += 360;
   if (theta2 > 180) theta2 -= 360;

   if (theta1 < 0) theta1 += 360;
   if (theta2 < 0) theta2 += 360;

   if (theta1 == theta2)
      coverage = 0xf;
   else if (dir == ARC_CCW)
   {
      angle = 0;
      while (angle < theta2 || !pass_theta1)
      {
         if (angle >= theta1 && !pass_theta1)
         {
            pass_theta1 = TRUE;
            if (theta2 > theta1 && angle >= theta2) break;
            if (theta2 < theta1) angle -= 360;
         }
         if (pass_theta1) coverage |= 1 << (((angle+360)/90) % 4);
         angle = (angle == 360) ? 0 : (angle+90);
      }
   }
   else
   {
      angle = 360;
      while (angle > theta2 || !pass_theta1)
      {
         if (angle <= theta1 && !pass_theta1)
         {
            pass_theta1 = TRUE;
            if (theta2 < theta1 && angle <= theta2) break;
            if (theta2 > theta1) angle += 360;
         }
         if (pass_theta1) coverage |= 1 << ((angle/90) % 4);
         angle = (angle == 0) ? 360 : (angle-90);
      }
   }
   if (coverage & 0x1) { EXPAND_BBOX(bbox,(int)(ltx+w),(int)(lty+h/2)); }
   if (coverage & 0x2) { EXPAND_BBOX(bbox,(int)(ltx+w/2),lty); }
   if (coverage & 0x4) { EXPAND_BBOX(bbox,ltx,(int)(lty+h/2)); }
   if (coverage & 0x8) { EXPAND_BBOX(bbox,(int)(ltx+w/2),(int)(lty+h)); }
}

void CalcArcBBox (arc_ptr, obbox, bbox)
   struct ArcRec	* arc_ptr;
   struct BBRec		obbox, * bbox;
{
   int		ltx=obbox.ltx, lty=obbox.lty, rbx=obbox.rbx, rby=obbox.rby;
   int		dx, dy, theta=0, dir=arc_ptr->dir;
   int		angle1=arc_ptr->angle1, angle2=arc_ptr->angle2;
   int		w=arc_ptr->w, h=arc_ptr->h, x, y;
   int		aw=arc_ptr->aw, ah=arc_ptr->ah;
   int		x1=arc_ptr->x1, y1=arc_ptr->y1, x2, y2;
   double	sine, cosine;

   if (arc_ptr->style & LS_LEFT)
   {
      switch (dir)
      {
         case ARC_CCW: theta = (int)(angle1/64)-90; break;
         case ARC_CW: theta = (int)(angle1/64)+90; break;
      }
      dx = -round(w*cos(theta*M_PI/180));
      dy = round(h*sin(theta*M_PI/180));
      if (dx==0 && dy==0)
      {
         sine = cosine = ((double)0.0);
         x = x1; y = y1;
         if (x < ltx) ltx = x; if (y < lty) lty = y;
         if (x > rbx) rbx = x; if (y > rby) rby = y;
      }
      else
      {
         double len = (double) sqrt ((double)(dx*dx+dy*dy));

         sine = dy/len;
         cosine = dx/len;

         x = round(x1 + aw*cosine - ah*sine);
         y = round(y1 + aw*sine + ah*cosine);
         if (x < ltx) ltx = x; if (y < lty) lty = y;
         if (x > rbx) rbx = x; if (y > rby) rby = y;

         x = round(x1 + aw*cosine + ah*sine);
         y = round(y1 + aw*sine - ah*cosine);
         if (x < ltx) ltx = x; if (y < lty) lty = y;
         if (x > rbx) rbx = x; if (y > rby) rby = y;
      }
   }

   if (arc_ptr->style & LS_RIGHT)
   {
      switch (dir)
      {
         case ARC_CCW: theta = (int)((angle1+angle2)/64)-90; break;
         case ARC_CW: theta = (int)((angle1+angle2)/64)+90; break;
      }
      dx = -round(w*cos(theta*M_PI/180));
      dy = round(h*sin(theta*M_PI/180));
      if (dx==0 && dy==0)
      {
         sine = cosine = ((double)0.0);
         ArcRealX2Y2 (arc_ptr, &x2, &y2);
         x = x2; y = y2;
         if (x < ltx) ltx = x; if (y < lty) lty = y;
         if (x > rbx) rbx = x; if (y > rby) rby = y;
      }
      else
      {
         double len = (double) sqrt ((double)(dx*dx+dy*dy));

         sine = dy/len;
         cosine = dx/len;

         ArcRealX2Y2 (arc_ptr, &x2, &y2);

         x = round(x2 - aw*cosine + ah*sine);
         y = round(y2 - aw*sine - ah*cosine);
         if (x < ltx) ltx = x; if (y < lty) lty = y;
         if (x > rbx) rbx = x; if (y > rby) rby = y;

         x = round(x2 - aw*cosine - ah*sine);
         y = round(y2 - aw*sine + ah*cosine);
         if (x < ltx) ltx = x; if (y < lty) lty = y;
         if (x > rbx) rbx = x; if (y > rby) rby = y;
      }
   }

   bbox->ltx = min(ltx, obbox.ltx-arc_ptr->width/2);
   bbox->lty = min(lty, obbox.lty-arc_ptr->width/2);
   bbox->rbx = max(rbx, obbox.rbx+arc_ptr->width/2);
   bbox->rby = max(rby, obbox.rby+arc_ptr->width/2);
}

static
void DumpArcPSPath (FP, xc, yc, xr, yr, dir, a1, a2, outline, blank1, blank2)
   FILE	* FP;
   int	xc, yc, xr, yr, dir, a1, a2, outline;
   char	* blank1, * blank2;
{
   fprintf (FP, "%snewpath\n", blank1);
   if (outline) fprintf (FP, "%s%1d %1d moveto\n", blank2, xc, yc);
#ifdef INVERT_CTM_BUG
   switch (dir)
   {
      case ARC_CCW:
         fprintf (FP, "%s%1d %1d %1d %s %1d %s %1d %1d tgifarc\n", blank2,
               xc, yc, xr, "tgif_min_radius", yr, "tgif_min_radius", a1, a2);
         break;
      case ARC_CW:
         fprintf (FP, "%s%1d %1d %1d %s %1d %s %1d %1d tgifarcn\n", blank2,
               xc, yc, xr, "tgif_min_radius", yr, "tgif_min_radius", a1, a2);
         break;
   }
#else
   switch (dir)
   {
      case ARC_CCW:
         fprintf (FP, "%s%1d %1d %1d %1d %1d %1d tgifarc\n", blank2,
               xc, yc, xr, yr, a1, a2);
         break;
      case ARC_CW:
         fprintf (FP, "%s%1d %1d %1d %1d %1d %1d tgifarcn\n", blank2,
               xc, yc, xr, yr, a1, a2);
         break;
   }
#endif
   if (outline) fprintf (FP, "%s%1d %1d lineto\n", blank2, xc, yc);
}

static
void DumpArcArrow (FP,Angle,Dir,X,Y,XRad,YRad,AW,AH,Pen,ColorIndex)
   FILE	* FP;
   int	Angle, Dir, X, Y, XRad, YRad, AW, AH, Pen, ColorIndex;
{
   int		i, dx, dy, theta=0;
   struct BBRec	bbox;
   XPoint	v[2];
   double	sine, cosine;

   switch (Dir)
   {
      case ARC_CCW: theta = Angle-90; break;
      case ARC_CW: theta = Angle+90; break;
   }
   dx = round(XRad*cos(theta*M_PI/180));
   dy = round(YRad*sin(theta*M_PI/180));

   fprintf (FP, "gsave\n");

   if (colorDump)
   {
      if (dx==0 && dy==0)
      {
         v[0].x = v[1].x = X;
         v[0].y = v[1].y = Y;
      }
      else
      {
         double len = (double) sqrt ((double)(dx*dx+dy*dy));

         sine = dy/len;
         cosine = dx/len;

         v[0].x = round(X - AW*cosine + AH*sine);
         v[0].y = round(Y - AW*sine - AH*cosine);
         v[1].x = round(X - AW*cosine - AH*sine);
         v[1].y = round(Y - AW*sine + AH*cosine);
      }

      bbox.ltx = bbox.rbx = X;
      bbox.lty = bbox.rby = Y;

      for (i = 0; i < 2; i++)
      {
         if (v[i].x < bbox.ltx) bbox.ltx = v[i].x;
         if (v[i].y < bbox.lty) bbox.lty = v[i].y;
         if (v[i].x > bbox.rbx) bbox.rbx = v[i].x;
         if (v[i].y > bbox.rby) bbox.rby = v[i].y;
      }
      fprintf (FP, "   newpath\n");
      fprintf (FP, "      %1d %1d %1d %1d %1d %1d tgifarrowtip\n", X, Y, AW, AH,
            dx, dy);
      fprintf (FP, "   1 setgray closepath fill\n");
      fprintf (FP, "   %.3f %.3f %.3f setrgbcolor\n",
            ((float)tgifColors[ColorIndex].red/maxRGB),
            ((float)tgifColors[ColorIndex].green/maxRGB),
            ((float)tgifColors[ColorIndex].blue/maxRGB));
   }
   else if (Pen > BACKPAT)
   {
      GrayCheck (Pen);
      if (useGray)
         fprintf (FP, "   %s setgray\n", GrayStr(Pen));
      else
         fprintf (FP, "   pat%1d %s\n", Pen, patternStr);
   }

   if (!(colorDump && Pen==BACKPAT))
   {
      fprintf (FP, "   newpath\n");
      fprintf (FP, "      %1d %1d %1d %1d %1d %1d tgifarrowtip\n", X, Y, AW, AH,
            dx, dy);
   }

   if (colorDump)
   {
      switch (Pen)
      {
         case SOLIDPAT: fprintf (FP, "   closepath fill\n"); break;
         case BACKPAT: break;
         default:
            fprintf (FP, "   closepath eoclip\n");
            DumpPatFill (FP, Pen, 8, bbox, "   ");
            break;
      }
   }
   else
   {
      switch (Pen)
      {
         case SOLIDPAT: fprintf (FP, "   closepath fill\n"); break;
         case BACKPAT: fprintf (FP, "   closepath 1 setgray fill\n"); break;
         default: fprintf (FP, "   closepath fill\n"); break;
      }
   }
   fprintf (FP, "grestore\n");
}

static
void DumpArcPath (FP,ObjPtr,xc,yc,xr,yr,dir,angle1,angle2,width,pen,dash)
   FILE			* FP;
   struct ObjRec	* ObjPtr;
   int			xc, yc, xr, yr, dir, angle1, angle2;
   int			width, pen, dash;
{
   register int	i;

   if (width != 1) fprintf (FP, "   %1d setlinewidth\n", width);
   if (dash != 0)
   {
      fprintf (FP, "   [");
      for (i = 0; i < dashListLength[dash]-1; i++)
         fprintf (FP, "%1d ", (int)(dashList[dash][i]));
      fprintf (FP, "%1d] 0 setdash\n",
            (int)(dashList[dash][dashListLength[dash]-1]));
   }

   if (!colorDump && pen > BACKPAT)
   {
      GrayCheck (pen);
      if (useGray)
         fprintf (FP, "   %s setgray\n", GrayStr(pen));
      else
         fprintf (FP, "   pat%1d %s\n", pen, patternStr);
   }

   DumpArcPSPath (FP,xc,yc,xr,yr,dir,angle1,angle2,FALSE,"   ","      ");

   switch (pen)
   {
      case SOLIDPAT: fprintf (FP, "   stroke\n"); break;
      case BACKPAT: fprintf (FP, "   1 setgray stroke 0 setgray\n"); break;
      default:
         if (colorDump)
         {
            fprintf (FP, "   flattenpath strokepath clip\n");
            DumpPatFill (FP, pen, 8, ObjPtr->bbox, "   ");
         }
         else
            fprintf (FP, "   stroke\n");
   }
   if (dash != 0) fprintf (FP, "   [] 0 setdash\n");
   if (width != 1) fprintf (FP, "   1 setlinewidth\n");
}

void DumpArcObj (FP, ObjPtr)
   FILE			* FP;
   struct ObjRec	* ObjPtr;
{
   register struct ArcRec	* arc_ptr = ObjPtr->detail.a;
   int				fill, width, pen, dash, color_index;
   int				xc, yc, xr, yr, dir, angle1, angle2, style;
   int				x1, y1, real_x2, real_y2, aw, ah;

   fill = arc_ptr->fill;
   width = arc_ptr->width;
   aw = arc_ptr->aw;
   ah = arc_ptr->ah;
   pen = arc_ptr->pen;
   dash = arc_ptr->dash;
   style = arc_ptr->style;
   xc = arc_ptr->xc; yc = arc_ptr->yc;
   x1 = arc_ptr->x1; y1 = arc_ptr->y1;
   xr = (int)(arc_ptr->w/2); yr = (int)(arc_ptr->h/2);
   dir = arc_ptr->dir;
   angle1 = -(int)(arc_ptr->angle1/64);
   angle2 = -(int)(arc_ptr->angle2/64) + angle1;

   if (fill == NONEPAT && pen == NONEPAT) return;

   fprintf (FP, "%% ARC\n");
   color_index = ObjPtr->color;
   if (colorDump)
      fprintf (FP, "%.3f %.3f %.3f setrgbcolor\n",
            ((float)tgifColors[color_index].red/maxRGB),
            ((float)tgifColors[color_index].green/maxRGB),
            ((float)tgifColors[color_index].blue/maxRGB));

   switch (fill)
   {
      case NONEPAT: break;
      case SOLIDPAT:
         DumpArcPSPath (FP,xc,yc,xr,yr,dir,angle1,angle2,TRUE,"","   ");
         fprintf (FP, "   fill\n");
         break;
      case BACKPAT:
         DumpArcPSPath (FP,xc,yc,xr,yr,dir,angle1,angle2,TRUE,"","   ");
         fprintf (FP, "   closepath 1 setgray fill\n");
         if (colorDump)
            fprintf (FP, "   %.3f %.3f %.3f setrgbcolor\n",
                  ((float)tgifColors[color_index].red/maxRGB),
                  ((float)tgifColors[color_index].green/maxRGB),
                  ((float)tgifColors[color_index].blue/maxRGB));
         else
            fprintf (FP, "   0 setgray\n");
         break;
      default:
         fprintf (FP, "gsave\n");
         if (!colorDump)
         {
            GrayCheck (fill);
            if (useGray)
               fprintf (FP, "   %s setgray\n", GrayStr(fill));
            else
               fprintf (FP, "   pat%1d %s\n", fill, patternStr);
         }
         else
         {
            DumpArcPSPath (FP, xc, yc, xr, yr, dir, angle1, angle2, TRUE,
                  "   ","      ");
            fprintf (FP, "   closepath 1 setgray fill\n");
            fprintf (FP, "   %.3f %.3f %.3f setrgbcolor\n",
                  ((float)tgifColors[color_index].red/maxRGB),
                  ((float)tgifColors[color_index].green/maxRGB),
                  ((float)tgifColors[color_index].blue/maxRGB));
         }
         DumpArcPSPath (FP,xc,yc,xr,yr,dir,angle1,angle2,TRUE,"   ","      ");
         if (colorDump)
         {
            fprintf (FP, "   closepath clip\n");
            DumpPatFill (FP, fill, 8, ObjPtr->bbox, "   ");
         }
         else
            fprintf (FP, "   fill\n");
         fprintf (FP, "grestore\n");
         break;
   }

   if (pen == NONEPAT) { fprintf (FP, "\n"); return; }

   fprintf (FP, "gsave\n");

   if (colorDump && pen > BACKPAT)
   {
      DumpArcPath (FP,ObjPtr,xc,yc,xr,yr,dir,angle1,angle2,
            width,BACKPAT,0);
      if (colorDump)
         fprintf (FP, "   %.3f %.3f %.3f setrgbcolor\n",
               ((float)tgifColors[color_index].red/maxRGB),
               ((float)tgifColors[color_index].green/maxRGB),
               ((float)tgifColors[color_index].blue/maxRGB));
   }
   DumpArcPath (FP,ObjPtr,xc,yc,xr,yr,dir,angle1,angle2,width,pen,dash);

   fprintf (FP, "grestore\n");

   ArcRealX2Y2 (arc_ptr, &real_x2, &real_y2);

   if (style&LS_LEFT)
      DumpArcArrow (FP,angle1,dir,x1,y1,xr,yr,-aw,-ah,pen,color_index);
   if (style&LS_RIGHT)
      DumpArcArrow (FP,angle2,dir,real_x2,real_y2,xr,yr,aw,ah,pen,
            color_index);

   fprintf (FP, "\n");
}

void DrawArcObj (window, XOff, YOff, ObjPtr)
   Window		window;
   int			XOff, YOff;
   struct ObjRec	* ObjPtr;
{
   struct ArcRec	* arc_ptr = ObjPtr->detail.a;
   int			fill, width, pen, dash, pixel, real_x_off, real_y_off;
   int			ltx, lty, w, h, angle1, angle2, x1, y1, x2, y2;
   int			real_x2, real_y2, aw, ah, dir;
   double		dx, dy, sine, cosine;
   XPoint		tmp_v[4];
   XGCValues		values;

   real_x_off = (zoomedIn ? XOff : (XOff>>zoomScale)<<zoomScale);
   real_y_off = (zoomedIn ? YOff : (YOff>>zoomScale)<<zoomScale);

   ltx = ZOOMED_SIZE(arc_ptr->ltx-real_x_off);
   lty = ZOOMED_SIZE(arc_ptr->lty-real_y_off);
   w = ZOOMED_SIZE(arc_ptr->ltx+arc_ptr->w-real_x_off)-ltx;
   h = ZOOMED_SIZE(arc_ptr->lty+arc_ptr->h-real_y_off)-lty;
   angle1 = arc_ptr->angle1;
   angle2 = arc_ptr->angle2;

   fill = ObjPtr->detail.a->fill;
   width = ObjPtr->detail.a->width;
   pen = ObjPtr->detail.a->pen;
   dash = ObjPtr->detail.a->dash;
   pixel = colorPixels[ObjPtr->color];

   if (fill != 0)
   {
      values.foreground = (fill == NONEPAT) ? myBgPixel : pixel;
      values.function = GXcopy;
      values.fill_style = FillOpaqueStippled;
      values.stipple = patPixmap[fill];
      XChangeGC (mainDisplay, drawGC,
            GCForeground | GCFunction | GCFillStyle | GCStipple, &values);
      XFillArc (mainDisplay, window, drawGC, ltx, lty, w, h, angle1, angle2);
   }

   if (pen == 0) return;

   values.foreground = (pen == NONEPAT) ? myBgPixel : pixel;
   values.function = GXcopy;
   values.fill_style = FillOpaqueStippled;
   values.stipple = patPixmap[pen];
   values.line_width = ZOOMED_SIZE(width);
#ifdef NO_THIN_LINE
   if (values.line_width < 1) values.line_width = 1;
#else
#ifdef THIN_OVAL_AND_ARC
   if (values.line_width <= 1) values.line_width = 0;
#endif
#endif
   if (dash != 0)
   {
      XSetDashes (mainDisplay, drawGC, 0, dashList[dash],
            dashListLength[dash]);
      values.line_style = LineOnOffDash;
   }
   else
      values.line_style = LineSolid;
   XChangeGC (mainDisplay, drawGC,
         GCForeground | GCFunction | GCFillStyle | GCStipple | GCLineWidth |
         GCLineStyle, &values);

   aw = ZOOMED_SIZE(arc_ptr->aw); if (aw == 0) aw = 1;
   ah = ZOOMED_SIZE(arc_ptr->ah); if (ah == 0) ah = 1;

   ArcRealX2Y2 (arc_ptr, &real_x2, &real_y2);

   dir = arc_ptr->dir;

/* xc = ZOOMED_SIZE(arc_ptr->xc-real_x_off); */
/* yc = ZOOMED_SIZE(arc_ptr->yc-real_y_off); */

   x1 = ZOOMED_SIZE(arc_ptr->x1-real_x_off);
   y1 = ZOOMED_SIZE(arc_ptr->y1-real_y_off);

   if (arc_ptr->style & LS_LEFT)
   {  /* the arrow should appear at angle1 */
      int	theta = 0;

      switch (dir)
      {
         case ARC_CCW: theta = (int)(angle1/64)-90; break;
         case ARC_CW: theta = (int)(angle1/64)+90; break;
      }

      dx = -round(w*cos(theta*M_PI/180));
      dy = round(h*sin(theta*M_PI/180));
      if (dx==0 && dy==0)
      {
         sine = cosine = ((double)0.0);

         tmp_v[0].x = tmp_v[1].x = tmp_v[2].x = tmp_v[3].x = x1;
         tmp_v[0].y = tmp_v[1].y = tmp_v[2].y = tmp_v[3].y = y1;
      }
      else
      {
         double len = (double) sqrt ((double)(dx*dx+dy*dy));

         sine = dy/len;
         cosine = dx/len;

         tmp_v[0].x = tmp_v[3].x = x1;
         tmp_v[0].y = tmp_v[3].y = y1;
         tmp_v[1].x = round(x1 + aw*cosine - ah*sine);
         tmp_v[1].y = round(y1 + aw*sine + ah*cosine);
         tmp_v[2].x = round(x1 + aw*cosine + ah*sine);
         tmp_v[2].y = round(y1 + aw*sine - ah*cosine);
      }

      XFillPolygon (mainDisplay, window, drawGC, tmp_v, 4, Convex,
            CoordModeOrigin);
   }

   x2 = ZOOMED_SIZE(real_x2-real_x_off);
   y2 = ZOOMED_SIZE(real_y2-real_y_off);

   if (arc_ptr->style & LS_RIGHT)
   {
      int	theta = 0;

      switch (dir)
      {
         case ARC_CCW: theta = (int)((angle1+angle2)/64)-90; break;
         case ARC_CW: theta = (int)((angle1+angle2)/64)+90; break;
      }

      dx = -round(w*cos(theta*M_PI/180));
      dy = round(h*sin(theta*M_PI/180));
      if (dx==0 && dy==0)
      {
         sine = cosine = ((double)0.0);

         tmp_v[0].x = tmp_v[1].x = tmp_v[2].x = tmp_v[3].x = x2;
         tmp_v[0].y = tmp_v[1].y = tmp_v[2].y = tmp_v[3].y = y2;
      }
      else
      {
         double len = (double) sqrt ((double)(dx*dx+dy*dy));

         sine = dy/len;
         cosine = dx/len;

         tmp_v[0].x = tmp_v[3].x = x2;
         tmp_v[0].y = tmp_v[3].y = y2;
         tmp_v[1].x = round(x2 - aw*cosine + ah*sine);
         tmp_v[1].y = round(y2 - aw*sine - ah*cosine);
         tmp_v[2].x = round(x2 - aw*cosine - ah*sine);
         tmp_v[2].y = round(y2 - aw*sine + ah*cosine);
      }

      XFillPolygon (mainDisplay, window, drawGC, tmp_v, 4, Convex,
            CoordModeOrigin);
   }

   XDrawArc (mainDisplay, window, drawGC, ltx, lty, w, h, angle1, angle2);
}

void UpdArcBBox (ObjPtr)
   struct ObjRec	* ObjPtr;
{
   struct ArcRec	* arc_ptr = ObjPtr->detail.a;

   ObjPtr->x = arc_ptr->xc;
   ObjPtr->y = arc_ptr->yc;

   CalcArcOBBox (arc_ptr, &(ObjPtr->obbox));
   CalcArcBBox (arc_ptr, ObjPtr->obbox, &(ObjPtr->bbox));
}

static
struct ObjRec * CreateArcObj (xc,yc,x1,y1,x2,y2,dir,ltx,lty,w,h,angle1,angle2)
   int	xc, yc, x1, y1, x2, y2, dir, ltx, lty, w, h, angle1, angle2;
{
   struct ArcRec	* arc_ptr;
   struct ObjRec	* obj_ptr;

   arc_ptr = (struct ArcRec *) calloc (1, sizeof(struct ArcRec));
   arc_ptr->fill = objFill;
   arc_ptr->width = curWidthOfLine[lineWidth];
   arc_ptr->aw = curArrowHeadW[lineWidth];
   arc_ptr->ah = curArrowHeadH[lineWidth];
   arc_ptr->pen = penPat;
   arc_ptr->dash = curDash;
   arc_ptr->style = lineStyle;

   arc_ptr->xc = ABS_X(xc);
   arc_ptr->yc = ABS_Y(yc);
   arc_ptr->x1 = ABS_X(x1);
   arc_ptr->y1 = ABS_Y(y1);
   arc_ptr->x2 = ABS_X(x2);
   arc_ptr->y2 = ABS_Y(y2);
   arc_ptr->dir = dir;
   arc_ptr->ltx = ABS_X(xc-w/2);
   arc_ptr->lty = ABS_Y(yc-h/2);
   arc_ptr->w = (arc_ptr->xc-arc_ptr->ltx)<<1;
   arc_ptr->h = (arc_ptr->yc-arc_ptr->lty)<<1;
   arc_ptr->angle1 = angle1; arc_ptr->angle2 = angle2;

   obj_ptr = (struct ObjRec *) calloc (1, sizeof(struct ObjRec));
   obj_ptr->detail.a = arc_ptr;

   UpdArcBBox (obj_ptr);

   obj_ptr->type = OBJ_ARC;
   obj_ptr->color = colorIndex;
   obj_ptr->id = objId++;
   obj_ptr->dirty = FALSE;
   obj_ptr->rotation = 0;
   obj_ptr->locked = FALSE;
   obj_ptr->fattr = obj_ptr->lattr = NULL;
   return (obj_ptr);
}
 
static
void ContinueArc (OrigX, OrigY)
   int	OrigX, OrigY;
{
   int 			grid_x, grid_y, first_x = 0, first_y = 0;
   int 			end_x, end_y, saved_x, saved_y;
   int 			done = FALSE, drawing_arc = FALSE;
   int			dir = INVALID, ltx, lty, w, h, angle1, angle2=0;
   char			msg[80];
   struct ObjRec	* obj_ptr;
   XGCValues		values;
   XEvent		input, ev;
   XMotionEvent		* motion_ev;

   values.foreground = xorColorPixels[colorIndex];
   values.function = GXxor;
   values.fill_style = FillSolid;
#ifdef NO_THIN_LINE
   values.line_width = 1;
#else
   values.line_width = 0;
#endif
   values.line_style = LineSolid;
   XChangeGC (mainDisplay, drawGC,
         GCForeground | GCFunction | GCFillStyle | GCLineWidth | GCLineStyle,
         &values);

   grid_x = saved_x = OrigX;
   grid_y = saved_y = OrigY; 
   XDrawLine (mainDisplay, drawWindow, drawGC, OrigX, OrigY, saved_x, saved_y);

   XGrabPointer (mainDisplay, drawWindow, FALSE,
         PointerMotionMask | ButtonPressMask | ButtonReleaseMask,
         GrabModeAsync, GrabModeAsync, None, handCursor, CurrentTime);
   
   Msg ("Please specify the start of an arc.");
   while (!done)
   {
      XNextEvent (mainDisplay, &input);

      if (input.type == Expose || input.type == VisibilityNotify)
         ExposeEventHandler (&input, TRUE);
      else if (input.type == ButtonPress)
      {
         if (drawing_arc)
         {
            XUngrabPointer (mainDisplay, CurrentTime);
            XDrawArc (mainDisplay, drawWindow, drawGC, ltx, lty, w, h,
                  angle1, angle2);
            done = TRUE;
            Msg ("");
         }
         else
         {
            XDrawLine (mainDisplay, drawWindow, drawGC, OrigX, OrigY,
                  saved_x, saved_y);
            first_x = saved_x;
            first_y = saved_y;
            drawing_arc = TRUE;
            if (OrigX == grid_x && OrigY == grid_y)
            {  /* fake it as if the 1st point is ok but the 2nd point is bad */
               XUngrabPointer (mainDisplay, CurrentTime);
               grid_x = first_x;
               grid_y = first_y;
               done = TRUE;
            }
            Msg ("Please specify the end of the arc.");
         }
      }
      else if (input.type == MotionNotify)
      {
         motion_ev = &(input.xmotion);
         end_x = motion_ev->x;
         end_y = motion_ev->y;
         GridXY (end_x, end_y, &grid_x, &grid_y);
         if (grid_x != saved_x || grid_y != saved_y)
         {
            if (drawing_arc)
            {  /* finished with the center and the first point on the arc */
               if (dir == INVALID)
               {
                  dir = ArcDirection (OrigX, OrigY, first_x, first_y,
                        grid_x, grid_y);
                  ltx = OrigX; lty = OrigY; w = 0; h = 0; angle1 = angle2 = 0;
                  if (dir == ARC_CW)
                     sprintf (msg, "Please specify the end of the arc.  (%s).",
                           "clockwise");
                  else
                     sprintf (msg, "Please specify the end of the arc.  (%s).",
                           "counter-clockwise");
                  Msg (msg);
               }
               XDrawArc (mainDisplay, drawWindow, drawGC, ltx, lty, w, h,
                     angle1, angle2);
               saved_x = grid_x;
               saved_y = grid_y;
               PointsToArc (OrigX, OrigY, first_x, first_y, saved_x, saved_y,
                     dir, &ltx, &lty, &w, &h, &angle1, &angle2);
               XDrawArc (mainDisplay, drawWindow, drawGC, ltx, lty, w, h,
                     angle1, angle2);
            }
            else
            {  /* looking for the first point on the arc */
               XDrawLine (mainDisplay, drawWindow, drawGC, OrigX, OrigY,
                     saved_x, saved_y);
               saved_x = grid_x;
               saved_y = grid_y;
               XDrawLine (mainDisplay, drawWindow, drawGC, OrigX, OrigY,
                     saved_x, saved_y);
            }
         }
         MarkRulers (grid_x, grid_y);
         while (XCheckMaskEvent (mainDisplay, PointerMotionMask, &ev)) ;
      }
   }
   if (angle2 == 0)
      Msg ("No arc created.");
   else
   {
      obj_ptr = CreateArcObj (OrigX, OrigY, first_x, first_y, saved_x,
            saved_y, dir, ltx, lty, w, h, angle1, angle2);
      AddObj (NULL, topObj, obj_ptr);
      RecordNewObjCmd ();
      DrawArcObj (drawWindow, drawOrigX, drawOrigY, topObj);
      arcDrawn = TRUE;
      SetFileModified (TRUE);
   }
}

void DrawArc (input)
   XEvent	* input;
{
   XButtonEvent	* button_ev;
   int		mouse_x, mouse_y, grid_x, grid_y;

   if (input->type != ButtonPress) return;

   button_ev = &(input->xbutton);
   if (button_ev->button == Button1)
   {
      mouse_x = button_ev->x;
      mouse_y = button_ev->y;
      GridXY (mouse_x, mouse_y, &grid_x, &grid_y);
      ContinueArc (grid_x, grid_y);
   }
}

void SaveArcObj (FP, ObjPtr)
   FILE				* FP;
   register struct ObjRec	* ObjPtr;
{
   register struct ArcRec	* arc_ptr = ObjPtr->detail.a;

   fprintf (FP, "arc('%s',", colorMenuItems[ObjPtr->color]);
   fprintf (FP, "%1d,%1d,%1d,%1d,%1d,%1d,%1d,%1d,%1d,%1d,\
%1d,%1d,%1d,%1d,%1d,%1d,%1d,%1d,%1d,%1d,%1d,%1d,%1d,",
         arc_ptr->fill, arc_ptr->width, arc_ptr->pen, arc_ptr->dash,
         arc_ptr->ltx, arc_ptr->lty, arc_ptr->xc, arc_ptr->yc,
         arc_ptr->x1, arc_ptr->y1, arc_ptr->x2, arc_ptr->y2,
         arc_ptr->dir, arc_ptr->w, arc_ptr->h, arc_ptr->angle1, arc_ptr->angle2,
         ObjPtr->id, ObjPtr->rotation, arc_ptr->style, arc_ptr->aw,
         arc_ptr->ah, ObjPtr->locked);
   SaveAttrs (FP, ObjPtr->lattr);
   fprintf (FP, ")");
}

#define GETVALUE(val,name) ScanValue("%d", (char *) &(val), name, "arc")

void ReadArcObj (Inbuf, ObjPtr)
   char			* Inbuf;
   struct ObjRec	* * ObjPtr;
{
   register struct ArcRec	* arc_ptr;
   char				color_str[20], * s, msg[MAXSTRING];
   int				fill, width=0, pen, dash, ltx, lty, w, h, id=0;
   int				rotation, new_alloc, style, locked=FALSE;
   int				aw=origArrowHeadW[0], ah=origArrowHeadH[0];
   int				xc, yc, x1, y1, x2, y2, dir, angle1, angle2;

   *ObjPtr = NULL;

   s = FindChar ('(', Inbuf);
   s = ParseStr (s, ',', color_str);

   InitScan (s, ", \t\n");

   style = LS_PLAIN;
   rotation = 0;
   if (fileVersion <= 8)
   {
      *ObjPtr = NULL;
      sprintf (msg, "Invalid arc version (%1d).", fileVersion);
      if (PRTGIF)
         fprintf (stderr, "%s\n", msg);
      else
         Msg (msg);
      return;
   }
   else if (fileVersion <= 13)
   {
      if (GETVALUE(fill,   "fill") == INVALID ||
          GETVALUE(width,  "width") == INVALID ||
          GETVALUE(pen,    "pen") == INVALID ||
          GETVALUE(dash,   "dash") == INVALID ||
          GETVALUE(ltx,    "ltx") == INVALID ||
          GETVALUE(lty,    "lty") == INVALID ||
          GETVALUE(xc,     "xc") == INVALID ||
          GETVALUE(yc,     "yc") == INVALID ||
          GETVALUE(x1,     "x1") == INVALID ||
          GETVALUE(y1,     "y1") == INVALID ||
          GETVALUE(x2,     "x2") == INVALID ||
          GETVALUE(y2,     "y2") == INVALID ||
          GETVALUE(dir,    "direction") == INVALID ||
          GETVALUE(w,      "width") == INVALID ||
          GETVALUE(h,      "height") == INVALID ||
          GETVALUE(angle1, "angle1") == INVALID ||
          GETVALUE(angle2, "angle2") == INVALID ||
          GETVALUE(id,     "id") == INVALID)
      {
         return;
      }
      if (id >= objId) objId = id+1;
   }
   else if (fileVersion <= 15)
   {
      if (GETVALUE(fill,     "fill") == INVALID ||
          GETVALUE(width,    "width") == INVALID ||
          GETVALUE(pen,      "pen") == INVALID ||
          GETVALUE(dash,     "dash") == INVALID ||
          GETVALUE(ltx,      "ltx") == INVALID ||
          GETVALUE(lty,      "lty") == INVALID ||
          GETVALUE(xc,       "xc") == INVALID ||
          GETVALUE(yc,       "yc") == INVALID ||
          GETVALUE(x1,       "x1") == INVALID ||
          GETVALUE(y1,       "y1") == INVALID ||
          GETVALUE(x2,       "x2") == INVALID ||
          GETVALUE(y2,       "y2") == INVALID ||
          GETVALUE(dir,      "direction") == INVALID ||
          GETVALUE(w,        "width") == INVALID ||
          GETVALUE(h,        "height") == INVALID ||
          GETVALUE(angle1,   "angle1") == INVALID ||
          GETVALUE(angle2,   "angle2") == INVALID ||
          GETVALUE(id,       "id") == INVALID ||
          GETVALUE(rotation, "rotation") == INVALID)
      {
         return;
      }
      if (id >= objId) objId = id+1;
   }
   else if (fileVersion <= 16)
   {
      if (GETVALUE(fill,     "fill") == INVALID ||
          GETVALUE(width,    "width") == INVALID ||
          GETVALUE(pen,      "pen") == INVALID ||
          GETVALUE(dash,     "dash") == INVALID ||
          GETVALUE(ltx,      "ltx") == INVALID ||
          GETVALUE(lty,      "lty") == INVALID ||
          GETVALUE(xc,       "xc") == INVALID ||
          GETVALUE(yc,       "yc") == INVALID ||
          GETVALUE(x1,       "x1") == INVALID ||
          GETVALUE(y1,       "y1") == INVALID ||
          GETVALUE(x2,       "x2") == INVALID ||
          GETVALUE(y2,       "y2") == INVALID ||
          GETVALUE(dir,      "direction") == INVALID ||
          GETVALUE(w,        "width") == INVALID ||
          GETVALUE(h,        "height") == INVALID ||
          GETVALUE(angle1,   "angle1") == INVALID ||
          GETVALUE(angle2,   "angle2") == INVALID ||
          GETVALUE(id,       "id") == INVALID ||
          GETVALUE(rotation, "rotation") == INVALID ||
          GETVALUE(style,    "style") == INVALID)
      {
         return;
      }
      if (id >= objId) objId = id+1;
   }
   else if (fileVersion <= 25)
   {
      if (GETVALUE(fill,     "fill") == INVALID ||
          GETVALUE(width,    "width") == INVALID ||
          GETVALUE(pen,      "pen") == INVALID ||
          GETVALUE(dash,     "dash") == INVALID ||
          GETVALUE(ltx,      "ltx") == INVALID ||
          GETVALUE(lty,      "lty") == INVALID ||
          GETVALUE(xc,       "xc") == INVALID ||
          GETVALUE(yc,       "yc") == INVALID ||
          GETVALUE(x1,       "x1") == INVALID ||
          GETVALUE(y1,       "y1") == INVALID ||
          GETVALUE(x2,       "x2") == INVALID ||
          GETVALUE(y2,       "y2") == INVALID ||
          GETVALUE(dir,      "direction") == INVALID ||
          GETVALUE(w,        "width") == INVALID ||
          GETVALUE(h,        "height") == INVALID ||
          GETVALUE(angle1,   "angle1") == INVALID ||
          GETVALUE(angle2,   "angle2") == INVALID ||
          GETVALUE(id,       "id") == INVALID ||
          GETVALUE(rotation, "rotation") == INVALID ||
          GETVALUE(style,    "style") == INVALID ||
          GETVALUE(aw,       "arrow head w") == INVALID ||
          GETVALUE(ah,       "arrow head h") == INVALID)
      {
         return;
      }
      if (id >= objId) objId = id+1;
   }
   else
   {
      if (GETVALUE(fill,     "fill") == INVALID ||
          GETVALUE(width,    "width") == INVALID ||
          GETVALUE(pen,      "pen") == INVALID ||
          GETVALUE(dash,     "dash") == INVALID ||
          GETVALUE(ltx,      "ltx") == INVALID ||
          GETVALUE(lty,      "lty") == INVALID ||
          GETVALUE(xc,       "xc") == INVALID ||
          GETVALUE(yc,       "yc") == INVALID ||
          GETVALUE(x1,       "x1") == INVALID ||
          GETVALUE(y1,       "y1") == INVALID ||
          GETVALUE(x2,       "x2") == INVALID ||
          GETVALUE(y2,       "y2") == INVALID ||
          GETVALUE(dir,      "direction") == INVALID ||
          GETVALUE(w,        "width") == INVALID ||
          GETVALUE(h,        "height") == INVALID ||
          GETVALUE(angle1,   "angle1") == INVALID ||
          GETVALUE(angle2,   "angle2") == INVALID ||
          GETVALUE(id,       "id") == INVALID ||
          GETVALUE(rotation, "rotation") == INVALID ||
          GETVALUE(style,    "style") == INVALID ||
          GETVALUE(aw,       "arrow head w") == INVALID ||
          GETVALUE(ah,       "arrow head h") == INVALID ||
          GETVALUE(locked,   "locked") == INVALID)
      {
         return;
      }
      if (id >= objId) objId = id+1;
   }

   if (dir == ARC_CCW && angle2 < 0)
   {
      sprintf (msg, "Warning:  Inconsistent arc direction.  Corrected.");
      if (PRTGIF)
         fprintf (stderr, "%s\n", msg);
      else
         Msg (msg);
      SetFileModified (TRUE);
      dir = ARC_CW;
   }
   else if (dir == ARC_CW && angle2 > 0)
   {
      sprintf (msg, "Warning:  Inconsistent arc direction.  Corrected.");
      if (PRTGIF)
         fprintf (stderr, "%s\n", msg);
      else
         Msg (msg);
      SetFileModified (TRUE);
      dir = ARC_CCW;
   }

   if (fileVersion <= 16 && width <= 6)
   {
      aw = origArrowHeadW[width];
      ah = origArrowHeadH[width];
      width = origWidthOfLine[width];
   }
   fill = UpgradePenFill (fill);
   pen = UpgradePenFill (pen);

   * ObjPtr = (struct ObjRec *) calloc (1, sizeof(struct ObjRec));
   arc_ptr = (struct ArcRec *) calloc (1, sizeof(struct ArcRec));

   arc_ptr->fill = fill;
   arc_ptr->width = width;
   arc_ptr->aw = aw;
   arc_ptr->ah = ah;
   arc_ptr->pen = pen;
   arc_ptr->dash = dash;
   arc_ptr->style = style;

   arc_ptr->xc = xc;         arc_ptr->yc = yc;
   arc_ptr->x1 = x1;         arc_ptr->y1 = y1;
   arc_ptr->x2 = x2;         arc_ptr->y2 = y2;
   arc_ptr->dir = dir;
   arc_ptr->ltx = ltx;       arc_ptr->lty = lty;
   arc_ptr->w = w;           arc_ptr->h = h;
   arc_ptr->angle1 = angle1; arc_ptr->angle2 = angle2;

   (*ObjPtr)->detail.a = arc_ptr;
   UpdArcBBox (*ObjPtr);

   (*ObjPtr)->type = OBJ_ARC;
   (*ObjPtr)->color = QuickFindColorIndex (color_str, &new_alloc);
   (*ObjPtr)->dirty = FALSE;
   (*ObjPtr)->id = id;
   (*ObjPtr)->rotation = rotation;
   (*ObjPtr)->locked = locked;
}

void FreeArcObj (ObjPtr)
   struct ObjRec	* ObjPtr;
{
   cfree (ObjPtr->detail.a);
   cfree (ObjPtr);
}

static
int ParseArcSpec (spec, seperator, radius, dir, theta1, theta2, error_str)
   char	* spec, seperator, * error_str;
   int	* radius, * dir, * theta1, * theta2;
{
   char	* s, buf[MAXSTRING], tmp_buf[MAXSTRING];

   strcpy (tmp_buf, spec);

   s = ParseStr (tmp_buf, seperator, buf);
   if (*s != '\0' && (*radius = atoi (buf)) <= 0) *s = '\0';
   if (*s == '\0') { strcpy (error_str, "radius"); return (FALSE); }

   s = ParseStr (s, seperator, buf);
   switch (*buf)
   {
      case '+': *dir = ARC_CW; break;
      case '-': *dir = ARC_CCW; break;
      default: *s = '\0'; break;
   }
   if (*s == '\0') { strcpy (error_str, "dir"); return (FALSE); }

   s = ParseStr (s, seperator, buf);
   if (*s == '\0') { strcpy (error_str, "theta1"); return (FALSE); }
   *theta1 = atoi (buf);
   *theta2 = atoi (s);

   return (TRUE);
}

void MakePreciseArc ()
{
   int			r = 0, dir = 0, x1, y1, x2, y2;
   int			theta1, theta2, angle2=0;
   char			spec[MAXSTRING], msg[MAXSTRING], error_str[MAXSTRING];
   double		angle_in_radian;
   struct ObjRec	* obj_ptr;

   Dialog ("Arc specification: [radius,dir(+/-),theta1,theta2]",
         "( <CR>: accept, <ESC>: cancel )", spec);
   if (*spec == '\0') return;

   TieLooseEnds ();
   SetCurChoice (NOTHING);
   if (topSel!=NULL) { HighLightReverse (); RemoveAllSel (); }

   if (!ParseArcSpec (spec, ',', &r, &dir, &theta1, &theta2, error_str) &&
         !ParseArcSpec (spec, ' ', &r, &dir, &theta1, &theta2, error_str))
   {
      sprintf (msg, "Invalid %s in arc specification '%s'.", error_str, spec);
      Msg (msg);
      return;
   }

   switch (dir)
   {
      case ARC_CCW: theta1 += 90;        theta2 += 90;        break;
      case ARC_CW:  theta1 = -theta1+90; theta2 = -theta2+90; break;
   }

   angle_in_radian = theta1 * M_PI / 180;
   x1 = round(r*cos(angle_in_radian));
   y1 = -round(r*sin(angle_in_radian));
   angle_in_radian = theta2 * M_PI / 180;
   x2 = round(r*cos(angle_in_radian));
   y2 = -round(r*sin(angle_in_radian));

   while (theta1 < 0) theta1 += 360;
   while (theta2 > theta1) theta2 -= 360;
   while (theta2 < theta1) theta2 += 360;

   switch (dir)
   {
      case ARC_CCW:
         angle2 = theta2-theta1;
         if (angle2 == 0) angle2 = 360;
         break;
      case ARC_CW:
         angle2 = theta2-theta1-360;
         break;
   }
   obj_ptr = CreateArcObj (0, 0, x1, y1, x2, y2, dir, -r, -r, r*2, r*2,
         theta1*64, angle2*64);

   PlaceTopObj (obj_ptr);
   AddObj (NULL, topObj, obj_ptr);
   SelectTopObj ();
   RecordNewObjCmd ();
   SetFileModified (TRUE);
   justDupped = FALSE;
}
