/*
 * Author:	William Chia-Wei Cheng (william@cs.ucla.edu)
 *
 * Copyright (C) 1990, 1991, 1992, William Cheng.
 * 
 * Permission limited to the use, copy, modify, and distribute this software
 * and its documentation for any purpose is hereby granted by the Author without
 * fee, provided that the above copyright notice appear in all copies and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation, and that the name of the Author not be used
 * in advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.  The Author makes no
 * representations about the suitability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.  All other
 * rights are reserved by the Author.
 *
 * THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF
 * USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
 * OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */
#ifndef lint
static char RCSid[] =
      "@(#)$Header: /amnt/kona/tangram/u/william/X11/TGIF2/RCS/exec.c,v 2.0 1993/06/07 03:07:09 william Exp $";
#endif

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/time.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include "const.h"
#include "types.h"

#include "attr.e"
#include "dialog.e"
#include "drawing.e"
#include "dup.e"
#include "eps.e"
#include "file.e"
#include "menu.e"
#include "obj.e"
#include "select.e"
#include "setup.e"
#include "stretch.e"
#include "xbitmap.e"

extern char *mktemp();

#define TOK_INVALID	(INVALID)
#define TOK_STR		0
#define TOK_LEFT_P	1
#define TOK_RIGHT_P	2
#define TOK_LEFT_B	3
#define TOK_RIGHT_B	4
#define TOK_LEFT_CB	5
#define TOK_RIGHT_CB	6
#define TOK_COMMA	7
#define TOK_SEMI	8

static char execDummyStr[MAXSTRING+1];

struct ObjRec	* FindObjWithName (BotObj, ObjName)
   struct ObjRec	* BotObj;
   char			* ObjName;
{
   register struct AttrRec	* attr_ptr;
   register struct ObjRec	* obj_ptr;

   for (obj_ptr=BotObj; obj_ptr!=NULL; obj_ptr=obj_ptr->prev)
      if (obj_ptr->fattr != NULL &&
            (attr_ptr=FindAttrWithName(obj_ptr,"name=")) != NULL &&
            strcmp (attr_ptr->s, ObjName) == 0)
         return (obj_ptr);
   return NULL;
}

static
char * GetToken (inbuf, outbuf, tok_type)
   char	* inbuf, * outbuf;
   int	* tok_type;
   /* returns NULL of the input string is invalid */
   /* otherwise, the return value points to the character */
   /*		immediately following the end of the token */
   /* *tok_type contains the token type */
{
   register char	* c_ptr, * obuf_ptr=outbuf;

   *tok_type = TOK_INVALID;
   while (*inbuf == ' ' || *inbuf == '\t') inbuf++;
   switch (*inbuf)
   {
      case '\0': return (NULL);

      case '\'':
      case '"':
         *tok_type = TOK_STR;
         for (c_ptr=(&inbuf[1]); *c_ptr!=(*inbuf) && *c_ptr!='\0';
               c_ptr++, obuf_ptr++)
         {
            switch (*c_ptr)
            {
               case '\\':
                  switch (*(++c_ptr))
                  {
                     case 'n': *obuf_ptr = '\n'; break;
                     case 'r': *obuf_ptr = '\r'; break;
                     case 't': *obuf_ptr = '\t'; break;
                     default: *obuf_ptr = *c_ptr; break;
                  }
                  break;
               default: *obuf_ptr = *c_ptr; break;
            }
         }
         if (*c_ptr == '\0') return (NULL);
         *obuf_ptr++ = '\0';
         return (++c_ptr);
      case '(': strcpy(obuf_ptr,"("); *tok_type=TOK_LEFT_P; return(&inbuf[1]);
      case ')': strcpy(obuf_ptr,")"); *tok_type=TOK_RIGHT_P; return(&inbuf[1]);
      case '[': strcpy(obuf_ptr,"["); *tok_type=TOK_LEFT_B; return(&inbuf[1]);
      case ']': strcpy(obuf_ptr,"]"); *tok_type=TOK_RIGHT_B; return(&inbuf[1]);
      case '{': strcpy(obuf_ptr,"{"); *tok_type=TOK_LEFT_CB; return(&inbuf[1]);
      case '}': strcpy(obuf_ptr,"}"); *tok_type=TOK_RIGHT_CB; return(&inbuf[1]);
      case ',': strcpy(obuf_ptr,","); *tok_type=TOK_COMMA; return(&inbuf[1]);
      case ';': strcpy(obuf_ptr,";"); *tok_type=TOK_SEMI; return(&inbuf[1]);
   }
   *tok_type = TOK_STR;
   c_ptr = inbuf;
   while (*c_ptr != '\0' && strchr (" \t()[]{},;", *c_ptr) == NULL)
   {
      if (*c_ptr == '$' && c_ptr[1] == '(')
      {
         *obuf_ptr++ = *c_ptr++;
         *obuf_ptr++ = *c_ptr++;
         while (*c_ptr != '\0' && *c_ptr != ')') *obuf_ptr++ = *c_ptr++;
         if (*c_ptr == '\0') return NULL;
      }
      else
         *obuf_ptr++ = *c_ptr++;
   }
   *obuf_ptr = '\0';
   return (c_ptr);
}

static
char * convert_str (inbuf, obj_ptr)
   char			* inbuf;
   struct ObjRec	* obj_ptr;
{
   register char	* buf_ptr;
   char			* new_c_ptr, * cp, * cp1;
   char			* return_str, * return_ptr;
   int			cur_size=(MAXSTRING<<1), count=0, return_len=0, n;
   struct AttrRec	* attr_ptr;

   return_ptr = return_str = (char *) calloc (cur_size+2, sizeof(char));
   buf_ptr = inbuf;
   while (*inbuf != '\0')
   {
      if ((new_c_ptr = strstr (buf_ptr, "$(")) == NULL)
      {
         count = strlen (inbuf);
         if (count != 0)
         {
            if (count+return_len >= cur_size)
            {
               n = return_ptr-return_str;
               cur_size += count+MAXSTRING;
               return_str = (char *) realloc (return_str,
                     (cur_size+2)*sizeof(char));
               return_ptr = &return_str[n];
            }
            strncpy (return_ptr, inbuf, count);
            return_ptr += count;
            *return_ptr = '\0';
         }
         return (return_str);
      }
      buf_ptr = new_c_ptr;
      count = buf_ptr-inbuf;
      return_len += count;
      if (count != 0)
      {
         if (count+return_len >= cur_size)
         {
            n = return_ptr-return_str;
            cur_size += count+MAXSTRING;
            return_str = (char *)realloc(return_str,(cur_size+2)*sizeof(char));
            return_ptr = &return_str[n];
         }
         strncpy (return_ptr, inbuf, count);
         return_ptr += count;
         *return_ptr = '\0';
      }
      if ((attr_ptr = ValidAttrArg (buf_ptr, obj_ptr, &new_c_ptr)) == NULL)
      {
         cfree (return_str);
         return NULL;
      }
      count = strlen (attr_ptr->s);
      if (count+return_len >= cur_size)
      {
         n = return_ptr-return_str;
         cur_size += count+MAXSTRING;
         return_str = (char *) realloc (return_str, (cur_size+2)*sizeof(char));
         return_ptr = &return_str[n];
      }
      strcpy (return_ptr, attr_ptr->s);
      if ((cp = strstr (return_ptr, "//")) != NULL)
      {
         *cp = '\0';
         count = cp-return_ptr;
      }
      if (*return_ptr == '"' && return_ptr[count-1] == '"')
      {
         for (n=1, cp1=return_ptr, cp=(&return_ptr[1]); n < count-1; n++)
            *cp1++ = *cp++;
         *cp1 = '\0';
         count -= 2;
      }
      return_ptr += count;
      inbuf = buf_ptr = ++new_c_ptr;
   }
   return (return_str);
}

static
void LaunchIt (cmd)
   char	* cmd;
{
   int	len=strlen(cmd);

   while (len > 0 && (cmd[len-1] == ' ')) cmd[--len] = '\0';
   if (cmd[0] != '\0')
   {
      SetWatchCursor (drawWindow);
      SetWatchCursor (mainWindow);
      if (cmd[len-1] == '&')
      {
         cmd[--len] = '\0';
         while (len > 0 && (cmd[len-1] == ' ')) cmd[--len] = '\0';
         if (cmd[0] != '\0')
         {
            int		pid;

            fprintf(stderr, "Backgrounding:  '%s'.\n", cmd);
            pid = fork();
            if (pid == 0) { system(cmd); exit(0); }
         }
      }
      else
      {
         FILE	*fp;

         if ((fp = popen (cmd, "r")) == NULL)
            fprintf(stderr, "Fail:  popen(%s).\n", cmd);
         else
         {
            char	tmp_str[MAXSTRING+1];
            XEvent	ev;

            while (fgets (tmp_str, MAXSTRING, fp) != NULL)
            {
               fprintf (stderr, "%s", tmp_str);
               if (XCheckMaskEvent (mainDisplay,
                     ExposureMask | VisibilityChangeMask, &ev))
               {
                  ExposeEventHandler (&ev, TRUE);
                  while (XCheckMaskEvent (mainDisplay,
                        ExposureMask | VisibilityChangeMask, &ev)) ;
               }
            }
            pclose (fp);
         }
      }
      SetDefaultCursor (mainWindow);
      SetDefaultCursor (drawWindow);
   }
}

int DoLaunch (launch_attr, obj_ptr)
   struct AttrRec	* launch_attr;
   struct ObjRec	* obj_ptr;
{
   register char	* c_ptr, * buf_ptr;
   char			* new_c_ptr;
   char			buf[MAXSTRING+1], * cmd, * cmd_ptr;
   char			msg[MAXSTRING+1];
   int			cur_size=2*MAXSTRING, count=0, cmd_len=0, n;
   int			first_time=TRUE;
   struct AttrRec	* attr_ptr;
   struct StrRec	* str_ptr;

   cmd = (char *) calloc (cur_size+1, sizeof(char));
   cmd_ptr = cmd;
   buf_ptr = buf;
   for (str_ptr = launch_attr->obj->detail.t->first; str_ptr != NULL;
         str_ptr = str_ptr->next)
   {
      if (first_time)
      {
         first_time = FALSE;
         c_ptr = launch_attr->s;
      }
      else
         c_ptr = str_ptr->s;

      for ( ; *c_ptr != '\0'; c_ptr++, count++)
      {
         switch (*c_ptr)
         {
            case '\\': c_ptr++; *buf_ptr++ = *c_ptr; break;
            case '$':
               if (count != 0)
               {
                  *buf_ptr = '\0';
                  if (count+cmd_len >= cur_size)
                  {
                     n = cmd_ptr-cmd;
                     cur_size += MAXSTRING;
                     cmd = (char *) realloc (cmd, (cur_size+2)*sizeof(char));
                     cmd_ptr = cmd+n;
                  }
                  strcpy (cmd_ptr, buf);
                  cmd_ptr += count;
                  count = 0;
                  buf_ptr = buf;
               }
               if ((attr_ptr = ValidAttrArg(c_ptr,obj_ptr,&new_c_ptr)) == NULL)
               {
                  cfree (cmd);
                  sprintf (msg, "Invalid attribute specification: '%s'.",
                        c_ptr);
                  Dialog (msg, "( <CR> or <ESC> to abort execution )",
                        execDummyStr );
                  return (FALSE);
               }
               count = strlen (attr_ptr->s);
               if (count+cmd_len >= cur_size)
               {
                  n = cmd_ptr-cmd;
                  cur_size += MAXSTRING;
                  cmd = (char *) realloc (cmd, (cur_size+2)*sizeof(char));
                  cmd_ptr = cmd+n;
               }
               strcpy (cmd_ptr, attr_ptr->s);
               cmd_ptr += count;
               count = -1;
               c_ptr = new_c_ptr;
               break;
            default: *buf_ptr++ = *c_ptr; break;
         }
      }
      if (count != 0)
      {
         *buf_ptr = '\0';
         if (count+cmd_len >= cur_size)
         {
            n = cmd_ptr-cmd;
            cur_size += MAXSTRING;
            cmd = (char *) realloc (cmd, (cur_size+2)*sizeof(char));
            cmd_ptr = cmd+n;
         }
         strcpy (cmd_ptr, buf);
         cmd_ptr += count;
         count = 0;
         buf_ptr = buf;
      }
      if (str_ptr->next != NULL) { *cmd_ptr++ = ' '; *cmd_ptr = '\0'; }
   }
   LaunchIt (cmd);
   cfree (cmd);
   return (TRUE);
}

static
int BadCmd (cmd_name)
   char	* cmd_name;
{
   char	msg[MAXSTRING+1];

   sprintf (msg, "Malformed '%s' command.  Command execution aborted.",
         cmd_name);
   Dialog (msg, "( <CR> or <ESC> to continue )", execDummyStr );
   return (FALSE);
}

static
int BadAttr (attr_name, cmd_name)
   char	* attr_name, * cmd_name;
{
   char	msg[MAXSTRING+1];

   sprintf (msg, "Can not find the '%s' %s '%s' command.",
         attr_name, "attribute while executing the", cmd_name);
   Dialog (msg, "( <CR> or <ESC> to abort execution )", execDummyStr );
   return (FALSE);
}

static
int ExecLaunch (buf, obj_ptr)
   char			* * buf;
   struct ObjRec	* obj_ptr;
   /* launch(attribute_to_launch); */
{
   char			* c_ptr=(*buf);
   char			attr_name[MAXSTRING+1];
   int			len, tok_type;
   struct AttrRec	* attr_ptr;

   if (!((c_ptr = GetToken (c_ptr, execDummyStr, &tok_type)) != NULL &&
         tok_type == TOK_LEFT_P &&
         (c_ptr = GetToken (c_ptr, attr_name, &tok_type)) != NULL &&
         tok_type == TOK_STR &&
         (c_ptr = GetToken (c_ptr, execDummyStr, &tok_type)) != NULL &&
         tok_type == TOK_RIGHT_P))
      return BadCmd ("launch");

   len = strlen (attr_name);
   attr_name[len++] = '=';
   attr_name[len] = '\0';
   attr_ptr = FindAttrWithName (obj_ptr, attr_name);
   if (attr_ptr == NULL) return (BadAttr (attr_name, "launch"));

   *buf = c_ptr;
   return (DoLaunch (attr_ptr, obj_ptr));
}

static
int ExecExec (buf, obj_ptr)
   char			* * buf;
   struct ObjRec	* obj_ptr;
   /* exec(attribute_to_exec); */
{
   char			* c_ptr=(*buf);
   char			attr_name[MAXSTRING+1];
   int			len, tok_type;
   struct AttrRec	* attr_ptr;

   if (!((c_ptr = GetToken (c_ptr, execDummyStr, &tok_type)) != NULL &&
         tok_type == TOK_LEFT_P &&
         (c_ptr = GetToken (c_ptr, attr_name, &tok_type)) != NULL &&
         tok_type == TOK_STR &&
         (c_ptr = GetToken (c_ptr, execDummyStr, &tok_type)) != NULL &&
         tok_type == TOK_RIGHT_P))
      return BadCmd ("exec");

   len = strlen (attr_name);
   attr_name[len++] = '=';
   attr_name[len] = '\0';
   attr_ptr = FindAttrWithName (obj_ptr, attr_name);
   if (attr_ptr == NULL) return (BadAttr (attr_name, "exec"));

   *buf = c_ptr;
   return (DoExec (attr_ptr, obj_ptr));
}

static
int ExecMktemp (buf, obj_ptr)
   char			* * buf;
   struct ObjRec	* obj_ptr;
   /* mktemp(tmp_file_template_string,result_attribute); */
{
   char			* c_ptr=(*buf);
   char			attr_name[MAXSTRING+1], file_name[MAXPATHLENGTH+1];
   int			len, tok_type;
   struct AttrRec	* attr_ptr;

   if (!((c_ptr = GetToken (c_ptr, execDummyStr, &tok_type)) != NULL &&
         tok_type == TOK_LEFT_P &&
         (c_ptr = GetToken (c_ptr, file_name, &tok_type)) != NULL &&
         tok_type == TOK_STR &&
         (c_ptr = GetToken (c_ptr, execDummyStr, &tok_type)) != NULL &&
         tok_type == TOK_COMMA &&
         (c_ptr = GetToken (c_ptr, attr_name, &tok_type)) != NULL &&
         tok_type == TOK_STR &&
         (c_ptr = GetToken (c_ptr, execDummyStr, &tok_type)) != NULL &&
         tok_type == TOK_RIGHT_P))
      return BadCmd ("mktemp");

   len = strlen (attr_name);
   attr_name[len++] = '=';
   attr_name[len] = '\0';
   attr_ptr = FindAttrWithName (obj_ptr, attr_name);
   if (attr_ptr == NULL) return (BadAttr (attr_name, "mktemp"));

   if (mktemp (file_name) == NULL)
   {
      Dialog ("Fail to mktemp().  Command execution aborted.",
            "( <CR> or <ESC> to abort execution )", attr_name );
      return (FALSE);
   }
   unlink (file_name);
   if (strcmp (attr_ptr->s, file_name) != 0)
   {
      int	ltx, lty, rbx, rby;

      ltx = obj_ptr->bbox.ltx; lty = obj_ptr->bbox.lty;
      rbx = obj_ptr->bbox.rbx; rby = obj_ptr->bbox.rby;
      PrepareToReplaceAnObj (obj_ptr);
      strcpy (attr_ptr->s, file_name);
      UpdAttr (attr_ptr);
      if (attr_ptr->shown)
      {
         AdjObjCache (obj_ptr);
         AdjObjBBox (obj_ptr);
      }
      RecordReplaceAnObj (obj_ptr);
      RedrawAreas (botObj, ltx-GRID_ABS_SIZE(1), lty-GRID_ABS_SIZE(1),
            rbx+GRID_ABS_SIZE(1), rby+GRID_ABS_SIZE(1),
            obj_ptr->bbox.ltx-GRID_ABS_SIZE(1),
            obj_ptr->bbox.lty-GRID_ABS_SIZE(1),
            obj_ptr->bbox.rbx+GRID_ABS_SIZE(1),
            obj_ptr->bbox.rby+GRID_ABS_SIZE(1));
      SetFileModified (TRUE);
   }
   *buf = c_ptr;
   return TRUE;
}

static
int ExecUseTemplate (buf, obj_ptr)
   char			* * buf;
   struct ObjRec	* obj_ptr;
   /* create_file_using_simple_template(template_file,        */
   /*	output_file, string_to_match,substitution_attribute); */
{
   char			* c_ptr=(*buf);
   char			attr_name[MAXSTRING+1], file_name[MAXPATHLENGTH+1];
   char			template_name[MAXSTRING+1], match_str[MAXPATHLENGTH+1];
   char			msg[MAXSTRING+1];
   char			* real_match_str=NULL;
   char			* real_file_name, * real_template_name;
   int			len, tok_type;
   FILE			* in_fp=NULL, * out_fp=NULL;
   struct AttrRec	* attr_ptr;

   if (!((c_ptr = GetToken (c_ptr, execDummyStr, &tok_type)) != NULL &&
         tok_type == TOK_LEFT_P &&
         (c_ptr = GetToken (c_ptr, template_name, &tok_type)) != NULL &&
         tok_type == TOK_STR &&
         (c_ptr = GetToken (c_ptr, execDummyStr, &tok_type)) != NULL &&
         tok_type == TOK_COMMA &&
         (c_ptr = GetToken (c_ptr, file_name, &tok_type)) != NULL &&
         tok_type == TOK_STR &&
         (c_ptr = GetToken (c_ptr, execDummyStr, &tok_type)) != NULL &&
         tok_type == TOK_COMMA &&
         (c_ptr = GetToken (c_ptr, match_str, &tok_type)) != NULL &&
         tok_type == TOK_STR &&
         (c_ptr = GetToken (c_ptr, execDummyStr, &tok_type)) != NULL &&
         tok_type == TOK_COMMA &&
         (c_ptr = GetToken (c_ptr, attr_name, &tok_type)) != NULL &&
         tok_type == TOK_STR &&
         (c_ptr = GetToken (c_ptr, execDummyStr, &tok_type)) != NULL &&
         tok_type == TOK_RIGHT_P))
      return BadCmd ("create_file_using_simple_template");

   *buf = c_ptr;

   len = strlen (attr_name);
   attr_name[len++] = '=';
   attr_name[len] = '\0';
   attr_ptr = FindAttrWithName (obj_ptr, attr_name);
   if (attr_ptr == NULL)
      return (BadAttr( attr_name, "create_file_using_simple_template"));

   if ((real_file_name = convert_str (file_name, obj_ptr)) != NULL &&
         (real_template_name = convert_str (template_name, obj_ptr)) != NULL &&
         (real_match_str = convert_str (match_str, obj_ptr)) != NULL)
   {
      if ((out_fp = fopen (real_file_name, "w")) != NULL &&
            (in_fp = fopen (real_template_name, "r")) != NULL)
      {
         int		len_to_match = strlen (real_match_str), len;
         char		tmp_buf[MAXSTRING+1];
         struct StrRec	* str_ptr;

         while (fgets (tmp_buf, MAXSTRING, in_fp) != NULL)
         {
            len = strlen (tmp_buf);
            if (tmp_buf[len-1] == '\n')
            {
               tmp_buf[--len] = '\0';
               if (len==len_to_match && strcmp(tmp_buf,real_match_str)==0)
               {
                  if (*attr_ptr->s != '\0')
                     fprintf (out_fp, "%s\n", attr_ptr->s);
                  if (attr_ptr->obj->detail.t->first != NULL)
                     for (str_ptr = attr_ptr->obj->detail.t->first->next;
                           str_ptr != NULL; str_ptr = str_ptr->next)
                        fprintf (out_fp, "%s\n", str_ptr->s);
               }
               else
                  fprintf (out_fp, "%s\n", tmp_buf);
            }
            else if (len==len_to_match && strcmp(tmp_buf,real_match_str)==0)
            {
               if (*attr_ptr->s != '\0')
                  fprintf (out_fp, "%s\n", attr_ptr->s);
               if (attr_ptr->obj->detail.t->first != NULL)
                  for (str_ptr = attr_ptr->obj->detail.t->first->next;
                        str_ptr != NULL; str_ptr = str_ptr->next)
                     fprintf (out_fp, "%s\n", str_ptr->s);
            }
            else
               fprintf (out_fp, "%s\n", tmp_buf);
         }
         fclose (in_fp);
         fclose (out_fp);
         cfree (match_str);
         cfree (real_template_name);
         cfree (real_file_name);
         return (TRUE);
      }
   }
   if (real_match_str == NULL || real_template_name == NULL ||
         real_file_name == NULL)
      BadAttr( attr_name, "create_file_using_simple_template");
   else if (out_fp == NULL)
   {
      sprintf (msg, "Can not open '%s' for write.", real_file_name );
      Dialog (msg, "( <CR> or <ESC> to abort execution )", execDummyStr );
   }
   else if (in_fp == NULL)
   {
      sprintf (msg, "Can not open '%s' for read.", real_template_name );
      Dialog (msg, "( <CR> or <ESC> to abort execution )", execDummyStr );
   }
   if (real_match_str) cfree (match_str);
   if (real_template_name) cfree (real_template_name);
   if (real_file_name) cfree (real_file_name);
   if (out_fp != NULL) fclose (out_fp);
   if (in_fp != NULL) fclose (in_fp);
   return (FALSE);
}

static
int ExecUpdEpsChild (buf, obj_ptr)
   char			* * buf;
   struct ObjRec	* obj_ptr;
   /* update_eps_child(eps_file_name); */
{
   char			* c_ptr=(*buf), file_name[MAXPATHLENGTH+1];
   char			msg[MAXPATHLENGTH+1];
   char			* real_file_name, * * lines=NULL, write_date[32];
   struct ObjRec	* eps_obj_ptr, * del_obj=NULL;
   int			rc, num_lines, epsf_level, image_w, image_h, tok_type;
   int			del_obj_ltx=0, del_obj_lty=0, ltx, lty, rbx, rby;
   float		llx, lly, urx, ury;
   Pixmap		bitmap;
   XImage		* image=NULL;

   if (obj_ptr->type != OBJ_GROUP && obj_ptr->type != OBJ_ICON &&
         obj_ptr->type != OBJ_SYM)
   {
      Dialog ("update_eps_child() only works with composite objects.",
            "( <CR> or <ESC> to abort execution )", execDummyStr );
      return (FALSE);
   }
   if (!((c_ptr = GetToken (c_ptr, execDummyStr, &tok_type)) != NULL &&
         tok_type == TOK_LEFT_P &&
         (c_ptr = GetToken (c_ptr, file_name, &tok_type)) != NULL &&
         tok_type == TOK_STR &&
         (c_ptr = GetToken (c_ptr, execDummyStr, &tok_type)) != NULL &&
         tok_type == TOK_RIGHT_P))
      return BadCmd ("update_eps_child");

   *buf = c_ptr;
   if ((real_file_name = convert_str (file_name, obj_ptr)) == NULL)
      return BadCmd ("update_eps_child");

   for (del_obj=obj_ptr->detail.r->last; del_obj!=NULL; del_obj=del_obj->prev)
      if (del_obj->type==OBJ_XBM && del_obj->detail.xbm->real_type!=XBM_XBM)
      {
         del_obj_ltx = del_obj->obbox.ltx;
         del_obj_lty = del_obj->obbox.lty;
         break;
      }
   if (del_obj == NULL)
   {
      del_obj_ltx = obj_ptr->bbox.ltx;
      del_obj_lty = obj_ptr->bbox.rby;
   }

   importingFile = TRUE;
   rc = MyReadEPSFile (real_file_name, &image_w, &image_h, &bitmap, &image,
         &num_lines, &lines, &epsf_level, &llx, &lly, &urx, &ury, write_date);

   if (rc != BitmapSuccess)
   {
      importingFile = FALSE;
      if (real_file_name) cfree (real_file_name);
      sprintf (msg, "update_eps_child:  Fail to import '%s'.", real_file_name );
      Dialog (msg, "( <CR> or <ESC> to abort execution )", execDummyStr );
      return (FALSE);
   }
   ltx = obj_ptr->bbox.ltx; lty = obj_ptr->bbox.lty;
   rbx = obj_ptr->bbox.rbx; rby = obj_ptr->bbox.rby;

   PrepareToReplaceAnObj (obj_ptr);

   saveEPSLines = TRUE;
   eps_obj_ptr = CreateEPSObj (real_file_name, image_w, image_h, bitmap, image,
            num_lines, lines, epsf_level, llx, lly, urx, ury, write_date);
   saveEPSLines = FALSE;

   if (strcmp(defaultEPSScalingStr,"1") != 0)
      ScaleAnEPSObj (eps_obj_ptr, defaultEPSScaling);

   MoveObj (eps_obj_ptr, del_obj_ltx-eps_obj_ptr->obbox.ltx,
         del_obj_lty-eps_obj_ptr->obbox.lty);

   if (del_obj != NULL)
   {
      if (del_obj == obj_ptr->detail.r->first)
         obj_ptr->detail.r->first = del_obj->next;
      else
         del_obj->prev->next = del_obj->next;
      if (del_obj == obj_ptr->detail.r->last)
         obj_ptr->detail.r->last = del_obj->prev;
      else
         del_obj->next->prev = del_obj->prev;
      FreeObj (del_obj);
   }
   eps_obj_ptr->prev = NULL;
   eps_obj_ptr->next = obj_ptr->detail.r->first;
   if (obj_ptr->detail.r->first == NULL)
      obj_ptr->detail.r->last = eps_obj_ptr;
   else
      obj_ptr->detail.r->first->prev = eps_obj_ptr;
   obj_ptr->detail.r->first = eps_obj_ptr;

   AdjObjBBox (obj_ptr);

   RecordNewObjCmd ();
   RedrawAreas (botObj, ltx-GRID_ABS_SIZE(1), lty-GRID_ABS_SIZE(1),
      rbx+GRID_ABS_SIZE(1), rby+GRID_ABS_SIZE(1),
      obj_ptr->bbox.ltx-GRID_ABS_SIZE(1), obj_ptr->bbox.lty-GRID_ABS_SIZE(1),
      obj_ptr->bbox.rbx+GRID_ABS_SIZE(1), obj_ptr->bbox.rby+GRID_ABS_SIZE(1));
   SetFileModified (TRUE);
   justDupped = FALSE;

   importingFile = FALSE;

   return (TRUE);
}

static
long ms_time (tv)
   struct timeval	* tv;
{
   return ((long)(tv->tv_usec / 1000.0)) + ((long)(tv->tv_sec * 1000));
}

#define FD_STYLE_LINEAR 0
#define FD_STYLE_PINGPONG 1

#define FD_FORWARD 0
#define FD_REVERSE 1

static
int ExecFlipDeck (buf, obj_ptr)
   char			* * buf;
   struct ObjRec	* obj_ptr;
   /* flip_deck(times,frames_per_second,style);         */
   /*		style=[linear | pingpong] */
{
   char			* c_ptr=(*buf);
   char			times_str[MAXPATHLENGTH+1];
   char			fps_str[MAXPATHLENGTH+1], style_str[MAXPATHLENGTH+1];
   char			* real_times, * real_fps, * real_style;
   int			tok_type, num_bm, iteration, times, fps, style;
   int			rc=TRUE;
   struct ObjRec	* bm_obj, * cur_obj, * prev_obj, * next_obj;
   struct BBRec		obbox;
   int			select_width=XConnectionNumber(mainDisplay)+1;
   struct timeval	start;
   struct timezone	zone;
   long			ms_start_time, ms_frame_interval;
   fd_set		fdset;

   if (obj_ptr->type != OBJ_GROUP && obj_ptr->type != OBJ_ICON &&
         obj_ptr->type != OBJ_SYM)
   {
      Dialog ("flip_deck() only works with composite objects.",
            "( <CR> or <ESC> to c abort execution )", execDummyStr );
      return (FALSE);
   }
   num_bm = 0;
   for (bm_obj=obj_ptr->detail.r->last; bm_obj != NULL; bm_obj=bm_obj->prev)
   {
      if (bm_obj->type != OBJ_XBM && bm_obj->type != OBJ_XPM)
      {
         Dialog ("flip_deck():  Object contains non-Bitmap/Pixmap sub-objects.",
               "( <CR> or <ESC> to abort execution )", execDummyStr );
         return (FALSE);
      }
      if (num_bm++ == 0)
      {
         obbox.ltx = bm_obj->obbox.ltx; obbox.lty = bm_obj->obbox.lty;
         obbox.rbx = bm_obj->obbox.rbx; obbox.rby = bm_obj->obbox.rby;
      }
      else
      {
         if (obbox.ltx!=bm_obj->obbox.ltx || obbox.lty!=bm_obj->obbox.lty ||
               obbox.rbx!=bm_obj->obbox.rbx || obbox.rby!=bm_obj->obbox.rby)
         {
            Dialog ("flip_deck():  Different sizes Bitmap/Pixmap sub-objects.",
                  "( <CR> or <ESC> to abort execution )", execDummyStr );
            return (FALSE);
         }
      }
   }
   if (num_bm < 2)
   {
      Dialog ("flip_deck():  Must have > 1 Bitmap/Pixmap sub-objects.",
            "( <CR> or <ESC> to abort execution )", execDummyStr );
      return (FALSE);
   }

   if (!((c_ptr = GetToken (c_ptr, execDummyStr, &tok_type)) != NULL &&
         tok_type == TOK_LEFT_P &&
         (c_ptr = GetToken (c_ptr, times_str, &tok_type)) != NULL &&
         tok_type == TOK_STR &&
         (c_ptr = GetToken (c_ptr, execDummyStr, &tok_type)) != NULL &&
         tok_type == TOK_COMMA &&
         (c_ptr = GetToken (c_ptr, fps_str, &tok_type)) != NULL &&
         tok_type == TOK_STR &&
         (c_ptr = GetToken (c_ptr, execDummyStr, &tok_type)) != NULL &&
         tok_type == TOK_COMMA &&
         (c_ptr = GetToken (c_ptr, style_str, &tok_type)) != NULL &&
         tok_type == TOK_STR &&
         (c_ptr = GetToken (c_ptr, execDummyStr, &tok_type)) != NULL &&
         tok_type == TOK_RIGHT_P))
      return BadCmd ("flip_deck");

   *buf = c_ptr;
   if ((real_fps = convert_str (fps_str, obj_ptr)) == NULL)
      fps = 12;
   else
      fps = atoi (real_fps);
   if (fps < 1 || fps > 60)
   {
      Dialog ("flip_deck():  frames_per_second must be >= 1 and <= 60.",
            "( <CR> or <ESC> to abort execution )", execDummyStr );
      if (real_fps) cfree (real_fps);
      return (FALSE);
   }
   ms_frame_interval = 1000 / fps - 1;
   if ((real_style = convert_str (style_str, obj_ptr)) == NULL ||
         strcmp (real_style, "linear") == 0)
      style = FD_STYLE_LINEAR;
   else if (strcmp (real_style, "ping_pong") == 0)
      style = FD_STYLE_PINGPONG;
   else
   {
      Dialog ("flip_deck():  undefined style.",
            "( <CR> or <ESC> to abort execution )", execDummyStr );
      if (real_fps) cfree (real_fps);
      if (real_style) cfree (real_style);
      return (FALSE);
   }
   if ((real_times = convert_str (times_str, obj_ptr)) == NULL ||
         strcmp (real_times, "infinite") == 0)
      times = (-1);
   else
      times = atoi (real_times);

   iteration = 0;
   gettimeofday (&start, &zone);
   ms_start_time = ms_time (&start);
   while (rc == TRUE && (times < 0 || iteration < times))
   {
      int	looping=TRUE;
      int	direction=FD_FORWARD;

      if (times >= 0) iteration++;
      cur_obj = obj_ptr->detail.r->first;
      while (looping)
      {
         struct timeval	timeout, now;
         long		ms_cur_time, ms_interval;
         int		status;

         prev_obj = cur_obj->prev;
         next_obj = cur_obj->next;
         cur_obj->prev = cur_obj->next = NULL;
         switch (cur_obj->type)
         {
            case OBJ_XBM:
               if (iconWindowShown)
                  DrawXBmObj (iconWindow, 0, 0, cur_obj);
               else
                  DrawXBmObj (drawWindow, drawOrigX, drawOrigY, cur_obj);
               break;
            case OBJ_XPM:
               if (iconWindowShown)
                  DrawXPmObj (iconWindow, 0, 0, cur_obj);
               else
                  DrawXPmObj (drawWindow, drawOrigX, drawOrigY, cur_obj);
               break;
         }
         XSync (mainDisplay, FALSE);
         cur_obj->prev = prev_obj;
         cur_obj->next = next_obj;
         switch (style)
         {
            case FD_STYLE_LINEAR:
               if ((cur_obj=cur_obj->next) == NULL)
                  looping = FALSE;
               break;
            case FD_STYLE_PINGPONG:
               switch (direction)
               {
                  case FD_FORWARD:
                     if (cur_obj->next == NULL)
                     {
                        if ((cur_obj=cur_obj->prev) == obj_ptr->detail.r->first)
                           looping = FALSE;
                        else
                           direction = FD_REVERSE;
                     }
                     else
                        cur_obj = cur_obj->next;
                     break;
                  case FD_REVERSE:
                     if ((cur_obj=cur_obj->prev) == obj_ptr->detail.r->first)
                        looping = FALSE;
                     break;
               }
               break;
         }
         do
         {
            FD_ZERO (&fdset);
            FD_SET (select_width-1, &fdset);

            gettimeofday (&now, &zone);
            ms_cur_time = ms_time (&now);
            ms_interval = ms_start_time + ms_frame_interval - ms_cur_time;
            while (ms_interval <= 0)
            {
               ms_start_time = ms_cur_time;
               ms_interval = ms_start_time + ms_frame_interval - ms_cur_time;
            }
            timeout.tv_sec = 0;
            timeout.tv_usec = 1000 * ms_interval;

            status = select (select_width, &fdset, NULL, NULL, &timeout);

            if (status < 0)
            {
               Msg ("flip_deck():  select() system call failed.");
               looping = FALSE;
               rc = FALSE;
               break;
            }
            else if (status == 0)
               break;
            else
            {
               XEvent	ev;

               if (XCheckMaskEvent (mainDisplay,
                     ExposureMask | VisibilityChangeMask, &ev))
               {
                  ExposeEventHandler (&ev, TRUE);
                  while (XCheckMaskEvent (mainDisplay,
                        ExposureMask | VisibilityChangeMask, &ev)) ;
               }
               else if (XCheckMaskEvent (mainDisplay,
                     ButtonPressMask | KeyPressMask, &ev))
               {
                  Msg ("flip_deck():  User interrupt.");
                  looping = FALSE;
                  rc = FALSE;
                  break;
               }
            }
         } while (ms_interval > 0);
      }
   }

/* RedrawAnArea (botObj, obj_ptr->bbox.ltx-GRID_ABS_SIZE(1),
      obj_ptr->bbox.lty-GRID_ABS_SIZE(1), obj_ptr->bbox.rbx+GRID_ABS_SIZE(1),
      obj_ptr->bbox.rby+GRID_ABS_SIZE(1)); */

   cur_obj = obj_ptr->detail.r->first;
   prev_obj = cur_obj->prev;
   next_obj = cur_obj->next;
   cur_obj->prev = cur_obj->next = NULL;
   switch (cur_obj->type)
   {
      case OBJ_XBM:
         if (iconWindowShown)
            DrawXBmObj (iconWindow, 0, 0, cur_obj);
         else
            DrawXBmObj (drawWindow, drawOrigX, drawOrigY, cur_obj);
         break;
      case OBJ_XPM:
         if (iconWindowShown)
            DrawXPmObj (iconWindow, 0, 0, cur_obj);
         else
            DrawXPmObj (drawWindow, drawOrigX, drawOrigY, cur_obj);
         break;
   }
   XSync (mainDisplay, FALSE);
   cur_obj->prev = prev_obj;
   cur_obj->next = next_obj;

   if (real_times) cfree (real_times);
   if (real_fps) cfree (real_fps);
   if (real_style) cfree (real_style);
   return (rc);
}

int DoExec (exec_attr, obj_ptr)
   struct AttrRec	* exec_attr;
   struct ObjRec	* obj_ptr;
{
   char			* c_ptr;
   char			* cmd, * cmd_ptr, buf[MAXSTRING+1];
   int			cur_size=2*MAXSTRING, count=0, cmd_len=0, n;
   int			first_time=TRUE, tok_type, rc=TRUE;
   struct StrRec	* str_ptr;

   cmd = (char *) calloc (cur_size+4, sizeof(char));
   cmd_ptr = cmd;
   for (str_ptr = exec_attr->obj->detail.t->first; str_ptr != NULL;
         str_ptr = str_ptr->next)
   {
      if (first_time)
      {
         first_time = FALSE;
         c_ptr = exec_attr->s;
      }
      else
         c_ptr = str_ptr->s;

      if ((count = strlen (c_ptr)) != 0)
      {
         if (count+cmd_len >= cur_size)
         {
            n = cmd_ptr-cmd;
            cur_size += count+MAXSTRING;
            cmd = (char *) realloc (cmd, (cur_size+4)*sizeof(char));
            cmd_ptr = &cmd[n];
         }
         strncpy (cmd_ptr, c_ptr, count);
         cmd_ptr += count;
         cmd_len += count;
         if (str_ptr->next != NULL)
         {
            *cmd_ptr++ = ' ';
            *cmd_ptr = '\0';
            cmd_len++;
         }
      }
   }
   *cmd_ptr = '\0';
   MakeQuiescent ();
   StartCompositeCmd ();

   cmd_ptr = cmd;
   while (*cmd_ptr != '\0')
   {
      if ((c_ptr = GetToken (cmd_ptr, buf, &tok_type)) == NULL) break;

      if (tok_type == TOK_STR)
      {
         if (strcmp ("launch", buf) == 0)
         {
            if (!ExecLaunch (&c_ptr, obj_ptr)) { rc=FALSE; break; }
         }
         else if (strcmp ("exec", buf) == 0)
         {
            if (!ExecExec (&c_ptr, obj_ptr)) { rc=FALSE; break; }
         }
         else if (strcmp ("mktemp", buf) == 0)
         {
            if (!ExecMktemp (&c_ptr, obj_ptr)) { rc=FALSE; break; }
         }
         else if (strcmp ("create_file_using_simple_template", buf) == 0)
         {
            if (!ExecUseTemplate (&c_ptr, obj_ptr)) { rc=FALSE; break; }
         }
         else if (strcmp ("update_eps_child", buf) == 0)
         {
            if (!ExecUpdEpsChild (&c_ptr, obj_ptr)) { rc=FALSE; break; }
         }
         else if (strcmp ("flip_deck", buf) == 0)
         {
            if (!ExecFlipDeck (&c_ptr, obj_ptr)) { rc=FALSE; break; }
         }
         else
         {
            Dialog ("Unrecognized command.  Command execution aborted.",
                  "( <CR> or <ESC> to abort execution )", buf);
            rc = FALSE;
            break;
         }
      }
      else if (tok_type != TOK_SEMI)
      {
         Dialog ("Unrecognized command.  Command execution aborted.",
               "( <CR> or <ESC> to abort execution )", buf);
         rc = FALSE;
         break;
      }
      cmd_ptr = c_ptr;
      if (CheckInterrupt ())
      {
         Msg ("User interrupt.");
         rc = FALSE;
         break;
      }
   }
   EndCompositeCmd ();
   cfree (cmd);
   return (rc);
}

