/*
 * Author:	William Chia-Wei Cheng (william@cs.ucla.edu)
 *
 * Copyright (C) 1990, 1991, 1992, William Cheng.
 * 
 * Permission limited to the use, copy, modify, and distribute this software
 * and its documentation for any purpose is hereby granted by the Author without
 * fee, provided that the above copyright notice appear in all copies and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation, and that the name of the Author not be used
 * in advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.  The Author makes no
 * representations about the suitability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.  All other
 * rights are reserved by the Author.
 *
 * THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF
 * USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
 * OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */
#ifndef lint
static char RCSid[] =
      "@(#)$Header: /amnt/kona/tangram/u/william/X11/TGIF2/RCS/polygon.c,v 2.58.1.4 1993/01/03 23:34:24 william Exp $";
#endif

#include <math.h>
#include <stdio.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>
#include "const.h"
#include "types.h"

#include "cmd.e"
#include "color.e"
#include "cursor.e"
#include "dialog.e"
#include "grid.e"
#include "file.e"
#include "mainloop.e"
#include "msg.e"
#include "obj.e"
#include "pattern.e"
#include "poly.e"
#include "raster.e"
#include "ruler.e"
#include "select.e"
#include "setup.e"
#include "spline.e"
#include "xpixmap.e"

int	polygonDrawn = FALSE;

static struct PtRec	* lastPtPtr = NULL;
static int		startPolygonX, startPolygonY;

XPoint	* MakePolygonVertex (XOff, YOff, NumVs, Vs)
   int			XOff, YOff, NumVs;
   register XPoint	* Vs;
{
   register XPoint	* v;
   register int		i;
   int			real_x_off, real_y_off;

   real_x_off = (zoomedIn ? XOff : (XOff>>zoomScale)<<zoomScale);
   real_y_off = (zoomedIn ? YOff : (YOff>>zoomScale)<<zoomScale);

   v = (XPoint *) calloc (NumVs+1, sizeof(XPoint));
   if (v == NULL) fprintf (stderr, "Can not calloc().\n");
   for (i = 0; i < NumVs; i++)
   {
      v[i].x = ZOOMED_SIZE(Vs[i].x-real_x_off);
      v[i].y = ZOOMED_SIZE(Vs[i].y-real_y_off);
   }
   return (v);
}

void DumpPoints (FP, NumPts, V, Indent)
   FILE		* FP;
   int		NumPts, Indent;
   XPoint	* V;
{
   register int	i, j;

   for (i = 1; i < NumPts; i++)
   {
      for (j = 0; j < Indent; j++) fprintf (FP, " ");
      fprintf (FP, "%1d %1d lineto\n", V[i].x, V[i].y);
   }
}

static
void DumpPolygonPath (FP, ObjPtr, Vs, NumPts, Width, Pen, Curved, Dash)
   FILE			* FP;
   struct ObjRec	* ObjPtr;
   XPoint		* Vs;
   int			NumPts;
   int			Width, Pen, Curved, Dash;
{
   register int	i;

   if (Width != 1) fprintf (FP, "   %1d setlinewidth\n", Width);
   if (Dash != 0)
   {
      fprintf (FP, "   [");
      for (i = 0; i < dashListLength[Dash]-1; i++)
         fprintf (FP, "%1d ", (int)(dashList[Dash][i]));
      fprintf (FP, "%1d] 0 setdash\n",
            (int)(dashList[Dash][dashListLength[Dash]-1]));
   }

   if (!colorDump && Pen > BACKPAT)
   {
      GrayCheck (Pen);
      if (useGray)
         fprintf (FP, "   %s setgray\n", GrayStr(Pen));
      else
         fprintf (FP, "   pat%1d %s\n", Pen, patternStr);
   }

   switch (Curved)
   {
      case LT_STRAIGHT:
         fprintf (FP, "   newpath\n   %1d %1d moveto\n", Vs[0].x, Vs[0].y);
         DumpPoints (FP, NumPts-1, Vs, 6);
         fprintf (FP, "   closepath ");
         break;
      case LT_SPLINE:
         fprintf (FP, "   newpath\n");
         DumpCurvedPolygonPoints (FP, NumPts, Vs, 6);
         fprintf (FP, "   closepath ");
         break;
   }

   switch (Pen)
   {
      case SOLIDPAT: fprintf (FP, "stroke\n"); break;
      case BACKPAT: fprintf (FP, "1 setgray stroke 0 setgray\n"); break;
      default:
         if (colorDump)
         {
            fprintf (FP, "flattenpath strokepath clip\n");
            DumpPatFill (FP, Pen, 8, ObjPtr->bbox, "   ");
         }
         else
            fprintf (FP, "stroke\n");
         break;
   }
   if (Dash != 0) fprintf (FP, "   [] 0 setdash\n");
   if (Width != 1) fprintf (FP, "   1 setlinewidth\n");
}

void DumpPolygonObj (FP, ObjPtr)
   FILE			* FP;
   struct ObjRec	* ObjPtr;
{
   XPoint	* v = ObjPtr->detail.g->vlist;
   int		num_pts = ObjPtr->detail.g->n;
   int		fill, width, pen, curved, dash, color_index;

   fill = ObjPtr->detail.g->fill;
   width = ObjPtr->detail.g->width;
   pen = ObjPtr->detail.g->pen;
   curved = ObjPtr->detail.g->curved;
   dash = ObjPtr->detail.g->dash;

   if (fill == NONEPAT && pen == NONEPAT) return;

   fprintf (FP, "%% POLYGON/CLOSED-SPLINE\n");
   color_index = ObjPtr->color;
   if (colorDump)
      fprintf (FP, "%.3f %.3f %.3f setrgbcolor\n",
            ((float)tgifColors[color_index].red/maxRGB),
            ((float)tgifColors[color_index].green/maxRGB),
            ((float)tgifColors[color_index].blue/maxRGB));

   if (fill != NONEPAT)
   {
      switch (curved)
      {
         case LT_STRAIGHT:
            switch (fill)
            {
               case SOLIDPAT: /* solid black polygon */
                  fprintf (FP, "newpath\n   %1d %1d moveto\n", v[0].x, v[0].y);
                  DumpPoints (FP, num_pts-1, v, 3);
                  fprintf (FP, "closepath eofill\n");
                  break;
               case BACKPAT: /* solid white polygon */
                  fprintf (FP, "newpath\n   %1d %1d moveto\n", v[0].x, v[0].y);
                  DumpPoints (FP, num_pts-1, v, 3);
                  fprintf (FP, "closepath 1 setgray eofill\n");
                  if (colorDump)
                     fprintf (FP, "   %.3f %.3f %.3f setrgbcolor\n",
                           ((float)tgifColors[color_index].red/maxRGB),
                           ((float)tgifColors[color_index].green/maxRGB),
                           ((float)tgifColors[color_index].blue/maxRGB));
                  else
                     fprintf (FP, "0 setgray\n");
                  break;
               default: /* patterned */
                  fprintf (FP, "gsave\n");
                  if (!colorDump)
                  {
                     GrayCheck (fill);
                     if (useGray)
                        fprintf (FP, "   %s setgray\n", GrayStr(fill));
                     else
                        fprintf (FP, "   pat%1d %s\n", fill, patternStr);
                  }
                  else
                  {
                     fprintf (FP, "   newpath\n");
                     fprintf (FP, "      %1d %1d moveto\n", v[0].x, v[0].y);
                     DumpPoints (FP, num_pts-1, v, 6);
                     fprintf (FP, "   closepath 1 setgray eofill\n");
                     fprintf (FP, "   %.3f %.3f %.3f setrgbcolor\n",
                           ((float)tgifColors[color_index].red/maxRGB),
                           ((float)tgifColors[color_index].green/maxRGB),
                           ((float)tgifColors[color_index].blue/maxRGB));
                  }
                  fprintf (FP, "   newpath\n");
                  fprintf (FP, "      %1d %1d moveto\n", v[0].x, v[0].y);
                  DumpPoints (FP, num_pts-1, v, 6);
                  if (colorDump)
                  {
                     fprintf (FP, "   closepath eoclip\n");
                     DumpPatFill (FP, fill, 8, ObjPtr->bbox, "   ");
                  }
                  else
                     fprintf (FP, "   closepath eofill\n");
                  fprintf (FP, "grestore\n");
                  break;
            }
            break;
         case LT_SPLINE:
            switch (fill)
            {
               case SOLIDPAT: /* solid black polygon */
                  fprintf (FP, "newpath\n");
                  DumpCurvedPolygonPoints (FP, num_pts, v, 3);
                  fprintf (FP, "closepath eofill\n");
                  break;
               case BACKPAT: /* solid white polygon */
                  fprintf (FP, "newpath\n");
                  DumpCurvedPolygonPoints (FP, num_pts, v, 3);
                  fprintf (FP, "closepath 1 setgray eofill\n");
                  if (colorDump)
                     fprintf (FP, "   %.3f %.3f %.3f setrgbcolor\n",
                           ((float)tgifColors[color_index].red/maxRGB),
                           ((float)tgifColors[color_index].green/maxRGB),
                           ((float)tgifColors[color_index].blue/maxRGB));
                  else
                     fprintf (FP, "0 setgray\n");
                  break;
               default: /* patterned */
                  fprintf (FP, "gsave\n");
                  if (!colorDump)
                  {
                     GrayCheck (fill);
                     if (useGray)
                        fprintf (FP, "   %s setgray\n", GrayStr(fill));
                     else
                        fprintf (FP, "   pat%1d %s\n", fill, patternStr);
                  }
                  else
                  {
                     fprintf (FP, "   newpath\n");
                     DumpCurvedPolygonPoints (FP, num_pts, v, 6);
                     fprintf (FP, "   closepath 1 setgray eofill\n");
                     fprintf (FP, "   %.3f %.3f %.3f setrgbcolor\n",
                           ((float)tgifColors[color_index].red/maxRGB),
                           ((float)tgifColors[color_index].green/maxRGB),
                           ((float)tgifColors[color_index].blue/maxRGB));
                  }
                  fprintf (FP, "   newpath\n");
                  DumpCurvedPolygonPoints (FP, num_pts, v, 6);
                  if (colorDump)
                  {
                     fprintf (FP, "   closepath eoclip\n");
                     DumpPatFill (FP, fill, 8, ObjPtr->bbox, "   ");
                  }
                  else
                     fprintf (FP, "   closepath eofill\n");
                  fprintf (FP, "grestore\n");
                  break;
            }
            break;
      }
   }

   if (pen == NONEPAT) { fprintf (FP, "\n"); return; }

   fprintf (FP, "gsave\n");

   if (colorDump && pen > BACKPAT)
   {
      DumpPolygonPath (FP, ObjPtr, v, num_pts, width, BACKPAT, curved, 0);
      if (colorDump)
         fprintf (FP, "   %.3f %.3f %.3f setrgbcolor\n",
               ((float)tgifColors[color_index].red/maxRGB),
               ((float)tgifColors[color_index].green/maxRGB),
               ((float)tgifColors[color_index].blue/maxRGB));
   }
   DumpPolygonPath (FP, ObjPtr, v, num_pts, width, pen, curved, dash);

   fprintf (FP, "grestore\n");

   fprintf (FP, "\n");
}

void DrawPolygonObj (Win, XOff, YOff, ObjPtr)
   Window		Win;
   int			XOff, YOff;
   struct ObjRec	* ObjPtr;
{
   struct PolygonRec	* polygon_ptr = ObjPtr->detail.g;
   XPoint		* v;
   int			fill, width, pen, dash, pixel;
   int			num_pts;
   XGCValues		values;

   fill = polygon_ptr->fill;
   width = polygon_ptr->width;
   pen = polygon_ptr->pen;
   dash = polygon_ptr->dash;
   pixel = colorPixels[ObjPtr->color];

   if (fill == NONEPAT && pen == NONEPAT) return;

   v = polygon_ptr->svlist;
   num_pts = polygon_ptr->sn;

   if (fill != 0)
   {
      values.foreground = (fill == BACKPAT) ? myBgPixel : pixel;
      values.function = GXcopy;
      values.fill_style = FillOpaqueStippled;
      values.stipple = patPixmap[fill];
      XChangeGC (mainDisplay, drawGC,
            GCForeground | GCFunction | GCFillStyle | GCStipple, &values);
      XFillPolygon (mainDisplay, Win, drawGC, v, num_pts, Complex,
            CoordModeOrigin);
   }

   if (pen == NONEPAT) return;

   values.foreground = (pen == BACKPAT) ? myBgPixel : pixel;
   values.function = GXcopy;
   values.fill_style = FillOpaqueStippled;
   values.stipple = patPixmap[pen];
   values.line_width = ZOOMED_SIZE(width);
#ifdef NO_THIN_LINE
   if (values.line_width < 1) values.line_width = 1;
#endif
   values.join_style = JoinBevel;
   if (dash != 0)
   {
      XSetDashes (mainDisplay, drawGC, 0, dashList[dash],
            dashListLength[dash]);
      values.line_style = LineOnOffDash;
   }
   else
      values.line_style = LineSolid;
   XChangeGC (mainDisplay, drawGC,
         GCForeground | GCFunction | GCFillStyle | GCStipple |
         GCLineWidth | GCLineStyle | GCJoinStyle, &values);

   XDrawLines (mainDisplay, Win, drawGC, v, num_pts, CoordModeOrigin);
}

static
void CreatePolygonObj (NumPts)
   int	NumPts;
{
   register int		i;
   struct PtRec		* pt_ptr, * next_pt;
   struct PolygonRec	* polygon_ptr;
   struct ObjRec	* obj_ptr;
   XPoint		* v;
   int			width, w, ltx, lty, rbx, rby;

   polygon_ptr = (struct PolygonRec *) calloc (1, sizeof(struct PolygonRec));
   polygon_ptr->n = NumPts;
   v = (XPoint *) calloc (NumPts+1, sizeof(XPoint));
   if (v == NULL) fprintf (stderr, "Can not calloc().\n");
   pt_ptr = lastPtPtr;
   ltx = rbx = pt_ptr->x;
   lty = rby = pt_ptr->y;
   for (i = NumPts-1; i >= 0; i--, lastPtPtr = next_pt)
   {
      next_pt = lastPtPtr->next;
      v[i].x = ABS_X(lastPtPtr->x);
      v[i].y = ABS_Y(lastPtPtr->y);
      if (lastPtPtr->x < ltx) ltx = lastPtPtr->x;
      if (lastPtPtr->y < lty) lty = lastPtPtr->y;
      if (lastPtPtr->x > rbx) rbx = lastPtPtr->x;
      if (lastPtPtr->y > rby) rby = lastPtPtr->y;
      cfree (lastPtPtr);
   }

   polygon_ptr->vlist = v;
   polygon_ptr->svlist = NULL;
   polygon_ptr->fill = objFill;
   polygon_ptr->width = width = curWidthOfLine[lineWidth];
   polygon_ptr->pen = penPat;
   polygon_ptr->curved = curSpline;
   polygon_ptr->dash = curDash;
   obj_ptr = (struct ObjRec *) calloc (1, sizeof(struct ObjRec));
   obj_ptr->color = colorIndex;
   obj_ptr->type = OBJ_POLYGON;
   obj_ptr->bbox.ltx = obj_ptr->obbox.ltx = obj_ptr->x = ABS_X(ltx);
   obj_ptr->bbox.lty = obj_ptr->obbox.lty = obj_ptr->y = ABS_Y(lty);
   obj_ptr->bbox.rbx = obj_ptr->obbox.rbx = ABS_X(rbx);
   obj_ptr->bbox.rby = obj_ptr->obbox.rby = ABS_Y(rby);
   w = HALF_W(width);
   obj_ptr->bbox.ltx -= w;
   obj_ptr->bbox.lty -= w;
   obj_ptr->bbox.rbx += w;
   obj_ptr->bbox.rby += w;
   obj_ptr->id = objId++;
   obj_ptr->dirty = FALSE;
   obj_ptr->rotation = 0;
   obj_ptr->locked = FALSE;
   obj_ptr->fattr = obj_ptr->lattr = NULL;
   obj_ptr->detail.g = polygon_ptr;
   AdjObjSplineVs (obj_ptr);
   AddObj (NULL, topObj, obj_ptr);
}

static XComposeStatus	c_stat;

static
void ContinuePolygon (OrigX, OrigY)
   int 	OrigX, OrigY;
   /* OrigX and OrigY are screen coordinates (scaled and translated). */
{
   XGCValues		values;
   XEvent		input, ev;
   XButtonEvent		* button_ev;
   XMotionEvent		* motion_ev;
   KeySym		key_sym;
   char			s[80];
   int 			end_x, end_y, grid_x, grid_y, done = FALSE;
   int 			saved_x, saved_y, closed=FALSE;
   int			xor_pixel, num_pts=1, n=2, sn=0, max_n=40;
   struct PtRec		* pt_ptr;
   XPoint		* v = NULL, * sv = NULL;

   xor_pixel = xorColorPixels[colorIndex];

   values.foreground = xor_pixel;
   values.function = GXxor;
   values.fill_style = FillSolid;
#ifdef NO_THIN_LINE
   values.line_width = 1;
#else
   values.line_width = 0;
#endif
   values.line_style = LineSolid;

   XChangeGC (mainDisplay, drawGC,
         GCForeground | GCFunction | GCFillStyle | GCLineWidth | GCLineStyle,
         &values);

   grid_x = end_x = saved_x = OrigX;
   grid_y = end_y = saved_y = OrigY;
   if (curSpline == LT_SPLINE && splineRubberband)
   {
      v = (XPoint *) calloc (max_n+1, sizeof (XPoint));
      if (v == NULL) fprintf (stderr, "Can not calloc().\n");
      v[0].x = v[1].x = v[2].x = ABS_X(OrigX);
      v[0].y = v[1].y = v[2].y = ABS_Y(OrigY);
      sv = MakeSplinePolygonVertex (&sn, drawOrigX, drawOrigY, n+1, v);
   }

   Msg ("Click middle or right button to end polygon.");
   XGrabPointer (mainDisplay, drawWindow, FALSE,
         PointerMotionMask | ButtonPressMask,
         GrabModeAsync, GrabModeAsync, None, handCursor, CurrentTime);
   
   while (!done)
   {
      XNextEvent (mainDisplay, &input);

      if (input.type == Expose || input.type == VisibilityNotify)
         ExposeEventHandler (&input, TRUE);
      else if ((input.type == KeyPress || input.type == KeyRelease) &&
            num_pts != 1 && !(curSpline == LT_SPLINE && splineRubberband))
      {
         XLookupString (&(input.xkey), s, 80-1, &key_sym, &c_stat);
         if (key_sym == XK_Shift_L || key_sym == XK_Shift_R ||
               key_sym == XK_Control_L || key_sym == XK_Control_R)
         {
            XDrawLine (mainDisplay, drawWindow, drawGC, saved_x, saved_y,
                  grid_x, grid_y);
            closed = !closed;
         }
      }
      else if (input.type == MotionNotify)
      {
         if (curSpline == LT_SPLINE && splineRubberband)
            XDrawLines (mainDisplay, drawWindow, drawGC, sv, sn,
                  CoordModeOrigin);
         else
         {
            XDrawLine (mainDisplay, drawWindow, drawGC, OrigX, OrigY, grid_x,
                  grid_y);
            if (num_pts != 1 && closed)
               XDrawLine (mainDisplay, drawWindow, drawGC, saved_x, saved_y,
                     grid_x, grid_y);
         }
         motion_ev = &(input.xmotion);
         end_x = motion_ev->x;
         end_y = motion_ev->y;
         GridXY (end_x, end_y, &grid_x, &grid_y);
         MarkRulers (grid_x, grid_y);
         if (curSpline == LT_SPLINE && splineRubberband)
         {
            cfree (sv);
            v[n-1].x = ABS_X(grid_x);
            v[n-1].y = ABS_Y(grid_y);
            sv = MakeSplinePolygonVertex (&sn, drawOrigX, drawOrigY, n+1, v);
            XDrawLines (mainDisplay, drawWindow, drawGC, sv, sn,
                  CoordModeOrigin);
         }
         else
         {
            if (num_pts != 1)
            {
               if (motion_ev->state & (ShiftMask | ControlMask))
               {
                  XDrawLine (mainDisplay, drawWindow, drawGC, saved_x, saved_y,
                        grid_x, grid_y);
                  closed = TRUE;
               }
               else
                  closed = FALSE;
            }
            XDrawLine (mainDisplay, drawWindow, drawGC, OrigX, OrigY, grid_x,
                  grid_y);
         }
         while (XCheckMaskEvent (mainDisplay, PointerMotionMask, &ev)) ;
      }
      else if (input.type == ButtonPress)
      {
         button_ev = &(input.xbutton);

         end_x = button_ev->x;
         end_y = button_ev->y;
         GridXY (end_x, end_y, &grid_x, &grid_y);

         if ((grid_x != OrigX || grid_y != OrigY) &&
               (button_ev->button == Button1 || num_pts != 1))
         {
            num_pts++;
            pt_ptr = (struct PtRec *) calloc (1, sizeof(struct PtRec));
            pt_ptr->next = lastPtPtr;
            lastPtPtr = pt_ptr;
            pt_ptr->x = grid_x;
            pt_ptr->y = grid_y;
            if (curSpline == LT_SPLINE && splineRubberband)
            {
               if (n >= max_n-3)
               {
                  max_n += 40;
                  v = (XPoint *) realloc (v, sizeof(XPoint)*max_n+1);
                  if (v == NULL) fprintf (stderr, "Can not realloc().\n");
               }
               XDrawLines (mainDisplay, drawWindow, drawGC, sv, sn,
                     CoordModeOrigin);
               cfree (sv);
               v[n].x = ABS_X(grid_x);
               v[n].y = ABS_Y(grid_y);
               n++;
               v[n].x = v[0].x;
               v[n].y = v[0].y;
               sv = MakeSplinePolygonVertex (&sn, drawOrigX, drawOrigY, n+1, v);
               XDrawLines (mainDisplay, drawWindow, drawGC, sv, sn,
                     CoordModeOrigin);
            }
         }

         if (num_pts == 2 && closed && button_ev->button == Button1 &&
               !(curSpline == LT_SPLINE && splineRubberband))
            XDrawLine (mainDisplay, drawWindow, drawGC, OrigX, OrigY, grid_x,
                  grid_y);

         if (grid_x == startPolygonX && grid_y == startPolygonY)
         {
            if (curSpline == LT_SPLINE && splineRubberband)
               XDrawLines (mainDisplay, drawWindow, drawGC, sv, sn,
                     CoordModeOrigin);
            else
            {
               XDrawLine (mainDisplay, drawWindow, drawGC, OrigX, OrigY,
                     grid_x, grid_y);
               if (num_pts != 1 && closed)
                  XDrawLine (mainDisplay, drawWindow, drawGC, saved_x,
                        saved_y, grid_x, grid_y);
               if (num_pts < 4)
                  XDrawLine (mainDisplay, drawWindow, drawGC, saved_x,
                        saved_y, OrigX, OrigY);
            }
            done = TRUE;
         }
         else
         {
            switch(button_ev->button)
            {
               case Button1 : OrigX = grid_x; OrigY = grid_y; break;
               case Button2 :
               case Button3 :
                  if (curSpline == LT_SPLINE && splineRubberband)
                     XDrawLines (mainDisplay, drawWindow, drawGC, sv, sn,
                           CoordModeOrigin);
                  else
                  {
                     XDrawLine (mainDisplay, drawWindow, drawGC, OrigX, OrigY,
                           grid_x, grid_y);
                     if (num_pts != 1 && closed)
                        XDrawLine (mainDisplay, drawWindow, drawGC, saved_x,
                              saved_y, grid_x, grid_y);
                     if (num_pts < 3)
                        XDrawLine (mainDisplay, drawWindow, drawGC, saved_x,
                              saved_y, OrigX, OrigY);
                  }
                  if (num_pts == 1)
                  {
                     done = TRUE;
                     break;
                  }
                  num_pts++;
                  pt_ptr = (struct PtRec *) calloc (1, sizeof(struct PtRec));
                  pt_ptr->next = lastPtPtr;
                  lastPtPtr = pt_ptr;
                  pt_ptr->x = startPolygonX;
                  pt_ptr->y = startPolygonY;
                  done = TRUE;
                  break;
            }
         }
      }
   }
   XUngrabPointer (mainDisplay, CurrentTime);
   Msg ("");

   if (curSpline == LT_SPLINE && splineRubberband) { cfree (v); cfree (sv); }

   switch (num_pts)
   {
      case 1: cfree (lastPtPtr); break;
      case 2: cfree (lastPtPtr->next); cfree (lastPtPtr); break;
      case 3:
         cfree (lastPtPtr->next->next);
         cfree (lastPtPtr->next);
         cfree (lastPtPtr);
         break;
      default:
         CreatePolygonObj (num_pts);
         RecordNewObjCmd ();
         RedrawAnArea (botObj, topObj->bbox.ltx-GRID_ABS_SIZE(1),
               topObj->bbox.lty-GRID_ABS_SIZE(1),
               topObj->bbox.rbx+GRID_ABS_SIZE(1),
               topObj->bbox.rby+GRID_ABS_SIZE(1));
         polygonDrawn = TRUE;
         SetFileModified (TRUE);
         break;
   }
}

void DrawPolygon (input)
   XEvent	* input;
{
   XButtonEvent	* button_ev;
   int		mouse_x, mouse_y, grid_x, grid_y;

   if (input->type != ButtonPress) return;

   button_ev = &(input->xbutton);
   if (button_ev->button == Button1)
   {
      mouse_x = button_ev->x;
      mouse_y = button_ev->y;
      GridXY (mouse_x, mouse_y, &grid_x, &grid_y);
      lastPtPtr = (struct PtRec *) calloc (1, sizeof(struct PtRec));
      lastPtPtr->x = startPolygonX = grid_x;
      lastPtPtr->y = startPolygonY = grid_y;
      lastPtPtr->next = NULL;
      ContinuePolygon (grid_x, grid_y);
   } 
}

void SavePolygonObj (FP, ObjPtr)
   FILE			* FP;
   struct ObjRec	* ObjPtr;
{
   register int		i, n;
   int			count;
   struct PolygonRec	* polygon_ptr = ObjPtr->detail.g;

   n = polygon_ptr->n;
   fprintf (FP, "polygon('%s',%1d,[\n\t",
         colorMenuItems[ObjPtr->color], polygon_ptr->n);
   for (i = 0, count = 0; i < n-1; i++)
   {
      fprintf (FP,"%1d,%1d,",polygon_ptr->vlist[i].x,polygon_ptr->vlist[i].y);
      if (++count == 8)
      {
         count = 0;
         fprintf (FP, "\n\t");
      }
   }
   fprintf (FP, "%1d,%1d],",
         polygon_ptr->vlist[n-1].x, polygon_ptr->vlist[n-1].y);

   fprintf (FP, "%1d,%1d,%1d,%1d,%1d,%1d,%1d,%1d,",
         polygon_ptr->fill, polygon_ptr->width, polygon_ptr->pen,
         polygon_ptr->curved, ObjPtr->id, polygon_ptr->dash, ObjPtr->rotation,
         ObjPtr->locked);
   SaveAttrs (FP, ObjPtr->lattr);
   fprintf (FP, ")");
}

#define GETVALUE(val,name) ScanValue("%d", (char *) &(val), name, "polygon")

void ReadPolygonObj (FP, Inbuf, ObjPtr)
   FILE			* FP;
   char			* Inbuf;
   struct ObjRec	* * ObjPtr;
{
   register int		i;
   struct PolygonRec	* polygon_ptr;
   XPoint		* v;
   char			color_str[20], * s, inbuf[MAXSTRING], msg[MAXSTRING];
   int			num_pts, ltx=0, lty=0, rbx=0, rby=0, x, y, id=0;
   int			fill, width, pen, w, new_alloc, locked=FALSE;
   int			curved, dash, initialized, rotation, count;

   *ObjPtr = NULL;

   s = FindChar ('(', Inbuf);
   s = ParseStr (s, ',', color_str);

   InitScan (s, "\t\n, []");

   if (GETVALUE (num_pts, "number of points") == INVALID)
      return;

   if (num_pts <= 0)
   {
      (void) sprintf (msg, "%s, %d:  Invalid number of points in polygon",
            scanFileName, scanLineNum);
      if (PRTGIF)
         fprintf (stderr, "%s\n", msg);
      else
         Msg (msg);
      return;
   }

   * ObjPtr = (struct ObjRec *) calloc (1, sizeof(struct ObjRec));
   polygon_ptr = (struct PolygonRec *) calloc (1, sizeof(struct PolygonRec));

   if (num_pts < 3)
   {
      v = (XPoint *) calloc (5, sizeof(XPoint));
      if (v == NULL) fprintf (stderr, "Can not calloc().\n");
   }
   else
   {
      v = (XPoint *) calloc (num_pts+1, sizeof(XPoint));
      if (v == NULL) fprintf (stderr, "Can not calloc().\n");
   }

   initialized = FALSE;

   if (fileVersion <= 13)
   {
      for (i = 0; i < num_pts; i++)
      {
         if (GETVALUE (x, "x") == INVALID || GETVALUE (y, "y") == INVALID)
         {
            cfree (*ObjPtr);
            cfree (polygon_ptr);
            cfree (v);
            *ObjPtr = NULL;
            return;
         }
         v[i].x = x; v[i].y = y;
         if (!initialized)
         {
            initialized = TRUE;
            ltx = rbx = x; lty = rby = y;
         }
         else
         {
            if (x < ltx) ltx = x; if (y < lty) lty = y;
            if (x > rbx) rbx = x; if (y > rby) rby = y;
         }
      }
   }
   else
   {
      fgets (inbuf, MAXSTRING, FP);
      scanLineNum++;
      s = inbuf;
      InitScan (s, "\t\n, []");
      for (i = 0, count = 0; i < num_pts; i++)
      {
         if (GETVALUE (x, "x") == INVALID || GETVALUE (y, "y") == INVALID)
         {
            cfree (*ObjPtr);
            cfree (polygon_ptr);
            cfree (v);
            *ObjPtr = NULL;
            return;
         }
         v[i].x = x; v[i].y = y;
         if (!initialized)
         {
            initialized = TRUE;
            ltx = rbx = x; lty = rby = y;
         }
         else
         {
            if (x < ltx) ltx = x; if (y < lty) lty = y;
            if (x > rbx) rbx = x; if (y > rby) rby = y;
         }
         if (++count == 8 && i != num_pts-1)
         {
            count = 0;
            fgets (inbuf, MAXSTRING, FP);
            scanLineNum++;
            s = inbuf;
            InitScan (s, "\t\n, []");
         }
      }
   }

   switch (num_pts)
   {
      case 1:
         sprintf (msg, "%s (%1d,%1d) converted to 3-point polygon.",
               "1-point polygon", v[0].x, v[0].y);
         if (PRTGIF)
            fprintf (stderr, "%s\n", msg);
         else
            Msg (msg);
         v[2].x = v[1].x = v[0].x;
         v[2].y = v[1].y = v[0].y;
         num_pts = 3;
         break;
      case 2:
         sprintf (msg, "%s [%1d,%1d,%1d,%1d] converted to 3-point polygon.",
               "2-point polygon", v[0].x, v[0].y, v[1].x, v[1].y);
         if (PRTGIF)
            fprintf (stderr, "%s\n", msg);
         else
            Msg (msg);
         v[2].x = v[0].x;
         v[2].y = v[0].y;
         num_pts = 3;
         break;
   }

   polygon_ptr->n = num_pts;

   dash = 0;
   rotation = 0;
   if (fileVersion <= 3)
   {
      if (GETVALUE (fill,     "fill") == INVALID ||
          GETVALUE (width,    "width") == INVALID ||
          GETVALUE (pen,      "pen") == INVALID)
      {
         cfree (*ObjPtr);
         cfree (polygon_ptr);
         cfree (v);
         *ObjPtr = NULL;
         return;
      }
      if (width == LINE_CURVED)
      {
         width = 0;
         curved = TRUE;
      }
      else
         curved = FALSE;
      switch (width)
      {
         case 1: width = 3; break;
         case 2: width = 6; break;
      }
      id = objId++;
   }
   else if (fileVersion <= 5)
   {
      if (GETVALUE (fill,     "fill") == INVALID ||
          GETVALUE (width,    "width") == INVALID ||
          GETVALUE (pen,      "pen") == INVALID ||
          GETVALUE (curved,   "curved") == INVALID)
      {
         cfree (*ObjPtr);
         cfree (polygon_ptr);
         cfree (v);
         *ObjPtr = NULL;
         return;
      }
      switch (width)
      {
         case 1: width = 3; break;
         case 2: width = 6; break;
      }
      id = objId++;
   }
   else if (fileVersion <= 7)
   {
      if (GETVALUE (fill,     "fill") == INVALID ||
          GETVALUE (width,    "width") == INVALID ||
          GETVALUE (pen,      "pen") == INVALID ||
          GETVALUE (curved,   "curved") == INVALID)
      {
         cfree (*ObjPtr);
         cfree (polygon_ptr);
         cfree (v);
         *ObjPtr = NULL;
         return;
      }
      id = objId++;
   }
   else if (fileVersion <= 8)
   {
      if (GETVALUE (fill,     "fill") == INVALID ||
          GETVALUE (width,    "width") == INVALID ||
          GETVALUE (pen,      "pen") == INVALID ||
          GETVALUE (curved,   "curved") == INVALID ||
          GETVALUE (id,       "id") == INVALID)
      {
         cfree (*ObjPtr);
         cfree (polygon_ptr);
         cfree (v);
         *ObjPtr = NULL;
         return;
      }
      if (id >= objId) objId = id+1;
   }
   else if (fileVersion <= 13)
   {
      if (GETVALUE (fill,     "fill") == INVALID ||
          GETVALUE (width,    "width") == INVALID ||
          GETVALUE (pen,      "pen") == INVALID ||
          GETVALUE (curved,   "curved") == INVALID ||
          GETVALUE (id,       "id") == INVALID ||
          GETVALUE (dash,     "dash") == INVALID)
      {
         cfree (*ObjPtr);
         cfree (polygon_ptr);
         cfree (v);
         *ObjPtr = NULL;
         return;
      }
      if (id >= objId) objId = id+1;
   }
   else if (fileVersion <= 25)
   {
      if (GETVALUE (fill,     "fill") == INVALID ||
          GETVALUE (width,    "width") == INVALID ||
          GETVALUE (pen,      "pen") == INVALID ||
          GETVALUE (curved,   "curved") == INVALID ||
          GETVALUE (id,       "id") == INVALID ||
          GETVALUE (dash,     "dash") == INVALID ||
          GETVALUE (rotation, "rotation") == INVALID)
      {
         cfree (*ObjPtr);
         cfree (polygon_ptr);
         cfree (v);
         *ObjPtr = NULL;
         return;
      }
      if (id >= objId) objId = id+1;
   }
   else
   {
      if (GETVALUE (fill,     "fill") == INVALID ||
          GETVALUE (width,    "width") == INVALID ||
          GETVALUE (pen,      "pen") == INVALID ||
          GETVALUE (curved,   "curved") == INVALID ||
          GETVALUE (id,       "id") == INVALID ||
          GETVALUE (dash,     "dash") == INVALID ||
          GETVALUE (rotation, "rotation") == INVALID ||
          GETVALUE (locked,   "locked") == INVALID)
      {
         cfree (*ObjPtr);
         cfree (polygon_ptr);
         cfree (v);
         *ObjPtr = NULL;
         return;
      }
      if (id >= objId) objId = id+1;
   }

   if (fileVersion <= 16 && width <= 6) width = origWidthOfLine[width];
   fill = UpgradePenFill (fill);
   pen = UpgradePenFill (pen);

   polygon_ptr->fill = fill;
   polygon_ptr->width = width;
   polygon_ptr->pen = pen;
   polygon_ptr->curved = curved;
   polygon_ptr->dash = dash;

   polygon_ptr->vlist = v;
   polygon_ptr->svlist = NULL;

   (*ObjPtr)->x = ltx;
   (*ObjPtr)->y = lty;
   (*ObjPtr)->color = QuickFindColorIndex (color_str, &new_alloc);
   (*ObjPtr)->dirty = FALSE;
   (*ObjPtr)->id = id;
   (*ObjPtr)->rotation = rotation;
   (*ObjPtr)->locked = locked;
   (*ObjPtr)->type = OBJ_POLYGON;
   (*ObjPtr)->obbox.ltx = ltx;
   (*ObjPtr)->obbox.lty = lty;
   (*ObjPtr)->obbox.rbx = rbx;
   (*ObjPtr)->obbox.rby = rby;
   w = HALF_W(width);
   (*ObjPtr)->bbox.ltx = ltx - w;
   (*ObjPtr)->bbox.lty = lty - w;
   (*ObjPtr)->bbox.rbx = rbx + w;
   (*ObjPtr)->bbox.rby = rby + w;
   (*ObjPtr)->detail.g = polygon_ptr;
   AdjObjSplineVs (*ObjPtr);
}

void FreePolygonObj (ObjPtr)
   struct ObjRec	* ObjPtr;
{
   if (ObjPtr->detail.g->svlist != NULL) cfree (ObjPtr->detail.g->svlist);
   cfree (ObjPtr->detail.g->vlist);
   cfree (ObjPtr->detail.g);
   cfree (ObjPtr);
}
