/*
 * Author:	William Chia-Wei Cheng (william@cs.ucla.edu)
 *
 * Copyright (C) 1990, 1991, 1992, William Cheng.
 * 
 * Permission limited to the use, copy, modify, and distribute this software
 * and its documentation for any purpose is hereby granted by the Author without
 * fee, provided that the above copyright notice appear in all copies and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation, and that the name of the Author not be used
 * in advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.  The Author makes no
 * representations about the suitability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.  All other
 * rights are reserved by the Author.
 *
 * THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF
 * USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
 * OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */
#ifndef lint
static char RCSid[] =
      "@(#)$Header: /amnt/kona/tangram/u/william/X11/TGIF2/RCS/spline.c,v 2.23 1993/01/08 04:33:14 william Exp $";
#endif

#include <math.h>
#include <stdio.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include "const.h"
#include "types.h"

#include "poly.e"
#include "raster.e"
#include "rect.e"
#include "setup.e"

int	splineTol = 9;
int	splineRubberband = TRUE;

void Spline (Win, Pixel, Func, X1, Y1, X2, Y2, X3, Y3, X4, Y4)
   Window	Win;
   int		Pixel, Func;
   double	X1, Y1, X2, Y2, X3, Y3, X4, Y4;
   /* X1, Y1, X2, Y2, X3, Y3, X4, Y4 are screen offsets */
{
   double	x, y;

   x = (X2 + X3) / 2.0;
   y = (Y2 + Y3) / 2.0;
   if (fabs (X1 - x) < splineTol && fabs (Y1 - y) < splineTol)
      XDrawLine (mainDisplay, Win, drawGC, round(X1), round(Y1), round(x),
            round(y));
   else
      Spline (Win, Pixel, Func, X1, Y1, ((X1+X2)/2.0), ((Y1+Y2)/2.0),
            ((3.0*X2+X3)/4.0), ((3.0*Y2+Y3)/4.0), x, y);

   if (fabs (x - X4) < splineTol && fabs (y - Y4) < splineTol)
      XDrawLine (mainDisplay, Win, drawGC, round(x), round(y), round(X4),
            round(Y4));
   else
      Spline (Win, Pixel, Func, x, y, ((X2+3.0*X3)/4.0), ((Y2+3.0*Y3)/4.0),
            ((X3+X4)/2.0), ((Y3+Y4)/2.0), X4, Y4);
}

static XPoint	* splineVs;

int AddSplinePt(N, MaxN, X, Y)
   int		* N, * MaxN, X, Y;
{
   if (*N == *MaxN)
   {
      splineVs = (XPoint *) realloc (splineVs, (*MaxN)*2*sizeof(XPoint)+1);
      if (splineVs == NULL)
      {
         fprintf (stderr, "Can not realloc() in AddSplinePt ().\n");
         return (FALSE);
      }
      *MaxN = (*MaxN) * 2;
   }
   splineVs[*N].x = X;
   splineVs[*N].y = Y;
   (*N)++;
   return (TRUE);
}

static
void SetSplineVs (N, MaxN, X1, Y1, X2, Y2, X3, Y3, X4, Y4)
   int		* N, * MaxN;
   double	X1, Y1, X2, Y2, X3, Y3, X4, Y4;
   /* X1, Y1, X2, Y2, X3, Y3, X4, Y4 are screen offsets */
{
   double	x, y;

   x = (X2 + X3) / 2.0;
   y = (Y2 + Y3) / 2.0;
   if (fabs (X1 - x) < splineTol && fabs (Y1 - y) < splineTol)
      AddSplinePt (N, MaxN, round(x), round(y));
   else
      SetSplineVs (N, MaxN, X1, Y1, ((X1+X2)/2.0), ((Y1+Y2)/2.0),
            ((3.0*X2+X3)/4.0), ((3.0*Y2+Y3)/4.0), x, y);

   if (fabs (x - X4) < splineTol && fabs (y - Y4) < splineTol)
      AddSplinePt (N, MaxN, round(X4), round(Y4));
   else
      SetSplineVs (N, MaxN, x, y, ((X2+3.0*X3)/4.0), ((Y2+3.0*Y3)/4.0),
            ((X3+X4)/2.0), ((Y3+Y4)/2.0), X4, Y4);
}

XPoint * MakeSplinePolyVertex (N, XOff, YOff, NumVs, Vs)
   int		* N, XOff, YOff, NumVs;
   XPoint	* Vs;
{
   double	mx1, my1, mx2, my2, mx3, my3, mx4, my4, x1, y1, x2, y2;
   int		i, x_off, y_off, max_n;

   x_off = (zoomedIn ? XOff : (XOff>>zoomScale)<<zoomScale);
   y_off = (zoomedIn ? YOff : (YOff>>zoomScale)<<zoomScale);

   splineVs = NULL;

   switch (NumVs)
   {
      case 0:
      case 1:
         break;
      case 2:
         splineVs = (XPoint *) calloc (NumVs+1, sizeof (XPoint));
         if (splineVs == NULL)
         {
            fprintf (stderr, "Can not calloc() for open splines.\n");
            *N = 0;
            return (splineVs);
         }
         splineVs[0].x = ZOOMED_SIZE(Vs[0].x-x_off);
         splineVs[0].y = ZOOMED_SIZE(Vs[0].y-y_off);
         splineVs[1].x = ZOOMED_SIZE(Vs[1].x-x_off);
         splineVs[1].y = ZOOMED_SIZE(Vs[1].y-y_off);
         *N = 2;
         break;
      case 3:
         mx1 = ZOOMED_SIZE(Vs->x-x_off); my1 = ZOOMED_SIZE((Vs++)->y-y_off);
         x1 = ZOOMED_SIZE(Vs->x-x_off);  y1 = ZOOMED_SIZE((Vs++)->y-y_off);
         mx2 = (mx1+x1)/2.0;             my2 = (my1+y1)/2.0;
         mx4 = ZOOMED_SIZE(Vs->x-x_off); my4 = ZOOMED_SIZE(Vs->y-y_off);
         mx3 = (x1+mx4)/2.0;             my3 = (y1+my4)/2.0;
         max_n = 100;
         splineVs = (XPoint *) calloc (max_n+1, sizeof (XPoint));
         if (splineVs == NULL)
         {
            fprintf (stderr, "Can not calloc() for open splines.\n");
            *N = 0;
            return (splineVs);
         }
         splineVs[0].x = mx1;
         splineVs[0].y = my1;
         *N = 1;
         SetSplineVs (N, &max_n, mx1, my1, mx2, my2, mx3, my3, mx4, my4);
         break;
      default:
         mx1 = ZOOMED_SIZE(Vs->x-x_off); my1 = ZOOMED_SIZE((Vs++)->y-y_off);
         x1 = ZOOMED_SIZE(Vs->x-x_off);  y1 = ZOOMED_SIZE((Vs++)->y-y_off);
         x2 = ZOOMED_SIZE(Vs->x-x_off);  y2 = ZOOMED_SIZE((Vs++)->y-y_off);
         mx2 = (mx1+x1)/2.0;             my2 = (my1+y1)/2.0;
         mx3 = (3.0*x1+x2)/4.0;          my3 = (3.0*y1+y2)/4.0;
         mx4 = (x1+x2)/2.0;              my4 = (y1+y2)/2.0;
         max_n = 100;
         splineVs = (XPoint *) calloc (max_n+1, sizeof (XPoint));
         if (splineVs == NULL)
         {
            fprintf (stderr, "Can not calloc() for open splines.\n");
            *N = 0;
            return (splineVs);
         }
         splineVs[0].x = mx1;
         splineVs[0].y = my1;
         *N = 1;
         SetSplineVs (N, &max_n, mx1, my1, mx2, my2, mx3, my3, mx4, my4);
      
         for (i = 2; i < NumVs-2; i++, Vs++)
         {
            mx1 = mx4;                     my1 = my4;
            mx2 = (x1 + 3.0*x2) / 4.0;     my2 = (y1 + 3.0*y2) / 4.0;
            x1 = x2;                       y1 = y2;
            x2 = ZOOMED_SIZE(Vs->x-x_off); y2 = ZOOMED_SIZE(Vs->y-y_off);
            mx3 = (3.0*x1 + x2) / 4.0;     my3 = (3.0*y1 + y2) / 4.0;
            mx4 = (x1 + x2) / 2.0;         my4 = (y1 + y2) / 2.0;
            SetSplineVs (N, &max_n, mx1, my1, mx2, my2, mx3, my3, mx4, my4);
         }
         mx1 = mx4;                      my1 = my4;
         mx2 = (x1 + 3.0*x2) / 4.0;      my2 = (y1 + 3.0*y2) / 4.0;
         x1 = x2;                        y1 = y2;
         mx4 = ZOOMED_SIZE(Vs->x-x_off); my4 = ZOOMED_SIZE(Vs->y-y_off);
         mx3 = (x1 + mx4) / 2.0;         my3 = (y1 + my4) / 2.0;
         SetSplineVs (N, &max_n, mx1, my1, mx2, my2, mx3, my3, mx4, my4);
         break;
   }
   return (splineVs);
}

XPoint * MakeSplinePolygonVertex (N, XOff, YOff, NumVs, Vs)
   int		* N, XOff, YOff, NumVs;
   XPoint	* Vs;
{
   double	mx1, my1, mx2, my2, mx3, my3, mx4, my4, x1, y1, x2, y2;
   int		i, max_n, x_off, y_off;

   x_off = (zoomedIn ? XOff : (XOff>>zoomScale)<<zoomScale);
   y_off = (zoomedIn ? YOff : (YOff>>zoomScale)<<zoomScale);

   splineVs = NULL;

   if (NumVs <= 3)
   {
      splineVs = (XPoint *) calloc (5, sizeof (XPoint));
      if (splineVs == NULL)
      {
         fprintf (stderr, "Can not calloc() for open splines.\n");
         *N = 0;
         return (splineVs);
      }
      splineVs[0].x = ZOOMED_SIZE(Vs[0].x-x_off);
      splineVs[0].y = ZOOMED_SIZE(Vs[0].y-y_off);
      splineVs[1].x = ZOOMED_SIZE(Vs[1].x-x_off);
      splineVs[1].y = ZOOMED_SIZE(Vs[1].y-y_off);
      *N = 2;
      return (splineVs);
   }

   Vs[NumVs].x = Vs[1].x; Vs[NumVs].y = Vs[1].y;
   x1 = ZOOMED_SIZE(Vs->x-x_off); y1 = ZOOMED_SIZE((Vs++)->y-y_off);
   x2 = ZOOMED_SIZE(Vs->x-x_off); y2 = ZOOMED_SIZE((Vs++)->y-y_off);
   mx4 = (x1 + x2) / 2.0;         my4 = (y1 + y2) / 2.0;

   max_n = 100;
   splineVs = (XPoint *) calloc (max_n+1, sizeof (XPoint));
   if (splineVs == NULL)
   {
      fprintf (stderr, "Can not calloc() for closed splines.\n");
      *N = 0;
      return (splineVs);
   }
   splineVs[0].x = mx4;
   splineVs[0].y = my4;
   *N = 1;

   for (i = 1; i < NumVs; i++, Vs++)
   {
      mx1 = mx4;                     my1 = my4;
      mx2 = (x1+3.0*x2)/4.0;         my2 = (y1+3.0*y2)/4.0;
      x1 = x2;                       y1 = y2;
      x2 = ZOOMED_SIZE(Vs->x-x_off); y2 = ZOOMED_SIZE(Vs->y-y_off);
      mx3 = (3.0*x1+x2)/4.0;         my3 = (3.0*y1+y2)/4.0;
      mx4 = (x1+x2)/2.0;             my4 = (y1+y2)/2.0;
      SetSplineVs (N, &max_n, mx1, my1, mx2, my2, mx3, my3, mx4, my4);
   }
   return (splineVs);
}

void DumpCurvedPolyPoints (FP, NumPts, V, Indent)
   FILE			* FP;
   int			NumPts, Indent;
   register XPoint	* V;
{
   register int	j, i;
   double	x1, y1, x2, y2;
   double	mx1, my1, mx2, my2, mx3, my3, mx4, my4;

   switch (NumPts)
   {
      case 0:
      case 1:
      case 2:
         break;
      case 3:
         mx1 = V->x; my1 = (V++)->y;
         x1 = V->x; y1 = (V++)->y;
         x2 = V->x; y2 = (V++)->y;
         mx2 = (mx1 + 2.0*x1) / 3.0; my2 = (my1 + 2.0*y1) / 3.0;
         mx3 = (2.0*x1 + x2) / 3.0; my3 = (2.0*y1 + y2) / 3.0;
         for (j = 0; j < Indent; j++) fprintf (FP, " ");
         fprintf (FP, "%.2f %.2f %.2f %.2f\n", mx2, my2, mx3, my3);
         break;
      default:
         mx1 = V->x; my1 = (V++)->y;
         x1 = V->x; y1 = (V++)->y;
         x2 = V->x; y2 = (V++)->y;
         mx2 = (mx1 + 2.0*x1) / 3.0; my2 = (my1 + 2.0*y1) / 3.0;
         mx3 = (5.0*x1 + x2) / 6.0; my3 = (5.0*y1 + y2) / 6.0;
         mx4 = (x1 + x2) / 2.0; my4 = (y1 + y2) / 2.0;
         for (j = 0; j < Indent; j++) fprintf (FP, " ");
         fprintf (FP, "%.2f %.2f %.2f %.2f %.2f %.2f curveto\n",
               mx2, my2, mx3, my3, mx4, my4);
      
         for (i = 2; i < NumPts-2; i++, V++)
         {
/*          mx1 = mx4; my1 = my4; */
            mx2 = (x1 + 5.0*x2) / 6.0; my2 = (y1 + 5.0*y2) / 6.0;
            x1 = x2; y1 = y2;
#ifdef stellar
            mx3 = (5.0*x1 + V->x) / 6.0; my3 = (5.0*y1 + V->y) / 6.0;
            mx4 = (x1 + V->x) / 2.0; my4 = (y1 + V->y) / 2.0;
#else
            x2 = V->x; y2 = V->y;
            mx3 = (5.0*x1 + x2) / 6.0; my3 = (5.0*y1 + y2) / 6.0;
            mx4 = (x1 + x2) / 2.0; my4 = (y1 + y2) / 2.0;
#endif
            for (j = 0; j < Indent; j++) fprintf (FP, " ");
            fprintf (FP, "%.2f %.2f %.2f %.2f %.2f %.2f curveto\n",
                  mx2, my2, mx3, my3, mx4, my4);
#ifdef stellar
            x2 = V->x; y2 = V->y;
#endif
         }
/*       mx1 = mx4; my1 = my4; */
         mx2 = (x1 + 5.0*x2) / 6.0; my2 = (y1 + 5.0*y2) / 6.0;
         x1 = x2; y1 = y2;
         mx3 = (2.0*x1 + V->x) / 3.0; my3 = (2.0*y1 + V->y) / 3.0;
         for (j = 0; j < Indent; j++) fprintf (FP, " ");
         fprintf (FP, "%.2f %.2f %.2f %.2f\n", mx2, my2, mx3, my3);
         break;
   }
}

void DumpCurvedPolygonPoints (FP, NumPts, V, Indent)
   FILE			* FP;
   int			NumPts, Indent;
   register XPoint	* V;
{
   register int	j;
   double	mx2, my2, mx3, my3, mx4, my4, x1, y1, x2, y2;
   int		i;

   V[NumPts].x = V[1].x; V[NumPts].y = V[1].y;
   x1 = V->x;             y1 = (V++)->y;
   x2 = V->x;             y2 = (V++)->y;
   mx4 = (x1 + x2) / 2.0; my4 = (y1 + y2) / 2.0;
   for (j = 0; j < Indent; j++) fprintf (FP, " ");
   fprintf (FP, "%.2f %.2f moveto\n", mx4, my4);

   for (i = 1; i < NumPts; i++, V++)
   {
/*    mx1 = mx4;             my1 = my4; */
      mx2 = (x1+5.0*x2)/6.0; my2 = (y1+5.0*y2)/6.0;
      x1 = x2;               y1 = y2;
#ifdef stellar
      mx3 = (5.0*x1+V->x)/6.0; my3 = (5.0*y1+V->y)/6.0;
      mx4 = (x1+V->x)/2.0;     my4 = (y1+V->y)/2.0;
#else
      x2 = V->x;             y2 = V->y;
      mx3 = (5.0*x1+x2)/6.0; my3 = (5.0*y1+y2)/6.0;
      mx4 = (x1+x2)/2.0;     my4 = (y1+y2)/2.0;
#endif
      for (j = 0; j < Indent; j++) fprintf (FP, " ");
      fprintf (FP, "%.2f %.2f %.2f %.2f %.2f %.2f curveto\n",
            mx2, my2, mx3, my3, mx4, my4);
#ifdef stellar
      x2 = V->x;             y2 = V->y;
#endif
   }
}
