/*
 * This file is part of the GROMACS molecular simulation package.
 *
 * Copyright 2019- The GROMACS Authors
 * and the project initiators Erik Lindahl, Berk Hess and David van der Spoel.
 * Consult the AUTHORS/COPYING files and https://www.gromacs.org for details.
 *
 * GROMACS is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1
 * of the License, or (at your option) any later version.
 *
 * GROMACS is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with GROMACS; if not, see
 * https://www.gnu.org/licenses, or write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.
 *
 * If you want to redistribute modifications to GROMACS, please
 * consider that scientific software is very special. Version
 * control is crucial - bugs must be traceable. We will be happy to
 * consider code for inclusion in the official distribution, but
 * derived work must not be called official GROMACS. Details are found
 * in the README & COPYING files - if they are missing, get the
 * official version at https://www.gromacs.org.
 *
 * To help us fund GROMACS development, we humbly ask that you cite
 * the research papers on the package. Check out https://www.gromacs.org.
 */
/*! \libinternal \file
 * \brief
 * Declares classes to calculate density fitting forces.
 *
 * \author Christian Blau <blau@kth.se>
 * \inlibraryapi
 * \ingroup module_math
 */

#include <memory>

#include "gromacs/math/gausstransform.h"
#include "gromacs/mdspan/extensions.h"
#include "gromacs/mdspan/mdspan.h"
#include "gromacs/utility/vectypes.h"

namespace gmx
{


/*! \libinternal
 * \brief
 * Manages evaluation of density-fitting forces for particles that were spread
 * with a kernel.
 *
 * The density fitting force is the inner product of the derivative of the
 * density-density goodness-of-fit function at all density values and the
 * gradient of the spreading kernel used to simulate the density.
 *
 * For a discrete Gauss transform spreading kernel and density derivative \f$D\f$
 * it is the Gauss transform of a single coordinate, weighted by a distance vector
 * and density derivative at each lattice point
 * \f[
 *  \sum_v D_v \frac{\vec{x_v} - \vec{x}}{\sigma^2}
 *  \frac{1}{\sigma^3 \sqrt(2^3*\pi^3)} * \exp(-\frac{(x_v-x)^2}{2 \sigma^2})
 * \f]
 *
 */
class DensityFittingForce
{
public:
    /*! \brief Construct density fitting force evaluator.
     * \param[in] kernelShapeParameters the global parameters of the density spreading kernel
     */
    DensityFittingForce(const GaussianSpreadKernelParameters::Shape& kernelShapeParameters);

    ~DensityFittingForce();
    //! Copy constructor
    DensityFittingForce(const DensityFittingForce& other);
    //! Copy assignment
    DensityFittingForce& operator=(const DensityFittingForce& other);
    //! Move constructor
    DensityFittingForce(DensityFittingForce&& other) noexcept;
    //! Move assignment
    DensityFittingForce& operator=(DensityFittingForce&& other) noexcept;
    /*! \brief
     * Evaluate density-fitting force between a reference density and the
     * density generated by a kernel.
     *
     * Implements the equation in the class description.
     *
     * \param[in] localParameters   local parameters of the spreading kernel
     * \param[in] densityDerivative the spatial derivative of the similarity
     *                              measure between the reference density and
     *                              a density that was generated using the
     *                              spreading kernel
     *
     * \returns the force that increases the measure of the goodness of fit used to calculate the density derivative
     */
    RVec evaluateForce(const GaussianSpreadKernelParameters::PositionAndAmplitude& localParameters,
                       basic_mdspan<const float, dynamicExtents3D> densityDerivative);

private:
    class Impl;
    std::unique_ptr<Impl> impl_;
};

} // namespace gmx
