/*
 *  $Id: anneal_synth.c 22362 2019-08-06 14:14:08Z yeti-dn $
 *  Copyright (C) 2019 David Necas (Yeti).
 *  E-mail: yeti@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <string.h>
#include <gtk/gtk.h>
#include <libgwyddion/gwymacros.h>
#include <libgwyddion/gwymath.h>
#include <libgwyddion/gwythreads.h>
#include <libgwyddion/gwyrandgenset.h>
#include <libprocess/stats.h>
#include <libprocess/filters.h>
#include <libgwydgets/gwystock.h>
#include <libgwymodule/gwymodule-process.h>
#include <app/gwyapp.h>
#include "dimensions.h"
#include "preview.h"
#include "libgwyddion/gwyomp.h"

#define ANNEAL_SYNTH_RUN_MODES (GWY_RUN_IMMEDIATE | GWY_RUN_INTERACTIVE)

enum {
    PAGE_DIMENSIONS = 0,
    PAGE_GENERATOR  = 1,
    PAGE_NPAGES
};

enum {
    CELL_STATUS_HAVE_RNUM    = (1 << 0),
    CELL_STATUS_TRY_SWAPPING = (1 << 1),
};

/* Cannot change this without losing reproducibility again! */
enum {
    NRANDOM_GENERATORS = 24
};

typedef struct _DomainSynthControls AnnealSynthControls;

typedef struct {
    gint active_page;
    gint seed;
    gboolean randomize;
    gboolean animated;
    guint nspecies;
    guint niters;
    gint average;
    gdouble height;
    gdouble fraction;
    gdouble B_fraction;
    gdouble T_init;
    gdouble T_final;
    gdouble deltaE[3];
} AnnealSynthArgs;

struct _DomainSynthControls {
    AnnealSynthArgs *args;
    GwyDimensions *dims;
    GtkWidget *dialog;
    GtkWidget *view;
    GtkWidget *update_now;
    GtkWidget *animated;
    GtkObject *seed;
    GtkWidget *randomize;
    GtkTable *table;
    GtkWidget *three_species;
    GtkObject *niters;
    GtkObject *T_init;
    GtkObject *T_final;
    GtkObject *fraction;
    GtkObject *B_fraction;
    GtkObject *deltaE[3];
    GtkObject *average;
    GtkObject *height;
    GtkWidget *height_units;
    GtkWidget *height_init;
    GwyContainer *mydata;
    GwyDataField *surface;
    gdouble pxsize;
    gdouble zscale;
    gboolean in_update;
};

static gboolean module_register        (void);
static void     anneal_synth           (GwyContainer *data,
                                        GwyRunType run);
static void     run_noninteractive     (AnnealSynthArgs *args,
                                        const GwyDimensionArgs *dimsargs,
                                        GwyContainer *data,
                                        GwyDataField *dfield,
                                        gint oldid,
                                        GQuark quark);
static gboolean anneal_synth_dialog    (AnnealSynthArgs *args,
                                        GwyDimensionArgs *dimsargs,
                                        GwyContainer *data,
                                        GwyDataField *dfield,
                                        gint id);
static void     update_controls        (AnnealSynthControls *controls,
                                        AnnealSynthArgs *args);
static void     page_switched          (AnnealSynthControls *controls,
                                        GtkNotebookPage *page,
                                        gint pagenum);
static void     update_values          (AnnealSynthControls *controls);
static void     T_init_changed         (GtkAdjustment *adj,
                                        AnnealSynthControls *controls);
static void     T_final_changed        (GtkAdjustment *adj,
                                        AnnealSynthControls *controls);
static void     three_species_toggled  (GtkToggleButton *toggle,
                                        AnnealSynthControls *controls);
static void     deltaE_AB_changed      (GtkAdjustment *adj,
                                        AnnealSynthControls *controls);
static void     deltaE_AC_changed      (GtkAdjustment *adj,
                                        AnnealSynthControls *controls);
static void     deltaE_BC_changed      (GtkAdjustment *adj,
                                        AnnealSynthControls *controls);
static void     update_sensitivity     (AnnealSynthControls *controls);
static void     height_init_clicked    (AnnealSynthControls *controls);
static void     anneal_synth_invalidate(AnnealSynthControls *controls);
static void     preview                (AnnealSynthControls *controls);
static gboolean anneal_synth_do        (AnnealSynthArgs *args,
                                        GwyDataField *dfield,
                                        GwyRandGenSet *rngset,
                                        gdouble preview_time);
static void     init_field_randomly    (GwyDataField *dfield,
                                        guint32 seed);
static void     anneal_synth_load_args (GwyContainer *container,
                                        AnnealSynthArgs *args,
                                        GwyDimensionArgs *dimsargs);
static void     anneal_synth_save_args (GwyContainer *container,
                                        const AnnealSynthArgs *args,
                                        const GwyDimensionArgs *dimsargs);

#define GWY_SYNTH_CONTROLS AnnealSynthControls
#define GWY_SYNTH_INVALIDATE(controls) anneal_synth_invalidate(controls)

#include "synth.h"

static const AnnealSynthArgs anneal_synth_defaults = {
    PAGE_DIMENSIONS,
    42, TRUE, TRUE,
    2, 5000, 1,
    1.0, 50.0, 100/3.0,
    1.25, 0.7,
    { 1.0, 1.0, 1.0 },
};

static const GwyDimensionArgs dims_defaults = GWY_DIMENSION_ARGS_INIT;

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Generates images by annealing a lattice gas model."),
    "Yeti <yeti@gwyddion.net>",
    "1.0",
    "David Nečas (Yeti)",
    "2019",
};

GWY_MODULE_QUERY2(module_info, anneal_synth)

static gboolean
module_register(void)
{
    gwy_process_func_register("anneal_synth",
                              (GwyProcessFunc)&anneal_synth,
                              N_("/S_ynthetic/_Anneal..."),
                              GWY_STOCK_SYNTHETIC_ANNEAL,
                              ANNEAL_SYNTH_RUN_MODES,
                              0,
                              N_("Generate image by annealing a lattice gas"));

    return TRUE;
}

static void
anneal_synth(GwyContainer *data, GwyRunType run)
{
    AnnealSynthArgs args;
    GwyDimensionArgs dimsargs;
    GwyDataField *dfield;
    GQuark quark;
    gint id;

    g_return_if_fail(run & ANNEAL_SYNTH_RUN_MODES);
    anneal_synth_load_args(gwy_app_settings_get(), &args, &dimsargs);
    gwy_app_data_browser_get_current(GWY_APP_DATA_FIELD, &dfield,
                                     GWY_APP_DATA_FIELD_ID, &id,
                                     GWY_APP_DATA_FIELD_KEY, &quark,
                                     0);

    if (run == GWY_RUN_IMMEDIATE
        || anneal_synth_dialog(&args, &dimsargs, data, dfield, id)) {
        run_noninteractive(&args, &dimsargs, data, dfield, id, quark);
    }

    gwy_dimensions_free_args(&dimsargs);
}

static void
run_noninteractive(AnnealSynthArgs *args,
                   const GwyDimensionArgs *dimsargs,
                   GwyContainer *data,
                   GwyDataField *dfield,
                   gint oldid,
                   GQuark quark)
{
    GwySIUnit *siunit;
    GwyRandGenSet *rngset;
    gboolean replace = dimsargs->replace && dfield;
    gboolean add = dimsargs->add && dfield;
    gint newid = -1;
    gboolean ok, wait_enabled;

    if (args->randomize)
        args->seed = g_random_int() & 0x7fffffff;

    rngset = gwy_rand_gen_set_new(NRANDOM_GENERATORS);
    gwy_rand_gen_set_init(rngset, args->seed);

    if (add) {
        dfield = gwy_data_field_duplicate(dfield);
    }
    else if (replace) {
        dfield = gwy_data_field_new_alike(dfield, TRUE);
        init_field_randomly(dfield, args->seed);
    }
    else {
        gdouble mag = pow10(dimsargs->xypow10) * dimsargs->measure;
        dfield = gwy_data_field_new(dimsargs->xres, dimsargs->yres,
                                    mag*dimsargs->xres, mag*dimsargs->yres,
                                    TRUE);
        init_field_randomly(dfield, args->seed);

        siunit = gwy_data_field_get_si_unit_xy(dfield);
        gwy_si_unit_set_from_string(siunit, dimsargs->xyunits);

        siunit = gwy_data_field_get_si_unit_z(dfield);
        gwy_si_unit_set_from_string(siunit, dimsargs->zunits);
    }

    wait_enabled = gwy_app_wait_get_enabled();
    if (wait_enabled && args->animated) {
        set_up_wait_data_field_preview(dfield, data, oldid);
    }
    gwy_app_wait_start(gwy_app_find_window_for_channel(data, oldid),
                       _("Initializing..."));
    ok = anneal_synth_do(args, dfield, rngset, wait_enabled ? 1.25 : 0.0);
    gwy_app_wait_finish();

    gwy_rand_gen_set_free(rngset);

    if (!ok) {
        g_object_unref(dfield);
        return;
    }

    gwy_data_field_renormalize(dfield,
                               pow10(dimsargs->zpow10) * args->height, 0.0);

    if (replace) {
        gwy_app_undo_qcheckpointv(data, 1, &quark);
        gwy_container_set_object(data, gwy_app_get_data_key_for_id(oldid),
                                 dfield);
        gwy_app_channel_log_add_proc(data, oldid, oldid);
        g_object_unref(dfield);
        return;
    }

    if (data) {
        newid = gwy_app_data_browser_add_data_field(dfield, data, TRUE);
    }
    else {
        data = gwy_container_new();
        newid = 0;
        gwy_container_set_object(data, gwy_app_get_data_key_for_id(newid),
                                 dfield);
        gwy_app_data_browser_add(data);
        gwy_app_data_browser_reset_visibility(data,
                                              GWY_VISIBILITY_RESET_SHOW_ALL);
        g_object_unref(data);
    }

    if (oldid != -1) {
        gwy_app_sync_data_items(data, data, oldid, newid, FALSE,
                                GWY_DATA_ITEM_GRADIENT,
                                0);
    }
    gwy_app_set_data_field_title(data, newid, _("Generated"));
    gwy_app_channel_log_add_proc(data, add ? oldid : -1, newid);
    g_object_unref(dfield);
}

static gboolean
anneal_synth_dialog(AnnealSynthArgs *args,
                    GwyDimensionArgs *dimsargs,
                    GwyContainer *data,
                    GwyDataField *dfield_template,
                    gint id)
{
    GtkWidget *dialog, *table, *vbox, *hbox, *notebook;
    AnnealSynthControls controls;
    GwyDataField *dfield;
    gboolean finished;
    gint row, response;

    gwy_clear(&controls, 1);
    controls.in_update = TRUE;
    controls.args = args;
    controls.pxsize = 1.0;
    dialog = gtk_dialog_new_with_buttons(_("Anneal"),
                                         NULL, 0,
                                         _("_Reset"), RESPONSE_RESET,
                                         GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                                         GTK_STOCK_OK, GTK_RESPONSE_OK,
                                         NULL);
    gtk_dialog_set_default_response(GTK_DIALOG(dialog), GTK_RESPONSE_OK);
    gwy_help_add_to_proc_dialog(GTK_DIALOG(dialog), GWY_HELP_DEFAULT);
    controls.dialog = dialog;

    hbox = gtk_hbox_new(FALSE, 2);
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), hbox,
                       FALSE, FALSE, 4);

    vbox = gtk_vbox_new(FALSE, 4);
    gtk_box_pack_start(GTK_BOX(hbox), vbox, FALSE, FALSE, 4);

    controls.mydata = gwy_container_new();
    dfield = gwy_data_field_new(PREVIEW_SIZE, PREVIEW_SIZE,
                                dimsargs->measure*PREVIEW_SIZE,
                                dimsargs->measure*PREVIEW_SIZE,
                                TRUE);
    gwy_container_set_object_by_name(controls.mydata, "/0/data", dfield);

    if (dfield_template) {
        gwy_app_sync_data_items(data, controls.mydata, id, 0, FALSE,
                                GWY_DATA_ITEM_PALETTE,
                                0);
        controls.surface = gwy_synth_surface_for_preview(dfield_template,
                                                         PREVIEW_SIZE);
        controls.zscale = 3.0*gwy_data_field_get_rms(dfield_template);
    }
    controls.view = create_preview(controls.mydata, 0, PREVIEW_SIZE,
                                   FALSE);
    gtk_box_pack_start(GTK_BOX(vbox), controls.view, FALSE, FALSE, 0);

    gtk_box_pack_start(GTK_BOX(vbox),
                       gwy_synth_progressive_preview_new(&controls,
                                                         &controls.update_now,
                                                         &controls.animated,
                                                         &args->animated),
                       FALSE, FALSE, 0);
    g_signal_connect_swapped(controls.update_now, "clicked",
                             G_CALLBACK(preview), &controls);

    gtk_box_pack_start(GTK_BOX(vbox),
                       gwy_synth_random_seed_new(&controls,
                                                 &controls.seed, &args->seed),
                       FALSE, FALSE, 0);

    controls.randomize = gwy_synth_randomize_new(&args->randomize);
    gtk_box_pack_start(GTK_BOX(vbox), controls.randomize, FALSE, FALSE, 0);

    notebook = gtk_notebook_new();
    gtk_box_pack_start(GTK_BOX(hbox), notebook, TRUE, TRUE, 4);
    g_signal_connect_swapped(notebook, "switch-page",
                             G_CALLBACK(page_switched), &controls);

    controls.dims = gwy_dimensions_new(dimsargs, dfield_template);
    gtk_notebook_append_page(GTK_NOTEBOOK(notebook),
                             gwy_dimensions_get_widget(controls.dims),
                             gtk_label_new(_("Dimensions")));

    table = gtk_table_new(14 + (dfield_template ? 1 : 0), 3, FALSE);
    controls.table = GTK_TABLE(table);
    gtk_table_set_row_spacings(GTK_TABLE(table), 2);
    gtk_table_set_col_spacings(GTK_TABLE(table), 6);
    gtk_container_set_border_width(GTK_CONTAINER(table), 4);
    gtk_notebook_append_page(GTK_NOTEBOOK(notebook), table,
                             gtk_label_new(_("Generator")));
    row = 0;

    gtk_table_attach(GTK_TABLE(table),
                     gwy_label_new_header(_("Simulation Parameters")),
                     0, 2, row, row+1, GTK_FILL, 0, 0, 0);
    row++;

    controls.niters = gtk_adjustment_new(args->niters, 1, 1000000, 1, 10, 0);
    g_object_set_data(G_OBJECT(controls.niters), "target", &args->niters);
    gwy_table_attach_adjbar(table, row++, _("_Number of iterations:"), NULL,
                            GTK_OBJECT(controls.niters), GWY_HSCALE_LOG);
    g_signal_connect_swapped(controls.niters, "value-changed",
                             G_CALLBACK(gwy_synth_int_changed), &controls);

    controls.T_init = gtk_adjustment_new(args->T_init,
                                         0.001, 2.0, 0.001, 0.1, 0);
    gwy_table_attach_adjbar(table, row++, _("_Initial temperature:"), NULL,
                            GTK_OBJECT(controls.T_init), GWY_HSCALE_SQRT);
    g_signal_connect(controls.T_init, "value-changed",
                     G_CALLBACK(T_init_changed), &controls);

    controls.T_final = gtk_adjustment_new(args->T_final,
                                          0.001, 2.0, 0.001, 0.1, 0);
    gwy_table_attach_adjbar(table, row++, _("Final _temperature:"), NULL,
                            GTK_OBJECT(controls.T_final), GWY_HSCALE_SQRT);
    g_signal_connect(controls.T_final, "value-changed",
                     G_CALLBACK(T_final_changed), &controls);

    controls.fraction = gtk_adjustment_new(args->fraction,
                                           0.01, 99.99, 0.01, 10.0, 0);
    g_object_set_data(G_OBJECT(controls.fraction), "target", &args->fraction);
    gwy_table_attach_adjbar(table, row++, _("Component _fraction:"), "%",
                            GTK_OBJECT(controls.fraction), GWY_HSCALE_LINEAR);
    g_signal_connect_swapped(controls.fraction, "value-changed",
                             G_CALLBACK(gwy_synth_double_changed), &controls);

    gtk_table_set_row_spacing(GTK_TABLE(table), row-1, 8);
    gtk_table_attach(GTK_TABLE(table),
                     gwy_label_new_header(_("Three Component Model")),
                     0, 2, row, row+1, GTK_FILL, 0, 0, 0);
    row++;

    controls.three_species
        = gtk_check_button_new_with_mnemonic(_("Enable three components"));
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(controls.three_species),
                                 args->nspecies == 3);
    gtk_table_attach(GTK_TABLE(table), controls.three_species,
                     0, 2, row, row+1, GTK_FILL, 0, 0, 0);
    g_signal_connect(controls.three_species, "toggled",
                     G_CALLBACK(three_species_toggled), &controls);
    row++;

    controls.B_fraction = gtk_adjustment_new(args->B_fraction,
                                             0.01, 99.99, 0.01, 10.0, 0);
    g_object_set_data(G_OBJECT(controls.B_fraction), "target",
                      &args->B_fraction);
    gwy_table_attach_adjbar(table, row++, _("F_raction of B:"), "%",
                            GTK_OBJECT(controls.B_fraction), GWY_HSCALE_LINEAR);
    g_signal_connect_swapped(controls.B_fraction, "value-changed",
                             G_CALLBACK(gwy_synth_double_changed), &controls);

    controls.deltaE[0] = gtk_adjustment_new(args->deltaE[0],
                                            0.0, 1.0, 0.001, 0.1, 0);
    gwy_table_attach_adjbar(table, row++, _("Mixing energy AB:"), NULL,
                            GTK_OBJECT(controls.deltaE[0]), GWY_HSCALE_LINEAR);
    g_signal_connect(controls.deltaE[0], "value-changed",
                     G_CALLBACK(deltaE_AB_changed), &controls);

    controls.deltaE[1] = gtk_adjustment_new(args->deltaE[1],
                                            0.0, 1.0, 0.001, 0.1, 0);
    gwy_table_attach_adjbar(table, row++, _("Mixing energy AC:"), NULL,
                            GTK_OBJECT(controls.deltaE[1]), GWY_HSCALE_LINEAR);
    g_signal_connect(controls.deltaE[1], "value-changed",
                     G_CALLBACK(deltaE_AC_changed), &controls);

    controls.deltaE[2] = gtk_adjustment_new(args->deltaE[2],
                                            0.0, 1.0, 0.001, 0.1, 0);
    gwy_table_attach_adjbar(table, row++, _("Mixing energy BC:"), NULL,
                            GTK_OBJECT(controls.deltaE[2]), GWY_HSCALE_LINEAR);
    g_signal_connect(controls.deltaE[2], "value-changed",
                     G_CALLBACK(deltaE_BC_changed), &controls);

    gtk_table_set_row_spacing(GTK_TABLE(table), row-1, 8);
    gtk_table_attach(GTK_TABLE(table),
                     gwy_label_new_header(_("Output Options")),
                     0, 3, row, row+1, GTK_FILL, 0, 0, 0);
    row++;

    row = gwy_synth_attach_height(&controls, row,
                                  &controls.height, &args->height,
                                  _("_Height:"), NULL, &controls.height_units);

    controls.average = gtk_adjustment_new(args->average,
                                          1.0, 10000.0, 1.0, 10.0, 0);
    g_object_set_data(G_OBJECT(controls.average), "target", &args->average);
    /* TRANSLATORS: Average is verb here, `perform averaging'. */
    gwy_table_attach_adjbar(table, row++, _("_Average iterations:"), NULL,
                            GTK_OBJECT(controls.average), GWY_HSCALE_SQRT);
    g_signal_connect_swapped(controls.average, "value-changed",
                             G_CALLBACK(gwy_synth_int_changed), &controls);

    if (dfield_template) {
        controls.height_init
            = gtk_button_new_with_mnemonic(_("_Like Current Image"));
        g_signal_connect_swapped(controls.height_init, "clicked",
                                 G_CALLBACK(height_init_clicked), &controls);
        gtk_table_attach(GTK_TABLE(table), controls.height_init,
                         0, 2, row, row+1, GTK_FILL, 0, 0, 0);
        row++;
    }

    gtk_widget_show_all(dialog);
    controls.in_update = FALSE;
    /* Must be done when widgets are shown, see GtkNotebook docs */
    gtk_notebook_set_current_page(GTK_NOTEBOOK(notebook), args->active_page);
    update_values(&controls);
    update_sensitivity(&controls);

    finished = FALSE;
    while (!finished) {
        response = gtk_dialog_run(GTK_DIALOG(dialog));
        switch (response) {
            case GTK_RESPONSE_CANCEL:
            case GTK_RESPONSE_DELETE_EVENT:
            case GTK_RESPONSE_OK:
            gtk_widget_destroy(dialog);
            case GTK_RESPONSE_NONE:
            finished = TRUE;
            break;

            case RESPONSE_RESET:
            {
                gint temp2 = args->active_page;
                *args = anneal_synth_defaults;
                args->active_page = temp2;
            }
            controls.in_update = TRUE;
            update_controls(&controls, args);
            controls.in_update = FALSE;
            break;

            default:
            g_assert_not_reached();
            break;
        }
    }

    anneal_synth_save_args(gwy_app_settings_get(), args, dimsargs);

    g_object_unref(controls.mydata);
    GWY_OBJECT_UNREF(controls.surface);
    gwy_dimensions_free(controls.dims);

    return response == GTK_RESPONSE_OK;
}

static void
update_controls(AnnealSynthControls *controls,
                AnnealSynthArgs *args)
{
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->seed), args->seed);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(controls->randomize),
                                 args->randomize);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(controls->animated),
                                 args->animated);
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->niters), args->niters);
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->T_final), args->T_final);
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->T_init), args->T_init);
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->fraction),
                             args->fraction);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(controls->three_species),
                                 args->nspecies == 3);
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->B_fraction),
                             args->B_fraction);
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->deltaE[0]),
                             args->deltaE[0]);
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->deltaE[1]),
                             args->deltaE[1]);
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->deltaE[2]),
                             args->deltaE[2]);
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->height), args->height);
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->average), args->average);
}

/* Ensure the maximum of the three deltaE values is always 1.0. */
static gboolean
fix_deltaE(AnnealSynthArgs *args, gint victim)
{
    gdouble s;

    s = fmax(fmax(args->deltaE[0], args->deltaE[1]), args->deltaE[2]);
    if (s == 1.0)
        return FALSE;

    if (!(s > 0.0)) {
        args->deltaE[victim] = 1.0;
        return TRUE;
    }

    args->deltaE[0] /= s;
    args->deltaE[1] /= s;
    args->deltaE[2] /= s;

    return TRUE;
}

static void
page_switched(AnnealSynthControls *controls,
              G_GNUC_UNUSED GtkNotebookPage *page,
              gint pagenum)
{
    if (controls->in_update)
        return;

    controls->args->active_page = pagenum;
    if (pagenum == PAGE_GENERATOR)
        update_values(controls);
}

static void
update_values(AnnealSynthControls *controls)
{
    GwyDimensions *dims = controls->dims;

    controls->pxsize = dims->args->measure * pow10(dims->args->xypow10);
    if (controls->height_units)
        gtk_label_set_markup(GTK_LABEL(controls->height_units),
                             dims->zvf->units);
}

static void
height_init_clicked(AnnealSynthControls *controls)
{
    gdouble mag = pow10(controls->dims->args->zpow10);
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->height),
                             controls->zscale/mag);
}

static void
anneal_synth_invalidate(G_GNUC_UNUSED AnnealSynthControls *controls)
{
}

static void
T_init_changed(GtkAdjustment *adj, AnnealSynthControls *controls)
{
    AnnealSynthArgs *args = controls->args;

    args->T_init = gtk_adjustment_get_value(adj);
    if (controls->in_update)
        return;

    if (args->T_final > args->T_init) {
        controls->in_update = TRUE;
        gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->T_final),
                                 args->T_init);
        controls->in_update = FALSE;
    }
}

static void
T_final_changed(GtkAdjustment *adj, AnnealSynthControls *controls)
{
    AnnealSynthArgs *args = controls->args;

    args->T_final = gtk_adjustment_get_value(adj);
    if (controls->in_update)
        return;

    if (args->T_final > args->T_init) {
        controls->in_update = TRUE;
        gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->T_init),
                                 args->T_final);
        controls->in_update = FALSE;
    }
}

static void
three_species_toggled(GtkToggleButton *toggle, AnnealSynthControls *controls)
{
    AnnealSynthArgs *args = controls->args;

    args->nspecies = (gtk_toggle_button_get_active(toggle) ? 3 : 2);
    update_sensitivity(controls);
}

static void
set_deltaE_adjustments(AnnealSynthControls *controls)
{
    AnnealSynthArgs *args = controls->args;
    guint i;

    controls->in_update = TRUE;
    for (i = 0; i < 3; i++) {
        gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->deltaE[i]),
                                 args->deltaE[i]);
    }
    controls->in_update = FALSE;
}

static void
deltaE_AB_changed(GtkAdjustment *adj, AnnealSynthControls *controls)
{
    AnnealSynthArgs *args = controls->args;

    args->deltaE[0] = gtk_adjustment_get_value(adj);
    if (controls->in_update)
        return;

    if (fix_deltaE(args, 2))
        set_deltaE_adjustments(controls);
}

static void
deltaE_AC_changed(GtkAdjustment *adj, AnnealSynthControls *controls)
{
    AnnealSynthArgs *args = controls->args;

    args->deltaE[1] = gtk_adjustment_get_value(adj);
    if (controls->in_update)
        return;

    if (fix_deltaE(args, 0))
        set_deltaE_adjustments(controls);
}

static void
deltaE_BC_changed(GtkAdjustment *adj, AnnealSynthControls *controls)
{
    AnnealSynthArgs *args = controls->args;

    args->deltaE[2] = gtk_adjustment_get_value(adj);
    if (controls->in_update)
        return;

    if (fix_deltaE(args, 1))
        set_deltaE_adjustments(controls);
}

static void
update_sensitivity(AnnealSynthControls *controls)
{
    gboolean sens = (controls->args->nspecies == 3);

    gwy_table_hscale_set_sensitive(controls->deltaE[0], sens);
    gwy_table_hscale_set_sensitive(controls->deltaE[1], sens);
    gwy_table_hscale_set_sensitive(controls->deltaE[2], sens);
    gwy_table_hscale_set_sensitive(controls->B_fraction, sens);
}

static void
preview(AnnealSynthControls *controls)
{
    AnnealSynthArgs *args = controls->args;
    GwyDataField *dfield;
    GwyRandGenSet *rngset;

    rngset = gwy_rand_gen_set_new(NRANDOM_GENERATORS);
    gwy_rand_gen_set_init(rngset, args->seed);

    dfield = GWY_DATA_FIELD(gwy_container_get_object_by_name(controls->mydata,
                                                             "/0/data"));

    if (controls->dims->args->add && controls->surface)
        gwy_data_field_copy(controls->surface, dfield, FALSE);
    else
        init_field_randomly(dfield, args->seed);

    gwy_app_wait_start(GTK_WINDOW(controls->dialog), _("Initializing..."));
    anneal_synth_do(args, dfield, rngset, 1.25);
    gwy_app_wait_finish();

    gwy_rand_gen_set_free(rngset);
}

/* We explicitly partition the image into NRANDOM_GENERATORS pieces, which are
 * independent on the number of threads and never changes.  Then each thread
 * takes a subset of the pieces and generates deterministically random numbers
 * for each of them. */
static void
replenish_random_numbers(guint32 *random_numbers,
                         guint *cell_status,
                         gint n,
                         GwyRandGenSet *rngset)
{
#ifdef _OPENMP
#pragma omp parallel if (gwy_threads_are_enabled()) default(none) \
            shared(random_numbers,cell_status,n,rngset)
#endif
    {
        guint irfrom = gwy_omp_chunk_start(NRANDOM_GENERATORS);
        guint irto = gwy_omp_chunk_end(NRANDOM_GENERATORS);
        guint ir, ifrom, ito, i, have_rbits, cs;
        guint32 rbits;
        GRand *rng;

        for (ir = irfrom; ir < irto; ir++) {
            rng = gwy_rand_gen_set_rng(rngset, ir);
            ifrom = ir*n/NRANDOM_GENERATORS;
            ito = (ir + 1)*n/NRANDOM_GENERATORS;
            have_rbits = 0;
            for (i = ifrom; i < ito; i++) {
                cs = cell_status[i];
                if (!(cs & CELL_STATUS_HAVE_RNUM)) {
                    random_numbers[i] = g_rand_int(rng);
                    cs |= CELL_STATUS_HAVE_RNUM;
                }
                if (!have_rbits) {
                    rbits = g_rand_int(rng);
                    have_rbits = 32;
                }
                /* Probability of choosing a cell is 1/4. */
                if (rbits & 0x3)
                    cs &= ~CELL_STATUS_TRY_SWAPPING;
                else
                    cs |= CELL_STATUS_TRY_SWAPPING;
                rbits >>= 2;
                have_rbits -= 2;
                cell_status[i] = cs;
            }
        }
    }
}

static void
update_exp_table2(gdouble delta, guint32 *exp_table, guint n)
{
    gint diff;

    exp_table[0] = G_MAXUINT32;
    for (diff = 1; diff < n; diff++) {
        exp_table[diff] = (guint)floor(G_MAXUINT32*exp(-delta*diff) + 0.1);
    }
}

/* The table hold probabilities for the case when first cell < second cell,
 * i.e. combinations AB, AC and BC.  The reverse combinations must be
 * obtained by inverting the differences. */
static void
update_exp_table3(const gdouble *deltaE, gdouble invT, guint32 *exp_table)
{
    /* dAB is is the change of energy when swapping A and B between mixed and
     * separated. */
    gdouble dAB = deltaE[0], dAC = deltaE[1], dBC = deltaE[2];
    gint dnA, dnB;

    /* Calculate swap probabilities for all combinations of changes dnA and dnB
     * of neighbours counts of the two swapped cells (change dnC is given by
     * the other two because the number of neighbours is fixed). */

    /* Cells to swap are A and B. */
    for (dnA = -3; dnA <= 3; dnA++) {
        for (dnB = -3; dnB <= 3; dnB++) {
            gdouble dE = dnA*(dBC - dAC - dAB) + dnB*(dBC - dAC + dAB) + 2*dAC;
            guint i = (dnA + 3)*7 + (dnB + 3);

            /* Always swap when the energy change is negative; otherwise
             * calculate probability. */
            if (dE <= 1e-9)
                exp_table[i] = G_MAXUINT32;
            else
                exp_table[i] = (guint)floor(G_MAXUINT32*exp(-dE*invT) + 0.1);
        }
    }

    /* Cells to swap are A and C. */
    exp_table += 7*7;
    for (dnA = -3; dnA <= 3; dnA++) {
        for (dnB = -3; dnB <= 3; dnB++) {
            gdouble dE = 2*dAC*(1 - dnA) + dnB*(dAB - dAC - dBC);
            guint i = (dnA + 3)*7 + (dnB + 3);

            /* Always swap when the energy change is negative; otherwise
             * calculate probability. */
            if (dE <= 1e-9)
                exp_table[i] = G_MAXUINT32;
            else
                exp_table[i] = (guint)floor(G_MAXUINT32*exp(-dE*invT) + 0.1);
        }
    }

    /* Cells to swap are B and C. */
    exp_table += 7*7;
    for (dnA = -3; dnA <= 3; dnA++) {
        for (dnB = -3; dnB <= 3; dnB++) {
            gdouble dE = 2*dBC*(1 - dnB) + dnA*(dAB - dAC - dBC);
            guint i = (dnA + 3)*7 + (dnB + 3);

            /* Always swap when the energy change is negative; otherwise
             * calculate probability. */
            if (dE <= 1e-9)
                exp_table[i] = G_MAXUINT32;
            else
                exp_table[i] = (guint)floor(G_MAXUINT32*exp(-dE*invT) + 0.1);
        }
    }
}

static inline guint
count_neighbours(const guint *domain, guint xres, guint yres,
                 guint i, guint j)
{
    guint idx = i*xres + j;
    guint idx_back = i ? idx - xres : idx + xres*(yres - 1);
    guint idx_forw = i < yres-1 ? idx + xres : idx - xres*(yres - 1);
    guint idx_left = j ? idx - 1 : idx + xres-1;
    guint idx_right = j < xres-1 ? idx + 1 : idx - (xres-1);

    return domain[idx_back] + domain[idx_left]
           + domain[idx_right] + domain[idx_forw];
}

/* Return TRUE if the random value was consumed. */
static guint
maybe_swap2(guint *domain, guint xres, guint yres,
            guint i1, guint j1, gboolean vertical,
            const guint32 *exp_table, guint32 random_value)
{
    guint i2 = i1, j2 = j1, idx1 = i1*xres + j1, idx2;
    guint a1, a2, neigh1, neigh2;
    gint Ef_before, Ef_after;

    if (vertical)
        i2 = i2 < yres-1 ? i2+1 : 0;
    else
        j2 = j1 < xres-1 ? j2+1 : 0;

    idx2 = i2*xres + j2;

    /* When the cells are the same swapping is no-op.  Quit early. */
    a1 = domain[idx1];
    a2 = domain[idx2];
    if (a1 == a2)
        return FALSE;

    neigh1 = count_neighbours(domain, xres, yres, i1, j1);
    neigh2 = count_neighbours(domain, xres, yres, i2, j2);

    /* Formation energy before. */
    Ef_before = (a1 ? 4 - neigh1 : neigh1) + (a2 ? 4 - neigh2 : neigh2);

    /* Formation energy after.  The cells are different so counting in the
     * other cell's position we count the cell itself as the same, but in
     * fact the cells will be swapped so the cell in the neighbour's position
     * will be always different.  We have to add 1 to each term. */
    Ef_after = (a2 ? 4 - neigh1 : neigh1) + (a1 ? 4 - neigh2 : neigh2) + 2;

    /* There are many Markov processes that have produce the same required
     * stationary probabilities 1/(1 + exp(-Δ)) and 1/(1 + exp(Δ))
     * corresponding to thermal equlibrium.  Of them we want the one that
     * changes states most frequently (assuming it means fastest convergence).
     * This is the same as used in simulated annealing: we always switch from
     * higher energy to lower, and we switch from lower energy to higher with
     * probability exp(-Δ). */
    if (Ef_after < Ef_before) {
        domain[idx1] = a2;
        domain[idx2] = a1;
        return FALSE;
    }

    if (random_value > exp_table[Ef_after - Ef_before])
        return TRUE;

    domain[idx1] = a2;
    domain[idx2] = a1;
    return TRUE;
}

static inline void
count_neighbours3(const guint *domain, guint xres, guint yres,
                  guint i, guint j,
                  gint *nA, gint *nB)
{
    guint idx = i*xres + j;
    guint idx_back = i ? idx - xres : idx + xres*(yres - 1);
    guint idx_forw = i < yres-1 ? idx + xres : idx - xres*(yres - 1);
    guint idx_left = j ? idx - 1 : idx + xres-1;
    guint idx_right = j < xres-1 ? idx + 1 : idx - (xres-1);

    *nA = ((domain[idx_back] == 0)
           + (domain[idx_left] == 0)
           + (domain[idx_right] == 0)
           + (domain[idx_forw] == 0));
    *nB = ((domain[idx_back] == 1)
           + (domain[idx_left] == 1)
           + (domain[idx_right] == 1)
           + (domain[idx_forw] == 1));
}

/* Return TRUE if the random value was consumed. */
static gboolean
maybe_swap3(guint *domain, guint xres, guint yres,
            guint i1, guint j1, gboolean vertical,
            const guint32 *exp_table, guint32 random_value)
{
    guint i2 = i1, j2 = j1, idx1 = i1*xres + j1, idx2, a1, a2, p, tidx;
    gint nA1, nB1, nA2, nB2, dnA, dnB;

    if (vertical)
        i2 = i2 < yres-1 ? i2+1 : 0;
    else
        j2 = j1 < xres-1 ? j2+1 : 0;

    idx2 = i2*xres + j2;

    /* When the cells are the same swapping is no-op.  Quit early. */
    a1 = domain[idx1];
    a2 = domain[idx2];
    if (a1 == a2)
        return FALSE;

    /* For correct exp_table() utilisation we need a1 < a2. */
    if (a1 > a2) {
        GWY_SWAP(guint, a1, a2);
        GWY_SWAP(guint, i1, i2);
        GWY_SWAP(guint, j1, j2);
        GWY_SWAP(guint, idx1, idx2);
    }

    count_neighbours3(domain, xres, yres, i1, j1, &nA1, &nB1);
    count_neighbours3(domain, xres, yres, i2, j2, &nA2, &nB2);
    dnA = nA2 - nA1;
    dnB = nB2 - nB1;

    if (dnA == 0 && dnB == 0)
        return FALSE;

    /* Choose the right probability table for the cell combination.  It is
     * the probability of swapping, when random_number > p corresponds we do
     * NOT swap the cells. */
    tidx = (dnA + 3)*7 + (dnB + 3);
    /* a1 + a2 is 1 for AB, 2 for AC, 3 for BC. */
    p = exp_table[tidx + 7*7*(a1 + a2 - 1)];

    /* Do not consume the random number when p = 1. */
    if (p == G_MAXUINT32) {
        domain[idx1] = a2;
        domain[idx2] = a1;
        return FALSE;
    }

    if (random_value > p)
        return TRUE;

    domain[idx1] = a2;
    domain[idx2] = a1;
    return TRUE;
}

static void
process_sublattice_vertical2(guint *domain, guint xres, guint yres,
                             guint sublattice, guint32 *exp_table,
                             const guint32 *random_numbers,
                             guint *cell_status)
{
    guint i;

#ifdef _OPENMP
#pragma omp parallel for if (gwy_threads_are_enabled()) default(none) \
            shared(domain,random_numbers,cell_status,exp_table,sublattice,xres,yres) \
            private(i)
#endif
    for (i = 0; i < yres/2; i++) {
        guint hoff = (i + sublattice) % 2, voff = (sublattice % 4)/2;
        guint j, k = (xres/2)*i;

        for (j = 0; j < xres/2; j++, k++) {
            if ((cell_status[k] & CELL_STATUS_TRY_SWAPPING)
                && maybe_swap2(domain, xres, yres,
                               2*i + voff, 2*j + hoff, TRUE,
                               exp_table, random_numbers[k]))
                cell_status[k] &= ~CELL_STATUS_HAVE_RNUM;
        }
    }
}

static void
process_sublattice_horizontal2(guint *domain, guint xres, guint yres,
                               guint sublattice, guint32 *exp_table,
                               const guint32 *random_numbers,
                               guint *cell_status)
{
    guint i;

#ifdef _OPENMP
#pragma omp parallel for if (gwy_threads_are_enabled()) default(none) \
            shared(domain,random_numbers,cell_status,exp_table,sublattice,xres,yres) \
            private(i)
#endif
    for (i = 0; i < yres/2; i++) {
        guint hoff = (sublattice % 4)/2;
        guint j, k = (xres/2)*i;

        for (j = 0; j < xres/2; j++, k++) {
            guint voff = (j + sublattice) % 2;

            if ((cell_status[k] & CELL_STATUS_TRY_SWAPPING)
                && maybe_swap2(domain, xres, yres,
                               2*i + voff, 2*j + hoff, FALSE,
                               exp_table, random_numbers[k]))
                cell_status[k] &= ~CELL_STATUS_HAVE_RNUM;
        }
    }
}

static void
process_sublattice_vertical3(guint *domain, guint xres, guint yres,
                             guint sublattice, guint32 *exp_table,
                             const guint32 *random_numbers,
                             guint *cell_status)
{
    guint i;

#ifdef _OPENMP
#pragma omp parallel for if (gwy_threads_are_enabled()) default(none) \
            shared(domain,random_numbers,cell_status,exp_table,sublattice,xres,yres) \
            private(i)
#endif
    for (i = 0; i < yres/2; i++) {
        guint hoff = (i + sublattice) % 2, voff = (sublattice % 4)/2;
        guint j, k = (xres/2)*i;

        for (j = 0; j < xres/2; j++, k++) {
            if ((cell_status[k] & CELL_STATUS_TRY_SWAPPING)
                && maybe_swap3(domain, xres, yres,
                               2*i + voff, 2*j + hoff, TRUE,
                               exp_table, random_numbers[k]))
                cell_status[k] &= ~CELL_STATUS_HAVE_RNUM;
        }
    }
}

static void
process_sublattice_horizontal3(guint *domain, guint xres, guint yres,
                               guint sublattice, guint32 *exp_table,
                               const guint32 *random_numbers,
                               guint *cell_status)
{
    guint i;

#ifdef _OPENMP
#pragma omp parallel for if (gwy_threads_are_enabled()) default(none) \
            shared(domain,random_numbers,cell_status,exp_table,sublattice,xres,yres) \
            private(i)
#endif
    for (i = 0; i < yres/2; i++) {
        guint hoff = (sublattice % 4)/2;
        guint j, k = (xres/2)*i;

        for (j = 0; j < xres/2; j++, k++) {
            guint voff = (j + sublattice) % 2;

            if ((cell_status[k] & CELL_STATUS_TRY_SWAPPING)
                && maybe_swap3(domain, xres, yres,
                               2*i + voff, 2*j + hoff, FALSE,
                               exp_table, random_numbers[k]))
                cell_status[k] &= ~CELL_STATUS_HAVE_RNUM;
        }
    }
}

/* XXX: This requires even-sized domain in both x and y. */
static void
process_sublattice(guint *domain, guint xres, guint yres, guint nspecies,
                   guint sublattice, guint32 *exp_table,
                   const guint32 *random_numbers, guint *cell_status)
{
    if (nspecies == 3) {
        if (sublattice < 4) {
            process_sublattice_vertical3(domain, xres, yres,
                                         sublattice, exp_table,
                                         random_numbers, cell_status);
        }
        else {
            process_sublattice_horizontal3(domain, xres, yres,
                                           sublattice, exp_table,
                                           random_numbers, cell_status);
        }
    }
    else if (nspecies == 2) {
        if (sublattice < 4) {
            process_sublattice_vertical2(domain, xres, yres,
                                         sublattice, exp_table,
                                         random_numbers, cell_status);
        }
        else {
            process_sublattice_horizontal2(domain, xres, yres,
                                           sublattice, exp_table,
                                           random_numbers, cell_status);
        }
    }
    else {
        g_assert_not_reached();
    }
}

static void
init_domain2_from_data_field(GwyDataField *dfield, guint *domain,
                             AnnealSynthArgs *args)
{
    guint xres = gwy_data_field_get_xres(dfield);
    guint yres = gwy_data_field_get_yres(dfield);
    gdouble *d = gwy_data_field_get_data(dfield);
    gdouble pvalue, threshold;
    guint xres2, yres2, i, j;
    gdouble *tmp;

    xres2 = (xres + 1)/2*2;
    yres2 = (yres + 1)/2*2;

    /* args->fraction is C-in-AC. */
    tmp = g_memdup(d, xres*yres*sizeof(gdouble));
    pvalue = 100.0 - args->fraction;
    gwy_math_percentiles(xres*yres, tmp, GWY_PERCENTILE_INTERPOLATION_MIDPOINT,
                         1, &pvalue, &threshold);
    g_free(tmp);

    for (i = 0; i < yres; i++) {
        for (j = 0; j < xres; j++)
            domain[i*xres2 + j] = (d[i*xres + j] >= threshold);
    }

    if (xres < xres2) {
        for (i = 0; i < yres; i++)
            domain[i*xres2 + xres2-1] = domain[i*xres2 + (i % 2 ? 0 : xres-1)];
    }
    if (yres < yres2) {
        for (j = 0; j < xres; j++)
            domain[yres*xres2 + j] = domain[j + (i % 2 ? 0 : yres-1)*xres];
    }
    if (xres < xres2 && yres < yres2)
        domain[xres2*yres2 - 1] = domain[0];
}

static inline guint
average3(guint a1, guint a2)
{
    if (a1 == a2)
        return a1;
    if (a1 == 1)
        return a2;
    if (a2 == 1)
        return a1;
    return 1;
}

static void
init_domain3_from_data_field(GwyDataField *dfield, guint *domain,
                             AnnealSynthArgs *args)
{
    guint xres = gwy_data_field_get_xres(dfield);
    guint yres = gwy_data_field_get_yres(dfield);
    gdouble *d = gwy_data_field_get_data(dfield);
    gdouble pvalues[2], thresholds[2];
    guint xres2, yres2, i, j;
    gdouble *tmp;

    xres2 = (xres + 1)/2*2;
    yres2 = (yres + 1)/2*2;

    /* Keep args->fraction as the C-in-AC fraction. */
    tmp = g_memdup(d, xres*yres*sizeof(gdouble));
    pvalues[0] = (100.0 - args->fraction)*(100.0 - args->B_fraction)/100.0;
    pvalues[1] = pvalues[0] + args->B_fraction;
    gwy_math_percentiles(xres*yres, tmp, GWY_PERCENTILE_INTERPOLATION_MIDPOINT,
                         2, pvalues, thresholds);
    g_free(tmp);

    for (i = 0; i < yres; i++) {
        for (j = 0; j < xres; j++) {
            domain[i*xres2 + j] = (d[i*xres + j] < thresholds[0]
                                   ? 0
                                   : (d[i*xres + j] >= thresholds[1]
                                      ? 2
                                      : 1));
        }
    }

    if (xres < xres2) {
        for (i = 0; i < yres; i++) {
            domain[i*xres2 + xres2-1] = average3(domain[i*xres2],
                                                 domain[i*xres2 + xres-1]);
        }
    }
    if (yres < yres2) {
        for (j = 0; j < xres; j++) {
            domain[yres*xres2 + j] = average3(domain[j],
                                              domain[j + (yres-1)*xres2]);
        }
    }
    if (xres < xres2 && yres < yres2) {
        domain[xres2*yres2 - 1] = average3(domain[0],
                                           domain[xres2*(yres-1) + xres-1]);
    }
}

static void
domain_add_to_data_field(const guint *u, GwyDataField *dfield)
{
    guint xres = gwy_data_field_get_xres(dfield);
    guint yres = gwy_data_field_get_yres(dfield);
    gdouble *d = gwy_data_field_get_data(dfield);
    guint xres2, i, j;

    xres2 = (xres + 1)/2*2;
    for (i = 0; i < yres; i++) {
        for (j = 0; j < xres; j++)
            d[i*xres + j] += u[i*xres2 + j];
    }
}

static gboolean
anneal_synth_do(AnnealSynthArgs *args,
                GwyDataField *dfield,
                GwyRandGenSet *rngset,
                gdouble preview_time)
{
    guint32 *random_numbers = NULL, *exp_table = NULL;
    guint *cell_status = NULL;
    guint *domain = NULL;
    guint xres, yres, xres2, yres2, l;
    guint nspecies = args->nspecies;
    /* Multiply niters by 4 since the probability of choosing a particular cell
     * is 1/4 in each iteration.  But if the user wants no averaging, then
     * really just use values from one iteration. */
    guint niters = 4*args->niters, navg = MIN(4*args->average - 3, niters);
    gdouble A = args->T_init, B = (A/args->T_final - 1.0)/niters;
    GwySynthUpdateType update;
    gboolean finished = FALSE;
    guint lattices[8];
    gulong i;
    GTimer *timer;
    GRand *rng;

    timer = g_timer_new();

    if (!args->animated)
        preview_time = 0.0;

    xres = gwy_data_field_get_xres(dfield);
    yres = gwy_data_field_get_yres(dfield);
    /* Create simulation domain with even dimensions. */
    xres2 = (xres + 1)/2*2;
    yres2 = (yres + 1)/2*2;

    gwy_synth_update_progress(NULL, 0, 0, 0);
    if (!gwy_app_wait_set_message(_("Running computation...")))
        goto fail;

    for (l = 0; l < 8; l++)
        lattices[l] = l;

    domain = g_new(guint, xres2*yres2);
    if (nspecies == 2) {
        init_domain2_from_data_field(dfield, domain, args);
        exp_table = g_new(guint, 2*4 + 1);
    }
    else if (nspecies == 3) {
        init_domain3_from_data_field(dfield, domain, args);
        exp_table = g_new(guint, 3*7*7);
    }
    else {
        g_assert_not_reached();
    }

    random_numbers = g_new(guint32, 2*xres2*yres2);
    cell_status = g_new0(guint, 2*xres2*yres2);
    rng = gwy_rand_gen_set_rng(rngset, 0);

    for (i = 0; i < niters; i++) {
        gdouble T = A/(1.0 + i*B);

        if (nspecies == 3)
            update_exp_table3(args->deltaE, 1.0/T, exp_table);
        else
            update_exp_table2(1.0/T, exp_table, 2*4 + 1);

        replenish_random_numbers(random_numbers, cell_status,
                                 2*xres2*yres2, rngset);

        /* Split the 2*n edges into 8 subsets, where in each the
         * edges are far enough from neighbours for updates to be independent.
         * The we can parallelise freely updates in one of the subset.  Always
         * run the update on the entire domain, but choose the order of
         * sublattice processing randomly. */
        for (l = 0; l < 8; l++) {
            guint ll = (l == 7 ? 7 : g_rand_int_range(rng, l, 8));
            guint roff = xres2*yres2/4 * l;

            GWY_SWAP(guint, lattices[l], lattices[ll]);
            ll = lattices[l];
            process_sublattice(domain, xres2, yres2, nspecies, ll, exp_table,
                               random_numbers + roff, cell_status + roff);
        }

        if (niters - i <= navg) {
            if (niters - i == navg)
                gwy_data_field_clear(dfield);
            domain_add_to_data_field(domain, dfield);
        }

        if (i % 100 == 0) {
            update = gwy_synth_update_progress(timer, preview_time, i, niters);
            if (update == GWY_SYNTH_UPDATE_CANCELLED)
                goto fail;
            if (update == GWY_SYNTH_UPDATE_DO_PREVIEW) {
                /* When we are already averaging, switch to the averaged
                 * field. */
                if (niters - i > navg) {
                    gwy_data_field_clear(dfield);
                    domain_add_to_data_field(domain, dfield);
                }
                gwy_data_field_invalidate(dfield);
                gwy_data_field_data_changed(dfield);
            }
        }
    }

    gwy_data_field_invalidate(dfield);
    gwy_data_field_data_changed(dfield);
    finished = TRUE;

fail:
    g_free(exp_table);
    g_free(random_numbers);
    g_free(cell_status);
    g_timer_destroy(timer);
    g_free(domain);

    return finished;
}

static void
init_field_randomly(GwyDataField *dfield, guint32 seed)
{
    gdouble *d = gwy_data_field_get_data(dfield);
    gint xres = gwy_data_field_get_xres(dfield);
    gint yres = gwy_data_field_get_yres(dfield);
    GRand *rng = g_rand_new();
    gint n = xres*yres, k;

    g_rand_set_seed(rng, seed);
    for (k = 0; k < n; k++)
        d[k] = g_rand_double(rng);

    g_rand_free(rng);
}

static const gchar prefix[]          = "/module/anneal_synth";
static const gchar active_page_key[] = "/module/anneal_synth/active_page";
static const gchar animated_key[]    = "/module/anneal_synth/animated";
static const gchar average_key[]     = "/module/anneal_synth/average";
static const gchar B_fraction_key[]  = "/module/anneal_synth/B_fraction";
static const gchar deltaE_AB_key[]   = "/module/anneal_synth/deltaE_AB";
static const gchar deltaE_AC_key[]   = "/module/anneal_synth/deltaE_AC";
static const gchar deltaE_BC_key[]   = "/module/anneal_synth/deltaE_BC";
static const gchar fraction_key[]    = "/module/anneal_synth/fraction";
static const gchar height_key[]      = "/module/anneal_synth/height";
static const gchar niters_key[]      = "/module/anneal_synth/niters";
static const gchar nspecies_key[]    = "/module/anneal_synth/nspecies";
static const gchar randomize_key[]   = "/module/anneal_synth/randomize";
static const gchar seed_key[]        = "/module/anneal_synth/seed";
static const gchar T_final_key[]     = "/module/anneal_synth/T_final";
static const gchar T_init_key[]      = "/module/anneal_synth/T_init";

static void
anneal_synth_sanitize_args(AnnealSynthArgs *args)
{
    args->active_page = CLAMP(args->active_page,
                              PAGE_DIMENSIONS, PAGE_NPAGES-1);
    args->seed = MAX(0, args->seed);
    args->randomize = !!args->randomize;
    args->animated = !!args->animated;
    args->niters = MIN(args->niters, 1000000);
    args->nspecies = CLAMP(args->nspecies, 2, 3);
    args->T_init = CLAMP(args->T_init, 0.001, 2.0);
    args->T_final = CLAMP(args->T_final, 0.001, 2.0);
    args->fraction = CLAMP(args->fraction, 0.01, 99.99);
    args->B_fraction = CLAMP(args->B_fraction, 0.01, 99.99);
    args->deltaE[0] = CLAMP(args->deltaE[0], 0.0, 1.0);
    args->deltaE[1] = CLAMP(args->deltaE[1], 0.0, 1.0);
    args->deltaE[2] = CLAMP(args->deltaE[2], 0.0, 1.0);
    fix_deltaE(args, 2);
    args->height = CLAMP(args->height, 0.001, 10000.0);
    args->average = CLAMP(args->average, 1, 10000);
}

static void
anneal_synth_load_args(GwyContainer *container,
                       AnnealSynthArgs *args,
                       GwyDimensionArgs *dimsargs)
{
    *args = anneal_synth_defaults;

    gwy_container_gis_int32_by_name(container, active_page_key,
                                    &args->active_page);
    gwy_container_gis_int32_by_name(container, seed_key, &args->seed);
    gwy_container_gis_boolean_by_name(container, randomize_key,
                                      &args->randomize);
    gwy_container_gis_boolean_by_name(container, animated_key,
                                      &args->animated);
    gwy_container_gis_int32_by_name(container, niters_key, &args->niters);
    gwy_container_gis_int32_by_name(container, nspecies_key, &args->nspecies);
    gwy_container_gis_double_by_name(container, T_init_key, &args->T_init);
    gwy_container_gis_double_by_name(container, T_final_key, &args->T_final);
    gwy_container_gis_double_by_name(container, fraction_key, &args->fraction);
    gwy_container_gis_double_by_name(container, B_fraction_key,
                                     &args->B_fraction);
    gwy_container_gis_double_by_name(container, deltaE_AB_key,
                                     args->deltaE + 0);
    gwy_container_gis_double_by_name(container, deltaE_AC_key,
                                     args->deltaE + 1);
    gwy_container_gis_double_by_name(container, deltaE_BC_key,
                                     args->deltaE + 2);
    gwy_container_gis_double_by_name(container, height_key, &args->height);
    gwy_container_gis_int32_by_name(container, average_key, &args->average);
    anneal_synth_sanitize_args(args);

    gwy_clear(dimsargs, 1);
    gwy_dimensions_copy_args(&dims_defaults, dimsargs);
    gwy_dimensions_load_args(dimsargs, container, prefix);
}

static void
anneal_synth_save_args(GwyContainer *container,
                       const AnnealSynthArgs *args,
                       const GwyDimensionArgs *dimsargs)
{
    gwy_container_set_int32_by_name(container, active_page_key,
                                    args->active_page);
    gwy_container_set_int32_by_name(container, seed_key, args->seed);
    gwy_container_set_boolean_by_name(container, randomize_key,
                                      args->randomize);
    gwy_container_set_boolean_by_name(container, animated_key,
                                      args->animated);
    gwy_container_set_int32_by_name(container, niters_key, args->niters);
    gwy_container_set_int32_by_name(container, nspecies_key, args->nspecies);
    gwy_container_set_double_by_name(container, T_init_key, args->T_init);
    gwy_container_set_double_by_name(container, T_final_key, args->T_final);
    gwy_container_set_double_by_name(container, fraction_key, args->fraction);
    gwy_container_set_double_by_name(container, B_fraction_key,
                                     args->B_fraction);
    gwy_container_set_double_by_name(container, deltaE_AB_key, args->deltaE[0]);
    gwy_container_set_double_by_name(container, deltaE_AC_key, args->deltaE[1]);
    gwy_container_set_double_by_name(container, deltaE_BC_key, args->deltaE[2]);
    gwy_container_set_double_by_name(container, height_key, args->height);
    gwy_container_set_int32_by_name(container, average_key, args->average);

    gwy_dimensions_save_args(dimsargs, container, prefix);
}

/* vim: set cin et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
