/*
 * JGlib -  FSlib compatible library for JG Font 
 *
 *  Programmmed by Hideo "Sir MaNMOS (and H.Kakugawa)
 *
 *  Edition History
 *  10 June 1993
 *  20 Jun. 1993   Ignored ZENKAKU-space by H.Kakugawa
 */

/*
 * Copyright (C) 1993 Hideo "Sir MaNMOS and Hirotsugu Kakugawa.  
 * All rights reserved.
 *
 * JGlib is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY.  No author or distributor accepts responsibility
 * to anyone for the consequences of using it or for whether it serves any
 * particular purpose or works at all, unless he says so in writing.  Refer
 * to the GNU General Public License for full details.
 *
 * Everyone is granted permission to copy, modify and redistribute
 * JGlib, but only under the conditions described in the GNU
 * General Public License.  A copy of this license is supposed to have been
 * given to you along with JGlib so you can know your rights and
 * responsibilities.  It should be in a file named COPYING.  Among other
 * things, the copyright notice and this notice must be preserved on all
 * copies. 
 */

#include  <stdio.h>
#include  <stdlib.h>
#include  <sys/types.h>
#include  "JGlib.h"
#include  "JGlib0.h"

#define		BEG_STEP	8

static	double	beg_para[BEG_STEP+1][4];

static JG_inited = 0;

static BUFFER  jg_buffer;
static int     jg_width;
static int     jg_height;
static int     jg_max_width;
static int     jg_max_height;
static int     jg_rast;
static int     jg_thin;
static int     jg_frame;
static int     jg_small;

#define		DATAMASK	((long)(0x7ff))
#define D_X_CL	500	/* horizontal center of data */
#define D_Y_CL	472	/* vertical center of data */
#define	D_Y_BL	872	/* base line of data */

#define B_X_CL	500	/* horizontal center in BuildChar */
#define B_Y_CL	400	/* vertical center in BuildChar */
#define B_Y_BL	0	/* base line in BuildChar */


#define Zerror_no_memory    (-100)
#define Zerror_cannot_read  (-101)
#define Zerror_cannot_write (-102)
#define Zerror_end_of_file  (-103)
#define Zerror_no_font      (-104)

static int   DrawFont();
static int   OpenFont();
static Font  *Zread_font();
static void  Zfree_font();
static int   Zread_header();
static Cycle *Zread_cycle();
static void  Zseek();
static long  Zread_long();
static int   Zread_byte();
static int   Zread_10_bits();
static int   jis2c();
static Free  *Zalloc();
static Cycle *Zalloc_cycle();
static Font  *Zalloc_font();
static void  Zfree();


#define  JG_FONT_TBL   16
typedef struct
{
  char        *font_name;
  FILE        *fd0;
  FILE        *fd1;
  FILE        *fd2;
  Zs_Header   read_header0;
  Zs_Header   read_header1;
  Zs_Header   read_header2;

} JGFont;
JGFont  JGFontTbl[JG_FONT_TBL];

static 
JG_init()
{
  register int  i;

  for (i = 0; i < JG_FONT_TBL; i++)
    JGFontTbl[i].font_name = NULL;
  JG_inited = 1;
}

static 
JG_find_slot()
{
  register int  i;

  for (i = 0; i < JG_FONT_TBL; i++)
    if (JGFontTbl[i].font_name == NULL)
      return i;
  return -1;
}


/*
 * JG_open_font()
 */
int
JG_open_font(fontname)
  char *fontname;
{
  int  fd; 

  if (JG_inited == 0)
    JG_init();

  if ((fd = JG_find_slot()) == -1)
    return -1;
  OpenFont(fd, fontname);

  return fd;
}


/*
 * JG_close_font()
 */
int
JG_close_font(fd)
  int  fd;
{
  char       *fontname;

  if ((fontname = JGFontTbl[fd].font_name) == NULL)
    return -1;

  free(JGFontTbl[fd].font_name);
  JGFontTbl[fd].font_name = NULL;
  fclose(JGFontTbl[fd].fd1);
  fclose(JGFontTbl[fd].fd2);
  return 0;
}


/*
 * JG_get_outline()
 */
int
JG_get_outline(c, fd, buf)
  int  c;     /* char code in serial code */
  int  fd;
  int  **buf;
{
  fprintf(stdout, "Not implemented yet. JG_get_outline()\n");
  return -1;
}

int
JG_scan_conv(vk_buf, w, h, bw, bo, bm_buf, thick, frame, smooth)
  long  *vk_buf;
  int   w;
  int   h;
  int   bw;
  int   bo;
  char  *bm_buf;
  int   thick;
  int   frame;
  int   smooth;
{
  fprintf(stdout, "Not implemented yet. JG_scan_conv()\n");
  return -1;
}

int
JG_get_bitmap(c, fd, w, h, bw, bo, bm_buf, thick, frame, smooth)
  int  c;
  int  fd;
  int  w;
  int  h;
  int  bw;
  int  bo;
  char *bm_buf;
  int  thick;
  int  frame;
  int  smooth;    /* ignored */
{
	char           *buff;
	int            ccode, rast, x, y;
	unsigned char  d;
	static int     scode2jis();
  
	rast = (w+7)/8;
	if ((buff = malloc(h*rast)) == NULL){
		fprintf(stderr, "No mem in JG_get_bitmap\n");
		exit(-1);
	}
	bzero(buff, rast*h);

	jg_width      = w;
	jg_height     = h;
	jg_buffer     = buff;
	jg_max_width  = MAX_VALUE & DATAMASK;
	jg_max_height = MAX_VALUE & DATAMASK;
	jg_rast       = rast;
	jg_frame	  = frame;
	if ( h + w < 48) {
		jg_frame = 2;
	}
	switch (thick){
	case 1:
		jg_thin = 60; break;
	case 0: default:
		jg_thin =  0; break;
	}

	ccode = scode2jis(c);
	if (ccode == 0x2121)    /* ZENKAKU space */
	  goto End;

	if (DrawFont(fd, ccode) < 0)
		return -1;

	for (y = 0; y < h; y++){
		for (x = 0; x < rast; x++)
			bm_buf[y*bw + x] = 0;
		for (x = 0; x < rast; x++){
			d = buff[y*rast + x];
			bm_buf[y*bw + x]   |= d >> bo;
			bm_buf[y*bw + x+1] |= d << (8-bo);
		}
	}
End:
	free(buff);

	return 0;
}

void
JG_set_f_xform(rotate, refx, refy)
  int rotate;
  int refx;
  int refy;
{
  return;
}

void
JG_fast_xform(vk, rotate, refx, refy)
  long *vk;
  int  rotate;
  int  refx;
  int  refy;
{
  return;
}

void
JG_set_m_xform(t1, t2, t3, t4, t5 ,t6)
  double  t1, t2, t3, t4, t5, t6;
{
  return;
}


void
JG_mat_xform(x, y, t1, t2, t3, t4, t5, t6)
   int    *x, *y;
  double  t1, t2, t3, t4, t5, t6;
{
  return;
}


/* ----------------------- */
/* Following routines are based on or snrafed 
 *   zfzeit.c (Ghostscript 2.4.1), and
 *   vfont.c  (dviout)
 */

static int
OpenFont(fd, fontname)
  int   fd;
  char  *fontname;
{
	int         err;
	FILE        *wfd0,*wfd1,*wfd2;
	Zs_Header   header0,header1,header2;
	char	      *the_name, *fn;

	if ((the_name = malloc(strlen(fontname))) == NULL){
		fprintf(stderr, "JGlib - no memory.");
		exit( -1 );
	}
	if ((fn = malloc(strlen(fontname)+10)) == NULL){
		fprintf(stderr, "JGlib - no memory.");
		exit( -1 );
	}
	strcpy(the_name, fontname);
  
	sprintf(fn, "%s", fontname);
	strcat(fn, ".fn0");
	if((wfd0 = fopen(fn, "rb")) == NULL){
		perror(fn);
		free(fn); free(the_name);
		return -1;
	}
	Zread_header(wfd0, &header0,CODE_SIZE0);

	sprintf(fn, "%s", fontname);
	strcat(fn, ".fn1");
	if((wfd1 = fopen(fn, "rb")) == NULL){
		perror(fn);
		free(fn); free(the_name);
		return -1;
	}
	Zread_header(wfd1, &header1,CODE_SIZE1);

	sprintf(fn, "%s", fontname);    
	strcat(fn, ".fn2");
	if((wfd2 = fopen(fn, "rb")) == NULL){
		perror(fn);
		free(fn); free(the_name);
		return -1;
	}
	Zread_header(wfd2, &header2,CODE_SIZE2);
  
	JGFontTbl[fd].font_name    = the_name;
	JGFontTbl[fd].fd0          = wfd0;
	JGFontTbl[fd].fd1          = wfd1;
	JGFontTbl[fd].fd2          = wfd2;
	JGFontTbl[fd].read_header0 = header0;
	JGFontTbl[fd].read_header1 = header1;
	JGFontTbl[fd].read_header2 = header2;
	free(fn);

	return fd;
}

static	make_begpara()
{
	int	i;
	double	t,u,t3,u3,tu2,t2u,t2,u2;

	for (i = 0 ; i < BEG_STEP+1;i++ ) {
		u = (double)i/BEG_STEP;
		t = 1.0-u;
		t2 = t*t; u2 = u*u; t3 = t2*t; u3 = u2*u; t2u = t2*u*3; tu2 = u2*t*3;
		beg_para[i][0] = t3;
		beg_para[i][1] = t2u;
		beg_para[i][2] = tu2;
		beg_para[i][3] = u3;
	}
}

static	Point	*begie(p0,p1,p2,p3,step)
Point	*p0,*p1,*p2,*p3;
int		step;
{
	int		i;
	Point	*ret,*ptr;
	static	first = 1;
	double	t,u,t3,u3,tu2,t2u,t2,u2;

	if ((ret = ptr = (Point *)malloc(sizeof(Point) * (step+1))) == NULL ) {
		return NULL;
	}
	if (first) {
		make_begpara();
		first = 0;
	}
	
	ptr[0] = *p0;
	ptr[step] = *p3;
	ptr++;
	for(i = 1; i <step ; i++ ,ptr++) {
		t3 = beg_para[i][0]; t2u = beg_para[i][1]; tu2 = beg_para[i][2]; u3 = beg_para[i][3] ;
		ptr->x = t3 * (p0->x) + t2u * (p1->x) + tu2 * (p2->x) + u3 *  (p3->x);
		ptr->y = t3 * (p0->y) + t2u * (p1->y) + tu2 * (p2->y) + u3 *  (p3->y);
	}
	return ret;
}

static	void	plot_beg(p0,p1,p2,p3,step,func)
Point	*p0,*p1,*p2,*p3;
int		step;
void	(*func)();
{
	Point	*ret,*free_ret;
	int      x, y, i;

	free_ret=ret = begie(p0,p1,p2,p3,step);
#ifdef TEST
	moveto(ret->x, ret->y);
#endif
	x = ret->x;
	y = ret->y;
	ret++;
	for(i = 1 ; i <= step ; i++,ret++) {
		(*func)(x,y,ret->x,ret->y);
#ifdef	TEST
		lineto(ret->x,ret->y);
#endif
		x = ret->x;
		y = ret->y;
	}
	free(free_ret);
}

static int 
DrawFont(fd, jiscode)
  int fd;
  int jiscode;
{
	char        *font_name;
	Font        *ptr, *fontptr;
	Cycle       *cptr, *cptr0;
	int         x, y, x1, y1,x2,y2,x3,y3, xbeg, ybeg;
	Point		*p0,*p1,*p2,*p3,*pp;
	int         func, h, j, beg;
	static void fill_edges(), trace_outline();
	static void (*DrawFuncTbl[])() = {fill_edges, trace_outline};
#ifdef TEST
	extern int  moveto(), lineto();
#endif

	if((fontptr = Zread_font(fd, jiscode)) == NULL)
		return -1;

	switch (jg_frame){
	case 0:
		h = 0;  j = 1;  break;
	case 1:
		default:
		h = 0;  j = 2;  break;
	case 2:
		h = 1;  j = 2;  break;
	}

	for (func = h; func < j; func++){
		for(ptr = fontptr; ptr; ptr = ptr->next){
			cptr0 = cptr = ptr->cycle;
			beg = 1;
#ifdef TEST
			printf("\n");
#endif
			do {
				p0 = pp = &(cptr->point);	cptr = cptr->next;
				p1 = pp = &(cptr->point);
				beg = 0;
				if ( !pp->xm ) {
#ifdef	TEST				
					moveto(p0->x,p0->y);
					lineto(pp->x,pp->y);
#endif
					(*DrawFuncTbl[func])(p0->x,p0->y,p1->x,p1->y);
					continue;
				}
				cptr = cptr->next;
				p2 = pp = &(cptr->point);	cptr = cptr->next;
				if ( !cptr ) {
#ifdef	TEST
					moveto(p0->x,p0->y);
					lineto(p1->x,p1->y);
					lineto(p2->x,p2->y);
#endif
					(*DrawFuncTbl[func])(p0->x,p0->y,p1->x,p1->y);
					(*DrawFuncTbl[func])(p2->x,p2->y,p1->x,p1->y);
					break;
				}
				p3 = pp = &(cptr->point);
				plot_beg(p0,p1,p2,p3,BEG_STEP,DrawFuncTbl[func]);
				if ( !pp->xm ) {
					do {
						cptr = cptr->next;
						pp = &(cptr->point);
#ifdef	TEST
						lineto(pp->x,pp->y);
#endif
						(*DrawFuncTbl[func])(p3->x,p3->y,pp->x,pp->y);
					} while(cptr->next && !pp->xm);
				}
				if (! cptr->next ) {
					pp = &(cptr0->point);
					if ( pp->x != cptr->point.x || pp->y != cptr->point.y) {
#ifdef	TEST
						lineto(pp->x,pp->y);
#endif
						(*DrawFuncTbl[func])(p3->x,p3->y,pp->x,pp->y);
					}
					break;
					
				}
			} while (cptr != cptr0);

		}
	}
  
	Zfree_font(fontptr);

	return 0;
}


static void 
trace_outline(x1, y1, x2, y2)
  int x1, y1, x2, y2;
{
  int     dx, dy, dx2, dy2, e, i, tmp;
  BUFFER  address;
  BUFFER  buffer     = jg_buffer;
  int     width      = jg_width;
  int     height     = jg_height;
  int     max_width  = jg_max_width;
  int     max_height = jg_max_height;
  int     thin       = jg_thin;
  int     y_factor   = jg_rast;
  long    offset_x, minus_offset_x, plus_offset_x;
  long    offset_y, minus_offset_y, plus_offset_y;
  unsigned char mask;
  
  offset_x       = (long)thin*(max_width+1)/100L/2L;
  minus_offset_x = -(max_width+1)/2L - offset_x;
  plus_offset_x  = -(max_width+1)/2L + offset_x;
  offset_y       = (long)thin*(max_height+1)/100L/2L;
  minus_offset_y = -(max_height+1)/2L - offset_y;
  plus_offset_y  = -(max_height+1)/2L + offset_y;
  
  /**printf("T.O.    (%5d,%5d)-(%5d,%5d)\n", x1,y1,x2,y2);**/
  dx = x2 - x1;
  dy = y2 - y1;
  if (dy < 0) {
    x1 = (int)(((long)x1 * width + minus_offset_x) / (max_width + 1));
    x2 = (int)(((long)x2 * width + minus_offset_x) / (max_width + 1));
  } else if (dy == 0) {
    x1 = (int)(((long)x1 * width - (max_width + 1) / 2) / (max_width + 1));
    x2 = (int)(((long)x2 * width - (max_width + 1) / 2) / (max_width + 1));
  } else {
    x1 = (int)(((long)x1 * width + plus_offset_x) / (max_width + 1));
    x2 = (int)(((long)x2 * width + plus_offset_x) / (max_width + 1));
  }
  if (dx > 0) {
    y1 = (int)( ((long)y1 * height + minus_offset_y) / (max_height + 1) );
    y2 = (int)( ((long)y2 * height + minus_offset_y) / (max_height + 1) );
  } else if (dx == 0) {
    y1 = (int)( ((long)y1 * height - (max_height + 1) / 2) 
	       / (max_height + 1) );
    y2 = (int)( ((long)y2 * height - (max_height + 1) / 2) 
	       / (max_height + 1) );
  } else {
    y1 = (int)(((long)y1 * height + plus_offset_y) / (max_height + 1));
    y2 = (int)(((long)y2 * height + plus_offset_y) / (max_height + 1));
  }
  if (x1 < 0) x1 = 0; else if (x1 >= max_width ) x1 = max_width - 1;
  if (x2 < 0) x2 = 0; else if (x2 >= max_width ) x2 = max_width - 1;
  if (y1 < 0) y1 = 0; else if (y1 >= max_height) y1 = max_height - 1;
  if (y2 < 0) y2 = 0; else if (y2 >= max_height) y2 = max_height - 1;
  
  dy = y2 - y1;
  if (dy < 0) {
    tmp = x1; x1 = x2; x2 = tmp;
    tmp = y1; y1 = y2; y2 = tmp;
    dy = -dy;
  }
  dx = x2 - x1;
  if (dx < 0) dx = -dx;
  if (dx == 0 && dy == 0) return;
  address = &buffer[y1 * y_factor + x1 / 8];
  mask = 0x80 >> (x1 & 7);
  dx2 = 2 * dx;
  dy2 = 2 * dy;
  
  if (dx < dy) {
    e = dx2 - dy;
    if (x1 < x2) {
      for (i = 0; i <= dy; i++) {
	*address |= mask;
	while (e >= 0) {
	  if ((mask >>= 1) == 0) {
	    address++;
	    mask = 0x80;
	  }
	  e -= dy2;
	}
	address += y_factor;
	e += dx2;
      }
    } else {
      for (i = 0; i <= dy; i++) {
	*address |= mask;
	while (e >= 0) {
	  if ((mask <<= 1) == 0) {
	    address--;
	    mask = 0x1;
	  }
	  e -= dy2;
	}
	address += y_factor;
	e += dx2;
      }
    }
  } else {
    e = dy2 - dx;
    if (x1 < x2) {
      for (i = 0; i <= dx; i++) {
	*address |= mask;
	while (e >= 0) {
	  address += y_factor;
	  e -= dx2;
	}
	if ((mask >>= 1) == 0) {
	  address++;
	  mask = 0x80;
	}
	e += dy2;
      }
    } else {
      for (i = 0; i <= dx; i++) {
	*address |= mask;
	while (e >= 0) {
	  address += y_factor;
	  e -= dx2;
	}
	if ((mask <<= 1) == 0) {
	  address--;
	  mask = 0x1;
	}
	e += dy2;
      }
    }
  }
}

static void 
fill_edges(x1, y1, x2, y2)
  int x1, y1, x2, y2;
{
  static unsigned char mask_pattern[8] =
      { 0xff, 0x7f, 0x3f, 0x1f, 0xf, 0x7, 0x3, 0x1 };
  int      dx, dy, dx2, dy2, sx, e, i, j, tmp;
  int      width      = jg_width;
  int      height     = jg_height;
  int      max_width  = jg_max_width;
  int      max_height = jg_max_height;
  int      y_factor   = jg_rast;
  BUFFER   buffer     = jg_buffer;
  BUFFER   address;
  int      bit, right_bytes;
  
  /**printf("F.E.    (%5d,%5d)-(%5d,%5d)\n", x1,y1,x2,y2);**/
  x1 = (int)( ((long)x1*width -(max_width+1)/2)  / (max_width+1) );
  y1 = (int)( ((long)y1*height-(max_height+1)/2) / (max_height+1) );
  x2 = (int)( ((long)x2*width -(max_width+1)/2)  / (max_width+1) );
  y2 = (int)( ((long)y2*height-(max_height+1)/2) / (max_height+1) );
  
  dy = y2 - y1;
  if (dy == 0) return;
  if (dy < 0) {
    tmp = x1; x1 = x2; x2 = tmp;
    tmp = y1; y1 = y2; y2 = tmp;
    dy = -dy;
  }
  dx = x2 - x1;
  sx = dx > 0 ? 1 : -1;
  if (dx < 0) dx = -dx;
  address = &buffer[y1 * y_factor + x1 / 8];
  right_bytes = (width - 1) / 8 - x1 / 8;
  bit = x1 & 7;
  dx2 = 2 * dx;
  dy2 = 2 * dy;
  e = dx2 - dy;
  
  for (i = 0; i < dy; i++) {
    /* Change right bits in this byte. */
    *address ^= mask_pattern[bit];
    /* Change all the bits in right bytes. */
    for (j = 1; j <= right_bytes; j++) {
      address[j] = ~address[j];
    }
    while (e >= 0) {
      bit += sx;
      if (bit & 0x8) {
	address += sx;
	right_bytes -= sx;
	bit &= 0x7;
      }
      e -= dy2;
    }
    address += y_factor;
    e += dx2;
  }
}

/*
 * scode2jis - serial code to jis code 
 */
static int 
scode2jis(scode)
int scode;
{
  int  jis;

  jis = ((scode/94)+0x21)*0x100 + ((scode%94)+0x21);
  return jis;
}



/***************************************************
 * Zeit Font Accsess Routines
 *     Original by Ext(T.Kawamoto)
 *     modifyed by K-ras
 **************************************************/
/*
 *	zeit font file library Ver 1.00
 *		for Human68k
 *	Copyright 1991 by Ext(T.Kawamoto)
 */

static int 
Zread_header(fp, hptr,code_size)
  FILE      *fp;
  Zs_Header *hptr;
{
	int i;
  
	Zseek(fp, 0);
	hptr->dummy[0] = Zread_byte(fp);
	hptr->dummy[1] = Zread_byte(fp);
	hptr->dummy[2] = Zread_byte(fp);
	hptr->dummy[3] = Zread_byte(fp);
	hptr->dummy[4] = Zread_byte(fp);
	hptr->dummy[5] = Zread_byte(fp);
	hptr->dummy[6] = Zread_byte(fp);
	hptr->dummy[7] = Zread_byte(fp);
	Zseek(fp,0x1ba);
	for (i = 0; i < code_size; ++i)
		hptr->offsets[i] = Zread_long(fp);
	return 0;
}

static	short	left;

static Font* 
Zread_font(fd, code)
int  fd;
int  code;
{
	Font  font, *fptr;
	Cycle *cptr;
	int   code_offset;
	long  seek_offset, no_font_flag;
	FILE  *fp;
	long	xoff;
  
	code_offset = jis2c(code);
	if (code < 0x3000){
		fp = JGFontTbl[fd].fd0;
		seek_offset  = no_font_flag = JGFontTbl[fd].read_header0.offsets[code_offset];
		seek_offset += (xoff = Zs0_Header_SIZE);
#ifdef	TEST
		printf("%8lx\n",JGFontTbl[fd].read_header0.offsets[code_offset+1]);
#endif	
	} else if (code < 0x5000){
		fp = JGFontTbl[fd].fd1;
		seek_offset  = no_font_flag = JGFontTbl[fd].read_header1.offsets[code_offset];
		seek_offset += (xoff = Zs1_Header_SIZE);
#ifdef	TEST
		printf("%8lx\n",JGFontTbl[fd].read_header1.offsets[code_offset+1]);
#endif
	} else {
		fp = JGFontTbl[fd].fd2;
		seek_offset  = no_font_flag = JGFontTbl[fd].read_header2.offsets[code_offset];
		seek_offset += (xoff = Zs2_Header_SIZE);
#ifdef	TEST
		printf("%8lx\n",JGFontTbl[fd].read_header1.offsets[code_offset+1]);
#endif
	}
	if (  no_font_flag == 0xffffffff)
		return NULL;
  
	Zseek(fp, seek_offset);
	fptr = &font;
	while (1){
		cptr = Zread_cycle(fp);
		if (cptr == NULL)
			break;
		if ((fptr->next = Zalloc_font()) == NULL)
			return fptr->next;
		fptr        = fptr->next;
		fptr->cycle = cptr;
	}
	fptr->next = NULL;
#ifdef	TEST
	printf  ("%8lx %d\n",ftell(fp) - xoff,left);
#endif
	return font.next;
}

static	int	Zread_12_bits();

static	Cycle	*Zread_point(fp)
FILE	*fp;
{
	Cycle	*c;
	int		x,y;
	x = Zread_12_bits(fp);
	y = Zread_12_bits(fp);
	if (x == MAX_VALUE && y == MAX_VALUE) return (Cycle *)-1;
	if ((c = Zalloc_cycle()) == NULL)	return NULL;
	c->point.x = x & DATAMASK;
	c->point.y = y & DATAMASK;
	c->point.xm = x & ~DATAMASK;
	c->point.ym = y & ~DATAMASK;
	if ( c->point.x > (jg_max_width +1)/2) {
		c->point.x = - (c->point.x) + (jg_max_width+1);
	} else {
		c->point.x += (jg_max_width+1)/2;
	}
	if ( c->point.y < (jg_max_height +1)/2) {
		c->point.y = - (c->point.y) + (jg_max_height+1)/2;
	}
	c->point.x -= (DATAMASK+1)/4;
	c->point.y -= (DATAMASK+1)*5/16; /* c->point.y -= (DATAMASK+1)/6; */
	c->point.x *=2;
	c->point.y *=2;
	return c;
}

static Cycle * 
Zread_cycle(fp)
  FILE *fp;
{
	int   x0, y0, x, y;
	int   x1, y2;
	int   x3, y3;
	int   x4, y4;
	int	loop= 0;
	Cycle *cptr0, *cptr,*cptrx,*cc;

	if ((cc = Zread_point(fp)) == (Cycle *)-1) return NULL;
	else if ( cc == NULL ) return NULL;
	cptrx = cptr0 = cptr = cc;
	while (1) {
		/* p1 */
		if ((cc = Zread_point(fp)) == (Cycle *)-1) break;
		else if ( cc == NULL ) return NULL;
		cptr = cptr->next = cc;
		loop = 1;
		if ( !cc->point.xm ) continue;
		

		/* p2 */

		if ((cc = Zread_point(fp)) == (Cycle *)-1) {
			cptrx->next = NULL;
			loop = 0;
			break;
		}
		else if ( cc == NULL ) return NULL;
		cptr = cptr->next = cc;
		loop = 1;

		/* p3 and p0 for next begie point */
		if ((cc = Zread_point(fp)) == (Cycle *)-1) break;
		else if ( cc == NULL ) return NULL;
		cptrx = cptr = cptr->next = cc;
		loop = 0;
	}
	
	if ( loop ) {
		cptr->next = cptr0;
	} else {
		cptr->next = NULL;
	}
	return cptr0;
}

static long
Zread_long(fp)
  FILE *fp;
{
	unsigned long dummy;
  
	dummy = Zread_byte(fp);
	dummy |= (unsigned long)Zread_byte(fp) << 8;
	dummy |= (unsigned long)Zread_byte(fp) << 16;
	dummy |= (unsigned long)Zread_byte(fp) << 24;
  
	return dummy;
}

static unsigned long divide;

static void 
Zseek(fp, offset)
  FILE *fp;
  long offset;
{
  fseek(fp, offset, 0);
  left   = 0;
  divide = 0;
}

static int 
Zread_12_bits(fp)
  FILE *fp;
{
	if (left < 12) {
		divide <<= 16;
		divide |= (unsigned long)Zread_byte(fp);
		divide |= (unsigned long)Zread_byte(fp) << 8;
		left   += 16;
	}
  
	left -= 12;
	return (divide >> left) & MAX_VALUE;
}

static int 
Zread_byte(fp)
  FILE *fp; 
{
	return fgetc(fp);
}


static int
jis2c(code)
  int  code;
{
	int  zscode;
  
	if (code < 0x3000){
		zscode = (((code >> 8) & 0xff) - 0x21) * 0x5e + (code & 0xff) - 0x21;
	} else if (code < 0x5000){
		zscode = (((code >> 8) & 0xff) - 0x30) * 0x5e + (code & 0xff) - 0x21;
	} else {
		zscode = (((code >> 8) & 0xff) - 0x50) * 0x5e + (code & 0xff) - 0x21;
	}
  
	return zscode;
}

/****************************************/
/*					*/
/*	alloc, free			*/
/*					*/
/****************************************/
static Free *free_ptr = NULL;

static Free *
Zalloc()
{
	Free *ptr;
  
	if (free_ptr == NULL){
		int i;
    
		if ((ptr = (Free *)malloc(sizeof(Free)*1024)) == NULL){
			fprintf(stderr,"\nZalloc malloc error\n");
			exit(0);
		}
		free_ptr = ptr;
		for (i = 0; i < 1023; ++i, ++ptr)
			ptr->next = ptr + 1;
		ptr->next = NULL;
	}
	ptr      = free_ptr;
	free_ptr = free_ptr->next;
  
	return ptr;
}

static Cycle * 
Zalloc_cycle()
{
	Free *ptr;
  
	ptr = Zalloc();
	if (ptr == NULL)
		return (Cycle *)ptr;
	return &(ptr->cycle);
}

static Font * 
Zalloc_font()
{
	Free *ptr;
  
	ptr = Zalloc();
	if (ptr == NULL)
		return (Font *)ptr;
	return &(ptr->font);
}

static void 
Zfree(ptr)
  Free *ptr;
{
	if (ptr == NULL)
		return;
	ptr->next = free_ptr;
	free_ptr  = ptr;
}

static void 
Zfree_cycle(ptr)
  Cycle *ptr;
{
	Cycle *ptr0, *ptr2;
  
	if (ptr == NULL)
		return;
	ptr0 = ptr;
	do {
		ptr2 = ptr->next;
		Zfree((Free *)ptr);
		ptr = ptr2;
	} while (ptr->next && ptr != ptr0);
}

static void 
Zfree_font(ptr)
  Font *ptr;
{
	Font *ptr2;
  
	while (ptr != NULL){
		Zfree_cycle(ptr->cycle);
		ptr2 = ptr->next;
		Zfree((Free *)ptr);
		ptr = ptr2;
	}
}

/* EOF */
