module fft_time_scheme
  use fft_force_solv
  use ffts

contains

subroutine fft_time_schematic( it, psiko, psikn, zopt1 )
!-- calculating time integration
  use fft_saveval_define
  use Math_Const

  implicit none
  integer, intent(in) :: it                               ! time step
  complex(kind(0d0)), dimension(nx,ny), intent(inout) :: psiko ! psi of zo
  complex(kind(0d0)), dimension(nx,ny), intent(out) :: psikn   ! psi of zn
  complex(kind(0d0)), dimension(nx,ny), intent(inout), optional :: zopt1
                                           ! additional term
                                           ! L-F => zeta in t=n-1
                                           ! AB2 => force in t=n-1
  integer :: i, j, ct
  complex(kind(0d0)), dimension(nx,ny) :: psikt, forcea_zeta, force_zeta

  psikt=psiko
  force_zeta=0.0d0
  forcea_zeta=0.0d0

  select case (time_flag(1:3))

  case ('EUL')  ! Euler scheme

     call fft_calc_FORCE( psiko, force_zeta )

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)

     do j=1,2*hynt+1
        do i=1,hxnt+1
           psikn(i,j)=psiko(i,j)+dt*force_zeta(i,j)
        end do
     end do

!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j)

     do j=2,hynt+1
        do i=2,hxnt+1
           psikn(nx-i+2,ny-j+2)=dconjg(psikn(i,j))
           psikn(nx-i+2,j)=dconjg(psikn(i,ny-j+2))
        end do
     end do

!$omp end do
!$omp end parallel

  case ('RK3')  ! 3rd order Runge-Kutta scheme

     do ct=1,3

        call fft_calc_FORCE( psiko, force_zeta )

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)

        do j=1,2*hynt+1
           do i=1,hxnt+1
              select case (ct)
              case (1)
                 forcea_zeta(i,j)=forcea_zeta(i,j)+force_zeta(i,j)
                 psiko(i,j)=psikt(i,j)+(0.5d0*dt)*force_zeta(i,j)
              case (2)
                 psiko(i,j)=psikt(i,j)+dt*(-forcea_zeta(i,j)+2.0d0*force_zeta(i,j))
                 forcea_zeta(i,j)=forcea_zeta(i,j)+4.0d0*force_zeta(i,j)
              case (3)
                 forcea_zeta(i,j)=forcea_zeta(i,j)+force_zeta(i,j)
                 psikn(i,j)=psikt(i,j)+(dt/6.0d0)*forcea_zeta(i,j)
              end select
           end do
        end do

!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j)

        do j=2,hynt+1
           do i=2,hxnt+1
              select case (ct)
              case (1,2)
                 psiko(nx-i+2,ny-j+2)=dconjg(psiko(i,j))
                 psiko(nx-i+2,j)=dconjg(psiko(i,ny-j+2))
              case (3)
                 psikn(nx-i+2,ny-j+2)=dconjg(psikn(i,j))
                 psikn(nx-i+2,j)=dconjg(psikn(i,ny-j+2))
              end select
           end do
        end do

!$omp end do
!$omp end parallel

     end do

  case ('RK4')  ! 4th order Runge-Kutta scheme

     do ct=1,4

        call fft_calc_FORCE( psiko, force_zeta )

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)

        do j=1,2*hynt+1
           do i=1,hxnt+1
              select case (ct)
              case (1)
                 forcea_zeta(i,j)=forcea_zeta(i,j)+force_zeta(i,j)
                 psiko(i,j)=psikt(i,j)+(0.5d0*dt)*force_zeta(i,j)
              case (2)                         
                 forcea_zeta(i,j)=forcea_zeta(i,j)+2.0d0*force_zeta(i,j)
                 psiko(i,j)=psikt(i,j)+(0.5d0*dt)*force_zeta(i,j)
              case (3)                         
                 forcea_zeta(i,j)=forcea_zeta(i,j)+2.0d0*force_zeta(i,j)
                 psiko(i,j)=psikt(i,j)+(dt)*force_zeta(i,j)
              case (4)                         
                 forcea_zeta(i,j)=forcea_zeta(i,j)+force_zeta(i,j)
                 psikn(i,j)=psikt(i,j)+(dt/6.0d0)*forcea_zeta(i,j)
              end select
           end do
        end do

!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j)

        do j=2,hynt+1
           do i=2,hxnt+1
              select case (ct)
              case (1,2,3)
                 psiko(nx-i+2,ny-j+2)=dconjg(psiko(i,j))
                 psiko(nx-i+2,j)=dconjg(psiko(i,ny-j+2))
              case (4)
                 psikn(nx-i+2,ny-j+2)=dconjg(psikn(i,j))
                 psikn(nx-i+2,j)=dconjg(psikn(i,ny-j+2))
              end select
           end do
        end do

!$omp end do
!$omp end parallel

     end do

  case ('AB2')  ! 2nd order Adams-Bashforse scheme

     call fft_calc_FORCE( psiko, force_zeta )

     if(it==1)then

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)

        do j=1,2*hynt+1
           do i=1,hxnt+1
              psikn(i,j)=psiko(i,j)+dt*force_zeta(i,j)
              zopt1(i,j)=force_zeta(i,j)
           end do
        end do

!$omp end do
!$omp end parallel

     else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)

        do j=1,2*hynt+1
           do i=1,hxnt+1
              psikn(i,j)=psiko(i,j)+dt*(1.5d0*force_zeta(i,j)-0.5d0*zopt1(i,j))
              zopt1(i,j)=force_zeta(i,j)
           end do
        end do

!$omp end do
!$omp end parallel

     end if

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)

     do j=2,hynt+1
        do i=2,hxnt+1
           psikn(nx-i+2,ny-j+2)=dconjg(psikn(i,j))
           psikn(nx-i+2,j)=dconjg(psikn(i,ny-j+2))
        end do
     end do

!$omp end do
!$omp end parallel

  case ('L-F')  ! Leap Frog scheme

     call fft_calc_FORCE( psiko, force_zeta )

     if(it==1)then

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)

        do j=1,2*hynt+1
           do i=1,hxnt+1
              psikn(i,j)=psiko(i,j)+dt*force_zeta(i,j)
              zopt1(i,j)=psiko(i,j)
           end do
        end do

!$omp end do
!$omp end parallel

     else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)

        do j=1,2*hynt+1
           do i=1,hxnt+1
              psikn(i,j)=zopt1(i,j)+2.0d0*dt*force_zeta(i,j)
              zopt1(i,j)=psiko(i,j)
           end do
        end do

!$omp end do
!$omp end parallel

     end if

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)

     do j=2,hynt+1
        do i=2,hxnt+1
           psikn(nx-i+2,ny-j+2)=dconjg(psikn(i,j))
           psikn(nx-i+2,j)=dconjg(psikn(i,ny-j+2))
        end do
     end do

!$omp end do
!$omp end parallel

!  case ('C-N')  ! Crank-Nicolson scheme

!  case ('BEU')  ! Backward Euler scheme

  end select

end subroutine fft_time_schematic

end module fft_time_scheme
