!-----------------------------------------------------------------------
!     Copyright (C) 2009-2015 GFD Dennou Club. All rights reserved.
!-----------------------------------------------------------------------

module radar_analysis  ! レーダデータ解析モジュール

  use Matrix_calc

  implicit none

contains

subroutine VVP( r, t, p, x, y, z, vr, u, v, w, undef )
! VVP 法を用いてドップラーレーダの動径速度データから風速場データを求積する.
! 計算の順番は
! 1. 座標変換を行う.
! 2. 計算候補点を探索する.
! 3. 座標変数行列 (G) を計算する.
! 4. 逆行列 (G-1) を計算する.
! 5. 行列計算を行う.
!    現段階で求めたい物理量が 3 次元速度場なので, 過程 5 は 3 成分計算のみ行う.
  implicit none
  integer, intent(in) :: r(:)  ! 動径方向格子座標 [m]
  integer, intent(in) :: t(:)  ! 同位角方向格子座標 [rad]
  integer, intent(in) :: p(:)  ! 仰角方向格子座標 [rad]
  integer, intent(in) :: x(:)  ! 東西方向格子座標 [m]
  integer, intent(in) :: y(:)  ! 南北方向格子座標 [m]
  integer, intent(in) :: z(:)  ! 鉛直方向格子座標 [m]
  integer, intent(in) :: vr(size(r),size(t),size(p))  ! ドップラーレーダ速度データ [m/s]
  integer, intent(inout) :: u(size(x),size(y),size(z))  ! 東西速度場データ [m/s]
  integer, intent(inout) :: v(size(x),size(y),size(z))  ! 東西速度場データ [m/s]
  integer, intent(inout) :: w(size(x),size(y),size(z))  ! 東西速度場データ [m/s]
  real, intent(in), optional :: undef  ! 未定義データ
  real, allocatable, dimension(:) :: Vo, Vv
  real, allocatable, dimension(:,:) :: a, b, c

  integer, parameter :: nm=11
  integer :: i, j, k, nx, ny, nz, nr, nt, np, nl

  nr=size(r)
  nt=size(t)
  np=size(p)
  nx=size(x)
  ny=size(y)
  nz=size(z)
  allocate(a(nm,nm))
  allocate(c(nm,nm))

!$omp parallel default(shared)
!$omp do private(i,j,k,a,b,c,nl,Vo)

  do k=1,nz
     do j=1,ny
        do i=1,nx
        !-- 1,2. 座標変換と計算候補点を探索する.
           call search_calc_point( r, t, p, x(i), y(j), z(k), ix, iy, iz, nl )
           allocate(b(nm,nl))
        !-- 3. 座標変数行列の計算
        !-- 4. 逆行列の計算
           call invert_mat( c, a )
        !-- 5. 行列計算
           u(i,j,k)=calc_G( 1, a, b, Vo )
           v(i,j,k)=calc_G( 2, a, b, Vo )
           w(i,j,k)=calc_G( 8, a, b, Vo )
           deallocate(b)
        end do
     end do
  end do

!$omp end do
!$omp end parallel

end subroutine VVP

end module radar_analysis
