!---------------------------------------------------------------
! Copyright (C) 2009-2015 GFD Dennou Club. All rights reserved.
!---------------------------------------------------------------

module Algebra   ! 代数演算を主に行うモジュール
! 固有値計算, 行列演算, ベクトル演算を主に行う.

  use Basis

  public :: rectangle_int
  public :: abst_2d
  public :: abst_3d
  public :: dot_prod_2d
  public :: dot_prod_3d
  public :: vec_prod_2d
  public :: vec_prod_3d

interface rectangle_int

  module procedure rectangle_intf, rectangle_intd, rectangle_intcp, rectangle_intcpd

end interface rectangle_int

interface abst_2d

  module procedure abst_2df, abst_2dd

end interface abst_2d

interface abst_3d

  module procedure abst_3df, abst_3dd

end interface abst_3d

interface dot_prod_2d

  module procedure dot_prod_2df, dot_prod_2dd

end interface dot_prod_2d

interface dot_prod_3d

  module procedure dot_prod_3df, dot_prod_3dd

end interface dot_prod_3d

interface vec_prod_2d

  module procedure vec_prod_2df, vec_prod_2dd

end interface vec_prod_2d

interface vec_prod_3d

  module procedure vec_prod_3df, vec_prod_3dd

end interface vec_prod_3d

contains

subroutine rectangle_intf( x, y, bot, top, res, undeff )  ! 1 次元台形積分
  ! 不等間隔でも計算可能であるが, 精度は保証しない.
  implicit none
  real, intent(in) :: bot  ! 積分区間左端
  real, intent(in) :: top  ! 積分区間右端
  real, intent(in) :: x(:)  ! 積分変数
  real, intent(in) :: y(size(x))  ! 非積分関数
  real, intent(inout) :: res  ! 台形積分の積分値
  real, intent(in), optional :: undeff
  integer :: i, nx, i_bot, i_top
  real :: y_bot, y_top

  nx=size(x)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_1d( nx, y ),  &
  &                                     "rectangle_int" )
  end if

  res=0.0

!-- bot < top でなければエラー
  if(bot>top)then
     write(*,*) "#### ERROR (algebra:rectangle_int) ####"
     write(*,*) "integrated interval must be bot < top. STOP", bot, top
     stop
  end if

!-- 以下で積分域を決定
!-- 実際には, 積分境界に最近する点

  if(present(undeff))then
     do i=1,nx
        if(x(i)/=undeff)then
           if(x(i)>bot)then
              write(*,*) "#### WARNING (algebra:rectangle_int) ####"
              write(*,*) "there is NOT the bot in the x(i).", bot, x(i), i
              ! このときは, i_bot=1としておいても, x(1)=bot なので, 余分計算はゼロとなり影響はない.
           end if
           exit
        end if
     end do

     do i=nx,1,-1
        if(x(i)/=undeff)then
           if(x(i)<top)then
              write(*,*) "#### WARNING (algebra:rectangle_int) ####"
              write(*,*) "there is NOT the top in the x(i).", top, x(i), i
              ! このとき, i_top=nx としても, x(nx)=top なので, 余分計算はゼロとなる.
           end if
           exit
        end if
     end do

     do i=1,nx
        if(x(i)/=undeff)then
           if(x(i)>=bot)then  ! i_bot は bot - top 内の最も bot に近づく格子
              i_bot=i
              exit
           end if
        end if
     end do

     do i=nx,1,-1
        if(x(i)/=undeff)then
           if(x(i)<=top)then  ! i_top は bot - top 内の最も top に近づく格子
              i_top=i
              exit
           end if
        end if
     end do

  else

     if(x(1)>bot)then
        write(*,*) "#### WARNING (algebra:rectangle_int) ####"
        write(*,*) "there is NOT the bot in the x(1).", bot, x(1)
        ! このときは, i_bot=1としておいても, x(1)=bot なので, 余分計算はゼロとなり影響はない.
     end if
     if(x(nx)<top)then
        write(*,*) "#### WARNING (algebra:rectangle_int) ####"
        write(*,*) "there is NOT the top in the x(nx).", top, x(nx)
        ! このとき, i_top=nx としても, x(nx)=top なので, 余分計算はゼロとなる.
     end if

     do i=1,nx
        if(x(i)>=bot)then  ! i_bot は bot - top 内の最も bot に近づく格子
           i_bot=i
           exit
        end if
     end do

     do i=nx,1,-1
        if(x(i)<=top)then  ! i_top は bot - top 内の最も top に近づく格子
           i_top=i
           exit
        end if
     end do

  end if

!-- 以下で格子に当てはまらない部分を内挿で補完
  if(present(undeff))then
     if(i_bot/=1)then
        if(y(i_bot)/=undeff.and.y(i_bot-1)/=undeff.and.x(i_bot)/=x(i_bot-1))then
           y_bot=y(i_bot-1)  &
     &           +((y(i_bot)-y(i_bot-1))/(x(i_bot)-x(i_bot-1)))*(bot-x(i_bot-1))
        else
           y_bot=-y(i_bot)  ! 最後の積分でゼロにするための措置
        end if
     else
        y_bot=-y(i_bot)  ! 最後の積分でゼロにするための措置
     end if
     if(i_top/=nx)then
        if(y(i_top)/=undeff.and.y(i_top+1)/=undeff.and.x(i_top+1)/=x(i_top))then
           y_top=y(i_top)  &
     &           +((y(i_top+1)-y(i_top))/(x(i_top+1)-x(i_top)))*(x(i_top+1)-top)
        else
           y_top=-y(i_top)
        end if
     else
        y_top=-y(i_top)  ! 最後の積分でゼロにするための措置
     end if
  else
     if(i_bot/=1)then
        y_bot=y(i_bot-1)  &
     &        +((y(i_bot)-y(i_bot-1))/(x(i_bot)-x(i_bot-1)))*(bot-x(i_bot-1))
     else
        y_bot=-y(i_bot)  ! 最後の積分でゼロにするための措置
     end if
     if(i_top/=nx)then
        y_top=y(i_top)  &
     &        +((y(i_top+1)-y(i_top))/(x(i_top+1)-x(i_top)))*(x(i_top+1)-top)
     else
        y_top=-y(i_top)  ! 最後の積分でゼロにするための措置
     end if
  end if

  if(i_bot<i_top)then  ! 積分区間内に格子が 2 つ以上あるとき

     if(present(undeff))then
        res=res+0.5*(x(i_bot)-bot)*(y(i_bot)+y_bot)  ! 下端の余り短冊分
        do i=i_bot,i_top-1
           if(y(i+1)/=undeff.and.y(i)/=undeff)then
              res=res+0.5*(x(i+1)-x(i))*(y(i+1)+y(i))  ! 通常の短冊
           end if
        end do
        res=res+0.5*(top-x(i_top))*(y_top+y(i_top))  ! 上端の余りのみ
     else
        res=res+0.5*(x(i_bot)-bot)*(y(i_bot)+y_bot)  ! 下端の余り短冊分
        do i=i_bot,i_top-1
           res=res+0.5*(x(i+1)-x(i))*(y(i+1)+y(i))  ! 通常の短冊
        end do
        res=res+0.5*(top-x(i_top))*(y_top+y(i_top))  ! 上端の余りのみ
     end if
  else
     if(present(undeff))then
        if(y(i_bot)/=undeff)then
           res=res+0.5*(x(i_bot)-bot)*(y(i_bot)+y_bot)  &
  &            +0.5*(top-x(i_top))*(y_top+y(i_top))
        else
           res=0.5*(top-bot)*(y_top+y_bot)
        end if
     else
        res=res+0.5*(x(i_bot)-bot)*(y(i_bot)+y_bot)  &
  &         +0.5*(top-x(i_bot))*(y_top+y(i_bot))
     end if
  end if

end subroutine rectangle_intf

!------------------------------------------
!------------------------------------------

subroutine rectangle_intd( x, y, bot, top, res, undeff )  ! 1 次元台形積分
  ! 不等間隔でも計算可能であるが, 精度は保証しない.
  implicit none
  double precision, intent(in) :: bot  ! 積分区間左端
  double precision, intent(in) :: top  ! 積分区間右端
  double precision, intent(in) :: x(:)  ! 積分変数
  double precision, intent(in) :: y(size(x))  ! 非積分関数
  double precision, intent(inout) :: res  ! 台形積分の積分値
  double precision, intent(in), optional :: undeff
  integer :: i, nx, i_bot, i_top
  double precision :: y_bot, y_top

  nx=size(x)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_1d( nx, y ),  &
  &                                     "rectangle_int" )
  end if

  res=0.0d0

!-- bot < top でなければエラー
  if(bot>top)then
     write(*,*) "#### ERROR (algebra:rectangle_int) ####"
     write(*,*) "integrated interval must be bot < top. STOP", bot, top
     stop
  end if

!-- 以下で積分域を決定
!-- 実際には, 積分境界に最近する点

  if(present(undeff))then
     do i=1,nx
        if(x(i)/=undeff)then
           if(x(i)>bot)then
              write(*,*) "#### WARNING (algebra:rectangle_int) ####"
              write(*,*) "there is NOT the bot in the x(i).", bot, x(i), i
              ! このときは, i_bot=1としておいても, x(1)=bot なので, 余分計算はゼロとなり影響はない.
           end if
           exit
        end if
     end do

     do i=nx,1,-1
        if(x(i)/=undeff)then
           if(x(i)<top)then
              write(*,*) "#### WARNING (algebra:rectangle_int) ####"
              write(*,*) "there is NOT the top in the x(i).", top, x(i), i
              ! このとき, i_top=nx としても, x(nx)=top なので, 余分計算はゼロとなる.
           end if
           exit
        end if
     end do

     do i=1,nx
        if(x(i)/=undeff)then
           if(x(i)>=bot)then  ! i_bot は bot - top 内の最も bot に近づく格子
              i_bot=i
              exit
           end if
        end if
     end do

     do i=nx,1,-1
        if(x(i)/=undeff)then
           if(x(i)<=top)then  ! i_top は bot - top 内の最も top に近づく格子
              i_top=i
              exit
           end if
        end if
     end do

  else

     if(x(1)>bot)then
        write(*,*) "#### WARNING (algebra:rectangle_int) ####"
        write(*,*) "there is NOT the bot in the x(1).", bot, x(1)
        ! このときは, i_bot=1としておいても, x(1)=bot なので, 余分計算はゼロとなり影響はない.
     end if
     if(x(nx)<top)then
        write(*,*) "#### WARNING (algebra:rectangle_int) ####"
        write(*,*) "there is NOT the top in the x(nx).", top, x(nx)
        ! このとき, i_top=nx としても, x(nx)=top なので, 余分計算はゼロとなる.
     end if

     do i=1,nx
        if(x(i)>=bot)then  ! i_bot は bot - top 内の最も bot に近づく格子
           i_bot=i
           exit
        end if
     end do

     do i=nx,1,-1
        if(x(i)<=top)then  ! i_top は bot - top 内の最も top に近づく格子
           i_top=i
           exit
        end if
     end do

  end if

!-- 以下で格子に当てはまらない部分を内挿で補完
  if(present(undeff))then
     if(i_bot/=1)then
        if(y(i_bot)/=undeff.and.y(i_bot-1)/=undeff.and.x(i_bot)/=x(i_bot-1))then
           y_bot=y(i_bot-1)  &
     &           +((y(i_bot)-y(i_bot-1))/(x(i_bot)-x(i_bot-1)))*(bot-x(i_bot-1))
        else
           y_bot=-y(i_bot)  ! 最後の積分でゼロにするための措置
        end if
     else
        y_bot=-y(i_bot)  ! 最後の積分でゼロにするための措置
     end if
     if(i_top/=nx)then
        if(y(i_top)/=undeff.and.y(i_top+1)/=undeff.and.x(i_top+1)/=x(i_top))then
           y_top=y(i_top)  &
     &           +((y(i_top+1)-y(i_top))/(x(i_top+1)-x(i_top)))*(x(i_top+1)-top)
        else
           y_top=-y(i_top)
        end if
     else
        y_top=-y(i_top)  ! 最後の積分でゼロにするための措置
     end if
  else
     if(i_bot/=1)then
        y_bot=y(i_bot-1)  &
     &        +((y(i_bot)-y(i_bot-1))/(x(i_bot)-x(i_bot-1)))*(bot-x(i_bot-1))
     else
        y_bot=-y(i_bot)  ! 最後の積分でゼロにするための措置
     end if
     if(i_top/=nx)then
        y_top=y(i_top)  &
     &        +((y(i_top+1)-y(i_top))/(x(i_top+1)-x(i_top)))*(x(i_top+1)-top)
     else
        y_top=-y(i_top)  ! 最後の積分でゼロにするための措置
     end if
  end if

  if(i_bot<i_top)then  ! 積分区間内に格子が 2 つ以上あるとき

     if(present(undeff))then
        res=res+0.5d0*(x(i_bot)-bot)*(y(i_bot)+y_bot)  ! 下端の余り短冊分
        do i=i_bot,i_top-1
           if(y(i+1)/=undeff.and.y(i)/=undeff)then
              res=res+0.5d0*(x(i+1)-x(i))*(y(i+1)+y(i))  ! 通常の短冊
           end if
        end do
        res=res+0.5d0*(top-x(i_top))*(y_top+y(i_top))  ! 上端の余りのみ
     else
        res=res+0.5d0*(x(i_bot)-bot)*(y(i_bot)+y_bot)  ! 下端の余り短冊分
        do i=i_bot,i_top-1
           res=res+0.5d0*(x(i+1)-x(i))*(y(i+1)+y(i))  ! 通常の短冊
        end do
        res=res+0.5d0*(top-x(i_top))*(y_top+y(i_top))  ! 上端の余りのみ
     end if
  else
     if(present(undeff))then
        if(y(i_bot)/=undeff)then
           res=res+0.5d0*(x(i_bot)-bot)*(y(i_bot)+y_bot)  &
  &            +0.5d0*(top-x(i_top))*(y_top+y(i_top))
        else
           res=0.5d0*(top-bot)*(y_top+y_bot)
        end if
     else
        res=res+0.5d0*(x(i_bot)-bot)*(y(i_bot)+y_bot)  &
  &         +0.5d0*(top-x(i_bot))*(y_top+y(i_bot))
     end if
  end if

end subroutine rectangle_intd

!------------------------------------------
!------------------------------------------

subroutine rectangle_intcp( x, y, bot, top, res, undeff )  ! 1 次元台形積分
  ! 不等間隔でも計算可能であるが, 精度は保証しない.
  implicit none
  real, intent(in) :: bot  ! 積分区間左端
  real, intent(in) :: top  ! 積分区間右端
  real, intent(in) :: x(:)  ! 積分変数
  complex, intent(in) :: y(size(x))  ! 非積分関数
  complex, intent(inout) :: res  ! 台形積分の積分値
  real, intent(in), optional :: undeff
  integer :: i, nx, i_bot, i_top
  real :: tmpr(size(x)), tmpi(size(x))
  real :: tmprres, tmpires
  complex, parameter :: img_unit=(0.0,1.0)

  nx=size(x)

  do i=1,nx
     tmpr(i)=real(y(i))
     tmpi(i)=aimag(y(i))
  end do

  if(present(undeff))then
     call rectangle_intf( x, tmpr, bot, top, tmprres, undeff )
     call rectangle_intf( x, tmpi, bot, top, tmpires, undeff )
     if(tmprres/=undeff.and.tmpires/=undeff)then
        res=tmprres+img_unit*tmpires
     else
        res=undeff+img_unit*undeff
     end if
  else
     call rectangle_intf( x, tmpr, bot, top, tmprres )
     call rectangle_intf( x, tmpi, bot, top, tmpires )
     res=tmprres+img_unit*tmpires
  end if

end subroutine rectangle_intcp

!------------------------------------------
!------------------------------------------

subroutine rectangle_intcpd( x, y, bot, top, res, undeff )  ! 1 次元台形積分
  ! 不等間隔でも計算可能であるが, 精度は保証しない.
  implicit none
  double precision, intent(in) :: bot  ! 積分区間左端
  double precision, intent(in) :: top  ! 積分区間右端
  double precision, intent(in) :: x(:)  ! 積分変数
  complex(kind(0d0)), intent(in) :: y(size(x))  ! 非積分関数
  complex(kind(0d0)), intent(inout) :: res  ! 台形積分の積分値
  double precision, intent(in), optional :: undeff
  integer :: i, nx, i_bot, i_top
  double precision :: tmpr(size(x)), tmpi(size(x))
  double precision :: tmprres, tmpires
  complex(kind(0d0)), parameter :: img_unit=(0.0d0,1.0d0)

  nx=size(x)

  do i=1,nx
     tmpr(i)=dble(y(i))
     tmpi(i)=dimag(y(i))
  end do

  if(present(undeff))then
     call rectangle_intd( x, tmpr, bot, top, tmprres, undeff )
     call rectangle_intd( x, tmpi, bot, top, tmpires, undeff )
     if(tmprres/=undeff.and.tmpires/=undeff)then
        res=tmprres+img_unit*tmpires
     else
        res=undeff+img_unit*undeff
     end if
  else
     call rectangle_intd( x(1:nx), tmpr(1:nx), bot, top, tmprres )
     call rectangle_intd( x(1:nx), tmpi(1:nx), bot, top, tmpires )
     res=tmprres+img_unit*tmpires
  end if

end subroutine rectangle_intcpd

!------------------------------------------
!------------------------------------------

subroutine rectangle_int_2d( x, y, z, botx, topx, boty, topy, res, undeff )
  ! 2 次元台形積分
  ! 不等間隔でも計算可能であるが, 精度は保証しない.
  ! なお, 本積分ルーチンは積分区間が空間に依存しない矩形積分にのみ対応している.
  ! つまり, \int^{y_2}_{y_1}{\int^{x_2}_{x_1}{f(x,y)dxdy}}
  ! を計算するときの, y_1, y_2, x_1, x_2 がそれぞれ x, y に依存しないことが条件である.

  implicit none
  real, intent(in) :: x(:)  ! 積分変数 1
  real, intent(in) :: y(:)  ! 積分変数 2
  real, intent(in) :: botx  ! 積分区間 1 左端
  real, intent(in) :: topx  ! 積分区間 1 右端
  real, intent(in) :: boty  ! 積分区間 2 左端
  real, intent(in) :: topy  ! 積分区間 2 右端
  real, intent(in) :: z(size(x),size(y))  ! 非積分関数
  real, intent(inout) :: res  ! 台形積分の積分値
  real, intent(in), optional :: undeff
  integer :: i, j, nx, ny, i_bot, i_top
  real :: resy(size(y))

  nx=size(x)
  ny=size(y)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, z ),  &
  &                                     "rectangle_int_2d" )
  end if

  resy=0.0
  res=0.0

!-- 以下で積分域を決定
!-- 実際には, 積分境界に最近する点

  if(y(1)>boty)then
     write(*,*) "#### WARNING ####"
     write(*,*) "there is NOT the bot in the y(i)."  ! このときは, i_bot=1としておいても, x(1)=bot なので, 余分計算はゼロとなり影響はない.
  end if
  if(y(ny)<topy)then
     write(*,*) "#### WARNING ####"
     write(*,*) "there is NOT the top in the y(i)."  ! このとき, i_top=nx としても, x(nx)=top なので, 余分計算はゼロとなる.
  end if

  do i=1,ny
     if(y(i)>=boty)then  ! i_bot は bot - top 内の最も bot に近づく格子
        i_bot=i
        exit
     end if
  end do

  if(i_bot>1)then  ! 領域の 1 格子外側を指定しておく.
     i_bot=i_bot-1
  end if

  do i=ny,1,-1
     if(y(i)<=topy)then  ! i_top は bot - top 内の最も top に近づく格子
        i_top=i
        exit
     end if
  end do

  if(i_top<ny)then  ! 領域の 1 格子外側を指定しておく.
     i_top=i_top+1
  end if

  if(present(undeff))then
     do j=i_bot,i_top
        call rectangle_int( x, z(:,j), botx, topx, resy(j), undeff=undeff )
     end do

     call rectangle_int( y, resy, boty, topy, res, undeff=undeff )

  else
     do j=i_bot,i_top
        call rectangle_int( x, z(:,j), botx, topx, resy(j) )
     end do

     call rectangle_int( y, resy, boty, topy, res )

  end if

end subroutine rectangle_int_2d

!------------------------------------------
!------------------------------------------

subroutine abst_2df(x,y,dis)
  ! 2 次元ベクトルの絶対値を計算するルーチン
  implicit none
  real, intent(in) :: x(:,:)  ! x 方向のベクトル成分
  real, intent(in) :: y(size(x,1),size(x,2))  ! y 方向のベクトル成分
  real, intent(inout) :: dis(size(x,1),size(x,2))  ! 各点での絶対値ベクトル
  integer :: i, j, nx, ny

  nx=size(x,1)
  ny=size(x,2)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, y ),  &
  &                                     "abst_2df" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, dis ),  &
  &                                     "abst_2df" )
  end if

  do j=1,ny
     do i=1,nx
        dis(i,j)=sqrt(x(i,j)**2+y(i,j)**2)
     end do
  end do

end subroutine abst_2df

!------------------------------------------
!------------------------------------------

subroutine abst_2dd(x,y,dis)
  ! 2 次元ベクトルの絶対値を計算するルーチン
  implicit none
  double precision, intent(in) :: x(:,:)  ! x 方向のベクトル成分
  double precision, intent(in) :: y(size(x,1),size(x,2))  ! y 方向のベクトル成分
  double precision, intent(inout) :: dis(size(x,1),size(x,2))  ! 各点での絶対値ベクトル
  integer :: i, j, nx, ny

  nx=size(x,1)
  ny=size(x,2)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, y ),  &
  &                                     "abst_2dd" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, dis ),  &
  &                                     "abst_2dd" )
  end if

  do j=1,ny
     do i=1,nx
        dis(i,j)=dsqrt(x(i,j)**2+y(i,j)**2)
     end do
  end do

end subroutine abst_2dd

!------------------------------------------
!------------------------------------------

subroutine abst_3df(x,y,z,dis)  ! 3 次元ベクトルの絶対値を計算するルーチン
  ! 配列数を調整することにより, 2 次元での計算も可能.
  implicit none
  real, intent(in) :: x(:,:,:)  ! x 方向のベクトル成分
  real, intent(in) :: y(size(x,1),size(x,2),size(x,3))  ! y 方向のベクトル成分
  real, intent(in) :: z(size(x,1),size(x,2),size(x,3))  ! z 方向のベクトル成分
  real, intent(inout) :: dis(size(x,1),size(x,2),size(x,3))  ! 各点での絶対値ベクトル
  integer :: i, j, k, nx, ny, nz

  nx=size(x,1)
  ny=size(x,2)
  nz=size(x,3)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, y ),  &
  &                                     "abst_3df" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, z ),  &
  &                                     "abst_3df" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, dis ),  &
  &                                     "abst_3df" )
  end if

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,k)
  do k=1,nz
     do j=1,ny
        do i=1,nx
           dis(i,j,k)=sqrt(x(i,j,k)**2+y(i,j,k)**2+z(i,j,k)**2)
        end do
     end do
  end do
!$omp end do
!$omp end parallel

end subroutine abst_3df

!------------------------------------------
!------------------------------------------

subroutine abst_3dd(x,y,z,dis)  ! 3 次元ベクトルの絶対値を計算するルーチン
  ! 配列数を調整することにより, 2 次元での計算も可能.
  implicit none
  double precision, intent(in) :: x(:,:,:)  ! x 方向のベクトル成分
  double precision, intent(in) :: y(size(x,1),size(x,2),size(x,3))  ! y 方向のベクトル成分
  double precision, intent(in) :: z(size(x,1),size(x,2),size(x,3))  ! z 方向のベクトル成分
  double precision, intent(inout) :: dis(size(x,1),size(x,2),size(x,3))  ! 各点での絶対値ベクトル
  integer :: i, j, k, nx, ny, nz

  nx=size(x,1)
  ny=size(x,2)
  nz=size(x,3)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, y ),  &
  &                                     "abst_3dd" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, z ),  &
  &                                     "abst_3dd" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, dis ),  &
  &                                     "abst_3dd" )
  end if

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,k)
  do k=1,nz
     do j=1,ny
        do i=1,nx
           dis(i,j,k)=dsqrt(x(i,j,k)**2+y(i,j,k)**2+z(i,j,k)**2)
        end do
     end do
  end do
!$omp end do
!$omp end parallel

end subroutine abst_3dd

!------------------------------------------
!------------------------------------------

subroutine calc_radius(xp,yp,zp,x,y,z,rad)
  ! ある位置からの距離を計算するルーチン
  ! 配列数を調整することにより, 2 次元での計算も可能.
  implicit none
  real, intent(in) :: xp  ! 中心位置座標の x 成分
  real, intent(in) :: yp  ! 中心位置座標の y 成分
  real, intent(in) :: zp  ! 中心位置座標の z 成分
  real, intent(in) :: x(:)  ! x 方向の位置座標
  real, intent(in) :: y(:)  ! y 方向の位置座標
  real, intent(in) :: z(:)  ! z 方向の位置座標
  real, intent(inout) :: rad(size(x),size(y),size(z))  ! 距離
  integer :: i, j, k, nx, ny, nz

  nx=size(x)
  ny=size(y)
  nz=size(z)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, rad ),  &
  &                                     "calc_radius" )
  end if

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,k)
  do k=1,nz
     do j=1,ny
        do i=1,nx
           rad(i,j,k)=sqrt((x(i)-xp)**2+(y(j)-yp)**2+(z(k)-zp)**2)
        end do
     end do
  end do
!$omp end do
!$omp end parallel


end subroutine calc_radius

!------------------------------------------
!------------------------------------------

subroutine dot_prod_2df(x,y,u,v,dot,undeff)
  ! 2ベクトルの内積計算ルーチン
  ! dot_prod の 2 次元版
  implicit none
  real, intent(in) :: x(:,:)  ! x 方向のベクトル成分
  real, intent(in) :: y(size(x,1),size(x,2))  ! y 方向のベクトル成分
  real, intent(in) :: u(size(x,1),size(x,2))  ! x 方向のベクトル成分
  real, intent(in) :: v(size(x,1),size(x,2))  ! y 方向のベクトル成分
  real, intent(inout) :: dot(size(x,1),size(x,2))  ! 内積
  real, intent(in), optional :: undeff        ! 未定義値
  integer :: i, j, nx, ny

  nx=size(x,1)
  ny=size(x,2)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, y ),  &
  &                                     "dot_prod_2df" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, u ),  &
  &                                     "dot_prod_2df" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, v ),  &
  &                                     "dot_prod_2df" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, dot ),  &
  &                                     "dot_prod_2df" )
  end if

  if(present(undeff))then

     do j=1,ny
        do i=1,nx
           if(x(i,j)==undeff.or.u(i,j)==undeff.or.  &
  &           y(i,j)==undeff.or.v(i,j)==undeff)then
              dot(i,j)=undeff
           else
              dot(i,j)=x(i,j)*u(i,j)+y(i,j)*v(i,j)
           end if
        end do
     end do

  else

     do j=1,ny
        do i=1,nx
           dot(i,j)=x(i,j)*u(i,j)+y(i,j)*v(i,j)
        end do
     end do

  end if

end subroutine dot_prod_2df

!------------------------------------------
!------------------------------------------

subroutine dot_prod_2dd(x,y,u,v,dot,undeff)
  ! 2ベクトルの内積計算ルーチン
  ! dot_prod の 2 次元版
  implicit none
  double precision, intent(in) :: x(:,:)  ! x 方向のベクトル成分
  double precision, intent(in) :: y(size(x,1),size(x,2))  ! y 方向のベクトル成分
  double precision, intent(in) :: u(size(x,1),size(x,2))  ! x 方向のベクトル成分
  double precision, intent(in) :: v(size(x,1),size(x,2))  ! y 方向のベクトル成分
  double precision, intent(inout) :: dot(size(x,1),size(x,2))  ! 内積
  double precision, intent(in), optional :: undeff        ! 未定義値
  integer :: i, j, nx, ny

  nx=size(x,1)
  ny=size(x,2)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, y ),  &
  &                                     "dot_prod_2dd" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, u ),  &
  &                                     "dot_prod_2dd" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, v ),  &
  &                                     "dot_prod_2dd" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, dot ),  &
  &                                     "dot_prod_2dd" )
  end if

  if(present(undeff))then

     do j=1,ny
        do i=1,nx
           if(x(i,j)==undeff.or.u(i,j)==undeff.or.  &
  &           y(i,j)==undeff.or.v(i,j)==undeff)then
              dot(i,j)=undeff
           else
              dot(i,j)=x(i,j)*u(i,j)+y(i,j)*v(i,j)
           end if
        end do
     end do

  else

     do j=1,ny
        do i=1,nx
           dot(i,j)=x(i,j)*u(i,j)+y(i,j)*v(i,j)
        end do
     end do

  end if

end subroutine dot_prod_2dd

!------------------------------------------
!------------------------------------------

subroutine dot_prod_3df(x,y,z,u,v,w,dot,undeff)
  ! 2ベクトルの内積計算ルーチン
  ! 配列を工夫すると, 2 次元での内積を計算することも可能
  implicit none
  real, intent(in) :: x(:,:,:)  ! x 方向のベクトル成分
  real, intent(in) :: y(size(x,1),size(x,2),size(x,3))  ! y 方向のベクトル成分
  real, intent(in) :: z(size(x,1),size(x,2),size(x,3))  ! z 方向のベクトル成分
  real, intent(in) :: u(size(x,1),size(x,2),size(x,3))  ! x 方向のベクトル成分
  real, intent(in) :: v(size(x,1),size(x,2),size(x,3))  ! y 方向のベクトル成分
  real, intent(in) :: w(size(x,1),size(x,2),size(x,3))  ! z 方向のベクトル成分
  real, intent(inout) :: dot(size(x,1),size(x,2),size(x,3))  ! 内積
  real, intent(in), optional :: undeff
  integer :: i, j, k, nx, ny, nz

  nx=size(x,1)
  ny=size(x,2)
  nz=size(x,3)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, y ),  &
  &                                     "dot_prod_3df" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, z ),  &
  &                                     "dot_prod_3df" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, u ),  &
  &                                     "dot_prod_3df" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, v ),  &
  &                                     "dot_prod_3df" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, w ),  &
  &                                     "dot_prod_3df" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, dot ),  &
  &                                     "dot_prod_3df" )
  end if

  if(present(undeff))then

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,k)
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(x(i,j,k)==undeff.or.u(i,j,k)==undeff.or.y(i,j,k)==undeff.or.  &
  &              v(i,j,k)==undeff.or.z(i,j,k)==undeff.or.w(i,j,k)==undeff)then
                 dot(i,j,k)=undeff
              else
                 dot(i,j,k)=x(i,j,k)*u(i,j,k)+y(i,j,k)*v(i,j,k)+z(i,j,k)*w(i,j,k)
              end if
           end do
        end do
     end do
!$omp end do
!$omp end parallel

  else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,k)
     do k=1,nz
        do j=1,ny
           do i=1,nx
              dot(i,j,k)=x(i,j,k)*u(i,j,k)+y(i,j,k)*v(i,j,k)+z(i,j,k)*w(i,j,k)
           end do
        end do
     end do
!$omp end do
!$omp end parallel
  end if

end subroutine dot_prod_3df

!------------------------------------------
!------------------------------------------

subroutine dot_prod_3dd(x,y,z,u,v,w,dot,undeff)
  ! 2ベクトルの内積計算ルーチン
  ! 配列を工夫すると, 2 次元での内積を計算することも可能
  implicit none
  double precision, intent(in) :: x(:,:,:)  ! x 方向のベクトル成分
  double precision, intent(in) :: y(size(x,1),size(x,2),size(x,3))  ! y 方向のベクトル成分
  double precision, intent(in) :: z(size(x,1),size(x,2),size(x,3))  ! z 方向のベクトル成分
  double precision, intent(in) :: u(size(x,1),size(x,2),size(x,3))  ! x 方向のベクトル成分
  double precision, intent(in) :: v(size(x,1),size(x,2),size(x,3))  ! y 方向のベクトル成分
  double precision, intent(in) :: w(size(x,1),size(x,2),size(x,3))  ! z 方向のベクトル成分
  double precision, intent(inout) :: dot(size(x,1),size(x,2),size(x,3))  ! 内積
  double precision, intent(in), optional :: undeff
  integer :: i, j, k, nx, ny, nz

  nx=size(x,1)
  ny=size(x,2)
  nz=size(x,3)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, y ),  &
  &                                     "dot_prod_3dd" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, z ),  &
  &                                     "dot_prod_3dd" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, u ),  &
  &                                     "dot_prod_3dd" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, v ),  &
  &                                     "dot_prod_3dd" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, w ),  &
  &                                     "dot_prod_3dd" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, dot ),  &
  &                                     "dot_prod_3dd" )
  end if

  if(present(undeff))then

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,k)
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(x(i,j,k)==undeff.or.u(i,j,k)==undeff.or.y(i,j,k)==undeff.or.  &
  &              v(i,j,k)==undeff.or.z(i,j,k)==undeff.or.w(i,j,k)==undeff)then
                 dot(i,j,k)=undeff
              else
                 dot(i,j,k)=x(i,j,k)*u(i,j,k)+y(i,j,k)*v(i,j,k)+z(i,j,k)*w(i,j,k)
              end if
           end do
        end do
     end do
!$omp end do
!$omp end parallel

  else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,k)
     do k=1,nz
        do j=1,ny
           do i=1,nx
              dot(i,j,k)=x(i,j,k)*u(i,j,k)+y(i,j,k)*v(i,j,k)+z(i,j,k)*w(i,j,k)
           end do
        end do
     end do
!$omp end do
!$omp end parallel
  end if

end subroutine dot_prod_3dd

!------------------------------------------
!------------------------------------------

subroutine vec_prod_2df(x,y,u,v,vecz,undeff)
  ! 2ベクトルの外積計算ルーチン
  ! vec_prod の 2 次元版
  ! 外積なので, 2 次元で計算すると, 外積の 1 成分しか現れないので,
  ! intent(inout) の引数が 1 つしかないことに注意.
  implicit none
  real, intent(in) :: x(:,:)  ! x 方向のベクトル成分
  real, intent(in) :: y(size(x,1),size(x,2))  ! y 方向のベクトル成分
  real, intent(in) :: u(size(x,1),size(x,2))  ! x 方向のベクトル成分
  real, intent(in) :: v(size(x,1),size(x,2))  ! y 方向のベクトル成分
  real, intent(inout) :: vecz(size(x,1),size(x,2))  ! 外積の直交成分
  real, intent(in), optional :: undeff
  integer :: i, j, nx, ny

  nx=size(x,1)
  ny=size(x,2)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, y ),  &
  &                                     "vec_prod_2df" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, u ),  &
  &                                     "vec_prod_2df" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, v ),  &
  &                                     "vec_prod_2df" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, vecz ),  &
  &                                     "vec_prod_2df" )
  end if

  if(present(undeff))then

        do j=1,ny
           do i=1,nx
              if(x(i,j)==undeff.or.u(i,j)==undeff.or.  &
  &              y(i,j)==undeff.or.v(i,j)==undeff)then
                 vecz(i,j)=undeff
              else
                 vecz(i,j)=x(i,j)*v(i,j)-y(i,j)*u(i,j)
              end if
           end do
        end do

  else

        do j=1,ny
           do i=1,nx
              vecz(i,j)=x(i,j)*v(i,j)-y(i,j)*u(i,j)
           end do
        end do

  end if

end subroutine vec_prod_2df

!-----------------------------------------
!------------------------------------------

subroutine vec_prod_2dd(x,y,u,v,vecz,undeff)
  ! 2ベクトルの外積計算ルーチン
  ! vec_prod の 2 次元版
  ! 外積なので, 2 次元で計算すると, 外積の 1 成分しか現れないので,
  ! intent(inout) の引数が 1 つしかないことに注意.
  implicit none
  double precision, intent(in) :: x(:,:)  ! x 方向のベクトル成分
  double precision, intent(in) :: y(size(x,1),size(x,2))  ! y 方向のベクトル成分
  double precision, intent(in) :: u(size(x,1),size(x,2))  ! x 方向のベクトル成分
  double precision, intent(in) :: v(size(x,1),size(x,2))  ! y 方向のベクトル成分
  double precision, intent(inout) :: vecz(size(x,1),size(x,2))  ! 外積の直交成分
  double precision, intent(in), optional :: undeff
  integer :: i, j, nx, ny

  nx=size(x,1)
  ny=size(x,2)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, y ),  &
  &                                     "vec_prod_2dd" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, u ),  &
  &                                     "vec_prod_2dd" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, v ),  &
  &                                     "vec_prod_2dd" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, vecz ),  &
  &                                     "vec_prod_2dd" )
  end if

  if(present(undeff))then

        do j=1,ny
           do i=1,nx
              if(x(i,j)==undeff.or.u(i,j)==undeff.or.  &
  &              y(i,j)==undeff.or.v(i,j)==undeff)then
                 vecz(i,j)=undeff
              else
                 vecz(i,j)=x(i,j)*v(i,j)-y(i,j)*u(i,j)
              end if
           end do
        end do

  else

        do j=1,ny
           do i=1,nx
              vecz(i,j)=x(i,j)*v(i,j)-y(i,j)*u(i,j)
           end do
        end do

  end if

end subroutine vec_prod_2dd

!-----------------------------------------
!-----------------------------------------

subroutine vec_prod_3df(x,y,z,u,v,w,vecx,vecy,vecz,undeff)
  ! 2ベクトルの外積計算ルーチン
  implicit none
  real, intent(in) :: x(:,:,:)  ! x 方向のベクトル成分
  real, intent(in) :: y(size(x,1),size(x,2),size(x,3))  ! y 方向のベクトル成分
  real, intent(in) :: z(size(x,1),size(x,2),size(x,3))  ! z 方向のベクトル成分
  real, intent(in) :: u(size(x,1),size(x,2),size(x,3))  ! x 方向のベクトル成分
  real, intent(in) :: v(size(x,1),size(x,2),size(x,3))  ! y 方向のベクトル成分
  real, intent(in) :: w(size(x,1),size(x,2),size(x,3))  ! z 方向のベクトル成分
  real, intent(inout) :: vecx(size(x,1),size(x,2),size(x,3))  ! 外積の x 成分
  real, intent(inout) :: vecy(size(x,1),size(x,2),size(x,3))  ! 外積の y 成分
  real, intent(inout) :: vecz(size(x,1),size(x,2),size(x,3))  ! 外積の z 成分
  real, intent(in), optional :: undeff
  integer :: i, j, k, nx, ny, nz

  nx=size(x,1)
  ny=size(x,2)
  nz=size(x,3)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, y ),  &
  &                                     "vec_prod_3df" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, z ),  &
  &                                     "vec_prod_3df" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, u ),  &
  &                                     "vec_prod_3df" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, v ),  &
  &                                     "vec_prod_3df" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, w ),  &
  &                                     "vec_prod_3df" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, vecx ),  &
  &                                     "vec_prod_3df" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, vecy ),  &
  &                                     "vec_prod_3df" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, vecz ),  &
  &                                     "vec_prod_3df" )
  end if

  if(present(undeff))then

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,k)
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(x(i,j,k)==undeff.or.u(i,j,k)==undeff.or.y(i,j,k)==undeff.or.  &
  &              v(i,j,k)==undeff.or.z(i,j,k)==undeff.or.w(i,j,k)==undeff)then
                 vecx(i,j,k)=undeff
                 vecy(i,j,k)=undeff
                 vecz(i,j,k)=undeff
              else
                 vecx(i,j,k)=y(i,j,k)*w(i,j,k)-z(i,j,k)*v(i,j,k)
                 vecy(i,j,k)=z(i,j,k)*u(i,j,k)-x(i,j,k)*w(i,j,k)
                 vecz(i,j,k)=x(i,j,k)*v(i,j,k)-y(i,j,k)*u(i,j,k)
              end if
           end do
        end do
     end do
!$omp end do
!$omp end parallel

  else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,k)
     do k=1,nz
        do j=1,ny
           do i=1,nx
              vecx(i,j,k)=y(i,j,k)*w(i,j,k)-z(i,j,k)*v(i,j,k)
              vecy(i,j,k)=z(i,j,k)*u(i,j,k)-x(i,j,k)*w(i,j,k)
              vecz(i,j,k)=x(i,j,k)*v(i,j,k)-y(i,j,k)*u(i,j,k)
           end do
        end do
     end do
!$omp end do
!$omp end parallel

  end if

end subroutine vec_prod_3df

!-----------------------------------------
!-----------------------------------------

subroutine vec_prod_3dd(x,y,z,u,v,w,vecx,vecy,vecz,undeff)
  ! 2ベクトルの外積計算ルーチン
  implicit none
  double precision, intent(in) :: x(:,:,:)  ! x 方向のベクトル成分
  double precision, intent(in) :: y(size(x,1),size(x,2),size(x,3))  ! y 方向のベクトル成分
  double precision, intent(in) :: z(size(x,1),size(x,2),size(x,3))  ! z 方向のベクトル成分
  double precision, intent(in) :: u(size(x,1),size(x,2),size(x,3))  ! x 方向のベクトル成分
  double precision, intent(in) :: v(size(x,1),size(x,2),size(x,3))  ! y 方向のベクトル成分
  double precision, intent(in) :: w(size(x,1),size(x,2),size(x,3))  ! z 方向のベクトル成分
  double precision, intent(inout) :: vecx(size(x,1),size(x,2),size(x,3))  ! 外積の x 成分
  double precision, intent(inout) :: vecy(size(x,1),size(x,2),size(x,3))  ! 外積の y 成分
  double precision, intent(inout) :: vecz(size(x,1),size(x,2),size(x,3))  ! 外積の z 成分
  double precision, intent(in), optional :: undeff
  integer :: i, j, k, nx, ny, nz

  nx=size(x,1)
  ny=size(x,2)
  nz=size(x,3)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, y ),  &
  &                                     "vec_prod_3dd" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, z ),  &
  &                                     "vec_prod_3dd" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, u ),  &
  &                                     "vec_prod_3dd" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, v ),  &
  &                                     "vec_prod_3dd" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, w ),  &
  &                                     "vec_prod_3dd" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, vecx ),  &
  &                                     "vec_prod_3dd" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, vecy ),  &
  &                                     "vec_prod_3dd" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, vecz ),  &
  &                                     "vec_prod_3dd" )
  end if

  if(present(undeff))then

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,k)
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(x(i,j,k)==undeff.or.u(i,j,k)==undeff.or.y(i,j,k)==undeff.or.  &
  &              v(i,j,k)==undeff.or.z(i,j,k)==undeff.or.w(i,j,k)==undeff)then
                 vecx(i,j,k)=undeff
                 vecy(i,j,k)=undeff
                 vecz(i,j,k)=undeff
              else
                 vecx(i,j,k)=y(i,j,k)*w(i,j,k)-z(i,j,k)*v(i,j,k)
                 vecy(i,j,k)=z(i,j,k)*u(i,j,k)-x(i,j,k)*w(i,j,k)
                 vecz(i,j,k)=x(i,j,k)*v(i,j,k)-y(i,j,k)*u(i,j,k)
              end if
           end do
        end do
     end do
!$omp end do
!$omp end parallel

  else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,k)
     do k=1,nz
        do j=1,ny
           do i=1,nx
              vecx(i,j,k)=y(i,j,k)*w(i,j,k)-z(i,j,k)*v(i,j,k)
              vecy(i,j,k)=z(i,j,k)*u(i,j,k)-x(i,j,k)*w(i,j,k)
              vecz(i,j,k)=x(i,j,k)*v(i,j,k)-y(i,j,k)*u(i,j,k)
           end do
        end do
     end do
!$omp end do
!$omp end parallel

  end if

end subroutine vec_prod_3dd

!-----------------------------------------
!-----------------------------------------

!-----------------------------------------
!-----------------------------------------

end module Algebra
