!---------------------------------------------------------------
! Copyright (C) 2009-2015 GFD Dennou Club. All rights reserved.
!---------------------------------------------------------------

module cloud
! CReSS の雲物理過程
  use Thermo_Function
  use Thermo_Const
  use special_function

contains

real function Src_theta_warm( temp, pres, qv, qr )
! warm rain 過程での theta のソース項
  use Thermo_Const
  implicit none
  real, intent(in) :: temp  ! 気温 [K]
  real, intent(in) :: pres  ! トータル気圧 [Pa]
  real, intent(in) :: qv  ! 水蒸気混合比 [kg/kg]
  real, intent(in) :: qr  ! 雨水混合比 [kg/kg]
  real :: vapor_qv, rhob

  vapor_qv=qvss( temp, pres )
  rhob=TP_2_rho( temp, pres )

  Src_theta_warm=Lv( temp )*EVrv( temp, qv, qr, pres )

  Src_theta_warm=rhob*Src_theta_warm/(cpd*exner_func_dry( pres ))

  return

end function Src_theta_warm

!---------------------------------

!!!real function Src_theta_cold( rhob, rhop, rho0, ptb, ptp, pb, pp, qv, qc, qr, qi, qs, qg )
!!!! warm rain 過程での theta のソース項
!!!  use Thermo_Const
!!!  implicit none
!!!  real, intent(in) :: rhob  ! 基本場の密度 [kg/m3]
!!!  real, intent(in) :: rhop  ! 密度の変動 [kg/m3]
!!!  real, intent(in) :: rho0  ! 地表面の基本場密度 [kg/m3]
!!!  real, intent(in) :: ptb  ! 温位の基本場 [K]
!!!  real, intent(in) :: ptp  ! 温位の変動 [K]
!!!  real, intent(in) :: pb  ! 温位の基本場 [K]
!!!  real, intent(in) :: pp  ! 温位の変動 [K]
!!!  real, intent(in) :: qv  ! 水蒸気混合比 [kg/kg]
!!!  real, intent(in) :: qc  ! 雲水混合比 [kg/kg]
!!!  real, intent(in) :: qr  ! 雨水混合比 [kg/kg]
!!!  real, intent(in) :: qi  ! 雲氷混合比 [kg/kg]
!!!  real, intent(in) :: qs  ! 雪混合比 [kg/kg]
!!!  real, intent(in) :: qg  ! 霰混合比 [kg/kg]
!!!  real :: vapor_qv
!!!
!!!  vapor_qv=qvs( temp )
!!!
!!!  Src_theta_cold=Lv( temp )*(VDvr())  &
!!!  &              +Ls( temp )*(NUAvi()+VDvi()+VDvs()+VDvg())  &
!!!  &              +Lf( temp )*(NUFci()+NUCci()+NUHci()  &
!!!  &              +CLri()-MLic()-MLsr()-MLgr()+FRrg())
!!!
!!!  Src_theta_cold=Src_theta_cold/(cpd*exner())
!!!
!!!  return
!!!
!!!end function Src_theta_cold
!!!
!!!!---------------------------------
! common function (in cloud module)
!---------------------------------

real function Lv( temp )
! 水 -> 水蒸気での潜熱加熱
  use Thermo_Const
  implicit none
  real, intent(in) :: temp  ! 温度 [K]

  Lv=2.50078e6*(ti0/temp)**(0.167+3.67e-4*temp)

  return

end function Lv

real function Ls( temp )
! 氷 -> 水蒸気での潜熱加熱
  use Thermo_Const
  implicit none
  real, intent(in) :: temp ! 温度 [K]

  Ls=2.834e6+100.0*(temp-ti0)

  return

end function Ls

real function Lf( temp )
! 水 -> 氷での融解熱
  use Thermo_Const
  implicit none
  real, intent(in) :: temp ! 温度 [K]

  Lf=3.34e5+2500.0*(temp-ti0)

  return

end function Lf

real function nu_air( pres, temp )
! 空気の動粘性係数
  use Thermo_Const
  implicit none
  real, intent(in) :: pres ! 気圧 [Pa]
  real, intent(in) :: temp ! 温度 [K]

  nu_air=1.328e-5*(p00/pres)*(temp/ti0)**1.754

  return

end function nu_air

real function mu_air( pres, temp )
! 空気の粘性係数
  implicit none
  real, intent(in) :: pres
  real, intent(in) :: temp
  real :: rho

  rho=TP_2_rho( temp, pres )

  mu_air=nu_air( pres, temp )*rho

  return

end function mu_air

real function Dv( pres, temp )
  use Thermo_Const
  implicit none
  real, intent(in) :: pres
  real, intent(in) :: temp

  Dv=2.23e-5*(p00/pres)*(temp/ti0)**1.81

  return

end function Dv

real function qvss( temp, pres )
  use Thermo_Const
  implicit none
  real, intent(in) :: temp
  real, intent(in) :: pres
  real :: eps

  eps=Rd/Rv
  qvss=eps*tetens(temp)/(pres)

  return

end function

!--------------------------------------
! warm bulk physics
!--------------------------------------

subroutine Moist_Sature_Adjust( pres, pt, qv, qc, err )
  use Thermo_Const
  implicit none
  real, intent(in) :: pres ! 圧力 [Pa]
  real, intent(inout) :: pt   ! 温位 [K]
  real, intent(inout) :: qv   ! 水蒸気混合比 [kg kg-1]
  real, intent(inout) :: qc   ! 雲水混合比 [kg kg-1]
  real, intent(in), optional :: err  ! convergence limit [default = 1.0e-8]
  real :: tmppt, tmpqv, tmpqc, dqc, temp, gam, coe, pt1, qv1, qc1, err_max
  real :: tmp_err

  if(qv<0.0.or.qc<0.0)then
     write(*,*) "ERROR : Argument in Moist_Sature_Adjust must be greater than zero."
     write(*,*) "STOP."
     stop
  end if

  if(present(err))then
     err_max=err
  else
     err_max=1.0e-8
  end if

  temp=thetaP_2_T( pt, pres )
  dqc=qv-qvss( temp, pres )
  gam=Lv( temp )/(Cpd*exner_func_dry( pres ))
  pt1=pt
  qv1=qv
  qc1=qc

  do while (dqc>0.0.or.qc>0.0)
     coe=qvss( temp, pres )*  &
  &      (Cpd/(pt1*Rd)+17.269*(t0-35.86)/(exner_func_dry( pres )*(temp-t0)**2))
     tmppt=pt1+gam*(dqc)/(1.0+gam*coe)
     tmpqv=qv1+(pt1-tmppt)/gam
     tmpqc=qv1+qc1-tmpqv
     if(tmpqc>0.0)then
        pt1=tmppt
        qv1=tmpqv
        qc1=tmpqc
     else if(tmpqc<=0.0)then
        pt1=pt1-gam*qc1
        qv1=qv1+qc1
        qc1=0.0
        exit
     end if
     temp=thetaP_2_T( pt1, pres )
     dqc=qv-qvss( temp, pres )

     tmp_err=abs(tmppt-pt1)
     if(err_max>tmp_err)then
        exit
     end if
  end do

  pt=pt1
  qv=qv1
  qc=qc1

end subroutine Moist_Sature_Adjust

real function CNcr( qc, a )
  implicit none
  real, intent(in) :: qc
  real, intent(in), optional :: a
  real, parameter :: k1=1.0e-3
  real :: tmp

  if(present(a))then
     tmp=a
  else
     tmp=1.0e-3
  end if

  if(qc>tmp)then

     CNcr=k1*(qc-tmp)

  else

     CNcr=0.0

  end if

  return

end function CNcr

real function CLcr( qc, qr )
  implicit none
  real, intent(in) :: qc
  real, intent(in) :: qr
  real, parameter :: k2=2.2

  CLcr=k2*qc*(qr**0.875)

  return

end function CLcr

real function EVrv( temp, qv, qr, p )
  implicit none
  real, intent(in) :: temp  ! 温度 [K]
  real, intent(in) :: qv    ! 水蒸気混合比 [kg/kg]
  real, intent(in) :: qr    ! 雨水混合比 [kg/kg]
  real, intent(in) :: p     ! 圧力 [Pa]
  real :: C, qvs, rhob

  rhob=TP_2_rho( temp, p )
  qvs=qvss( temp, p )
  C=1.6+124.9*(rhob*qr)**0.2046

  EVrv=((1.0-qv/qvs)*C*(rhob*qr)**0.525)/(rhob*(5.4e5+2.55e6/(p*qvs)))

  return

end function EVrv

real function term_v_warm( rhob, qr, rho0 )
  implicit none
  real, intent(in) :: rhob
  real, intent(in) :: qr
  real, intent(in) :: rho0

  term_v=36.34*(rho0/rhob)*(rhob*qr)**0.1346

  return

end function term_v_warm

!-----------------------------------------
!  cold bulk parameterization
!-----------------------------------------

subroutine Moist_Sature_Adjust_cold( pres, pt, qv, qc, err )
  use Thermo_Const
  implicit none
  real, intent(in) :: pres ! 圧力 [Pa]
  real, intent(inout) :: pt   ! 温位 [K]
  real, intent(inout) :: qv   ! 水蒸気混合比 [kg kg-1]
  real, intent(inout) :: qc   ! 雲水混合比 [kg kg-1]
  real, intent(in), optional :: err  ! convergence limit [default = 1.0e-8]
  real :: tmppt, tmpqv, tmpqc, dqc, temp, gam, coe, pt1, qv1, qc1, err_max
  real :: tmp_err

  if(qv<0.0.or.qc<0.0)then
     write(*,*) "ERROR : Argument in Moist_Sature_Adjust must be greater than zero."
     write(*,*) "STOP."
     stop
  end if

  if(present(err))then
     err_max=err
  else
     err_max=1.0e-8
  end if

  temp=thetaP_2_T( pt, pres )
  dqc=qv-qvss( temp, pres )
  gam=Lv( temp )/(Cpd*exner_func_dry( pres ))
  pt1=pt
  qv1=qv
  qc1=qc

  do while (dqc>0.0.or.qc>0.0)
     coe=qvss( temp, pres )*  &
  &      (Cpd/(pt1*Rd)+17.269*(t0-35.86)/(exner_func_dry( pres )*(temp-t0)**2))
     tmppt=pt1+gam*(dqc)/(1.0+gam*coe)
     tmpqv=qv1+(pt1-tmppt)/gam
     tmpqc=qv1+qc1-tmpqv
     if(tmpqc>0.0)then
        pt1=tmppt
        qv1=tmpqv
        qc1=tmpqc
     else if(tmpqc<=0.0)then
        pt1=pt1-gam*qc1
        qv1=qv1+qc1
        qc1=0.0
        exit
     end if
     temp=thetaP_2_T( pt1, pres )
     dqc=qv-qvss( temp, pres )

     tmp_err=abs(tmppt-pt1)
     if(err_max>tmp_err)then
        exit
     end if
  end do

  pt=pt1
  qv=qv1
  qc=qc1

end subroutine Moist_Sature_Adjust_cold


!!!real function FRrg( temp, rhob, lambdar )
!!!! 雨粒の凍結
!!!  use Thermo_Const
!!!  use Cloud_Const
!!!  use Math_Const
!!!  implicit none
!!!  real, intent(in) :: temp  ! 温度 [K]
!!!  real, intent(in) :: rhob  ! 基本場の密度 [kg/m3]
!!!  real, intent(in) :: lambdar  ! 粒径分布の傾き
!!!  real :: tsp  ! 過冷却温度 [K]
!!!
!!!  tsp=ti0-temp
!!!
!!!  FRrg=20.0*pi*pi*bd*nr0*rhow*(exp(ad*tsp)-1.0)/(rhob*lambdar**7)
!!!
!!!  return
!!!
!!!end function FRrg
!!!
!!!!-----------------------------------------
!!!
!!!real function MLxr( temp, rhob, pres, qv, cat )
!!!! 雪, 霰の融解
!!!  use Math_Const
!!!  use Thermo_Const
!!!  use Cloud_Const
!!!  implicit none
!!!  real, intent(in) :: temp  ! 温度 [K]
!!!  real, intent(in) :: rhob  ! 基本場の密度 [kg/m3]
!!!  real, intent(in) :: pres  ! 圧力 [Pa]
!!!  real, intent(in) :: qv  ! 水蒸気混合比 [kg/kg]
!!!  character(1), intent(in) :: cat  ! 変換のカテゴリ
!!!                ! 's' -> 雪, 'g' -> 霰
!!!  real :: tdg  ! セルシウス温度
!!!
!!!  tdg=temp-t0
!!!
!!!  if(tdg>t0)then
!!!     MKxr=(2.0*pi*VENTx( temp, rhob, rhos, trim(cat) )  &
!!!  &       *(kpa*tdg/rhob+Lv( temp )*Dv( pres, temp )*(qv-qvs(ti0)))  &
!!!  &       +cw*tdg*(CLxy( 'c'//trim(cat) )+CLxy( 'r'//trim(cat) )))/(Lf( temp ))
!!!  else
!!!     MLxr=0.0
!!!  end if
!!!
!!!  return
!!!
!!!end function MLxr
!!!
!!!!-----------------------------------------
!!!
!!!real function MLic( dt, qi )
!!!! 雲氷の融解
!!!  implicit none
!!!  real, intent(in) :: dt  ! dtbig [s]
!!!  real, intent(in) :: qi  ! 雲氷混合比 [kg/kg]
!!!
!!!  MLic=0.5*qi/dt
!!!
!!!  return
!!!
!!!end function MLic
!!!
!!!!-----------------------------------------
!!!
!!!!!real function CLxy( )
!!!! 
!!!
!!!!!end function CLxy
!!!
!!!!-----------------------------------------
!!!


real function NUAvi()
  ! 昇華核形成での氷晶の生成を計算する.
  use Cloud_Const
  use Thermo_Const
  implicit none
  real, intent(in) :: temp  ! 温度 [K]
  real, intent(in) :: pres  ! 圧力 [Pa]
  real, intent(in) :: qv    ! 水蒸気混合比 [kg/kg]
  real :: tmpa, tmpb, tmpc


  NUAvi=

  return

end function NUAvi

real function MLic( temp, qi, dt )
  implicit none
  real, intent(in) :: temp  ! 温度 [K]
  real, intent(in) :: qi    ! 氷晶混合比 [kg/kg]
  real, intent(in) :: dt    ! 時間間隔 [s]

  if(temp>ti0)then
     MLic=0.5*qi/dt  ! Leap Frog スキームのとき.
  else
     MLic=0.0
  end if

  return

end function MLic

real function MLxr( q_type, temp, rhob, pres, qv )
  use Math_Const
  use Thermo_Const
  use Cloud_Const
  implicit none
  character(1), intent(in) :: q_type  ! s, g
  real, intent(in) :: temp  ! 温度 [K]
  real, intent(in) :: rhob  ! 平均場の密度 [kg/m^3]
  real, intent(in) :: pres  ! 圧力 [Pa]
  real, intent(in) :: qv    ! 水蒸気混合比 [kg/kg]
  real :: tc
  real, parameter :: Cww=4.17e3
! 水の融点は圧力変化に対してほとんど変化しないと仮定する. (qvss を求めるとき)

  if(temp>ti0)then
     tc=temp-ti0
     MLxr=(2.0*pi*VENTx(trim(q_type),)/(rhob*Lf(temp)))*
  &       (kappaa*tc+Lv(temp)*Dv(pres,temp)*rhob*(qv-qvss(ti0,pres)))  &
  &       +(Cww*tc*(CLcx(trim(q_type),)+CLrx(trim(q_type),)))/(Lf(temp))
  else
     MLxr=0.0
  end if

  return

end function MLxr

real function FRrg( temp, rhob )
  use Math_Const
  use Thermo_Const
  use Cloud_Const
  implicit none
  real, intent(in) :: temp  ! 温度 [K]
  real, intent(in) :: rhob  ! 平均場の密度 [kg/m^3]
  real :: ts

  ts=ti0-temp
  FRrg=20.0*pi*pi*nr0*rhow*(exp(Bigga*ts)-1.0)/(rhob*(lamr)**7)

  return

end function FRrg

real function term_vq_cold( rhob, q, rho0, q_type )
! under construction
  use Cloud_Const
  use Thermo_Const
  implicit none
  real, intent(in) :: rhob
  real, intent(in) :: qr
  real, intent(in) :: rho0


  return

end function term_v_warm

!-----------------------------------------
!  cold rain number concentration scheme
!-----------------------------------------

real function FRrgN( temp, rhob )
  use Math_Const
  use Thermo_Const
  use Cloud_Const
  implicit none
  real, intent(in) :: temp  ! 温度 [K]
  real, intent(in) :: rhob  ! 平均場の密度 [kg/m^3]
  real :: ts

  ts=ti0-temp
  FRrg=pi*Biggb*nr0*(exp(Bigga*ts)-1.0)/(6.0*rhob*(lamr)**4)

  return

end function FRrg

real function CNsg( Ns, lams, qc, rhob )  ! ここ, qc が正しいのかは不明.
  use Math_Const
  use Thermo_Const
  use Cloud_Const
  implicit none
  real, intent(in) :: Ns    ! 雪の数濃度 [1/m^3]
  real, intent(in) :: lams  ! 雪の数濃度傾き
  real, intent(in) :: qc    ! 雲水の混合比 [kg/kg]
  real, intent(in) :: rhob  ! 大気の密度 [kg/m^3]
  real :: tmpa, tmpb, tmpc, Ecs

  Ecs=(Stk()/(Stk()+0.5))**2
  tmpa=rhog/(rhog-rhos)
  tmpb=3.0*pi*rho0*((rhob*qc*Ecs*aus)**2)*gamma_func(2.0*bus+2.0)*Ns
  tmpc=8.0*rhob*(rhog-rhos)*(lams**(2.0*bus+1.0))

  CNsg=tmpa*tmpb/tmpc

  return

end function CNsg

real function CNsgN( Ns, qc, rhob )  ! ここ, qc が正しいのかは不明.
  use Math_Const
  use Thermo_Const
  use Cloud_Const
  implicit none
  real, intent(in) :: Ns    ! 雪の数濃度 [1/m^3]
  real, intent(in) :: qc    ! 雲水の混合比 [kg/kg]
  real, intent(in) :: rhob  ! 大気の密度 [kg/m^3]
  real :: tmpa, tmpb, tmpc, Ecs

  Ecs=(Stk()/(Stk()+0.5))**2
  tmpa=rho0/rhob
  tmpb=3.0*pi*aus*Ecs*rhob*qc*Ns
  tmpc=2.0*(rhog-rhos)

  CNsg=tmpa*tmpb/tmpc

  return

end function CNsgN

real function CNis( Ns, lams, qc, rhob )  ! under construction
  use Math_Const
  use Thermo_Const
  use Cloud_Const
  implicit none
  real, intent(in) :: Ns    ! 雪の数濃度 [1/m^3]
  real, intent(in) :: lams  ! 雪の数濃度傾き
  real, intent(in) :: qc    ! 雲水の混合比 [kg/kg]
  real, intent(in) :: rhob  ! 大気の密度 [kg/m^3]

  return

end function CNis

real function CNcr( Nc, qc, rhob, sigma, qcm )  ! 定式が正しいか原論文確認
! ここでは, Lin スキームを記述している.
  use Math_Const
  use Thermo_Const
  use Cloud_Const
  implicit none
  real, intent(in) :: Nc    ! 雲水の数濃度 [1/m^3]
  real, intent(in) :: qc    ! 雲水の混合比 [kg/kg]
  real, intent(in) :: rhob  ! 大気の密度 [kg/m^3]
  real, intent(in), optional :: sigma  ! 雲水数濃度の分散, default=sigma**2=0.15
  real, intent(in), optional :: qcm  ! 雲水混合比の閾値 [kg/kg], default=2.0e-3
  real :: tmp_qcr, tmp_sigma, coe

  if(present(sigma))then
     tmp_sigma=sigma
  else
     tmp_sigma=0.15
  end if

  if(present(qcm))then
     tmp_qcm=qcm
  else
     tmp_qcm=2.0e-3
  end if

  if(qc/=tmp_qcm)
     coe=Nc/(tmp_sigma*tmp_sigma*(qc-tmp_qcm))
     CNcr=(rhob*(qc-tmp_qcm)**2)*(1.2e-4+1.569e-12*(10.0**(coe)))
  else
     CNcr=0.0
  end if

  return

end function CNcr

real function AGiN( Ni, qi, rhob )
  use Math_Const
  use Thermo_Const
  use Cloud_Const
  implicit none
  real, intent(in) :: Ni    ! 雲氷の数濃度 [1/m^3]
  real, intent(in) :: qi    ! 雲氷の混合比 [kg/kg]
  real, intent(in) :: rhob  ! 大気の密度 [kg/m^3]
  real :: c1

  c1=(qi*aui*Eii*spii/rhoi)*((rho0/rhob)**(1.0/3.0))
  AGiN=-0.5*c1*Ni  ! c1 に本来かかる rhob と AGiN にかかる 1/rhob で相殺.

  return

end function AGiN

real function AGsN( Ns, qs, rhob )
  use Math_Const
  use Thermo_Const
  use Cloud_Const
  implicit none
  real, intent(in) :: Ns    ! 雪の数濃度 [1/m^3]
  real, intent(in) :: qs    ! 雪の混合比 [kg/kg]
  real, intent(in) :: rhob  ! 大気の密度 [kg/m^3]
  real, parameter :: Ibus=1610.0
  real :: c1, c2, c3

  c1=(aus*Ess*Ibus)/(2880.0*rhob)
  c2=(rhob*qs/(pi*Ns*rhos))**((2.0+bus)/3.0)
  c3=(pi**2)*(Ns**6)
  AGiN=-c1*c2*c3

  return

end function AGsN

real function VDvr( temp, pres, qv, rhob )
  use Cloud_Const
  use Math_Const
  implicit none
  real, intent(in) :: temp  ! 温度 [K]
  real, intent(in) :: pres  ! 圧力 [Pa]
  real, intent(in) :: qv    ! 水蒸気混合比 [kg/kg]
  real, intent(in) :: rhob  ! 基準密度 [kg/m3]
  real :: Sw

  Sw=qv/qvss( temp, pres )
  if(Sw<1.0)then
     VDvr=2.0*pi*(Sw-1.0)*Gw(temp, pres)*VENTr(rhob)/rhob
  else
     VDvr=0.0
  end if

  return

end function VDvr

real function Gw( temp, pres, rhob )
  use Cloud_Const
  use Math_Const
  use Thermo_Const
  implicit none
  real, intent(in) :: temp  ! 温度 [K]
  real, intent(in) :: pres  ! 圧力 [Pa]
  real, intent(in) :: rhob  ! 基準密度 [kg/m3]
  real :: tmpa, tmpb, tmpc, tmpd

  tmpa=Lv(temp)
  tmpb=kappaa*Rv*temp*temp
  tmpc=rhob*qvss(temp,pres)*Dv(pres,temp)
  tmpd=tmpa*tmpa/tmpb+1.0/tmpc
  Gw=1.0/tmpd

  return

end function Gw

real function VENTr( temp, pres, lamr, rhob, rho0 )
  use Cloud_Const
  use Math_Const
  use Thermo_Const
  implicit none
  real, intent(in) :: temp  ! 温度 [K]
  real, intent(in) :: pres  ! 圧力 [Pa]
  real, intent(in) :: lamr  ! 雨の傾き
  real, intent(in) :: rhob  ! 基準密度 [kg/m3]
  real, intent(in) :: rho0  ! 地表面での基準密度 [kg/m3]
  real :: tmp, tmpa, tmpb, tmpc

  tmp=0.5*(5.0+bur)
  tmpa=0.78/(lamr*lamr)
  tmpb=0.31*(Sc**(1.0/3.0))*sqrt(aur/nuair(pres,temp))  &
  &    *gamma_func(tmp)*sqrt(sqrt(rho0/rhob))
  tmpc=lamr**tmp
  VENTr=nr0*(tmpa+tmpb/tmpc)

  return

end function VENTr


real function UbxN( types, qx, rhob, rho0, Nx )
! 数濃度の重みをかけたカテゴリー x の終端落下速度を計算する.
! 数濃度がない場合は, y 切片を固定値で仮定して計算する.
  use Math_Const
  use Cloud_Const
  use Thermo_Const
  implicit none
  character(1), intent(in) :: types  ! 凝結物質の種類 'r' = 雨, 's' = 雪, 'g' = 霰
  real, intent(in) :: qx     ! 凝結物質の混合比 [kg/kg]
  real, intent(in) :: rhob   ! 基本場の密度 [kg/m3]
  real, intent(in) :: rhob   ! 地上における基本場の密度 [kg/m3]
  real, intent(in), optional :: Nx  ! qx という凝結物質の数濃度
  real :: tmpa, lamxx

  if(present(Nx))then
     lamxx=lamx( types(1:1), qx, rhob, Nx )
  else
     lamxx=lamx( types(1:1), qx, rhob )
  end if

  select case (types(1:1))  ! 'r' は Nx を使わないので, 場合分けしない.
  case ('r')
     tmpa=aur*gamma_func(1.0+bur)
     tmpb=lamxx**(bur)
     tmpc=(rho0/rhob)**(cur)
  case ('s')
     tmpa=aus*gamma_func(1.0+bus)
     tmpb=lamxx**(bus)
     tmpc=(rho0/rhob)**(cus)
  case ('g')
     tmpa=aug*gamma_func(1.0+bug)
     tmpb=lamxx**(bug)
     tmpc=(rho0/rhob)**(cug)
  end select

  UbxN=tmpa*tmpc/tmpb

  return

end function UbxN


real function UhxN( types, qx, rhob, rho0, Nx )
! 重量の重みをかけたカテゴリー x の終端落下速度を計算する.
! 数濃度がない場合は, y 切片を固定値で仮定して計算する.
  use Math_Const
  use Cloud_Const
  use Thermo_Const
  implicit none
  character(1), intent(in) :: types  ! 凝結物質の種類 'r' = 雨, 's' = 雪, 'g' = 霰
  real, intent(in) :: qx     ! 凝結物質の混合比 [kg/kg]
  real, intent(in) :: rhob   ! 基本場の密度 [kg/m3]
  real, intent(in) :: rhob   ! 地上における基本場の密度 [kg/m3]
  real, intent(in), optional :: Nx  ! qx という凝結物質の数濃度
  real :: tmpa, lamxx

  if(present(Nx))then
     lamxx=lamx( types(1:1), qx, rhob, Nx )
  else
     lamxx=lamx( types(1:1), qx, rhob )
  end if

  select case (types(1:1))  ! 'r' は Nx を使わないので, 場合分けしない.
  case ('r')
     tmpa=aur*gamma_func(4.0+bur)
     tmpb=6.0*lamxx**(bur)
     tmpc=(rho0/rhob)**(cur)
  case ('s')
     tmpa=aus*gamma_func(4.0+bus)
     tmpb=6.0*lamxx**(bus)
     tmpc=(rho0/rhob)**(cus)
  case ('g')
     tmpa=aug*gamma_func(4.0+bug)
     tmpb=6.0*lamxx**(bug)
     tmpc=(rho0/rhob)**(cug)
  end select

  UhxN=tmpa*tmpc/tmpb

  return

end function UhxN

!---------------------------------
! 以下, プライベート関数
!---------------------------------

real function lamx( types, qx, rhob, Nx )
! 凝結物質の指数関数分布における傾きを計算する.
! 数濃度がない場合は, y 切片を固定値で仮定して計算する.
  use Math_Const
  use Cloud_Const
  use Thermo_Const
  implicit none
  character(1), intent(in) :: types  ! 凝結物質の種類 'r' = 雨, 's' = 雪, 'g' = 霰
  real, intent(in) :: qx     ! 凝結物質の混合比 [kg/kg]
  real, intent(in) :: rhob   ! 基本場の密度 [kg/m3]
  real, intent(in), optional :: Nx  ! qx という凝結物質の数濃度
  real :: ncond, tmpa, tmpb, tmpc

  if(present(Nx))then
     tmpa=pi*Nx/rhob/qx
     select case (types(1:1))  ! 'r' は Nx を使わないので, 場合分けしない.
     case ('s')
        lamx=(tmpa*rhos)**(1.0/3.0)
     case ('g')
        lamx=(tmpa*rhog)**(1.0/3.0)
     end select
  else
     tmpa=pi/rhob/qx
     select case (types(1:1))  ! 'r' は Nx を使わないので, 場合分けしない.
     case ('r')
        lamx=(tmpa*rhow*nr0)**(0.25)
     case ('s')
        lamx=(tmpa*rhos*ns0)**(0.25)
     case ('g')
        lamx=(tmpa*rhog*ng0)**(0.25)
     end select
  end if

  return

end function lamx

real function nx0( types, qx, rhob, Nx )
! 凝結物質の指数関数分布における y 切片を計算する.
! 数濃度がない場合は, y 切片を固定値で仮定して計算する.
  use Math_Const
  use Cloud_Const
  use Thermo_Const
  implicit none
  character(1), intent(in) :: types  ! 凝結物質の種類 'r' = 雨, 's' = 雪, 'g' = 霰
  real, intent(in) :: qx     ! 凝結物質の混合比 [kg/kg]
  real, intent(in) :: rhob   ! 基本場の密度 [kg/m3]
  real, intent(in), optional :: Nx  ! qx という凝結物質の数濃度
  real :: ncond, tmpa, tmpb, tmpc

  if(present(Nx))then
     tmpa=pi*Nx/rhob/qx
     select case (types(1:1))  ! 'r' は Nx を使わないので, 場合分けしない.
     case ('s')
        lamx=Nx*(tmpa*rhos)**(1.0/3.0)
     case ('g')
        lamx=Nx*(tmpa*rhog)**(1.0/3.0)
     end select
  else
     select case (types(1:1))  ! 'r' は Nx を使わないので, 場合分けしない.
     case ('r')
        nx0=nr0
     case ('s')
        nx0=ns0
     case ('g')
        nx0=ng0
     end select
  end if

  return

end function nx0

end module
