program BOGUS
! 本プログラムは Yoshino et al (2009) における PV inversion を用いた
! 力学的にバランスした台風ボーガス渦を初期値データに埋め込むための
! 初期値変換プログラムである.
! ボーガスの埋め込み方は以下のようになっている.
! = 前準備
! * 初期値大気データ (MSM or GSM or CReSS)
! * 理想化台風渦の r-z 面での力学および熱力学場データ
! * 対象とする台風のベストトラックデータ (テキスト)
! = 変換の順番
! [1] : 各データを QGPV, EPV 変換する.
!       この際, 理想化台風データは 2->3 次元に展開, 同一座標系変換しておく.
! [2] : 埋め込む台風渦のデータと初期値 GPV データの時刻対応をさせる.
!       対応づけは cores_list にテキスト形式で記載する.
!       テキストのフォーマットは
!       初期 GPV, 台風渦, BST TCcenter (lon, lat) [deg], ideal TC center (nx,ny)
!       である.
! [3] : 対応ファイルをもとに, 台風渦の QGPV, EPV をそれぞれ初期値 GPV の
!       QGPV, EPV に足し合わせる.
!       ただし, 足し合わせ方は, 初期値 GPV における PV を
!       PV=PV(x,y,z)+PV'(x,y,z,t)=PV(x,y,z)+(PVa(x,y,z,t)+P''(x,y,z,t))
!       という時間平均と偏差に分け, PVa 成分が台風渦に伴う PV であると判断する.
!       対して, 台風渦の方は
!       PV=PV(z)+PV'(r,z)
!       と分解し, PV' の方を 3 次元展開して初期値 GPV の PV' と置き換える.
! [4] : QGPV, EPV の場の両方について [3] の作業を行い,
!       修正された PV 場を持った初期値 GPV データを作成する.
! [5] : 修正 PV 場の初期値 GPV データを QGPV inversion する.
! [6] : [5] で得られた力学, 熱力学場を第一推定値として EPV inversion する.
! [7] : 得られた力学, 熱力学場を修正された初期値 GPV データとして, 数値モデルの
!       初期値・境界値データにする.
! [注意] : 本プログラムでは, inversion の際に EPV のオリジナル形式を用いている
!          ため, 水蒸気場については修正することができない.
!          Davis and Emanuel (1999) などには EPV を用いて水蒸気も修正する
!          方法が提案されているが, ある程度大きな仮定を置かなければならないため,
!          現段階では実装を行っていない.
!          この点については, Schebert et al. (2001) の Mpisture PV を
!          EPV の代わりに用いて, 水蒸気場も修正することが考えられ,
!          現在, 実装を試みている.
!          理想化台風渦と初期値 GPV は任意の時刻で埋め込みが可能である.
!          現在, 理想化台風渦の最低中心気圧とベストトラックの最低中心気圧が
!          もっとも近い値になっている際の台風渦を埋め込みに用いることにしている.
! 
! = 各データの PV 変換
! * 理想化台風渦と初期値データの PV を QG 系とプリミティブ系で作成する.
!   * 台風渦データについては, r-z の 2 次元データを 3 次元に展開する.

  use file_operate
  use Basis
  use Statistics
  use Math_Const
  use Phys_Const
  use Thermo_Const
  use Map_Function
  use Thermo_Function
  use Thermo_Advanced_Routine
  use sub_calc
  use typhoon_analy

  implicit none

  integer, parameter :: nthetar=100
  real, parameter :: undef=-999.0
  real, parameter :: eps=1.0e-1
  real, parameter :: eppsi=1.0e-3

  !-- namelist variables
  ! &main
  character(1000) :: cores_list

  ! &tcdata
  integer :: nrr, nzr, nxr, nyr
  real :: drr, dxr, dyr, corilr
  real :: xmin, ymin
  real, allocatable, dimension(:) :: z_tc, z_tc_rev
  character(2) :: coor
  character(20) :: ordr
  logical :: zr_flag

  ! &gpvdata
  integer :: nxg, nyg, nzg
  integer, dimension(2) :: nnxg, nnyg, nnzg
  real :: dlon, dlat
  real :: lonmin, latmin
  real, allocatable, dimension(:) :: z_gpv, z_gpv_rev
  character(2) :: coog
  character(20) :: ordg
  character(1000) :: corilg_file
  logical :: zg_flag

  integer :: i, j, k, l, nl, ncr, ncg, itmpr, iz
  integer, dimension(10) :: counterg, counterr
  integer, allocatable, dimension(:) :: itcx, itcy, igx, igy

  real :: dist, tmpr, val, arg, meanpsi
  real, allocatable, dimension(:) :: rr, xr, yr, thetar
  real, allocatable, dimension(:) :: lon, lat, pexg
  real, allocatable, dimension(:) :: t_refg, rho_refg, rho_refrg
  real, allocatable, dimension(:) :: pgrmean, phigrmean
  real, allocatable, dimension(:,:) :: sxr, syr, sxg, syg, a1, a2, corr, corg
  real, allocatable, dimension(:,:) :: posix, posiy, posixg, posiyg
  real, allocatable, dimension(:,:) :: utr, vtr, ptr, ttr, phitr, qtr
  real, allocatable, dimension(:,:) :: epvtr, qgpvtr
  real, allocatable, dimension(:,:,:) :: ur, vr, zr, phir, pr, tr, sr, qr
  real, allocatable, dimension(:,:,:) :: urrg, vrrg, phirg, prg, trg, srg, qrg
  real, allocatable, dimension(:,:,:) :: epvrg, qgpvrg, rhocg, rhog, pr_rev
  real, allocatable, dimension(:,:,:) :: urg, vrg, rhorg, zrg
  real, allocatable, dimension(:,:,:) :: utga, vtga, pga, phiga, tga, qga
  real, allocatable, dimension(:,:,:) :: epvga, qgpvga, utg, vtg
  real, allocatable, dimension(:,:,:) :: rub, rvb, rtb, qgpvb, rsb
  real, allocatable, dimension(:,:,:) :: rphib, qgpsib, tmp, zeta
  real, allocatable, dimension(:,:,:) :: bo_psi, tmpg, tmprr
  real, allocatable, dimension(:,:,:,:) :: ug, vg, zg, phig, pg, tg, sg, qg
  real, allocatable, dimension(:,:,:,:) :: qgpvg, epvg, qgphit, qgpvt, qgtt
  real, allocatable, dimension(:,:,:,:) :: qgpsit

  character(2) :: moistcg, moistcr
  character(1000), allocatable, dimension(:,:) :: cval

  logical :: sg_flag, tg_flag, sr_flag, tr_flag

  sg_flag=.false.
  sr_flag=.false.
  tg_flag=.false.
  tr_flag=.false.
  counterg=0
  counterr=0

  write(*,*) "Starting program."

  write(*,*) "Reading namelist."
  !-- reading namelist
  namelist /main /cores_list
  namelist /tcdata /nrr, nxr, nyr, nzr, xmin, ymin, drr, dxr, dyr,  &
  &                 ordr, zr_flag, coor, corilr
  namelist /gpvdata /nxg, nyg, nzg, lonmin, latmin, dlon, dlat,  &
  &                  ordg, zg_flag, coog, nnxg, nnyg, nnzg, corilg_file
  read(5,nml=main)
  read(5,nml=tcdata)
  read(5,nml=gpvdata)

  allocate(z_tc(nzr))
  allocate(z_tc_rev(nzr))
  allocate(z_gpv(nzg))
  allocate(z_gpv_rev(nzg))

  namelist /option /z_tc, z_gpv
  read(5,nml=option)

  z_gpv_rev=-z_gpv
  z_tc_rev=-z_tc

  ncg=len_trim(adjustl(ordg))
  ncr=len_trim(adjustl(ordr))

  nl=line_number_counter( trim(cores_list) )

  write(*,*) "Allocate all arrays."
  !-- allocate arrays
  allocate(xr(nxr))
  allocate(yr(nyr))
  allocate(rr(nrr))
  allocate(thetar(nthetar))
  allocate(lon(nxg))
  allocate(lat(nyg))
  allocate(pexg(nzg))
  allocate(t_refg(nzg))
  allocate(rho_refg(nzg))
  allocate(rho_refrg(nzg))
  allocate(pgrmean(nzg))
  allocate(phigrmean(nzg))
  allocate(itcx(nl))
  allocate(itcy(nl))
  allocate(igx(nl))
  allocate(igy(nl))

  allocate(cval(6,nl))

  allocate(sxr(nxr,nyr))
  allocate(syr(nxr,nyr))
  allocate(corr(nxr,nyr))
  allocate(posix(nxr,nyr))
  allocate(posiy(nxr,nyr))
  allocate(sxg(nxg,nyg))
  allocate(syg(nxg,nyg))
  allocate(corg(nxg,nyg))
  allocate(a1(nxg,nyg))
  allocate(a2(nxg,nyg))
  allocate(posixg(nxg,nyg))
  allocate(posiyg(nxg,nyg))

  allocate(ur(nxr,nyr,nzr))
  allocate(vr(nxr,nyr,nzr))
  allocate(zr(nxr,nyr,nzr))
  allocate(phir(nxr,nyr,nzr))
  allocate(pr(nxr,nyr,nzr))
  allocate(pr_rev(nxr,nyr,nzr))
  allocate(tr(nxr,nyr,nzr))
  allocate(sr(nxr,nyr,nzr))
  allocate(qr(nxr,nyr,nzr))
  allocate(tmprr(nxr,nyr,nzr))
  allocate(rhorg(nxr,nyr,nzg))
  allocate(urg(nxr,nyr,nzg))
  allocate(vrg(nxr,nyr,nzg))
  allocate(urrg(nxr,nyr,nzg))
  allocate(vrrg(nxr,nyr,nzg))
  allocate(phirg(nxr,nyr,nzg))
  allocate(zrg(nxr,nyr,nzg))
  allocate(prg(nxr,nyr,nzg))
  allocate(trg(nxr,nyr,nzg))
  allocate(srg(nxr,nyr,nzg))
  allocate(qrg(nxr,nyr,nzg))
  allocate(epvrg(nxr,nyr,nzg))
  allocate(qgpvrg(nxr,nyr,nzg))
  allocate(rhocg(nxg,nyg,nzg))
  allocate(rhog(nxg,nyg,nzg))
  allocate(utg(nxg,nyg,nzg))
  allocate(vtg(nxg,nyg,nzg))
  allocate(utga(nxg,nyg,nzg))
  allocate(vtga(nxg,nyg,nzg))
  allocate(pga(nxg,nyg,nzg))
  allocate(phiga(nxg,nyg,nzg))
  allocate(tga(nxg,nyg,nzg))
  allocate(qga(nxg,nyg,nzg))
  allocate(epvga(nxg,nyg,nzg))
  allocate(qgpvga(nxg,nyg,nzg))
  allocate(rub(nxg,nyg,nzg))
  allocate(rvb(nxg,nyg,nzg))
  allocate(rtb(nxg,nyg,nzg))
  allocate(qgpvb(nxg,nyg,nzg))
  allocate(rsb(nxg,nyg,nzg))
  allocate(rphib(nxg,nyg,nzg))
  allocate(qgpsib(nxg,nyg,nzg))
  allocate(tmpg(nxg,nyg,nzg))
  allocate(ug(nxg,nyg,nzg,nl))
  allocate(vg(nxg,nyg,nzg,nl))
  allocate(zg(nxg,nyg,nzg,nl))
  allocate(phig(nxg,nyg,nzg,nl))
  allocate(pg(nxg,nyg,nzg,nl))
  allocate(tg(nxg,nyg,nzg,nl))
  allocate(sg(nxg,nyg,nzg,nl))
  allocate(qg(nxg,nyg,nzg,nl))
  allocate(epvg(nxg,nyg,nzg,nl))
  allocate(qgpvg(nxg,nyg,nzg,nl))
  allocate(qgphit(nxg,nyg,nzg,nl))
  allocate(qgpvt(nxg,nyg,nzg,nl))
  allocate(qgtt(nxg,nyg,nzg,nl))
  allocate(qgpsit(nxg,nyg,nzg,nl))
  allocate(tmp(nxg,nyg,nzg))
  allocate(zeta(nxg,nyg,nzg))
  allocate(bo_psi(nxg,nyg,nzg))

  allocate(utr(nrr,nzg))
  allocate(vtr(nrr,nzg))
  allocate(ptr(nrr,nzg))
  allocate(phitr(nrr,nzg))
  allocate(ttr(nrr,nzg))
  allocate(qtr(nrr,nzg))
  allocate(epvtr(nrr,nzg))
  allocate(qgpvtr(nrr,nzg))

  !-- setting coordinates
  xr=(/((xmin+dxr*real(i-1)),i=1,nxr)/)
  yr=(/((ymin+dyr*real(i-1)),i=1,nyr)/)
  rr=(/((drr*real(i-1)),i=1,nrr)/)
  thetar=(/((2.0*pi*real(i-1)/real(nthetar)),i=1,nthetar)/)
  lon=(/((lonmin+dlon*real(i-1)),i=1,nxg)/)
  lat=(/((latmin+dlat*real(i-1)),i=1,nyg)/)
  if(zg_flag.eqv..false.)then
     do k=1,nzg
        pexg(k)=exner_func_dry( z_gpv(k) )
     end do
     rhocg=-g
  end if

  !-- setting scaling factor and coriolis parameter
  if(coor(1:2)=='ll')then
     xr=xr*pi/180.0
     yr=yr*pi/180.0
     do j=1,nyr
        do i=1,nxr
           sxr(i,j)=radius*cos(yr(j))
           syr(i,j)=radius
        end do
     end do
  else if(coor(1:2)=='xy')then
     sxr=1.0
     syr=1.0
  end if

  corr=2.0*omega*sin(corilr*pi/180.0)

  if(coog(1:2)=='ll')then
     lon=lon*pi/180.0
     lat=lat*pi/180.0
     do j=1,nyg
        do i=1,nxg
           sxg(i,j)=radius*cos(lat(j))
           syg(i,j)=radius
           corg(i,j)=2.0*omega*sin(lat(j))
           a1(i,j)=cos(lat(j))
           a2(i,j)=-sin(lat(j))*cos(lat(j))
        end do
     end do
  else if(coog(1:2)=='xy')then
     sxg=1.0
     syg=1.0
     call read_file( trim(corilg_file), nxg, nyg, 1, corg )
  end if

  write(*,*) "Reading cores_list."
  !-- reading cores_list
  call read_file_text( trim(cores_list), 6, nl, cval )

  write(*,*) "Reading tcdata and converting to QGPV and EPV ..."
  do i=1,nl
     write(*,*) "Reading gpvdata : "//trim(cval(1,i))//' ...'
     !-- reading gpvdata
     do l=1,ncg
        select case(ordg(l:l))
        case ('z')
           counterg(1)=counterg(1)+1
           call read_file_3d( trim(cval(1,i)), nxg, nyg, nzg, (l-1)*nzg+1,  &
  &                           zg(:,:,:,i) )
           phig(:,:,:,i)=g*zg(:,:,:,i)
        case ('g')
           counterg(2)=counterg(2)+1
           call read_file_3d( trim(cval(1,i)), nxg, nyg, nzg, (l-1)*nzg+1,  &
  &                           phig(:,:,:,i) )
           zg(:,:,:,i)=phig(:,:,:,i)/g
        case ('u')
           counterg(3)=counterg(3)+1
           call read_file_3d( trim(cval(1,i)), nxg, nyg, nzg, (l-1)*nzg+1,  &
  &                           ug(:,:,:,i) )
        case ('v')
           counterg(4)=counterg(4)+1
           call read_file_3d( trim(cval(1,i)), nxg, nyg, nzg, (l-1)*nzg+1,  &
  &                           vg(:,:,:,i) )
!        case ('w')
!           call read_file_3d( trim(cval(1,i)), nxg, nyg, nzg, (l-1)*nzg+1,  &
!  &                           wg(:,:,:,i) )
        case ('p')
           counterg(5)=counterg(5)+1
           if(counterg(5)==1)then
              call read_file_3d( trim(cval(1,i)), nxg, nyg, nzg, (l-1)*nzg+1,  &
  &                              pg(:,:,:,i) )
           else if(counterg(5)>1)then
              tmpg(:,:,:)=pg(:,:,:,i)
              call read_file_3d( trim(cval(1,i)), nxg, nyg, nzg, (l-1)*nzg+1,  &
  &                              pg(:,:,:,i) )
              pg(:,:,:,i)=pg(:,:,:,i)+tmpg(:,:,:)
           end if
        case ('t')
           counterg(6)=counterg(6)+1
           if(counterg(6)==1)then
              call read_file_3d( trim(cval(1,i)), nxg, nyg, nzg, (l-1)*nzg+1,  &
  &                              tg(:,:,:,i) )
           else if(counterg(6)>1)then
              tmpg(:,:,:)=tg(:,:,:,i)
              call read_file_3d( trim(cval(1,i)), nxg, nyg, nzg, (l-1)*nzg+1,  &
  &                              tg(:,:,:,i) )
              tg(:,:,:,i)=tg(:,:,:,i)+tmpg(:,:,:)
           end if
           tg_flag=.true.
        case ('s')
           counterg(7)=counterg(7)+1
           if(counterg(7)==1)then
              call read_file_3d( trim(cval(1,i)), nxg, nyg, nzg, (l-1)*nzg+1,  &
  &                              sg(:,:,:,i) )
           else if(counterg(7)>1)then
              tmpg(:,:,:)=sg(:,:,:,i)
              call read_file_3d( trim(cval(1,i)), nxg, nyg, nzg, (l-1)*nzg+1,  &
  &                              sg(:,:,:,i) )
              sg(:,:,:,i)=sg(:,:,:,i)+tmpg(:,:,:)
           end if
           sg_flag=.true.
        case ('q','r')
           counterg(8)=counterg(8)+1
           call read_file_3d( trim(cval(1,i)), nxg, nyg, nzg, (l-1)*nzg+1,  &
  &                           qg(:,:,:,i) )
           if(ordg(l:l)=='q')then
              moistcg='QV'
           else
              moistcg='RH'
           end if
        end select
     end do

     do k=1,nzg
        do j=1,nyg
           do l=1,nxg
              if(zg_flag.eqv..true.)then
                 zg(l,j,k,i)=z_gpv(k)
                 phig(l,j,k,i)=zg(l,j,k,i)*g
              else
                 pg(l,j,k,i)=z_gpv(k)
              end if
              if((tg_flag.eqv..true.).and.(sg_flag.eqv..false.))then
                 if(zg_flag.eqv..true.)then
                    sg(l,j,k,i)=theta_dry( tg(l,j,k,i), pg(l,j,k,i) )
                 else
                    sg(l,j,k,i)=theta_dry( tg(l,j,k,i), z_gpv(k) )
                 end if
              else if((tg_flag.eqv..false.).and.(sg_flag.eqv..true.))then
                 if(zg_flag.eqv..false.)then
                    tg(l,j,k,i)=thetaP_2_T( sg(l,j,k,i), pg(l,j,k,i) )
                 else
                    tg(l,j,k,i)=thetaP_2_T( sg(l,j,k,i), z_gpv(k) )
                 end if
              end if
           end do
        end do
     end do

     !-- setting TC center data from BST.
     igx(i)=c2r_convert( trim(adjustl(cval(3,i))) )
     igy(i)=c2r_convert( trim(adjustl(cval(4,i))) )
     itcx(i)=c2i_convert( trim(adjustl(cval(5,i))) )
     itcy(i)=c2i_convert( trim(adjustl(cval(6,i))) )

  end do

  !-- calculating basic state of GPV.
!$omp parallel default(shared)
!$omp do schedule(runtime) private(k)
  do k=1,nzg
     call Mean_3d( tg(nnxg(1):nnxg(2),nnyg(1):nnyg(2),k,1:nl), t_refg(k) )
  end do
!$omp end do
!$omp end parallel

  if(zg_flag.eqv..true.)then
     do k=1,nzg
        do l=1,nl
           do j=nnyg(1),nnyg(2)
              do i=nnxg(1),nnxg(2)
                 rhog(i,j,l)=TP_2_rho( thetaP_2_T( tg(i,j,k,l), pg(i,j,k,l) ),  &
  &                                    pg(i,j,k,l) )
              end do
           end do
        end do
        call Mean_3d( rhog(nnxg(1):nnxg(2),nnyg(1):nnyg(2),1:nl), rho_refg(k) )
     end do
  end if

  do i=1,nl
     counterr=0
     write(*,*) "Reading tcfile : "//trim(cval(2,i))//' ...'
     !-- reading tcdata
     do l=1,ncr
        select case(ordr(l:l))
        case ('z')
           counterr(1)=counterr(1)+1
           call read_file_3d( trim(cval(2,i)), nxr, nyr, nzr, (l-1)*nzr+1,  &
  &                           zr )
           phir=zr*g
        case ('g')
           counterr(2)=counterr(2)+1
           call read_file_3d( trim(cval(2,i)), nxr, nyr, nzr, (l-1)*nzr+1,  &
  &                           phir )
           zr=phir/g
        case ('u')
           counterr(3)=counterr(3)+1
           call read_file_3d( trim(cval(2,i)), nxr, nyr, nzr, (l-1)*nzr+1,  &
  &                           ur )
        case ('v')
           counterr(4)=counterr(4)+1
           call read_file_3d( trim(cval(2,i)), nxr, nyr, nzr, (l-1)*nzr+1,  &
  &                           vr )
!        case ('w')
!           call read_file_3d( trim(cval(2,i)), nxr, nyr, nzr, (l-1)*nzr+1,  &
!  &                           wr )
        case ('p')
           counterr(5)=counterr(5)+1
           if(counterr(5)==1)then
              call read_file_3d( trim(cval(2,i)), nxr, nyr, nzr, (l-1)*nzr+1,  &
  &                              pr )
           else if(counterr(5)>1)then
              tmprr=pr
              call read_file_3d( trim(cval(2,i)), nxr, nyr, nzr, (l-1)*nzr+1,  &
  &                              pr )
              pr=pr+tmprr
           end if
           pr_rev=-pr
        case ('t')
           counterr(6)=counterr(6)+1
           if(counterr(6)==1)then
              call read_file_3d( trim(cval(2,i)), nxr, nyr, nzr, (l-1)*nzr+1,  &
  &                              tr )
           else if(counterr(6)>1)then
              tmprr=tr
              call read_file_3d( trim(cval(2,i)), nxr, nyr, nzr, (l-1)*nzr+1,  &
  &                              tr )
              tr=tr+tmprr
           end if
           tr_flag=.true.
        case ('s')
           counterr(7)=counterr(7)+1
           if(counterr(7)==1)then
              call read_file_3d( trim(cval(2,i)), nxr, nyr, nzr, (l-1)*nzr+1,  &
  &                              sr )
           else if(counterr(7)>1)then
              tmprr=sr
              call read_file_3d( trim(cval(2,i)), nxr, nyr, nzr, (l-1)*nzr+1,  &
  &                              sr )
              sr=sr+tmprr
           end if
           sr_flag=.true.
        case ('q','r')
           counterr(8)=counterr(8)+1
           call read_file_3d( trim(cval(2,i)), nxr, nyr, nzr, (l-1)*nzr+1,  &
  &                           qr )
           if(ordr(l:l)=='q')then
              moistcr='QV'
           else
              moistcr='RH'
           end if
        end select
     end do

     do k=1,nzr
        do j=1,nyr
           do l=1,nxr
              if(zr_flag.eqv..true.)then
                 zr(l,j,k)=z_tc(k)
                 phir(l,j,k)=zr(l,j,k)*g
              else
                 pr(l,j,k)=z_tc(k)
              end if
              if((tr_flag.eqv..true.).and.(sr_flag.eqv..false.))then
                 if(zr_flag.eqv..true.)then
                    sr(l,j,k)=theta_dry( tr(l,j,k), pr(l,j,k) )
                 else
                    sr(l,j,k)=theta_dry( tr(l,j,k), z_tc(k) )
                 end if
              else if((tr_flag.eqv..false.).and.(sr_flag.eqv..true.))then
                 if(zr_flag.eqv..true.)then
                    tr(l,j,k)=thetaP_2_T( sr(l,j,k), pr(l,j,k) )
                 else
                    tr(l,j,k)=thetaP_2_T( sr(l,j,k), z_tc(k) )
                 end if
              end if
              if(moistcg(1:2)=='RH'.and.moistcr(1:2)=='QV')then
                 if(zr_flag.eqv..true.)then
                    qr(l,j,k)=qvTP_2_RH( qr(l,j,k), tr(l,j,k), pr(l,j,k) )
                 else
                    qr(l,j,k)=qvTP_2_RH( qr(l,j,k), tr(l,j,k), z_tc(k) )
                 end if
              else if(moistcg(1:2)=='QV'.and.moistcr(1:2)=='RH')then
                 if(zr_flag.eqv..true.)then
                    qr(l,j,k)=RHTP_2_qv( qr(l,j,k), tr(l,j,k), pr(l,j,k) )
                 else
                    qr(l,j,k)=RHTP_2_qv( qr(l,j,k), tr(l,j,k), z_tc(k) )
                 end if
              end if
           end do
        end do
     end do

!-- calculating each PV
     if(zg_flag.eqv..true.)then  ! gpv : Z-coord (Now Not Supported)

        do k=1,nzg
           do j=1,nyg
              do l=1,nxg
                 rhocg(l,j,k)=1.0/TP_2_rho( tg(l,j,k,i), pg(l,j,k,i) )
              end do
           end do
        end do
        !-- sg = potent. temp., phig = pres.
        call HEPV( lon, lat, z_gpv, ug(:,:,:,i), vg(:,:,:,i), rhocg,  &
  &                sg(:,:,:,i), corg, epvg(:,:,:,i), sx=sxg, sy=syg )
        call HQGPV( lon, lat, z_gpv, pg(:,:,:,i), t_refg, corg,  &
  &                 qgpvg(:,:,:,i), hx=sxg, hy=syg, rhoc=rho_refg )

        if(zr_flag.eqv..false.)then  ! tc : P-coord (P->Z)

           do j=1,nyr
              do l=1,nxr
                 call auto_interpolation_1d( zr(l,j,1:nzr), z_gpv(nnzg(1):nnzg(2)),  &
  &                                          ur(l,j,1:nzr), urg(l,j,nnzg(1):nnzg(2)) )
                 call auto_interpolation_1d( zr(l,j,1:nzr), z_gpv(nnzg(1):nnzg(2)),  &
  &                                          vr(l,j,1:nzr), vrg(l,j,nnzg(1):nnzg(2)) )
                 call auto_interpolation_1d( zr(l,j,1:nzr), z_gpv(nnzg(1):nnzg(2)),  &
  &                                          z_tc(1:nzr), prg(l,j,nnzg(1):nnzg(2)) )
                 call auto_interpolation_1d( zr(l,j,1:nzr), z_gpv(nnzg(1):nnzg(2)),  &
  &                                          tr(l,j,1:nzr), trg(l,j,nnzg(1):nnzg(2)) )
                 call auto_interpolation_1d( zr(l,j,1:nzr), z_gpv(nnzg(1):nnzg(2)),  &
  &                                          sr(l,j,1:nzr), srg(l,j,nnzg(1):nnzg(2)) )
                 call auto_interpolation_1d( zr(l,j,1:nzr), z_gpv(nnzg(1):nnzg(2)),  &
  &                                          qr(l,j,1:nzr), qrg(l,j,nnzg(1):nnzg(2)) )
                 do k=nnzg(2),nnzg(1),-1
                    call interpo_search_1d( zr(l,j,1:nzr), z_gpv(k), iz )
                    if(iz==0)then
                       trg(l,j,k)=tr(l,j,1)+6.5e-3*(zr(l,j,1)-z_gpv(k))
                       prg(l,j,k)=hypsometric_form( z_tc(1), zr(l,j,1),  &
  &                                                 tr(l,j,1), z_gpv(k) )
                       srg(l,j,k)=theta_dry( trg(l,j,k), prg(l,j,k) )
                       urg(l,j,k)=ur(l,j,1)
                       vrg(l,j,k)=vr(l,j,1)
                       qrg(l,j,k)=qr(l,j,1)
                    else if(z_gpv(k)>zr(l,j,nzr))then
                       urg(l,j,k)=undef
                       vrg(l,j,k)=undef
                       trg(l,j,k)=undef
                       prg(l,j,k)=undef
                       srg(l,j,k)=undef
                       qrg(l,j,k)=undef
                    end if
                 end do
              end do
           end do

        else  ! tc : Z-coord

           do j=1,nyr
              do l=1,nxr
                 call auto_interpolation_1d( z_tc(1:nzr), z_gpv(nnzg(1):nnzg(2)),  &
  &                                          ur(l,j,1:nzr), urg(l,j,nnzg(1):nnzg(2)) )
                 call auto_interpolation_1d( z_tc(1:nzr), z_gpv(nnzg(1):nnzg(2)),  &
  &                                          vr(l,j,1:nzr), vrg(l,j,nnzg(1):nnzg(2)) )
                 call auto_interpolation_1d( z_tc(1:nzr), z_gpv(nnzg(1):nnzg(2)),  &
  &                                          pr(l,j,1:nzr), prg(l,j,nnzg(1):nnzg(2)) )
                 call auto_interpolation_1d( z_tc(1:nzr), z_gpv(nnzg(1):nnzg(2)),  &
  &                                          tr(l,j,1:nzr), trg(l,j,nnzg(1):nnzg(2)) )
                 call auto_interpolation_1d( z_tc(1:nzr), z_gpv(nnzg(1):nnzg(2)),  &
  &                                          sr(l,j,1:nzr), srg(l,j,nnzg(1):nnzg(2)) )
                 call auto_interpolation_1d( z_tc(1:nzr), z_gpv(nnzg(1):nnzg(2)),  &
  &                                          qr(l,j,1:nzr), qrg(l,j,nnzg(1):nnzg(2)) )
                 do k=nnzg(2),nnzg(1),-1
                    call interpo_search_1d( z_tc(1:nzr), z_gpv(k), iz )
                    if(iz==0)then
                       trg(l,j,k)=tr(l,j,1)+6.5e-3*(z_tc(1)-z_gpv(k))
                       prg(l,j,k)=hypsometric_form( pr(l,j,1), z_tc(1),  &
  &                                                 tr(l,j,1), z_gpv(k) )
                       srg(l,j,k)=theta_dry( trg(l,j,k), prg(l,j,k) )
                       urg(l,j,k)=ur(l,j,1)
                       vrg(l,j,k)=vr(l,j,1)
                       qrg(l,j,k)=qr(l,j,1)
                    else if(z_gpv(k)>z_tc(nzr))then
                       urg(l,j,k)=undef
                       vrg(l,j,k)=undef
                       trg(l,j,k)=undef
                       prg(l,j,k)=undef
                       srg(l,j,k)=undef
                       qrg(l,j,k)=undef
                    end if
                 end do
              end do
           end do

        end if

        !-- calculating basic state of TR
        do k=nnzg(1),nnzg(2)
           do j=1,nyr
              do l=1,nxr
                 rhorg(l,j,k)=TP_2_rho( thetaP_2_T( trg(l,j,k), prg(l,j,k) ),  &
  &                                     prg(l,j,k) )
              end do
           end do
           call Mean_2d( rhorg(:,:,k), rho_refrg(k) )
           do j=1,nyr
              do l=1,nxr
                 rhorg(l,j,k)=1.0/rhorg(l,j,k)
              end do
           end do
        end do

        call HEPV( xr, yr, z_gpv(nnzg(1):nnzg(2)), urg(:,:,nnzg(1):nnzg(2)),  &
  &                vrg(:,:,nnzg(1):nnzg(2)), rhorg(:,:,nnzg(1):nnzg(2)),  &
  &                srg(:,:,nnzg(1):nnzg(2)), corr, epvrg(:,:,nnzg(1):nnzg(2)),  &
  &                sx=sxr, sy=syr, undeff=undef )
        call HQGPV( xr, yr, z_gpv(nnzg(1):nnzg(2)), prg(:,:,nnzg(1):nnzg(2)),  &
  &                 t_refg(nnzg(1):nnzg(2)), corr, qgpvrg(:,:,nnzg(1):nnzg(2)),  &
  &                 hx=sxr, hy=syr, rhoc=rho_refrg(nnzg(1):nnzg(2)),  &
  &                 undef=undef )

     else if(zg_flag.eqv..false.)then  ! gpv : P-coord

        !-- tg = temp., phig = gp.
        call HEPV( lon, lat, z_gpv, ug(:,:,:,i), vg(:,:,:,i), rhocg,  &
  &                sg(:,:,:,i), corg, epvg(:,:,:,i), sx=sxg, sy=syg )
        call HQGPV( lon, lat, z_gpv, phig(:,:,:,i), t_refg, corg,  &
  &                 qgpvg(:,:,:,i), hx=sxg, hy=syg )

        if(zr_flag.eqv..true.)then  ! tc : Z-coord(Z->P)
           do j=1,nyr
              do l=1,nxr
                 call auto_interpolation_1d( pr_rev(l,j,1:nzr), z_gpv_rev(nnzg(1):nnzg(2)),  &
  &                                          ur(l,j,1:nzr), urg(l,j,nnzg(1):nnzg(2)),  &
  &                                          stdopt=.true. )
                 call auto_interpolation_1d( pr_rev(l,j,1:nzr), z_gpv_rev(nnzg(1):nnzg(2)),  &
  &                                          vr(l,j,1:nzr), vrg(l,j,nnzg(1):nnzg(2)),  &
  &                                          stdopt=.true. )
                 call auto_interpolation_1d( pr_rev(l,j,1:nzr), z_gpv_rev(nnzg(1):nnzg(2)),  &
  &                                          z_tc(1:nzr), zrg(l,j,nnzg(1):nnzg(2)),  &
  &                                          stdopt=.true. )
                 call auto_interpolation_1d( pr_rev(l,j,1:nzr), z_gpv_rev(nnzg(1):nnzg(2)),  &
  &                                          tr(l,j,1:nzr), trg(l,j,nnzg(1):nnzg(2)),  &
  &                                          stdopt=.true. )
                 call auto_interpolation_1d( pr_rev(l,j,1:nzr), z_gpv_rev(nnzg(1):nnzg(2)),  &
  &                                          qr(l,j,1:nzr), qrg(l,j,nnzg(1):nnzg(2)),  &
  &                                          stdopt=.true. )
                 do k=nnzg(2),nnzg(1),-1
                    call interpo_search_1d( pr_rev(l,j,1:nzr), z_gpv_rev(k),  &
  &                                         iz, stdopt=.true. )
                    if(iz==0)then
                       trg(l,j,k)=tr(l,j,1)*(z_gpv(k)/pr(l,j,1))**(Rd*6.5e-3/g)
                       zrg(l,j,k)=z_hypsometric_form( pr(l,j,1), z_tc(1),  &
  &                                                   tr(l,j,1), z_gpv(k) )
                       srg(l,j,k)=theta_dry( trg(l,j,k), z_gpv(k) )
                       urg(l,j,k)=ur(l,j,1)
                       vrg(l,j,k)=vr(l,j,1)
                       qrg(l,j,k)=qr(l,j,1)
                       phirg(l,j,k)=g*zrg(l,j,k)
                    else if(z_gpv_rev(k)>pr_rev(l,j,nzr))then
                       urg(l,j,k)=undef
                       vrg(l,j,k)=undef
                       trg(l,j,k)=undef
                       zrg(l,j,k)=undef
                       srg(l,j,k)=undef
                       phirg(l,j,k)=undef
                       qrg(l,j,k)=undef
                    else
                       phirg(l,j,k)=g*zrg(l,j,k)
                       srg(l,j,k)=theta_dry( trg(l,j,k), z_gpv(k) )
                    end if
                 end do
              end do
           end do

        else  ! tc : P-coord

!$omp parallel default(shared)
!$omp do schedule(runtime) private(l,j)
           do j=1,nyr
              do l=1,nxr
                 call auto_interpolation_1d( z_tc_rev(1:nzr), z_gpv_rev(nnzg(1):nnzg(2)),  &
  &                                          ur(l,j,1:nzr), urg(l,j,nnzg(1):nnzg(2)) )
                 call auto_interpolation_1d( z_tc_rev(1:nzr), z_gpv_rev(nnzg(1):nnzg(2)),  &
  &                                          vr(l,j,1:nzr), vrg(l,j,nnzg(1):nnzg(2)) )
                 call auto_interpolation_1d( z_tc_rev(1:nzr), z_gpv_rev(nnzg(1):nnzg(2)),  &
  &                                          phir(l,j,1:nzr), phirg(l,j,nnzg(1):nnzg(2)) )
                 call auto_interpolation_1d( z_tc_rev(1:nzr), z_gpv_rev(nnzg(1):nnzg(2)),  &
  &                                          tr(l,j,1:nzr), trg(l,j,nnzg(1):nnzg(2)) )
                 call auto_interpolation_1d( z_tc_rev(1:nzr), z_gpv_rev(nnzg(1):nnzg(2)),  &
  &                                          qr(l,j,1:nzr), qrg(l,j,nnzg(1):nnzg(2)) )
                 do k=nnzg(2),nnzg(1),-1
                    call interpo_search_1d( z_tc_rev(1:nzr), z_gpv_rev(k), iz )
                    if(iz==0)then
                       trg(l,j,k)=tr(l,j,1)*(z_gpv(k)/z_tc(1))**(Rd*6.5e-3/g)
                       zrg(l,j,k)=z_hypsometric_form( pr(l,j,1), z_tc(1),  &
  &                                                   tr(l,j,1), z_gpv(k) )
                       srg(l,j,k)=theta_dry( trg(l,j,k), z_gpv(k) )
                       urg(l,j,k)=ur(l,j,1)
                       vrg(l,j,k)=vr(l,j,1)
                       qrg(l,j,k)=qr(l,j,1)
                       phirg(l,j,k)=g*zrg(l,j,k)
                    else if(z_gpv_rev(k)>z_tc_rev(nzr))then
                       urg(l,j,k)=undef
                       vrg(l,j,k)=undef
                       trg(l,j,k)=undef
                       zrg(l,j,k)=undef
                       srg(l,j,k)=undef
                       phirg(l,j,k)=undef
                       qrg(l,j,k)=undef
                    else
                       zrg(l,j,k)=phirg(l,j,k)/g
                       srg(l,j,k)=theta_dry( trg(l,j,k), z_gpv(k) )
                    end if
                 end do
              end do
           end do
!$omp end do
!$omp end parallel

        end if

        !-- calculating basic state of TR
        rhorg=-g

        call HEPV( xr, yr, z_gpv(nnzg(1):nnzg(2)), urg(:,:,nnzg(1):nnzg(2)),  &
  &                vrg(:,:,nnzg(1):nnzg(2)), rhorg(:,:,nnzg(1):nnzg(2)),  &
  &                srg(:,:,nnzg(1):nnzg(2)), corr, epvrg(:,:,nnzg(1):nnzg(2)),  &
  &                sx=sxr, sy=syr, undeff=undef )
        call HQGPV( xr, yr, z_gpv(nnzg(1):nnzg(2)), prg(:,:,nnzg(1):nnzg(2)),  &
  &                 t_refg(nnzg(1):nnzg(2)), corr, qgpvrg(:,:,nnzg(1):nnzg(2)),  &
  &                 hx=sxr, hy=syr, undef=undef )

     end if

!$omp parallel default(shared)
!$omp do schedule(runtime) private(l,j,dist)

     do j=1,nyr
        do l=1,nxr
           posix(l,j)=xr(l)-xr(itcx(i))
           posiy(l,j)=yr(j)-yr(itcy(i))
           dist=sqrt(posix(l,j)**2+posiy(l,j)**2)
           if(dist/=0.0)then
              posix(l,j)=posix(l,j)/dist
              posiy(l,j)=posiy(l,j)/dist
           else
              posix(l,j)=0.0
              posiy(l,j)=0.0
           end if
        end do
     end do

!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(l,j,dist)

     do j=nnyg(1),nnyg(2)
        do l=nnxg(1),nnxg(2)
           posixg(l,j)=lon(l)-lon(igx(i))
           posiyg(l,j)=lat(j)-lat(igy(i))
           dist=sqrt(posixg(l,j)**2+posiyg(l,j)**2)
           if(dist/=0.0)then
              posixg(l,j)=posixg(l,j)/dist
              posiyg(l,j)=posiyg(l,j)/dist
           else
              posixg(l,j)=0.0
              posiyg(l,j)=0.0
           end if
        end do
     end do

!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(k)

     do k=nnzg(1),nnzg(2)
        !-- tc data (but gpv coordinate, so not zr_flag, but zg_flag)
        call dot_prod_2d( posix, posiy, urg(:,:,k), vrg(:,:,k), urrg(:,:,k),  &
  &                       undeff=undef )
        call vec_prod_2d( posix, posiy, urg(:,:,k), vrg(:,:,k), vrrg(:,:,k),  &
  &                       undeff=undef )
        call tangent_mean_scal( xr, yr, xr(itcx(i)), yr(itcy(i)), urrg(:,:,k),  &
  &                             rr, thetar, utr(:,k), undefgc='inc',  &
  &                             undefg=undef, axis=coor )
        call tangent_mean_scal( xr, yr, xr(itcx(i)), yr(itcy(i)), vrrg(:,:,k),  &
  &                             rr, thetar, vtr(:,k), undefgc='inc',  &
  &                             undefg=undef, axis=coor )
        if(zg_flag.eqv..true.)then  ! This flag is correct for zg_flag.
           call tangent_mean_scal( xr, yr, xr(itcx(i)), yr(itcy(i)), prg(:,:,k),  &
  &                                rr, thetar, ptr(:,k), undefgc='inc',  &
  &                                undefg=undef, axis=coor )
        else
           call tangent_mean_scal( xr, yr, xr(itcx(i)), yr(itcy(i)), phirg(:,:,k),  &
  &                                rr, thetar, phitr(:,k), undefgc='inc',  &
  &                                undefg=undef, axis=coor )
        end if
        call tangent_mean_scal( xr, yr, xr(itcx(i)), yr(itcy(i)), trg(:,:,k),  &
  &                             rr, thetar, ttr(:,k), undefgc='inc',  &
  &                             undefg=undef, axis=coor )
        call tangent_mean_scal( xr, yr, xr(itcx(i)), yr(itcy(i)), qrg(:,:,k),  &
  &                             rr, thetar, qtr(:,k), undefgc='inc',  &
  &                             undefg=undef, axis=coor )
        call tangent_mean_scal( xr, yr, xr(itcx(i)), yr(itcy(i)), epvrg(:,:,k),  &
  &                             rr, thetar, epvtr(:,k), undefgc='inc',  &
  &                             undefg=undef, axis=coor )
        call tangent_mean_scal( xr, yr, xr(itcx(i)), yr(itcy(i)), qgpvrg(:,:,k),  &
  &                             rr, thetar, qgpvtr(:,k), undefgc='inc',  &
  &                             undefg=undef, axis=coor )

        !-- GPV data
        call dot_prod_2d( posixg(nnxg(1):nnxg(2),nnyg(1):nnyg(2)),  &
  &                       posiyg(nnxg(1):nnxg(2),nnyg(1):nnyg(2)),  &
  &                       ug(nnxg(1):nnxg(2),nnyg(1):nnyg(2),k,i),  &
  &                       vg(nnxg(1):nnxg(2),nnyg(1):nnyg(2),k,i),  &
  &                       utg(nnxg(1):nnxg(2),nnyg(1):nnyg(2),k), undeff=undef )
        call vec_prod_2d( posixg(nnxg(1):nnxg(2),nnyg(1):nnyg(2)),  &
  &                       posiyg(nnxg(1):nnxg(2),nnyg(1):nnyg(2)),  &
  &                       ug(nnxg(1):nnxg(2),nnyg(1):nnyg(2),k,i),  &
  &                       vg(nnxg(1):nnxg(2),nnyg(1):nnyg(2),k,i),  &
  &                       vtg(nnxg(1):nnxg(2),nnyg(1):nnyg(2),k), undeff=undef )
        call tangent_mean_anom_scal_Cart( lon(nnxg(1):nnxg(2)),  &
  &                                       lat(nnyg(1):nnyg(2)),  &
  &                                       lon(igx(i)), lat(igy(i)),  &
  &                            utg(nnxg(1):nnxg(2),nnyg(1):nnyg(2),k),  &
  &                            rr, thetar,  &
  &                            utga(nnxg(1):nnxg(2),nnyg(1):nnyg(2),k),  &
  &                            undefgc='inc', undefg=undef, axis=coog,  &
  &                            undef=undef )
        call tangent_mean_anom_scal_Cart( lon(nnxg(1):nnxg(2)),  &
  &                                       lat(nnyg(1):nnyg(2)),  &
  &                                       lon(igx(i)), lat(igy(i)),  &
  &                            vtg(nnxg(1):nnxg(2),nnyg(1):nnyg(2),k),  &
  &                            rr, thetar,  &
  &                            vtga(nnxg(1):nnxg(2),nnyg(1):nnyg(2),k),  &
  &                            undefgc='inc', undefg=undef, axis=coog,  &
  &                            undef=undef )
        if(zg_flag.eqv..true.)then
           call tangent_mean_anom_scal_Cart( lon(nnxg(1):nnxg(2)),  &
  &                                          lat(nnyg(1):nnyg(2)),  &
  &                                          lon(igx(i)), lat(igy(i)),  &
  &                               pg(nnxg(1):nnxg(2),nnyg(1):nnyg(2),k,i),  &
  &                               rr, thetar,  &
  &                               pga(nnxg(1):nnxg(2),nnyg(1):nnyg(2),k),  &
  &                               undefgc='inc', undefg=undef, axis=coog,  &
  &                               undef=undef )
           call tangent_mean_scal( lon(nnxg(1):nnxg(2)),  &
  &                                lat(nnyg(1):nnyg(2)),  &
  &                                lon(igx(i)), lat(igy(i)),  &
  &                               pg(nnxg(1):nnxg(2),nnyg(1):nnyg(2),k,i),  &
  &                               rr(nrr:nrr), thetar,  &
  &                               pgrmean(k),  &
  &                               undefgc='inc', undefg=undef, axis=coog,  &
  &                               undef=undef )
           pgrmean(k)=ptr(nrr,k)-pgrmean(k)  ! Adjusting phi field
        else
           call tangent_mean_anom_scal_Cart( lon(nnxg(1):nnxg(2)),  &
  &                                          lat(nnyg(1):nnyg(2)),  &
  &                                          lon(igx(i)), lat(igy(i)),  &
  &                               phig(nnxg(1):nnxg(2),nnyg(1):nnyg(2),k,i),  &
  &                               rr, thetar,  &
  &                               phiga(nnxg(1):nnxg(2),nnyg(1):nnyg(2),k),  &
  &                               undefgc='inc', undefg=undef, axis=coog,  &
  &                               undef=undef )
           call tangent_mean_scal( lon(nnxg(1):nnxg(2)),  &
  &                                lat(nnyg(1):nnyg(2)),  &
  &                                lon(igx(i)), lat(igy(i)),  &
  &                               phig(nnxg(1):nnxg(2),nnyg(1):nnyg(2),k,i),  &
  &                               rr(nrr:nrr), thetar,  &
  &                               phigrmean(k),  &
  &                               undefgc='inc', undefg=undef, axis=coog,  &
  &                               undef=undef )
           phigrmean(k)=phitr(nrr,k)-phigrmean(k)  ! Adjusting phi field
        end if
        call tangent_mean_anom_scal_Cart( lon(nnxg(1):nnxg(2)),  &
  &                                       lat(nnyg(1):nnyg(2)),  &
  &                                       lon(igx(i)), lat(igy(i)),  &
  &                            tg(nnxg(1):nnxg(2),nnyg(1):nnyg(2),k,i),  &
  &                            rr, thetar,  &
  &                            tga(nnxg(1):nnxg(2),nnyg(1):nnyg(2),k),  &
  &                            undefgc='inc', undefg=undef, axis=coog,  &
  &                            undef=undef )
        call tangent_mean_anom_scal_Cart( lon(nnxg(1):nnxg(2)),  &
  &                                       lat(nnyg(1):nnyg(2)),  &
  &                                       lon(igx(i)), lat(igy(i)),  &
  &                            qg(nnxg(1):nnxg(2),nnyg(1):nnyg(2),k,i),  &
  &                            rr, thetar,  &
  &                            qga(nnxg(1):nnxg(2),nnyg(1):nnyg(2),k),  &
  &                            undefgc='inc', undefg=undef, axis=coog,  &
  &                            undef=undef )
        call tangent_mean_anom_scal_Cart( lon(nnxg(1):nnxg(2)),  &
  &                                       lat(nnyg(1):nnyg(2)),  &
  &                                       lon(igx(i)), lat(igy(i)),  &
  &                            epvg(nnxg(1):nnxg(2),nnyg(1):nnyg(2),k,i),  &
  &                            rr, thetar,  &
  &                            epvga(nnxg(1):nnxg(2),nnyg(1):nnyg(2),k),  &
  &                            undefgc='inc', undefg=undef, axis=coog,  &
  &                            undef=undef )
        call tangent_mean_anom_scal_Cart( lon(nnxg(1):nnxg(2)),  &
  &                                       lat(nnyg(1):nnyg(2)),  &
  &                                       lon(igx(i)), lat(igy(i)),  &
  &                            qgpvg(nnxg(1):nnxg(2),nnyg(1):nnyg(2),k,i),  &
  &                            rr, thetar,  &
  &                            qgpvga(nnxg(1):nnxg(2),nnyg(1):nnyg(2),k),  &
  &                            undefgc='inc', undefg=undef, axis=coog,  &
  &                            undef=undef )

     end do

!$omp end do
!$omp barrier

     !-- merging tc -> gpv

!$omp do schedule(runtime) private(l,j,k,itmpr,tmpr,val)
     do k=nnzg(1),nnzg(2)
        do j=nnyg(1),nnyg(2)
           do l=nnxg(1),nnxg(2)
              if(coog(1:2)=='xy')then
                 tmpr=sqrt((lon(l)-lon(igx(i)))**2+(lat(j)-lat(igy(i)))**2)
              else
                 tmpr=ll2radi( lon(igx(i)), lat(igy(i)), lon(l), lat(j) )
              end if

              if(rr(nrr)>tmpr)then

                 call interpo_search_1d( rr, tmpr, itmpr )

                 if(utr(itmpr,k)/=undef.and.utr(itmpr+1,k)/=undef)then
                    call interpolation_1d( rr(itmpr:itmpr+1),  &
  &                                        utr(itmpr:itmpr+1,k), tmpr, val )
                    utga(l,j,k)=utga(l,j,k)+val

                    call interpolation_1d( rr(itmpr:itmpr+1),  &
  &                                        vtr(itmpr:itmpr+1,k), tmpr, val )
                    vtga(l,j,k)=vtga(l,j,k)+val

                    if(zg_flag.eqv..true.)then
                       call interpolation_1d( rr(itmpr:itmpr+1),  &
  &                                           ptr(itmpr:itmpr+1,k), tmpr, val )
                       pga(l,j,k)=pga(l,j,k)+val-pgrmean(k)
                    else
                       call interpolation_1d( rr(itmpr:itmpr+1),  &
  &                                           phitr(itmpr:itmpr+1,k), tmpr, val )
                       phiga(l,j,k)=phiga(l,j,k)+val-phigrmean(k)
                    end if

                    call interpolation_1d( rr(itmpr:itmpr+1),  &
  &                                        ttr(itmpr:itmpr+1,k), tmpr, val )
                    tga(l,j,k)=tga(l,j,k)+val

                    call interpolation_1d( rr(itmpr:itmpr+1),  &
  &                                        qtr(itmpr:itmpr+1,k), tmpr, val )
                    qga(l,j,k)=qga(l,j,k)+val

                    call interpolation_1d( rr(itmpr:itmpr+1),  &
  &                                        epvtr(itmpr:itmpr+1,k), tmpr, val )
                    epvga(l,j,k)=epvga(l,j,k)+val

                    call interpolation_1d( rr(itmpr:itmpr+1),  &
  &                                        qgpvtr(itmpr:itmpr+1,k), tmpr, val )
                    qgpvga(l,j,k)=qgpvga(l,j,k)+val
                    !-- velocity is converting from anular to Cart.
                    if(lon(igx(i))==lon(l))then
                       if(lat(igy(i))<=lat(j))then
                          arg=0.5*pi
                       else
                          arg=1.5*pi
                       end if
                    else if(lat(igy(i))==lat(j))then
                       if(lon(igx(i))<=lon(l))then
                          arg=0.0
                       else
                          arg=pi
                       end if
                    else if((lon(igx(i))<lon(l)).and.(lat(igy(i))<lat(j)))then
                       arg=atan(abs(lat(j)-lat(igy(i)))/abs(lon(l)-lon(igx(i))))
                    else if((lon(igx(i))>lon(l)).and.(lat(igy(i))<lat(j)))then
                       arg=atan(abs(lat(j)-lat(igy(i)))/abs(lon(l)-lon(igx(i))))
                       arg=pi-arg
                    else if((lon(igx(i))<lon(l)).and.(lat(igy(i))>lat(j)))then
                       arg=atan(abs(lat(j)-lat(igy(i)))/abs(lon(l)-lon(igx(i))))
                       arg=2.0*pi-arg
                    else if((lon(igx(i))>lon(l)).and.(lat(igy(i))>lat(j)))then
                       arg=atan(abs(lat(j)-lat(igy(i)))/abs(lon(l)-lon(igx(i))))
                       arg=pi+arg
                    end if
                    ug(l,j,k,i)=utga(l,j,k)*cos(arg)-vtga(l,j,k)*sin(arg)
                    vg(l,j,k,i)=utga(l,j,k)*sin(arg)+vtga(l,j,k)*cos(arg)
                    if(zg_flag.eqv..true.)then
                       pg(l,j,k,i)=pga(l,j,k)
                    else
                       phig(l,j,k,i)=phiga(l,j,k)
                    end if
                    tg(l,j,k,i)=tga(l,j,k)
                    if(qga(l,j,k)<0.0)then
                       qg(l,j,k,i)=0.0
                    else
                       if(moistcg(1:2)=='RH'.and.qga(l,j,k)>100.0)then
                          qg(l,j,k,i)=100.0
                       else
                          qg(l,j,k,i)=qga(l,j,k)
                       end if
                    end if
                    epvg(l,j,k,i)=epvga(l,j,k)
                    qgpvg(l,j,k,i)=qgpvga(l,j,k)
                 end if
              end if
           end do
        end do
     end do
!$omp end do
!$omp end parallel

     zg(:,:,:,i)=phig(:,:,:,i)/g
     !-- dumping temporary file
     call write_file_3d( trim(cval(1,i))//'.tmp', nxg, nyg, nzg, 1,  &
  &                      zg(:,:,:,i), mode='replace' )
     call write_file_3d( trim(cval(1,i))//'.tmp', nxg, nyg, nzg, nzg+1,  &
  &                      ug(:,:,:,i) )
     call write_file_3d( trim(cval(1,i))//'.tmp', nxg, nyg, nzg, 2*nzg+1,  &
  &                      vg(:,:,:,i) )
     call write_file_3d( trim(cval(1,i))//'.tmp', nxg, nyg, nzg, 3*nzg+1,  &
  &                      pg(:,:,:,i) )
     call write_file_3d( trim(cval(1,i))//'.tmp', nxg, nyg, nzg, 4*nzg+1,  &
  &                      tg(:,:,:,i) )
     call write_file_3d( trim(cval(1,i))//'.tmp', nxg, nyg, nzg, 5*nzg+1,  &
  &                      qg(:,:,:,i) )
     call write_file_3d( trim(cval(1,i))//'.tmp', nxg, nyg, nzg, 6*nzg+1,  &
  &                      epvg(:,:,:,i) )
     call write_file_3d( trim(cval(1,i))//'.tmp', nxg, nyg, nzg, 7*nzg+1,  &
  &                      qgpvg(:,:,:,i) )

     write(*,*) "Finish : "//trim(cval(1,i))//'.tmp ...'
  end do

  do k=1,nzg
     do j=nnyg(1),nnyg(2)
        do i=nnxg(1),nnxg(2)
           call Mean_1d( phig(i,j,k,:), rphib(i,j,k) )
           call Mean_1d( ug(i,j,k,:), rub(i,j,k) )
           call Mean_1d( vg(i,j,k,:), rvb(i,j,k) )
           call Mean_1d( tg(i,j,k,:), rtb(i,j,k) )
           call Mean_1d( qgpvg(i,j,k,:), qgpvb(i,j,k) )
           call Mean_1d( sg(i,j,k,:), rsb(i,j,k) )
           qgpsib(i,j,k)=rphib(i,j,k)/(corg((nnxg(2)+nnxg(1))/2,(nnyg(2)+nnyg(1))/2))
           do l=1,nl
              qgphit(i,j,k,l)=phig(i,j,k,l)-rphib(i,j,k)
              qgtt(i,j,k,l)=tg(i,j,k,l)-rtb(i,j,k)
              qgpvt(i,j,k,l)=qgpvg(i,j,k,l)-qgpvb(i,j,k)
           end do
        end do
     end do
  end do

  if(zg_flag.eqv..true.)then
     write(*,*) "*** MESSAGE (main) ***"
     write(*,*) "Now, EPV Bogus in this program is not supported for Z-coord."
     write(*,*) "Stop."
     stop
  end if

  do i=1,nl
     write(*,*) "Starting QGPV inversion ..."
  !-- calculating inversion dynamics and thermodynamics fields.
     call QGPV_inv( lon(nnxg(1):nnxg(2)), lat(nnyg(1):nnyg(2)),  &
  &                 z_gpv(nnzg(1):nnzg(2)), t_refg(nnzg(1):nnzg(2)), eps,  &
  &                 qgpvt(nnxg(1):nnxg(2),nnyg(1):nnyg(2),nnzg(1):nnzg(2),i),  &
  &                 coog, corg(nnxg(1):nnxg(2),nnyg(1):nnyg(2)),  &
  &                 qgphit(nnxg(1):nnxg(2),nnyg(1):nnyg(2),nnzg(1):nnzg(2),i),  &
  &                 qgtt(nnxg(1):nnxg(2),nnyg(1):nnyg(2),nnzg(1):nnzg(2),i),  &
  &                 tmp(nnxg(1):nnxg(2),nnyg(1):nnyg(2),nnzg(1):nnzg(2)),  &
  &                 tmp(nnxg(1):nnxg(2),nnyg(1):nnyg(2),nnzg(1):nnzg(2)),  &
  &                 qgpsit(nnxg(1):nnxg(2),nnyg(1):nnyg(2),nnzg(1):nnzg(2),i) )
     write(*,*) "Passing QGPV inversion ..."

!$omp parallel default(shared)
!$omp do schedule(runtime) private(l,j,k)
     do k=nnzg(1),nnzg(2)
        do j=nnyg(1),nnyg(2)
           do l=nnxg(1),nnxg(2)
              qgphit(l,j,k,i)=qgphit(l,j,k,i)+rphib(l,j,k)
              qgpsit(l,j,k,i)=qgpsit(l,j,k,i)+qgpsib(l,j,k)
           end do
        end do
     end do
!$omp end do
!$omp end parallel

  !-- setting psi (u,v -> zeta -> psi)
     write(*,*) "Starting calc psi ..."
     do k=nnzg(1),nnzg(2)
        call curl( lon(nnxg(1):nnxg(2)), lat(nnyg(1):nnyg(2)),  &
  &                ug(nnxg(1):nnxg(2),nnyg(1):nnyg(2),k,i),  &
  &                vg(nnxg(1):nnxg(2),nnyg(1):nnyg(2),k,i),  &
  &                zeta(nnxg(1):nnxg(2),nnyg(1):nnyg(2),k),  &
  &                hx=sxg(nnxg(1):nnxg(2),nnyg(1):nnyg(2)),  &
  &                hy=syg(nnxg(1):nnxg(2),nnyg(1):nnyg(2)) )
        do j=nnyg(1),nnyg(2)
           do l=nnxg(1),nnxg(2)
              zeta(l,j,k)=zeta(l,j,k)*(sxg(l,j))**2
           end do
        end do
        do j=nnyg(1),nnyg(2)
           bo_psi(nnxg(1),j,k)=vg(nnxg(1),j,k,i)*sxg(nnxg(1),j)
           bo_psi(nnxg(2),j,k)=vg(nnxg(2),j,k,i)*sxg(nnxg(2),j)
        end do
        do l=nnxg(1),nnxg(2)
           bo_psi(l,nnyg(1),k)=-ug(l,nnyg(1),k,i)*syg(l,nnyg(1))
           bo_psi(l,nnyg(2),k)=-ug(l,nnyg(2),k,i)*syg(l,nnyg(2))
        end do
     end do

     if(coog(1:2)=='ll')then
        do k=nnzg(1),nnzg(2)
           call Mean_2d( qgpsit(nnxg(1):nnxg(2),nnyg(1):nnyg(2),k,i), meanpsi )
write(*,*) "check psint", nnxg(1),nnxg(2),nnyg(1),nnyg(2),k,i
write(*,*) "check psit", qgpsit(nnxg(1):nnxg(2),nnyg(1):nnyg(2),k,i)
           write(*,*) "psi eps = ", eppsi*meanpsi, meanpsi
           call Ellip_Jacobi_2d( lon(nnxg(1):nnxg(2)), lat(nnyg(1):nnyg(2)),  &
  &                              zeta(nnxg(1):nnxg(2),nnyg(1):nnyg(2),k),  &
  &                              eppsi*meanpsi, '2222',  &
  &                              qgpsit(nnxg(1):nnxg(2),nnyg(1):nnyg(2),k,i),  &
  &                              bound_opt=bo_psi(nnxg(1):nnxg(2),nnyg(1):nnyg(2),k),  &
  &                              c=a1(nnxg(1):nnxg(2),nnyg(1):nnyg(2)),  &
  &                              e=a2(nnxg(1):nnxg(2),nnyg(1):nnyg(2)),  &
  &                              init_flag=.false. )
        end do
     else
        do k=nnzg(1),nnzg(2)
           call Mean_2d( qgpsit(nnxg(1):nnxg(2),nnyg(1):nnyg(2),k,i), meanpsi )
           write(*,*) "psi eps = ", eppsi*meanpsi, meanpsi
           call Ellip_Jacobi_2d( lon(nnxg(1):nnxg(2)), lat(nnyg(1):nnyg(2)),  &
  &                              zeta(nnxg(1):nnxg(2),nnyg(1):nnyg(2),k),  &
  &                              eppsi*meanpsi, '2222',  &
  &                              qgpsit(nnxg(1):nnxg(2),nnyg(1):nnyg(2),k,i),  &
  &                              bound_opt=bo_psi(nnxg(1):nnxg(2),nnyg(1):nnyg(2),k),  &
  &                              init_flag=.false. )
        end do
     end if

     write(*,*) "Finish : calculating psi."

     call EPV_varinv( lon(nnxg(1):nnxg(2)), lat(nnyg(1):nnyg(2)),  &
  &                   pexg(nnzg(1):nnzg(2)),  &
  &                   epvg(nnxg(1):nnxg(2),nnyg(1):nnyg(2),nnzg(1):nnzg(2),i),  &
  &                   qgphit(nnxg(1):nnxg(2),nnyg(1):nnyg(2),nnzg(1):nnzg(2),i),  &
  &                   sg(nnxg(1):nnxg(2),nnyg(1):nnyg(2),nnzg(1):nnzg(2),i),  &
  &                   ug(nnxg(1):nnxg(2),nnyg(1):nnyg(2),nnzg(1):nnzg(2),i),  &
  &                   vg(nnxg(1):nnxg(2),nnyg(1):nnyg(2),nnzg(1):nnzg(2),i),  &
  &                   qgpsit(nnxg(1):nnxg(2),nnyg(1):nnyg(2),nnzg(1):nnzg(2),i),  &
  &                   ini_flag=.false. )

write(*,*) "Passing EPV"
     zg(:,:,:,i)=qgphit(:,:,:,i)/g

     call write_file_3d( trim(cval(1,i))//'.einv', nxg, nyg, nzg, 1,  &
  &                      zg(:,:,:,i), mode='replace' )
     call write_file_3d( trim(cval(1,i))//'.einv', nxg, nyg, nzg, nzg+1, ug(:,:,:,i) )
     call write_file_3d( trim(cval(1,i))//'.einv', nxg, nyg, nzg, 2*nzg+1, vg(:,:,:,i) )
     call write_file_3d( trim(cval(1,i))//'.einv', nxg, nyg, nzg, 3*nzg+1, sg(:,:,:,i) )
     call write_file_3d( trim(cval(1,i))//'.einv', nxg, nyg, nzg, 4*nzg+1, qg(:,:,:,i) )
     call write_file_3d( trim(cval(1,i))//'.einv', nxg, nyg, nzg, 5*nzg+1, qgpsit(:,:,:,i) )
!     call write_file_3d( trim(cval(1,i))//'.einv', nxg, nyg, nzg, 5*nzg+1, bomg(:,:,:,i) )
     write(*,*) "Finish : ", trim(cval(1,i))//'.einv'

  end do

  write(*,*) "Stop, normally."

end program
