program calc_tendency
! SEQ で得られた値から地表面気圧の時間変化 (発達率) を計算する.
! Budget_Formula の式および, Pendergrass Willoughby (2009) の式参照.
  use Algebra
  use Statistics
  use derivation
  use Ellip_Slv
  use file_operate
  use gtool_history
  use Thermo_Const
  use Thermo_Function
  use Math_Const
  use Phys_Const

  implicit none

!-- namelist valiables
  integer :: nr, nz
  integer :: mom_flag
  real :: coril
  real :: dr, dz
  character(4) :: bc
  character(80) :: fname
  character(80) :: sound_name  ! サウンディングファイルの名前

!-- internal valiables
  real, allocatable :: r(:)  ! 動径座標
  real, allocatable :: z(:)  ! 鉛直座標
  real, allocatable :: v0(:,:)  ! 接線風速
  real, allocatable :: pres(:,:)  ! 気圧
  real, allocatable :: theta0(:,:)  ! 温位
  real, allocatable :: rho(:,:)  ! 密度
  real, allocatable :: Q(:,:)  ! 
  real, allocatable :: u2(:,:)  ! 
  real, allocatable :: w2(:,:)  ! 
  real, allocatable :: mom(:,:)  ! 
  real, allocatable :: dmomdz(:,:)  ! 
  real, allocatable :: taur(:,:)  ! 
  real, allocatable :: tauz(:,:)  ! 
  real, allocatable :: nu(:,:)  ! 
  real, allocatable :: VADVH(:,:)  ! 
  real, allocatable :: VADVV(:,:)  ! 
  real, allocatable :: VCORI(:,:)  ! 
  real, allocatable :: TADVH(:,:)  ! 
  real, allocatable :: TADVV(:,:)  ! 
  real, allocatable :: NETV(:,:)  ! 
  real, allocatable :: NETT(:,:)  ! 
  real, allocatable :: intTADV(:,:)  ! 
  real, allocatable :: intDIAQ(:,:)  ! 
  real, allocatable :: dvdr(:,:)  ! 
  real, allocatable :: dvdz(:,:)  ! 
  real, allocatable :: dptdr(:,:)  ! 
  real, allocatable :: dptdz(:,:)  ! 
  real, allocatable :: adia(:)  ! 
  real, allocatable :: diaq(:)  ! 
  real, allocatable :: dp_adia(:)  ! 
  real, allocatable :: dp_diaq(:)  ! 
  real, allocatable :: dp_totl(:)  ! 
  real :: dpmean_adia, dpmean_diaq, dpmean_totl

!-- parameter
  integer, parameter :: nnr(2)=(/1,20/)  ! dp/dt の平均を行う水平方向の領域
  real, parameter :: Cd=1.5e-3  ! 運動量交換係数
  real, parameter :: theta_const=273.16
  real, parameter :: eps=1.0e4  ! original (Tsujino et al. 2017)
  real, parameter :: dpscale=3600.0  ! Pa/h
  character(10), parameter :: cdpscale=' [hPa/h]  '  ! Pa/h

!-- tmp valiables
  integer :: i, j
  real :: coriol

!-- namelist からのパラメータの代入
  namelist /input /nr, nz, dr, dz, bc, fname, sound_name, coril, mom_flag
  read(5,nml=input)

!-- 配列の割り当て
  allocate(r(nr))
  allocate(z(nz))
  allocate(v0(nr,nz))
  allocate(pres(nr,nz))
  allocate(theta0(nr,nz))
  allocate(rho(nr,nz)) 
  allocate(Q(nr,nz)) 
  allocate(u2(nr,nz)) 
  allocate(w2(nr,nz)) 
  allocate(mom(nr,nz))
  allocate(dmomdz(nr,nz))
  allocate(taur(nr,nz))
  allocate(tauz(nr,nz))
  allocate(nu(nr,nz))
  allocate(VADVH(nr,nz))
  allocate(VADVV(nr,nz))
  allocate(VCORI(nr,nz))
  allocate(TADVH(nr,nz))
  allocate(TADVV(nr,nz))
  allocate(NETV(nr,nz))
  allocate(NETT(nr,nz))
  allocate(intTADV(nr,nz))
  allocate(intDIAQ(nr,nz))
  allocate(dvdr(nr,nz))
  allocate(dvdz(nr,nz))
  allocate(dptdr(nr,nz))
  allocate(dptdz(nr,nz))
  allocate(adia(nr))
  allocate(diaq(nr))
  allocate(dp_adia(nr))
  allocate(dp_diaq(nr))
  allocate(dp_totl(nr))

  coriol=2.0*omega*sin(coril*pi/180.0)

!-- 座標値の設定
!  r=(/(dr*real(i-1),i=1,nr)/)
!  z=(/(dz*real(i-1),i=1,nz)/)

!-- 初期値の読み込み

!  call read_file( trim(fname), nr, nz, 1, v0 )
!  call read_file( trim(fname), nr, nz, 2, p0 )
!  call read_file( trim(fname), nr, nz, 3, Q )
!  call read_file( trim(fname), nr, nz, 4, v0 )

!-- gtool による読み込み

  call HistoryGet( trim(fname), 'r', r(1:nr) )
  call HistoryGet( trim(fname), 'z', z(1:nz) )
  call HistoryGet( trim(fname), 'pres', pres(1:nr,1:nz) )
  call HistoryGet( trim(fname), 'Q', Q(1:nr,1:nz) )
  call HistoryGet( trim(fname), 'theta', theta0(1:nr,1:nz) )
  call HistoryGet( trim(fname), 'v', v0(1:nr,1:nz) )
  call HistoryGet( trim(fname), 'rho', rho(1:nr,1:nz) )

  call HistoryGet( 'result_'//trim(fname), 'u', u2(1:nr,1:nz) )
  call HistoryGet( 'result_'//trim(fname), 'w', w2(1:nr,1:nz) )

!-- Q を過熱率 (J/s) から実際の計算で必要な (K/s) に換算.
  do j=1,nz
     do i=1,nr
        Q(i,j)=Q(i,j)/Cpd
     end do
  end do

!-- 運動量ソースの計算

  mom=0.0

  select case (mom_flag)
  case (0)
     write(*,*) "### momentum source is zero."
  case (1)  ! 速度場から計算する.
     write(*,*) "### momentum source is calculating with Vt."
     dmomdz=0.0
     nu=0.0

     do i=1,nr
        call grad_1d( z, v0(i,:), tauz(i,:) )
     end do

     do j=1,nz
        call grad_1d( r, v0(:,j), taur(:,j) )
     end do

     do j=1,nz
        do i=2,nr
           nu(i,j)=200.0**2*sqrt((taur(i,j)-v0(i,j)/r(i))**2+tauz(i,j)**2)
        end do
     end do

     do j=1,nz
        do i=2,nr
           taur(i,j)=nu(i,j)*(taur(i,j)-v0(i,j)/r(i))
        end do
     end do

     do j=1,nz
        do i=1,nr
           tauz(i,j)=nu(i,j)*tauz(i,j)
        end do
     end do

     do i=1,nr
        tauz(i,1)=Cd*abs(v0(i,1))*v0(i,1)
     end do

!-- ここまでで, 応力テンソルの計算終了.
!-- 以降で乱流フラックスの計算.

     do i=1,nr
        call grad_1d( z, tauz(i,:), dmomdz(i,:) )
        do j=1,nz
           mom(i,j)=dmomdz(i,j)
        end do
     end do

     do j=2,nz
        call grad_1d( r, taur(:,j), dmomdz(:,j) )
        do i=1,nr
           mom(i,j)=mom(i,j)+dmomdz(i,j)
        end do
        do i=2,nr
           mom(i,j)=mom(i,j)+2.0*taur(i,j)/r(i)
        end do
        do i=1,nr
           mom(i,j)=mom(i,j)/rho(i,j)
        end do
     end do

!     mom=0.0
!     do i=1,nr
!        mom(i,1)=-Cd*abs(v0(i,1))*v0(i,1)
!        call grad_1d( z, mom(i,:), dmomdz(i,:) )
!     end do

  case (2)
     write(*,*) "### momentum source is writing the external file."
     call HistoryGet( trim(fname), 'mom', mom(1:nr,1:nz) )

  end select

!-- 空間勾配量の計算

  call grad_2d( r, z, v0, dvdr, dvdz )
  call grad_2d( r, z, theta0, dptdr, dptdz )

!-- 収支式の各項を計算

  do j=1,nz
     do i=1,nr
        VADVH(i,j)=-u2(i,j)*dvdr(i,j)
        VADVV(i,j)=-w2(i,j)*dvdz(i,j)
        TADVH(i,j)=-u2(i,j)*dptdr(i,j)
        TADVV(i,j)=-w2(i,j)*dptdz(i,j)
        if(r(i)==0.0)then
           VCORI(i,j)=-coriol*u2(i,j)
        else
           VCORI(i,j)=-(coriol+v0(i,j)/r(i))*u2(i,j)
        end if
        NETV(i,j)=VADVH(i,j)+VADVV(i,j)+VCORI(i,j)
        NETT(i,j)=TADVH(i,j)+TADVV(i,j)+Q(i,j)
        intTADV(i,j)=(TADVH(i,j)+TADVV(i,j))/(theta0(i,j)**2)
        intDIAQ(i,j)=Q(i,j)/(theta0(i,j)**2)
     end do
  end do

!-- 鉛直積算量の計算

  do i=1,nr
     call rectangle_int( z(1:nz), intTADV(i,1:nz), z(1), z(nz), adia(i) )
     call rectangle_int( z(1:nz), intDIAQ(i,1:nz), z(1), z(nz), diaq(i) )
     adia(i)=-g*adia(i)/(exner_func_dry( pres(i,1) )*Rd)
     diaq(i)=-g*diaq(i)/(exner_func_dry( pres(i,1) )*Rd)
     dp_adia(i)=-pres(i,1)*(1.0-exp(adia(i)))
     dp_diaq(i)=-pres(i,1)*(1.0-exp(diaq(i)))
     dp_totl(i)=-pres(i,1)*(1.0-exp(adia(i)+diaq(i)))
  end do

!-- 任意領域での水平平均

  call Mean_1d( dp_adia(nnr(1):nnr(2)), dpmean_adia )
  call Mean_1d( dp_diaq(nnr(1):nnr(2)), dpmean_diaq )
  call Mean_1d( dp_totl(nnr(1):nnr(2)), dpmean_totl )

!-- gtool による書き出し

  call HistoryCreate( file='tend_'//trim(fname), title='Diagnostic tendencies', &
    & source='test', institution='test', dims=(/'r','z'/), dimsizes=(/nr,nz/),  &
    & longnames=(/'r-coordinate','z-coordinate'/),  &
    & units=(/'m','m'/), origin=0.0, interval=0.0 )

  call HistoryPut( 'r', r )
  call HistoryPut( 'z', z )

  call HistoryAddVariable( varname='DVDT', dims=(/'r','z'/), &
    & longname='Total tendency of tangential wind', units='m/s2', xtype='float')

  call HistoryPut( 'DVDT', NETV )

  call HistoryAddVariable( varname='VADVH', dims=(/'r','z'/), &
    & longname='Horizontal advection of tangential wind',  &
    & units='m/s2', xtype='float')

  call HistoryPut( 'VADVH', VADVH )

  call HistoryAddVariable( varname='VADVV', dims=(/'r','z'/), &
    & longname='Vertical advection of tangential wind',  &
    & units='m/s2', xtype='float')

  call HistoryPut( 'VADVV', VADVV )

  call HistoryAddVariable( varname='VCORI', dims=(/'r','z'/), &
    & longname='Coriolis and centrifugal forcings',  &
    & units='m/s2', xtype='float')

  call HistoryPut( 'VCORI', VCORI )

  call HistoryAddVariable( varname='DTDT', dims=(/'r','z'/), &
    & longname='Total tendency of potential temperature',  &
    & units='K/s', xtype='float')

  call HistoryPut( 'DTDT', NETT )

  call HistoryAddVariable( varname='TADVH', dims=(/'r','z'/), &
    & longname='Horizontal advection of potential temperature',  &
    & units='K/s', xtype='float')

  call HistoryPut( 'TADVH', TADVH )

  call HistoryAddVariable( varname='TADVV', dims=(/'r','z'/), &
    & longname='Vertical advection of potential temperature',  &
    & units='K/s', xtype='float')

  call HistoryPut( 'TADVV', TADVV )

  call HistoryAddVariable( varname='TDIAQ', dims=(/'r','z'/), &
    & longname='Diabatic heating rate',  &
    & units='K/s', xtype='float')

  call HistoryPut( 'TDIAQ', Q )

  call HistoryClose

  write(*,'(a38)') "--------------- Result ---------------"
  write(*,'(a10,1PE16.8,a10)') "Dp adia = ", dpmean_adia*0.01*dpscale, cdpscale
  write(*,'(a10,1PE16.8,a10)') "Dp diaq = ", dpmean_diaq*0.01*dpscale, cdpscale
  write(*,'(a10,1PE16.8,a10)') "Delta p = ", dpmean_totl*0.01*dpscale, cdpscale
  write(*,'(a38)') "--------------- Result ---------------"

!-- 終了宣言

  write(*,*) "Program is terminated, normally."

end program
