!= Arakawa-Schubert scheme by Lord et al. (1982)
!
!= Arakawa-Schubert scheme by Lord et al. (1982)
!
! Authors::   Yoshiyuki O. Takahashi
! Version::   $Id: arakawa_schubert_L1982.f90,v 1.2 2014/02/04 10:24:42 yot Exp $ 
! Tag Name::  $Name:  $
! Copyright:: Copyright (C) GFD Dennou Club, 2008. All rights reserved.
! License::   See COPYRIGHT[link:../../../COPYRIGHT]
!

module arakawa_schubert_L1982
  !
  != Arakawa-Schubert scheme by Lord et al. (1982)
  !
  != Arakawa-Schubert scheme by Lord et al. (1982)
  !
  ! <b>Note that Japanese and English are described in parallel.</b>
  !
  ! Lord et al. (1982) による Arakawa-Schubert scheme の実装.
  !
  ! Implementation of Arakawa-Schubert scheme by Lord et al. (1982).
  !
  !
  !== References
  !
  !  Lord, S. J., W. C. Chao, and A. Arakawa,
  !    Interaction of a cumulus cloud ensemble with the large-scale environment.
  !    Part IV: The discrete model, 
  !    J. Atmos. Sci., 39, 104-113, 1982.
  !
  !== Procedures List
  !
  ! ArakawaSchubertL1982CalcCWFCrtl :: 
  ! ------------------------------- :: ------------
  ! ArakawaSchubertL1982CalcCWFCrtl :: Set critical value of cloud work function 
  !                                    presented by Lord et al. (1982)
  !
  !== NAMELIST
  !
  ! NAMELIST#moist_conv_adjust_nml
  !

  ! モジュール引用 ; USE statements
  !

  ! 格子点設定
  ! Grid points settings
  !
  use gridset, only: imax, & ! 経度格子点数. 
                             ! Number of grid points in longitude
    &                jmax, & ! 緯度格子点数. 
                             ! Number of grid points in latitude
    &                kmax    ! 鉛直層数. 
                             ! Number of vertical level

  ! 種別型パラメタ
  ! Kind type parameter
  !
  use dc_types, only: DP, &      ! 倍精度実数型. Double precision. 
    &                 STRING     ! 文字列.       Strings. 


  ! メッセージ出力
  ! Message output
  !
  use dc_message, only: MessageNotify

  ! 宣言文 ; Declaration statements
  !
  implicit none
  private


  ! 公開手続き
  ! Public procedure
  !
  public :: ArakawaSchubertL1982CalcCWFCrtl
  public :: ASL1982CalcCWFCrtl1D
  public :: ArakawaSchubertL1982Init

  ! 公開変数
  ! Public variables
  !

  ! 非公開変数
  ! Private variables
  !
  logical, save :: arakawa_schubert_L1982_inited = .false.
                              ! 初期設定フラグ. 
                              ! Initialization flag

  ! Values in Table 1 in Lord et al. (1982)
  !
  integer , parameter :: nTbl1 = 15
  real(DP), save      :: a_Tbl1Press( nTbl1 )  ! Pressure (Pa)
  real(DP), save      :: a_Tbl1CWF  ( nTbl1 )  ! Cloud work function (J kg-1 Pa-1)

  data a_Tbl1Press &
    & / 150.0d2  , 200.0d2  , 250.0d2  , 300.0d2  , 350.0d2  , 400.0d2  , 450.0d2  , &
    &   500.0d2  , 550.0d2  , 600.0d2  , 650.0d2  , 700.0d2  , 750.0d2  , 800.0d2  , &
    &   850.0d2   /

  data a_Tbl1CWF   &
    & / 1.6851d-2, 1.1686d-2, 0.7663d-2, 0.5255d-2, 0.4100d-2, 0.3677d-2, 0.3151d-2, &
    &   0.2216d-2, 0.1521d-2, 0.1082d-2, 0.0750d-2, 0.0664d-2, 0.0553d-2, 0.0445d-2, &
    &   0.0633d-2 /


  character(*), parameter:: module_name = 'arakawa_schubert_L1982'
                              ! モジュールの名称. 
                              ! Module name
  character(*), parameter:: version = &
    & '$Name:  $' // &
    & '$Id: arakawa_schubert_L1982.f90,v 1.2 2014/02/04 10:24:42 yot Exp $'
                              ! モジュールのバージョン
                              ! Module version

contains

  subroutine ArakawaSchubertL1982CalcCWFCrtl( &
    & xy_PressCldBase, xyz_PressCldTop,  &  ! (in)
    & xyz_CWFCrtl                        &  ! (out)
    & )
    !
    ! 
    !
    ! Calculate critical cloud work function by using interpolation of Table 1 
    ! by Lord et al. (1982)
    !

    ! モジュール引用 ; USE statements
    !



    ! 宣言文 ; Declaration statements
    !
    implicit none

    real(DP), intent(in ) :: xy_PressCldBase (0:imax-1, 1:jmax)
                              ! pressure at cloud base
    real(DP), intent(in ) :: xyz_PressCldTop (0:imax-1, 1:jmax, 1:kmax)
                              ! pressure at cloud top
    real(DP), intent(out) :: xyz_CWFCrtl     (0:imax-1, 1:jmax, 1:kmax)
                              ! critical cloud work function

    ! 作業変数
    ! Work variables
    !
    integer  :: xyz_Index     (0:imax-1, 1:jmax, 1:kmax)
    real(DP) :: xyz_CWFIntpled(0:imax-1, 1:jmax, 1:kmax)

    integer :: i               ! 経度方向に回る DO ループ用作業変数
                               ! Work variables for DO loop in longitude
    integer :: j               ! 緯度方向に回る DO ループ用作業変数
                               ! Work variables for DO loop in latitude
    integer :: k               ! 鉛直方向に回る DO ループ用作業変数
                               ! Work variables for DO loop in vertical direction
    integer :: l


    ! 実行文 ; Executable statement
    !

    ! 初期化確認
    ! Initialization check
    !
    if ( .not. arakawa_schubert_L1982_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    do k = 1, kmax
      do j = 1, jmax
        do i = 0, imax-1

          xyz_Index(i,j,k) = 1
          loop_search : do l = 2, nTbl1-1
            if ( a_Tbl1Press(l) < xyz_PressCldTop(i,j,k) ) then
              xyz_Index(i,j,k) = l
            end if
          end do loop_search

        end do
      end do
    end do

    do k = 1, kmax
      do j = 1, jmax
        do i = 0, imax-1

          xyz_CWFIntpled(i,j,k) =                                                   &
            &   ( a_Tbl1CWF  (xyz_Index(i,j,k)+1) - a_Tbl1CWF  (xyz_Index(i,j,k)) ) &
            & / ( a_Tbl1Press(xyz_Index(i,j,k)+1) - a_Tbl1Press(xyz_Index(i,j,k)) ) &
            & * ( xyz_PressCldTop(i,j,k)          - a_Tbl1Press(xyz_Index(i,j,k)) ) &
            & + a_Tbl1CWF(xyz_Index(i,j,k))

        end do
      end do
    end do

    do k = 1, kmax
      xyz_CWFCrtl(:,:,k) = &
        & xyz_CWFIntpled(:,:,k) * ( xy_PressCldBase - xyz_PressCldTop(:,:,k) )
    end do


  end subroutine ArakawaSchubertL1982CalcCWFCrtl

  !-------------------------------------------------------------------

  subroutine ASL1982CalcCWFCrtl1D( &
    & PressCldBase, z_PressCldTop,  &  ! (in)
    & z_CWFCrtl                        &  ! (out)
    & )
    !
    ! 
    !
    ! Calculate critical cloud work function by using interpolation of Table 1 
    ! by Lord et al. (1982)
    !

    ! モジュール引用 ; USE statements
    !



    ! 宣言文 ; Declaration statements
    !
    implicit none

    real(DP), intent(in ) :: PressCldBase
                              ! pressure at cloud base
    real(DP), intent(in ) :: z_PressCldTop (1:kmax)
                              ! pressure at cloud top
    real(DP), intent(out) :: z_CWFCrtl     (1:kmax)
                              ! critical cloud work function

    ! 作業変数
    ! Work variables
    !
    integer  :: z_Index     (1:kmax)
    real(DP) :: z_CWFIntpled(1:kmax)

    integer :: k               ! 鉛直方向に回る DO ループ用作業変数
                               ! Work variables for DO loop in vertical direction
    integer :: l


    ! 実行文 ; Executable statement
    !

    ! 初期化確認
    ! Initialization check
    !
    if ( .not. arakawa_schubert_L1982_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    do k = 1, kmax
      z_Index(k) = 1
      loop_search : do l = 2, nTbl1-1
        if ( a_Tbl1Press(l) < z_PressCldTop(k) ) then
          z_Index(k) = l
        end if
      end do loop_search
    end do

    do k = 1, kmax
      z_CWFIntpled(k) =                                                   &
        &   ( a_Tbl1CWF  (z_Index(k)+1) - a_Tbl1CWF  (z_Index(k)) ) &
        & / ( a_Tbl1Press(z_Index(k)+1) - a_Tbl1Press(z_Index(k)) ) &
        & * ( z_PressCldTop(k)          - a_Tbl1Press(z_Index(k)) ) &
        & + a_Tbl1CWF(z_Index(k))
    end do

    do k = 1, kmax
      z_CWFCrtl(k) = &
        & z_CWFIntpled(k) * ( PressCldBase - z_PressCldTop(k) )
    end do


  end subroutine ASL1982CalcCWFCrtl1D

  !-------------------------------------------------------------------

  subroutine ArakawaSchubertL1982Init
    !
    ! arakawa_schubert_L1982 モジュールの初期化を行います. 
    ! NAMELIST#arakawa_schubert_L1982_nml の読み込みはこの手続きで行われます. 
    !
    ! "arakawa_schubert_L1982" module is initialized. 
    ! "NAMELIST#arakawa_schubert_L1982_nml" is loaded in this procedure. 
    !

    ! モジュール引用 ; USE statements
    !

    ! NAMELIST ファイル入力に関するユーティリティ
    ! Utilities for NAMELIST file input
    !
!!$    use namelist_util, only: namelist_filename, NmlutilMsg, NmlutilAryValid

    ! ファイル入出力補助
    ! File I/O support
    !
    use dc_iounit, only: FileOpen


    ! 宣言文 ; Declaration statements
    !
    implicit none

!!$    integer:: unit_nml        ! NAMELIST ファイルオープン用装置番号. 
!!$                              ! Unit number for NAMELIST file open
!!$    integer:: iostat_nml      ! NAMELIST 読み込み時の IOSTAT. 
!!$                              ! IOSTAT of NAMELIST read

    ! NAMELIST 変数群
    ! NAMELIST group name
    !
!!$    namelist /arakawa_schubert_L1982_nml/ &
!!$      & 

          ! デフォルト値については初期化手続 "arakawa_schubert_L1982#CumAdjInit" 
          ! のソースコードを参照のこと. 
          !
          ! Refer to source codes in the initialization procedure
          ! "arakawa_schubert_L1982#ArakawaSchubertL1982Init" for the default values. 
          !

    ! 実行文 ; Executable statement
    !

    if ( arakawa_schubert_L1982_inited ) return

    ! デフォルト値の設定
    ! Default values settings
    !

    ! NAMELIST の読み込み
    ! NAMELIST is input
    !
!!$    if ( trim(namelist_filename) /= '' ) then
!!$      call FileOpen( unit_nml, &          ! (out)
!!$        & namelist_filename, mode = 'r' ) ! (in)
!!$
!!$      rewind( unit_nml )
!!$      read( unit_nml,                  &  ! (in)
!!$        & nml = moist_conv_adjust_nml, &  ! (out)
!!$        & iostat = iostat_nml )           ! (out)
!!$      close( unit_nml )
!!$
!!$      call NmlutilMsg( iostat_nml, module_name ) ! (in)
!!$    end if



    ! 印字 ; Print
    !
    call MessageNotify( 'M', module_name, '----- Initialization Messages -----' )
    call MessageNotify( 'M', module_name, '' )
    call MessageNotify( 'M', module_name, '-- version = %c', c1 = trim(version) )

    arakawa_schubert_L1982_inited = .true.

  end subroutine ArakawaSchubertL1982Init

  !-------------------------------------------------------------------

end module arakawa_schubert_L1982
