!= 
!
!= CO2 condensation temperature (Pollack et al., 1981)
!
! Authors::   Yoshiyuki O. Takahashi
! Version::   $Id: saturate_co2_p81.f90,v 1.2 2014/05/07 09:39:21 murashin Exp $ 
! Tag Name::  $Name:  $
! Copyright:: Copyright (C) GFD Dennou Club, 2008. All rights reserved.
! License::   See COPYRIGHT[link:../../../COPYRIGHT]
!

module saturate_co2_p81
  !
  != 
  !
  != CO2 condensation temperature (Pollack et al., 1981)
  !
  ! <b>Note that Japanese and English are described in parallel.</b>
  !
  !== References
  !
  ! * Pollack, J. B., C. B. Leovy, P. W. Greiman, and Y. Mintz, 1981:
  !   A Martian general circulation experiment with large topography, 
  !   <i>J. Atmos. Sci.</i>, <b>38</b>, 3--29.
  !
  !== Procedures List
  ! 
!!$  ! DryConvAdjust :: 乾燥対流調節
!!$  ! ------------  :: ------------
!!$  ! DryConvAdjust :: Dry convective adjustment
  !
  !== NAMELIST
  !
  ! NAMELIST#major_comp_saturate_nml
  !

  ! モジュール引用 ; USE statements
  !

  ! 種別型パラメタ
  ! Kind type parameter
  !
  use dc_types, only: DP, &      ! 倍精度実数型. Double precision. 
    &                 STRING     ! 文字列.       Strings. 

  ! NAMELIST ファイル入力に関するユーティリティ
  ! Utilities for NAMELIST file input
  !
  use namelist_util, only: MaxNmlArySize
                              ! NAMELIST から読み込む配列の最大サイズ. 
                              ! Maximum size of arrays loaded from NAMELIST

  ! メッセージ出力
  ! Message output
  !
  use dc_message, only: MessageNotify

  ! 宣言文 ; Declaration statements
  !
  implicit none
  private


  ! 公開手続き
  ! Public procedure
  !
  public :: xyz_SaturateCO2P81TempCond
  public :: xyz_SaturateCO2P81PressSat
  public :: xyz_SaturateCO2P81DPressSatDT
  public :: SaturateCO2P81Init


  ! 公開変数
  ! Public variables
  !
  logical, save, public:: saturate_co2_p81_inited = .false.
                              ! 初期設定フラグ. 
                              ! Initialization flag

  ! 非公開変数
  ! Private variables
  !
  real(DP), parameter :: ConstTemp0 = 149.2_DP
  real(DP), parameter :: ConstAlpha = 6.48_DP
  real(DP), parameter :: ConstBeta  = 0.135e-2_DP


  character(*), parameter:: module_name = 'saturate_co2_p81'
                              ! モジュールの名称. 
                              ! Module name
  character(*), parameter:: version = &
    & '$Name:  $' // &
    & '$Id: saturate_co2_p81.f90,v 1.2 2014/05/07 09:39:21 murashin Exp $'
                              ! モジュールのバージョン
                              ! Module version


contains

  !--------------------------------------------------------------------------------------

  function xyz_SaturateCO2P81TempCond( xyz_Press ) result( xyz_TempCond )
    !
    ! CO2 相変化
    !
    ! CO2 phase change
    !

    ! モジュール引用 ; USE statements
    !


    ! 宣言文 ; Declaration statements
    !
    implicit none

    real(DP), intent(in ):: xyz_Press   (:,:,:)
                              ! $ p $ . 気圧 (整数レベル). 
                              ! Air pressure (full level)
    real(DP) :: xyz_TempCond(size(xyz_Press,1), size(xyz_Press,2), size(xyz_Press,3))
                              !
                              ! Condensation temperature

    ! 作業変数
    ! Work variables
    !

    ! 実行文 ; Executable statement
    !


    ! 初期化
    ! Initialization
    !
    if ( .not. saturate_co2_p81_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if

    ! CO2 condensation temperature (Pollack et al., 1981)

!!$    xyz_TempCond = 149.2_DP + 6.48_DP * log( 0.135_DP * xyz_Press * 1.0e-2_DP )
    xyz_TempCond = ConstTemp0 + ConstAlpha * log( ConstBeta * xyz_Press )


  end function xyz_SaturateCO2P81TempCond

  !--------------------------------------------------------------------------------------

  function xyz_SaturateCO2P81PressSat( xyz_Temp ) result( xyz_Press )
    !
    ! CO2 相変化
    !
    ! CO2 phase change
    !

    ! モジュール引用 ; USE statements
    !


    ! 宣言文 ; Declaration statements
    !
    implicit none

    real(DP), intent(in ):: xyz_Temp (:,:,:)
                              ! $ T $ . 
                              ! Air temperature
    real(DP) :: xyz_Press(size(xyz_Temp,1), size(xyz_Temp,2), size(xyz_Temp,3))
                              !
                              ! Saturation pressure

    ! 作業変数
    ! Work variables
    !

    ! 実行文 ; Executable statement
    !


    ! 初期化
    ! Initialization
    !
    if ( .not. saturate_co2_p81_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if

    ! CO2 condensation temperature (Pollack et al., 1981)

    xyz_Press = 1.0_DP / ConstBeta * exp( ( xyz_Temp - ConstTemp0 ) / ConstAlpha )


  end function xyz_SaturateCO2P81PressSat

  !--------------------------------------------------------------------------------------

  function xyz_SaturateCO2P81DPressSatDT( xyz_Temp ) result( xyz_DPressSatDT )
    !
    ! CO2 相変化
    !
    ! CO2 phase change
    !

    ! モジュール引用 ; USE statements
    !


    ! 宣言文 ; Declaration statements
    !
    implicit none

    real(DP), intent(in ):: xyz_Temp (:,:,:)
                              ! $ T $ . 
                              ! Air temperature
    real(DP) :: xyz_DPressSatDT(size(xyz_Temp,1), size(xyz_Temp,2), size(xyz_Temp,3))
                              !
                              ! Derivative of Condensation temperature with temperature

    ! 作業変数
    ! Work variables
    !

    ! 実行文 ; Executable statement
    !


    ! 初期化
    ! Initialization
    !
    if ( .not. saturate_co2_p81_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if

    ! CO2 condensation temperature (Pollack et al., 1981)

    xyz_DPressSatDT = xyz_SaturateCO2P81PressSat( xyz_Temp ) / ConstAlpha


  end function xyz_SaturateCO2P81DPressSatDT

  !--------------------------------------------------------------------------------------

  subroutine SaturateCO2P81Init
    !
    ! saturate_co2_p81 モジュールの初期化を行います. 
    ! NAMELIST#saturate_co2_p81_nml の読み込みはこの手続きで行われます. 
    !
    ! "saturate_co2_p81" module is initialized. 
    ! "NAMELIST#saturate_co2_p81_nml" is loaded in this procedure. 
    !

    ! モジュール引用 ; USE statements
    !

!!$    ! NAMELIST ファイル入力に関するユーティリティ
!!$    ! Utilities for NAMELIST file input
!!$    !
!!$    use namelist_util, only: namelist_filename, NmlutilMsg, NmlutilAryValid
!!$
!!$    ! ファイル入出力補助
!!$    ! File I/O support
!!$    !
!!$    use dc_iounit, only: FileOpen
!!$
!!$    ! 種別型パラメタ
!!$    ! Kind type parameter
!!$    !
!!$    use dc_types, only: STDOUT ! 標準出力の装置番号. Unit number of standard output

    ! 文字列操作
    ! Character handling
    !
    use dc_string, only: StoA


    ! 宣言文 ; Declaration statements
    !
    implicit none


!!$    integer:: unit_nml        ! NAMELIST ファイルオープン用装置番号. 
!!$                              ! Unit number for NAMELIST file open
!!$    integer:: iostat_nml      ! NAMELIST 読み込み時の IOSTAT. 
!!$                              ! IOSTAT of NAMELIST read

    ! NAMELIST 変数群
    ! NAMELIST group name
    !
!!$    namelist /major_comp_saturate_nml/ &
!!$      & FlagUse

          ! デフォルト値については初期化手続 "major_comp_saturate#MajorCompSaturateInit" 
          ! のソースコードを参照のこと. 
          !
          ! Refer to source codes in the initialization procedure
          ! "major_comp_saturate#MajorCompSaturateInit" for the default values. 
          !

    ! 実行文 ; Executable statement
    !

    if ( saturate_co2_p81_inited ) return


    ! デフォルト値の設定
    ! Default values settings
    !


    ! NAMELIST の読み込み
    ! NAMELIST is input
    !
!!$    if ( trim(namelist_filename) /= '' ) then
!!$      call FileOpen( unit_nml, &          ! (out)
!!$        & namelist_filename, mode = 'r' ) ! (in)
!!$
!!$      rewind( unit_nml )
!!$      read( unit_nml,                    &  ! (in)
!!$        & nml = major_comp_saturate_nml, &  ! (out)
!!$        & iostat = iostat_nml )             ! (out)
!!$      close( unit_nml )
!!$
!!$      call NmlutilMsg( iostat_nml, module_name ) ! (in)
!!$!      if ( iostat_nml == 0 ) write( STDOUT, nml = cumulus_adjust_nml )
!!$    end if


    ! 印字 ; Print
    !
    call MessageNotify( 'M', module_name, '----- Initialization Messages -----' )
    call MessageNotify( 'M', module_name, '-- version = %c', c1 = trim(version) )


    saturate_co2_p81_inited = .true.

  end subroutine SaturateCO2P81Init

  !-------------------------------------------------------------------

end module saturate_co2_p81
