
// -*- mode: c++; c-basic-offset:4 -*-

// This file is part of libdap, A C++ implementation of the OPeNDAP Data
// Access Protocol.

// Copyright (c) 2002,2003,2005 OPeNDAP, Inc.
// Author: James Gallagher <jgallagher@opendap.org>
//         Reza Nekovei <reza@intcomm.net>
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
// You can contact OPeNDAP, Inc. at PO Box 112, Saunderstown, RI. 02874-0112.

// (c) COPYRIGHT URI/MIT 1994-1996
// Please read the full copyright statement in the file COPYRIGHT.
//
// Authors:
//      reza            Reza Nekovei (reza@ieee.org)

// DODS-netCDF surrogate library functions for file handling. Note that with
// the preprocessor symbol LOCAL defined this library will access local files
// without using the DODS network services. Defining LOCAL will add support
// for local files through a slightly modifed version of the original netCDF
// library. This library has the same functionality as the original (it can
// access only files on a local hard disk) but has renamed entry points to
// avoid linker conflicts.
// 
// ReZa 10/13/94
//
// Rewrite to support netCDF version 3 interface; 5/12/99, Reza
//

#include "config_nc.h"

static char rcsid[] not_used =
    { "$Id: Dnc.cc 16710 2007-06-19 16:44:47Z jimg $" };

#include <string>

#include "Connections.h"
#include "Connections.cc"

#include "ClientParams.h"
#include "NCStructure.h"
#include "NCConnect.h"

#include "Dnetcdf.h"
#include "nc_util.h"

// Make these symbols external so as to avoid link-time conflicts with their
// definitions in the localized version of the original
// implementation of netcdf.

#ifdef LOCAL
extern char *cdf_routine_name;
extern int ncopts;
extern int ncerr;
#else
char *cdf_routine_name;
int ncopts = NC_VERBOSE;
int ncerr;
#endif

/** Defined in lnetcdf/lerror. This is not included in lnetcdf.h because
    doing so would require all sources that use lnetcdf.h to also include
    Error.h 03/02/05 jhrg */
extern void set_opendap_error(const Error &);

int rcode;

Connections < NCConnect * >*conns = 0;

static void delete_conns()
{
    delete conns;
    conns = 0;
}

int
nc__create_mp(const char *path, int ioflags, size_t initialsz, int basepe,
              size_t * chunksizehintp, int *ncid_ptr)
{
    // ClientParams is allocated outside of NCConnect so that NCConnect's
    // ctor can get the path stripped of the URL params. It then passes this
    // URL to AISConnect. The pointer 'cp' is passed to NCConnect so that the
    // memory can be recovered when this instance is deleted.
    ClientParams *cp = new ClientParams(path);
    NCConnect *c = new NCConnect(cp->url_without_params(), cp);

#ifdef LOCAL
    if (c->is_local()) {
        rcode = lnc__create_mp(path, ioflags, initialsz, basepe,
                               chunksizehintp, ncid_ptr);
        if (rcode == NC_NOERR) {
            c->set_ncid(*ncid_ptr);
            if (!conns) {
                DBG2(cerr << "Allocating conns" << endl);
                conns = new Connections < NCConnect * >(MAX_NC_OPEN);
                atexit(delete_conns);
            }
            DBG(cerr << "nc__create_mp: Adding object " << c
                << " to Connections." << endl);
            DBG(cerr << "nc__create_mp: local ncid is " << *ncid_ptr <<
                endl);
            *ncid_ptr = conns->add_connect(c);
            DBG(cerr << "nc__create_mp: Conns id is " << *ncid_ptr <<
                endl);
            return NC_NOERR;
        } else {
            delete c;
            c = 0;
            return rcode;
        }
    }
#endif

    delete c;
    c = 0;
    return NC_EPERM;
}

int
nc__create(const char *path, int ioflags, size_t initialsz,
           size_t * chunksizehintp, int *ncid_ptr)
{
    return nc__create_mp(path, ioflags, initialsz, 0,
                         chunksizehintp, ncid_ptr);
}


int nc_create(const char *path, int ioflags, int *ncid_ptr)
{
    return nc__create(path, ioflags, 0, NULL, ncid_ptr);
}

/* This function sets a default create flag that will be logically
   or'd to whatever flags are passed into lnc_create for all future
   calls to lnc_create.
   Valid default create flags are NC_64BIT_OFFSET, NC_CLOBBER,
   NC_LOCK, NC_SHARE. */
int nc_set_default_format(int format, int *old_formatp)
{
    return lnc_set_default_format(format, old_formatp);
}

int
nc__open(const char *path, int ioflags,
         size_t * chunksizehintp, int *ncid_ptr)
{
    return nc__open_mp(path, ioflags, 0, chunksizehintp, ncid_ptr);
}

int
nc__open_mp(const char *urlpath, int ioflags, int basepe,
            size_t * chunksizehintp, int *ncid_ptr)
{
    string path = urlpath;
    string var_name;

    size_t q_mark = path.find("?");
    string ce = "";
    if (q_mark != string::npos) {
        ce = path.substr(q_mark + 1);
        path = path.substr(0, q_mark);
    }

    ClientParams *cp = new ClientParams(path);
    NCConnect *c = new NCConnect(cp->url_without_params(), cp);

#ifdef LOCAL
    if (c->is_local()) {
        try {
            rcode = lnc__open_mp(path.c_str(), ioflags, basepe,
                                 chunksizehintp, ncid_ptr);
        }
        catch(Error & e) {
            delete c;
            c = 0;
            set_opendap_error(e);
            return rcode;
        }

        if (rcode == NC_NOERR) {
            c->set_ncid(*ncid_ptr);
            if (!conns) {
                DBG2(cerr << "Allocating conns" << endl);
                conns = new Connections < NCConnect * >(MAX_NC_OPEN);
                atexit(delete_conns);
            }
            DBG(cerr << "nc__open_mp (local): Adding object " << c
                << " to Connections." << endl);
            DBG(cerr << "nc__open_mp (local): ncid is " << *ncid_ptr <<
                endl);
            *ncid_ptr = conns->add_connect(c);
            DBG(cerr << "nc__open_mp (local): Conns id is " << *ncid_ptr
                << endl);
            return NC_NOERR;
        } else {
            delete c;
            c = 0;
            return rcode;
        }
    }
#endif

    // !_CRAYMPP, only pe 0 is valid
    if (basepe != 0) {
        delete c;
        c = 0;
        return NC_EINVAL;
    }
    // Read only network access
    if (ioflags != NC_NOWRITE) {
        delete c;
        c = 0;
        return NC_EPERM;
    }

    try {
        c->init_remote_source(ce);
    }
    catch(Error & e) {
        DBG(cerr << "Caught error while initializing remote source:" <<
            endl << e.get_error_message() << endl);
        set_opendap_error(e);
        delete c;
        c = 0;
        if (e.get_error_code() != NC_NOERR)
            return e.get_error_code();
        else
            return NC_EINVAL;
    }

    if (!conns) {
        DBG2(cerr << "Allocating conns" << endl);
        conns = new Connections < NCConnect * >(MAX_NC_OPEN);
        atexit(delete_conns);
    }

    DBG(cerr << "nc__open_mp (remote): Adding object " << c
        << " to Connections." << endl);
    *ncid_ptr = conns->add_connect(c);
    DBG(cerr << "nc__open_mp (remote): Conns id is " << *ncid_ptr << endl);

    return NC_NOERR;
}


int nc_open(const char *path, int ioflags, int *ncid_ptr)
{
    return nc__open(path, ioflags, NULL, ncid_ptr);
}


int nc_close(int cdfid)
{
    DBG(cerr << "Inside nc_close() with cdfid: " << cdfid << endl);

    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN || (*conns)[cdfid] == NULL)  // does the file id exist?
        return NC_EBADID;

    int rcode = NC_NOERR;

#ifdef LOCAL
    if ((*conns)[cdfid]->is_local()) {
        rcode = lnc_close((*conns)[cdfid]->get_ncid());
    }
#endif

    DBG(cerr << "        number of conns on free list: "
        << conns->free_elements() << endl);

    delete(*conns)[cdfid];
    conns->del_connect(cdfid);
    // Let delete_conns(), which is registered using atexit(), 
    // take care of deleting conns itself.

    DBG(cerr << "        number of conns on free list after delete: "
        << conns->free_elements() << endl);

    return rcode;
}

int nc_delete(const char *path)
{
    return nc_delete_mp(path, 0);
}

int nc_delete_mp(const char *path, int basepe)
{
    ClientParams *cp = new ClientParams(path);
    NCConnect *c = new NCConnect(cp->url_without_params(), cp);

#ifdef LOCAL
    if (c->is_local()) {
        rcode = lnc_delete_mp(path, basepe);
        return rcode;
    }
#endif

    delete c;
    c = 0;
    return NC_EPERM;
}

int nc_inq_base_pe(int cdfid, int *pe)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN || (*conns)[cdfid] == NULL)  // does the file id exist?
        return NC_EBADID;

#ifdef LOCAL
    if ((*conns)[cdfid]->is_local()) {
        rcode = lnc_inq_base_pe((*conns)[cdfid]->get_ncid(), pe);
        return rcode;
    }
#endif
    *pe = 0;                    /* !_CRAYMPP, only pe 0 is valid */
    return NC_NOERR;
}

int nc_set_base_pe(int cdfid, int pe)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN || (*conns)[cdfid] == NULL)  // does the file id exist?
        return NC_EBADID;

#ifdef LOCAL
    if ((*conns)[cdfid]->is_local()) {
        rcode = lnc_set_base_pe((*conns)[cdfid]->get_ncid(), pe);
        return rcode;
    }
#endif
    return NC_EPERM;
}

int nc_inq(int cdfid, int *ndimsp, int *nvarsp, int *nattsp, int *xtendimp)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN || (*conns)[cdfid] == NULL)  // does the file id exist?
        return NC_EBADID;

#ifdef LOCAL
    if ((*conns)[cdfid]->is_local()) {
        rcode =
            lnc_inq((*conns)[cdfid]->get_ncid(), ndimsp, nvarsp, nattsp,
                    xtendimp);
        return rcode;
    }
#endif

    if (nvarsp != NULL)
        *nvarsp = (*conns)[cdfid]->get_nvars();

    if (nattsp != NULL)
        *nattsp = (*conns)[cdfid]->get_num_attr(NC_GLOBAL);

    if (ndimsp != NULL)
        *ndimsp = (*conns)[cdfid]->get_ndims();

    if (xtendimp != NULL)
        *xtendimp = (*conns)[cdfid]->recdim();

    return NC_NOERR;
}


int nc_inq_ndims(int cdfid, int *ndimsp)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN || (*conns)[cdfid] == NULL)  // does the file id exist?
        return NC_EBADID;

#ifdef LOCAL
    if ((*conns)[cdfid]->is_local()) {
        rcode = lnc_inq_ndims((*conns)[cdfid]->get_ncid(), ndimsp);
        return rcode;
    }
#endif

    if (ndimsp != NULL)
        *ndimsp = (*conns)[cdfid]->get_ndims();

    return NC_NOERR;
}


int nc_inq_nvars(int cdfid, int *nvarsp)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN || (*conns)[cdfid] == NULL)  // does the file id exist?
        return NC_EBADID;

#ifdef LOCAL
    if ((*conns)[cdfid]->is_local()) {
        rcode = lnc_inq_nvars((*conns)[cdfid]->get_ncid(), nvarsp);
        return rcode;
    }
#endif

    if (nvarsp != NULL)
        *nvarsp = (*conns)[cdfid]->get_nvars();

    return NC_NOERR;
}


int nc_inq_natts(int cdfid, int *nattsp)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN || (*conns)[cdfid] == NULL)  // does the file id exist?
        return NC_EBADID;

#ifdef LOCAL
    if ((*conns)[cdfid]->is_local()) {
        rcode = lnc_inq_natts((*conns)[cdfid]->get_ncid(), nattsp);
        return rcode;
    }
#endif

    if (nattsp != NULL)
        *nattsp = (*conns)[cdfid]->get_num_attr(NC_GLOBAL);

    return NC_NOERR;
}


int nc_inq_unlimdim(int cdfid, int *xtendimp)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN || (*conns)[cdfid] == NULL)  // does the file id exist?
        return NC_EBADID;

#ifdef LOCAL
    if ((*conns)[cdfid]->is_local()) {
        rcode = lnc_inq_unlimdim((*conns)[cdfid]->get_ncid(), xtendimp);
        return rcode;
    }
#endif

    if (xtendimp != NULL)
        *xtendimp = (*conns)[cdfid]->recdim();

    return NC_NOERR;
}



int nc_sync(int cdfid)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN || (*conns)[cdfid] == NULL)  // does the file id exist?
        return NC_EBADID;

#ifdef LOCAL
    if ((*conns)[cdfid]->is_local()) {
        rcode = lnc_sync((*conns)[cdfid]->get_ncid());
        return rcode;
    }
#endif

    return NC_EPERM;
}

int nc_abort(int cdfid)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN || (*conns)[cdfid] == NULL)  // does the file id exist?
        return NC_EBADID;

#ifdef LOCAL
    if ((*conns)[cdfid]->is_local()) {
        rcode = lnc_abort((*conns)[cdfid]->get_ncid());
        return rcode;
    }
#endif
    rcode = nc_close(cdfid);
    return rcode;
}

/* 
 * Deprecated function ;
 */
int ncnobuf(int cdfid)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN || (*conns)[cdfid] == NULL)  // does the file id exist?
        return NC_EBADID;

    cdf_routine_name = "ncnobuf";

    /* NOOP */

    return (0);
}


int nc_redef(int cdfid)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN || (*conns)[cdfid] == NULL)  // does the file id exist?
        return NC_EBADID;

#ifdef LOCAL
    if ((*conns)[cdfid]->is_local()) {
        rcode = lnc_redef((*conns)[cdfid]->get_ncid());
        return rcode;
    }
#endif

    return NC_EPERM;
}


int
nc__enddef(int cdfid, size_t h_minfree, size_t v_align,
           size_t v_minfree, size_t r_align)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN || (*conns)[cdfid] == NULL)  // does the file id exist?
        return NC_EBADID;

#ifdef LOCAL
    if ((*conns)[cdfid]->is_local()) {
        rcode = lnc__enddef((*conns)[cdfid]->get_ncid(), h_minfree,
                            v_align, v_minfree, r_align);
        return rcode;
    }
#endif

    return NC_NOERR;            // no effect on a remote file

}


int nc_enddef(int cdfid)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN || (*conns)[cdfid] == NULL)  // does the file id exist?
        return NC_EBADID;

#ifdef LOCAL
    if ((*conns)[cdfid]->is_local()) {
        rcode = lnc_enddef((*conns)[cdfid]->get_ncid());
        return rcode;
    }
#endif

    return NC_NOERR;            // no effect on a remote file
}

int nc_set_fill(int cdfid, int fillmode, int *old_mode_ptr)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN || (*conns)[cdfid] == NULL)  // does the file id exist?
        return NC_EBADID;

    if ((*conns)[cdfid] == NULL)        // does the file id exist?
        return NC_EBADID;

#ifdef LOCAL
    if ((*conns)[cdfid]->is_local()) {
        rcode =
            lnc_set_fill((*conns)[cdfid]->get_ncid(), fillmode,
                         old_mode_ptr);
        return rcode;
    }
#endif

    return NC_EPERM;
}

const char *nc_inq_libvers(void)
{
    return lnc_inq_libvers();   // local netCDF library version (not OPeNDAP)
}

// I've modified the 'local version' of this so that it works for both local
// and remote accesses. jhrg
const char *nc_strerror(int err)
{
    return lnc_strerror(err);
}

int nc_inq_format(int ncid, int *formatp)
{
    // does the file id exist?
    if (!conns || ncid < 0 || ncid > MAX_NC_OPEN || (*conns)[ncid] == NULL)
        return NC_EBADID;

    if ((*conns)[ncid] == NULL) // does the file id exist?
        return NC_EBADID;

#ifdef LOCAL
    if ((*conns)[ncid]->is_local()) {
        rcode = lnc_inq_format((*conns)[ncid]->get_ncid(), formatp);
        return rcode;
    }
#endif
    // The new (3.6.2 beta 4) from Unidata looks at the NC_64BIT_OFFSET flag
    // and sets formatp to NC_FORMAT_64BIT if it's true, otherwise is sets it
    // to NC_FORMAT_CLASSIC. I'm just going to return 64BIT for now since I
    // think that building the library as 64BIT means everything in memory is
    // represented using the new format. This probably only matters for
    // creating files anyway, in which case only the local code is critical
    // and that's still using the 'real' netcdf library. jhrg 11/01/06
#if 0
    *formatp = fIsSet(ncp->flags, NC_64BIT_OFFSET) ? NC_FORMAT_64BIT
        : NC_FORMAT_CLASSIC;
#endif
    *formatp = NC_FORMAT_64BIT;
    return NC_NOERR;
}
