require "vtk/Gtk"

VTK_DATA_ROOT = ENV["VTK_DATA_ROOT"]


VTK_DATA_ROOT || raise("ERROR: Please define environment variable VTK_DATA_ROOT")

###
# Now actually create the GUI
Gtk.init
Gtk::GL.init
root = Gtk::Window.new(Gtk::Window::TOPLEVEL)
top = Gtk::VBox.new
root.add(top)
@renWin = GtkGLExtVTKRenderWindowInteractor.new
top.pack_start(@renWin)


# Start by loading some data.
v16 = Vtk::Volume16Reader.new
v16.SetDataDimensions(64, 64)
v16.SetDataByteOrderToLittleEndian()
v16.SetFilePrefix(VTK_DATA_ROOT + "/Data/headsq/quarter")
v16.SetImageRange(1, 93)
v16.SetDataSpacing(3.2, 3.2, 1.5)
v16.Update()

@xMin, @xMax, @yMin, @yMax, @zMin, @zMax = v16.GetOutput().GetWholeExtent()

spacing = v16.GetOutput().GetSpacing()
@sx, @sy, @sz = spacing

origin = v16.GetOutput().GetOrigin()
@ox, @oy, @oz = origin

# An outline is shown for context.
outline = Vtk::OutlineFilter.new
outline.SetInput(v16.GetOutput())

outlineMapper = Vtk::PolyDataMapper.new
outlineMapper.SetInput(outline.GetOutput())

outlineActor = Vtk::Actor.new
outlineActor.SetMapper(outlineMapper)

# The shared picker enables us to use 3 planes at one time
# and gets the picking order right
picker = Vtk::CellPicker.new
picker.SetTolerance(0.005)

# The 3 image plane widgets are used to probe the dataset.
@planeWidgetX = Vtk::ImagePlaneWidget.new
@planeWidgetX.DisplayTextOn()
@planeWidgetX.SetInput(v16.GetOutput())
@planeWidgetX.SetPlaneOrientationToXAxes()
@planeWidgetX.SetSliceIndex(32)
@planeWidgetX.SetPicker(picker)
@planeWidgetX.SetKeyPressActivationValue("x")
prop1 = @planeWidgetX.GetPlaneProperty()
prop1.SetColor(1, 0, 0)

@planeWidgetY = Vtk::ImagePlaneWidget.new
@planeWidgetY.DisplayTextOn()
@planeWidgetY.SetInput(v16.GetOutput())
@planeWidgetY.SetPlaneOrientationToYAxes()
@planeWidgetY.SetSliceIndex(32)
@planeWidgetY.SetPicker(picker)
@planeWidgetY.SetKeyPressActivationValue("y")
prop2 = @planeWidgetY.GetPlaneProperty()
prop2.SetColor(1, 1, 0)
@planeWidgetY.SetLookupTable(@planeWidgetX.GetLookupTable())

# for the z-slice, turn off texture interpolation:
# interpolation is now nearest neighbour, to demonstrate
# cross-hair cursor snapping to pixel centers
@planeWidgetZ = Vtk::ImagePlaneWidget.new
@planeWidgetZ.DisplayTextOn()
@planeWidgetZ.SetInput(v16.GetOutput())
@planeWidgetZ.SetPlaneOrientationToZAxes()
@planeWidgetZ.SetSliceIndex(46)
@planeWidgetZ.SetPicker(picker)
@planeWidgetZ.SetKeyPressActivationValue("z")
prop3 = @planeWidgetZ.GetPlaneProperty()
prop3.SetColor(0, 0, 1)
@planeWidgetZ.SetLookupTable(@planeWidgetX.GetLookupTable())

# Create the RenderWindow and Renderer
@ren = Vtk::Renderer.new
@renWin.GetRenderWindow.AddRenderer(@ren)

# Add the outline actor to the renderer, set the background color and size
@ren.AddActor(outlineActor)
@renWin.set_size_request(500, 500)
@ren.SetBackground(0.1, 0.1, 0.2)

@current_widget = @planeWidgetZ
@mode_widget = @planeWidgetZ

# Create the GUI
# We first create the supporting functions (callbacks) for the GUI
#
# Align the camera so that it faces the desired widget
def AlignCamera
  cx = @ox+(0.5*(@xMax-@xMin))*@sx
  cy = @oy+(0.5*(@yMax-@yMin))*@sy
  cz = @oy+(0.5*(@zMax-@zMin))*@sz
  vx, vy, vz = 0, 0, 0
  nx, ny, nz = 0, 0, 0
  iaxis = @current_widget.GetPlaneOrientation()
  if iaxis == 0
    vz = -1
    nx = @ox + @xMax*@sx
    cx = @ox + @x_adjust.value*@sx
  elsif iaxis == 1
    vz = -1
    ny = @oy+@yMax*@sy
    cy = @oy+@y_adjust.value*@sy
  else
    vy = 1
    nz = @oz+@zMax*@sz
    cz = @oz+@z_adjust.value*@sz
  end
 
  px = cx+nx*2
  py = cy+ny*2
  pz = cz+nz*3

  camera = @ren.GetActiveCamera()
  camera.SetViewUp(vx, vy, vz)
  camera.SetFocalPoint(cx, cy, cz)
  camera.SetPosition(px, py, pz)
  camera.OrthogonalizeViewUp()
  @ren.ResetCameraClippingRange()
  @renWin.Render()
end
 
# Capture the display and place in a tiff
def captureImage()
  w2i = Vtk::WindowToImageFilter.new
  writer = Vtk::TIFFWriter.new
  w2i.SetInput(@renWin.GetRenderWindow)
  w2i.Update()
  writer.SetInput(w2i.GetOutput())
  writer.SetFileName("image.tif")
  @renWin.Render()
  writer.Write()
end
 

# Align the widget back into orthonormal position,
# set the slider to reflect the widget's position,
# call AlignCamera to set the camera facing the widget
def alignXaxis()
  po = @planeWidgetX.GetPlaneOrientation()
  if po == 3
    @planeWidgetX.SetPlaneOrientationToXAxes()
  end
 
  @current_widget = @planeWidgetX

  AlignCamera()
end

def alignYaxis()
  po = @planeWidgetY.GetPlaneOrientation()
  if po == 3
    @planeWidgetY.SetPlaneOrientationToYAxes()
  end

  @current_widget = @planeWidgetY

  AlignCamera()
end
 
def alignZaxis()
  po = @planeWidgetZ.GetPlaneOrientation()
  if po == 3
    @planeWidgetZ.SetPlaneOrientationToZAxes()
  end
 
  @current_widget = @planeWidgetZ

  AlignCamera()
end


# Set the widget's reslice interpolation mode
# to the corresponding popup menu choice
def SetInterpolation(mode)
  @popm.popdown
  if mode == 0
    @mode_widget.TextureInterpolateOff()
  else
    @mode_widget.TextureInterpolateOn()
  end

  @mode_widget.SetResliceInterpolate(mode)
  @renWin.Render()
end

# Share the popup menu among buttons, keeping track of associated
# widget's interpolation mode
def buttonEvent(event, arg)
p event,arg
  if (event.state&Gdk::Window::ModifierType::BUTTON3_MASK)==Gdk::Window::ModifierType::BUTTON3_MASK
    if arg == 0
      @mode_widget = @planeWidgetX
    elsif arg == 1
      @mode_widget = @planeWidgetY
    elsif arg == 2
      @mode_widget = @planeWidgetZ
    else
      return nil
    end
    @popm.popup(nil, nil, event.button, event.time)
  end
end
        
def SetSlice(adj,i)
  case i
  when 0
    @planeWidgetX.SetSliceIndex(adj.value)
  when 1
    @planeWidgetY.SetSliceIndex(adj.value)
  when 2
    @planeWidgetZ.SetSliceIndex(adj.value)
  end
  @ren.ResetCameraClippingRange()
  @renWin.Render()
end




# Popup menu
@popm = Gtk::Menu.new
item0 = Gtk::MenuItem.new("nearest")
item0.signal_connect("activate"){ SetInterpolation(0) }
item1 = Gtk::MenuItem.new("linear")
item1.signal_connect("activate"){ SetInterpolation(1) }
item2 = Gtk::MenuItem.new("cubic")
item2.signal_connect("activate"){ SetInterpolation(2) }
@popm.append(item0)
@popm.append(item1)
@popm.append(item2)

# Buttons
ctrl_buttons = Gtk::HBox.new

quit_button = Gtk::Button.new("Quit")
quit_button.signal_connect("clicked"){ Gtk.main_quit }
capture_button = Gtk::Button.new("Tif")
capture_button.signal_connect("clicked"){ captureImage }
x_button = Gtk::Button.new("x")
x_button.signal_connect("clicked"){ alignXaxis }
#x_button.signal_connect("pressed"){|w,e| buttonEvent(e,0) }
y_button = Gtk::Button.new("y")
y_button.signal_connect("clicked"){ alignYaxis }
#y_button.signal_connect("pressed"){|w,e| buttonEvent(e,1) }
z_button = Gtk::Button.new("z")
z_button.signal_connect("clicked"){ alignZaxis }
#z_button.signal_connect("pressed"){|w,e| buttonEvent(e,1) }
ctrl_buttons.pack_start(quit_button)
ctrl_buttons.pack_start(capture_button)
ctrl_buttons.pack_start(x_button)
ctrl_buttons.pack_start(y_button)
ctrl_buttons.pack_start(z_button)
top.pack_start(ctrl_buttons)



# Add a slice scale to browse the current slice stack
@x_adjust = Gtk::Adjustment.new(@planeWidgetX.GetSliceIndex,@xMin,@xMax,1,1,0)
@y_adjust = Gtk::Adjustment.new(@planeWidgetY.GetSliceIndex,@yMin,@yMax,1,1,0)
@z_adjust = Gtk::Adjustment.new(@planeWidgetZ.GetSliceIndex,@zMin,@zMax,1,1,0)
@x_slice = Gtk::HScale.new(@x_adjust)
@y_slice = Gtk::HScale.new(@y_adjust)
@z_slice = Gtk::HScale.new(@z_adjust)
@x_adjust.signal_connect("value-changed"){|w| SetSlice(w,0) }
@y_adjust.signal_connect("value-changed"){|w| SetSlice(w,1) }
@z_adjust.signal_connect("value-changed"){|w| SetSlice(w,2) }
top.pack_start(@x_slice)
top.pack_start(@y_slice)
top.pack_start(@z_slice)

# Done with the GUI. 
###

# Set the interactor for the widgets
#iact = renWin.GetRenderWindow().GetInteractor()
iact = @renWin.GetInteractor
@planeWidgetX.SetInteractor(iact)
@planeWidgetX.On()
@planeWidgetY.SetInteractor(iact)
@planeWidgetY.On()
@planeWidgetZ.SetInteractor(iact)
@planeWidgetZ.On()

# Create an initial interesting view
cam1 = @ren.GetActiveCamera()
cam1.Elevation(110)
cam1.SetViewUp(0, 0, -1)
cam1.Azimuth(45)
@ren.ResetCameraClippingRange()



# Start Tkinter event loop
root.show_all
Gtk.main
