=begin
  gettext.rb - GetText module

  Copyright (C) 2001-2003  Masahiro Sakai, Masao Mutoh

      Masahiro Sakai    <s01397ms@sfc.keio.ac.jp>
      Masao Mutoh       <mutoh@highway.ne.jp>

  You may redistribute it and/or modify it under the same
  license terms as Ruby.

  $Id: gettext.rb,v 1.5 2003/12/01 17:08:49 mutoh Exp $
=end

require 'rbconfig'
require 'gettext/mo'
require 'gettext/locale'

module GetText
  class TextDomain
    def initialize(name, path, locale, charset)
      @name, @path = name, path
      @search_files = Array.new
    
      if path
        @locale_dirs = [path]
      else
        @locale_dirs = []
        prefix = Config::CONFIG["prefix"]
        if prefix != "/usr" and prefix != "/usr/local"
          @locale_dirs += [prefix + "/share/locale", 
                           prefix + "/local/share/locale"]
        end
        @locale_dirs += ["/usr/share/locale", "/usr/local/share/locale"]
      end
    
      @mofiles = Hash.new
      set_charset(charset, false)
      set_locale(locale, false)
      load_mo(false)
    
      if $DEBUG
        print "Search path:"
        p @locale_dirs
        print "locale:"
        p locale
      end
    end
    
    def set_locale(locale, reload = true)
      @locales = []
      matched = /^([A-Za-z]+)(_[A-Za-z]+)?(\.\w+)?((?:@[^@]*)*)/.match(locale)
      if matched.size > 1
        matched = matched.to_a
        matched.shift
        while not matched.empty?
          @locales.push(matched.join(''))
          matched.pop
        end
        @locales.uniq!
      else
        @locales.push(locale)
      end
      load_mo(false) if reload
    end
    
    def gettext(msgid)
      if @mo
        result = @mo[msgid] ? @mo[msgid] : msgid
      else
        no_mo_file if $DEBUG
        result = msgid
      end
      result
    end

    def ngettext(msgid, msgid_plural, n)
      msg = gettext(msgid + "\000" + msgid_plural)
      if msg.include?("\000")
        ary = msg.split("\000")
        if @mo
          plural = eval(@mo.plural)
          if plural.kind_of?(Numeric)
            msg = ary[plural]
          else
            msg = plural ? ary[1] : ary[0]
          end
        else
          msg = n == 1 ? ary[0] : ary[1]
        end
      end
      msg
    end

    def no_mo_file
      print("\nMO file is not found in\n")
      @search_files.each do |fname|
        print("\t#{fname}\n")
      end
    end

    def same_property?(name, path, locale, charset)
      @name == name and @path == path and @locales.include?(locale) and @charset == charset
    end
    
    def set_charset(charset, reload = true)
      if charset and charset != ""
        @charset = charset
      else
        @charset = ENV["OUTPUT_CHARSET"] ? ENV["OUTPUT_CHARSET"] : Locale.codeset
      end
      load_mo(false) if reload
    end

    def load_mo(cache = true)
      @mo = nil
      if cache 
        @locales.each do |locale|
          if @mo = @mofiles[locale]
            return @mo
          end
        end
      end
      @locale_dirs.each do |dir|
      @locales.each{|locale|
        fname = File.join(dir, locale, "LC_MESSAGES", @name + ".mo")
        @search_files << fname
        if File.exist?(fname)
          @mo = MOFile.open(fname, @charset)
          @mofiles[locale] = @mo
         break
        end
      }
      end
    end
  end

  @@__textdomain = Hash.new
  @@__textdomain_key = Hash.new

  def bindtextdomain(domainname, path = nil, locale = nil, charset = nil)
    locale ||= Locale.get
    charset ||= ENV["OUTPUT_CHARSET"] ? ENV["OUTPUT_CHARSET"] : Locale.codeset
    src = callersrc
    textdomain = @@__textdomain[src]
    if textdomain
      @@__textdomain_key[domainname] = textdomain
    else
      textdomain = @@__textdomain_key[domainname]
      @@__textdomain[src] = textdomain
    end
    if ! textdomain or ! textdomain.same_property?(domainname, path, locale, charset)
      textdomain = TextDomain.new(domainname, path, locale, charset)
      @@__textdomain_key[domainname] = textdomain
      @@__textdomain[src] = textdomain
    end
    @@__textdomain[src]
  end
  
  def gettext(msgid)
    @@__textdomain[callersrc].gettext(msgid)
  end
  
  def ngettext(msgid, msgid_plural, n)
    @@__textdomain[callersrc].ngettext(msgid, msgid_plural, n)
  end
  
  def N_(msgid)
    msgid
  end

  def callersrc
    caller(2)[0].sub(/:\d+(?::in \`\S+\')?\Z/, '')
  end

  def locale=(locale)
    @@__textdomain[callersrc].set_locale(locale)
  end

  def charset=(cs)
    @@__textdomain[callersrc].set_charset(cs)
  end
  
  alias :setlocale :locale=
  alias :_ :gettext
  alias :n_ :ngettext

  module_function :bindtextdomain, :N_, :gettext, :_, :ngettext, :n_, 
    :setlocale, :locale=, :charset=, :callersrc
end
