module fftsub_mod

  use ffts
  use typhoon_analy

contains


subroutine rearrange_3to2( val3, val2 )
!-- rearranging J array to N array (3hn+1 -> 2hn+1)
  use fft_saveval_define
  implicit none
  complex(kind(0d0)), dimension(jxnt,jynt), intent(in) :: val3
  complex(kind(0d0)), dimension(nx,ny), intent(inout) :: val2
  integer :: i, j

  val2(1,1)=val3(1,1)

!$omp parallel default(shared)
!$omp do schedule(runtime) private(j)

  do j=2,hynt+1
     val2(1,j)=val3(1,j)
     val2(1,ny-j+2)=val3(1,jynt-j+2)
  end do

!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i)

  do i=2,hxnt+1
     val2(i,1)=val3(i,1)
     val2(nx-i+2,1)=val3(jxnt-i+2,1)
  end do

!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j)

  do j=2,hynt+1
     do i=2,hxnt+1
        val2(i,j)=val3(i,j)
        val2(nx-i+2,ny-j+2)=val3(jxnt-i+2,jynt-j+2)
        val2(i,ny-j+2)=val3(i,jynt-j+2)
        val2(nx-i+2,j)=val3(jxnt-i+2,j)
     end do
  end do

!$omp end do
!$omp end parallel

end subroutine rearrange_3to2


subroutine rearrange_2to3( val2, val3 )
!-- rearranging J array to N array (2hn+1 -> 3hn+1)
  use fft_saveval_define
  implicit none
  complex(kind(0d0)), dimension(nx,ny), intent(in) :: val2
  complex(kind(0d0)), dimension(jxnt,jynt), intent(inout) :: val3
  integer :: i, j

  val3(1,1)=val2(1,1)

!$omp parallel default(shared)
!$omp do schedule(runtime) private(j)

  do j=2,hynt+1
     val3(1,j)=val2(1,j)
     val3(1,jynt-j+2)=val2(1,ny-j+2)
  end do

!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i)

  do i=2,hxnt+1
     val3(i,1)=val2(i,1)
     val3(jxnt-i+2,1)=val2(nx-i+2,1)
  end do

!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j)

  do j=2,hynt+1
     do i=2,hxnt+1
        val3(i,j)=val2(i,j)
        val3(jxnt-i+2,jynt-j+2)=val2(nx-i+2,ny-j+2)
        val3(i,jynt-j+2)=val2(i,ny-j+2)
        val3(jxnt-i+2,j)=val2(nx-i+2,j)
     end do
  end do

!$omp end do
!$omp end parallel

end subroutine rearrange_2to3


subroutine psik2ukvk( psik, uk, vk, ukopt, vkopt )
!-- converting psi to u and v
  use Math_Const
  use fft_saveval_define
  implicit none
  complex(kind(0d0)), dimension(:,:), intent(in) :: psik      ! stream function
  complex(kind(0d0)), dimension(size(psik,1),size(psik,2)), intent(inout) :: uk
                                        ! advecting speed of x direction
  complex(kind(0d0)), dimension(size(psik,1),size(psik,2)), intent(inout) :: vk
                                        ! advecting speed of y direction
  complex(kind(0d0)), intent(in), optional :: ukopt  ! 0,0 component of uk
  complex(kind(0d0)), intent(in), optional :: vkopt  ! 0,0 component of uk

  integer :: i, j, ix, jy
  double precision :: lxi, lyi, pi2

  uk=0.0d0
  vk=0.0d0

  if(present(ukopt))then
     uk(1,1)=ukopt
  end if
  if(present(vkopt))then
     vk(1,1)=vkopt
  end if

  ix=size(psik,1)
  jy=size(psik,2)

  lxi=1.0d0/Lx
  lyi=1.0d0/Ly
  pi2=2.0d0*pi_dp

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i)

  do i=2,hxnt+1
! uk=0.0
     vk(i,1)=pi2*img_cdp*dble(i-1)*lxi*psik(i,1)
     vk(ix-i+2,1)=dconjg(vk(i,1))
  end do

!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(j)

  do j=2,hynt+1
     uk(1,j)=-pi2*img_cdp*dble(j-1)*lyi*psik(1,j)
     uk(1,jy-j+2)=pi2*img_cdp*dble(j-1)*lyi*psik(1,jy-j+2)
! vk=0.0
  end do

!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j)

  do j=2,hynt+1
     do i=2,hxnt+1
        uk(i,j)=-pi2*img_cdp*dble(j-1)*lyi*psik(i,j)
        uk(ix-i+2,jy-j+2)=dconjg(uk(i,j))
        uk(i,jy-j+2)=pi2*img_cdp*dble(j-1)*lyi*psik(i,jy-j+2)
        uk(ix-i+2,j)=dconjg(uk(i,jy-j+2))
        vk(i,j)=pi2*img_cdp*dble(i-1)*lxi*psik(i,j)
        vk(ix-i+2,jy-j+2)=dconjg(vk(i,j))
        vk(i,jy-j+2)=pi2*img_cdp*dble(i-1)*lxi*psik(i,jy-j+2)
        vk(ix-i+2,j)=dconjg(vk(i,jy-j+2))
     end do
  end do

!$omp end do
!$omp end parallel

end subroutine psik2ukvk


subroutine zetak2psik( zk, psik, psikopt )
!-- converting zeta to psi using fft with poisson.
  use Math_Const
  use fft_saveval_define
  implicit none
  complex(kind(0d0)), dimension(:,:), intent(in) :: zk
  complex(kind(0d0)), dimension(size(zk,1),size(zk,2)), intent(inout) :: psik
  complex(kind(0d0)), intent(in), optional :: psikopt   ! 0,0 component of psik
  integer :: i, j, ix, jy
  double precision :: lxi, lyi, pii4

  psik=0.0d0

  if(present(psikopt))then
     psik(1,1)=psikopt
  end if

  ix=size(zk,1)
  jy=size(zk,2)

  lxi=1.0d0/Lx
  lyi=1.0d0/Ly
  pii4=0.25d0/((pi_dp)**2)

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i)

  do i=2,hxnt+1
     psik(i,1)=-pii4*zk(i,1)/((dble(i-1)*lxi)**2)
     psik(ix-i+2,1)=dconjg(psik(i,1))
  end do

!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(j)

  do j=2,hynt+1
     psik(1,j)=-pii4*zk(1,j)/((dble(j-1)*lyi)**2)
     psik(1,jy-j+2)=-pii4*zk(1,jy-j+2)/((dble(j-1)*lyi)**2)
  end do

!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j)

  do j=2,hynt+1
     do i=2,hxnt+1
        psik(i,j)=-pii4*zk(i,j)/((dble(i-1)*lxi)**2+(dble(j-1)*lyi)**2)
        psik(ix-i+2,jy-j+2)=dconjg(psik(i,j))
        psik(i,jy-j+2)=-pii4*zk(i,jy-j+2)/((dble(i-1)*lxi)**2+(dble(j-1)*lyi)**2)
        psik(ix-i+2,j)=dconjg(psik(i,jy-j+2))
     end do
  end do

!$omp end do
!$omp end parallel

end subroutine zetak2psik


subroutine psik2zetak( psik, zk, zkopt )
!-- converting psi to zeta using fft.
  use Math_Const
  use fft_saveval_define
  implicit none
  complex(kind(0d0)), dimension(:,:), intent(in) :: psik
  complex(kind(0d0)), dimension(size(psik,1),size(psik,2)), intent(inout) :: zk
  complex(kind(0d0)), intent(in), optional :: zkopt   ! 0,0 component of zk
  integer :: i, j, ix, jy
  double precision :: lxi, lyi, pi4

  zk=0.0d0

  if(present(zkopt))then
     zk(1,1)=zkopt
  end if

  ix=size(psik,1)
  jy=size(psik,2)

  lxi=1.0d0/Lx
  lyi=1.0d0/Ly
  pi4=4.0d0*pi_dp*pi_dp

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i)

  do i=2,hxnt+1
     zk(i,1)=-pi4*psik(i,1)*((dble(i-1)*lxi)**2)
     zk(ix-i+2,1)=dconjg(zk(i,1))
  end do

!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(j)

  do j=2,hynt+1
     zk(1,j)=-pi4*psik(1,j)*((dble(j-1)*lyi)**2)
     zk(1,jy-j+2)=-pi4*psik(1,jy-j+2)*((dble(j-1)*lyi)**2)
  end do

!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j)

  do j=2,hynt+1
     do i=2,hxnt+1
        zk(i,j)=-pi4*psik(i,j)*((dble(i-1)*lxi)**2+(dble(j-1)*lyi)**2)
        zk(ix-i+2,jy-j+2)=dconjg(zk(i,j))
        zk(i,jy-j+2)=-pi4*psik(i,jy-j+2)*((dble(i-1)*lxi)**2+(dble(j-1)*lyi)**2)
        zk(ix-i+2,j)=dconjg(zk(i,jy-j+2))
     end do
  end do

!$omp end do
!$omp end parallel

end subroutine psik2zetak


subroutine azimuth_filter( ioval )
!-- Filter a specified azimuthal wavenumber (force_wn) in ioval and return ioval
  use Math_Const
  use fft_saveval_define
  implicit none
  double precision, intent(inout) :: ioval(:,:)
  double precision, allocatable, dimension(:) :: rad, theta
  double precision, allocatable, dimension(:,:) :: ioval_rt
  complex(kind(0d0)), allocatable, dimension(:,:) :: cpval_rt, sp_rt, spout_rt
  double precision :: dxtmp, dytmp
  integer :: ii, jj, ix, jy, kk, kr
  integer :: ixc, jyc, ixmin, ixmax, jymin, jymax
!-- OpenMP variables
!$ integer :: OMP_GET_THREAD_NUM, OMP_GET_MAX_THREADS
  integer :: ompnum, omppe

  ix=size(ioval,1)  ! =jxnt
  jy=size(ioval,2)  ! =jynt
  dxtmp=dx*dble(nx-1)/dble(jxnt-1)
  dytmp=dy*dble(ny-1)/dble(jynt-1)

!-- internal variables for OpenMP
  ompnum=1
  omppe=1
!$   ompnum=OMP_GET_MAX_THREADS()  ! Activated in using OpenMP

  !-- search the minimum grid number in radius
  !kr=nint(min(min(tcx-xj(1),xj(jxnt)-tcx)/dxtmp,min(tcy-yj(1),yj(jynt)-tcy)/dytmp))
  kr=force_kr(2)
  allocate(rad(kr))
  allocate(theta(jy))
  allocate(ioval_rt(kr,jy))
  allocate(cpval_rt(kr,jy))

  rad(1:kr)=(/((dxtmp*dble(ii)),ii=1,kr)/)
  theta(1:jy)=(/((2.0d0*pi_dp*dble(jj-1)/dble(jy)),jj=1,jy)/)
  ioval_rt=0.0d0
  cpval_rt=0.0d0

  allocate(sp_rt(jy,ompnum))
  allocate(spout_rt(jy,ompnum))

  call interpo_search_1d( xj, tcx, ixc, stdopt=.true. )
  call interpo_search_1d( yj, tcy, jyc, stdopt=.true. )
  ixmin=ixc-kr-1
  ixmax=ixc+kr+1
  jymin=jyc-kr-1
  jymax=jyc+kr+1
  if(ixmin<1)then
     ixmin=1
  end if
  if(ixmax>jxnt)then
     ixmax=jxnt
  end if
  if(jymin<1)then
     jymin=1
  end if
  if(jymax>jynt)then
     jymax=jynt
  end if

  !-- 1. convert ioval from X-Y to R-T
  call tangent_conv_scal( xj(ixmin:ixmax), yj(jymin:jymax), tcx, tcy,  &
  &                       ioval(ixmin:ixmax,jymin:jymax),  &
  &                       rad, theta, ioval_rt(1:kr,1:jy), undef=0.0d0,  &
  &                       undefg=0.0d0, undefgc='inc',  &
  &                       axis='xy', stdopt=.true. )

  do jj=1,jy
     do kk=force_kr(1),force_kr(2)
        cpval_rt(kk,jj)=ioval_rt(kk,jj)
     end do
  end do
  ioval=0.0d0
  ioval_rt=0.0d0
  sp_rt=0.0d0
  spout_rt=0.0d0

  !-- 2. filter force_wn
!$omp parallel default(shared)
!$omp do schedule(runtime) private(kk,omppe)
  do kk=force_kr(1),force_kr(2)
     !-- For OpenMP
!$   omppe=OMP_GET_THREAD_NUM()+1  ! Activated in using OpenMP

     !-- r_FFT
     call ffttp_1d( jy, cpval_rt(kk,1:jy), sp_rt(1:jy,omppe), csign='r',  &
  &                 prim='o', prim_fact=pyfact,  &
  &                 omega_fix=omegayjbr, omegan_fix=omegayjnr )
     !-- Filtering
     spout_rt(force_wn+1,omppe)=sp_rt(force_wn+1,omppe)
     spout_rt(jy-(force_wn-1),omppe)=sp_rt(jy-(force_wn-1),omppe)
     !-- i_FFT
     call ffttp_1d( jy, spout_rt(1:jy,omppe), cpval_rt(kk,1:jy), csign='i',  &
  &                 prim='o', prim_fact=pyfact,  &
  &                 omega_fix=omegayjbi, omegan_fix=omegayjni )
  end do
!$omp end do
!$omp end parallel

  do jj=1,jy
     do kk=force_kr(1),force_kr(2)
        ioval_rt(kk,jj)=dble(cpval_rt(kk,jj))
     end do
  end do

  !-- 3. convert ioval (filterred) from R-T to X-Y
  call Cart_conv_scal( rad, theta, ioval_rt(1:kr,1:jy),  &
  &                    xj(ixmin:ixmax), yj(jymin:jymax), tcx, tcy,  &
  &                    ioval(ixmin:ixmax,jymin:jymax),  &
  &                    undef=0.0d0, undefg=0.0d0, undefgc='inc',  &
  &                    axis='xy', stdopt=.true. )

  deallocate(rad)
  deallocate(theta)
  deallocate(ioval_rt)
  deallocate(cpval_rt)
  deallocate(sp_rt)
  deallocate(spout_rt)

end subroutine azimuth_filter


subroutine make_restart( itn, rtn, psik, zk, zopt )
!-- output restart file
  use fft_saveval_define
  use gtool_history
  implicit none
  integer, intent(in) :: itn
  real, intent(in) :: rtn
  complex(kind(0d0)), dimension(:,:), intent(in) :: psik
  complex(kind(0d0)), dimension(size(psik,1),size(psik,2)), intent(in) :: zk
  complex(kind(0d0)), dimension(size(psik,1),size(psik,2)), optional, intent(in) :: zopt
  double precision, dimension(size(psik,1),size(psik,2)) :: tmpd
  type(GT_HISTORY) :: res_hst
  integer :: ix, iy

  call HistoryCreate( file=trim(adjustl(resfname)),  &
  &    title='BAROTRO result data', &
  &    source='test', institution='test', dims=(/'x', 'y'/),  &
  &    dimsizes=(/ nx, ny /),  & 
  &    longnames=(/'X-coordinate','Y-coordinate'/),  &
  &    units=(/'m', 'm'/), history=res_hst )

  call HistoryAddVariable( varname=trim(adjustl(respsir)), dims=(/'x','y'/), &
  &                        longname='stream line function',  &
  &                        units='m2 s-1', xtype='double', history=res_hst )

  call HistoryAddVariable( varname=trim(adjustl(respsii)), dims=(/'x','y'/), &
  &                        longname='stream line function',  &
  &                        units='m2 s-1', xtype='double', history=res_hst )

  call HistoryAddVariable( varname=trim(adjustl(reszetar)), dims=(/'x','y'/), &
  &                        longname='vorticity', units='s-1',  &
  &                        xtype='double', history=res_hst )

  call HistoryAddVariable( varname=trim(adjustl(reszetai)), dims=(/'x','y'/), &
  &                        longname='vorticity', units='s-1',  &
  &                        xtype='double', history=res_hst )

  call HistoryAddVariable( varname='xd', dims=(/'x'/), &
  &                        longname='X-coord double',  &
  &                        units='m', xtype='double', history=res_hst )

  call HistoryAddVariable( varname='yd', dims=(/'y'/), &
  &                        longname='Y-coord double',  &
  &                        units='m', xtype='double', history=res_hst )

  if(present(zopt))then
     call HistoryAddVariable( varname=trim(adjustl(reszoptr)),  &
  &                           dims=(/'x','y'/), longname='vorticity',  &
  &                           units='s-1', xtype='double', history=res_hst )

     call HistoryAddVariable( varname=trim(adjustl(reszopti)),  &
  &                           dims=(/'x','y'/), longname='vorticity',  &
  &                           units='s-1', xtype='double', history=res_hst )
  end if

  call HistoryPut( 'x', xd, history=res_hst )
  call HistoryPut( 'y', yd, history=res_hst )

  call HistoryAddAttr( trim(adjustl(respsir)), trim(adjustl(rest)),  &
  &                    rtn, history=res_hst )
  call HistoryAddAttr( trim(adjustl(respsir)), trim(adjustl(restn)),  &
  &                    itn, history=res_hst )

  do iy=1,ny
     do ix=1,nx
        tmpd(ix,iy)=dble(psik(ix,iy))
     end do
  end do

  call HistoryPut( trim(adjustl(respsir)), tmpd, history=res_hst )

  do iy=1,ny
     do ix=1,nx
        tmpd(ix,iy)=dimag(psik(ix,iy))
     end do
  end do

  call HistoryPut( trim(adjustl(respsii)), tmpd, history=res_hst )

  do iy=1,ny
     do ix=1,nx
        tmpd(ix,iy)=dble(zk(ix,iy))
     end do
  end do

  call HistoryPut( trim(adjustl(reszetar)), tmpd, history=res_hst )

  do iy=1,ny
     do ix=1,nx
        tmpd(ix,iy)=dimag(zk(ix,iy))
     end do
  end do

  call HistoryPut( trim(adjustl(reszetai)), tmpd, history=res_hst )

  call HistoryPut( 'xd', x, history=res_hst )
  call HistoryPut( 'yd', y, history=res_hst )

  if(present(zopt))then

     do iy=1,ny
        do ix=1,nx
           tmpd(ix,iy)=dble(zopt(ix,iy))
        end do
     end do

     call HistoryPut( trim(adjustl(reszoptr)), tmpd, history=res_hst )

     do iy=1,ny
        do ix=1,nx
           tmpd(ix,iy)=dimag(zopt(ix,iy))
        end do
     end do

     call HistoryPut( trim(adjustl(reszopti)), tmpd, history=res_hst )

  end if

  call HistoryClose( history=res_hst )

end subroutine make_restart


subroutine fft_all_clear()
  use fft_saveval_define
  use fft_val_define
  implicit none

  zi=0.0d0

  psid=0.0
  ud=0.0
  vd=0.0
  zd=0.0
  tmpr=0.0d0
  tmpi=0.0d0
  
  ur=0.0d0
  vr=0.0d0
  zor=0.0d0
  psior=0.0d0
  psiko=0.0d0
  zko=0.0d0
  psikn=0.0d0
  zopt1=0.0d0
  uk=0.0d0
  vk=0.0d0

  xi=0.0d0
  yi=0.0d0
  xd=0.0
  yd=0.0
  x=0.0d0
  y=0.0d0
!ORG  t=0.0d0

end subroutine fft_all_clear


end module fftsub_mod
