program make_init
!-- producing initial data of zeta
!-- Version Kuo et al. (2008,MWR).

  use gtool_history
  use Matrix_calc

  implicit none

  integer, parameter :: nc=100

!-- namelist variables

  integer :: nx, ny
  integer :: nnum
  integer, dimension(nc) :: ndip
  real, allocatable, dimension(:) :: xr, yr
  double precision :: xmin, ymin, dx, dy
  double precision :: zcore, R1, alpha
  double precision :: ri, ro
  double precision, dimension(nc) :: xdip, ydip, zdip, R2, adip
  character(1000) :: fname

  integer :: i, j, k
  double precision :: r, theta
  double precision :: tcx, tcy
  double precision, allocatable, dimension(:) :: x, y
  double precision, allocatable, dimension(:,:) :: zeta
  double precision :: makez
  double precision, dimension(4) :: a, b, c
  double precision, dimension(4,4) :: gau

  double precision :: d, gam, rast, del

  namelist /initial /nx, ny, xmin, ymin, dx, dy, fname,  &
  &                  tcx, tcy
  namelist /vprof /zcore, R1, alpha, ri, ro
  namelist /dipole /nnum, xdip, ydip, zdip, adip, ndip, R2
  read(5,nml=initial)
  read(5,nml=vprof)
  read(5,nml=dipole)

  if(nnum>nc)then
     write(*,*) "*** ERROR (main) *** : namelist variable 'nnum' is <=", nnum, '.'
     write(*,*) "STOP."
     stop
  end if

  gam=zcore/zdip(1)
  d=dsqrt((tcx-xdip(1))**2+(tcy-ydip(1))**2)
  del=(d-(R1+R2(1)))/R1
  rast=R1/R2(1)

  write(*,'(a46)') "Initial parameters are set to the following :"
  write(*,'(a23,1PE16.7E3)') "gamma (vor1/vor2)    = ", gam
  write(*,'(a23,1PE16.7E3)') "alpha                = ", alpha
  write(*,'(a23,1PE16.7E3)') "delta (d-(R1+R2))/R1 = ", del
  write(*,'(a23,1PE16.7E3)') "rast R1/R2           = ", rast
  write(*,'(a23,1PE16.7E3)') "d                    = ", d

!-- calculating the function in the intermediate region
  a=0.0d0
  b=0.0d0
  gau=0.0d0
  gau(1:4,1)=(/1.0d0,ro,ro**2,ro**3/)
  gau(1:4,2)=(/0.0d0,1.0d0,2.0d0*ro,3.0d0*(ro**2)/)
  gau(1:4,3)=(/1.0d0,ri,ri**2,ri**3/)
  gau(1:4,4)=(/0.0d0,1.0d0,2.0d0*ri,3.0d0*(ri**2)/)
  b(1)=0.5d0*zcore*(1.0d0-alpha)/(ro**(1.0d0+alpha))
  b(2)=-0.5d0*zcore*(1.0d0-alpha**2)/(ro**(2.0d0+alpha))
  b(3)=zcore

  call gausss( gau, b, a )

  allocate(x(nx))
  allocate(y(ny))
  allocate(xr(nx))
  allocate(yr(ny))
  allocate(zeta(nx,ny))

  x=(/((xmin+dx*dble(i-1)),i=1,nx)/)
  y=(/((ymin+dy*dble(j-1)),j=1,ny)/)

  xr=(/((xmin+dx*real(i-1)),i=1,nx)/)
  yr=(/((ymin+dy*real(j-1)),j=1,ny)/)

  zeta=0.0d0

!-- producing mean zeta.

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,r,theta)

  do j=1,ny
     do i=1,nx
        r=dsqrt((x(i)-tcx)**2+(y(j)-tcy)**2)/R1
        if(x(i)-tcx>0.0d0.and.y(j)-tcy>0.0d0)then
           theta=datan((y(j)-tcy)/(x(i)-tcx))
        else if(x(i)-tcx<0.0d0.and.y(j)-tcy>0.0d0)then
           theta=dabs(dacos(-1.0d0))-datan((y(j)-tcy)/dabs(x(i)-tcx))
        else if(x(i)-tcx<0.0d0.and.y(j)-tcy<0.0d0)then
           theta=dabs(dacos(-1.0d0))+datan(dabs(y(j)-tcy)/dabs(x(i)-tcx))
        else if(x(i)-tcx>0.0d0.and.y(j)-tcy<0.0d0)then
           theta=2.0d0*dabs(dacos(-1.0d0))-datan(dabs(y(j)-tcy)/(x(i)-tcx))
        else if(x(i)-tcx==0.0d0.and.y(j)-tcy<0.0d0)then
           theta=1.5d0*dabs(dacos(-1.0d0))
        else if(x(i)-tcx==0.0d0.and.y(j)-tcy>0.0d0)then
           theta=0.5d0*dabs(dacos(-1.0d0))
        else if(x(i)-tcx>0.0d0.and.y(j)-tcy==0.0d0)then
           theta=0.0d0
        else if(x(i)-tcx<0.0d0.and.y(j)-tcy==0.0d0)then
           theta=dabs(dacos(-1.0d0))
        end if

        if(ri<=r.and.ro>=r)then
           zeta(i,j)=a(1)+a(2)*r+a(3)*(r**2)+a(4)*(r**3)
        else if(r<ri)then
           zeta(i,j)=zcore
        else
           zeta(i,j)=0.5d0*zcore*(1.0d0-alpha)/(r**(1.0d0+alpha))
        end if
     end do
  end do

!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j,k,r,b,c)

!-- producing perturbation zeta.

  do j=1,ny
     do i=1,nx
        do k=1,nnum
           r=dsqrt((x(i)-xdip(k))**2+(y(j)-ydip(k))**2)/R2(k)
           b=0.0d0
           c=0.0d0

           !-- for dip, determining the cubic function.
           b(3)=zdip(k)

           call gausss( gau, b, c )

           if(ri<=r.and.ro>=r)then
              zeta(i,j)=zeta(i,j)+(c(1)+c(2)*r+c(3)*(r**2)+c(4)*(r**3))
           else if(r<ri)then
              zeta(i,j)=zeta(i,j)+zdip(k)
           end if
        end do
     end do
  end do

!$omp end do
!$omp end parallel

  call HistoryCreate( file=trim(adjustl(fname)), title='BAROTRO initial data', &
  & source='test', institution='test', dims=(/'x', 'y'/),  &
  & dimsizes=(/ nx, ny /),  &
  & longnames=(/'X-coordinate','Y-coordinate'/),  &
  & units=(/'m', 'm'/) )

  call HistoryPut( 'x', xr )
  call HistoryPut( 'y', yr )

  call HistoryAddVariable( varname='zeta', dims=(/'x','y'/), &
  &    longname='vorticity', units='s-1', xtype='double' )

  call HistoryAddVariable( varname='xd', dims=(/'x'/), &
  &    longname='X-coord double', units='m', xtype='double' )

  call HistoryAddVariable( varname='yd', dims=(/'y'/), &
  &    longname='Y-coord double', units='m', xtype='double' )

  call HistoryPut( 'zeta', zeta )
  call HistoryPut( 'xd', x )
  call HistoryPut( 'yd', y )

  call HistoryClose()

end program
