program solver
! 簡易版ボックスラグランジュシミュレーションを行う.
! 時間積分は RK4. 
! サウンディングデータを読み込んで, 以下の時間発展方程式を計算する.
! Dz/Dt = w
! Dw/Dt = g*(theta-thetab)/thetab
! Dtheta/Dt = (theta/(cp*T))*D(L*qvs)/Dt
! 
! サウンディングデータのフォーマットは sound_conv で変換されたものと仮定

  use basis
  use Thermo_Const
  use Phys_Const
  use file_operate
  use statistics
  use Thermo_Function

  implicit none

!-- namelist variables
  integer :: lskip    ! サウンディングファイルのヘッダスキップ行
  integer :: nt       ! 時間ステップ数
  integer :: dmpstep  ! 出力ファイルへの出力間隔ステップ数
  double precision :: dt  ! 1 ステップの時間間隔 [s]
  double precision :: z0  ! 初期高度
  double precision :: w0  ! 初期の鉛直速度
  double precision :: undef    ! 出力結果の未定義値
  character(1000) :: sndfile  ! サウンディングファイル
  character(1000) :: outfile  ! 計算結果の出力ファイル
  character(100) :: sundef    ! サウンディングファイルでの未定義値

!-- internal variables
  integer :: i, j, nz, nl, intpz
  double precision :: one6
  double precision, allocatable, dimension(:) :: zsnd, psnd, ptsnd, qvsnd
  double precision :: t, oldqv, pte, ptenv
  double precision, dimension(4) :: z, w, p, pt, temp, qv, qvs, dz, dw, dpt, dqv
  character(100), allocatable, dimension(:,:) :: cval
  character(20) :: forma, formb
  character(1000) :: chead, cunit

!-- input namelist variables
  namelist /input /lskip, nt, dmpstep, dt, z0, w0, undef,  &
  &                sndfile, outfile, sundef
  read(5,nml=input)

  nl=4
  one6=1.0d0/6.0d0

!-- read sndfile
  nz=line_number_counter( trim(adjustl(sndfile)) )-lskip
  allocate(cval(nl,nz))
  allocate(zsnd(nz))
  allocate(psnd(nz))
  allocate(ptsnd(nz))
  allocate(qvsnd(nz))
  call read_file_text( trim(adjustl(sndfile)), nl, nz, cval, skip=lskip )
  call convert_snd_c2r( cval, zsnd, psnd, ptsnd, qvsnd, sundef, undef )

  open(unit=100,file=trim(adjustl(outfile)),status='unknown')
  forma='(96a)'
  formb='(1P6E16.8)'
  chead='Time            Height          W               Theta           '  &
      //'QV              ThetaE          '
  cunit='s               m               ms-1            K               '  &
      //'kgkg-1          K               '
  write(100,trim(adjustl(forma))) trim(adjustl(chead))
  write(100,trim(adjustl(forma))) trim(adjustl(cunit))

!-- calculate the governing equations by RK4
  !-- setting the initial variables
  z(1)=z0
  w(1)=w0
  call interpo_search_1d( zsnd, z(1), intpz )
  call interpolation_1d( zsnd(intpz:intpz+1), psnd(intpz:intpz+1), z(1), p(1) )
  call interpolation_1d( zsnd(intpz:intpz+1), ptsnd(intpz:intpz+1), z(1), pt(1) )
  call interpolation_1d( zsnd(intpz:intpz+1), qvsnd(intpz:intpz+1), z(1), qv(1) )

  t=0.0
  oldqv=qv(1)
  temp(1)=dble(thetaP_2_T( real(pt(1)), real(p(1)) ))
  pte=dble(thetae_Bolton( real(temp(1)), real(qv(1)), real(p(1)) ))
  qvs(1)=dble(TP_2_qvs( real(temp(1)), real(p(1)), 't' ))
  ptenv=pt(1)

  write(100,trim(adjustl(formb))) t, z(1), w(1), pt(1), qv(1), pte

  do i=1,nt
     t=t+dt

     do j=1,4
     !-- step j
        dz(j)=dt*w(j)
        dw(j)=dt*(pt(j)-ptenv)*g/ptenv
        if(qv(j)>qvs(j))then
           dpt(j)=(pt(j)/(cpd_dp*temp(j)))*dble(LH(real(temp(j))))*(qv(j)-qvs(j))
           dqv(j)=qvs(j)-qv(j)
        else
           dpt(j)=0.0d0
           dqv(j)=0.0d0
        end if

        select case (j)
        case (1,2)
           z(j+1)=z(1)+dz(j)*0.5d0
           w(j+1)=w(1)+dw(j)*0.5d0
           pt(j+1)=pt(1)+dpt(j)*0.5d0
           qv(j+1)=qv(1)+dqv(j)*0.5d0

        case (3)
           z(j+1)=z(1)+dz(j)
           w(j+1)=w(1)+dw(j)
           pt(j+1)=pt(1)+dpt(j)
           qv(j+1)=qv(1)+dqv(j)

        case (4)
           z(1)=z(1)+one6*(dz(1)+2.0d0*(dz(2)+dz(3))+dz(4))
           w(1)=w(1)+one6*(dw(1)+2.0d0*(dw(2)+dw(3))+dw(4))
           pt(1)=pt(1)+one6*(dpt(1)+2.0d0*(dpt(2)+dpt(3))+dpt(4))
           qv(1)=qv(1)+one6*(dqv(1)+2.0d0*(dqv(2)+dqv(3))+dqv(4))
        end select

        if(j<4)then
           call interpo_search_1d( zsnd, z(j+1), intpz )
           call interpolation_1d( zsnd(intpz:intpz+1), ptsnd(intpz:intpz+1),  &
  &                               z(j+1), ptenv )
           call interpolation_1d( zsnd(intpz:intpz+1), psnd(intpz:intpz+1),  &
  &                               z(j+1), p(j+1) )
           temp(j+1)=dble(thetaP_2_T( real(pt(j+1)), real(p(j+1)) ))
           qvs(j+1)=dble(TP_2_qvs( real(temp(j+1)), real(p(j+1)), 't' ))

        else if(j==4)then
           call interpo_search_1d( zsnd, z(1), intpz )
           call interpolation_1d( zsnd(intpz:intpz+1), ptsnd(intpz:intpz+1),  &
  &                               z(1), ptenv )
           call interpolation_1d( zsnd(intpz:intpz+1), psnd(intpz:intpz+1),  &
  &                               z(1), p(1) )
           temp(1)=dble(thetaP_2_T( real(pt(1)), real(p(1)) ))
           qvs(1)=dble(TP_2_qvs( real(temp(1)), real(p(1)), 't' ))
           oldqv=qv(1)

        end if
     end do

     if(mod(i,dmpstep)==0)then
        pte=dble(thetae_Bolton( real(temp(1)), real(qv(1)), real(p(1)) ))
        write(100,trim(adjustl(formb))) t, z(1), w(1), pt(1), qv(1), pte
        write(*,*) "file dmp = ", t, "[s]"
     else
        write(*,*) "Passing = ", t, "[s]"
     end if

  end do

  close(unit=100)
  write(*,*) "Stop normally."

contains

subroutine convert_snd_c2r( cprof, zprof, pprof, ptprof, qvprof,  &
  &                         iundef, oundef )
! 読み込んだ文字データを実数データに変換する.
  implicit none
  character(*), dimension(:,:), intent(in) :: cprof
  double precision, dimension(size(cprof,2)), intent(out) :: zprof
  double precision, dimension(size(cprof,2)), intent(out) :: pprof
  double precision, dimension(size(cprof,2)), intent(out) :: ptprof
  double precision, dimension(size(cprof,2)), intent(out) :: qvprof
  character(*), intent(in) :: iundef
  double precision, intent(in) :: oundef
  integer :: kk, kz

  kz=size(cprof,2)

  do kk=1,kz
     if(trim(adjustl(cprof(3,kk)))/=trim(adjustl(iundef)))then
        zprof(kk)=c2r_convert( trim(adjustl(cprof(1,kk))) )
        pprof(kk)=c2r_convert( trim(adjustl(cprof(2,kk))) )
        ptprof(kk)=c2r_convert( trim(adjustl(cprof(3,kk))) )  ! temperature
        ptprof(kk)=dble(theta_dry( real(ptprof(kk)), real(pprof(kk)) ))
        qvprof(kk)=c2r_convert( trim(adjustl(cprof(4,kk))) )
     else
        zprof(kk)=undef
        pprof(kk)=undef
        ptprof(kk)=undef
        qvprof(kk)=undef
     end if
  end do

end subroutine convert_snd_c2r

end program
