program initial_make
!  モデルの計算に必要な初期値ファイルを生成するプログラム
!  必要に応じてソースの書き換えが可能.
!  読み込むサウンディングで鉛直格子を決定する.
  use file_operate
  use Thermo_Const
  use Math_Const
  use Phys_Const
  use Thermo_Function
  use Thermo_Routine
  use Algebra
  use Typhoon_analy
  use gtool_history
  use derivation
  use max_min

  implicit none

!-- namelist valiables
  integer :: nr, nz
  integer :: mom_flag
  real :: coril
  real :: dr, dz
  character(4) :: bc
  character(80) :: fname
  character(80) :: sound_name  ! サウンディングファイルの名前

!-- internal valiables
  integer :: i, j
  real :: n
  real, allocatable, dimension(:) :: r, z, zd, z_logP  ! 軸対称 2 次元の動径, 鉛直座標
  real, allocatable, dimension(:) :: rm, vm  ! 各高度での最大風速と MWR
  real, allocatable, dimension(:) :: r1, r2  ! inner, outer core boundary
  real, allocatable, dimension(:) :: pres_s, temp_s, pt_s, rho_s  ! サウンディング
  real, allocatable, dimension(:,:) :: v0, zph, qv  ! 初期の接線風, 気圧, 加熱率
  real, allocatable, dimension(:,:) :: mom  ! 運動量ソース
  real, allocatable, dimension(:,:) :: theta0, rho, dtheta  ! 初期の温位, 密度
  real, allocatable, dimension(:,:) :: temp0, dtemp  ! 初期の温位, 温度偏差 (モニター)
  real, allocatable, dimension(:,:) :: x  ! 座標変数
  real, allocatable, dimension(:,:) :: coriol  ! コリオリパラメータ
  real, allocatable, dimension(:,:) :: N2  ! 浮力振動数 **2
  character(20), allocatable :: val(:,:)
  real :: nibun_left, nibun_right, func_left, func_right
  real :: err, max_val
  real, allocatable, dimension(:) :: ac

!-- constant value
  integer, parameter :: sline=2  ! サウンディングファイルの読み飛ばし行
  real, parameter :: n_low=1.0  ! eye での速度の傾き
  real, parameter :: n_up=1.3  ! eye での速度の傾き
  integer, parameter :: m_bell=2  ! 非断熱加熱のベル関数の次数
  real, parameter :: x1=300.0e3
  real, parameter :: r_qv=-2.0e3   ! qv と rm の距離
  real, parameter :: dRm=5.0/5.0  ! Rmax の傾き (dr/dz)
!ORG  real, parameter :: dRm=16.0/18.0  ! Rmax の傾き (dr/dz)
  real, parameter :: rmax=15.0e3  ! 地表面での Rmax
  real, parameter :: vmax=63.0e0  ! 地表面での Vmax
  real, parameter :: z_v0=18.0e3  ! vmax がゼロになる高度 (地表からここまで線形に減少する)
  real, parameter :: z_qv=18.0e3  ! qv がゼロになる高度
  real, parameter :: dr12=20.0e3  ! r1 と r2 の距離
  real, parameter :: err_max=1.0e-5  ! 二分法の収束条件

!-- function name
  real :: func_a, Bell

!-- namelist からのパラメータの代入
  namelist /input /nr, nz, dr, dz, bc, fname, sound_name, coril, mom_flag
  read(5,nml=input)

!-- 配列の割付
  allocate(r(nr))
  allocate(z(nz))
  allocate(z_logP(nz))  ! 非断熱加熱の分布計算で用いる.
  allocate(zd(nz))  ! 非断熱加熱の分布計算で用いる.
  allocate(rm(nz))
  allocate(vm(nz))
  allocate(r1(nz))
  allocate(r2(nz))
  allocate(pres_s(nz))
  allocate(temp_s(nz))
  allocate(pt_s(nz))
  allocate(rho_s(nz))
  allocate(val(4,nz))
  allocate(v0(nr,nz))
  allocate(zph(nr,nz))
  allocate(qv(nr,nz))
  allocate(mom(nr,nz))
  allocate(theta0(nr,nz))
  allocate(dtheta(nr,nz))
  allocate(temp0(nr,nz))
  allocate(dtemp(nr,nz))
  allocate(rho(nr,nz))
  allocate(x(nr,nz))
  allocate(coriol(nr,nz))
  allocate(N2(nr,nz))
  allocate(ac(nz))

  coriol=2.0*omega*sin(coril*pi/180.0)
  mom=0.0  ! 現在のところ, 運動量ソースは initial_make では陽に作成しない.

!-- サウンディングから基本場の熱力学変数・鉛直座標を設定.
!-- サウンディングのデータ数は namelist で設定されているのと同じ値.

  call read_file_text( trim(sound_name), 4, nz, val, skip=sline )

  do i=1,nz
     read(val(1,i),*) z(i)
     read(val(2,i),*) temp_s(i)
     read(val(3,i),*) pres_s(i)
     read(val(4,i),*) pt_s(i)
     rho_s(i)=TP_2_rho( temp_s(i), pres_s(i) )
     z_logP(i)=-(Rd*300.0/g)*log(pres_s(i)/1000.0e2)
write(*,*) rho_s(i)
  end do

!-- 座標値の設定
  r=(/(dr*real(i-1),i=1,nr)/)

!-- 傾度風場の生成に必要なパラメータの設定
  rm(1)=rmax
  vm(1)=vmax
  do j=2,nz
!ORG     rm(j)=rm(1)+dRm*z(j)
     if(z_logP(j)<5000.0)then
        rm(j)=rm(1)
        vm(j)=vm(1)
     else
        rm(j)=rm(1)+dRm*(z_logP(j)-5000.0)
        vm(j)=vm(1)-(z_logP(j)/5.0e3)
!        vm(j)=vm(1)
     end if
!ORG     vm(j)=vm(1)-vm(1)*(z(j)/z_v0)
!     vm(j)=vm(1)*func_a((z_v0-z(j))/z_v0)
  end do

!-- r1, r2 の決定 (Willoughby 2006)
  do j=1,nz
     if(z_logP(j)<5000.0)then
        n=n_low+(n_up-n_low)*(z_logP(j)/5000.0)
     else
        n=n_up
     end if
     ac(j)=(real(n)*x1)/(real(n)*x1+rm(j))
!-- r1 を求めるため, 初期値を 0, 1 とした二分法開始
     nibun_left=0.0
     nibun_right=1.0
     func_left=func_a(nibun_left)-ac(j)
     func_right=func_a(nibun_right)-ac(j)
     do while(func_left>err_max)
        err=0.0
        func_left=func_a((nibun_left+nibun_right)*0.5)-ac(j)
        func_right=func_a(nibun_right)-ac(j)
        if(func_left*func_right<0.0)then
           nibun_left=(nibun_left+nibun_right)*0.5
        else
           nibun_right=(nibun_left+nibun_right)*0.5
        end if
     end do
     r1(j)=rm(j)-dr12*0.5*(nibun_left+nibun_right)
     r2(j)=r1(j)+dr12
  end do

  do j=1,nz
     do i=1,nr
        x(i,j)=(r(i)-r1(j))/(r2(j)-r1(j))
     end do
  end do

!-- 初期値の生成
!-- 傾度風場の生成
  do j=1,nz
     do i=1,nr
        if(z_logP(j)<5000.0.and.z_logP(j)>=2000.0)then
           n=n_low+(n_up-n_low)*((z_logP(j)-2000.0)/3000.0)
        else if(z_logP(j)<2000.0)then
           n=n_low
        else
           n=n_up
        end if
!ORG        if(r(i)<r1(j))then
        if(r(i)<rm(j))then
           v0(i,j)=vm(j)*(r(i)/rm(j))**n
        else
!ORG           if(r1(j)<=r(i).and.r(i)<=r2(j))then
!ORG              v0(i,j)=(vm(j)*(r(i)/rm(j))**n_low)*(1.0-func_a(x(i,j)))  &
!ORG  &                   +(vm(j)*exp(-(r(i)-rm(j))/x1))*func_a(x(i,j))
!ORG           else
!ORG              v0(i,j)=vm(j)*exp(-(r(i)-rm(j))/x1)
           v0(i,j)=vm(j)*(rm(j)/r(i))**0.3
!ORG           end if
        end if
        if(z_logP(j)>=z_v0)then
           v0(i,j)=0.0
        end if
     end do
  end do

!-- サウンディングと軸対称流から静力学・傾度風平衡場の生成.

  call hydro_grad_eqbp( r, pres_s, coriol, v0, z, temp_s, zph, temp0 )

!-- ソルバに必要な pt を計算

  do j=1,nz
     do i=1,nr
        theta0(i,j)=theta_dry( temp0(i,j), pres_s(j) )
     end do
  end do

!-- モニター変数として, 外側半径からの温位偏差を計算

  do j=1,nz
     do i=1,nr
        dtheta(i,j)=theta0(i,j)-theta0(nr,j)
        dtemp(i,j)=temp0(i,j)-temp0(nr,j)
     end do
  end do

!-- 加熱率の計算
!-- 浮力振動数の計算
  do i=1,nr
     call grad_1d( z, theta0(i,:), N2(i,:) )
     do j=1,nz
        N2(i,j)=g*N2(i,j)/theta0(i,j)
     end do
  end do

!  call max_val_2d( N2, maxnx, maxny, max_val )  ! この値は適宜変更
  max_val=1.5e-4

!-- zd の設定
  zd=(/(((z_qv-z(j))/(z_qv-5.0e3)),j=1,nz)/)

  do j=1,nz
     do i=1,nr
        qv(i,j)=max_val*func_a(zd(j))*Bell( m_bell, x(i,j) )
        qv(i,j)=qv(i,j)*Cpd*theta0(i,j)/g
     end do
  end do


!-- 初期ファイルへの書き込み

!-- 以下は GrADS 形式
!  call write_file( trim(fname), nr, nz, 1, v0 )
!  call write_file( trim(fname), nr, nz, 2, pres, mode='old' )
!  call write_file( trim(fname), nr, nz, 3, theta0, mode='old' )
!  call write_file( trim(fname), nr, nz, 4, rho, mode='old' )
!  call write_file( trim(fname), nr, nz, 5, qv, mode='old' )

!-- 以下は netcdf 形式
  call HistoryCreate( file=trim(fname), title='SEQ initial data', &
    & source='test', institution='test', dims=(/'r','z'/),  &
    & dimsizes=(/nr,nz/),  &
    & longnames=(/'Radius       ','Pseudo-height'/),  &
    & units=(/'m','m'/), origin=0.0, interval=0.0 )

  call HistoryPut( 'r', r )
  call HistoryPut( 'z', z_logP )

  call HistoryAddVariable( varname='p', dims=(/'z'/), &
    & longname='Pressure', units='Pa', xtype='float')

  call HistoryPut( 'p', pres_s )

  call HistoryAddVariable( varname='v', dims=(/'r','z'/), &
    & longname='tangential wind', units='m/s', xtype='float')

  call HistoryPut( 'v',v0 )

  call HistoryAddVariable( varname='zph', dims=(/'r','z'/), &
    & longname='geopotential height', units='m', xtype='float')

  call HistoryPut( 'zph', zph )

  call HistoryAddVariable( varname='theta', dims=(/'r','z'/), &
    & longname='potential temperature', units='K', xtype='float')

  call HistoryPut( 'theta', theta0 )

  call HistoryAddVariable( varname='dtheta', dims=(/'r','z'/), &
    & longname='potential temperature perturbation from the outermost radius', units='K', xtype='float')

  call HistoryPut( 'dtheta', dtheta )

  call HistoryAddVariable( varname='temp', dims=(/'r','z'/), &
    & longname='temperature', units='K', xtype='float')

  call HistoryPut( 'temp', temp0 )

  call HistoryAddVariable( varname='dtemp', dims=(/'r','z'/), &
    & longname='temperature perturbation from the outermost radius', units='K', xtype='float')

  call HistoryPut( 'dtemp', dtemp )

  call HistoryAddVariable( varname='rho', dims=(/'r','z'/), &
    & longname='density', units='kg/m3', xtype='float')

  call HistoryPut( 'rho', rho )

  call HistoryAddVariable( varname='Q', dims=(/'r','z'/), &
    & longname='heating rate', units='J/s', xtype='float')

  call HistoryPut( 'Q', qv )

  call HistoryAddVariable( varname='mom', dims=(/'r','z'/), &
    & longname='momentum source rate', units='m/s2', xtype='float')

  call HistoryPut( 'mom', mom )

  call HistoryClose

!-- 終了のお知らせ

  write(*,*) "initial_make is normally conplete."

end program


real function func_a( val )
  implicit none
  real, intent(in) :: val  ! 座標変数

  if(val<=0.0)then
     func_a=0.0
  end if

  if(val>=1.0)then
     func_a=1.0
  end if

  if(val>0.0.and.val<1.0)then
     func_a=(10.0-15.0*val+6.0*val*val)*val*val*val
  end if

  return

end function


real function Bell( n, val )
!  Willoughby 2006 で提唱されたベル型関数を計算する.
  implicit none
  integer, intent(in) :: n  ! 次数
  real, intent(in) :: val  ! 引数

  if(val>=0.0.and.val<=1.0)then
     Bell=(2.0**(2*n))*((val*(1.0-val))**n)
  else
     Bell=0.0
  end if

  return

end function
