!---------------------------------------------------------------
! Copyright (C) 2009-2015 GFD Dennou Club. All rights reserved.
!---------------------------------------------------------------

module Geometry  ! 幾何学に関するルーチン

  public :: rt_2_xy,  &
  &         xy_2_rt

  interface rt_2_xy
  module procedure rt_2_xy_r,  &
  &                rt_2_xy_d
  end interface rt_2_xy

  interface xy_2_rt
  module procedure xy_2_rt_r,  &
  &                xy_2_rt_d
  end interface xy_2_rt

contains

subroutine rt_2_xy_r( r, t, x, y )
! 平面極座標から平面デカルト座標への変換
!
! この場合は座標の原点が r で自動的に指定されているので,
! 原点を指定する引数は必要ない.
! 逆に返される x, y の値は, r=0 を原点とした
! デカルト座標系として返されることに注意する.
  implicit none
  real, intent(in) :: r  ! 中心からの距離
  real, intent(in) :: t  ! x 軸からの角度 [rad]
  real, intent(inout) :: x  ! x 座標
  real, intent(inout) :: y  ! y 座標

  x=r*cos(t)
  y=r*sin(t)

end subroutine


subroutine rt_2_xy_d( r, t, x, y )
! 平面極座標から平面デカルト座標への変換
!
! この場合は座標の原点が r で自動的に指定されているので,
! 原点を指定する引数は必要ない.
! 逆に返される x, y の値は, r=0 を原点とした
! デカルト座標系として返されることに注意する.
  implicit none
  double precision, intent(in) :: r  ! 中心からの距離
  double precision, intent(in) :: t  ! x 軸からの角度 [rad]
  double precision, intent(inout) :: x  ! x 座標
  double precision, intent(inout) :: y  ! y 座標

  x=r*dcos(t)
  y=r*dsin(t)

end subroutine


subroutine xy_2_rt_r( x, y, xc, yc, r, t )
! 平面デカルト座標から平面極座標への変換
!
! この場合は座標の原点が r で自動的に指定されているので,
! 原点を指定する引数は必要ない.
! 逆に返される x, y の値は, r=0 を原点とした
! デカルト座標系として返されることに注意する.
  use Math_Const
  implicit none
  real, intent(in) :: x  ! x 座標
  real, intent(in) :: y  ! y 座標
  real, intent(in) :: xc ! x 座標中心点
  real, intent(in) :: yc ! y 座標中心点
  real, intent(inout) :: r  ! 中心からの距離
  real, intent(inout) :: t  ! x 軸からの角度 [rad]
  real :: rx, ry

  rx=x-xc
  ry=y-yc

  r=sqrt(rx**2+ry**2)

  if(rx==0.0.and.ry==0.0)then
     t=0.0
  else if(rx==0.0.and.ry/=0.0)then
     if(ry>0.0)then
        t=0.5*pi
     else
        t=1.5*pi
     end if
  else if(rx/=0.0.and.ry==0.0)then
     if(rx>0.0)then
        t=0.0
     else
        t=pi
     end if
  else
     if(rx>0.0.and.ry>0.0)then
        t=atan(ry/rx)
     else if(rx<0.0.and.ry>0.0)then
        t=pi-atan(ry/abs(rx))
     else if(rx>0.0.and.ry<0.0)then
        t=2.0*pi-atan(abs(ry)/rx)
     else if(rx<0.0.and.ry<0.0)then
        t=pi+atan(ry/rx)
     end if
  end if

end subroutine


subroutine xy_2_rt_d( x, y, xc, yc, r, t )
! 平面デカルト座標から平面極座標への変換
!
! この場合は座標の原点が r で自動的に指定されているので,
! 原点を指定する引数は必要ない.
! 逆に返される x, y の値は, r=0 を原点とした
! デカルト座標系として返されることに注意する.
  use Math_Const
  implicit none
  double precision, intent(in) :: x  ! x 座標
  double precision, intent(in) :: y  ! y 座標
  double precision, intent(in) :: xc ! x 座標中心点
  double precision, intent(in) :: yc ! y 座標中心点
  double precision, intent(inout) :: r  ! 中心からの距離
  double precision, intent(inout) :: t  ! x 軸からの角度 [rad]
  double precision :: rx, ry

  rx=x-xc
  ry=y-yc

  r=dsqrt(rx**2+ry**2)

  if(rx==0.0d0.and.ry==0.0d0)then
     t=0.0d0
  else if(rx==0.0d0.and.ry/=0.0d0)then
     if(ry>0.0d0)then
        t=0.5d0*pi_dp
     else
        t=1.5d0*pi_dp
     end if
  else if(rx/=0.0d0.and.ry==0.0d0)then
     if(rx>0.0d0)then
        t=0.0d0
     else
        t=pi_dp
     end if
  else
     if(rx>0.0d0.and.ry>0.0d0)then
        t=datan(ry/rx)
     else if(rx<0.0d0.and.ry>0.0d0)then
        t=pi_dp-datan(ry/dabs(rx))
     else if(rx>0.0d0.and.ry<0.0d0)then
        t=2.0d0*pi_dp-datan(dabs(ry)/rx)
     else if(rx<0.0d0.and.ry<0.0d0)then
        t=pi_dp+datan(ry/rx)
     end if
  end if

end subroutine


subroutine product_circle( xp, yp, rad, num, xd, yd )
! デカルト点 xp, yp から半径 rad の円を描き, その値を
! 極座標の同位角方向に num 個の配列要素に格納する.
! xd, yd がその円のデカルト座標での値.
  use math_const
  implicit none
  real, intent(in) :: xp  ! 円の中心 x 座標
  real, intent(in) :: yp  ! 円の中心 y 座標
  real, intent(in) :: rad  ! 円の半径
  integer, intent(in) :: num  ! 配列要素数
  real, intent(inout) :: xd(num)  ! 円の x 座標データ
  real, intent(inout) :: yd(num)  ! 円の y 座標データ
  integer :: i

  do i=1,num
     xd(i)=xp+rad*cos(real(i-1)*2.0*pi/real(num-1))
     yd(i)=yp+rad*sin(real(i-1)*2.0*pi/real(num-1))
  end do

end subroutine

!subroutine xy_2_rt( x, y, r, t )
! 平面デカルト座標から平面極座標への変換
!
! x=0, y=0 の点が r=0 の原点となって返されることに注意.
! この点が平面極座標での原点となる.
!  implicit none
!  real, intent(in) :: x  ! x 座標
!  real, intent(in) :: y  ! y 座標
!  real, intent(inout) :: r  ! 中心からの距離
!  real, intent(inout) :: t  ! x 軸からの角度 [rad]

!  r=sqrt(x**2+y**2)  ! これはどの象限でも同じ計算式で評価できる.

!  if(x=0.0.and.y=0.0)then
!     t=0.0
!  else
!     if(x*y>0.0)then  ! x*y>0 なら, 第 1, 3 象限領域を表す.
!        t=
!     else
!     end if
!  end if

!end subroutine

!-------------------------------------------


end module
