!---------------------------------------------------------------
! Copyright (C) 2009-2015 GFD Dennou Club. All rights reserved.
!---------------------------------------------------------------

module Statistics  ! 統計解析関係のルーチン集

  use Basis
  use Matrix_Calc
  use Max_Min
  use Map_Function

  private :: summ, summf, summd
  private :: replace_undef_f, replace_undef_d

interface Mean_1d

  module procedure Mean_1df, Mean_1dd

end interface Mean_1d

interface Mean_2d

  module procedure Mean_2df, Mean_2dd

end interface Mean_2d

interface Mean_3d

  module procedure Mean_3df, Mean_3dd

end interface Mean_3d

interface Median_1d

  module procedure Median_1df, Median_1dd

end interface Median_1d

interface Median_2d

  module procedure Median_2df, Median_2dd

end interface Median_2d

interface Median_3d

  module procedure Median_3df, Median_3dd

end interface Median_3d

interface Quartile_1d

  module procedure Quartile_1df, Quartile_1dd

end interface Quartile_1d

interface Quartile_2d

  module procedure Quartile_2df, Quartile_2dd

end interface Quartile_2d

interface Quartile_3d

  module procedure Quartile_3df, Quartile_3dd

end interface Quartile_3d

interface Quantile_1d

  module procedure Quantile_1df, Quantile_1dd

end interface Quantile_1d

interface Quantile_2d

  module procedure Quantile_2df, Quantile_2dd

end interface Quantile_2d

interface Quantile_3d

  module procedure Quantile_3df, Quantile_3dd

end interface Quantile_3d

interface Freqbin_count_1d

  module procedure Freqbin_count_1df, Freqbin_count_1dd

end interface Freqbin_count_1d

interface Freqbin_count_2d

  module procedure Freqbin_count_2df, Freqbin_count_2dd

end interface Freqbin_count_2d

interface Freqbin_count_3d

  module procedure Freqbin_count_3df, Freqbin_count_3dd

end interface Freqbin_count_3d

interface Anomaly_1d

  module procedure Anomaly_1df, Anomaly_1dd

end interface Anomaly_1d

interface Anomaly_2d

  module procedure Anomaly_2df, Anomaly_2dd

end interface Anomaly_2d

interface Anomaly_3d

  module procedure Anomaly_3df, Anomaly_3dd

end interface Anomaly_3d

interface stand_devi_1d

  module procedure stand_devi_1df, stand_devi_1dd

end interface stand_devi_1d

interface stand_devi_2d

  module procedure stand_devi_2df, stand_devi_2dd

end interface stand_devi_2d

interface stand_devi_3d

  module procedure stand_devi_3df, stand_devi_3dd

end interface stand_devi_3d

interface covariance_1d

  module procedure covariance_1df, covariance_1dd

end interface covariance_1d

interface covariance_2d

  module procedure covariance_2df, covariance_2dd

end interface covariance_2d

interface covariance_3d

  module procedure covariance_3df, covariance_3dd

end interface covariance_3d

interface nearest_neighbor_search_1d

  module procedure nearest_neighbor_search_1df, nearest_neighbor_search_1dd

end interface nearest_neighbor_search_1d

interface nearest_neighbor_search_2d

  module procedure nearest_neighbor_search_2df, nearest_neighbor_search_2dd

end interface nearest_neighbor_search_2d

interface nearest_neighbor_search_3d

  module procedure nearest_neighbor_search_3df, nearest_neighbor_search_3dd

end interface nearest_neighbor_search_3d

interface nearest_search_1d

  module procedure nearest_search_1df, nearest_search_1dd

end interface nearest_search_1d

interface nearest_search_2d

  module procedure nearest_search_2df, nearest_search_2dd

end interface nearest_search_2d

interface nearest_search_3d

  module procedure nearest_search_3df, nearest_search_3dd

end interface nearest_search_3d

interface interpo_search_1d

  module procedure interpo_search_1df, interpo_search_1dd

end interface interpo_search_1d

interface interpo_search_2d

  module procedure interpo_search_2df, interpo_search_2dd

end interface interpo_search_2d

interface interpo_search_3d

  module procedure interpo_search_3df, interpo_search_3dd

end interface interpo_search_3d

interface auto_interpo_search_1d

  module procedure auto_interpo_search_1df, auto_interpo_search_1dd

end interface auto_interpo_search_1d

interface auto_interpo_search_2d

  module procedure auto_interpo_search_2df, auto_interpo_search_2dd

end interface auto_interpo_search_2d

interface auto_interpo_search_3d

  module procedure auto_interpo_search_3df, auto_interpo_search_3dd

end interface auto_interpo_search_3d

interface interpolation_1d

  module procedure interpolation_1df, interpolation_1dd

end interface interpolation_1d

interface interpolation_2d

  module procedure interpolation_2df, interpolation_2dd

end interface interpolation_2d

interface interpolation_3d

  module procedure interpolation_3df, interpolation_3dd

end interface interpolation_3d

interface interpo_undef_1d

  module procedure interpo_undef_1df, interpo_undef_1dd

end interface interpo_undef_1d

interface auto_interpolation_1d

  module procedure auto_interpolation_1df, auto_interpolation_1dd

end interface auto_interpolation_1d

interface auto_interpolation_2d

  module procedure auto_interpolation_2df, auto_interpolation_2dd

end interface auto_interpolation_2d

interface auto_interpolation_3d

  module procedure auto_interpolation_3df, auto_interpolation_3dd

end interface auto_interpolation_3d

interface tri_interpolation

  module procedure tri_interpolation_f, tri_interpolation_d

end interface tri_interpolation

interface tri_interpolation_2d

  module procedure tri_interpolation_2df, tri_interpolation_2dd

end interface tri_interpolation_2d

interface LSM_1d

  module procedure LSM_1df, LSM_1dd

end interface LSM_1d

interface LSM_2d

  module procedure LSM_2df, LSM_2dd

end interface LSM_2d

interface LSM_3d

  module procedure LSM_3df, LSM_3dd

end interface LSM_3d

interface LSM_poly_1d

  module procedure LSM_poly_1df, LSM_poly_1dd

end interface LSM_poly_1d

interface LSM_poly_2d

  module procedure LSM_poly_2df, LSM_poly_2dd

end interface LSM_poly_2d

interface LSM_poly_3d

  module procedure LSM_poly_3df, LSM_poly_3dd

end interface LSM_poly_3d

interface LSM_multi

  module procedure LSM_multi_f, LSM_multi_d

end interface LSM_multi

interface Cor_Coe_1d

  module procedure Cor_Coe_1df, Cor_Coe_1dd

end interface Cor_Coe_1d

interface Cor_Coe_2d

  module procedure Cor_Coe_2df, Cor_Coe_2dd

end interface Cor_Coe_2d

interface Cor_Coe_3d

  module procedure Cor_Coe_3df, Cor_Coe_3dd

end interface Cor_Coe_3d

interface Reg_Line_1d

  module procedure Reg_Line_1df, Reg_Line_1dd

end interface Reg_Line_1d

interface Reg_Line_2d

  module procedure Reg_Line_2df, Reg_Line_2dd

end interface Reg_Line_2d

interface Reg_Line_3d

  module procedure Reg_Line_3df, Reg_Line_3dd

end interface Reg_Line_3d

interface spline_3

  module procedure spline_3_f, spline_3_d

end interface spline_3

interface Bubble_Sort
! Bubble_Sort

  module procedure Bubble_Sort_i, Bubble_Sort_f, Bubble_Sort_d

end interface Bubble_Sort

interface Quick_Sort
! Quick_Sort

  module procedure Quick_Sort_i, Quick_Sort_f, Quick_Sort_d

end interface Quick_Sort

interface detrend_1d
! detrend_1d

  module procedure detrend_1df, detrend_1dd

end interface detrend_1d

interface detrend_2d
! detrend_2d

  module procedure detrend_2df, detrend_2dd

end interface detrend_2d

!--- interface

contains

!--- subroutines

subroutine Mean_1df( x, ave, error, nc )  ! 1 次元配列平均値計算ルーチン
  implicit none
  real, intent(in) :: x(:)  ! データ
  real, intent(inout) :: ave  ! 計算する平均値
  real, intent(in), optional :: error  ! 欠損値が存在するデータセットの欠損値
  integer, intent(inout), optional :: nc  ! 欠損値が存在するデータセットの平均
                                          ! に用いたデータ数.
  integer :: i, nt
  integer :: nx  ! データの要素数
  real :: summ

  summ=0.0
  nt=0
  nx=size(x)

  if(present(error))then
     do i=1,nx
        if(x(i)/=error)then
           summ=summ+x(i)
           nt=1+nt
        end if
     end do

     if(nt/=0)then
        ave=summ/real(nt)
     else
        ave=error
     end if

     if(present(nc))then
        nc=nt
     end if

  else

     do i=1,nx
        summ=summ+x(i)
     end do

     ave=summ/real(nx)

  end if

end subroutine Mean_1df


subroutine Mean_1dd( x, ave, error, nc )  ! 1 次元配列平均値計算ルーチン
  implicit none
  double precision, intent(in) :: x(:)  ! データ
  double precision, intent(inout) :: ave  ! 計算する平均値
  double precision, intent(in), optional :: error  ! 欠損値が存在するデータセットの欠損値
  integer, intent(inout), optional :: nc  ! 欠損値が存在するデータセットの平均
                                          ! に用いたデータ数.
  integer :: i, nt
  integer :: nx  ! データの要素数
  double precision :: summ

  summ=0.0d0
  nt=0
  nx=size(x)

  if(present(error))then
     do i=1,nx
        if(x(i)/=error)then
           summ=summ+x(i)
           nt=1+nt
        end if
     end do

     if(nt/=0)then
        ave=summ/dble(nt)
     else
        ave=error
     end if

     if(present(nc))then
        nc=nt
     end if

  else

     do i=1,nx
        summ=summ+x(i)
     end do

     ave=summ/dble(nx)

  end if

end subroutine Mean_1dd


subroutine Mean_2df( x, ave, error, nc )  ! 2 次元配列平均値計算ルーチン
  implicit none
  real, intent(in) :: x(:,:)  ! データ
  real, intent(inout) :: ave  ! 計算する平均値
  real, intent(in), optional :: error  ! 欠損値が存在するデータセットの欠損値
  integer, intent(inout), optional :: nc  ! 欠損値が存在するデータセットの平均
                                          ! に用いたデータ数.
  integer :: i, j, nt
  integer :: nx  ! データの要素数 1
  integer :: ny  ! データの要素数 2
  real :: summ

  summ=0.0
  nt=0
  nx=size(x,1)
  ny=size(x,2)

  if(present(error))then
     do j=1,ny
        do i=1,nx
           if(x(i,j)/=error)then
              summ=summ+x(i,j)
              nt=1+nt
           end if
        end do
     end do

     if(nt/=0)then
        ave=summ/real(nt)
     else
        ave=error
     end if

     if(present(nc))then
        nc=nt
     end if

  else

     do j=1,ny
        do i=1,nx
           summ=summ+x(i,j)
        end do
     end do

     ave=summ/real(nx*ny)

  end if

end subroutine Mean_2df


subroutine Mean_2dd( x, ave, error, nc )  ! 2 次元配列平均値計算ルーチン
  implicit none
  double precision, intent(in) :: x(:,:)  ! データ
  double precision, intent(inout) :: ave  ! 計算する平均値
  double precision, intent(in), optional :: error  ! 欠損値が存在するデータセットの欠損値
  integer, intent(inout), optional :: nc  ! 欠損値が存在するデータセットの平均
                                          ! に用いたデータ数.
  integer :: i, j, nt
  integer :: nx  ! データの要素数 1
  integer :: ny  ! データの要素数 2
  double precision :: summ

  summ=0.0d0
  nt=0
  nx=size(x,1)
  ny=size(x,2)

  if(present(error))then
     do j=1,ny
        do i=1,nx
           if(x(i,j)/=error)then
              summ=summ+x(i,j)
              nt=1+nt
           end if
        end do
     end do

     if(nt/=0)then
        ave=summ/dble(nt)
     else
        ave=error
     end if

     if(present(nc))then
        nc=nt
     end if

  else

     do j=1,ny
        do i=1,nx
           summ=summ+x(i,j)
        end do
     end do

     ave=summ/dble(nx*ny)

  end if

end subroutine Mean_2dd


subroutine Mean_3df( x, ave, error, nc )  ! 3 次元配列平均値計算ルーチン
  implicit none
  real, intent(in) :: x(:,:,:)  ! データ
  real, intent(inout) :: ave  ! 計算する平均値
  real, intent(in), optional :: error  ! 欠損値が存在するデータセットの欠損値
  integer, intent(inout), optional :: nc  ! 欠損値が存在するデータセットの平均
                                          ! に用いたデータ数.
  integer :: i, j, k, nt
  integer :: nx  ! データの要素数 1
  integer :: ny  ! データの要素数 2
  integer :: nz  ! データの要素数 2
  real :: summ

  summ=0.0
  nt=0
  nx=size(x,1)
  ny=size(x,2)
  nz=size(x,3)

  if(present(error))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(x(i,j,k)/=error)then
                 summ=summ+x(i,j,k)
                 nt=1+nt
              end if
           end do
        end do
     end do

     if(nt/=0)then
        ave=summ/real(nt)
     else
        ave=error
     end if

     if(present(nc))then
        nc=nt
     end if

  else

     do k=1,nz
        do j=1,ny
           do i=1,nx
              summ=summ+x(i,j,k)
           end do
        end do
     end do

     ave=summ/real(nx*ny*nz)

  end if

end subroutine Mean_3df


subroutine Mean_3dd( x, ave, error, nc )  ! 3 次元配列平均値計算ルーチン
  implicit none
  double precision, intent(in) :: x(:,:,:)  ! データ
  double precision, intent(inout) :: ave  ! 計算する平均値
  double precision, intent(in), optional :: error  ! 欠損値が存在するデータセットの欠損値
  integer, intent(inout), optional :: nc  ! 欠損値が存在するデータセットの平均
                                          ! に用いたデータ数.
  integer :: i, j, k, nt
  integer :: nx  ! データの要素数 1
  integer :: ny  ! データの要素数 2
  integer :: nz  ! データの要素数 2
  double precision :: summ

  summ=0.0d0
  nt=0
  nx=size(x,1)
  ny=size(x,2)
  nz=size(x,3)

  if(present(error))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(x(i,j,k)/=error)then
                 summ=summ+x(i,j,k)
                 nt=1+nt
              end if
           end do
        end do
     end do

     if(nt/=0)then
        ave=summ/dble(nt)
     else
        ave=error
     end if

     if(present(nc))then
        nc=nt
     end if

  else

     do k=1,nz
        do j=1,ny
           do i=1,nx
              summ=summ+x(i,j,k)
           end do
        end do
     end do

     ave=summ/dble(nx*ny*nz)

  end if

end subroutine Mean_3dd


subroutine Median_1df( x, med, error, nc )  ! 1 次元配列中央値計算ルーチン
  implicit none
  real, intent(in) :: x(:)  ! データ
  real, intent(inout) :: med  ! 計算する中央値
  real, intent(in), optional :: error  ! 欠損値が存在するデータセットの欠損値
  integer, intent(inout), optional :: nc  ! 欠損値が存在するデータセットの平均
                                          ! に用いたデータ数.
  integer :: i, nt
  integer :: nx  ! データの要素数
  real :: rx(size(x)), sx(size(x))

  nt=0
  rx=0.0
  nx=size(x)

  if(present(error))then
     do i=1,nx
        if(x(i)/=error)then
           nt=1+nt
           rx(nt)=x(i)
        end if
     end do

     if(nt/=0)then
        call Quick_Sort_f( rx(1:nt), sx(1:nt), 'i' )
        if(mod(nt,2)==0)then  ! データ数が偶数
           med=0.5*(sx(nt/2)+sx(nt/2+1))
        else  ! データ数が奇数
           med=sx(nt/2+1)
        end if
     else
        med=error
     end if

     if(present(nc))then
        nc=nt
     end if

  else

     call Quick_Sort_f( x(1:nx), sx(1:nx), 'i' )
     if(mod(nx,2)==0)then  ! データ数が偶数
        med=0.5*(sx(nx/2)+sx(nx/2+1))
     else  ! データ数が奇数
        med=sx(nx/2+1)
     end if

     if(present(nc))then
        nc=nx
     end if

  end if

end subroutine Median_1df


subroutine Median_1dd( x, med, error, nc )  ! 1 次元配列中央値計算ルーチン
  implicit none
  double precision, intent(in) :: x(:)  ! データ
  double precision, intent(inout) :: med  ! 計算する中央値
  double precision, intent(in), optional :: error  ! 欠損値が存在するデータセットの欠損値
  integer, intent(inout), optional :: nc  ! 欠損値が存在するデータセットの平均
                                          ! に用いたデータ数.
  integer :: i, nt
  integer :: nx  ! データの要素数
  double precision :: rx(size(x)), sx(size(x))

  nt=0
  rx=0.0
  nx=size(x)

  if(present(error))then
     do i=1,nx
        if(x(i)/=error)then
           nt=1+nt
           rx(nt)=x(i)
        end if
     end do

     if(nt/=0)then
        call Quick_Sort_d( rx(1:nt), sx(1:nt), 'i' )
        if(mod(nt,2)==0)then  ! データ数が偶数
           med=0.5d0*(sx(nt/2)+sx(nt/2+1))
        else  ! データ数が奇数
           med=sx(nt/2+1)
        end if
     else
        med=error
     end if

     if(present(nc))then
        nc=nt
     end if

  else

     call Quick_Sort_d( x(1:nx), sx(1:nx), 'i' )
     if(mod(nx,2)==0)then  ! データ数が偶数
        med=0.5d0*(sx(nx/2)+sx(nx/2+1))
     else  ! データ数が奇数
        med=sx(nx/2+1)
     end if

     if(present(nc))then
        nc=nx
     end if

  end if

end subroutine Median_1dd


subroutine Median_2df( x, med, error, nc )  ! 2 次元配列中央値計算ルーチン
  implicit none
  real, intent(in) :: x(:,:)  ! データ
  real, intent(inout) :: med  ! 計算する中央値
  real, intent(in), optional :: error  ! 欠損値が存在するデータセットの欠損値
  integer, intent(inout), optional :: nc  ! 欠損値が存在するデータセットの平均
                                          ! に用いたデータ数.
  integer :: i, j, nt
  integer :: nx  ! データの要素数 1
  integer :: ny  ! データの要素数 2
  logical :: error_flag  ! 未定義を返すかどうかの処理.
  real :: rx(size(x,1)*size(x,2))

  !-- 配列を 2 次元から 1 次元へ置き換え

  nt=0
  nx=size(x,1)
  ny=size(x,2)
  error_flag=.false.

  if(present(error))then
     do j=1,ny
        do i=1,nx
           if(x(i,j)/=error)then
              nt=1+nt
              rx(nt)=x(i,j)
           end if
        end do
     end do

     if(nt==0)then
        med=error
        error_flag=.true.
     end if

  else

     do j=1,ny
        do i=1,nx
           nt=1+nt
           rx(nt)=x(i,j)
        end do
     end do

  end if

  if(present(nc))then
     nc=nt
  end if

  if(error_flag.eqv..false.)then
     call Median_1df( rx(1:nt), med )
  end if

end subroutine Median_2df


subroutine Median_2dd( x, med, error, nc )  ! 2 次元配列中央値計算ルーチン
  implicit none
  double precision, intent(in) :: x(:,:)  ! データ
  double precision, intent(inout) :: med  ! 計算する中央値
  double precision, intent(in), optional :: error  ! 欠損値が存在するデータセットの欠損値
  integer, intent(inout), optional :: nc  ! 欠損値が存在するデータセットの平均
                                          ! に用いたデータ数.
  integer :: i, j, nt
  integer :: nx  ! データの要素数 1
  integer :: ny  ! データの要素数 2
  logical :: error_flag  ! 未定義を返すかどうかの処理.
  double precision :: rx(size(x,1)*size(x,2))

  !-- 配列を 2 次元から 1 次元へ置き換え

  nt=0
  nx=size(x,1)
  ny=size(x,2)
  error_flag=.false.

  if(present(error))then
     do j=1,ny
        do i=1,nx
           if(x(i,j)/=error)then
              nt=1+nt
              rx(nt)=x(i,j)
           end if
        end do
     end do

     if(nt==0)then
        med=error
        error_flag=.true.
     end if

  else

     do j=1,ny
        do i=1,nx
           nt=1+nt
           rx(nt)=x(i,j)
        end do
     end do

  end if

  if(present(nc))then
     nc=nt
  end if

  if(error_flag.eqv..false.)then
     call Median_1dd( rx(1:nt), med )
  end if

end subroutine Median_2dd


subroutine Median_3df( x, med, error, nc )  ! 3 次元配列中央値計算ルーチン
  implicit none
  real, intent(in) :: x(:,:,:)  ! データ
  real, intent(inout) :: med  ! 計算する中央値
  real, intent(in), optional :: error  ! 欠損値が存在するデータセットの欠損値
  integer, intent(inout), optional :: nc  ! 欠損値が存在するデータセットの平均
                                          ! に用いたデータ数.
  integer :: i, j, k, nt
  integer :: nx  ! データの要素数 1
  integer :: ny  ! データの要素数 2
  integer :: nz  ! データの要素数 3
  logical :: error_flag  ! 未定義を返すかどうかの処理.
  real :: rx(size(x,1)*size(x,2)*size(x,3))

  !-- 配列を 2 次元から 1 次元へ置き換え

  nt=0
  nx=size(x,1)
  ny=size(x,2)
  nz=size(x,3)
  error_flag=.false.

  if(present(error))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(x(i,j,k)/=error)then
                 nt=1+nt
                 rx(nt)=x(i,j,k)
              end if
           end do
        end do
     end do

     if(nt==0)then
        med=error
        error_flag=.true.
     end if

  else

     do k=1,nz
        do j=1,ny
           do i=1,nx
              nt=1+nt
              rx(nt)=x(i,j,k)
           end do
        end do
     end do

  end if

  if(present(nc))then
     nc=nt
  end if

  if(error_flag.eqv..false.)then
     call Median_1df( rx(1:nt), med )
  end if

end subroutine Median_3df


subroutine Median_3dd( x, med, error, nc )  ! 3 次元配列中央値計算ルーチン
  implicit none
  double precision, intent(in) :: x(:,:,:)  ! データ
  double precision, intent(inout) :: med  ! 計算する中央値
  double precision, intent(in), optional :: error  ! 欠損値が存在するデータセットの欠損値
  integer, intent(inout), optional :: nc  ! 欠損値が存在するデータセットの平均
                                          ! に用いたデータ数.
  integer :: i, j, k, nt
  integer :: nx  ! データの要素数 1
  integer :: ny  ! データの要素数 2
  integer :: nz  ! データの要素数 3
  logical :: error_flag  ! 未定義を返すかどうかの処理.
  double precision :: rx(size(x,1)*size(x,2)*size(x,3))

  !-- 配列を 2 次元から 1 次元へ置き換え

  nt=0
  nx=size(x,1)
  ny=size(x,2)
  nz=size(x,3)
  error_flag=.false.

  if(present(error))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(x(i,j,k)/=error)then
                 nt=1+nt
                 rx(nt)=x(i,j,k)
              end if
           end do
        end do
     end do

     if(nt==0)then
        med=error
        error_flag=.true.
     end if

  else

     do k=1,nz
        do j=1,ny
           do i=1,nx
              nt=1+nt
              rx(nt)=x(i,j,k)
           end do
        end do
     end do

  end if

  if(present(nc))then
     nc=nt
  end if

  if(error_flag.eqv..false.)then
     call Median_1dd( rx(1:nt), med )
  end if

end subroutine Median_3dd


subroutine Quartile_1df( x, qua, error, nc )  ! 1 次元配列四分位計算ルーチン
  implicit none
  real, intent(in) :: x(:)  ! データ
  real, intent(inout) :: qua(3)  ! 計算する四分位値 (25, 50, 75)
  real, intent(in), optional :: error  ! 欠損値が存在するデータセットの欠損値
  integer, intent(inout), optional :: nc  ! 欠損値が存在するデータセットの平均
                                          ! に用いたデータ数.
  integer :: i, nt
  integer :: nx  ! データの要素数
  real :: rx(size(x)), sx(size(x))

  nt=0
  rx=0.0
  nx=size(x)

  if(present(error))then
     do i=1,nx
        if(x(i)/=error)then
           nt=1+nt
           rx(nt)=x(i)
        end if
     end do
     if(nt==0)then
        qua=error
        if(present(nc))then
           nc=nt
        end if
        write(*,*) "*** WARNING (Quartile_1d) ***: data size is zero.", nt
        return
     end if
  else
     nt=nx
     rx(1:nt)=x(1:nt)
  end if

  call Quick_Sort_f( rx(1:nt), sx(1:nt), 'i' )
  if(mod(nt,2)==0)then  ! データ数が偶数
     qua(2)=0.5*(sx(nt/2)+sx(nt/2+1))
     if(mod(nt/2,2)==0)then  ! データ数が偶数
        qua(1)=0.5*(sx(nt/4)+sx(nt/4+1))
        qua(3)=0.5*(sx(3*nt/4)+sx(3*nt/4+1))
     else
        qua(1)=sx(nt/4+1)
        qua(3)=sx(3*nt/4+1)
     end if
  else  ! データ数が奇数
     qua(2)=sx(nt/2+1)
     if(mod(nt/2,2)==0)then  ! データ数が偶数
        qua(1)=0.5*(sx(nt/4)+sx(nt/4+1))
        qua(3)=0.5*(sx(3*nt/4)+sx(3*nt/4+1))
     else
        qua(1)=sx(nt/4+1)
        qua(3)=sx(3*nt/4+1)
     end if
  end if

end subroutine Quartile_1df


subroutine Quartile_1dd( x, qua, error, nc )  ! 1 次元配列四分位計算ルーチン
  implicit none
  double precision, intent(in) :: x(:)  ! データ
  double precision, intent(inout) :: qua(3)  ! 計算する四分位値 (25, 50, 75)
  double precision, intent(in), optional :: error  ! 欠損値が存在するデータセットの欠損値
  integer, intent(inout), optional :: nc  ! 欠損値が存在するデータセットの平均
                                          ! に用いたデータ数.
  integer :: i, nt
  integer :: nx  ! データの要素数
  double precision :: rx(size(x)), sx(size(x))

  nt=0
  rx=0.0d0
  nx=size(x)

  if(present(error))then
     do i=1,nx
        if(x(i)/=error)then
           nt=1+nt
           rx(nt)=x(i)
        end if
     end do
     if(nt==0)then
        qua=error
        if(present(nc))then
           nc=nt
        end if
        write(*,*) "*** WARNING (Quartile_1d) ***: data size is zero.", nt
        return
     end if
  else
     nt=nx
     rx(1:nt)=x(1:nt)
  end if

  call Quick_Sort_d( rx(1:nt), sx(1:nt), 'i' )
  if(mod(nt,2)==0)then  ! データ数が偶数
     qua(2)=0.5d0*(sx(nt/2)+sx(nt/2+1))
     if(mod(nt/2,2)==0)then  ! データ数が偶数
        qua(1)=0.5d0*(sx(nt/4)+sx(nt/4+1))
        qua(3)=0.5d0*(sx(3*nt/4)+sx(3*nt/4+1))
     else
        qua(1)=sx(nt/4+1)
        qua(3)=sx(3*nt/4+1)
     end if
  else  ! データ数が奇数
     qua(2)=sx(nt/2+1)
     if(mod(nt/2,2)==0)then  ! データ数が偶数
        qua(1)=0.5d0*(sx(nt/4)+sx(nt/4+1))
        qua(3)=0.5d0*(sx(3*nt/4)+sx(3*nt/4+1))
     else
        qua(1)=sx(nt/4+1)
        qua(3)=sx(3*nt/4+1)
     end if
  end if

end subroutine Quartile_1dd


subroutine Quartile_2df( x, qua, error, nc )  ! 2 次元配列四分位値計算ルーチン
  implicit none
  real, intent(in) :: x(:,:)  ! データ
  real, intent(inout) :: qua(3)  ! 計算する四分位値
  real, intent(in), optional :: error  ! 欠損値が存在するデータセットの欠損値
  integer, intent(inout), optional :: nc  ! 欠損値が存在するデータセットの平均
                                          ! に用いたデータ数.
  integer :: i, j, nt
  integer :: nx  ! データの要素数 1
  integer :: ny  ! データの要素数 2
  logical :: error_flag  ! 未定義を返すかどうかの処理.
  real :: rx(size(x,1)*size(x,2))

  !-- 配列を 2 次元から 1 次元へ置き換え

  nt=0
  nx=size(x,1)
  ny=size(x,2)
  error_flag=.false.

  if(present(error))then
     do j=1,ny
        do i=1,nx
           if(x(i,j)/=error)then
              nt=1+nt
              rx(nt)=x(i,j)
           end if
        end do
     end do

     if(nt==0)then
        qua=error
        error_flag=.true.
     end if

  else

     do j=1,ny
        do i=1,nx
           nt=1+nt
           rx(nt)=x(i,j)
        end do
     end do

  end if

  if(present(nc))then
     nc=nt
  end if

  if(error_flag.eqv..false.)then
     call Quartile_1df( rx(1:nt), qua )
  end if

end subroutine Quartile_2df


subroutine Quartile_2dd( x, qua, error, nc )  ! 2 次元配列四分位値計算ルーチン
  implicit none
  double precision, intent(in) :: x(:,:)  ! データ
  double precision, intent(inout) :: qua(3)  ! 計算する四分位値
  double precision, intent(in), optional :: error  ! 欠損値が存在するデータセットの欠損値
  integer, intent(inout), optional :: nc  ! 欠損値が存在するデータセットの平均
                                          ! に用いたデータ数.
  integer :: i, j, nt
  integer :: nx  ! データの要素数 1
  integer :: ny  ! データの要素数 2
  logical :: error_flag  ! 未定義を返すかどうかの処理.
  double precision :: rx(size(x,1)*size(x,2))

  !-- 配列を 2 次元から 1 次元へ置き換え

  nt=0
  nx=size(x,1)
  ny=size(x,2)
  error_flag=.false.

  if(present(error))then
     do j=1,ny
        do i=1,nx
           if(x(i,j)/=error)then
              nt=1+nt
              rx(nt)=x(i,j)
           end if
        end do
     end do

     if(nt==0)then
        qua=error
        error_flag=.true.
     end if

  else

     do j=1,ny
        do i=1,nx
           nt=1+nt
           rx(nt)=x(i,j)
        end do
     end do

  end if

  if(present(nc))then
     nc=nt
  end if

  if(error_flag.eqv..false.)then
     call Quartile_1dd( rx(1:nt), qua )
  end if

end subroutine Quartile_2dd


subroutine Quartile_3df( x, qua, error, nc )  ! 3 次元配列四分位値計算ルーチン
  implicit none
  real, intent(in) :: x(:,:,:)  ! データ
  real, intent(inout) :: qua(3)  ! 計算する四分位値
  real, intent(in), optional :: error  ! 欠損値が存在するデータセットの欠損値
  integer, intent(inout), optional :: nc  ! 欠損値が存在するデータセットの平均
                                          ! に用いたデータ数.
  integer :: i, j, k, nt
  integer :: nx  ! データの要素数 1
  integer :: ny  ! データの要素数 2
  integer :: nz  ! データの要素数 3
  logical :: error_flag  ! 未定義を返すかどうかの処理.
  real :: rx(size(x,1)*size(x,2)*size(x,3))

  !-- 配列を 2 次元から 1 次元へ置き換え

  nt=0
  nx=size(x,1)
  ny=size(x,2)
  nz=size(x,3)
  error_flag=.false.

  if(present(error))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(x(i,j,k)/=error)then
                 nt=1+nt
                 rx(nt)=x(i,j,k)
              end if
           end do
        end do
     end do

     if(nt==0)then
        qua=error
        error_flag=.true.
     end if

  else

     do k=1,nz
        do j=1,ny
           do i=1,nx
              nt=1+nt
              rx(nt)=x(i,j,k)
           end do
        end do
     end do

  end if

  if(present(nc))then
     nc=nt
  end if

  if(error_flag.eqv..false.)then
     call Quartile_1df( rx(1:nt), qua )
  end if

end subroutine Quartile_3df


subroutine Quartile_3dd( x, qua, error, nc )  ! 3 次元配列四分位値計算ルーチン
  implicit none
  double precision, intent(in) :: x(:,:,:)  ! データ
  double precision, intent(inout) :: qua(3)  ! 計算する四分位値
  double precision, intent(in), optional :: error  ! 欠損値が存在するデータセットの欠損値
  integer, intent(inout), optional :: nc  ! 欠損値が存在するデータセットの平均
                                          ! に用いたデータ数.
  integer :: i, j, k, nt
  integer :: nx  ! データの要素数 1
  integer :: ny  ! データの要素数 2
  integer :: nz  ! データの要素数 3
  logical :: error_flag  ! 未定義を返すかどうかの処理.
  double precision :: rx(size(x,1)*size(x,2)*size(x,3))

  !-- 配列を 2 次元から 1 次元へ置き換え

  nt=0
  nx=size(x,1)
  ny=size(x,2)
  nz=size(x,3)
  error_flag=.false.

  if(present(error))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(x(i,j,k)/=error)then
                 nt=1+nt
                 rx(nt)=x(i,j,k)
              end if
           end do
        end do
     end do

     if(nt==0)then
        qua=error
        error_flag=.true.
     end if

  else

     do k=1,nz
        do j=1,ny
           do i=1,nx
              nt=1+nt
              rx(nt)=x(i,j,k)
           end do
        end do
     end do

  end if

  if(present(nc))then
     nc=nt
  end if

  if(error_flag.eqv..false.)then
     call Quartile_1dd( rx(1:nt), qua )
  end if

end subroutine Quartile_3dd


subroutine Quantile_1df( q, x, qua, error, nc, opt )  ! 1 次元配列q分位計算ルーチン
  implicit none
  real, intent(in) :: q     ! 分位数 (0<=q<=1)
  real, intent(in) :: x(:)  ! データ
  real, intent(inout) :: qua  ! 計算する分位値
  real, intent(in), optional :: error  ! 欠損値が存在するデータセットの欠損値
  integer, intent(inout), optional :: nc  ! 欠損値が存在するデータセットの平均
                                          ! に用いたデータ数.
  character(1), intent(in), optional :: opt
                            ! 分位点が整数でない場合の処理.
                            ! 'i' = 内挿補間する, 'l' = 下側データを返す.
                            ! 'u' = 上側データを返す. default = 'i'
  integer :: i, nt
  integer :: nx  ! データの要素数
  integer :: itpoint
  real :: tpoint
  real :: rx(size(x)), sx(size(x))
  character(1) :: copt

  nt=0
  rx=0.0
  nx=size(x)

  if(present(opt))then
     copt(1:1)=opt(1:1)
  else
     copt(1:1)='i'
  end if

  if(present(error))then
     do i=1,nx
        if(x(i)/=error)then
           nt=1+nt
           rx(nt)=x(i)
        end if
     end do
     if(nt==0)then
        qua=error
        if(present(nc))then
           nc=nt
        end if
        write(*,*) "*** WARNING (Quantile_1d) ***: data size is zero.", nt
        return
     end if
  else
     nt=nx
     rx(1:nt)=x(1:nt)
  end if

  call Quick_Sort_f( rx(1:nt), sx(1:nt), 'i' )

  !-- 分位点の計算
  tpoint=1.0-q+q*real(nt)
  itpoint=aint(tpoint)  ! tpoint を超えない最大の整数

  if(itpoint<1)then  ! データの下端以下

     qua=sx(1)

  else if(itpoint>nt)then  ! データの上端以上

     qua=sx(nt)

  else  ! データ配列の範囲内

     if(real(itpoint)==tpoint)then  ! 分位点が整数
        qua=sx(itpoint)
     else  ! 分位点が実数 (データ配列の丁度の位置にない)
        select case (copt(1:1))  ! データの返し方
        case ('i')  ! 実数位置に内挿
           qua=(real(itpoint+1)-tpoint)*sx(itpoint)  &
  &           +(tpoint-real(itpoint))*sx(itpoint+1)
        case ('l')  ! 下側データを返す
           qua=sx(itpoint)
        case ('u')  ! 上側データを返す
           qua=sx(itpoint+1)
        end select
     end if

  end if

  if(present(nc))then
     nc=nt
  end if

end subroutine Quantile_1df


subroutine Quantile_1dd( q, x, qua, error, nc, opt )  ! 1 次元配列q分位計算ルーチン
  implicit none
  double precision, intent(in) :: q     ! 分位数 (0<=q<=1)
  double precision, intent(in) :: x(:)  ! データ
  double precision, intent(inout) :: qua  ! 計算する分位値
  double precision, intent(in), optional :: error  ! 欠損値が存在するデータセットの欠損値
  integer, intent(inout), optional :: nc  ! 欠損値が存在するデータセットの平均
                                          ! に用いたデータ数.
  character(1), intent(in), optional :: opt
                            ! 分位点が整数でない場合の処理.
                            ! 'i' = 内挿補間する, 'l' = 下側データを返す.
                            ! 'u' = 上側データを返す. default = 'i'
  integer :: i, nt
  integer :: nx  ! データの要素数
  integer :: itpoint
  double precision :: tpoint
  double precision :: rx(size(x)), sx(size(x))
  character(1) :: copt

  nt=0
  rx=0.0d0
  nx=size(x)

  if(present(opt))then
     copt(1:1)=opt(1:1)
  else
     copt(1:1)='i'
  end if

  if(present(error))then
     do i=1,nx
        if(x(i)/=error)then
           nt=1+nt
           rx(nt)=x(i)
        end if
     end do
     if(nt==0)then
        qua=error
        if(present(nc))then
           nc=nt
        end if
        write(*,*) "*** WARNING (Quantile_1d) ***: data size is zero.", nt
        return
     end if
  else
     nt=nx
     rx(1:nt)=x(1:nt)
  end if

  call Quick_Sort_d( rx(1:nt), sx(1:nt), 'i' )

  !-- 分位点の計算
  tpoint=1.0d0-q+q*dble(nt)
  itpoint=aint(tpoint)  ! tpoint を超えない最大の整数

  if(itpoint<1)then  ! データの下端以下

     qua=sx(1)

  else if(itpoint>nt)then  ! データの上端以上

     qua=sx(nt)

  else  ! データ配列の範囲内

     if(dble(itpoint)==tpoint)then  ! 分位点が整数
        qua=sx(itpoint)
     else  ! 分位点が実数 (データ配列の丁度の位置にない)
        select case (copt(1:1))  ! データの返し方
        case ('i')  ! 実数位置に内挿
           qua=(dble(itpoint+1)-tpoint)*sx(itpoint)  &
  &           +(tpoint-dble(itpoint))*sx(itpoint+1)
        case ('l')  ! 下側データを返す
           qua=sx(itpoint)
        case ('u')  ! 上側データを返す
           qua=sx(itpoint+1)
        end select
     end if

  end if

  if(present(nc))then
     nc=nt
  end if

end subroutine Quantile_1dd


subroutine Quantile_2df( q, x, qua, error, nc, opt )  ! 2 次元配列q分位値計算ルーチン
  implicit none
  real, intent(in) :: q       ! 分位数 (0<=q<=1)
  real, intent(in) :: x(:,:)  ! データ
  real, intent(inout) :: qua  ! 計算するq分位値
  real, intent(in), optional :: error  ! 欠損値が存在するデータセットの欠損値
  integer, intent(inout), optional :: nc  ! 欠損値が存在するデータセットの平均
                                          ! に用いたデータ数.
  character(1), intent(in), optional :: opt
                            ! 分位点が整数でない場合の処理.
                            ! 'i' = 内挿補間する, 'l' = 下側データを返す.
                            ! 'u' = 上側データを返す. default = 'i'
  integer :: i, j, nt
  integer :: nx  ! データの要素数 1
  integer :: ny  ! データの要素数 2
  logical :: error_flag  ! 未定義を返すかどうかの処理.
  real :: rx(size(x,1)*size(x,2))
  character(1) :: copt

  !-- 配列を 2 次元から 1 次元へ置き換え

  nt=0
  nx=size(x,1)
  ny=size(x,2)
  error_flag=.false.

  if(present(opt))then
     copt(1:1)=opt(1:1)
  else
     copt(1:1)='i'
  end if

  if(present(error))then
     do j=1,ny
        do i=1,nx
           if(x(i,j)/=error)then
              nt=1+nt
              rx(nt)=x(i,j)
           end if
        end do
     end do

     if(nt==0)then
        qua=error
        error_flag=.true.
     end if

  else

     do j=1,ny
        do i=1,nx
           nt=1+nt
           rx(nt)=x(i,j)
        end do
     end do

  end if

  if(present(nc))then
     nc=nt
  end if

  if(error_flag.eqv..false.)then
     call Quantile_1df( q, rx(1:nt), qua, opt=copt(1:1) )
  end if

end subroutine Quantile_2df


subroutine Quantile_2dd( q, x, qua, error, nc, opt )  ! 2 次元配列q分位値計算ルーチン
  implicit none
  double precision, intent(in) :: q       ! 分位数 (0<=q<=1)
  double precision, intent(in) :: x(:,:)  ! データ
  double precision, intent(inout) :: qua  ! 計算するq分位値
  double precision, intent(in), optional :: error  ! 欠損値が存在するデータセットの欠損値
  integer, intent(inout), optional :: nc  ! 欠損値が存在するデータセットの平均
                                          ! に用いたデータ数.
  character(1), intent(in), optional :: opt
                            ! 分位点が整数でない場合の処理.
                            ! 'i' = 内挿補間する, 'l' = 下側データを返す.
                            ! 'u' = 上側データを返す. default = 'i'
  integer :: i, j, nt
  integer :: nx  ! データの要素数 1
  integer :: ny  ! データの要素数 2
  logical :: error_flag  ! 未定義を返すかどうかの処理.
  double precision :: rx(size(x,1)*size(x,2))
  character(1) :: copt

  !-- 配列を 2 次元から 1 次元へ置き換え

  nt=0
  nx=size(x,1)
  ny=size(x,2)
  error_flag=.false.

  if(present(opt))then
     copt(1:1)=opt(1:1)
  else
     copt(1:1)='i'
  end if

  if(present(error))then
     do j=1,ny
        do i=1,nx
           if(x(i,j)/=error)then
              nt=1+nt
              rx(nt)=x(i,j)
           end if
        end do
     end do

     if(nt==0)then
        qua=error
        error_flag=.true.
     end if

  else

     do j=1,ny
        do i=1,nx
           nt=1+nt
           rx(nt)=x(i,j)
        end do
     end do

  end if

  if(present(nc))then
     nc=nt
  end if

  if(error_flag.eqv..false.)then
     call Quantile_1dd( q, rx(1:nt), qua, opt=copt(1:1) )
  end if

end subroutine Quantile_2dd


subroutine Quantile_3df( q, x, qua, error, nc, opt )  ! 3 次元配列q分位値計算ルーチン
  implicit none
  real, intent(in) :: q         ! 分位数 (0<=q<=1)
  real, intent(in) :: x(:,:,:)  ! データ
  real, intent(inout) :: qua  ! 計算するq分位値
  real, intent(in), optional :: error  ! 欠損値が存在するデータセットの欠損値
  integer, intent(inout), optional :: nc  ! 欠損値が存在するデータセットの平均
                                          ! に用いたデータ数.
  character(1), intent(in), optional :: opt
                            ! 分位点が整数でない場合の処理.
                            ! 'i' = 内挿補間する, 'l' = 下側データを返す.
                            ! 'u' = 上側データを返す. default = 'i'
  integer :: i, j, k, nt
  integer :: nx  ! データの要素数 1
  integer :: ny  ! データの要素数 2
  integer :: nz  ! データの要素数 3
  logical :: error_flag  ! 未定義を返すかどうかの処理.
  real :: rx(size(x,1)*size(x,2)*size(x,3))
  character(1) :: copt

  !-- 配列を 2 次元から 1 次元へ置き換え

  nt=0
  nx=size(x,1)
  ny=size(x,2)
  nz=size(x,3)
  error_flag=.false.

  if(present(opt))then
     copt(1:1)=opt(1:1)
  else
     copt(1:1)='i'
  end if

  if(present(error))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(x(i,j,k)/=error)then
                 nt=1+nt
                 rx(nt)=x(i,j,k)
              end if
           end do
        end do
     end do

     if(nt==0)then
        qua=error
        error_flag=.true.
     end if

  else

     do k=1,nz
        do j=1,ny
           do i=1,nx
              nt=1+nt
              rx(nt)=x(i,j,k)
           end do
        end do
     end do

  end if

  if(present(nc))then
     nc=nt
  end if

  if(error_flag.eqv..false.)then
     call Quantile_1df( q, rx(1:nt), qua, opt=copt(1:1) )
  end if

end subroutine Quantile_3df


subroutine Quantile_3dd( q, x, qua, error, nc, opt )  ! 3 次元配列q分位値計算ルーチン
  implicit none
  double precision, intent(in) :: q         ! 分位数 (0<=q<=1)
  double precision, intent(in) :: x(:,:,:)  ! データ
  double precision, intent(inout) :: qua  ! 計算するq分位値
  double precision, intent(in), optional :: error  ! 欠損値が存在するデータセットの欠損値
  integer, intent(inout), optional :: nc  ! 欠損値が存在するデータセットの平均
                                          ! に用いたデータ数.
  character(1), intent(in), optional :: opt
                            ! 分位点が整数でない場合の処理.
                            ! 'i' = 内挿補間する, 'l' = 下側データを返す.
                            ! 'u' = 上側データを返す. default = 'i'
  integer :: i, j, k, nt
  integer :: nx  ! データの要素数 1
  integer :: ny  ! データの要素数 2
  integer :: nz  ! データの要素数 3
  logical :: error_flag  ! 未定義を返すかどうかの処理.
  double precision :: rx(size(x,1)*size(x,2)*size(x,3))
  character(1) :: copt

  !-- 配列を 2 次元から 1 次元へ置き換え

  nt=0
  nx=size(x,1)
  ny=size(x,2)
  nz=size(x,3)
  error_flag=.false.

  if(present(opt))then
     copt(1:1)=opt(1:1)
  else
     copt(1:1)='i'
  end if

  if(present(error))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(x(i,j,k)/=error)then
                 nt=1+nt
                 rx(nt)=x(i,j,k)
              end if
           end do
        end do
     end do

     if(nt==0)then
        qua=error
        error_flag=.true.
     end if

  else

     do k=1,nz
        do j=1,ny
           do i=1,nx
              nt=1+nt
              rx(nt)=x(i,j,k)
           end do
        end do
     end do

  end if

  if(present(nc))then
     nc=nt
  end if

  if(error_flag.eqv..false.)then
     call Quantile_1dd( q, rx(1:nt), qua, opt=copt(1:1) )
  end if

end subroutine Quantile_3dd


subroutine Freqbin_count_1df( qbot, qtop, x, opt, qua, error, nc )
  ! 1 次元配列指定ビン頻度カウント
  ! CFAD を作成する際に利用可能. 
  implicit none
  real, intent(in) :: qbot(:)           ! データ x を分ける際のビン閾値下端.
  real, intent(in) :: qtop(size(qbot))  ! データ x を分ける際のビン閾値下端.
  real, intent(in) :: x(:)  ! データ
  character(1), intent(in) :: opt
                            ! ビン閾値上にデータがある場合の処理.
                            ! 'l' = qbot と同じ値なら, そのビンに含める.
                            ! 'u' = qtop と同じ値なら, そのビンに含める.
  integer, intent(inout) :: qua(size(qbot))  ! q の範囲内でカウントされたデータ個数
  real, intent(in), optional :: error  ! 欠損値が存在するデータセットの欠損値
  integer, intent(inout), optional :: nc  ! 欠損値が存在するデータセットの平均
                                          ! に用いたデータ数.
  integer :: i, j, nt, nq, qcounter
  integer :: nx  ! データの要素数
  real :: rx(size(x)), sx(size(x))
  character(1) :: copt

  nt=0
  rx=0.0
  nx=size(x)
  nq=size(qbot)
  qua=0

  copt(1:1)=opt(1:1)

  if(present(error))then
     do i=1,nx
        if(x(i)/=error)then
           nt=1+nt
           rx(nt)=x(i)
        end if
     end do
     if(nt==0)then
        if(present(nc))then
           nc=nt
        end if
        write(*,*) "*** WARNING (Quantile_1d) ***: data size is zero.", nt
        return
     end if
  else
     nt=nx
     rx(1:nt)=x(1:nt)
  end if

  call Quick_Sort_f( rx(1:nt), sx(1:nt), 'i' )

  !-- ビン毎にデータをカウント.

  qcounter=1

  if(opt(1:1)=='l')then
     do j=1,nq
        do i=qcounter,nt
           if(qbot(j)<=sx(i).and.qtop(j)>sx(i))then
              qua(j)=qua(j)+1
           else if(qtop(j)<=sx(i))then
              qcounter=i
              exit
           end if
        end do
     end do
  else if(opt(1:1)=='u')then
     do j=1,nq
        do i=qcounter,nt
           if(qbot(j)<sx(i).and.qtop(j)>=sx(i))then
              qua(j)=qua(j)+1
           else if(qtop(j)<sx(i))then
              qcounter=i
              exit
           end if
        end do
     end do
  end if

  if(present(nc))then
     nc=nt
  end if

end subroutine Freqbin_count_1df


subroutine Freqbin_count_1dd( qbot, qtop, x, opt, qua, error, nc )
  ! 1 次元配列指定ビン頻度カウント
  ! CFAD を作成する際に利用可能. 
  implicit none
  double precision, intent(in) :: qbot(:)           ! データ x を分ける際のビン閾値下端.
  double precision, intent(in) :: qtop(size(qbot))  ! データ x を分ける際のビン閾値下端.
  double precision, intent(in) :: x(:)  ! データ
  character(1), intent(in) :: opt
                            ! ビン閾値上にデータがある場合の処理.
                            ! 'l' = qbot と同じ値なら, そのビンに含める.
                            ! 'u' = qtop と同じ値なら, そのビンに含める.
  integer, intent(inout) :: qua(size(qbot))  ! q の範囲内でカウントされたデータ個数
  double precision, intent(in), optional :: error  ! 欠損値が存在するデータセットの欠損値
  integer, intent(inout), optional :: nc  ! 欠損値が存在するデータセットの平均
                                          ! に用いたデータ数.
  integer :: i, j, nt, nq, qcounter
  integer :: nx  ! データの要素数
  double precision :: rx(size(x)), sx(size(x))
  character(1) :: copt

  nt=0
  rx=0.0d0
  nx=size(x)
  nq=size(qbot)
  qua=0

  copt(1:1)=opt(1:1)

  if(present(error))then
     do i=1,nx
        if(x(i)/=error)then
           nt=1+nt
           rx(nt)=x(i)
        end if
     end do
     if(nt==0)then
        if(present(nc))then
           nc=nt
        end if
        write(*,*) "*** WARNING (Quantile_1d) ***: data size is zero.", nt
        return
     end if
  else
     nt=nx
     rx(1:nt)=x(1:nt)
  end if

  call Quick_Sort_d( rx(1:nt), sx(1:nt), 'i' )

  !-- ビン毎にデータをカウント.

  qcounter=1

  if(opt(1:1)=='l')then
     do j=1,nq
        do i=qcounter,nt
           if(qbot(j)<=sx(i).and.qtop(j)>sx(i))then
              qua(j)=qua(j)+1
           else if(qtop(j)<=sx(i))then
              qcounter=i
              exit
           end if
        end do
     end do
  else if(opt(1:1)=='u')then
     do j=1,nq
        do i=qcounter,nt
           if(qbot(j)<sx(i).and.qtop(j)>=sx(i))then
              qua(j)=qua(j)+1
           else if(qtop(j)<sx(i))then
              qcounter=i
              exit
           end if
        end do
     end do
  end if

  if(present(nc))then
     nc=nt
  end if

end subroutine Freqbin_count_1dd


subroutine Freqbin_count_2df( qbot, qtop, x, opt, qua, error, nc )
  ! 2 次元配列指定ビン頻度カウント
  ! CFAD を作成する際に利用可能. 
  implicit none
  real, intent(in) :: qbot(:)           ! データ x を分ける際のビン閾値下端.
  real, intent(in) :: qtop(size(qbot))  ! データ x を分ける際のビン閾値下端.
  real, intent(in) :: x(:,:)  ! データ
  character(1), intent(in) :: opt
                            ! ビン閾値上にデータがある場合の処理.
                            ! 'l' = qbot と同じ値なら, そのビンに含める.
                            ! 'u' = qtop と同じ値なら, そのビンに含める.
  integer, intent(inout) :: qua(size(qbot))  ! q の範囲内でカウントされたデータ個数
  real, intent(in), optional :: error  ! 欠損値が存在するデータセットの欠損値
  integer, intent(inout), optional :: nc  ! 欠損値が存在するデータセットの平均
                                          ! に用いたデータ数.
  integer :: i, j, nt, nq, qcounter
  integer :: nx  ! データの要素数 1
  integer :: ny  ! データの要素数 2
  logical :: error_flag  ! 未定義を返すかどうかの処理.
  real :: rx(size(x,1)*size(x,2))
  character(1) :: copt

  nt=0
  rx=0.0
  nx=size(x,1)
  ny=size(x,2)
  nq=size(qbot)
  qua=0
  error_flag=.false.

  copt(1:1)=opt(1:1)

  !-- 配列を 2 次元から 1 次元へ置き換え

  if(present(error))then
     do j=1,ny
        do i=1,nx
           if(x(i,j)/=error)then
              nt=1+nt
              rx(nt)=x(i,j)
           end if
        end do
     end do

     if(nt==0)then
        qua=error
        error_flag=.true.
     end if

  else

     do j=1,ny
        do i=1,nx
           nt=1+nt
           rx(nt)=x(i,j)
        end do
     end do

  end if

  if(present(nc))then
     nc=nt
  end if

  if(error_flag.eqv..false.)then
     call Freqbin_count_1df( qbot, qtop, rx(1:nt), copt(1:1), qua )
  end if

end subroutine Freqbin_count_2df


subroutine Freqbin_count_2dd( qbot, qtop, x, opt, qua, error, nc )
  ! 2 次元配列指定ビン頻度カウント
  ! CFAD を作成する際に利用可能. 
  implicit none
  double precision, intent(in) :: qbot(:)           ! データ x を分ける際のビン閾値下端.
  double precision, intent(in) :: qtop(size(qbot))  ! データ x を分ける際のビン閾値下端.
  double precision, intent(in) :: x(:,:)  ! データ
  character(1), intent(in) :: opt
                            ! ビン閾値上にデータがある場合の処理.
                            ! 'l' = qbot と同じ値なら, そのビンに含める.
                            ! 'u' = qtop と同じ値なら, そのビンに含める.
  integer, intent(inout) :: qua(size(qbot))  ! q の範囲内でカウントされたデータ個数
  double precision, intent(in), optional :: error  ! 欠損値が存在するデータセットの欠損値
  integer, intent(inout), optional :: nc  ! 欠損値が存在するデータセットの平均
                                          ! に用いたデータ数.
  integer :: i, j, nt, nq, qcounter
  integer :: nx  ! データの要素数 1
  integer :: ny  ! データの要素数 2
  logical :: error_flag  ! 未定義を返すかどうかの処理.
  double precision :: rx(size(x,1)*size(x,2))
  character(1) :: copt

  nt=0
  rx=0.0d0
  nx=size(x,1)
  ny=size(x,2)
  nq=size(qbot)
  qua=0
  error_flag=.false.

  copt(1:1)=opt(1:1)

  !-- 配列を 2 次元から 1 次元へ置き換え

  if(present(error))then
     do j=1,ny
        do i=1,nx
           if(x(i,j)/=error)then
              nt=1+nt
              rx(nt)=x(i,j)
           end if
        end do
     end do

     if(nt==0)then
        qua=error
        error_flag=.true.
     end if

  else

     do j=1,ny
        do i=1,nx
           nt=1+nt
           rx(nt)=x(i,j)
        end do
     end do

  end if

  if(present(nc))then
     nc=nt
  end if

  if(error_flag.eqv..false.)then
     call Freqbin_count_1dd( qbot, qtop, rx(1:nt), copt(1:1), qua )
  end if

end subroutine Freqbin_count_2dd


subroutine Freqbin_count_3df( qbot, qtop, x, opt, qua, error, nc )
  ! 3 次元配列指定ビン頻度カウント
  ! CFAD を作成する際に利用可能. 
  implicit none
  real, intent(in) :: qbot(:)           ! データ x を分ける際のビン閾値下端.
  real, intent(in) :: qtop(size(qbot))  ! データ x を分ける際のビン閾値下端.
  real, intent(in) :: x(:,:,:)  ! データ
  character(1), intent(in) :: opt
                            ! ビン閾値上にデータがある場合の処理.
                            ! 'l' = qbot と同じ値なら, そのビンに含める.
                            ! 'u' = qtop と同じ値なら, そのビンに含める.
  integer, intent(inout) :: qua(size(qbot))  ! q の範囲内でカウントされたデータ個数
  real, intent(in), optional :: error  ! 欠損値が存在するデータセットの欠損値
  integer, intent(inout), optional :: nc  ! 欠損値が存在するデータセットの平均
                                          ! に用いたデータ数.
  integer :: i, j, k, nt, nq, qcounter
  integer :: nx  ! データの要素数 1
  integer :: ny  ! データの要素数 2
  integer :: nz  ! データの要素数 3
  logical :: error_flag  ! 未定義を返すかどうかの処理.
  real :: rx(size(x,1)*size(x,2)*size(x,3))
  character(1) :: copt

  nt=0
  rx=0.0
  nx=size(x,1)
  ny=size(x,2)
  nz=size(x,3)
  nq=size(qbot)
  qua=0
  error_flag=.false.

  copt(1:1)=opt(1:1)

  !-- 配列を 2 次元から 1 次元へ置き換え

  if(present(error))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(x(i,j,k)/=error)then
                 nt=1+nt
                 rx(nt)=x(i,j,k)
              end if
           end do
        end do
     end do

     if(nt==0)then
        qua=error
        error_flag=.true.
     end if

  else

     do k=1,nz
        do j=1,ny
           do i=1,nx
              nt=1+nt
              rx(nt)=x(i,j,k)
           end do
        end do
     end do

  end if

  if(present(nc))then
     nc=nt
  end if

  if(error_flag.eqv..false.)then
     call Freqbin_count_1df( qbot, qtop, rx(1:nt), copt(1:1), qua )
  end if

end subroutine Freqbin_count_3df


subroutine Freqbin_count_3dd( qbot, qtop, x, opt, qua, error, nc )
  ! 3 次元配列指定ビン頻度カウント
  ! CFAD を作成する際に利用可能. 
  implicit none
  double precision, intent(in) :: qbot(:)           ! データ x を分ける際のビン閾値下端.
  double precision, intent(in) :: qtop(size(qbot))  ! データ x を分ける際のビン閾値下端.
  double precision, intent(in) :: x(:,:,:)  ! データ
  character(1), intent(in) :: opt
                            ! ビン閾値上にデータがある場合の処理.
                            ! 'l' = qbot と同じ値なら, そのビンに含める.
                            ! 'u' = qtop と同じ値なら, そのビンに含める.
  integer, intent(inout) :: qua(size(qbot))  ! q の範囲内でカウントされたデータ個数
  double precision, intent(in), optional :: error  ! 欠損値が存在するデータセットの欠損値
  integer, intent(inout), optional :: nc  ! 欠損値が存在するデータセットの平均
                                          ! に用いたデータ数.
  integer :: i, j, k, nt, nq, qcounter
  integer :: nx  ! データの要素数 1
  integer :: ny  ! データの要素数 2
  integer :: nz  ! データの要素数 3
  logical :: error_flag  ! 未定義を返すかどうかの処理.
  double precision :: rx(size(x,1)*size(x,2)*size(x,3))
  character(1) :: copt

  nt=0
  rx=0.0d0
  nx=size(x,1)
  ny=size(x,2)
  nz=size(x,3)
  nq=size(qbot)
  qua=0
  error_flag=.false.

  copt(1:1)=opt(1:1)

  !-- 配列を 2 次元から 1 次元へ置き換え

  if(present(error))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(x(i,j,k)/=error)then
                 nt=1+nt
                 rx(nt)=x(i,j,k)
              end if
           end do
        end do
     end do

     if(nt==0)then
        qua=error
        error_flag=.true.
     end if

  else

     do k=1,nz
        do j=1,ny
           do i=1,nx
              nt=1+nt
              rx(nt)=x(i,j,k)
           end do
        end do
     end do

  end if

  if(present(nc))then
     nc=nt
  end if

  if(error_flag.eqv..false.)then
     call Freqbin_count_1dd( qbot, qtop, rx(1:nt), copt(1:1), qua )
  end if

end subroutine Freqbin_count_3dd


subroutine Anomaly_1df( x, anor, error, nc )  ! 1 次元データ配列の偏差を返す
  implicit none
  real, intent(in) :: x(:)  ! データ
  real, intent(inout) :: anor(size(x))  ! 各 x(i) に対応する偏差 anor(i)
  real, intent(in), optional :: error  ! 欠損値が存在するデータセットの場合の欠損値
  integer, intent(inout), optional :: nc  ! 欠損値が存在するデータセットの平均
                                          ! に用いたデータ数.
  integer :: i
  integer :: nx  ! データの要素数
  real :: ave

  nx=size(x)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_1d( nx, anor ),  &
  &                                     "Anomaly_1d" )
  end if

  if(present(error))then
     call Mean_1df( x, ave, error, nc=nc )
     do i=1,nx
        if(x(i)==error)then
           anor(i)=error
        else
           anor(i)=x(i)-ave
        end if
     end do
  else
     call Mean_1df( x, ave )
     do i=1,nx
        anor(i)=x(i)-ave
     end do
  end if

end subroutine Anomaly_1df


subroutine Anomaly_1dd( x, anor, error, nc )  ! 1 次元データ配列の偏差を返す
  implicit none
  double precision, intent(in) :: x(:)  ! データ
  double precision, intent(inout) :: anor(size(x))  ! 各 x(i) に対応する偏差 anor(i)
  double precision, intent(in), optional :: error  ! 欠損値が存在するデータセットの場合の欠損値
  integer, intent(inout), optional :: nc  ! 欠損値が存在するデータセットの平均
                                          ! に用いたデータ数.
  integer :: i
  integer :: nx  ! データの要素数
  double precision :: ave

  nx=size(x)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_1d( nx, anor ),  &
  &                                     "Anomaly_1d" )
  end if

  if(present(error))then
     call Mean_1dd( x, ave, error, nc=nc )
     do i=1,nx
        if(x(i)==error)then
           anor(i)=error
        else
           anor(i)=x(i)-ave
        end if
     end do
  else
     call Mean_1dd( x, ave )
     do i=1,nx
        anor(i)=x(i)-ave
     end do
  end if

end subroutine Anomaly_1dd


subroutine Anomaly_2df( x, anor, error, nc )  ! 2 次元データ配列の偏差を返す
  implicit none
  real, intent(in) :: x(:,:)  ! データ
  real, intent(inout) :: anor(size(x,1),size(x,2))  ! 各 x(i,j) に対応する偏差 anor(i,j)
  real, intent(in), optional :: error  ! 欠損値が存在するデータセットの場合の欠損値
  integer, intent(inout), optional :: nc  ! 欠損値が存在するデータセットの平均
                                          ! に用いたデータ数.
  integer :: i, j
  integer :: nx  ! データの要素数 1
  integer :: ny  ! データの要素数 2
  real :: ave

  nx=size(x,1)
  ny=size(x,2)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, anor ),  &
  &                                     "Anomaly_2d" )
  end if

  if(present(error))then
     call Mean_2df( x, ave, error, nc=nc )
     do j=1,ny
        do i=1,nx
           if(x(i,j)==error)then
              anor(i,j)=error
           else
              anor(i,j)=x(i,j)-ave
           end if
        end do
     end do
  else
     call Mean_2df( x, ave )
     do j=1,ny
        do i=1,nx
           anor(i,j)=x(i,j)-ave
        end do
     end do
  end if

end subroutine Anomaly_2df


subroutine Anomaly_2dd( x, anor, error, nc )  ! 2 次元データ配列の偏差を返す
  implicit none
  double precision, intent(in) :: x(:,:)  ! データ
  double precision, intent(inout) :: anor(size(x,1),size(x,2))  ! 各 x(i,j) に対応する偏差 anor(i,j)
  double precision, intent(in), optional :: error  ! 欠損値が存在するデータセットの場合の欠損値
  integer, intent(inout), optional :: nc  ! 欠損値が存在するデータセットの平均
                                          ! に用いたデータ数.
  integer :: i, j
  integer :: nx  ! データの要素数 1
  integer :: ny  ! データの要素数 2
  double precision :: ave

  nx=size(x,1)
  ny=size(x,2)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, anor ),  &
  &                                     "Anomaly_2d" )
  end if

  if(present(error))then
     call Mean_2dd( x, ave, error, nc=nc )
     do j=1,ny
        do i=1,nx
           if(x(i,j)==error)then
              anor(i,j)=error
           else
              anor(i,j)=x(i,j)-ave
           end if
        end do
     end do
  else
     call Mean_2dd( x, ave )
     do j=1,ny
        do i=1,nx
           anor(i,j)=x(i,j)-ave
        end do
     end do
  end if

end subroutine Anomaly_2dd


subroutine Anomaly_3df( x, anor, error, nc )  ! 3 次元データ配列の偏差を返す
  implicit none
  real, intent(in) :: x(:,:,:)  ! データ
  real, intent(inout) :: anor(size(x,1),size(x,2),size(x,3))  ! 各 x(i,j,k) に対応する偏差 anor(i,j,k)
  real, intent(in), optional :: error  ! 欠損値が存在するデータセットの場合の欠損値
  integer, intent(inout), optional :: nc  ! 欠損値が存在するデータセットの平均
                                          ! に用いたデータ数.
  integer :: i, j, k
  integer :: nx  ! データの要素数 1
  integer :: ny  ! データの要素数 2
  integer :: nz  ! データの要素数 3
  real :: ave

  nx=size(x,1)
  ny=size(x,2)
  nz=size(x,3)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_3d(  &
  &                                          nx, ny, nz, anor ),  &
  &                                     "Anomaly_3d" )
  end if

  if(present(error))then
     call Mean_3df( x, ave, error, nc=nc )
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(x(i,j,k)==error)then
                 anor(i,j,k)=error
              else
                 anor(i,j,k)=x(i,j,k)-ave
              end if
           end do
        end do
     end do
  else
     call Mean_3df( x, ave )
     do k=1,nz
        do j=1,ny
           do i=1,nx
              anor(i,j,k)=x(i,j,k)-ave
           end do
        end do
     end do
  end if

end subroutine Anomaly_3df


subroutine Anomaly_3dd( x, anor, error, nc )  ! 3 次元データ配列の偏差を返す
  implicit none
  double precision, intent(in) :: x(:,:,:)  ! データ
  double precision, intent(inout) :: anor(size(x,1),size(x,2),size(x,3))  ! 各 x(i,j,k) に対応する偏差 anor(i,j,k)
  double precision, intent(in), optional :: error  ! 欠損値が存在するデータセットの場合の欠損値
  integer, intent(inout), optional :: nc  ! 欠損値が存在するデータセットの平均
                                          ! に用いたデータ数.
  integer :: i, j, k
  integer :: nx  ! データの要素数 1
  integer :: ny  ! データの要素数 2
  integer :: nz  ! データの要素数 3
  double precision :: ave

  nx=size(x,1)
  ny=size(x,2)
  nz=size(x,3)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_3d(  &
  &                                          nx, ny, nz, anor ),  &
  &                                     "Anomaly_3d" )
  end if

  if(present(error))then
     call Mean_3dd( x, ave, error, nc=nc )
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(x(i,j,k)==error)then
                 anor(i,j,k)=error
              else
                 anor(i,j,k)=x(i,j,k)-ave
              end if
           end do
        end do
     end do
  else
     call Mean_3dd( x, ave )
     do k=1,nz
        do j=1,ny
           do i=1,nx
              anor(i,j,k)=x(i,j,k)-ave
           end do
        end do
     end do
  end if

end subroutine Anomaly_3dd

!---------------------------------------
!---------------------------------------

subroutine stand_devi_1df( x, anor, error )  ! 1 次元データの標準偏差を計算
  ! 標準偏差$\sigma $の定義は,
  ! $$\sigma =\sum^{nx}_{i=1}{epsilon ^2} $$
  ! ただし, $\epsilon $は平均値からのずれ$x-\bar{x}$である.
  implicit none
  real, intent(in) :: x(:)  ! データ
  real, intent(inout) :: anor  ! 標準偏差
  real, intent(in), optional :: error  ! 欠損値
  integer :: i
  integer :: nx  ! データ数
  integer :: nt
  real :: an(size(x))

  nx=size(x)
  anor=0.0

  if(present(error))then
     call Anomaly_1df( x, an, error, nc=nt )
     do i=1,nx
        if(x(i)/=error)then
           anor=anor+an(i)**2
        end if
     end do
     if(anor/=error.and.nt/=0)then
        anor=sqrt(anor/real(nt))
     end if
  else
     call Anomaly_1df( x, an )
     do i=1,nx
        anor=anor+an(i)**2
     end do
     anor=sqrt(anor/real(nx))
  end if

end subroutine stand_devi_1df

!---------------------------------------
!---------------------------------------

subroutine stand_devi_1dd( x, anor, error )  ! 1 次元データの標準偏差を計算
  ! 標準偏差$\sigma $の定義は,
  ! $$\sigma =\sum^{nx}_{i=1}{epsilon ^2} $$
  ! ただし, $\epsilon $は平均値からのずれ$x-\bar{x}$である.
  implicit none
  double precision, intent(in) :: x(:)  ! データ
  double precision, intent(inout) :: anor  ! 標準偏差
  double precision, intent(in), optional :: error  ! 欠損値
  integer :: i
  integer :: nx  ! データ数
  integer :: nt
  double precision :: an(size(x))

  nx=size(x)
  anor=0.0d0

  if(present(error))then
     call Anomaly_1dd( x, an, error, nc=nt )
     do i=1,nx
        if(x(i)/=error)then
           anor=anor+an(i)**2
        end if
     end do
     if(anor/=error.and.nt/=0)then
        anor=dsqrt(anor/dble(nt))
     end if
  else
     call Anomaly_1dd( x, an )
     do i=1,nx
        anor=anor+an(i)**2
     end do
     anor=dsqrt(anor/dble(nx))
  end if

end subroutine stand_devi_1dd

!---------------------------------------
!---------------------------------------

subroutine stand_devi_2df( x, anor, error )  ! 2 次元データの標準偏差を計算
  implicit none
  real, intent(in) :: x(:,:)  ! データ
  real, intent(inout) :: anor  ! 標準偏差
  real, intent(in), optional :: error  ! 欠損値
  integer :: i, j, counter
  integer :: nx  ! データ数 1
  integer :: ny  ! データ数 2
  real :: val(size(x,1)*size(x,2))

  nx=size(x,1)
  ny=size(x,2)

  counter=0
  do j=1,ny
     do i=1,nx
        counter=counter+1
        val(counter)=x(i,j)
     end do
  end do

  anor=0.0

  if(present(error))then
     call stand_devi_1df( val, anor, error )
  else
     call stand_devi_1df( val, anor )
  end if

end subroutine stand_devi_2df

!---------------------------------------
!---------------------------------------

subroutine stand_devi_2dd( x, anor, error )  ! 2 次元データの標準偏差を計算
  implicit none
  double precision, intent(in) :: x(:,:)  ! データ
  double precision, intent(inout) :: anor  ! 標準偏差
  double precision, intent(in), optional :: error  ! 欠損値
  integer :: i, j, counter
  integer :: nx  ! データ数 1
  integer :: ny  ! データ数 2
  double precision :: val(size(x,1)*size(x,2))

  nx=size(x,1)
  ny=size(x,2)

  counter=0
  do j=1,ny
     do i=1,nx
        counter=counter+1
        val(counter)=x(i,j)
     end do
  end do

  anor=0.0d0

  if(present(error))then
     call stand_devi_1dd( val, anor, error )
  else
     call stand_devi_1dd( val, anor )
  end if

end subroutine stand_devi_2dd

!---------------------------------------
!---------------------------------------

subroutine stand_devi_3df( x, anor, error )  ! 3 次元データの標準偏差を計算
  implicit none
  real, intent(in) :: x(:,:,:)  ! データ
  real, intent(inout) :: anor  ! 標準偏差
  real, intent(in), optional :: error  ! 欠損値
  integer :: i, j, k, counter
  integer :: nx  ! データ数 1
  integer :: ny  ! データ数 2
  integer :: nz  ! データ数 3
  real :: val(size(x,1)*size(x,2)*size(x,3))

  nx=size(x,1)
  ny=size(x,2)
  nz=size(x,3)

  counter=0
  do k=1,nz
     do j=1,ny
        do i=1,nx
           counter=counter+1
           val(counter)=x(i,j,k)
        end do
     end do
  end do

  anor=0.0

  if(present(error))then
     call stand_devi_1df( val, anor, error )
  else
     call stand_devi_1df( val, anor )
  end if

end subroutine stand_devi_3df

!---------------------------------------
!---------------------------------------

subroutine stand_devi_3dd( x, anor, error )  ! 3 次元データの標準偏差を計算
  implicit none
  double precision, intent(in) :: x(:,:,:)  ! データ
  double precision, intent(inout) :: anor  ! 標準偏差
  double precision, intent(in), optional :: error  ! 欠損値
  integer :: i, j, k, counter
  integer :: nx  ! データ数 1
  integer :: ny  ! データ数 2
  integer :: nz  ! データ数 3
  double precision :: val(size(x,1)*size(x,2)*size(x,3))

  nx=size(x,1)
  ny=size(x,2)
  nz=size(x,3)

  counter=0
  do k=1,nz
     do j=1,ny
        do i=1,nx
           counter=counter+1
           val(counter)=x(i,j,k)
        end do
     end do
  end do

  anor=0.0d0

  if(present(error))then
     call stand_devi_1dd( val, anor, error )
  else
     call stand_devi_1dd( val, anor )
  end if

end subroutine stand_devi_3dd

!---------------------------------------
!---------------------------------------

subroutine covariance_1df( x, y, cov, error, nc )  ! 2 つの 1 次元データの共分散を計算
  ! 共分散$\sigma $の定義は,
  ! $$\sigma =\sum^{nx}_{i=1}{(x-\bar{x})(y-\bar{y})} $$
  implicit none
  real, intent(in) :: x(:)  ! データ 1
  real, intent(in) :: y(size(x))  ! データ 2
  real, intent(inout) :: cov  ! 標準偏差
  real, intent(in), optional :: error  ! 欠損値
  integer, intent(inout), optional :: nc  ! 欠損値が存在するデータセットの平均
                                          ! に用いたデータ数.
  integer :: i
  integer :: nx  ! データ数
  integer :: nt
  real :: an1(size(x)), an2(size(x))
  real :: rx(size(x)), ry(size(x))

  nx=size(x)
  cov=0.0

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_1d( nx, y ),  &
  &                                     "covariance_1d" )
  end if

  if(present(error))then
     call replace_undef_f( x, y, rx, ry, error )
     nt=0
     call Anomaly_1df( rx, an1, error )
     call Anomaly_1df( ry, an2, error )
     do i=1,nx
        if(an1(i)/=error.and.an2(i)/=error)then
           cov=cov+an1(i)*an2(i)
           nt=nt+1
        end if
     end do

     if(present(nc))then
        nc=nt
     end if

     if(cov/=error.and.nt/=0)then
        cov=cov/real(nt)
     end if
  else
     call Anomaly_1df( x, an1 )
     call Anomaly_1df( y, an2 )
     do i=1,nx
        cov=cov+an1(i)*an2(i)
     end do
     cov=cov/real(nx)
  end if

end subroutine covariance_1df

!---------------------------------------
!---------------------------------------

subroutine covariance_1dd( x, y, cov, error, nc )  ! 2 つの 1 次元データの共分散を計算
  ! 共分散$\sigma $の定義は,
  ! $$\sigma =\sum^{nx}_{i=1}{(x-\bar{x})(y-\bar{y})} $$
  implicit none
  double precision, intent(in) :: x(:)  ! データ 1
  double precision, intent(in) :: y(size(x))  ! データ 2
  double precision, intent(inout) :: cov  ! 標準偏差
  double precision, intent(in), optional :: error  ! 欠損値
  integer, intent(inout), optional :: nc  ! 欠損値が存在するデータセットの平均
                                          ! に用いたデータ数.
  integer :: i
  integer :: nx  ! データ数
  integer :: nt
  double precision :: an1(size(x)), an2(size(x))
  double precision :: rx(size(x)), ry(size(x))

  nx=size(x)
  cov=0.0d0

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_1d( nx, y ),  &
  &                                     "covariance_1d" )
  end if

  if(present(error))then
     call replace_undef_d( x, y, rx, ry, error )
     nt=0
     call Anomaly_1dd( rx, an1, error )
     call Anomaly_1dd( ry, an2, error )
     do i=1,nx
        if(an1(i)/=error.and.an2(i)/=error)then
           cov=cov+an1(i)*an2(i)
           nt=nt+1
        end if
     end do

     if(present(nc))then
        nc=nt
     end if

     if(cov/=error.and.nt/=0)then
        cov=cov/dble(nt)
     end if
  else
     call Anomaly_1dd( x, an1 )
     call Anomaly_1dd( y, an2 )
     do i=1,nx
        cov=cov+an1(i)*an2(i)
     end do
     cov=cov/dble(nx)
  end if

end subroutine covariance_1dd

!---------------------------------------
!---------------------------------------

subroutine covariance_2df( x, y, cov, error, nc )  ! 2 つの 2 次元データの共分散を計算
  implicit none
  real, intent(in) :: x(:,:)  ! データ 1
  real, intent(in) :: y(size(x,1),size(x,2))  ! データ 2
  real, intent(inout) :: cov  ! 標準偏差
  real, intent(in), optional :: error  ! 欠損値
  integer, intent(inout), optional :: nc  ! 欠損値が存在するデータセットの平均
                                          ! に用いたデータ数.
  integer :: i, j, counter
  integer :: nx  ! データ数 1
  integer :: ny  ! データ数 2
  real :: val1(size(x,1)*size(x,2)), val2(size(x,1)*size(x,2))

  nx=size(x,1)
  ny=size(x,2)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, y ),  &
  &                                     "covariance_2d" )
  end if

  counter=0

  do j=1,ny
     do i=1,nx
        counter=counter+1
        val1(counter)=x(i,j)
        val2(counter)=y(i,j)
     end do
  end do

  cov=0.0

  if(present(error))then
     if(present(nc))then
        call covariance_1df( val1, val2, cov, error, nc=nc )
     else
        call covariance_1df( val1, val2, cov, error )
     end if
  else
     call covariance_1df( val1, val2, cov )
  end if

end subroutine covariance_2df

!---------------------------------------
!---------------------------------------

subroutine covariance_2dd( x, y, cov, error, nc )  ! 2 つの 2 次元データの共分散を計算
  implicit none
  double precision, intent(in) :: x(:,:)  ! データ 1
  double precision, intent(in) :: y(size(x,1),size(x,2))  ! データ 2
  double precision, intent(inout) :: cov  ! 標準偏差
  double precision, intent(in), optional :: error  ! 欠損値
  integer, intent(inout), optional :: nc  ! 欠損値が存在するデータセットの平均
                                          ! に用いたデータ数.
  integer :: i, j, counter
  integer :: nx  ! データ数 1
  integer :: ny  ! データ数 2
  double precision :: val1(size(x,1)*size(x,2)), val2(size(x,1)*size(x,2))

  nx=size(x,1)
  ny=size(x,2)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, y ),  &
  &                                     "covariance_2d" )
  end if

  counter=0

  do j=1,ny
     do i=1,nx
        counter=counter+1
        val1(counter)=x(i,j)
        val2(counter)=y(i,j)
     end do
  end do

  cov=0.0d0

  if(present(error))then
     if(present(nc))then
        call covariance_1dd( val1, val2, cov, error, nc=nc )
     else
        call covariance_1dd( val1, val2, cov, error )
     end if
  else
     call covariance_1dd( val1, val2, cov )
  end if

end subroutine covariance_2dd

!---------------------------------------
!---------------------------------------

subroutine covariance_3df( x, y, cov, error, nc )  ! 2 つの 3 次元データの共分散を計算
  implicit none
  real, intent(in) :: x(:,:,:)  ! データ 1
  real, intent(in) :: y(size(x,1),size(x,2),size(x,3))  ! データ 2
  real, intent(inout) :: cov  ! 標準偏差
  real, intent(in), optional :: error  ! 欠損値
  integer, intent(inout), optional :: nc  ! 欠損値が存在するデータセットの平均
                                          ! に用いたデータ数.
  integer :: i, j, k, counter
  integer :: nx  ! データ数 1
  integer :: ny  ! データ数 2
  integer :: nz  ! データ数 3
  real :: val1(size(x,1)*size(x,2)*size(x,3)), val2(size(x,1)*size(x,2)*size(x,3))

  nx=size(x,1)
  ny=size(x,2)
  nz=size(x,3)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, y ),  &
  &                                     "covariance_3d" )
  end if

  counter=0

  do k=1,nz
     do j=1,ny
        do i=1,nx
           counter=counter+1
           val1(counter)=x(i,j,k)
           val2(counter)=y(i,j,k)
        end do
     end do
  end do

  cov=0.0

  if(present(error))then
     if(present(nc))then
        call covariance_1df( val1, val2, cov, error, nc=nc )
     else
        call covariance_1df( val1, val2, cov, error )
     end if
  else
     call covariance_1df( val1, val2, cov )
  end if

end subroutine covariance_3df

!---------------------------------------
!---------------------------------------

subroutine covariance_3dd( x, y, cov, error, nc )  ! 2 つの 3 次元データの共分散を計算
  implicit none
  double precision, intent(in) :: x(:,:,:)  ! データ 1
  double precision, intent(in) :: y(size(x,1),size(x,2),size(x,3))  ! データ 2
  double precision, intent(inout) :: cov  ! 標準偏差
  double precision, intent(in), optional :: error  ! 欠損値
  integer, intent(inout), optional :: nc  ! 欠損値が存在するデータセットの平均
                                          ! に用いたデータ数.
  integer :: i, j, k, counter
  integer :: nx  ! データ数 1
  integer :: ny  ! データ数 2
  integer :: nz  ! データ数 3
  double precision :: val1(size(x,1)*size(x,2)*size(x,3)), val2(size(x,1)*size(x,2)*size(x,3))

  nx=size(x,1)
  ny=size(x,2)
  nz=size(x,3)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, y ),  &
  &                                     "covariance_3d" )
  end if

  counter=0

  do k=1,nz
     do j=1,ny
        do i=1,nx
           counter=counter+1
           val1(counter)=x(i,j,k)
           val2(counter)=y(i,j,k)
        end do
     end do
  end do

  cov=0.0d0

  if(present(error))then
     if(present(nc))then
        call covariance_1dd( val1, val2, cov, error, nc=nc )
     else
        call covariance_1dd( val1, val2, cov, error )
     end if
  else
     call covariance_1dd( val1, val2, cov )
  end if

end subroutine covariance_3dd

!---------------------------------------
!---------------------------------------

subroutine nearest_search_1df( x, point, i, hx, hp )
  ! 1 次元最近傍探索ルーチン
  ! interpo_search_1d から値を求め, その値と +1 した値の距離を比較して
  ! 距離の短い方を選択する.
  implicit none
  real, intent(in) :: x(:)  ! 漸増配列
  real, intent(in) :: point  ! この点
  integer, intent(inout) :: i  ! point の最近傍地点の要素番号
  real, intent(in), optional :: hx(size(x))  ! x 座標のスケール因子
  real, intent(in), optional :: hp  ! point でのスケール因子 !! まだ用意しただけ
  real :: tmp1, tmp2
  integer :: j, nx

  nx=size(x)

  call interpo_search_1df( x, point, j )

  if(j==0)then
     i=1
  else if(j==nx)then
     i=nx
  else
     tmp1=x(j)
     tmp2=x(j+1)

     if(abs(point-tmp1)>abs(tmp2-point))then
        i=j+1
     else
        i=j
     end if
  end if

end subroutine nearest_search_1df

!---------------------------------------
!---------------------------------------

subroutine nearest_search_1dd( x, point, i, hx, hp )
  ! 1 次元最近傍探索ルーチン
  ! interpo_search_1d から値を求め, その値と +1 した値の距離を比較して
  ! 距離の短い方を選択する.
  implicit none
  double precision, intent(in) :: x(:)  ! 漸増配列
  double precision, intent(in) :: point  ! この点
  integer, intent(inout) :: i  ! point の最近傍地点の要素番号
  double precision, intent(in), optional :: hx(size(x))  ! x 座標のスケール因子
  double precision, intent(in), optional :: hp  ! point でのスケール因子 !! まだ用意しただけ
  double precision :: tmp1, tmp2
  integer :: j, nx

  nx=size(x)

  call interpo_search_1dd( x, point, j )

  if(j==0)then
     i=1
  else if(j==nx)then
     i=nx
  else
     tmp1=x(j)
     tmp2=x(j+1)

     if(abs(point-tmp1)>abs(tmp2-point))then
        i=j+1
     else
        i=j
     end if
  end if

end subroutine nearest_search_1dd

!---------------------------------------
!---------------------------------------

subroutine nearest_search_2df( x, y, pointx, pointy, i, j )
  ! 2 次元最近傍探索ルーチン
  ! nearest_search_1d から値を求める.
  ! 本来, 2 次元であるため, 周囲 4 点の最近を計算する必要があるが,
  ! ここでは直交直線座標を考えているので, 各軸方向独立で最近点を計算し,
  ! どちらも最近の点が求めたい 2 次元の最近点となる.
  implicit none
  real, intent(in) :: x(:)  ! 漸増配列 x
  real, intent(in) :: y(:)  ! 漸増配列 y
  real, intent(in) :: pointx  ! この点 x
  real, intent(in) :: pointy  ! この点 y
  integer, intent(inout) :: i  ! pointx の最近要素番号
  integer, intent(inout) :: j  ! pointy の最近要素番号

  call nearest_search_1df( x, pointx, i )
  call nearest_search_1df( y, pointy, j )

end subroutine nearest_search_2df

!---------------------------------------
!---------------------------------------

subroutine nearest_search_2dd( x, y, pointx, pointy, i, j )
  ! 2 次元最近傍探索ルーチン
  ! nearest_search_1d から値を求める.
  ! 本来, 2 次元であるため, 周囲 4 点の最近を計算する必要があるが,
  ! ここでは直交直線座標を考えているので, 各軸方向独立で最近点を計算し,
  ! どちらも最近の点が求めたい 2 次元の最近点となる.
  implicit none
  double precision, intent(in) :: x(:)  ! 漸増配列 x
  double precision, intent(in) :: y(:)  ! 漸増配列 y
  double precision, intent(in) :: pointx  ! この点 x
  double precision, intent(in) :: pointy  ! この点 y
  integer, intent(inout) :: i  ! pointx の最近要素番号
  integer, intent(inout) :: j  ! pointy の最近要素番号

  call nearest_search_1dd( x, pointx, i )
  call nearest_search_1dd( y, pointy, j )

end subroutine nearest_search_2dd

!---------------------------------------
!---------------------------------------

subroutine nearest_search_3df( x, y, z, pointx, pointy, pointz, i, j, k )
  ! 2 次元最近傍探索ルーチン
  ! nearest_search_1d から値を求める.
  ! 本来, 2 次元であるため, 周囲 4 点の最近を計算する必要があるが,
  ! ここでは直交直線座標を考えているので, 各軸方向独立で最近点を計算し,
  ! どちらも最近の点が求めたい 2 次元の最近点となる.
  implicit none
  real, intent(in) :: x(:)  ! 漸増配列 x
  real, intent(in) :: y(:)  ! 漸増配列 y
  real, intent(in) :: z(:)  ! 漸増配列 z
  real, intent(in) :: pointx  ! この点 x
  real, intent(in) :: pointy  ! この点 y
  real, intent(in) :: pointz  ! この点 z
  integer, intent(inout) :: i  ! pointx の最近要素番号
  integer, intent(inout) :: j  ! pointy の最近要素番号
  integer, intent(inout) :: k  ! pointz の最近要素番号

  call nearest_search_1df( x, pointx, i )
  call nearest_search_1df( y, pointy, j )
  call nearest_search_1df( z, pointz, k )

end subroutine nearest_search_3df

!---------------------------------------
!---------------------------------------

subroutine nearest_search_3dd( x, y, z, pointx, pointy, pointz, i, j, k )
  ! 2 次元最近傍探索ルーチン
  ! nearest_search_1d から値を求める.
  ! 本来, 2 次元であるため, 周囲 4 点の最近を計算する必要があるが,
  ! ここでは直交直線座標を考えているので, 各軸方向独立で最近点を計算し,
  ! どちらも最近の点が求めたい 2 次元の最近点となる.
  implicit none
  double precision, intent(in) :: x(:)  ! 漸増配列 x
  double precision, intent(in) :: y(:)  ! 漸増配列 y
  double precision, intent(in) :: z(:)  ! 漸増配列 z
  double precision, intent(in) :: pointx  ! この点 x
  double precision, intent(in) :: pointy  ! この点 y
  double precision, intent(in) :: pointz  ! この点 z
  integer, intent(inout) :: i  ! pointx の最近要素番号
  integer, intent(inout) :: j  ! pointy の最近要素番号
  integer, intent(inout) :: k  ! pointz の最近要素番号

  call nearest_search_1dd( x, pointx, i )
  call nearest_search_1dd( y, pointy, j )
  call nearest_search_1dd( z, pointz, k )

end subroutine nearest_search_3dd

!---------------------------------------
!---------------------------------------

subroutine nearest_neighbor_search_1df( x, point, i, undef )
  ! 1 次元最近傍探索ルーチン
  ! 座標 x が漸増関数でなくてもよい.
  ! | point - x | の最小値となる点を i として返す.
  implicit none
  real, intent(in) :: x(:)  ! 漸増配列
  real, intent(in) :: point  ! この点
  integer, intent(inout) :: i  ! point の最近傍地点の要素番号
  real, intent(in), optional :: undef   ! 未定義点
  real, dimension(size(x)) :: dx
  integer :: j
  real :: tmpv

  if(present(undef))then
     dx=undef
     do j=1,size(x)
        if(x(j)/=undef)then
           dx(j)=abs(x(j)-point)
        end if
     end do

     call min_val_1d( dx, i, tmpv, undef=undef )

  else

     do j=1,size(x)
        dx(j)=abs(x(j)-point)
     end do

     call min_val_1d( dx, i, tmpv )

  end if

end subroutine nearest_neighbor_search_1df

!---------------------------------------
!---------------------------------------

subroutine nearest_neighbor_search_1dd( x, point, i, undef )
  ! 1 次元最近傍探索ルーチン
  ! 座標 x が漸増関数でなくてもよい.
  ! | point - x | の最小値となる点を i として返す.
  implicit none
  double precision, intent(in) :: x(:)  ! 漸増配列
  double precision, intent(in) :: point  ! この点
  integer, intent(inout) :: i  ! point の最近傍地点の要素番号
  double precision, intent(in), optional :: undef   ! 未定義点
  double precision, dimension(size(x)) :: dx
  integer :: j
  double precision :: tmpv

  if(present(undef))then
     dx=undef
     do j=1,size(x)
        if(x(j)/=undef)then
           dx(j)=abs(x(j)-point)
        end if
     end do

     call min_val_1d( dx, i, tmpv, undef=undef )

  else

     do j=1,size(x)
        dx(j)=abs(x(j)-point)
     end do

     call min_val_1d( dx, i, tmpv )

  end if

end subroutine nearest_neighbor_search_1dd

!---------------------------------------
!---------------------------------------

subroutine nearest_neighbor_search_2df( x, y, pointx, pointy, i, j, undef )
  ! 2 次元最近傍探索ルーチン
  ! nearest_neighbor_search_1d から値を求める.
  ! x, y は漸増関数でなくてよい.
  ! 本来, 2 次元であるため, 周囲 4 点の最近を計算する必要があるが,
  ! ここでは直交直線座標を考えているので, 各軸方向独立で最近点を計算し,
  ! どちらも最近の点が求めたい 2 次元の最近点となる.
  implicit none
  real, intent(in) :: x(:)  ! 漸増配列 x
  real, intent(in) :: y(:)  ! 漸増配列 y
  real, intent(in) :: pointx  ! この点 x
  real, intent(in) :: pointy  ! この点 y
  integer, intent(inout) :: i  ! pointx の最近要素番号
  integer, intent(inout) :: j  ! pointy の最近要素番号
  real, optional :: undef  ! 未定義値

  if(present(undef))then
     call nearest_neighbor_search_1df( x, pointx, i, undef=undef )
     call nearest_neighbor_search_1df( y, pointy, j, undef=undef )
  else
     call nearest_neighbor_search_1df( x, pointx, i )
     call nearest_neighbor_search_1df( y, pointy, j )
  end if

end subroutine nearest_neighbor_search_2df

!---------------------------------------
!---------------------------------------

subroutine nearest_neighbor_search_2dd( x, y, pointx, pointy, i, j, undef )
  ! 2 次元最近傍探索ルーチン
  ! nearest_neighbor_search_1d から値を求める.
  ! x, y は漸増関数でなくてよい.
  ! 本来, 2 次元であるため, 周囲 4 点の最近を計算する必要があるが,
  ! ここでは直交直線座標を考えているので, 各軸方向独立で最近点を計算し,
  ! どちらも最近の点が求めたい 2 次元の最近点となる.
  implicit none
  double precision, intent(in) :: x(:)  ! 漸増配列 x
  double precision, intent(in) :: y(:)  ! 漸増配列 y
  double precision, intent(in) :: pointx  ! この点 x
  double precision, intent(in) :: pointy  ! この点 y
  integer, intent(inout) :: i  ! pointx の最近要素番号
  integer, intent(inout) :: j  ! pointy の最近要素番号
  double precision, optional :: undef  ! 未定義値

  if(present(undef))then
     call nearest_neighbor_search_1dd( x, pointx, i, undef=undef )
     call nearest_neighbor_search_1dd( y, pointy, j, undef=undef )
  else
     call nearest_neighbor_search_1dd( x, pointx, i )
     call nearest_neighbor_search_1dd( y, pointy, j )
  end if

end subroutine nearest_neighbor_search_2dd

!---------------------------------------
!---------------------------------------

subroutine nearest_neighbor_search_3df( x, y, z, pointx, pointy, pointz,  &
  &                                     i, j, k, undef )
  ! 3 次元最近傍探索ルーチン
  ! nearest_neighbor_search_1d から値を求める.
  ! x, y は漸増関数でなくてよい.
  ! 本来, 3 次元であるため, 周囲 8 点の最近を計算する必要があるが,
  ! ここでは直交直線座標を考えているので, 各軸方向独立で最近点を計算し,
  ! どちらも最近の点が求めたい 3 次元の最近点となる.
  implicit none
  real, intent(in) :: x(:)  ! 漸増配列 x
  real, intent(in) :: y(:)  ! 漸増配列 y
  real, intent(in) :: z(:)  ! 漸増配列 z
  real, intent(in) :: pointx  ! この点 x
  real, intent(in) :: pointy  ! この点 y
  real, intent(in) :: pointz  ! この点 z
  integer, intent(inout) :: i  ! pointx の最近要素番号
  integer, intent(inout) :: j  ! pointy の最近要素番号
  integer, intent(inout) :: k  ! pointz の最近要素番号
  real, optional :: undef  ! 未定義値

  if(present(undef))then
     call nearest_neighbor_search_1df( x, pointx, i, undef=undef )
     call nearest_neighbor_search_1df( y, pointy, j, undef=undef )
     call nearest_neighbor_search_1df( z, pointz, k, undef=undef )
  else
     call nearest_neighbor_search_1df( x, pointx, i )
     call nearest_neighbor_search_1df( y, pointy, j )
     call nearest_neighbor_search_1df( z, pointz, k )
  end if

end subroutine nearest_neighbor_search_3df

!---------------------------------------
!---------------------------------------

subroutine nearest_neighbor_search_3dd( x, y, z, pointx, pointy, pointz,  &
  &                                     i, j, k, undef )
  ! 3 次元最近傍探索ルーチン
  ! nearest_neighbor_search_1d から値を求める.
  ! x, y は漸増関数でなくてよい.
  ! 本来, 3 次元であるため, 周囲 8 点の最近を計算する必要があるが,
  ! ここでは直交直線座標を考えているので, 各軸方向独立で最近点を計算し,
  ! どちらも最近の点が求めたい 3 次元の最近点となる.
  implicit none
  double precision, intent(in) :: x(:)  ! 漸増配列 x
  double precision, intent(in) :: y(:)  ! 漸増配列 y
  double precision, intent(in) :: z(:)  ! 漸増配列 z
  double precision, intent(in) :: pointx  ! この点 x
  double precision, intent(in) :: pointy  ! この点 y
  double precision, intent(in) :: pointz  ! この点 z
  integer, intent(inout) :: i  ! pointx の最近要素番号
  integer, intent(inout) :: j  ! pointy の最近要素番号
  integer, intent(inout) :: k  ! pointz の最近要素番号
  double precision, optional :: undef  ! 未定義値

  if(present(undef))then
     call nearest_neighbor_search_1dd( x, pointx, i, undef=undef )
     call nearest_neighbor_search_1dd( y, pointy, j, undef=undef )
     call nearest_neighbor_search_1dd( z, pointz, k, undef=undef )
  else
     call nearest_neighbor_search_1dd( x, pointx, i )
     call nearest_neighbor_search_1dd( y, pointy, j )
     call nearest_neighbor_search_1dd( z, pointz, k )
  end if

end subroutine nearest_neighbor_search_3dd

!---------------------------------------
!---------------------------------------

subroutine interpo_search_1df( x, point, i, undeff, stdopt )
  ! 漸増配列（要素数が増えるごとに値が大きくなる配列）のなかで,
  ! point の前に来る要素番号を出力する.
  implicit none
  real, intent(in) :: x(:)  ! 漸増配列
  real, intent(in) :: point  ! この点
  integer, intent(inout) :: i  ! point の値を越えない最大の値をもつ要素番号
  integer, intent(in), optional :: undeff  ! 探索範囲の配列要素より小さい値を探索しようとした際, undef を返すが, その undef 値を設定する. default では 0.
  logical, intent(in), optional :: stdopt  ! 探索範囲が見つからない旨の標準出力を表示させないようにする.
                                           ! default では .false. (表示させる)
  integer :: nx, j
  integer :: just
  logical :: stderr

  nx=size(x)
  if(present(undeff))then
     just=undeff
  else
     just=0
  end if

  if(present(stdopt))then
     stderr=stdopt
  else
     stderr=.false.
  end if

  if(x(1)>point)then

     if(stderr.eqv..false.)then
        write(*,*) "****** WARNING ******"
        write(*,*) "searching point was not found :", x(1), point
        write(*,*) "Abort. Exit.!!!"
     end if
     i=just

  else

     do j=1,nx
        if(x(j)<=point)then
           i=j
        else
           exit
        end if
     end do

  end if

end subroutine interpo_search_1df

!---------------------------------------
!---------------------------------------

subroutine interpo_search_1dd( x, point, i, undeff, stdopt )
  ! 漸増配列（要素数が増えるごとに値が大きくなる配列）のなかで,
  ! point の前に来る要素番号を出力する.
  implicit none
  double precision, intent(in) :: x(:)  ! 漸増配列
  double precision, intent(in) :: point  ! この点
  integer, intent(inout) :: i  ! point の値を越えない最大の値をもつ要素番号
  integer, intent(in), optional :: undeff  ! 探索範囲の配列要素より小さい値を探索しようとした際, undef を返すが, その undef 値を設定する. default では 0.
  logical, intent(in), optional :: stdopt  ! 探索範囲が見つからない旨の標準出力を表示させないようにする.
                                           ! default では .false. (表示させる)
  integer :: nx, j
  integer :: just
  logical :: stderr

  nx=size(x)
  if(present(undeff))then
     just=undeff
  else
     just=0
  end if

  if(present(stdopt))then
     stderr=stdopt
  else
     stderr=.false.
  end if

  if(x(1)>point)then

     if(stderr.eqv..false.)then
        write(*,*) "****** WARNING ******"
        write(*,*) "searching point was not found :", x(1), point
        write(*,*) "Abort. Exit.!!!"
     end if
     i=just

  else

     do j=1,nx
        if(x(j)<=point)then
           i=j
        else
           exit
        end if
     end do

  end if

end subroutine interpo_search_1dd


subroutine interpo_search_2df( x, y, pointx, pointy, i, j, undeff, stdopt )
  ! 漸増配列（要素数が増えるごとに値が大きくなる配列）のなかで,
  ! point の前に来る要素番号を出力する.
  implicit none
  real, intent(in) :: x(:)  ! 漸増配列 x
  real, intent(in) :: y(:)  ! 漸増配列 y
  real, intent(in) :: pointx  ! この点 x
  real, intent(in) :: pointy  ! この点 y
  integer, intent(inout) :: i  ! pointx の値を越えない最大の値をもつ要素番号
  integer, intent(inout) :: j  ! pointy の値を越えない最大の値をもつ要素番号
  integer, intent(in), optional :: undeff  ! 探索範囲の配列要素より小さい値を探索しようとした際, undef を返すが, その undef 値を設定する. default では 0.
  logical, intent(in), optional :: stdopt  ! 探索範囲が見つからない旨の標準出力を表示させないようにする.
  integer :: just
  logical :: stderr

  if(present(stdopt))then
     stderr=stdopt
  else
     stderr=.false.
  end if

  if(present(undeff))then
     just=undeff
     call interpo_search_1df( x, pointx, i, just, stdopt=stderr )
     call interpo_search_1df( y, pointy, j, just, stdopt=stderr )
  else
     call interpo_search_1df( x, pointx, i, stdopt=stderr )
     call interpo_search_1df( y, pointy, j, stdopt=stderr )
  end if

end subroutine interpo_search_2df


subroutine interpo_search_2dd( x, y, pointx, pointy, i, j, undeff, stdopt )
  ! 漸増配列（要素数が増えるごとに値が大きくなる配列）のなかで,
  ! point の前に来る要素番号を出力する.
  implicit none
  double precision, intent(in) :: x(:)  ! 漸増配列 x
  double precision, intent(in) :: y(:)  ! 漸増配列 y
  double precision, intent(in) :: pointx  ! この点 x
  double precision, intent(in) :: pointy  ! この点 y
  integer, intent(inout) :: i  ! pointx の値を越えない最大の値をもつ要素番号
  integer, intent(inout) :: j  ! pointy の値を越えない最大の値をもつ要素番号
  integer, intent(in), optional :: undeff  ! 探索範囲の配列要素より小さい値を探索しようとした際, undef を返すが, その undef 値を設定する. default では 0.
  logical, intent(in), optional :: stdopt  ! 探索範囲が見つからない旨の標準出力を表示させないようにする.
  integer :: just
  logical :: stderr

  if(present(stdopt))then
     stderr=stdopt
  else
     stderr=.false.
  end if

  if(present(undeff))then
     just=undeff
     call interpo_search_1dd( x, pointx, i, just, stdopt=stderr )
     call interpo_search_1dd( y, pointy, j, just, stdopt=stderr )
  else
     call interpo_search_1dd( x, pointx, i, stdopt=stderr )
     call interpo_search_1dd( y, pointy, j, stdopt=stderr )
  end if

end subroutine interpo_search_2dd


subroutine interpo_search_3df( x, y, z, pointx, pointy, pointz, i, j, k,  &
  &                            undeff, stdopt )
  ! 漸増配列（要素数が増えるごとに値が大きくなる配列）のなかで,
  ! point の前に来る要素番号を出力する.
  implicit none
  real, intent(in) :: x(:)  ! 漸増配列 x
  real, intent(in) :: y(:)  ! 漸増配列 y
  real, intent(in) :: z(:)  ! 漸増配列 z
  real, intent(in) :: pointx  ! この点 x
  real, intent(in) :: pointy  ! この点 y
  real, intent(in) :: pointz  ! この点 z
  integer, intent(inout) :: i  ! pointx の値を越えない最大の値をもつ要素番号
  integer, intent(inout) :: j  ! pointy の値を越えない最大の値をもつ要素番号
  integer, intent(inout) :: k  ! pointz の値を越えない最大の値をもつ要素番号
  integer, intent(in), optional :: undeff  ! 探索範囲の配列要素より小さい値を探索しようとした際, undef を返すが, その undef 値を設定する. default では 0.
  logical, intent(in), optional :: stdopt  ! 探索範囲が見つからない旨の標準出力を表示させないようにする.
  integer :: just
  logical :: stderr

  if(present(stdopt))then
     stderr=stdopt
  else
     stderr=.false.
  end if

  if(present(undeff))then
     just=int(undeff)
     call interpo_search_1df( x, pointx, i, just, stdopt=stderr )
     call interpo_search_1df( y, pointy, j, just, stdopt=stderr )
     call interpo_search_1df( z, pointz, k, just, stdopt=stderr )
  else
     call interpo_search_1df( x, pointx, i, stdopt=stderr )
     call interpo_search_1df( y, pointy, j, stdopt=stderr )
     call interpo_search_1df( z, pointz, k, stdopt=stderr )
  end if

end subroutine interpo_search_3df


subroutine interpo_search_3dd( x, y, z, pointx, pointy, pointz, i, j, k,  &
  &                            undeff, stdopt )
  ! 漸増配列（要素数が増えるごとに値が大きくなる配列）のなかで,
  ! point の前に来る要素番号を出力する.
  implicit none
  double precision, intent(in) :: x(:)  ! 漸増配列 x
  double precision, intent(in) :: y(:)  ! 漸増配列 y
  double precision, intent(in) :: z(:)  ! 漸増配列 z
  double precision, intent(in) :: pointx  ! この点 x
  double precision, intent(in) :: pointy  ! この点 y
  double precision, intent(in) :: pointz  ! この点 z
  integer, intent(inout) :: i  ! pointx の値を越えない最大の値をもつ要素番号
  integer, intent(inout) :: j  ! pointy の値を越えない最大の値をもつ要素番号
  integer, intent(inout) :: k  ! pointz の値を越えない最大の値をもつ要素番号
  integer, intent(in), optional :: undeff  ! 探索範囲の配列要素より小さい値を探索しようとした際, undef を返すが, その undef 値を設定する. default では 0.
  logical, intent(in), optional :: stdopt  ! 探索範囲が見つからない旨の標準出力を表示させないようにする.
  integer :: just
  logical :: stderr

  if(present(stdopt))then
     stderr=stdopt
  else
     stderr=.false.
  end if

  if(present(undeff))then
     just=int(undeff)
     call interpo_search_1dd( x, pointx, i, just, stdopt=stderr )
     call interpo_search_1dd( y, pointy, j, just, stdopt=stderr )
     call interpo_search_1dd( z, pointz, k, just, stdopt=stderr )
  else
     call interpo_search_1dd( x, pointx, i, stdopt=stderr )
     call interpo_search_1dd( y, pointy, j, stdopt=stderr )
     call interpo_search_1dd( z, pointz, k, stdopt=stderr )
  end if

end subroutine interpo_search_3dd

!---------------------------------------
!---------------------------------------

subroutine auto_interpo_search_1df( x, point, i, undeff, stdopt )
  ! 漸増配列（要素数が増えるごとに値が大きくなる配列）のなかで,
  ! 漸増配列 point の前に来る要素番号を出力する.
  implicit none
  real, intent(in) :: x(:)  ! 漸増配列
  real, intent(in) :: point(:)  ! 求めたい点 (複数)
  integer, intent(inout) :: i(size(point)) ! 各 point の値を越えない最大の値をもつ要素番号
  integer, intent(in), optional :: undeff  ! 探索範囲の配列要素より小さい値を探索しようとした際, undef を返すが, その undef 値を設定する. default では 0.
  logical, intent(in), optional :: stdopt  ! 探索範囲が見つからない旨の標準出力を表示させないようにする.
                                           ! default では .false. (表示させる)
  integer :: nx, ni, j, icount, jcount
  integer :: just
  logical :: stderr

  nx=size(x)
  ni=size(point)

  if(present(undeff))then
     just=undeff
  else
     just=0
  end if

  if(present(stdopt))then
     stderr=stdopt
  else
     stderr=.false.
  end if

!-- はじめに point のうちどれくらい x の最低値より下か (候補がないか)

  jcount=1

  do j=1,ni
     if(x(1)>point(j))then
        if(stderr.eqv..false.)then
           write(*,*) "****** WARNING ******"
           write(*,*) "searching point was not found :", x(1), point(j)
           write(*,*) "Abort. Exit.!!!"
        end if
        i(j)=just
     else
        jcount=j
        exit
     end if
  end do

!-- jcount (point の x(1) に最も近接する点) から x についてループを回し
!-- カウントを開始する.

  icount=1

  do j=jcount,ni
     if(x(icount)<=point(j))then
        do while(x(icount)<=point(j))
           i(j)=icount
           icount=icount+1

           if(icount>nx)then
              i(j:ni)=nx
              exit
           end if
        end do
        icount=icount-1
     else
        icount=icount+1
     end if
  end do

end subroutine auto_interpo_search_1df

!---------------------------------------
!---------------------------------------

subroutine auto_interpo_search_1dd( x, point, i, undeff, stdopt )
  ! 漸増配列（要素数が増えるごとに値が大きくなる配列）のなかで,
  ! 漸増配列 point の前に来る要素番号を出力する.
  implicit none
  double precision, intent(in) :: x(:)  ! 漸増配列
  double precision, intent(in) :: point(:)  ! 求めたい点 (複数)
  integer, intent(inout) :: i(size(point)) ! 各 point の値を越えない最大の値をもつ要素番号
  integer, intent(in), optional :: undeff  ! 探索範囲の配列要素より小さい値を探索しようとした際, undef を返すが, その undef 値を設定する. default では 0.
  logical, intent(in), optional :: stdopt  ! 探索範囲が見つからない旨の標準出力を表示させないようにする.
                                           ! default では .false. (表示させる)
  integer :: nx, ni, j, icount, jcount
  integer :: just
  logical :: stderr

  nx=size(x)
  ni=size(point)

  if(present(undeff))then
     just=undeff
  else
     just=0
  end if

  if(present(stdopt))then
     stderr=stdopt
  else
     stderr=.false.
  end if

!-- はじめに point のうちどれくらい x の最低値より下か (候補がないか)

  jcount=1

  do j=1,ni
     if(x(1)>point(j))then
        if(stderr.eqv..false.)then
           write(*,*) "****** WARNING ******"
           write(*,*) "searching point was not found :", x(1), point(j)
           write(*,*) "Abort. Exit.!!!"
        end if
        i(j)=just
     else
        jcount=j
        exit
     end if
  end do

!-- jcount (point の x(1) に最も近接する点) から x についてループを回し
!-- カウントを開始する.

  icount=1

  do j=jcount,ni
     if(x(icount)<=point(j))then
        do while(x(icount)<=point(j))
           i(j)=icount
           icount=icount+1

           if(icount>nx)then
              i(j:ni)=nx
              exit
           end if
        end do
        icount=icount-1
     else
        icount=icount+1
     end if
  end do

end subroutine auto_interpo_search_1dd

!---------------------------------------
!---------------------------------------

subroutine auto_interpo_search_2df( x, y, pointx, pointy, i, j, undeff, stdopt )
  ! 漸増配列（要素数が増えるごとに値が大きくなる配列）のなかで,
  ! pointx, pointy の前に来る要素番号を出力する.
  implicit none
  real, intent(in) :: x(:)  ! 漸増配列 x
  real, intent(in) :: y(:)  ! 漸増配列 y
  real, intent(in) :: pointx(:)  ! x についての位置を求めたい漸増配列
  real, intent(in) :: pointy(:)  ! y についての位置を求めたい漸増配列
  integer, intent(inout) :: i(size(pointx))  ! pointx の値を越えない最大の値をもつ要素番号
  integer, intent(inout) :: j(size(pointy))  ! pointy の値を越えない最大の値をもつ要素番号
  integer, intent(in), optional :: undeff  ! 探索範囲の配列要素より小さい値を探索しようとした際, undef を返すが, その undef 値を設定する. default では 0.
  logical, intent(in), optional :: stdopt  ! 探索範囲が見つからない旨の標準出力を表示させないようにする.
  integer :: just
  logical :: stderr

  if(present(stdopt))then
     stderr=stdopt
  else
     stderr=.false.
  end if

  if(present(undeff))then
     just=undeff
     call auto_interpo_search_1df( x, pointx, i, just, stdopt=stderr )
     call auto_interpo_search_1df( y, pointy, j, just, stdopt=stderr )
  else
     call auto_interpo_search_1df( x, pointx, i, stdopt=stderr )
     call auto_interpo_search_1df( y, pointy, j, stdopt=stderr )
  end if

end subroutine auto_interpo_search_2df

!---------------------------------------
!---------------------------------------

subroutine auto_interpo_search_2dd( x, y, pointx, pointy, i, j, undeff, stdopt )
  ! 漸増配列（要素数が増えるごとに値が大きくなる配列）のなかで,
  ! pointx, pointy の前に来る要素番号を出力する.
  implicit none
  double precision, intent(in) :: x(:)  ! 漸増配列 x
  double precision, intent(in) :: y(:)  ! 漸増配列 y
  double precision, intent(in) :: pointx(:)  ! x についての位置を求めたい漸増配列
  double precision, intent(in) :: pointy(:)  ! y についての位置を求めたい漸増配列
  integer, intent(inout) :: i(size(pointx))  ! pointx の値を越えない最大の値をもつ要素番号
  integer, intent(inout) :: j(size(pointy))  ! pointy の値を越えない最大の値をもつ要素番号
  integer, intent(in), optional :: undeff  ! 探索範囲の配列要素より小さい値を探索しようとした際, undef を返すが, その undef 値を設定する. default では 0.
  logical, intent(in), optional :: stdopt  ! 探索範囲が見つからない旨の標準出力を表示させないようにする.
  integer :: just
  logical :: stderr

  if(present(stdopt))then
     stderr=stdopt
  else
     stderr=.false.
  end if

  if(present(undeff))then
     just=undeff
     call auto_interpo_search_1dd( x, pointx, i, just, stdopt=stderr )
     call auto_interpo_search_1dd( y, pointy, j, just, stdopt=stderr )
  else
     call auto_interpo_search_1dd( x, pointx, i, stdopt=stderr )
     call auto_interpo_search_1dd( y, pointy, j, stdopt=stderr )
  end if

end subroutine auto_interpo_search_2dd

!---------------------------------------
!---------------------------------------

subroutine auto_interpo_search_3df( x, y, z, pointx, pointy, pointz, i, j, k,  &
  &                                 undeff, stdopt )
  ! 漸増配列（要素数が増えるごとに値が大きくなる配列）のなかで,
  ! point の前に来る要素番号を出力する.
  implicit none
  real, intent(in) :: x(:)  ! 漸増配列 x
  real, intent(in) :: y(:)  ! 漸増配列 y
  real, intent(in) :: z(:)  ! 漸増配列 z
  real, intent(in) :: pointx(:)  ! x についての位置を求めたい漸増配列
  real, intent(in) :: pointy(:)  ! y についての位置を求めたい漸増配列
  real, intent(in) :: pointz(:)  ! z についての位置を求めたい漸増配列
  integer, intent(inout) :: i(size(pointx))  ! pointx の値を越えない最大の値をもつ要素番号
  integer, intent(inout) :: j(size(pointy))  ! pointy の値を越えない最大の値をもつ要素番号
  integer, intent(inout) :: k(size(pointz))  ! pointz の値を越えない最大の値をもつ要素番号
  integer, intent(in), optional :: undeff  ! 探索範囲の配列要素より小さい値を探索しようとした際, undef を返すが, その undef 値を設定する. default では 0.
  logical, intent(in), optional :: stdopt  ! 探索範囲が見つからない旨の標準出力を表示させないようにする.
  integer :: just
  logical :: stderr

  if(present(stdopt))then
     stderr=stdopt
  else
     stderr=.false.
  end if

  if(present(undeff))then
     just=int(undeff)
     call auto_interpo_search_1df( x, pointx, i, just, stdopt=stderr )
     call auto_interpo_search_1df( y, pointy, j, just, stdopt=stderr )
     call auto_interpo_search_1df( z, pointz, k, just, stdopt=stderr )
  else
     call auto_interpo_search_1df( x, pointx, i, stdopt=stderr )
     call auto_interpo_search_1df( y, pointy, j, stdopt=stderr )
     call auto_interpo_search_1df( z, pointz, k, stdopt=stderr )
  end if

end subroutine auto_interpo_search_3df

!---------------------------------------
!---------------------------------------

subroutine auto_interpo_search_3dd( x, y, z, pointx, pointy, pointz, i, j, k,  &
  &                                 undeff, stdopt )
  ! 漸増配列（要素数が増えるごとに値が大きくなる配列）のなかで,
  ! point の前に来る要素番号を出力する.
  implicit none
  double precision, intent(in) :: x(:)  ! 漸増配列 x
  double precision, intent(in) :: y(:)  ! 漸増配列 y
  double precision, intent(in) :: z(:)  ! 漸増配列 z
  double precision, intent(in) :: pointx(:)  ! x についての位置を求めたい漸増配列
  double precision, intent(in) :: pointy(:)  ! y についての位置を求めたい漸増配列
  double precision, intent(in) :: pointz(:)  ! z についての位置を求めたい漸増配列
  integer, intent(inout) :: i(size(pointx))  ! pointx の値を越えない最大の値をもつ要素番号
  integer, intent(inout) :: j(size(pointy))  ! pointy の値を越えない最大の値をもつ要素番号
  integer, intent(inout) :: k(size(pointz))  ! pointz の値を越えない最大の値をもつ要素番号
  integer, intent(in), optional :: undeff  ! 探索範囲の配列要素より小さい値を探索しようとした際, undef を返すが, その undef 値を設定する. default では 0.
  logical, intent(in), optional :: stdopt  ! 探索範囲が見つからない旨の標準出力を表示させないようにする.
  integer :: just
  logical :: stderr

  if(present(stdopt))then
     stderr=stdopt
  else
     stderr=.false.
  end if

  if(present(undeff))then
     just=int(undeff)
     call auto_interpo_search_1dd( x, pointx, i, just, stdopt=stderr )
     call auto_interpo_search_1dd( y, pointy, j, just, stdopt=stderr )
     call auto_interpo_search_1dd( z, pointz, k, just, stdopt=stderr )
  else
     call auto_interpo_search_1dd( x, pointx, i, stdopt=stderr )
     call auto_interpo_search_1dd( y, pointy, j, stdopt=stderr )
     call auto_interpo_search_1dd( z, pointz, k, stdopt=stderr )
  end if

end subroutine auto_interpo_search_3dd

!---------------------------------------
!---------------------------------------

subroutine interpolation_1df( x, y, point, val, h, hp )
  ! 1 次の線形内挿ルーチン
  implicit none
  real, intent(in) :: x(2)  ! 内挿点の左右端
  real, intent(in) :: y(2)  ! x の点で定義されている値
  real, intent(in) :: point  ! 内挿点
  real, intent(inout) :: val  ! 内挿点での値
  real, intent(in), optional :: h(2)  ! 内挿点の左右でのスケール因子
  real, intent(in), optional :: hp  ! 内挿点でのスケール因子
  real :: fd, dt
  real :: tmin
  real :: tmax
  real :: xmin
  real :: xmax

  if(present(h))then
     tmin=x(1)*h(1)
     tmax=x(2)*h(2)
  else
     tmin=x(1)
     tmax=x(2)
  end if

  xmin=y(1)
  xmax=y(2)

  if(present(hp))then
     dt=hp*point-tmin
  else
     dt=point-tmin
  end if

  fd=(xmax-xmin)/(tmax-tmin)

  val=xmin+dt*fd

end subroutine interpolation_1df

!---------------------------------------
!---------------------------------------

subroutine interpolation_1dd( x, y, point, val, h, hp )
  ! 1 次の線形内挿ルーチン
  implicit none
  double precision, intent(in) :: x(2)  ! 内挿点の左右端
  double precision, intent(in) :: y(2)  ! x の点で定義されている値
  double precision, intent(in) :: point  ! 内挿点
  double precision, intent(inout) :: val  ! 内挿点での値
  double precision, intent(in), optional :: h(2)  ! 内挿点の左右でのスケール因子
  double precision, intent(in), optional :: hp  ! 内挿点でのスケール因子
  double precision :: fd, dt
  double precision :: tmin
  double precision :: tmax
  double precision :: xmin
  double precision :: xmax

  if(present(h))then
     tmin=x(1)*h(1)
     tmax=x(2)*h(2)
  else
     tmin=x(1)
     tmax=x(2)
  end if

  xmin=y(1)
  xmax=y(2)

  if(present(hp))then
     dt=hp*point-tmin
  else
     dt=point-tmin
  end if

  fd=(xmax-xmin)/(tmax-tmin)

  val=xmin+dt*fd

end subroutine interpolation_1dd

!---------------------------------------
!---------------------------------------

subroutine interpolation_2df( x, y, z, point, val, h, hp )
  ! 2 次の重線形内挿ルーチン
  ! 本ルーチンは直線直交座標空間でのみ使用可能.
  implicit none
  real, intent(in) :: x(2)  ! 内挿の空間点 x 方向の左右端
  real, intent(in) :: y(2)  ! 内挿の空間点 y 方向の左右端
  real, intent(in) :: z(2,2)  ! x, y での各点での値, (i,j) について, i<=x, j<=y
  real, intent(in) :: point(2)  ! 内挿点 point(1)<=x 座標, point(2)<=y 座標
  real, intent(inout) :: val  ! 内挿点での値
  real, intent(in), optional :: h(2,2)  ! 内挿点の四隅でのスケール因子
  real, intent(in), optional :: hp(2)  ! 内挿点でのスケール因子
  real :: valx(2)

  if(present(h))then
     ! y(1) での x 方向の内挿点での値
     call interpolation_1df( x, (/z(1,1), z(2,1)/), point(1), valx(1),  &
  &                          (/h(1,1), h(2,1)/), hp(1) )

     ! y(2) での x 方向の内挿点での値
     call interpolation_1df( x, (/z(1,2), z(2,2)/), point(1), valx(2),  &
  &                          (/h(1,2), h(2,2)/), hp(1) )
   
     ! x の内挿点からの y 方向の内挿点での値(これが求める内挿点)
     call interpolation_1df( y, valx, point(2), val, (/h(1,1), h(1,2)/), hp(2) )
  else
     ! y(1) での x 方向の内挿点での値
     call interpolation_1df( x, (/z(1,1), z(2,1)/), point(1), valx(1) )

     ! y(2) での x 方向の内挿点での値
     call interpolation_1df( x, (/z(1,2), z(2,2)/), point(1), valx(2) )
   
     ! x の内挿点からの y 方向の内挿点での値(これが求める内挿点)
     call interpolation_1df( y, valx, point(2), val )
  end if

end subroutine interpolation_2df

!---------------------------------------
!---------------------------------------

subroutine interpolation_2dd( x, y, z, point, val, h, hp )
  ! 2 次の重線形内挿ルーチン
  ! 本ルーチンは直線直交座標空間でのみ使用可能.
  implicit none
  double precision, intent(in) :: x(2)  ! 内挿の空間点 x 方向の左右端
  double precision, intent(in) :: y(2)  ! 内挿の空間点 y 方向の左右端
  double precision, intent(in) :: z(2,2)  ! x, y での各点での値, (i,j) について, i<=x, j<=y
  double precision, intent(in) :: point(2)  ! 内挿点 point(1)<=x 座標, point(2)<=y 座標
  double precision, intent(inout) :: val  ! 内挿点での値
  double precision, intent(in), optional :: h(2,2)  ! 内挿点の四隅でのスケール因子
  double precision, intent(in), optional :: hp(2)  ! 内挿点でのスケール因子
  double precision :: valx(2)

  if(present(h))then
     ! y(1) での x 方向の内挿点での値
     call interpolation_1dd( x, (/z(1,1), z(2,1)/), point(1), valx(1),  &
  &                          (/h(1,1), h(2,1)/), hp(1) )

     ! y(2) での x 方向の内挿点での値
     call interpolation_1dd( x, (/z(1,2), z(2,2)/), point(1), valx(2),  &
  &                          (/h(1,2), h(2,2)/), hp(1) )
   
     ! x の内挿点からの y 方向の内挿点での値(これが求める内挿点)
     call interpolation_1dd( y, valx, point(2), val, (/h(1,1), h(1,2)/), hp(2) )
  else
     ! y(1) での x 方向の内挿点での値
     call interpolation_1dd( x, (/z(1,1), z(2,1)/), point(1), valx(1) )

     ! y(2) での x 方向の内挿点での値
     call interpolation_1dd( x, (/z(1,2), z(2,2)/), point(1), valx(2) )
   
     ! x の内挿点からの y 方向の内挿点での値(これが求める内挿点)
     call interpolation_1dd( y, valx, point(2), val )
  end if

end subroutine interpolation_2dd

!---------------------------------------
!---------------------------------------

subroutine interpolation_3df( x, y, z, u, point, val, h, hp )
  ! 3 次の重線形内挿ルーチン
  ! 本ルーチンは直線直交座標空間でのみ使用可能.
  implicit none
  real, intent(in) :: x(2)  ! 内挿の空間点 x 方向の左右端
  real, intent(in) :: y(2)  ! 内挿の空間点 y 方向の左右端
  real, intent(in) :: z(2)  ! 内挿の空間点 z 方向の左右端
  real, intent(in) :: u(2,2,2)  ! x, y, z での各点での値, (i,j,k) について, i<=x, j<=y, k<=z
  real, intent(in) :: point(3)  ! 内挿点 point(1)<=x 座標, point(2)<=y 座標, point(3)<=z 座標
  real, intent(inout) :: val  ! 内挿点での値
  real, intent(in), optional :: h(2,2,2)  ! 内挿点の八隅でのスケール因子
  real, intent(in), optional :: hp(3)  ! 内挿点でのスケール因子
  real :: valx(2)

  if(present(h))then
     ! z(1) での x-y 平面での重線形内挿の値
     call interpolation_2df( x, y, u(:,:,1), point(1:2), valx(1),  &
  &                          h(:,:,1), hp(1:2) )
   
     ! z(2) での x 方向の内挿点での値
     call interpolation_2df( x, y, u(:,:,2), point(1:2), valx(2),  &
  &                          h(:,:,2), hp(1:2) )
   
     ! z(1) の内挿点からの z 方向の内挿点での値(これが求める内挿点)
     call interpolation_1df( z, valx, point(3), val, h(1,1,:), hp(3) )
  else
     ! z(1) での x-y 平面での重線形内挿の値
     call interpolation_2df( x, y, u(:,:,1), point(1:2), valx(1) )
   
     ! z(2) での x 方向の内挿点での値
     call interpolation_2df( x, y, u(:,:,2), point(1:2), valx(2) )
   
     ! z(1) の内挿点からの z 方向の内挿点での値(これが求める内挿点)
     call interpolation_1df( (/z(1), z(2)/), (/valx(1), valx(2)/), point(3), val )
  end if

end subroutine interpolation_3df

!---------------------------------------
!---------------------------------------

subroutine interpolation_3dd( x, y, z, u, point, val, h, hp )
  ! 3 次の重線形内挿ルーチン
  ! 本ルーチンは直線直交座標空間でのみ使用可能.
  implicit none
  double precision, intent(in) :: x(2)  ! 内挿の空間点 x 方向の左右端
  double precision, intent(in) :: y(2)  ! 内挿の空間点 y 方向の左右端
  double precision, intent(in) :: z(2)  ! 内挿の空間点 z 方向の左右端
  double precision, intent(in) :: u(2,2,2)  ! x, y, z での各点での値, (i,j,k) について, i<=x, j<=y, k<=z
  double precision, intent(in) :: point(3)  ! 内挿点 point(1)<=x 座標, point(2)<=y 座標, point(3)<=z 座標
  double precision, intent(inout) :: val  ! 内挿点での値
  double precision, intent(in), optional :: h(2,2,2)  ! 内挿点の八隅でのスケール因子
  double precision, intent(in), optional :: hp(3)  ! 内挿点でのスケール因子
  double precision :: valx(2)

  if(present(h))then
     ! z(1) での x-y 平面での重線形内挿の値
     call interpolation_2dd( x, y, u(:,:,1), point(1:2), valx(1),  &
  &                          h(:,:,1), hp(1:2) )
   
     ! z(2) での x 方向の内挿点での値
     call interpolation_2dd( x, y, u(:,:,2), point(1:2), valx(2),  &
  &                          h(:,:,2), hp(1:2) )
   
     ! z(1) の内挿点からの z 方向の内挿点での値(これが求める内挿点)
     call interpolation_1dd( z, valx, point(3), val, h(1,1,:), hp(3) )
  else
     ! z(1) での x-y 平面での重線形内挿の値
     call interpolation_2dd( x, y, u(:,:,1), point(1:2), valx(1) )
   
     ! z(2) での x 方向の内挿点での値
     call interpolation_2dd( x, y, u(:,:,2), point(1:2), valx(2) )
   
     ! z(1) の内挿点からの z 方向の内挿点での値(これが求める内挿点)
     call interpolation_1dd( (/z(1), z(2)/), (/valx(1), valx(2)/), point(3), val )
  end if

end subroutine interpolation_3dd

!------------------------------------
!------------------------------------

subroutine interpo_undef_1df( x, undef, u )
  ! 座標 x で定義されているデータ u の中に undef がある場合,
  ! 隣接する値の入った格子のデータから内挿を行う.
  ! 隣接格子がない場合の外挿は行わない.
  implicit none
  real, intent(in) :: x(:)  ! 元座標
  real, intent(in) :: undef  ! 未定義値
  real, intent(inout) :: u(size(x))  ! 元データ

  integer :: i, j, nx, counts, counte
  integer :: ns(size(x)), ne(size(x))
  integer, allocatable, dimension(:) :: nt
  real :: du

  nx=size(x)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_1d( nx, u ),  &
  &                                     "interpo_undef_1d" )
  end if

  counts=0
  counte=0

  if(u(1)==undef)then
     if(u(2)==undef)then
        counte=-1
     end if
  end if
  if(u(nx)==undef)then
     if(u(nx-1)==undef)then
        counts=-1
     end if
  end if

  do i=2,nx-1
     if(u(i)==undef)then
        if(u(i+1)/=undef)then
           counte=counte+1
           if(counte>0)then
              ne(counte)=i
           end if
        end if
     end if
  end do

  do i=nx-1,2,-1
     if(u(i)==undef)then
        if(u(i-1)/=undef)then
           counts=counts+1
           if(counts>0)then
              ns(counts)=i
           end if
        end if
     end if
  end do

  !-- ns loop is opposite, so ns order reverse.

  if(counts>0)then
     allocate(nt(counts))
     do i=1,counts
        nt(i)=ns(counts-i+1)
     end do
     do i=1,counts
        ns(i)=nt(i)
     end do
  end if

  if(counts>0.and.counte>0)then
     do i=1,counts
        du=(u(ne(i)+1)-u(ns(i)-1))/(x(ne(i)+1)-x(ns(i)-1))
        do j=ns(i),ne(i)
           u(j)=u(j-1)+du*(x(j)-x(j-1))
        end do
     end do
  end if

end subroutine interpo_undef_1df

!------------------------------------
!------------------------------------

subroutine interpo_undef_1dd( x, undef, u )
  ! 座標 x で定義されているデータ u の中に undef がある場合,
  ! 隣接する値の入った格子のデータから内挿を行う.
  ! 隣接格子がない場合の外挿は行わない.
  implicit none
  double precision, intent(in) :: x(:)  ! 元座標
  double precision, intent(in) :: undef  ! 未定義値
  double precision, intent(inout) :: u(size(x))  ! 元データ

  integer :: i, j, nx, counts, counte
  integer :: ns(size(x)), ne(size(x))
  integer, allocatable, dimension(:) :: nt
  double precision :: du

  nx=size(x)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_1d( nx, u ),  &
  &                                     "interpo_undef_1d" )
  end if

  counts=0
  counte=0

  if(u(1)==undef)then
     if(u(2)==undef)then
        counte=-1
     end if
  end if
  if(u(nx)==undef)then
     if(u(nx-1)==undef)then
        counts=-1
     end if
  end if

  do i=2,nx-1
     if(u(i)==undef)then
        if(u(i+1)/=undef)then
           counte=counte+1
           if(counte>0)then
              ne(counte)=i
           end if
        end if
     end if
  end do

  do i=nx-1,2,-1
     if(u(i)==undef)then
        if(u(i-1)/=undef)then
           counts=counts+1
           if(counts>0)then
              ns(counts)=i
           end if
        end if
     end if
  end do

  !-- ns loop is opposite, so ns order reverse.

  if(counts>0)then
     allocate(nt(counts))
     do i=1,counts
        nt(i)=ns(counts-i+1)
     end do
     do i=1,counts
        ns(i)=nt(i)
     end do
  end if

  if(counts>0.and.counte>0)then
     do i=1,counts
        du=(u(ne(i)+1)-u(ns(i)-1))/(x(ne(i)+1)-x(ns(i)-1))
        do j=ns(i),ne(i)
           u(j)=u(j-1)+du*(x(j)-x(j-1))
        end do
     end do
  end if

end subroutine interpo_undef_1dd

!------------------------------------
!------------------------------------

subroutine auto_interpolation_1df( x, r, u, v, undef, undefr, stdopt, extopt )
  ! 座標 x で定義されているデータ u を
  ! 座標 r で定義されるデータ v に自動で内挿する.
  implicit none
  real, intent(in) :: x(:)  ! 元座標
  real, intent(in) :: r(:)  ! 内挿座標
  real, intent(in) :: u(size(x))  ! 元データ
  real, intent(inout) :: v(size(r))  ! 内挿したデータ
  real, intent(in), optional :: undef  ! 未定義値
  real, intent(in), optional :: undefr  ! 内挿領域内での未定義値.
        ! 内挿点の隣接点が undefr なら, 内挿点も undefr が代入される
  logical, intent(in), optional :: stdopt  ! 探索範囲が見つからない旨の標準出力を表示させないようにする.
  logical, intent(in), optional :: extopt  ! 探索範囲外の場合, データのある最近接点の値をコピーする.
                                           ! デフォルトではコピーしない .false.
  integer :: ir(size(r))
  integer :: i, nx, nr
  real :: defun
  real :: rdefun
  logical :: stderr, extra

  nx=size(x)
  nr=size(r)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_1d( nx, u ),  &
  &                                     "auto_interpolation_1d" )
     call check_array_size_dmp_message( check_array_size_1d( nr, v ),  &
  &                                     "auto_interpolation_1d" )
  end if

  if(present(undef))then
     defun=undef
  else
     defun=-999.0
  end if

  if(present(undefr))then
     rdefun=undefr
  else
     rdefun=-999.0
  end if

  if(present(stdopt))then
     stderr=stdopt
  else
     stderr=.false.
  end if

  if(present(extopt))then
     extra=extopt
  else
     extra=.false.
  end if

  call auto_interpo_search_1df( x, r, ir, undeff=0, stdopt=stderr )

  do i=1, nr
     if(ir(i)/=0)then
        if(ir(i)<nx)then
           if(u(ir(i))/=rdefun.and.u(ir(i)+1)/=rdefun)then
              call interpolation_1df( x(ir(i):ir(i)+1),  &
  &                                   u(ir(i):ir(i)+1), r(i), v(i) )
           else
              v(i)=rdefun
           end if
        else if(ir(i)==nx.and.x(nx)==r(i))then
           v(i)=u(ir(i))
        else
           if(extra.eqv..true.)then
              v(i)=u(nx)
           else
              v(i)=defun
           end if
        end if
     else
        if(extra.eqv..true.)then
           v(i)=u(1)
        else
           v(i)=defun
        end if
     end if
  end do

end subroutine auto_interpolation_1df

!------------------------------------
!------------------------------------

subroutine auto_interpolation_1dd( x, r, u, v, undef, undefr, stdopt, extopt )
  ! 座標 x で定義されているデータ u を
  ! 座標 r で定義されるデータ v に自動で内挿する.
  implicit none
  double precision, intent(in) :: x(:)  ! 元座標
  double precision, intent(in) :: r(:)  ! 内挿座標
  double precision, intent(in) :: u(size(x))  ! 元データ
  double precision, intent(inout) :: v(size(r))  ! 内挿したデータ
  double precision, intent(in), optional :: undef  ! 未定義値
  double precision, intent(in), optional :: undefr  ! 内挿領域内での未定義値.
        ! 内挿点の隣接点が undefr なら, 内挿点も undefr が代入される
  logical, intent(in), optional :: stdopt  ! 探索範囲が見つからない旨の標準出力を表示させないようにする.
  logical, intent(in), optional :: extopt  ! 探索範囲外の場合, データのある最近接点の値をコピーする.
                                           ! デフォルトではコピーしない .false.
  integer :: ir(size(r))
  integer :: i, nx, nr
  double precision :: defun
  double precision :: rdefun
  logical :: stderr, extra

  nx=size(x)
  nr=size(r)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_1d( nx, u ),  &
  &                                     "auto_interpolation_1d" )
     call check_array_size_dmp_message( check_array_size_1d( nr, v ),  &
  &                                     "auto_interpolation_1d" )
  end if

  if(present(undef))then
     defun=undef
  else
     defun=-999.0d0
  end if

  if(present(undefr))then
     rdefun=undefr
  else
     rdefun=-999.0d0
  end if

  if(present(stdopt))then
     stderr=stdopt
  else
     stderr=.false.
  end if

  if(present(extopt))then
     extra=extopt
  else
     extra=.false.
  end if

  call auto_interpo_search_1dd( x, r, ir, undeff=0, stdopt=stderr )

  do i=1, nr
     if(ir(i)/=0)then
        if(ir(i)<nx)then
           if(u(ir(i))/=rdefun.and.u(ir(i)+1)/=rdefun)then
              call interpolation_1dd( x(ir(i):ir(i)+1),  &
  &                                   u(ir(i):ir(i)+1), r(i), v(i) )
           else
              v(i)=rdefun
           end if
        else if(ir(i)==nx.and.x(nx)==r(i))then
           v(i)=u(ir(i))
        else
           if(extra.eqv..true.)then
              v(i)=u(nx)
           else
              v(i)=defun
           end if
        end if
     else
        if(extra.eqv..true.)then
           v(i)=u(1)
        else
           v(i)=defun
        end if
     end if
  end do

end subroutine auto_interpolation_1dd

!------------------------------------
!------------------------------------

subroutine auto_interpolation_2df( x, y, r, q, u, v, undef, undefr, stdopt,  &
  &                                extopt )
  ! 座標 x, y で定義されているデータ u を
  ! 座標 r, q で定義されるデータ v に自動で内挿する.
  implicit none
  real, intent(in) :: x(:)  ! 元座標 1
  real, intent(in) :: y(:)  ! 元座標 2
  real, intent(in) :: r(:)  ! 内挿座標 1
  real, intent(in) :: q(:)  ! 内挿座標 2
  real, intent(in) :: u(size(x),size(y))  ! 元データ
  real, intent(inout) :: v(size(r),size(q))  ! 内挿したデータ
  real, intent(in), optional :: undef  ! 未定義値
  real, intent(in), optional :: undefr  ! 内挿領域内での未定義値.
        ! 内挿点の隣接点が undefr なら, 内挿点も undefr が代入される
  logical, intent(in), optional :: stdopt  ! 探索範囲が見つからない旨の標準出力を表示させないようにする.
  logical, intent(in), optional :: extopt  ! 探索範囲外の場合, データのある最近接点の値をコピーする.
                                           ! デフォルトではコピーしない .false.
  integer :: ir(size(r)), iq(size(q))
  integer :: i, j, nx, ny, nr, nq
  real :: defun
  real :: rdefun
  logical :: stderr, extra

  nx=size(x)
  ny=size(y)
  nr=size(r)
  nq=size(q)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, u ),  &
  &                                     "auto_interpolation_2d" )
     call check_array_size_dmp_message( check_array_size_2d( nr, nq, v ),  &
  &                                     "auto_interpolation_2d" )
  end if

  if(present(undef))then
     defun=undef
  else
     defun=-999.0
  end if

  if(present(undefr))then
     rdefun=undefr
  else
     rdefun=-999.0
  end if

  if(present(stdopt))then
     stderr=stdopt
  else
     stderr=.false.
  end if

  if(present(extopt))then
     extra=extopt
  else
     extra=.false.
  end if

  call auto_interpo_search_2df( x, y, r, q, ir, iq, undeff=0, stdopt=stderr )

  do j=1, nq
     do i=1, nr
        if(ir(i)/=0.and.iq(j)/=0)then
           if(u(ir(i),iq(j))/=rdefun)then
              if(ir(i)<nx.and.iq(j)<ny)then
                 if(u(ir(i),iq(j)+1)/=rdefun.and.  &
  &                 u(ir(i)+1,iq(j))/=rdefun.and.  &
  &                 u(ir(i)+1,iq(j)+1)/=rdefun)then
                    call interpolation_2df( x(ir(i):ir(i)+1),  &
  &                                         y(iq(j):iq(j)+1),  &
  &                                         u(ir(i):ir(i)+1,iq(j):iq(j)+1),  &
  &                                         (/r(i), q(j)/), v(i,j) )
                 else
                    v(i,j)=rdefun
                 end if

              else if(x(nx)==r(i).and.y(ny)==q(j))then
                 v(i,j)=u(nx,ny)

              else if(x(nx)==r(i).and.iq(j)<ny)then
                 if(u(nx,iq(j)+1)/=rdefun)then
                    call interpolation_1df( y(iq(j):iq(j)+1),  &
  &                                         u(nx,iq(j):iq(j)+1),  &
  &                                         q(j), v(i,j) )
                 else
                    v(i,j)=rdefun
                 end if

              else if(y(ny)==q(j).and.ir(i)<nx)then
                 if(u(ir(i)+1,ny)/=rdefun)then
                    call interpolation_1df( x(ir(i):ir(i)+1),  &
  &                                         u(ir(i):ir(i)+1,ny),  &
  &                                         r(i), v(i,j) )
                 else
                    v(i,j)=rdefun
                 end if
              else
                 if(extra.eqv..true.)then
                    if(ir(i)==nx.and.iq(j)==ny)then
                       v(i,j)=u(nx,ny)
                    else if(ir(i)==nx.and.iq(j)<ny)then
                       if(u(nx,iq(j))/=rdefun.and.u(nx,iq(j)+1)/=rdefun)then
                          call interpolation_1df( y(iq(j):iq(j)+1),  &
  &                                               u(nx,iq(j):iq(j)+1),  &
  &                                               q(j), v(i,j) )
                       else
                          v(i,j)=rdefun
                       end if
                    else if(ir(i)<nx.and.iq(j)==ny)then
                       if(u(ir(i),ny)/=rdefun.and.u(ir(i)+1,ny)/=rdefun)then
                          call interpolation_1df( x(ir(i):ir(i)+1),  &
  &                                               u(ir(i):ir(i)+1,ny),  &
  &                                               r(i), v(i,j) )
                       else
                          v(i,j)=rdefun
                       end if
                    end if
                 else
                    v(i,j)=defun
                 end if
              end if
           else
              v(i,j)=rdefun
           end if
        else
           if(extra.eqv..true.)then
              if(ir(i)==0.and.iq(j)==0)then
                 v(i,j)=u(1,1)
              else if(ir(i)==0.and.iq(j)/=0)then
                 if(u(1,iq(j))/=rdefun.and.u(1,iq(j)+1)/=rdefun)then
                    call interpolation_1df( y(iq(j):iq(j)+1),  &
  &                                         u(1,iq(j):iq(j)+1),  &
  &                                         q(j), v(i,j) )
                 else
                    v(i,j)=rdefun
                 end if
              else if(ir(i)/=0.and.iq(j)==0)then
                 if(u(ir(i),1)/=rdefun.and.u(ir(i)+1,1)/=rdefun)then
                    call interpolation_1df( x(ir(i):ir(i)+1),  &
  &                                         u(ir(i):ir(i)+1,1),  &
  &                                         r(i), v(i,j) )
                 else
                    v(i,j)=rdefun
                 end if
              end if
           else
              v(i,j)=defun
           end if
        end if
     end do
  end do

end subroutine auto_interpolation_2df

!------------------------------------
!------------------------------------

subroutine auto_interpolation_2dd( x, y, r, q, u, v, undef, undefr, stdopt,  &
  &                                extopt )
  ! 座標 x, y で定義されているデータ u を
  ! 座標 r, q で定義されるデータ v に自動で内挿する.
  implicit none
  double precision, intent(in) :: x(:)  ! 元座標 1
  double precision, intent(in) :: y(:)  ! 元座標 2
  double precision, intent(in) :: r(:)  ! 内挿座標 1
  double precision, intent(in) :: q(:)  ! 内挿座標 2
  double precision, intent(in) :: u(size(x),size(y))  ! 元データ
  double precision, intent(inout) :: v(size(r),size(q))  ! 内挿したデータ
  double precision, intent(in), optional :: undef  ! 未定義値
  double precision, intent(in), optional :: undefr  ! 内挿領域内での未定義値.
        ! 内挿点の隣接点が undefr なら, 内挿点も undefr が代入される
  logical, intent(in), optional :: stdopt  ! 探索範囲が見つからない旨の標準出力を表示させないようにする.
  logical, intent(in), optional :: extopt  ! 探索範囲外の場合, データのある最近接点の値をコピーする.
                                           ! デフォルトではコピーしない .false.
  integer :: ir(size(r)), iq(size(q))
  integer :: i, j, nx, ny, nr, nq
  double precision :: defun
  double precision :: rdefun
  logical :: stderr, extra

  nx=size(x)
  ny=size(y)
  nr=size(r)
  nq=size(q)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, u ),  &
  &                                     "auto_interpolation_2d" )
     call check_array_size_dmp_message( check_array_size_2d( nr, nq, v ),  &
  &                                     "auto_interpolation_2d" )
  end if

  if(present(undef))then
     defun=undef
  else
     defun=-999.0d0
  end if

  if(present(undefr))then
     rdefun=undefr
  else
     rdefun=-999.0d0
  end if

  if(present(stdopt))then
     stderr=stdopt
  else
     stderr=.false.
  end if

  if(present(extopt))then
     extra=extopt
  else
     extra=.false.
  end if

  call auto_interpo_search_2dd( x, y, r, q, ir, iq, undeff=0, stdopt=stderr )

  do j=1, nq
     do i=1, nr
        if(ir(i)/=0.and.iq(j)/=0)then
           if(u(ir(i),iq(j))/=rdefun)then
              if(ir(i)<nx.and.iq(j)<ny)then
                 if(u(ir(i),iq(j)+1)/=rdefun.and.  &
  &                 u(ir(i)+1,iq(j))/=rdefun.and.  &
  &                 u(ir(i)+1,iq(j)+1)/=rdefun)then
                    call interpolation_2dd( x(ir(i):ir(i)+1),  &
  &                                         y(iq(j):iq(j)+1),  &
  &                                         u(ir(i):ir(i)+1,iq(j):iq(j)+1),  &
  &                                         (/r(i), q(j)/), v(i,j) )
                 else
                    v(i,j)=rdefun
                 end if

              else if(x(nx)==r(i).and.y(ny)==q(j))then
                 v(i,j)=u(nx,ny)

              else if(x(nx)==r(i).and.iq(j)<ny)then
                 if(u(nx,iq(j)+1)/=rdefun)then
                    call interpolation_1dd( y(iq(j):iq(j)+1),  &
  &                                         u(nx,iq(j):iq(j)+1),  &
  &                                         q(j), v(i,j) )
                 else
                    v(i,j)=rdefun
                 end if

              else if(y(ny)==q(j).and.ir(i)<nx)then
                 if(u(ir(i)+1,ny)/=rdefun)then
                    call interpolation_1dd( x(ir(i):ir(i)+1),  &
  &                                         u(ir(i):ir(i)+1,ny),  &
  &                                         r(i), v(i,j) )
                 else
                    v(i,j)=rdefun
                 end if
              else
                 if(extra.eqv..true.)then
                    if(ir(i)==nx.and.iq(j)==ny)then
                       v(i,j)=u(nx,ny)
                    else if(ir(i)==nx.and.iq(j)<ny)then
                       if(u(nx,iq(j))/=rdefun.and.u(nx,iq(j)+1)/=rdefun)then
                          call interpolation_1dd( y(iq(j):iq(j)+1),  &
  &                                               u(nx,iq(j):iq(j)+1),  &
  &                                               q(j), v(i,j) )
                       else
                          v(i,j)=rdefun
                       end if
                    else if(ir(i)<nx.and.iq(j)==ny)then
                       if(u(ir(i),ny)/=rdefun.and.u(ir(i)+1,ny)/=rdefun)then
                          call interpolation_1dd( x(ir(i):ir(i)+1),  &
  &                                               u(ir(i):ir(i)+1,ny),  &
  &                                               r(i), v(i,j) )
                       else
                          v(i,j)=rdefun
                       end if
                    end if
                 else
                    v(i,j)=defun
                 end if
              end if
           else
              v(i,j)=rdefun
           end if
        else
           if(extra.eqv..true.)then
              if(ir(i)==0.and.iq(j)==0)then
                 v(i,j)=u(1,1)
              else if(ir(i)==0.and.iq(j)/=0)then
                 if(u(1,iq(j))/=rdefun.and.u(1,iq(j)+1)/=rdefun)then
                    call interpolation_1dd( y(iq(j):iq(j)+1),  &
  &                                         u(1,iq(j):iq(j)+1),  &
  &                                         q(j), v(i,j) )
                 else
                    v(i,j)=rdefun
                 end if
              else if(ir(i)/=0.and.iq(j)==0)then
                 if(u(ir(i),1)/=rdefun.and.u(ir(i)+1,1)/=rdefun)then
                    call interpolation_1dd( x(ir(i):ir(i)+1),  &
  &                                         u(ir(i):ir(i)+1,1),  &
  &                                         r(i), v(i,j) )
                 else
                    v(i,j)=rdefun
                 end if
              end if
           else
              v(i,j)=defun
           end if
        end if
     end do
  end do

end subroutine auto_interpolation_2dd

!------------------------------------
!------------------------------------

subroutine auto_interpolation_3df( x, y, z, r, q, p, u, v, undef, undefr,  &
  &                                stdopt, extopt )
  ! 座標 x, y, z で定義されているデータ u を
  ! 座標 r, q, p で定義されるデータ v に自動で内挿する.
  implicit none
  real, intent(in) :: x(:)  ! 元座標 1
  real, intent(in) :: y(:)  ! 元座標 2
  real, intent(in) :: z(:)  ! 元座標 3
  real, intent(in) :: r(:)  ! 内挿座標 1
  real, intent(in) :: q(:)  ! 内挿座標 2
  real, intent(in) :: p(:)  ! 内挿座標 3
  real, intent(in) :: u(size(x),size(y),size(z))  ! 元データ
  real, intent(inout) :: v(size(r),size(q),size(p))  ! 内挿したデータ
  real, intent(in), optional :: undef  ! 未定義値
  real, intent(in), optional :: undefr  ! 内挿領域内での未定義値.
        ! 内挿点の隣接点が undefr なら, 内挿点も undefr が代入される
  logical, intent(in), optional :: stdopt  ! 探索範囲が見つからない旨の標準出力を表示させないようにする.
  logical, intent(in), optional :: extopt  ! 探索範囲外の場合, データのある最近接点の値をコピーする.
                                           ! デフォルトではコピーしない .false.
  integer :: ir(size(r)), iq(size(q)), ip(size(p))
  integer :: i, j, k, nx, ny, nz, nr, nq, np
  real :: defun
  real :: rdefun
  logical :: stderr, extra

  nx=size(x)
  ny=size(y)
  nz=size(z)
  nr=size(r)
  nq=size(q)
  np=size(p)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, u ),  &
  &                                     "auto_interpolation_3d" )
     call check_array_size_dmp_message( check_array_size_3d( nr, nq, np, v ),  &
  &                                     "auto_interpolation_3d" )
  end if

  if(present(undef))then
     defun=undef
  else
     defun=-999.0
  end if

  if(present(undefr))then
     rdefun=undefr
  else
     rdefun=-999.0
  end if

  if(present(stdopt))then
     stderr=stdopt
  else
     stderr=.false.
  end if

  if(present(extopt))then
     extra=extopt
  else
     extra=.false.
  end if

  call auto_interpo_search_3df( x, y, z, r, q, p, ir, iq, ip,  &
  &                             undeff=0, stdopt=stderr )

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,k)

  do k=1, np
     do j=1, nq
        do i=1, nr
           if(ir(i)/=0.and.iq(j)/=0.and.ip(k)/=0)then
              if(u(ir(i),iq(j),ip(k))/=rdefun)then
                 if(ir(i)<nx.and.iq(j)<ny.and.ip(k)<nz)then
                    if(u(ir(i)+1,iq(j),ip(k))/=rdefun.and.  &
  &                    u(ir(i),iq(j)+1,ip(k))/=rdefun.and.  &
  &                    u(ir(i)+1,iq(j)+1,ip(k))/=rdefun.and.  &
  &                    u(ir(i),iq(j),ip(k)+1)/=rdefun.and.  &
  &                    u(ir(i)+1,iq(j),ip(k)+1)/=rdefun.and.  &
  &                    u(ir(i),iq(j)+1,ip(k)+1)/=rdefun.and.  &
  &                    u(ir(i)+1,iq(j)+1,ip(k)+1)/=rdefun)then
                       call interpolation_3df( x(ir(i):ir(i)+1),  &
  &                                            y(iq(j):iq(j)+1),  &
  &                                            z(ip(k):ip(k)+1),  &
  &                                            u(ir(i):ir(i)+1,iq(j):iq(j)+1,ip(k):ip(k)+1),  &
  &                                            (/r(i), q(j), p(k)/), v(i,j,k) )
                    else
                       v(i,j,k)=rdefun
                    end if

                 else if(x(nx)==r(i).and.y(ny)==q(j).and.z(nz)==p(k))then
                    v(i,j,k)=u(ir(i),iq(j),ip(k))

                 else if(x(nx)==r(i).and.iq(j)<ny.and.ip(k)<nz)then
                    if(u(nx,iq(j)+1,ip(k))/=rdefun.and.u(nx,iq(j),ip(k)+1)/=rdefun.and.  &
  &                    u(nx,iq(j)+1,ip(k)+1)/=rdefun)then
                       call interpolation_2df( y(iq(j):iq(j)+1),  &
  &                                            z(ip(k):ip(k)+1),  &
  &                                            u(nx,iq(j):iq(j)+1,ip(k):ip(k)+1),  &
  &                                            (/q(j), p(k)/), v(i,j,k) )
                    else
                       v(i,j,k)=rdefun
                    end if

                 else if(y(ny)==q(j).and.ir(i)<nx.and.ip(k)<nz)then
                    if(u(ir(i)+1,ny,ip(k))/=rdefun.and.u(ir(i),ny,ip(k)+1)/=rdefun.and.  &
  &                    u(ir(i)+1,ny,ip(k)+1)/=rdefun)then
                       call interpolation_2df( x(ir(i):ir(i)+1),  &
  &                                            z(ip(k):ip(k)+1),  &
  &                                            u(ir(i):ir(i)+1,ny,ip(k):ip(k)+1),  &
  &                                            (/r(i), p(k)/), v(i,j,k) )
                    else
                       v(i,j,k)=rdefun
                    end if

                 else if(z(nz)==p(k).and.ir(i)<nx.and.iq(j)<ny)then
                    if(u(ir(i)+1,iq(j),nz)/=rdefun.and.u(ir(i),iq(j)+1,nz)/=rdefun.and.  &
  &                    u(ir(i)+1,iq(j)+1,nz)/=rdefun)then
                       call interpolation_2df( x(ir(i):ir(i)+1),  &
  &                                            y(iq(j):iq(j)+1),  &
  &                                            u(ir(i):ir(i)+1,iq(j):iq(j)+1,nz),  &
  &                                            (/r(i), q(j)/), v(i,j,k) )
                    else
                       v(i,j,k)=rdefun
                    end if

                 else if(x(nx)==r(i).and.y(ny)==q(j).and.ip(k)<nz)then
                    if(u(nx,ny,ip(k)+1)/=rdefun)then
                       call interpolation_1df( z(ip(k):ip(k)+1),  &
  &                                            u(nx,ny,ip(k):ip(k)+1),  &
  &                                            p(k), v(i,j,k) )
                    else
                       v(i,j,k)=rdefun
                    end if

                 else if(x(nx)==r(i).and.z(nz)==p(k).and.iq(j)<ny)then
                    if(u(nx,iq(j)+1,nz)/=rdefun)then
                       call interpolation_1df( y(iq(j):iq(j)+1),  &
  &                                            u(nx,iq(j):iq(j)+1,nz),  &
  &                                            q(j), v(i,j,k) )
                    else
                       v(i,j,k)=rdefun
                    end if

                 else if(y(ny)==q(j).and.z(nz)==p(k).and.ir(i)<nx)then
                    if(u(ir(i)+1,ny,nz)/=rdefun)then
                       call interpolation_1df( x(ir(i):ir(i)+1),  &
  &                                            u(ir(i):ir(i)+1,ny,nz),  &
  &                                            r(i), v(i,j,k) )
                    else
                       v(i,j,k)=rdefun
                    end if

                 else

                    if(extra.eqv..true.)then    ! 以下, 上位の if 文から (1,1,1) はない.
                       if(ir(i)==nx)then
                          if(iq(j)==ny)then
                             if(ip(k)==nz)then   ! (0,0,0)
                                v(i,j,k)=u(nx,ny,nz)
                             else               ! (0,0,1)
                                if(u(nx,ny,ip(k))/=rdefun.and.  &
  &                                u(nx,ny,ip(k)+1)/=rdefun)then
                                   call interpolation_1df( z(ip(k):ip(k)+1),  &
  &                                                        u(nx,ny,ip(k):ip(k)+1),  &
  &                                                        p(k), v(i,j,k) )
                                else
                                   v(i,j,k)=rdefun
                                end if
                             end if
                          else
                             if(ip(k)==nz)then   ! (0,1,0)
                                if(u(nx,iq(j),nz)/=rdefun.and.  &
  &                                u(nx,iq(j)+1,nz)/=rdefun)then
                                   call interpolation_1df( y(iq(j):iq(j)+1),  &
  &                                                        u(nx,iq(j):iq(j)+1,nz),  &
  &                                                        q(j), v(i,j,k) )
                                else
                                   v(i,j,k)=rdefun
                                end if
                             else               ! (0,1,1)
                                if(u(nx,iq(j),ip(k))/=rdefun.and.  &
  &                                u(nx,iq(j)+1,ip(k))/=rdefun.and.  &
  &                                u(nx,iq(j),ip(k)+1)/=rdefun.and.  &
  &                                u(nx,iq(j)+1,ip(k)+1)/=rdefun)then
                                   call interpolation_2df( y(iq(j):iq(j)+1),  &
  &                                                        z(ip(k):ip(k)+1),  &
  &                                                        u(nx,iq(j):iq(j)+1,ip(k):ip(k)+1),  &
  &                                                        (/q(j), p(k)/), v(i,j,k) )
                                else
                                   v(i,j,k)=rdefun
                                end if
                             end if
                          end if
                       else
                          if(iq(j)==ny)then
                             if(ip(k)==nz)then   ! (1,0,0)
                                if(u(ir(i),ny,nz)/=rdefun.and.  &
  &                                u(ir(i)+1,ny,nz)/=rdefun)then
                                   call interpolation_1df( x(ir(i):ir(i)+1),  &
  &                                                        u(ir(i):ir(i)+1,ny,nz),  &
  &                                                        r(i), v(i,j,k) )
                                else
                                   v(i,j,k)=rdefun
                                end if
                             else               ! (1,0,1)
                                if(u(ir(i),ny,ip(k))/=rdefun.and.  &
  &                                u(ir(i)+1,ny,ip(k))/=rdefun.and.  &
  &                                u(ir(i),ny,ip(k)+1)/=rdefun.and.  &
  &                                u(ir(i)+1,ny,ip(k)+1)/=rdefun)then
                                   call interpolation_2df( x(ir(i):ir(i)+1),  &
  &                                                        z(ip(k):ip(k)+1),  &
  &                                                        u(ir(i):ir(i)+1,ny,ip(k):ip(k)+1),  &
  &                                                        (/r(i), p(k)/), v(i,j,k) )
                                else
                                   v(i,j,k)=rdefun
                                end if
                             end if
                          else
                             if(ip(k)==nz)then   ! (1,1,0)
                                if(u(ir(i),iq(j),nz)/=rdefun.and.  &
  &                                u(ir(i)+1,iq(j),nz)/=rdefun.and.  &
  &                                u(ir(i),iq(j)+1,nz)/=rdefun.and.  &
  &                                u(ir(i)+1,iq(j)+1,nz)/=rdefun)then
                                   call interpolation_2df( x(ir(i):ir(i)+1),  &
  &                                                        y(iq(j):iq(j)+1),  &
  &                                                        u(ir(i):ir(i)+1,iq(j):iq(j)+1,nz),  &
  &                                                        (/r(i), q(j)/), v(i,j,k) )
                                else
                                   v(i,j,k)=rdefun
                                end if
                             end if
                          end if
                       end if
                    else
                       v(i,j,k)=defun
                    end if
                 end if
              else
                 v(i,j,k)=rdefun
              end if
           else
              if(extra.eqv..true.)then    ! 以下, 上位の if 文から (1,1,1) はない.
                 if(ir(i)==0)then
                    if(iq(j)==0)then
                       if(ip(k)==0)then   ! (0,0,0)
                          v(i,j,k)=u(1,1,1)
                       else               ! (0,0,1)
                          if(u(1,1,ip(k))/=rdefun.and.  &
  &                          u(1,1,ip(k)+1)/=rdefun)then
                             call interpolation_1df( z(ip(k):ip(k)+1),  &
  &                                                  u(1,1,ip(k):ip(k)+1),  &
  &                                                  p(k), v(i,j,k) )
                          else
                             v(i,j,k)=rdefun
                          end if
                       end if
                    else
                       if(ip(k)==0)then   ! (0,1,0)
                          if(u(1,iq(j),1)/=rdefun.and.  &
  &                          u(1,iq(j)+1,1)/=rdefun)then
                             call interpolation_1df( y(iq(j):iq(j)+1),  &
  &                                                  u(1,iq(j):iq(j)+1,1),  &
  &                                                  q(j), v(i,j,k) )
                          else
                             v(i,j,k)=rdefun
                          end if
                       else               ! (0,1,1)
                          if(u(1,iq(j),ip(k))/=rdefun.and.  &
  &                          u(1,iq(j)+1,ip(k))/=rdefun.and.  &
  &                          u(1,iq(j),ip(k)+1)/=rdefun.and.  &
  &                          u(1,iq(j)+1,ip(k)+1)/=rdefun)then
                             call interpolation_2df( y(iq(j):iq(j)+1),  &
  &                                                  z(ip(k):ip(k)+1),  &
  &                                                  u(1,iq(j):iq(j)+1,ip(k):ip(k)+1),  &
  &                                                  (/q(j), p(k)/), v(i,j,k) )
                          else
                             v(i,j,k)=rdefun
                          end if
                       end if
                    end if
                 else
                    if(iq(j)==0)then
                       if(ip(k)==0)then   ! (1,0,0)
                          if(u(ir(i),1,1)/=rdefun.and.  &
  &                          u(ir(i)+1,1,1)/=rdefun)then
                             call interpolation_1df( x(ir(i):ir(i)+1),  &
  &                                                  u(ir(i):ir(i)+1,1,1),  &
  &                                                  r(i), v(i,j,k) )
                          else
                             v(i,j,k)=rdefun
                          end if
                       else               ! (1,0,1)
                          if(u(ir(i),1,ip(k))/=rdefun.and.  &
  &                          u(ir(i)+1,1,ip(k))/=rdefun.and.  &
  &                          u(ir(i),1,ip(k)+1)/=rdefun.and.  &
  &                          u(ir(i)+1,1,ip(k)+1)/=rdefun)then
                             call interpolation_2df( x(ir(i):ir(i)+1),  &
  &                                                  z(ip(k):ip(k)+1),  &
  &                                                  u(ir(i):ir(i)+1,1,ip(k):ip(k)+1),  &
  &                                                  (/r(i), p(k)/), v(i,j,k) )
                          else
                             v(i,j,k)=rdefun
                          end if
                       end if
                    else
                       if(ip(k)==0)then   ! (1,1,0)
                          if(u(ir(i),iq(j),1)/=rdefun.and.  &
  &                          u(ir(i)+1,iq(j),1)/=rdefun.and.  &
  &                          u(ir(i),iq(j)+1,1)/=rdefun.and.  &
  &                          u(ir(i)+1,iq(j)+1,1)/=rdefun)then
                             call interpolation_2df( x(ir(i):ir(i)+1),  &
  &                                                  y(iq(j):iq(j)+1),  &
  &                                                  u(ir(i):ir(i)+1,iq(j):iq(j)+1,1),  &
  &                                                  (/r(i), q(j)/), v(i,j,k) )
                          else
                             v(i,j,k)=rdefun
                          end if
                       end if
                    end if
                 end if
              else
                 v(i,j,k)=defun
              end if
           end if
        end do
     end do
  end do

!$omp end do
!$omp end parallel

end subroutine auto_interpolation_3df

!------------------------------------
!------------------------------------

subroutine auto_interpolation_3dd( x, y, z, r, q, p, u, v, undef, undefr,  &
  &                                stdopt, extopt )
  ! 座標 x, y, z で定義されているデータ u を
  ! 座標 r, q, p で定義されるデータ v に自動で内挿する.
  implicit none
  double precision, intent(in) :: x(:)  ! 元座標 1
  double precision, intent(in) :: y(:)  ! 元座標 2
  double precision, intent(in) :: z(:)  ! 元座標 3
  double precision, intent(in) :: r(:)  ! 内挿座標 1
  double precision, intent(in) :: q(:)  ! 内挿座標 2
  double precision, intent(in) :: p(:)  ! 内挿座標 3
  double precision, intent(in) :: u(size(x),size(y),size(z))  ! 元データ
  double precision, intent(inout) :: v(size(r),size(q),size(p))  ! 内挿したデータ
  double precision, intent(in), optional :: undef  ! 未定義値
  double precision, intent(in), optional :: undefr  ! 内挿領域内での未定義値.
        ! 内挿点の隣接点が undefr なら, 内挿点も undefr が代入される
  logical, intent(in), optional :: stdopt  ! 探索範囲が見つからない旨の標準出力を表示させないようにする.
  logical, intent(in), optional :: extopt  ! 探索範囲外の場合, データのある最近接点の値をコピーする.
                                           ! デフォルトではコピーしない .false.
  integer :: ir(size(r)), iq(size(q)), ip(size(p))
  integer :: i, j, k, nx, ny, nz, nr, nq, np
  double precision :: defun
  double precision :: rdefun
  logical :: stderr, extra

  nx=size(x)
  ny=size(y)
  nz=size(z)
  nr=size(r)
  nq=size(q)
  np=size(p)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, u ),  &
  &                                     "auto_interpolation_3d" )
     call check_array_size_dmp_message( check_array_size_3d( nr, nq, np, v ),  &
  &                                     "auto_interpolation_3d" )
  end if

  if(present(undef))then
     defun=undef
  else
     defun=-999.0d0
  end if

  if(present(undefr))then
     rdefun=undefr
  else
     rdefun=-999.0d0
  end if

  if(present(stdopt))then
     stderr=stdopt
  else
     stderr=.false.
  end if

  if(present(extopt))then
     extra=extopt
  else
     extra=.false.
  end if

  call auto_interpo_search_3dd( x, y, z, r, q, p, ir, iq, ip,  &
  &                             undeff=0, stdopt=stderr )

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,k)

  do k=1, np
     do j=1, nq
        do i=1, nr
           if(ir(i)/=0.and.iq(j)/=0.and.ip(k)/=0)then
              if(u(ir(i),iq(j),ip(k))/=rdefun)then
                 if(ir(i)<nx.and.iq(j)<ny.and.ip(k)<nz)then
                    if(u(ir(i)+1,iq(j),ip(k))/=rdefun.and.  &
  &                    u(ir(i),iq(j)+1,ip(k))/=rdefun.and.  &
  &                    u(ir(i)+1,iq(j)+1,ip(k))/=rdefun.and.  &
  &                    u(ir(i),iq(j),ip(k)+1)/=rdefun.and.  &
  &                    u(ir(i)+1,iq(j),ip(k)+1)/=rdefun.and.  &
  &                    u(ir(i),iq(j)+1,ip(k)+1)/=rdefun.and.  &
  &                    u(ir(i)+1,iq(j)+1,ip(k)+1)/=rdefun)then
                       call interpolation_3dd( x(ir(i):ir(i)+1),  &
  &                                            y(iq(j):iq(j)+1),  &
  &                                            z(ip(k):ip(k)+1),  &
  &                                            u(ir(i):ir(i)+1,iq(j):iq(j)+1,ip(k):ip(k)+1),  &
  &                                            (/r(i), q(j), p(k)/), v(i,j,k) )
                    else
                       v(i,j,k)=rdefun
                    end if

                 else if(x(nx)==r(i).and.y(ny)==q(j).and.z(nz)==p(k))then
                    v(i,j,k)=u(ir(i),iq(j),ip(k))

                 else if(x(nx)==r(i).and.iq(j)<ny.and.ip(k)<nz)then
                    if(u(nx,iq(j)+1,ip(k))/=rdefun.and.u(nx,iq(j),ip(k)+1)/=rdefun.and.  &
  &                    u(nx,iq(j)+1,ip(k)+1)/=rdefun)then
                       call interpolation_2dd( y(iq(j):iq(j)+1),  &
  &                                            z(ip(k):ip(k)+1),  &
  &                                            u(nx,iq(j):iq(j)+1,ip(k):ip(k)+1),  &
  &                                            (/q(j), p(k)/), v(i,j,k) )
                    else
                       v(i,j,k)=rdefun
                    end if

                 else if(y(ny)==q(j).and.ir(i)<nx.and.ip(k)<nz)then
                    if(u(ir(i)+1,ny,ip(k))/=rdefun.and.u(ir(i),ny,ip(k)+1)/=rdefun.and.  &
  &                    u(ir(i)+1,ny,ip(k)+1)/=rdefun)then
                       call interpolation_2dd( x(ir(i):ir(i)+1),  &
  &                                            z(ip(k):ip(k)+1),  &
  &                                            u(ir(i):ir(i)+1,ny,ip(k):ip(k)+1),  &
  &                                            (/r(i), p(k)/), v(i,j,k) )
                    else
                       v(i,j,k)=rdefun
                    end if

                 else if(z(nz)==p(k).and.ir(i)<nx.and.iq(j)<ny)then
                    if(u(ir(i)+1,iq(j),nz)/=rdefun.and.u(ir(i),iq(j)+1,nz)/=rdefun.and.  &
  &                    u(ir(i)+1,iq(j)+1,nz)/=rdefun)then
                       call interpolation_2dd( x(ir(i):ir(i)+1),  &
  &                                            y(iq(j):iq(j)+1),  &
  &                                            u(ir(i):ir(i)+1,iq(j):iq(j)+1,nz),  &
  &                                            (/r(i), q(j)/), v(i,j,k) )
                    else
                       v(i,j,k)=rdefun
                    end if

                 else if(x(nx)==r(i).and.y(ny)==q(j).and.ip(k)<nz)then
                    if(u(nx,ny,ip(k)+1)/=rdefun)then
                       call interpolation_1dd( z(ip(k):ip(k)+1),  &
  &                                            u(nx,ny,ip(k):ip(k)+1),  &
  &                                            p(k), v(i,j,k) )
                    else
                       v(i,j,k)=rdefun
                    end if

                 else if(x(nx)==r(i).and.z(nz)==p(k).and.iq(j)<ny)then
                    if(u(nx,iq(j)+1,nz)/=rdefun)then
                       call interpolation_1dd( y(iq(j):iq(j)+1),  &
  &                                            u(nx,iq(j):iq(j)+1,nz),  &
  &                                            q(j), v(i,j,k) )
                    else
                       v(i,j,k)=rdefun
                    end if

                 else if(y(ny)==q(j).and.z(nz)==p(k).and.ir(i)<nx)then
                    if(u(ir(i)+1,ny,nz)/=rdefun)then
                       call interpolation_1dd( x(ir(i):ir(i)+1),  &
  &                                            u(ir(i):ir(i)+1,ny,nz),  &
  &                                            r(i), v(i,j,k) )
                    else
                       v(i,j,k)=rdefun
                    end if

                 else

                    if(extra.eqv..true.)then    ! 以下, 上位の if 文から (1,1,1) はない.
                       if(ir(i)==nx)then
                          if(iq(j)==ny)then
                             if(ip(k)==nz)then   ! (0,0,0)
                                v(i,j,k)=u(nx,ny,nz)
                             else               ! (0,0,1)
                                if(u(nx,ny,ip(k))/=rdefun.and.  &
  &                                u(nx,ny,ip(k)+1)/=rdefun)then
                                   call interpolation_1dd( z(ip(k):ip(k)+1),  &
  &                                                        u(nx,ny,ip(k):ip(k)+1),  &
  &                                                        p(k), v(i,j,k) )
                                else
                                   v(i,j,k)=rdefun
                                end if
                             end if
                          else
                             if(ip(k)==nz)then   ! (0,1,0)
                                if(u(nx,iq(j),nz)/=rdefun.and.  &
  &                                u(nx,iq(j)+1,nz)/=rdefun)then
                                   call interpolation_1dd( y(iq(j):iq(j)+1),  &
  &                                                        u(nx,iq(j):iq(j)+1,nz),  &
  &                                                        q(j), v(i,j,k) )
                                else
                                   v(i,j,k)=rdefun
                                end if
                             else               ! (0,1,1)
                                if(u(nx,iq(j),ip(k))/=rdefun.and.  &
  &                                u(nx,iq(j)+1,ip(k))/=rdefun.and.  &
  &                                u(nx,iq(j),ip(k)+1)/=rdefun.and.  &
  &                                u(nx,iq(j)+1,ip(k)+1)/=rdefun)then
                                   call interpolation_2dd( y(iq(j):iq(j)+1),  &
  &                                                        z(ip(k):ip(k)+1),  &
  &                                                        u(nx,iq(j):iq(j)+1,ip(k):ip(k)+1),  &
  &                                                        (/q(j), p(k)/), v(i,j,k) )
                                else
                                   v(i,j,k)=rdefun
                                end if
                             end if
                          end if
                       else
                          if(iq(j)==ny)then
                             if(ip(k)==nz)then   ! (1,0,0)
                                if(u(ir(i),ny,nz)/=rdefun.and.  &
  &                                u(ir(i)+1,ny,nz)/=rdefun)then
                                   call interpolation_1dd( x(ir(i):ir(i)+1),  &
  &                                                        u(ir(i):ir(i)+1,ny,nz),  &
  &                                                        r(i), v(i,j,k) )
                                else
                                   v(i,j,k)=rdefun
                                end if
                             else               ! (1,0,1)
                                if(u(ir(i),ny,ip(k))/=rdefun.and.  &
  &                                u(ir(i)+1,ny,ip(k))/=rdefun.and.  &
  &                                u(ir(i),ny,ip(k)+1)/=rdefun.and.  &
  &                                u(ir(i)+1,ny,ip(k)+1)/=rdefun)then
                                   call interpolation_2dd( x(ir(i):ir(i)+1),  &
  &                                                        z(ip(k):ip(k)+1),  &
  &                                                        u(ir(i):ir(i)+1,ny,ip(k):ip(k)+1),  &
  &                                                        (/r(i), p(k)/), v(i,j,k) )
                                else
                                   v(i,j,k)=rdefun
                                end if
                             end if
                          else
                             if(ip(k)==nz)then   ! (1,1,0)
                                if(u(ir(i),iq(j),nz)/=rdefun.and.  &
  &                                u(ir(i)+1,iq(j),nz)/=rdefun.and.  &
  &                                u(ir(i),iq(j)+1,nz)/=rdefun.and.  &
  &                                u(ir(i)+1,iq(j)+1,nz)/=rdefun)then
                                   call interpolation_2dd( x(ir(i):ir(i)+1),  &
  &                                                        y(iq(j):iq(j)+1),  &
  &                                                        u(ir(i):ir(i)+1,iq(j):iq(j)+1,nz),  &
  &                                                        (/r(i), q(j)/), v(i,j,k) )
                                else
                                   v(i,j,k)=rdefun
                                end if
                             end if
                          end if
                       end if
                    else
                       v(i,j,k)=defun
                    end if
                 end if
              else
                 v(i,j,k)=rdefun
              end if
           else
              if(extra.eqv..true.)then    ! 以下, 上位の if 文から (1,1,1) はない.
                 if(ir(i)==0)then
                    if(iq(j)==0)then
                       if(ip(k)==0)then   ! (0,0,0)
                          v(i,j,k)=u(1,1,1)
                       else               ! (0,0,1)
                          if(u(1,1,ip(k))/=rdefun.and.  &
  &                          u(1,1,ip(k)+1)/=rdefun)then
                             call interpolation_1dd( z(ip(k):ip(k)+1),  &
  &                                                  u(1,1,ip(k):ip(k)+1),  &
  &                                                  p(k), v(i,j,k) )
                          else
                             v(i,j,k)=rdefun
                          end if
                       end if
                    else
                       if(ip(k)==0)then   ! (0,1,0)
                          if(u(1,iq(j),1)/=rdefun.and.  &
  &                          u(1,iq(j)+1,1)/=rdefun)then
                             call interpolation_1dd( y(iq(j):iq(j)+1),  &
  &                                                  u(1,iq(j):iq(j)+1,1),  &
  &                                                  q(j), v(i,j,k) )
                          else
                             v(i,j,k)=rdefun
                          end if
                       else               ! (0,1,1)
                          if(u(1,iq(j),ip(k))/=rdefun.and.  &
  &                          u(1,iq(j)+1,ip(k))/=rdefun.and.  &
  &                          u(1,iq(j),ip(k)+1)/=rdefun.and.  &
  &                          u(1,iq(j)+1,ip(k)+1)/=rdefun)then
                             call interpolation_2dd( y(iq(j):iq(j)+1),  &
  &                                                  z(ip(k):ip(k)+1),  &
  &                                                  u(1,iq(j):iq(j)+1,ip(k):ip(k)+1),  &
  &                                                  (/q(j), p(k)/), v(i,j,k) )
                          else
                             v(i,j,k)=rdefun
                          end if
                       end if
                    end if
                 else
                    if(iq(j)==0)then
                       if(ip(k)==0)then   ! (1,0,0)
                          if(u(ir(i),1,1)/=rdefun.and.  &
  &                          u(ir(i)+1,1,1)/=rdefun)then
                             call interpolation_1dd( x(ir(i):ir(i)+1),  &
  &                                                  u(ir(i):ir(i)+1,1,1),  &
  &                                                  r(i), v(i,j,k) )
                          else
                             v(i,j,k)=rdefun
                          end if
                       else               ! (1,0,1)
                          if(u(ir(i),1,ip(k))/=rdefun.and.  &
  &                          u(ir(i)+1,1,ip(k))/=rdefun.and.  &
  &                          u(ir(i),1,ip(k)+1)/=rdefun.and.  &
  &                          u(ir(i)+1,1,ip(k)+1)/=rdefun)then
                             call interpolation_2dd( x(ir(i):ir(i)+1),  &
  &                                                  z(ip(k):ip(k)+1),  &
  &                                                  u(ir(i):ir(i)+1,1,ip(k):ip(k)+1),  &
  &                                                  (/r(i), p(k)/), v(i,j,k) )
                          else
                             v(i,j,k)=rdefun
                          end if
                       end if
                    else
                       if(ip(k)==0)then   ! (1,1,0)
                          if(u(ir(i),iq(j),1)/=rdefun.and.  &
  &                          u(ir(i)+1,iq(j),1)/=rdefun.and.  &
  &                          u(ir(i),iq(j)+1,1)/=rdefun.and.  &
  &                          u(ir(i)+1,iq(j)+1,1)/=rdefun)then
                             call interpolation_2dd( x(ir(i):ir(i)+1),  &
  &                                                  y(iq(j):iq(j)+1),  &
  &                                                  u(ir(i):ir(i)+1,iq(j):iq(j)+1,1),  &
  &                                                  (/r(i), q(j)/), v(i,j,k) )
                          else
                             v(i,j,k)=rdefun
                          end if
                       end if
                    end if
                 end if
              else
                 v(i,j,k)=defun
              end if
           end if
        end do
     end do
  end do

!$omp end do
!$omp end parallel

end subroutine auto_interpolation_3dd

!------------------------------------
!------------------------------------

subroutine tri_interpolation_f( x, y, val, point, oval )
  ! 三角形の要素内の線形内挿ルーチン
  ! 要素内における内挿点 point(1:2) によって分割される領域面積に対して,
  ! oval = \sum^{3}_{i}(S_(i)*val(i)) / S, 
  ! S = \sum^{3}_{i}(S_(i))
  ! 各面積はベクトルの外積が平行四辺形の面積になるという性質より, 
  ! 三角形の各点と要素点の座標から求める.
  implicit none
  real, intent(in) :: x(3)    ! 三角形の各頂点 x 座標
  real, intent(in) :: y(3)    ! 三角形の各頂点 y 座標
  real, intent(in) :: val(3)  ! 三角形の各頂点での値
  real, intent(in) :: point(2)! 内挿点 point(1)<=x 座標, point(2)<=y 座標
  real, intent(inout) :: oval  ! 内挿点での値
  real :: Stot, S(3), xp, yp

  xp=point(1)
  yp=point(2)
  Stot=(x(2)-x(1))*(y(3)-y(1))-(x(3)-x(1))*(y(2)-y(1))
  S(1)=(x(3)-x(2))*(yp-y(3))-(xp-x(3))*(y(3)-y(2))
  S(2)=(x(1)-x(3))*(yp-y(1))-(xp-x(1))*(y(1)-y(3))
  S(3)=(x(2)-x(1))*(yp-y(2))-(xp-x(2))*(y(2)-y(1))

  oval=(val(1)*S(1)+val(2)*S(2)+val(3)*S(3))/Stot

end subroutine tri_interpolation_f

!------------------------------------
!------------------------------------

subroutine tri_interpolation_d( x, y, val, point, oval )
  ! 三角形の要素内の線形内挿ルーチン
  ! 要素内における内挿点 point(1:2) によって分割される領域面積に対して,
  ! oval = \sum^{3}_{i}(S_(i)*val(i)) / S, 
  ! S = \sum^{3}_{i}(S_(i))
  ! 各面積はベクトルの外積が平行四辺形の面積になるという性質より, 
  ! 三角形の各点と要素点の座標から求める.
  implicit none
  double precision, intent(in) :: x(3)    ! 三角形の各頂点 x 座標
  double precision, intent(in) :: y(3)    ! 三角形の各頂点 y 座標
  double precision, intent(in) :: val(3)  ! 三角形の各頂点での値
  double precision, intent(in) :: point(2)! 内挿点 point(1)<=x 座標, point(2)<=y 座標
  double precision, intent(inout) :: oval  ! 内挿点での値
  double precision :: Stot, S(3), xp, yp

  xp=point(1)
  yp=point(2)
  Stot=(x(2)-x(1))*(y(3)-y(1))-(x(3)-x(1))*(y(2)-y(1))
  S(1)=(x(3)-x(2))*(yp-y(3))-(xp-x(3))*(y(3)-y(2))
  S(2)=(x(1)-x(3))*(yp-y(1))-(xp-x(1))*(y(1)-y(3))
  S(3)=(x(2)-x(1))*(yp-y(2))-(xp-x(2))*(y(2)-y(1))

  oval=(val(1)*S(1)+val(2)*S(2)+val(3)*S(3))/Stot

end subroutine tri_interpolation_d

!------------------------------------
!------------------------------------

subroutine tri_interpolation_2df( x_in, y_in, iv, ivad, x_out, y_out, ov,  &
  &                               ovad, undef, jflag )
!-- 2 次元不等間隔格子点 (x_in,y_in) 上で定義された物理量 iv を
!-- 同じ次元の等間隔格子 (x_out,y_out) に
!-- 三角点補間 (tri_interpolation) するルーチン (補間値は ov).
!-- [注意]: x_out, y_out は等間隔でなければならない.
  implicit none
  real, dimension(:,:), intent(in) :: x_in  ! 補間前のオリジナル座標 x 成分
  real, dimension(size(x_in,1),size(x_in,2)), intent(in) :: y_in  ! 補間前のオリジナル座標 y 成分
  real, dimension(size(x_in,1),size(x_in,2)), intent(in) :: iv  ! x_in, y_in で定義された物理量
  real, dimension(size(x_in,1),size(x_in,2)), intent(in) :: ivad  ! iv 以外に追加物理量
  real, dimension(:), intent(in) :: x_out  ! 補間する座標 x 成分 (x_in と同じ単位, 等間隔)
  real, dimension(:), intent(in) :: y_out  ! 補間する座標 y 成分 (y_in と同じ単位, 等間隔)
  real, dimension(size(x_out),size(y_out)), intent(inout) :: ov  ! 補間された値
  real, dimension(size(x_out),size(y_out)), intent(inout) :: ovad  ! ivad の補間された値
  real, intent(in) :: undef
  character(1), intent(in) :: jflag  ! 内挿点の値が更新される基準.
                                     ! 'u' : 値が大きいと更新, 'l' : 値が小さいと更新.

  integer :: k, l, m, ix, jy, icounter
  integer :: nsi, nti, nxo, nyo, ixmin, ixmax, jymin, jymax, itmp
  integer, dimension(2) :: isqr
  real :: intx_out(size(x_out)), inty_out(size(y_out))
  real, dimension(size(x_in,1),size(y_in,2)) :: intx_in, inty_in
  real :: dlon, dlat, x_outmin, y_outmin, ov_tmp, ovad_tmp
  real, dimension(4) :: sqrlon, sqrlat, sqrval, sqrvalad
  real, dimension(3) :: interp_lon, interp_lat, interp_val, interp_valad
  logical :: calc_flag

  nsi=size(x_in,1)
  nti=size(x_in,2)
  nxo=size(x_out)
  nyo=size(y_out)

  dlon=x_out(2)-x_out(1)
  dlat=y_out(2)-y_out(1)
  x_outmin=x_out(1)
  y_outmin=y_out(1)

  ov=undef
  ovad=undef

!-- x_out, y_out を x_out(1) = 1, y_out(1) = 1 として格子点番号に変換

  intx_out=(/((x_out(ix)-x_outmin)/dlon+1.0d0,ix=1,nxo)/)
  inty_out=(/((y_out(jy)-y_outmin)/dlat+1.0d0,jy=1,nyo)/)

!-- x_in, y_in を x_out, y_out 系での格子点番号 (実数) に変換

  intx_in=undef  ! x_in, y_in は undef が入っている (特に領域外側).
  inty_in=undef
  do l=1,nti
     do k=1,nsi
        if(x_in(k,l)/=undef.and.y_in(k,l)/=undef)then
           intx_in(k,l)=(x_in(k,l)-x_outmin)/dlon+1.0d0
           inty_in(k,l)=(y_in(k,l)-y_outmin)/dlat+1.0d0
        end if
     end do
  end do

!-- x_in, y_in を左下から順に三角形分割し, その三角形内に
!-- 1. 標的格子 (x_out, y_out) が含まれているかチェック,
!-- 2. 含まれていれば三角形で線形内挿.

  do l=1,nti-1
     do k=1,nsi-1
        sqrlon=(/intx_in(k,l), intx_in(k+1,l), intx_in(k,l+1), intx_in(k+1,l+1)/)
        sqrlat=(/inty_in(k,l), inty_in(k+1,l), inty_in(k,l+1), inty_in(k+1,l+1)/)
        sqrval=(/iv(k,l), iv(k+1,l), iv(k,l+1), iv(k+1,l+1)/)
        sqrvalad=(/ivad(k,l), ivad(k+1,l), ivad(k,l+1), ivad(k+1,l+1)/)

        !-- 対象とする隣接 4 点が全て未定義でないことの確認
        calc_flag=.true.
        do m=1,4
           if(sqrlon(m)==undef)then
              calc_flag=.false.
              exit
           end if
           if(sqrlat(m)==undef)then
              calc_flag=.false.
              exit
           end if
           if(sqrval(m)==undef)then
              calc_flag=.false.
              exit
           end if
        end do
        if(calc_flag.eqv..false.)then  ! 計算しないなら, cycle で次の k,l へ
           cycle
        end if

        !-- 隣接 4 点から三角形に分けるため, 対角線を特定
        !-- (線分の交点の有無で判断)
        !-- selopt で長い方の対角点を取得している. 
        !-- 対角線の短い三角形を使うので, 後の処理のため長い方の対角点を取得.
        call check_square_intersect( sqrlon, sqrlat, isqr, selopt='l' )

        !-- 分けた三角形で三角形内に内挿点候補があるかのチェック
        do m=1,2  ! 三角形は 2 つある.

           icounter=1
           do itmp=1,4
              if(isqr(m)/=itmp)then  ! 対角点以外で m 番目の三角形を構築
                 interp_lon(icounter)=sqrlon(itmp)
                 interp_lat(icounter)=sqrlat(itmp)
                 interp_val(icounter)=sqrval(itmp)
                 interp_valad(icounter)=sqrvalad(itmp)
                 icounter=icounter+1
              end if
           end do

           ixmin=int(amin1(interp_lon(1),interp_lon(2),interp_lon(3)))-1
           ixmax=int(amax1(interp_lon(1),interp_lon(2),interp_lon(3)))+2
           jymin=int(amin1(interp_lat(1),interp_lat(2),interp_lat(3)))-1
           jymax=int(amax1(interp_lat(1),interp_lat(2),interp_lat(3)))+2

           if(ixmin<1.or.ixmax>nxo.or.jymin<1.or.jymax>nyo)then
              ! ここの範囲は以下の ix, jy ループの端と連動する.
              !-- 内挿点候補がそもそもない. -> 次のループへ (reported by Tsukada).
              exit
           end if

           if(ixmin<=ixmax.and.jymin<=jymax)then  ! 内挿点候補がある場合.
              do jy=jymin,jymax  ! この jymin, jymax, ixmin, ixmax は ov の点
                 do ix=ixmin,ixmax
                    if(check_triclose( interp_lon, interp_lat,  &
  &                              (/intx_out(ix),inty_out(jy)/) ).eqv..true.)then
                    ! 実際に内挿点がある場合.
                       call tri_interpolation( interp_lon, interp_lat, interp_val,  &
  &                                             (/intx_out(ix), inty_out(jy)/), ov_tmp )
                       call tri_interpolation( interp_lon, interp_lat, interp_valad,  &
  &                                             (/intx_out(ix), inty_out(jy)/), ovad_tmp )
                       if(ov(ix,jy)/=undef)then
                          if(jflag(1:1)=='l'.and.(ov(ix,jy)>ov_tmp))then  ! ov_tmp の方が小さいとき更新.
                             ov(ix,jy)=ov_tmp
                             ovad(ix,jy)=ovad_tmp
                          else if(jflag(1:1)=='u'.and.(ov(ix,jy)<ov_tmp))then  ! ov_tmp の方が大きいとき更新.
                             ov(ix,jy)=ov_tmp
                             ovad(ix,jy)=ovad_tmp
                          end if
                       else
                          ov(ix,jy)=ov_tmp
                          ovad(ix,jy)=ovad_tmp
                       end if
                    end if
                 end do
              end do
           end if

        end do

     end do
  end do

end subroutine tri_interpolation_2df

!------------------------------------
!------------------------------------

subroutine tri_interpolation_2dd( x_in, y_in, iv, ivad, x_out, y_out, ov,  &
  &                               ovad, undef, jflag )
!-- 2 次元不等間隔格子点 (x_in,y_in) 上で定義された物理量 iv を
!-- 同じ次元の等間隔格子 (x_out,y_out) に
!-- 三角点補間 (tri_interpolation) するルーチン (補間値は ov).
!-- [注意]: x_out, y_out は等間隔でなければならない.
  implicit none
  double precision, dimension(:,:), intent(in) :: x_in  ! 補間前のオリジナル座標 x 成分
  double precision, dimension(size(x_in,1),size(x_in,2)), intent(in) :: y_in  ! 補間前のオリジナル座標 y 成分
  double precision, dimension(size(x_in,1),size(x_in,2)), intent(in) :: iv  ! x_in, y_in で定義された物理量
  double precision, dimension(size(x_in,1),size(x_in,2)), intent(in) :: ivad  ! iv 以外に追加物理量
  double precision, dimension(:), intent(in) :: x_out  ! 補間する座標 x 成分 (x_in と同じ単位, 等間隔)
  double precision, dimension(:), intent(in) :: y_out  ! 補間する座標 y 成分 (y_in と同じ単位, 等間隔)
  double precision, dimension(size(x_out),size(y_out)), intent(inout) :: ov  ! 補間された値
  double precision, dimension(size(x_out),size(y_out)), intent(inout) :: ovad  ! ivad の補間された値
  double precision, intent(in) :: undef
  character(1), intent(in) :: jflag  ! 内挿点の値が更新される基準.
                                     ! 'u' : 値が大きいと更新, 'l' : 値が小さいと更新.

  integer :: k, l, m, ix, jy, icounter
  integer :: nsi, nti, nxo, nyo, ixmin, ixmax, jymin, jymax, itmp
  integer, dimension(2) :: isqr
  double precision :: intx_out(size(x_out)), inty_out(size(y_out))
  double precision, dimension(size(x_in,1),size(y_in,2)) :: intx_in, inty_in
  double precision :: dlon, dlat, x_outmin, y_outmin, ov_tmp, ovad_tmp
  double precision, dimension(4) :: sqrlon, sqrlat, sqrval, sqrvalad
  double precision, dimension(3) :: interp_lon, interp_lat, interp_val, interp_valad
  logical :: calc_flag

  nsi=size(x_in,1)
  nti=size(x_in,2)
  nxo=size(x_out)
  nyo=size(y_out)

  dlon=x_out(2)-x_out(1)
  dlat=y_out(2)-y_out(1)
  x_outmin=x_out(1)
  y_outmin=y_out(1)

  ov=undef
  ovad=undef

!-- x_out, y_out を x_out(1) = 1, y_out(1) = 1 として格子点番号に変換

  intx_out=(/((x_out(ix)-x_outmin)/dlon+1.0d0,ix=1,nxo)/)
  inty_out=(/((y_out(jy)-y_outmin)/dlat+1.0d0,jy=1,nyo)/)

!-- x_in, y_in を x_out, y_out 系での格子点番号 (実数) に変換

  intx_in=undef  ! x_in, y_in は undef が入っている (特に領域外側).
  inty_in=undef
  do l=1,nti
     do k=1,nsi
        if(x_in(k,l)/=undef.and.y_in(k,l)/=undef)then
           intx_in(k,l)=(x_in(k,l)-x_outmin)/dlon+1.0d0
           inty_in(k,l)=(y_in(k,l)-y_outmin)/dlat+1.0d0
        end if
     end do
  end do

!-- x_in, y_in を左下から順に三角形分割し, その三角形内に
!-- 1. 標的格子 (x_out, y_out) が含まれているかチェック,
!-- 2. 含まれていれば三角形で線形内挿.

  do l=1,nti-1
     do k=1,nsi-1
        sqrlon=(/intx_in(k,l), intx_in(k+1,l), intx_in(k,l+1), intx_in(k+1,l+1)/)
        sqrlat=(/inty_in(k,l), inty_in(k+1,l), inty_in(k,l+1), inty_in(k+1,l+1)/)
        sqrval=(/iv(k,l), iv(k+1,l), iv(k,l+1), iv(k+1,l+1)/)
        sqrvalad=(/ivad(k,l), ivad(k+1,l), ivad(k,l+1), ivad(k+1,l+1)/)

        !-- 対象とする隣接 4 点が全て未定義でないことの確認
        calc_flag=.true.
        do m=1,4
           if(sqrlon(m)==undef)then
              calc_flag=.false.
              exit
           end if
           if(sqrlat(m)==undef)then
              calc_flag=.false.
              exit
           end if
           if(sqrval(m)==undef)then
              calc_flag=.false.
              exit
           end if
        end do
        if(calc_flag.eqv..false.)then  ! 計算しないなら, cycle で次の k,l へ
           cycle
        end if

        !-- 隣接 4 点から三角形に分けるため, 対角線を特定
        !-- (線分の交点の有無で判断)
        !-- selopt で長い方の対角点を取得している. 
        !-- 対角線の短い三角形を使うので, 後の処理のため長い方の対角点を取得.
        call check_square_intersect( sqrlon, sqrlat, isqr, selopt='l' )

        !-- 分けた三角形で三角形内に内挿点候補があるかのチェック
        do m=1,2  ! 三角形は 2 つある.

           icounter=1
           do itmp=1,4
              if(isqr(m)/=itmp)then  ! 対角点以外で m 番目の三角形を構築
                 interp_lon(icounter)=sqrlon(itmp)
                 interp_lat(icounter)=sqrlat(itmp)
                 interp_val(icounter)=sqrval(itmp)
                 interp_valad(icounter)=sqrvalad(itmp)
                 icounter=icounter+1
              end if
           end do

           ixmin=idint(dmin1(interp_lon(1),interp_lon(2),interp_lon(3)))-1
           ixmax=idint(dmax1(interp_lon(1),interp_lon(2),interp_lon(3)))+2
           jymin=idint(dmin1(interp_lat(1),interp_lat(2),interp_lat(3)))-1
           jymax=idint(dmax1(interp_lat(1),interp_lat(2),interp_lat(3)))+2

           if(ixmin<1.or.ixmax>nxo.or.jymin<1.or.jymax>nyo)then
              ! ここの範囲は以下の ix, jy ループの端と連動する.
              !-- 内挿点候補がそもそもない. -> 次のループへ (reported by Tsukada).
              exit
           end if

           if(ixmin<=ixmax.and.jymin<=jymax)then  ! 内挿点候補がある場合.
              do jy=jymin,jymax  ! この jymin, jymax, ixmin, ixmax は ov の点
                 do ix=ixmin,ixmax
                    if(check_triclose( interp_lon, interp_lat,  &
  &                              (/intx_out(ix),inty_out(jy)/) ).eqv..true.)then
                    ! 実際に内挿点がある場合.
                       call tri_interpolation( interp_lon, interp_lat, interp_val,  &
  &                                            (/intx_out(ix), inty_out(jy)/), ov_tmp )
                       call tri_interpolation( interp_lon, interp_lat, interp_valad,  &
  &                                            (/intx_out(ix), inty_out(jy)/), ovad_tmp )
                       if(ov(ix,jy)/=undef)then
                          if(jflag(1:1)=='l'.and.(ov(ix,jy)>ov_tmp))then  ! ov_tmp の方が小さいとき更新.
                             ov(ix,jy)=ov_tmp
                             ovad(ix,jy)=ovad_tmp
                          else if(jflag(1:1)=='u'.and.(ov(ix,jy)<ov_tmp))then  ! ov_tmp の方が大きいとき更新.
                             ov(ix,jy)=ov_tmp
                             ovad(ix,jy)=ovad_tmp
                          end if
                       else
                          ov(ix,jy)=ov_tmp
                          ovad(ix,jy)=ovad_tmp
                       end if
                    end if
                 end do
              end do
           end if

        end do

     end do
  end do

end subroutine tri_interpolation_2dd

!------------------------------------
!------------------------------------

subroutine LSM_1df( x, y, slope, intercept, undef )  ! 最小二乗法による傾きと切片計算 (1 次元データ版)
  implicit none
  real, intent(in) :: x(:)  ! データ要素 1
  real, intent(in) :: y(size(x))  ! データ要素 2
  real, intent(inout) :: slope  ! 最適な傾き
  real, intent(inout) :: intercept  ! 最適な切片
  real, intent(in), optional :: undef ! undef 
  real :: u(size(x)), v(size(x)), tmpx(size(x)), tmpy(size(y))
  integer :: i
  integer :: nx, nc  ! データ数
  real :: a, b, c, d

  nx=size(x)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_1d( nx, y ),  &
  &                                     "LSM_1d" )
  end if

  a=0.0
  b=0.0
  c=0.0
  d=0.0

  if(present(undef))then

     u=undef
     v=undef
     tmpx=undef
     tmpy=undef

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i)
     do i=1,nx
        if(x(i)/=undef.and.y(i)/=undef)then
           u(i)=x(i)*x(i)
           v(i)=x(i)*y(i)
           tmpx(i)=x(i)
           tmpy(i)=y(i)
        end if
     end do
!$omp end do
!$omp end parallel

     call summf(v,a,undeff=undef,nc=nc)
     call summf(tmpx,b,undeff=undef,nc=nc)
     call summf(tmpy,c,undeff=undef,nc=nc)
     call summf(u,d,undeff=undef,nc=nc)

  else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i)
     do i=1,nx
        u(i)=x(i)*x(i)
        v(i)=x(i)*y(i)
     end do
!$omp end do
!$omp end parallel

     call summf(v,a)
     call summf(x,b)
     call summf(y,c)
     call summf(u,d)

     nc=nx

  end if

  slope=(nc*a-b*c)/(nc*d-b**2)
  intercept=(c*d-a*b)/(nc*d-b**2)

end subroutine LSM_1df

!------------------------------------
!------------------------------------

subroutine LSM_1dd( x, y, slope, intercept, undef )  ! 最小二乗法による傾きと切片計算 (1 次元データ版)
  implicit none
  double precision, intent(in) :: x(:)  ! データ要素 1
  double precision, intent(in) :: y(size(x))  ! データ要素 2
  double precision, intent(inout) :: slope  ! 最適な傾き
  double precision, intent(inout) :: intercept  ! 最適な切片
  double precision, intent(in), optional :: undef ! undef 
  double precision :: u(size(x)), v(size(x)), tmpx(size(x)), tmpy(size(y))
  integer :: i
  integer :: nx, nc  ! データ数
  double precision :: a, b, c, d

  nx=size(x)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_1d( nx, y ),  &
  &                                     "LSM_1d" )
  end if

  a=0.0d0
  b=0.0d0
  c=0.0d0
  d=0.0d0

  if(present(undef))then

     u=undef
     v=undef
     tmpx=undef
     tmpy=undef

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i)
     do i=1,nx
        if(x(i)/=undef.and.y(i)/=undef)then
           u(i)=x(i)*x(i)
           v(i)=x(i)*y(i)
           tmpx(i)=x(i)
           tmpy(i)=y(i)
        end if
     end do
!$omp end do
!$omp end parallel

     call summd(v,a,undeff=undef,nc=nc)
     call summd(tmpx,b,undeff=undef,nc=nc)
     call summd(tmpy,c,undeff=undef,nc=nc)
     call summd(u,d,undeff=undef,nc=nc)

  else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i)
     do i=1,nx
        u(i)=x(i)*x(i)
        v(i)=x(i)*y(i)
     end do
!$omp end do
!$omp end parallel

     call summd(v,a)
     call summd(x,b)
     call summd(y,c)
     call summd(u,d)

     nc=nx

  end if

  slope=(nc*a-b*c)/(nc*d-b**2)
  intercept=(c*d-a*b)/(nc*d-b**2)

end subroutine LSM_1dd

!------------------------------------
!------------------------------------

subroutine LSM_2df( x, y, slope, intercept, undef )  ! 最小二乗法による傾きと切片計算 (2 次元データ版)
  implicit none
  real, intent(in) :: x(:,:)  ! データ要素 1
  real, intent(in) :: y(size(x,1),size(x,2))  ! データ要素 2
  real, intent(inout) :: slope  ! 最適な傾き
  real, intent(inout) :: intercept  ! 最適な切片
  real, intent(in), optional :: undef ! undef 
  real :: u(size(x,1)*size(x,2)), v(size(x,1)*size(x,2))
  integer :: i, j, counter
  integer :: nx  ! データ数 1
  integer :: ny  ! データ数 2

  nx=size(x,1)
  ny=size(x,2)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, y ),  &
  &                                     "LSM_2d" )
  end if

  counter=0

  do j=1,ny
     do i=1,nx
        counter=counter+1
        u(counter)=x(i,j)
        v(counter)=y(i,j)
     end do
  end do

  if(present(undef))then
     call LSM_1df( u, v, slope, intercept, undef )
  else
     call LSM_1df( u, v, slope, intercept )
  end if

end subroutine LSM_2df

!------------------------------------
!------------------------------------

subroutine LSM_2dd( x, y, slope, intercept, undef )  ! 最小二乗法による傾きと切片計算 (2 次元データ版)
  implicit none
  double precision, intent(in) :: x(:,:)  ! データ要素 1
  double precision, intent(in) :: y(size(x,1),size(x,2))  ! データ要素 2
  double precision, intent(inout) :: slope  ! 最適な傾き
  double precision, intent(inout) :: intercept  ! 最適な切片
  double precision, intent(in), optional :: undef ! undef 
  double precision :: u(size(x,1)*size(x,2)), v(size(x,1)*size(x,2))
  integer :: i, j, counter
  integer :: nx  ! データ数 1
  integer :: ny  ! データ数 2

  nx=size(x,1)
  ny=size(x,2)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, y ),  &
  &                                     "LSM_2d" )
  end if

  counter=0

  do j=1,ny
     do i=1,nx
        counter=counter+1
        u(counter)=x(i,j)
        v(counter)=y(i,j)
     end do
  end do

  if(present(undef))then
     call LSM_1dd( u, v, slope, intercept, undef )
  else
     call LSM_1dd( u, v, slope, intercept )
  end if

end subroutine LSM_2dd

!------------------------------------
!------------------------------------

subroutine LSM_3df( x, y, slope, intercept, undef )  ! 最小二乗法による傾きと切片計算 (3 次元データ版)
  implicit none
  real, intent(in) :: x(:,:,:)  ! データ要素 1
  real, intent(in) :: y(size(x,1),size(x,2),size(x,3))  ! データ要素 2
  real, intent(inout) :: slope  ! 最適な傾き
  real, intent(inout) :: intercept  ! 最適な切片
  real, intent(in), optional :: undef ! undef 
  real :: u(size(x,1)*size(x,2)*size(x,3)), v(size(x,1)*size(x,2)*size(x,3))
  integer :: i, j, k, counter
  integer :: nx  ! データ数 1
  integer :: ny  ! データ数 2
  integer :: nz  ! データ数 3

  nx=size(x,1)
  ny=size(x,2)
  nz=size(x,3)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, y ),  &
  &                                     "LSM_3d" )
  end if

  counter=0

  do k=1,nz
     do j=1,ny
        do i=1,nx
           counter=counter+1
           u(counter)=x(i,j,k)
           v(counter)=y(i,j,k)
        end do
     end do
  end do

  if(present(undef))then
     call LSM_1df( u, v, slope, intercept, undef )
  else
     call LSM_1df( u, v, slope, intercept )
  end if

end subroutine LSM_3df

!------------------------------------
!------------------------------------

subroutine LSM_3dd( x, y, slope, intercept, undef )  ! 最小二乗法による傾きと切片計算 (3 次元データ版)
  implicit none
  double precision, intent(in) :: x(:,:,:)  ! データ要素 1
  double precision, intent(in) :: y(size(x,1),size(x,2),size(x,3))  ! データ要素 2
  double precision, intent(inout) :: slope  ! 最適な傾き
  double precision, intent(inout) :: intercept  ! 最適な切片
  double precision, intent(in), optional :: undef ! undef 
  double precision :: u(size(x,1)*size(x,2)*size(x,3)), v(size(x,1)*size(x,2)*size(x,3))
  integer :: i, j, k, counter
  integer :: nx  ! データ数 1
  integer :: ny  ! データ数 2
  integer :: nz  ! データ数 3

  nx=size(x,1)
  ny=size(x,2)
  nz=size(x,3)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, y ),  &
  &                                     "LSM_3d" )
  end if

  counter=0

  do k=1,nz
     do j=1,ny
        do i=1,nx
           counter=counter+1
           u(counter)=x(i,j,k)
           v(counter)=y(i,j,k)
        end do
     end do
  end do

  if(present(undef))then
     call LSM_1dd( u, v, slope, intercept, undef )
  else
     call LSM_1dd( u, v, slope, intercept )
  end if

end subroutine LSM_3dd

!------------------------------------
!------------------------------------

subroutine LSM_poly_1df( x, y, a, intercept, undef )
! LSM の多項式近似バージョン.
! LSM では, F(x)=a_0+a_1x の直線近似を行っていたが,
! LSM_poly では, F(x)=\sum^{N}_{n=0}{a_nx^n}
! の任意次数の多項式曲線近似を行うことが可能.
! アルゴリズムは最小二乗法を用いており, 係数のソルバには gausss ルーチンを使用.
  implicit none
  real, intent(in) :: x(:)  ! データ要素配列 1
  real, intent(in) :: y(size(x))  ! データ要素配列 2
  real, intent(inout) :: a(:)  ! 多項式の係数
  real, intent(inout) :: intercept  ! y 切片. 
                         ! a に組み込むと引数を渡すとき, poly_n+1 で渡す必要が
                         ! あり, 紛らわしいと判断したため, a_0 である y 切片を
                         ! 独立で引数として渡すことにした.
  real, intent(in), optional :: undef  ! 未定義値.
  integer :: i, j, k
  integer :: nx  ! データの個数
  integer :: poly_n  ! 近似する曲線の最高次数. 1 なら, LSM と同じ.
  real :: coe(0:size(a)), tmpa_coe(0:size(a),0:size(a)), tmpb_coe(0:size(a))
          ! coe は a_n が入る. tmp_coe はデータの総和が入る.
          ! [注意] : 第一要素が行. 第二要素が列.
  real :: tmp(size(x))  ! べき乗計算の一時配列

  nx=size(x)
  poly_n=size(a)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_1d( nx, y ),  &
  &                                     "LSM_poly_1d" )
  end if

!-- gausss に渡しやすいように, 用意した配列に引数を代入.
  if(present(undef))then
     do k=0,poly_n  ! 列成分の計算
        do j=0,poly_n  ! 行成分の計算. 行成分の計算が先に回ることに注意.
           if(j >= k)then  ! 行成分(j)より列成分(k)の要素数が小さい場合, 値を
                           ! まじめに計算する.
              do i=1,nx
                 if(x(i)/=undef)then
                    tmp(i)=x(i)**(j+k)
                 else
                    tmp(i)=undef
                 end if
              end do
              call summf( tmp, tmpa_coe(j,k), undef )
           else  ! 行成分(j)より列成分(k)の要素数が大きい場合, 解く係数行列が
                 ! 対称行列であることから, 値の参照代入のみ行う.
              tmpa_coe(j,k)=tmpa_coe(k,j)  ! 対称成分の代入（すでに計算済み）
           end if
        end do
     end do
     do j=0,poly_n
        do i=1,nx
           if(x(i)/=undef)then
              tmp(i)=y(i)*(x(i)**j)
           else
              tmp(i)=undef
           end if
        end do
        call summf( tmp, tmpb_coe(j), undef )
     end do
  else  ! undef 処理がないとき.
     do k=0,poly_n  ! 列成分の計算
        do j=0,poly_n  ! 行成分の計算. 行成分の計算が先に回ることに注意.
           if(j >= k)then  ! 行成分(j)より列成分(k)の要素数が小さい場合, 値を
                           ! まじめに計算する.
              do i=1,nx
                 tmp(i)=x(i)**(j+k)
              end do
              call summf( tmp, tmpa_coe(j,k), undef )
           else  ! 行成分(j)より列成分(k)の要素数が大きい場合, 解く係数行列が
                 ! 対称行列であることから, 値の参照代入のみ行う.
              tmpa_coe(j,k)=tmpa_coe(k,j)  ! 対称成分の代入（すでに計算済み）
           end if
        end do
     end do
     do j=0,poly_n
        do i=1,nx
           tmp(i)=y(i)*(x(i)**j)
        end do
        call summf( tmp, tmpb_coe(j), undef )
     end do
  end if

!  以上で係数行列に値が入った.

  call gausssf( tmpa_coe(0:poly_n,0:poly_n), tmpb_coe(0:poly_n),  &
  &            coe(0:poly_n) )

  do i=1,poly_n
     a(i)=coe(i)
  end do
  intercept=coe(0)

end subroutine LSM_poly_1df

!------------------------------------
!------------------------------------

subroutine LSM_poly_1dd( x, y, a, intercept, undef )
! LSM の多項式近似バージョン.
! LSM では, F(x)=a_0+a_1x の直線近似を行っていたが,
! LSM_poly では, F(x)=\sum^{N}_{n=0}{a_nx^n}
! の任意次数の多項式曲線近似を行うことが可能.
! アルゴリズムは最小二乗法を用いており, 係数のソルバには gausss ルーチンを使用.
  implicit none
  double precision, intent(in) :: x(:)  ! データ要素配列 1
  double precision, intent(in) :: y(size(x))  ! データ要素配列 2
  double precision, intent(inout) :: a(:)  ! 多項式の係数
  double precision, intent(inout) :: intercept  ! y 切片. 
                         ! a に組み込むと引数を渡すとき, poly_n+1 で渡す必要が
                         ! あり, 紛らわしいと判断したため, a_0 である y 切片を
                         ! 独立で引数として渡すことにした.
  double precision, intent(in), optional :: undef  ! 未定義値.
  integer :: i, j, k
  integer :: nx  ! データの個数
  integer :: poly_n  ! 近似する曲線の最高次数. 1 なら, LSM と同じ.
  double precision :: coe(0:size(a)), tmpa_coe(0:size(a),0:size(a)), tmpb_coe(0:size(a))
          ! coe は a_n が入る. tmp_coe はデータの総和が入る.
          ! [注意] : 第一要素が行. 第二要素が列.
  double precision :: tmp(size(x))  ! べき乗計算の一時配列

  nx=size(x)
  poly_n=size(a)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_1d( nx, y ),  &
  &                                     "LSM_poly_1d" )
  end if

!-- gausss に渡しやすいように, 用意した配列に引数を代入.
  if(present(undef))then
     do k=0,poly_n  ! 列成分の計算
        do j=0,poly_n  ! 行成分の計算. 行成分の計算が先に回ることに注意.
           if(j >= k)then  ! 行成分(j)より列成分(k)の要素数が小さい場合, 値を
                           ! まじめに計算する.
              do i=1,nx
                 if(x(i)/=undef)then
                    tmp(i)=x(i)**(j+k)
                 else
                    tmp(i)=undef
                 end if
              end do
              call summd( tmp, tmpa_coe(j,k), undef )
           else  ! 行成分(j)より列成分(k)の要素数が大きい場合, 解く係数行列が
                 ! 対称行列であることから, 値の参照代入のみ行う.
              tmpa_coe(j,k)=tmpa_coe(k,j)  ! 対称成分の代入（すでに計算済み）
           end if
        end do
     end do
     do j=0,poly_n
        do i=1,nx
           if(x(i)/=undef)then
              tmp(i)=y(i)*(x(i)**j)
           else
              tmp(i)=undef
           end if
        end do
        call summd( tmp, tmpb_coe(j), undef )
     end do
  else  ! undef 処理がないとき.
     do k=0,poly_n  ! 列成分の計算
        do j=0,poly_n  ! 行成分の計算. 行成分の計算が先に回ることに注意.
           if(j >= k)then  ! 行成分(j)より列成分(k)の要素数が小さい場合, 値を
                           ! まじめに計算する.
              do i=1,nx
                 tmp(i)=x(i)**(j+k)
              end do
              call summd( tmp, tmpa_coe(j,k), undef )
           else  ! 行成分(j)より列成分(k)の要素数が大きい場合, 解く係数行列が
                 ! 対称行列であることから, 値の参照代入のみ行う.
              tmpa_coe(j,k)=tmpa_coe(k,j)  ! 対称成分の代入（すでに計算済み）
           end if
        end do
     end do
     do j=0,poly_n
        do i=1,nx
           tmp(i)=y(i)*(x(i)**j)
        end do
        call summd( tmp, tmpb_coe(j), undef )
     end do
  end if

!  以上で係数行列に値が入った.

  call gausssd( tmpa_coe(0:poly_n,0:poly_n), tmpb_coe(0:poly_n),  &
  &             coe(0:poly_n) )

  do i=1,poly_n
     a(i)=coe(i)
  end do
  intercept=coe(0)

end subroutine LSM_poly_1dd

!------------------------------------
!------------------------------------

subroutine LSM_poly_2df( x, y, a, intercept, undef )
! LSM の多項式近似バージョン. (2 次元データ版)
  implicit none
  real, intent(in) :: x(:,:)  ! データ要素配列 1
  real, intent(in) :: y(size(x,1),size(x,2))  ! データ要素配列 2
  real, intent(inout) :: a(:)  ! 多項式の係数
  real, intent(inout) :: intercept  ! y 切片. 
                         ! a に組み込むと引数を渡すとき, poly_n+1 で渡す必要が
                         ! あり, 紛らわしいと判断したため, a_0 である y 切片を
                         ! 独立で引数として渡すことにした.
  real, intent(in), optional :: undef  ! 未定義値.
  integer :: i, j, counter
  integer :: nx  ! データの個数 1
  integer :: ny  ! データの個数 2
  real, dimension(size(x,1)*size(x,2)) :: val1, val2

  nx=size(x,1)
  ny=size(x,2)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, y ),  &
  &                                     "LSM_poly_2d" )
  end if

  counter=0

  do j=1,ny
     do i=1,nx
        counter=counter+1
        val1(counter)=x(i,j)
        val2(counter)=y(i,j)
     end do
  end do

  if(present(undef))then
     call LSM_poly_1df( val1, val2, a, intercept, undef )
  else
     call LSM_poly_1df( val1, val2, a, intercept )
  end if

end subroutine LSM_poly_2df

!------------------------------------
!------------------------------------

subroutine LSM_poly_2dd( x, y, a, intercept, undef )
! LSM の多項式近似バージョン. (2 次元データ版)
  implicit none
  double precision, intent(in) :: x(:,:)  ! データ要素配列 1
  double precision, intent(in) :: y(size(x,1),size(x,2))  ! データ要素配列 2
  double precision, intent(inout) :: a(:)  ! 多項式の係数
  double precision, intent(inout) :: intercept  ! y 切片. 
                         ! a に組み込むと引数を渡すとき, poly_n+1 で渡す必要が
                         ! あり, 紛らわしいと判断したため, a_0 である y 切片を
                         ! 独立で引数として渡すことにした.
  double precision, intent(in), optional :: undef  ! 未定義値.
  integer :: i, j, counter
  integer :: nx  ! データの個数 1
  integer :: ny  ! データの個数 2
  double precision, dimension(size(x,1)*size(x,2)) :: val1, val2

  nx=size(x,1)
  ny=size(x,2)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, y ),  &
  &                                     "LSM_poly_2d" )
  end if

  counter=0

  do j=1,ny
     do i=1,nx
        counter=counter+1
        val1(counter)=x(i,j)
        val2(counter)=y(i,j)
     end do
  end do

  if(present(undef))then
     call LSM_poly_1dd( val1, val2, a, intercept, undef )
  else
     call LSM_poly_1dd( val1, val2, a, intercept )
  end if

end subroutine LSM_poly_2dd

!------------------------------------
!------------------------------------

subroutine LSM_poly_3df( x, y, a, intercept, undef )
! LSM の多項式近似バージョン. (3 次元データ版)
  implicit none
  real, intent(in) :: x(:,:,:)  ! データ要素配列 1
  real, intent(in) :: y(size(x,1),size(x,2),size(x,3))  ! データ要素配列 2
  real, intent(inout) :: a(:)  ! 多項式の係数
  real, intent(inout) :: intercept  ! y 切片. 
                         ! a に組み込むと引数を渡すとき, poly_n+1 で渡す必要が
                         ! あり, 紛らわしいと判断したため, a_0 である y 切片を
                         ! 独立で引数として渡すことにした.
  real, intent(in), optional :: undef  ! 未定義値.
  integer :: i, j, k, counter
  integer :: nx  ! データの個数 1
  integer :: ny  ! データの個数 2
  integer :: nz  ! データの個数 3
  real, dimension(size(x,1)*size(x,2)*size(x,3)) :: val1, val2

  nx=size(x,1)
  ny=size(x,2)
  nz=size(x,3)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, y ),  &
  &                                     "LSM_poly_3d" )
  end if

  counter=0

  do k=1,nz
     do j=1,ny
        do i=1,nx
           counter=counter+1
           val1(counter)=x(i,j,k)
           val2(counter)=y(i,j,k)
        end do
     end do
  end do

  if(present(undef))then
     call LSM_poly_1df( val1, val2, a, intercept, undef )
  else
     call LSM_poly_1df( val1, val2, a, intercept )
  end if

end subroutine LSM_poly_3df

!------------------------------------
!------------------------------------

subroutine LSM_poly_3dd( x, y, a, intercept, undef )
! LSM の多項式近似バージョン. (3 次元データ版)
  implicit none
  double precision, intent(in) :: x(:,:,:)  ! データ要素配列 1
  double precision, intent(in) :: y(size(x,1),size(x,2),size(x,3))  ! データ要素配列 2
  double precision, intent(inout) :: a(:)  ! 多項式の係数
  double precision, intent(inout) :: intercept  ! y 切片. 
                         ! a に組み込むと引数を渡すとき, poly_n+1 で渡す必要が
                         ! あり, 紛らわしいと判断したため, a_0 である y 切片を
                         ! 独立で引数として渡すことにした.
  double precision, intent(in), optional :: undef  ! 未定義値.
  integer :: i, j, k, counter
  integer :: nx  ! データの個数 1
  integer :: ny  ! データの個数 2
  integer :: nz  ! データの個数 3
  double precision, dimension(size(x,1)*size(x,2)*size(x,3)) :: val1, val2

  nx=size(x,1)
  ny=size(x,2)
  nz=size(x,3)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, y ),  &
  &                                     "LSM_poly_3d" )
  end if

  counter=0

  do k=1,nz
     do j=1,ny
        do i=1,nx
           counter=counter+1
           val1(counter)=x(i,j,k)
           val2(counter)=y(i,j,k)
        end do
     end do
  end do

  if(present(undef))then
     call LSM_poly_1dd( val1, val2, a, intercept, undef )
  else
     call LSM_poly_1dd( val1, val2, a, intercept )
  end if

end subroutine LSM_poly_3dd

!------------------------------------
!------------------------------------

subroutine LSM_multi_f( x, y, a, undef )
! LSM の多変数多項式近似バージョン.
! LSM_multi では, F(x_0,x_1,...,x_n)=\sum^{N}_{n=0}{a_nx_n}
! の任意次数の多項式曲線近似を行うことが可能.
! アルゴリズムは最小二乗法を用いており, 係数のソルバには gausss ルーチンを使用.
  implicit none
  real, intent(in) :: x(:,:)  ! データ要素配列 1
  real, intent(in) :: y(size(x,1))  ! データ要素配列 2
  real, intent(inout) :: a(size(x,2))  ! 多項式の係数
  real, intent(in), optional :: undef  ! 未定義値.
  integer :: i, j, k
  integer :: nx  ! データの個数
  integer :: multi_n  ! 近似する多変数の数. 1 なら, LSM と同じ.
  real :: coe(size(x,2)), tmpa_coe(size(x,2),size(x,2)), tmpb_coe(size(x,2))
          ! coe は a_n が入る. tmp_coe はデータの総和が入る.
          ! [注意] : 第一要素が行. 第二要素が列.
  real :: tmp(size(x,1))  ! べき乗計算の一時配列
  ! [注意] ここで, データの個数 nx がデータ点の数であり, この点 1 つずつに
  !        multi_n 個の種類の独立なデータが得られていることを想定する.

  nx=size(x,1)
  multi_n=size(x,2)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_1d( nx, y ),  &
  &                                     "LSM_multi" )
  end if

!-- gausss に渡しやすいように, 用意した配列に引数を代入.
  if(present(undef))then
     do k=1,multi_n  ! 列成分の計算
        do j=1,multi_n  ! 行成分の計算. 行成分の計算が先に回ることに注意.
           if(j >= k)then  ! 行成分(j)より列成分(k)の要素数が小さい場合, 値を
                           ! まじめに計算する.
              do i=1,nx
                 if(x(i,j)/=undef.and.x(i,k)/=undef)then
                    tmp(i)=x(i,j)*x(i,k)
                 else
                    tmp(i)=undef
                 end if
              end do
              call summf( tmp, tmpa_coe(j,k), undef )
           else  ! 行成分(j)より列成分(k)の要素数が大きい場合, 解く係数行列が
                 ! 対称行列であることから, 値の参照代入のみ行う.
              tmpa_coe(j,k)=tmpa_coe(k,j)  ! 対称成分の代入（すでに計算済み）
           end if
        end do
     end do
     do j=1,multi_n
        do i=1,nx
           if(x(i,j)/=undef)then
              tmp(i)=y(i)*x(i,j)
           else
              tmp(i)=undef
           end if
        end do
        call summf( tmp, tmpb_coe(j), undef )
     end do
  else  ! undef 処理がないとき.
     do k=1,multi_n  ! 列成分の計算
        do j=1,multi_n  ! 行成分の計算. 行成分の計算が先に回ることに注意.
           if(j >= k)then  ! 行成分(j)より列成分(k)の要素数が小さい場合, 値を
                           ! まじめに計算する.
              do i=1,nx
                 tmp(i)=x(i,j)*x(i,k)
              end do
              call summf( tmp, tmpa_coe(j,k), undef )
           else  ! 行成分(j)より列成分(k)の要素数が大きい場合, 解く係数行列が
                 ! 対称行列であることから, 値の参照代入のみ行う.
              tmpa_coe(j,k)=tmpa_coe(k,j)  ! 対称成分の代入（すでに計算済み）
           end if
        end do
     end do
     do j=1,multi_n
        do i=1,nx
           tmp(i)=y(i)*x(i,j)
        end do
        call summf( tmp, tmpb_coe(j), undef )
     end do
  end if

!  以上で係数行列に値が入った.

  call gausssf( tmpa_coe(1:multi_n,1:multi_n), tmpb_coe(1:multi_n),  &
  &            coe(1:multi_n) )

  do i=1,multi_n
     a(i)=coe(i)
  end do

end subroutine LSM_multi_f

!------------------------------------
!------------------------------------

subroutine LSM_multi_d( x, y, a, undef )
! LSM の多変数多項式近似バージョン.
! LSM_multi では, F(x_0,x_1,...,x_n)=\sum^{N}_{n=0}{a_nx_n}
! の任意次数の多項式曲線近似を行うことが可能.
! アルゴリズムは最小二乗法を用いており, 係数のソルバには gausss ルーチンを使用.
  implicit none
  double precision, intent(in) :: x(:,:)  ! データ要素配列 1
  double precision, intent(in) :: y(size(x,1))  ! データ要素配列 2
  double precision, intent(inout) :: a(size(x,2))  ! 多項式の係数
  double precision, intent(in), optional :: undef  ! 未定義値.
  integer :: i, j, k
  integer :: nx  ! データの個数
  integer :: multi_n  ! 近似する多変数の数. 1 なら, LSM と同じ.
  double precision :: coe(size(x,2)), tmpa_coe(size(x,2),size(x,2)), tmpb_coe(size(x,2))
          ! coe は a_n が入る. tmp_coe はデータの総和が入る.
          ! [注意] : 第一要素が行. 第二要素が列.
  double precision :: tmp(size(x,1))  ! べき乗計算の一時配列
  ! [注意] ここで, データの個数 nx がデータ点の数であり, この点 1 つずつに
  !        multi_n 個の種類の独立なデータが得られていることを想定する.

  nx=size(x,1)
  multi_n=size(x,2)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_1d( nx, y ),  &
  &                                     "LSM_multi" )
  end if

!-- gausss に渡しやすいように, 用意した配列に引数を代入.
  if(present(undef))then
     do k=1,multi_n  ! 列成分の計算
        do j=1,multi_n  ! 行成分の計算. 行成分の計算が先に回ることに注意.
           if(j >= k)then  ! 行成分(j)より列成分(k)の要素数が小さい場合, 値を
                           ! まじめに計算する.
              do i=1,nx
                 if(x(i,j)/=undef.and.x(i,k)/=undef)then
                    tmp(i)=x(i,j)*x(i,k)
                 else
                    tmp(i)=undef
                 end if
              end do
              call summd( tmp, tmpa_coe(j,k), undef )
           else  ! 行成分(j)より列成分(k)の要素数が大きい場合, 解く係数行列が
                 ! 対称行列であることから, 値の参照代入のみ行う.
              tmpa_coe(j,k)=tmpa_coe(k,j)  ! 対称成分の代入（すでに計算済み）
           end if
        end do
     end do
     do j=1,multi_n
        do i=1,nx
           if(x(i,j)/=undef)then
              tmp(i)=y(i)*x(i,j)
           else
              tmp(i)=undef
           end if
        end do
        call summd( tmp, tmpb_coe(j), undef )
     end do
  else  ! undef 処理がないとき.
     do k=1,multi_n  ! 列成分の計算
        do j=1,multi_n  ! 行成分の計算. 行成分の計算が先に回ることに注意.
           if(j >= k)then  ! 行成分(j)より列成分(k)の要素数が小さい場合, 値を
                           ! まじめに計算する.
              do i=1,nx
                 tmp(i)=x(i,j)*x(i,k)
              end do
              call summd( tmp, tmpa_coe(j,k), undef )
           else  ! 行成分(j)より列成分(k)の要素数が大きい場合, 解く係数行列が
                 ! 対称行列であることから, 値の参照代入のみ行う.
              tmpa_coe(j,k)=tmpa_coe(k,j)  ! 対称成分の代入（すでに計算済み）
           end if
        end do
     end do
     do j=1,multi_n
        do i=1,nx
           tmp(i)=y(i)*x(i,j)
        end do
        call summd( tmp, tmpb_coe(j), undef )
     end do
  end if

!  以上で係数行列に値が入った.

  call gausssd( tmpa_coe(1:multi_n,1:multi_n), tmpb_coe(1:multi_n),  &
  &            coe(1:multi_n) )

  do i=1,multi_n
     a(i)=coe(i)
  end do

end subroutine LSM_multi_d

!------------------------------------
!------------------------------------

subroutine Reg_Line_1df( x, y, slope, intercept, undef )
  ! LSM を用いて回帰直線の傾き slope と切片 intercept を計算するルーチン
  implicit none
  real, intent(in) :: x(:)  ! データ要素 1
  real, intent(in) :: y(size(x))  ! データ要素 2
  real, intent(inout) :: slope  ! 最適な傾き
  real, intent(inout) :: intercept  ! 最適な切片
  real, intent(in), optional :: undef  ! 未定義値
  real :: cov, sig, xm, ym
  integer :: nx  ! データ数

  nx=size(x)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_1d( nx, y ),  &
  &                                     "Reg_Line_1d" )
  end if

  if(present(undef))then
     call Mean_1df( x, xm, error=undef )
     call Mean_1df( y, ym, error=undef )
     call covariance_1df( x, y, cov, error=undef )  ! sum[(x-xm)*(y-ym)]/n
     call stand_devi_1df( x, sig, error=undef )  ! sqrt(sum[(x-xm)^2]/n)
     if(sig/=0.0.and.cov/=undef.and.sig/=undef)then
        slope=cov/(sig**2)
        intercept=ym-xm*slope
     else
        slope=undef
        intercept=undef
     end if
  else
     call Mean_1df( x, xm )
     call Mean_1df( y, ym )
     call covariance_1df( x, y, cov )  ! sum[(x-xm)*(y-ym)]/n
     call stand_devi_1df( x, sig )  ! sqrt(sum[(x-xm)^2]/n)
     slope=cov/(sig**2)
     intercept=ym-xm*slope
  end if

end subroutine Reg_Line_1df

!------------------------------------
!------------------------------------

subroutine Reg_Line_1dd( x, y, slope, intercept, undef )
  ! LSM を用いて回帰直線の傾き slope と切片 intercept を計算するルーチン
  implicit none
  double precision, intent(in) :: x(:)  ! データ要素 1
  double precision, intent(in) :: y(size(x))  ! データ要素 2
  double precision, intent(inout) :: slope  ! 最適な傾き
  double precision, intent(inout) :: intercept  ! 最適な切片
  double precision, intent(in), optional :: undef  ! 未定義値
  double precision :: cov, sig, xm, ym
  integer :: nx  ! データ数

  nx=size(x)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_1d( nx, y ),  &
  &                                     "Reg_Line_1d" )
  end if

  if(present(undef))then
     call Mean_1dd( x, xm, error=undef )
     call Mean_1dd( y, ym, error=undef )
     call covariance_1dd( x, y, cov, error=undef )  ! sum[(x-xm)*(y-ym)]/n
     call stand_devi_1dd( x, sig, error=undef )  ! sqrt(sum[(x-xm)^2]/n)
     if(sig/=0.0.and.cov/=undef.and.sig/=undef)then
        slope=cov/(sig**2)
        intercept=ym-xm*slope
     else
        slope=undef
        intercept=undef
     end if
  else
     call Mean_1dd( x, xm )
     call Mean_1dd( y, ym )
     call covariance_1dd( x, y, cov )  ! sum[(x-xm)*(y-ym)]/n
     call stand_devi_1dd( x, sig )  ! sqrt(sum[(x-xm)^2]/n)
     slope=cov/(sig**2)
     intercept=ym-xm*slope
  end if

end subroutine Reg_Line_1dd

!------------------------------------
!------------------------------------

subroutine Reg_Line_2df( x, y, slope, intercept, error )
  ! LSM を用いて回帰直線の傾き slope と切片 intercept を計算するルーチン (2 次元版)
  implicit none
  real, intent(in) :: x(:,:)  ! データ要素 1
  real, intent(in) :: y(size(x,1),size(x,2))  ! データ要素 2
  real, intent(inout) :: slope  ! 最適な傾き
  real, intent(inout) :: intercept  ! 最適な切片
  real, intent(in), optional :: error
  real, dimension(size(x,1)*size(x,2)) :: u, v
  integer :: i, j, counter
  integer :: nx  ! データ数 1
  integer :: ny  ! データ数 2

  nx=size(x,1)
  ny=size(x,2)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, y ),  &
  &                                     "Reg_Line_2d" )
  end if

  counter=0

  do j=1,ny
     do i=1,nx
        counter=counter+1
        u(counter)=x(i,j)
        v(counter)=y(i,j)
     end do
  end do

  if(present(error))then
     call Reg_Line_1df( u, v, slope, intercept, error )
  else
     call Reg_Line_1df( u, v, slope, intercept )
  end if

end subroutine Reg_Line_2df

!------------------------------------
!------------------------------------

subroutine Reg_Line_2dd( x, y, slope, intercept, error )
  ! LSM を用いて回帰直線の傾き slope と切片 intercept を計算するルーチン (2 次元版)
  implicit none
  double precision, intent(in) :: x(:,:)  ! データ要素 1
  double precision, intent(in) :: y(size(x,1),size(x,2))  ! データ要素 2
  double precision, intent(inout) :: slope  ! 最適な傾き
  double precision, intent(inout) :: intercept  ! 最適な切片
  double precision, intent(in), optional :: error
  double precision, dimension(size(x,1)*size(x,2)) :: u, v
  integer :: i, j, counter
  integer :: nx  ! データ数 1
  integer :: ny  ! データ数 2

  nx=size(x,1)
  ny=size(x,2)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, y ),  &
  &                                     "Reg_Line_2d" )
  end if

  counter=0

  do j=1,ny
     do i=1,nx
        counter=counter+1
        u(counter)=x(i,j)
        v(counter)=y(i,j)
     end do
  end do

  if(present(error))then
     call Reg_Line_1dd( u, v, slope, intercept, error )
  else
     call Reg_Line_1dd( u, v, slope, intercept )
  end if

end subroutine Reg_Line_2dd

!------------------------------------
!------------------------------------

subroutine Reg_Line_3df( x, y, slope, intercept, error )
  ! LSM を用いて回帰直線の傾き slope と切片 intercept を計算するルーチン (3 次元版)
  implicit none
  real, intent(in) :: x(:,:,:)  ! データ要素 1
  real, intent(in) :: y(size(x,1),size(x,2),size(x,3))  ! データ要素 2
  real, intent(inout) :: slope  ! 最適な傾き
  real, intent(inout) :: intercept  ! 最適な切片
  real, intent(in), optional :: error
  real, dimension(size(x,1)*size(x,2)*size(x,3)) :: u, v
  integer :: i, j, k, counter
  integer :: nx  ! データ数 1
  integer :: ny  ! データ数 2
  integer :: nz  ! データ数 3

  nx=size(x,1)
  ny=size(x,2)
  nz=size(x,3)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, y ),  &
  &                                     "Reg_Line_3d" )
  end if

  counter=0

  do k=1,nz
     do j=1,ny
        do i=1,nx
           counter=counter+1
           u(counter)=x(i,j,k)
           v(counter)=y(i,j,k)
        end do
     end do
  end do

  if(present(error))then
     call Reg_Line_1df( u, v, slope, intercept, error )
  else
     call Reg_Line_1df( u, v, slope, intercept )
  end if

end subroutine Reg_Line_3df

!------------------------------------
!------------------------------------

subroutine Reg_Line_3dd( x, y, slope, intercept, error )
  ! LSM を用いて回帰直線の傾き slope と切片 intercept を計算するルーチン (3 次元版)
  implicit none
  double precision, intent(in) :: x(:,:,:)  ! データ要素 1
  double precision, intent(in) :: y(size(x,1),size(x,2),size(x,3))  ! データ要素 2
  double precision, intent(inout) :: slope  ! 最適な傾き
  double precision, intent(inout) :: intercept  ! 最適な切片
  double precision, intent(in), optional :: error
  double precision, dimension(size(x,1)*size(x,2)*size(x,3)) :: u, v
  integer :: i, j, k, counter
  integer :: nx  ! データ数 1
  integer :: ny  ! データ数 2
  integer :: nz  ! データ数 3

  nx=size(x,1)
  ny=size(x,2)
  nz=size(x,3)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, y ),  &
  &                                     "Reg_Line_3d" )
  end if

  counter=0

  do k=1,nz
     do j=1,ny
        do i=1,nx
           counter=counter+1
           u(counter)=x(i,j,k)
           v(counter)=y(i,j,k)
        end do
     end do
  end do

  if(present(error))then
     call Reg_Line_1dd( u, v, slope, intercept, error )
  else
     call Reg_Line_1dd( u, v, slope, intercept )
  end if

end subroutine Reg_Line_3dd

!------------------------------------
!------------------------------------

subroutine Cor_Coe_1df( x, y ,cc, error )  ! 2 データの相関係数を計算するルーチン
  implicit none
  real, intent(in) :: x(:)  ! データ要素 1
  real, intent(in) :: y(size(x))  ! データ要素 2
  real, intent(inout) :: cc  ! 相関係数
  real, intent(in), optional :: error  ! 欠損値
  integer :: nx  ! データ個数
  real :: cov, anor1, anor2
  real :: rx(size(x)), ry(size(x))

  nx=size(x)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_1d( nx, y ),  &
  &                                     "Cor_Coe_1d" )
  end if

  if(present(error))then
     call replace_undef_f( x, y, rx, ry, error )
     call covariance_1df( rx, ry, cov, error )
     call stand_devi_1df( rx, anor1, error )
     call stand_devi_1df( ry, anor2, error )
  else
     call covariance_1df( x, y, cov )
     call stand_devi_1df( x, anor1 )
     call stand_devi_1df( y, anor2 )
  end if

  cc=cov/(anor1*anor2)
end subroutine Cor_Coe_1df

!---------------------------------
!------------------------------------

subroutine Cor_Coe_1dd( x, y ,cc, error )  ! 2 データの相関係数を計算するルーチン
  implicit none
  double precision, intent(in) :: x(:)  ! データ要素 1
  double precision, intent(in) :: y(size(x))  ! データ要素 2
  double precision, intent(inout) :: cc  ! 相関係数
  double precision, intent(in), optional :: error  ! 欠損値
  integer :: nx  ! データ個数
  double precision :: cov, anor1, anor2
  double precision :: rx(size(x)), ry(size(x))

  nx=size(x)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_1d( nx, y ),  &
  &                                     "Cor_Coe_1d" )
  end if

  if(present(error))then
     call replace_undef_d( x, y, rx, ry, error )
     call covariance_1dd( rx, ry, cov, error )
     call stand_devi_1dd( rx, anor1, error )
     call stand_devi_1dd( ry, anor2, error )
  else
     call covariance_1dd( x, y, cov )
     call stand_devi_1dd( x, anor1 )
     call stand_devi_1dd( y, anor2 )
  end if

  cc=cov/(anor1*anor2)

end subroutine Cor_Coe_1dd

!---------------------------------
!------------------------------------

subroutine Cor_Coe_2df( x, y ,cc, error )  ! 2 データの相関係数を計算するルーチン (2 次元版)
  implicit none
  real, intent(in) :: x(:,:)  ! データ要素 1
  real, intent(in) :: y(size(x,1),size(x,2))  ! データ要素 2
  real, intent(inout) :: cc  ! 相関係数
  real, intent(in), optional :: error  ! 欠損値
  integer :: i, j, counter
  integer :: nx  ! データ個数 1
  integer :: ny  ! データ個数 2
  real, dimension(size(x,1)*size(x,2)) :: val1, val2

  nx=size(x,1)
  ny=size(x,2)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, y ),  &
  &                                     "Cor_Coe_2d" )
  end if

  counter=0

  do j=1,ny
     do i=1,nx
        counter=counter+1
        val1(counter)=x(i,j)
        val2(counter)=y(i,j)
     end do
  end do

  if(present(error))then
     call Cor_Coe_1df( val1, val2, cc, error )
  else
     call Cor_Coe_1df( val1, val2, cc )
  end if

end subroutine Cor_Coe_2df

!---------------------------------
!------------------------------------

subroutine Cor_Coe_2dd( x, y ,cc, error )  ! 2 データの相関係数を計算するルーチン (2 次元版)
  implicit none
  double precision, intent(in) :: x(:,:)  ! データ要素 1
  double precision, intent(in) :: y(size(x,1),size(x,2))  ! データ要素 2
  double precision, intent(inout) :: cc  ! 相関係数
  double precision, intent(in), optional :: error  ! 欠損値
  integer :: i, j, counter
  integer :: nx  ! データ個数 1
  integer :: ny  ! データ個数 2
  double precision, dimension(size(x,1)*size(x,2)) :: val1, val2

  nx=size(x,1)
  ny=size(x,2)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, y ),  &
  &                                     "Cor_Coe_2d" )
  end if

  counter=0

  do j=1,ny
     do i=1,nx
        counter=counter+1
        val1(counter)=x(i,j)
        val2(counter)=y(i,j)
     end do
  end do

  if(present(error))then
     call Cor_Coe_1dd( val1, val2, cc, error )
  else
     call Cor_Coe_1dd( val1, val2, cc )
  end if

end subroutine Cor_Coe_2dd

!---------------------------------
!------------------------------------

subroutine Cor_Coe_3df( x, y ,cc, error )  ! 2 データの相関係数を計算するルーチン (3 次元版)
  implicit none
  real, intent(in) :: x(:,:,:)  ! データ要素 1
  real, intent(in) :: y(size(x,1),size(x,2),size(x,3))  ! データ要素 2
  real, intent(inout) :: cc  ! 相関係数
  real, intent(in), optional :: error  ! 欠損値
  integer :: i, j, k, counter
  integer :: nx  ! データ個数 1
  integer :: ny  ! データ個数 2
  integer :: nz  ! データ個数 2
  real, dimension(size(x,1)*size(x,2)*size(x,3)) :: val1, val2

  nx=size(x,1)
  ny=size(x,2)
  nz=size(x,3)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, y ),  &
  &                                     "Cor_Coe_3d" )
  end if

  counter=0

  do k=1,nz
     do j=1,ny
        do i=1,nx
           counter=counter+1
           val1(counter)=x(i,j,k)
           val2(counter)=y(i,j,k)
        end do
     end do
  end do

  if(present(error))then
     call Cor_Coe_1df( val1, val2, cc, error )
  else
     call Cor_Coe_1df( val1, val2, cc )
  end if

end subroutine Cor_Coe_3df

!---------------------------------
!------------------------------------

subroutine Cor_Coe_3dd( x, y ,cc, error )  ! 2 データの相関係数を計算するルーチン (3 次元版)
  implicit none
  double precision, intent(in) :: x(:,:,:)  ! データ要素 1
  double precision, intent(in) :: y(size(x,1),size(x,2),size(x,3))  ! データ要素 2
  double precision, intent(inout) :: cc  ! 相関係数
  double precision, intent(in), optional :: error  ! 欠損値
  integer :: i, j, k, counter
  integer :: nx  ! データ個数 1
  integer :: ny  ! データ個数 2
  integer :: nz  ! データ個数 2
  double precision, dimension(size(x,1)*size(x,2)*size(x,3)) :: val1, val2

  nx=size(x,1)
  ny=size(x,2)
  nz=size(x,3)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, y ),  &
  &                                     "Cor_Coe_3d" )
  end if

  counter=0

  do k=1,nz
     do j=1,ny
        do i=1,nx
           counter=counter+1
           val1(counter)=x(i,j,k)
           val2(counter)=y(i,j,k)
        end do
     end do
  end do

  if(present(error))then
     call Cor_Coe_1dd( val1, val2, cc, error )
  else
     call Cor_Coe_1dd( val1, val2, cc )
  end if

end subroutine Cor_Coe_3dd

!---------------------------------
!---------------------------------

subroutine smooth_1d( x, n, y, method, weight, error )
! 1 次元データについて, スムージングするルーチン
! 現在, error オプションは機能していない.
  implicit none
  real, intent(in) :: x(:)  ! スムージングするデータ
  integer, intent(in) :: n  ! スムーズの影響格子数(中心を含めた左右幅)
  real, intent(inout) :: y(size(x))  ! スムージングされたデータ
  character(3), intent(in) :: method  ! スムージングの方法
                            ! "SMP" = 単純平均, "OPT" = オプション重み
                            ! "MAX" = 最大値, "MIN" = 最小値
  real, intent(in), optional :: weight(n)  ! method 引数が "OPT" の場合
                            ! weight(1) が左端, weight(n) が右端として重み
  real, intent(in), optional :: error  ! 未定義値
  integer :: ix, j, mx, half, val
  real :: div_fact
  real :: wg(n)

  y=0.0
  mx=size(x)
  half=(n-1)/2

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_1d( mx, y ),  &
  &                                     "smooth_1d" )
  end if

  select case (method(1:3))
  case ("SMP")
     do j=1,n
        wg(j)=1.0
     end do

  case ("OPT")
     if(present(weight))then
        do j=1,n
           wg(j)=weight(j)
        end do
     end if
  end select

  if(method(1:3)/="MIN".and.method(1:3)/="MAX")then
!-- determining dividing factor

     div_fact=0.0

     do j=1,n
        if(wg(j)<0.0)then
           div_fact=1.0
           exit
        else
           div_fact=div_fact+wg(j)
        end if
     end do

!-- avoiding zero dividing

     if(div_fact==0.0)then
        div_fact=1.0
     end if

     do ix=half+1,mx-half
        do j=1,n
           y(ix)=y(ix)+x(ix-half-1+j)*wg(j)
        end do
        y(ix)=y(ix)/div_fact
     end do

  else

     select case (method(1:3))
     case ("MAX")
        do ix=half+1,mx-half
           val=x(ix-half)
           do j=2,n
              if(val<x(ix-half-1+j))then
                 val=x(ix-half-1+j)
              end if
           end do
           y(ix-half)=val
        end do

     case ("MIN")
        do ix=half+1,mx-half
           val=x(ix-half)
           do j=2,n
              if(val>x(ix-half-1+j))then
                 val=x(ix-half-1+j)
              end if
           end do
           y(ix-half)=val
        end do

     end select

  end if

end subroutine smooth_1d

!---------------------------------
!---------------------------------

subroutine smooth_2d( x, n, y, method, weight, error )
! 2 次元データについて, スムージングするルーチン
! 現在, error オプションは機能していない.
  implicit none
  real, intent(in) :: x(:,:)  ! スムージングするデータ
  integer, intent(in) :: n  ! スムーズの影響格子数(中心を含めた左右幅)
  real, intent(inout) :: y(size(x,1),size(x,2))  ! スムージングされたデータ
  character(3), intent(in) :: method  ! スムージングの方法
                            ! "SMP" = 単純平均, "OPT" = オプション重み
                            ! "MAX" = 最大値, "MIN" = 最小値
  real, intent(in), optional :: weight(n,n)  ! method 引数が "OPT" の場合
                          ! weight(1,1) が左下端, weight(n,n) が右上端として重み
  real, intent(in), optional :: error  ! 未定義値
  integer :: ix, iy, j, k, mx, my, half, val
  real :: div_fact
  real :: wg(n,n)

  y=0.0
  mx=size(x,1)
  my=size(x,2)
  half=(n-1)/2

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( mx, my, y ),  &
  &                                     "smooth_2d" )
  end if

  select case (method(1:3))
  case ("SMP")
     do k=1,n
        do j=1,n
           wg(j,k)=1.0
        end do
     end do

  case ("OPT")
     if(present(weight))then
        do k=1,n
           do j=1,n
              wg(j,k)=weight(j,k)
           end do
        end do
     end if
  end select

  if(method(1:3)/="MIN".and.method(1:3)/="MAX")then
!-- determining dividing factor

     div_fact=0.0

     do k=1,n
        do j=1,n
           if(wg(j,k)<0.0)then
              div_fact=1.0
              exit
           else
              div_fact=div_fact+wg(j,k)
           end if
        end do
     end do

!-- avoiding zero dividing

     if(div_fact==0.0)then
        div_fact=1.0
     end if

     do iy=half+1,my-half
        do ix=half+1,mx-half
           do k=1,n
              do j=1,n
                 y(ix,iy)=y(ix,iy)+x(ix-half-1+j,iy-half-1+k)*wg(j,k)
              end do
           end do

           y(ix,iy)=y(ix,iy)/div_fact
        end do
     end do

  else

     select case (method(1:3))
     case ("MAX")
        do iy=half+1,my-half
           do ix=half+1,mx-half
              val=x(ix-half,iy-half)
              do k=2,n
                 do j=2,n
                    if(val<x(ix-half-1+j,iy-half-1+k))then
                       val=x(ix-half-1+j,iy-half-1+k)
                    end if
                 end do
              end do
              y(ix-half,iy-half)=val
           end do
        end do

     case ("MIN")
        do iy=half+1,my-half
           do ix=half+1,mx-half
              val=x(ix-half,iy-half)
              do k=2,n
                 do j=2,n
                    if(val>x(ix-half-1+j,iy-half-1+k))then
                       val=x(ix-half-1+j,iy-half-1+k)
                    end if
                 end do
              end do
              y(ix-half,iy-half)=val
           end do
        end do

     end select

  end if

end subroutine smooth_2d

!---------------------------------
!---------------------------------

subroutine smooth_3d( x, n, y, method, weight, error )
! 3 次元データについて, スムージングするルーチン
! 現在, error オプションは機能していない.
  implicit none
  real, intent(in) :: x(:,:,:)  ! スムージングするデータ
  integer, intent(in) :: n  ! スムーズの影響格子数(中心を含めた左右幅)
  real, intent(inout) :: y(size(x,1),size(x,2),size(x,3))
                            ! スムージングされたデータ
  character(3), intent(in) :: method  ! スムージングの方法
                            ! "SMP" = 単純平均, "OPT" = オプション重み
                            ! "MAX" = 最大値, "MIN" = 最小値
  real, intent(in), optional :: weight(n,n,n)  ! method 引数が "OPT" の場合
  real, intent(in), optional :: error  ! 未定義値
  integer :: ix, iy, iz, j, k, l, mx, my, mz, half, val
  real :: div_fact
  real :: wg(n,n,n)

  y=0.0
  mx=size(x,1)
  my=size(x,2)
  mz=size(x,3)
  half=(n-1)/2

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_3d( mx, my, mz, y ),  &
  &                                     "smooth_3d" )
  end if

  select case (method(1:3))
  case ("SMP")
     do l=1,n
        do k=1,n
           do j=1,n
              wg(j,k,l)=1.0
           end do
        end do
     end do

  case ("OPT")
     if(present(weight))then
        do l=1,n
           do k=1,n
              do j=1,n
                 wg(j,k,l)=weight(j,k,l)
              end do
           end do
        end do
     end if
  end select

  if(method(1:3)/="MIN".and.method(1:3)/="MAX")then
!-- determining dividing factor

     div_fact=0.0

     do l=1,n
        do k=1,n
           do j=1,n
              if(wg(j,k,l)<0.0)then
                 div_fact=1.0
                 exit
              else
                 div_fact=div_fact+wg(j,k,l)
              end if
           end do
        end do
     end do

!-- avoiding zero dividing

     if(div_fact==0.0)then
        div_fact=1.0
     end if

     do iz=half+1,mz-half
        do iy=half+1,my-half
           do ix=half+1,mx-half
              do l=1,n
                 do k=1,n
                    do j=1,n
                       y(ix,iy,iz)=y(ix,iy,iz)  &
  &                                +x(ix-half-1+j,iy-half-1+k,iz-half-1+l)  &
  &                                *wg(j,k,l)
                    end do
                 end do
              end do

              y(ix,iy,iz)=y(ix,iy,iz)/div_fact
           end do
        end do
     end do

  else

     select case (method(1:3))
     case ("MAX")
        do iz=half+1,mz-half
           do iy=half+1,my-half
              do ix=half+1,mx-half
                 val=x(ix-half,iy-half,iz-half)
                 do l=2,n
                    do k=2,n
                       do j=2,n
                          if(val<x(ix-half-1+j,iy-half-1+k,iz-half-1+l))then
                             val=x(ix-half-1+j,iy-half-1+k,iz-half-1+l)
                          end if
                       end do
                    end do
                 end do
                 y(ix-half,iy-half,iz-half)=val
              end do
           end do
        end do

     case ("MIN")
        do iz=half+1,mz-half
           do iy=half+1,my-half
              do ix=half+1,mx-half
                 val=x(ix-half,iy-half,iz-half)
                 do l=2,n
                    do k=2,n
                       do j=2,n
                          if(val>x(ix-half-1+j,iy-half-1+k,iz-half-1+l))then
                             val=x(ix-half-1+j,iy-half-1+k,iz-half-1+l)
                          end if
                       end do
                    end do
                 end do
                 y(ix-half,iy-half,iz-half)=val
              end do
           end do
        end do

     end select

  end if

end subroutine smooth_3d

!---------------------------------
!---------------------------------

subroutine Move_ave( x, n, y, error, offset )
! 移動平均を計算するルーチン
  implicit none
  real, intent(in) :: x(:)  ! データ
  integer, intent(in) :: n  ! 平均をとる数
  real, intent(inout) :: y(size(x))  ! 平均化した後のデータ.
                      ! 実際は, y(1:n-1) までの配列にはゼロが入る.
  real, intent(in), optional :: error  ! 欠損値
  integer, intent(in), optional :: offset  ! 移動平均を開始する要素番号.
                      ! default = n
  integer :: nx, i, ioff
  real :: tmp

  nx=size(x)
  y=0.0

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_1d( nx, y ),  &
  &                                     "Move_ave" )
  end if

  if(nx<n.or.n<2)then
     write(*,*) "### ERROR ### (Move_ave)"
     write(*,*) "x(nx) : nx must be more than n or n must be more than 2."
     write(*,*) "nx is ", nx, ", n is ", n, "."
     write(*,*) "STOP"
     stop
  end if

  if(present(offset))then
     if(offset>0)then
        ioff=offset
     else
        write(*,*) "### ERROR ### (Move_ave)"
        write(*,*) "offset must be more than 1."
        write(*,*) "STOP"
        stop
     end if
  else
     ioff=n
  end if

  if(present(error))then
     call Mean_1d( x(1:n), tmp, error )
     if(ioff>2)then
        y(1:ioff-1)=error
        y(nx-n+ioff+1:nx)=error
     end if
     y(ioff)=tmp

     do i=ioff+1,nx-n+ioff
!        if(x(i+n-ioff)/=error.and.x(i-ioff)/=error)then
!           y(i)=y(i-1)+(x(i+n-ioff)-x(i-ioff))/real(n)
!        else
           call Mean_1d( x(i-ioff+1:i+n-ioff), y(i), error )
!        end if
     end do
  else
     call Mean_1d( x(1:n), tmp )
     if(ioff>2)then
        y(1:ioff-1)=0.0
        y(nx-n+ioff+1:nx)=0.0
     end if
     y(ioff)=tmp

     do i=ioff+1,nx-n+ioff
        y(i)=y(i-1)+(x(i+n-ioff)-x(i-ioff))/real(n)
     end do
  end if

end subroutine Move_ave

!---------------------------------
!---------------------------------

subroutine Move_anom( x, n, y, error, offset )
! 移動平均からのアノマリを計算するルーチン
  implicit none
  real, intent(in) :: x(:)  ! データ
  integer, intent(in) :: n  ! 平均をとる数
  real, intent(inout) :: y(size(x))  ! 平均化した後のデータ.
                      ! 実際は, y(1:n-1) までの配列にはゼロが入る.
  real, intent(in), optional :: error  ! 欠損値
  integer, intent(in), optional :: offset  ! 移動平均を開始する要素番号.
                      ! default = n
  integer :: nx, i, ioff
  real :: tmp, undef
  real :: bar(size(x))

  nx=size(x)
  y=0.0

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_1d( nx, y ),  &
  &                                     "Move_anom" )
  end if

  if(nx<n.or.n<2)then
     write(*,*) "### ERROR ### (Move_anom)"
     write(*,*) "x(nx) : nx must be more than n or n must be more than 2."
     write(*,*) "nx is ", nx, ", n is ", n, "."
     write(*,*) "STOP"
     stop
  end if

  if(present(offset))then
     if(offset>0)then
        ioff=offset
     else
        write(*,*) "### ERROR ### (Move_anom)"
        write(*,*) "offset must be more than 1."
        write(*,*) "STOP"
        stop
     end if
  else
     ioff=n
  end if

  if(present(error))then
     undef=error
  else
     undef=0.0
  end if

  call Move_ave( x, n, bar, error=undef, offset=ioff )

  if(ioff>2)then
     y(1:ioff-1)=0.0
     y(nx-n+ioff+1:nx)=0.0
  end if

  if(present(error))then
     y(ioff:nx-n+ioff)=error
     do i=ioff,nx-n+ioff
        if(x(i)/=error)then
           y(i)=x(i)-bar(i)
        end if
     end do
  else
     do i=ioff,nx-n+ioff
        y(i)=x(i)-bar(i)
     end do
  end if

end subroutine Move_anom

!---------------------------------
!---------------------------------

subroutine spline_3_f( xi, xo, yi, yo, itype, t1, tn )
! xi 点で定義されたデータ yi を3 次スプライン関数で xo という座標点に
! yo という値で補間するルーチン.
! itype, t1, tn は両端で設定される追加条件であり, これらがまったく
! 指定されない場合, 自然スプラインで補間するように設定されている.
! [注意]
! xi, xo はともに同じ座標軸上に存在し, 単位は同じもので定義されていなければ
! ならない.
  implicit none
  real, intent(in) :: xi(:)  ! データの定義されている点
  real, intent(in) :: xo(:)  ! 補間を行う点
  real, intent(in) :: yi(size(xi))  ! データの定義されている点
  real, intent(inout) :: yo(size(xo))  ! データの定義されている点
  integer, intent(in), optional :: itype  ! 両端条件の種類.
                             ! 1 = 1 回微分で指定.
                             ! 2 = 2 回微分で指定. [default]
  real, intent(in), optional :: t1  ! xo(1) での微分値. [default = 0.0]
  real, intent(in), optional :: tn  ! xo(n) での微分値. [default = 0.0]

  integer :: i, j, k, ni, no, inter_type, ix
  real :: ti1, ti2
  real, dimension(size(xi)) :: ai, bi, ci, di, zi, hi
  real, dimension(size(xi),size(xi)) :: a

  ni=size(xi)
  no=size(xo)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_1d( ni, yi ),  &
  &                                     "spline_3" )
     call check_array_size_dmp_message( check_array_size_1d( no, yo ),  &
  &                                     "spline_3" )
  end if

  a=0.0
  ai=0.0
  bi=0.0
  ci=0.0
  di=0.0
  zi=0.0
  hi=0.0

  if(present(t1))then
     ti1=t1
  else
     ti1=0.0
  end if

  if(present(tn))then
     ti2=tn
  else
     ti2=0.0
  end if

  if(present(itype))then
     inter_type=itype
  else
     inter_type=2
  end if

!-- 内挿範囲のチェック
  if(xi(1)>xo(1))then
     write(*,*) "*** ERROR (spline_3) *** : xo(1) must be G.E. to xi(1)."
     write(*,*) "stop."
     stop
  end if
  if(xi(ni)<xo(no))then
     write(*,*) "*** ERROR (spline_3) *** : xo(N) must be L.E. to xi(N)."
     write(*,*) "stop."
     stop
  end if

!-- 係数の設定

  do i=2,ni
     hi(i)=xi(i)-xi(i-1)
  end do

  do i=2,ni-2
     a(i,i)=2.0*(hi(i+1)+hi(i))
     a(i,i-1)=hi(i)
     a(i,i+1)=hi(i+1)
     zi(i)=3.0*(yi(i+1)-yi(i))/hi(i+1)-3.0*(yi(i)-yi(i-1))/hi(i)
  end do

  if(inter_type==1)then
     a(1,1)=2.0  ! 定義しただけで使わない
     a(1,2)=1.0  ! 定義しただけで使わない
     a(ni-1,ni-2)=2.0*hi(ni-1)
     a(ni-1,ni-1)=4.0*hi(ni-1)+3.0*hi(ni)
     zi(1)=3.0*(yi(2)-yi(1))/(hi(2)**2)-3.0*ti1/hi(2)
     zi(ni-1)=-3.0*ti2+9.0*(yi(ni)-yi(ni-1))/hi(ni)  &
  &           -6.0*(yi(ni-1)-yi(ni-2))/hi(ni-1)
     !-- bi(1) は独立に求める.
     zi(2)=zi(2)-0.5*zi(1)
     a(2,2)=(2.0*hi(3)/hi(2)+1.5)
     a(2,3)=hi(3)/hi(2)
  else if(inter_type==2)then
     a(1,1)=1.0  ! 定義しただけで使わない
     a(ni-1,ni-2)=hi(ni-1)
     a(ni-1,ni-1)=2.0*(hi(ni-1)+hi(ni))
     zi(1)=0.5*ti1
     zi(ni-1)=-0.5*ti2*hi(ni)+3.0*(yi(ni)-yi(ni-1))/hi(ni)  &
  &           -3.0*(yi(ni-1)-yi(ni-2))/hi(ni-1)
     !-- bi(1) は独立に求める.
     bi(1)=zi(1)
     zi(2)=zi(2)-hi(2)*bi(1)
  end if

!-- 係数 b を計算.

  call tri_gauss( a(2:ni-1,2:ni-1), zi(2:ni-1), bi(2:ni-1) )

!-- b を元に, a, c, d を算出.

  if(inter_type==1)then
     ci(ni-1)=-0.5*hi(ni)*bi(ni-1)+1.5*(yi(ni)-yi(ni-1))/hi(ni)-0.5*ti2
     ai(ni-1)=(ti2-ci(ni-1))/(3.0*hi(ni)*hi(ni))-2.0*bi(ni-1)/(3.0*hi(ni))
     !-- bi(1) は独立に求める.
     bi(1)=0.5*(zi(1)-bi(2))
  else if(inter_type==2)then
     ci(ni-1)=-2.0*hi(ni)*bi(ni-1)/3.0+(yi(ni)-yi(ni-1))/hi(ni)-ti2*hi(ni)/6.0
     ai(ni-1)=(0.5*ti2-bi(ni-1))/(3.0*hi(ni))
  end if

  do i=2,ni-1
     ai(i-1)=(bi(i)-bi(i-1))/(3.0*hi(i))
     ci(i-1)=(yi(i)-yi(i-1))/hi(i)-hi(i)*(bi(i)+2.0*bi(i-1))/3.0
     di(i-1)=yi(i-1)
  end do

  di(ni-1)=yi(ni-1)

!-- 最終的なスプライン関数から xo での補間値を計算.

  do i=1,no
     call interpo_search_1df( xi, xo(i), ix )
     if(ix==ni)then
        ix=ni-1
     end if
     yo(i)=ai(ix)*((xo(i)-xi(ix))**3)  &
  &       +bi(ix)*((xo(i)-xi(ix))**2)  &
  &       +ci(ix)*(xo(i)-xi(ix))+di(ix)
  end do

end subroutine spline_3_f

!---------------------------------
!---------------------------------

subroutine spline_3_d( xi, xo, yi, yo, itype, t1, tn )
! xi 点で定義されたデータ yi を3 次スプライン関数で xo という座標点に
! yo という値で補間するルーチン.
! itype, t1, tn は両端で設定される追加条件であり, これらがまったく
! 指定されない場合, 自然スプラインで補間するように設定されている.
! [注意]
! xi, xo はともに同じ座標軸上に存在し, 単位は同じもので定義されていなければ
! ならない.
  implicit none
  double precision, intent(in) :: xi(:)  ! データの定義されている点
  double precision, intent(in) :: xo(:)  ! 補間を行う点
  double precision, intent(in) :: yi(size(xi))  ! データの定義されている点
  double precision, intent(inout) :: yo(size(xo))  ! データの定義されている点
  integer, intent(in), optional :: itype  ! 両端条件の種類.
                             ! 1 = 1 回微分で指定.
                             ! 2 = 2 回微分で指定. [default]
  double precision, intent(in), optional :: t1  ! xo(1) での微分値. [default = 0.0]
  double precision, intent(in), optional :: tn  ! xo(n) での微分値. [default = 0.0]

  integer :: i, j, k, ni, no, inter_type, ix
  double precision :: ti1, ti2
  double precision, dimension(size(xi)) :: ai, bi, ci, di, zi, hi
  double precision, dimension(size(xi),size(xi)) :: a

  ni=size(xi)
  no=size(xo)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_1d( ni, yi ),  &
  &                                     "spline_3" )
     call check_array_size_dmp_message( check_array_size_1d( no, yo ),  &
  &                                     "spline_3" )
  end if

  a=0.0
  ai=0.0
  bi=0.0
  ci=0.0
  di=0.0
  zi=0.0
  hi=0.0

  if(present(t1))then
     ti1=t1
  else
     ti1=0.0
  end if

  if(present(tn))then
     ti2=tn
  else
     ti2=0.0
  end if

  if(present(itype))then
     inter_type=itype
  else
     inter_type=2
  end if

!-- 内挿範囲のチェック
  if(xi(1)>xo(1))then
     write(*,*) "*** ERROR (spline_3) *** : xo(1) must be G.E. to xi(1)."
     write(*,*) "stop."
     stop
  end if
  if(xi(ni)<xo(no))then
     write(*,*) "*** ERROR (spline_3) *** : xo(N) must be L.E. to xi(N)."
     write(*,*) "stop."
     stop
  end if

!-- 係数の設定

  do i=2,ni
     hi(i)=xi(i)-xi(i-1)
  end do

  do i=2,ni-2
     a(i,i)=2.0d0*(hi(i+1)+hi(i))
     a(i,i-1)=hi(i)
     a(i,i+1)=hi(i+1)
     zi(i)=3.0d0*(yi(i+1)-yi(i))/hi(i+1)-3.0d0*(yi(i)-yi(i-1))/hi(i)
  end do

  if(inter_type==1)then
     a(1,1)=2.0d0  ! 定義しただけで使わない
     a(1,2)=1.0d0  ! 定義しただけで使わない
     a(ni-1,ni-2)=2.0d0*hi(ni-1)
     a(ni-1,ni-1)=4.0d0*hi(ni-1)+3.0d0*hi(ni)
     zi(1)=3.0d0*(yi(2)-yi(1))/(hi(2)**2)-3.0d0*ti1/hi(2)
     zi(ni-1)=-3.0d0*ti2+9.0d0*(yi(ni)-yi(ni-1))/hi(ni)  &
  &           -6.0d0*(yi(ni-1)-yi(ni-2))/hi(ni-1)
     !-- bi(1) は独立に求める.
     zi(2)=zi(2)-0.5d0*zi(1)
     a(2,2)=(2.0d0*hi(3)/hi(2)+1.5d0)
     a(2,3)=hi(3)/hi(2)
  else if(inter_type==2)then
     a(1,1)=1.0d0  ! 定義しただけで使わない
     a(ni-1,ni-2)=hi(ni-1)
     a(ni-1,ni-1)=2.0d0*(hi(ni-1)+hi(ni))
     zi(1)=0.5d0*ti1
     zi(ni-1)=-0.5d0*ti2*hi(ni)+3.0d0*(yi(ni)-yi(ni-1))/hi(ni)  &
  &           -3.0d0*(yi(ni-1)-yi(ni-2))/hi(ni-1)
     !-- bi(1) は独立に求める.
     bi(1)=zi(1)
     zi(2)=zi(2)-hi(2)*bi(1)
  end if

!-- 係数 b を計算.

  call tri_gauss( a(2:ni-1,2:ni-1), zi(2:ni-1), bi(2:ni-1) )

!-- b を元に, a, c, d を算出.

  if(inter_type==1)then
     ci(ni-1)=-0.5d0*hi(ni)*bi(ni-1)+1.5d0*(yi(ni)-yi(ni-1))/hi(ni)-0.5d0*ti2
     ai(ni-1)=(ti2-ci(ni-1))/(3.0d0*hi(ni)*hi(ni))-2.0d0*bi(ni-1)/(3.0d0*hi(ni))
     !-- bi(1) は独立に求める.
     bi(1)=0.5d0*(zi(1)-bi(2))
  else if(inter_type==2)then
     ci(ni-1)=-2.0d0*hi(ni)*bi(ni-1)/3.0d0+(yi(ni)-yi(ni-1))/hi(ni)-ti2*hi(ni)/6.0d0
     ai(ni-1)=(0.5d0*ti2-bi(ni-1))/(3.0d0*hi(ni))
  end if

  do i=2,ni-1
     ai(i-1)=(bi(i)-bi(i-1))/(3.0d0*hi(i))
     ci(i-1)=(yi(i)-yi(i-1))/hi(i)-hi(i)*(bi(i)+2.0d0*bi(i-1))/3.0d0
     di(i-1)=yi(i-1)
  end do

  di(ni-1)=yi(ni-1)

!-- 最終的なスプライン関数から xo での補間値を計算.

  do i=1,no
     call interpo_search_1dd( xi, xo(i), ix )
     if(ix==ni)then
        ix=ni-1
     end if
     yo(i)=ai(ix)*((xo(i)-xi(ix))**3)  &
  &       +bi(ix)*((xo(i)-xi(ix))**2)  &
  &       +ci(ix)*(xo(i)-xi(ix))+di(ix)
  end do

end subroutine spline_3_d

!---------------------------------
!---------------------------------

subroutine Bubble_Sort_i( a, b, sig )
! バブルソートを用いて数値データを sig の方向にソートする.
  implicit none
  integer, intent(in) :: a(:)  ! ソートする配列
  integer, intent(inout) :: b(size(a))  ! ソートした結果を格納する配列
  character(1), intent(in) :: sig  ! ソートの順番
                                   ! 'i' = 要素番号の若いものに小さい値が入る
                                   ! 'r' = 要素番号の若いものに大きい値が入る
  integer :: i, j, n
  integer :: tmp

  n=size(a)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_1d( n, b ),  &
  &                                     "Bubble_Sort" )
  end if

  if(sig/='i'.and.sig/='r')then
     write(*,*) "### ERROR ###"
     write(*,*) "sig flag is 'r' .or. 'i', STOP."
     stop
  end if

  do i=1,n
     b(i)=a(i)
  end do

  if(sig=='i')then  ! 昇べきソート
     do i=1,n
        do j=1,n-1
           if(b(j)>b(j+1))then
              tmp=b(j+1)
              b(j+1)=b(j)
              b(j)=tmp
           end if
        end do
     end do
  else
     do i=1,n
        do j=1,n-1
           if(b(j)<b(j+1))then
              tmp=b(j+1)
              b(j+1)=b(j)
              b(j)=tmp
           end if
        end do
     end do
  end if

end subroutine Bubble_Sort_i

!---------------------------------
!---------------------------------

subroutine Bubble_Sort_f( a, b, sig )
! バブルソートを用いて数値データを sig の方向にソートする.
  implicit none
  real, intent(in) :: a(:)  ! ソートする配列
  real, intent(inout) :: b(size(a))  ! ソートした結果を格納する配列
  character(1), intent(in) :: sig  ! ソートの順番
                                   ! 'i' = 要素番号の若いものに小さい値が入る
                                   ! 'r' = 要素番号の若いものに大きい値が入る
  integer :: i, j, n
  real :: tmp

  n=size(a)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_1d( n, b ),  &
  &                                     "Bubble_Sort" )
  end if

  if(sig/='i'.and.sig/='r')then
     write(*,*) "### ERROR ###"
     write(*,*) "sig flag is 'r' .or. 'i', STOP."
     stop
  end if

  do i=1,n
     b(i)=a(i)
  end do

  if(sig=='i')then  ! 昇べきソート
     do i=1,n
        do j=1,n-1
           if(b(j)>b(j+1))then
              tmp=b(j+1)
              b(j+1)=b(j)
              b(j)=tmp
           end if
        end do
     end do
  else
     do i=1,n
        do j=1,n-1
           if(b(j)<b(j+1))then
              tmp=b(j+1)
              b(j+1)=b(j)
              b(j)=tmp
           end if
        end do
     end do
  end if

end subroutine Bubble_Sort_f

!---------------------------------
!---------------------------------

subroutine Bubble_Sort_d( a, b, sig )
! バブルソートを用いて数値データを sig の方向にソートする.
  implicit none
  double precision, intent(in) :: a(:)  ! ソートする配列
  double precision, intent(inout) :: b(size(a))  ! ソートした結果を格納する配列
  character(1), intent(in) :: sig  ! ソートの順番
                                   ! 'i' = 要素番号の若いものに小さい値が入る
                                   ! 'r' = 要素番号の若いものに大きい値が入る
  integer :: i, j, n
  double precision :: tmp

  n=size(a)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_1d( n, b ),  &
  &                                     "Bubble_Sort" )
  end if

  if(sig/='i'.and.sig/='r')then
     write(*,*) "### ERROR ###"
     write(*,*) "sig flag is 'r' .or. 'i', STOP."
     stop
  end if

  do i=1,n
     b(i)=a(i)
  end do

  if(sig=='i')then  ! 昇べきソート
     do i=1,n
        do j=1,n-1
           if(b(j)>b(j+1))then
              tmp=b(j+1)
              b(j+1)=b(j)
              b(j)=tmp
           end if
        end do
     end do
  else
     do i=1,n
        do j=1,n-1
           if(b(j)<b(j+1))then
              tmp=b(j+1)
              b(j+1)=b(j)
              b(j)=tmp
           end if
        end do
     end do
  end if

end subroutine Bubble_Sort_d

!---------------------------------
!---------------------------------

subroutine Quick_Sort_i( a, b, sig )
! Quick ソートを用いて数値データを sig の方向にソートする.
  implicit none
  integer, intent(in) :: a(:)  ! ソートする配列
  integer, intent(inout) :: b(size(a))  ! ソートした結果を格納する配列
  character(1), intent(in) :: sig  ! ソートの順番
                                   ! 'i' = 要素番号の若いものに小さい値が入る
                                   ! 'r' = 要素番号の若いものに大きい値が入る
  integer :: i, j, k, n, itmp, nt, isplt
  integer :: vtmp, refv
  integer :: c(size(a))  ! ソートした結果を格納する配列
  logical :: diflag
  logical :: splitflag(size(a))

  nt=size(a)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_1d( n, b ),  &
  &                                     "Bubble_Sort" )
  end if

  if(sig/='i'.and.sig/='r')then
     write(*,*) "### ERROR ###"
     write(*,*) "sig flag is 'r' .or. 'i', STOP."
     stop
  end if

  c=a
  n=nt
  isplt=nt
  splitflag=.false.
  splitflag(1)=.true.

  do k=1,nt

     if(k>1.and.splitflag(k).eqv..true.)then
        do j=k+1,nt
           if(splitflag(j).eqv..true.)then
              isplt=j
              n=isplt-1
              exit
           else if(j==nt)then
              isplt=nt+1
              n=nt
              exit
           end if
        end do
     end if

     ! パーティションフラグがあるかチェック
     do while(splitflag(k).eqv..true.)

        !-- 1 パーティションでの数値比較 (同じ数値のみでまとめられているか)
        diflag=.false.

        if(isplt-1>k)then
           do i=k+1,isplt-1
              if(c(i-1)/=c(i))then
                 diflag=.true.
                 exit
              end if
           end do
        end if

        if(diflag.eqv..true.)then

           itmp=k
           refv=c(itmp)

           !-- 1 パーティションでのソートと 2 分割化
           !-- 最初の前方探索は, refv = パーティション先頭としているので不要.
           !-- k = 前方からの探索 (itmp), n = 後方からの探索 (isplt)
           !-- [パーティションの位置]
           !-- 1. 前方探索が見つからず, 後方探索と出会う場合, 
           !--    出会った地点の 1 つ後ろがパーティション位置.
           !-- 2. 後方探索が見つからず, 前方探索と出会う場合,
           !--    出会った地点の 1 つ前がパーティション位置.
           do j=n,k,-1
              if(j>itmp)then
                 if(refv>c(j))then  ! 後方探索, refv 未満を検出
                    vtmp=c(j)
                    c(j)=c(itmp)
                    c(itmp)=vtmp  ! 該当すれば, c(itmp) と入れ替え

                    if(itmp+1==j)then  ! 入れ替えが隣接で起こる場合, 
                       isplt=j
                       n=j-1
                       splitflag(j)=.true.
                       exit
                    else
                       do i=itmp+1,n  ! 次段階の前方探索, refv 以上を検出
                          if(i<j)then
                             if(refv<=c(i))then  ! 検知すれば, その番号を取得.
                                itmp=i
                                exit
                             end if
                          else  ! 検知しなければ, itmp は j の直前に (go to 20)
                             itmp=j
                             exit
                          end if
                       end do
                    end if
                 end if
              else if(j==itmp)then  ! 後方探索が前方探索と出会う
                 if(itmp==k)then  ! 出会った場所がパーティション先頭の場合,
                    isplt=j+1  ! 分割された後方パーティションの先頭番号
                    n=j
                    splitflag(j+1)=.true.
                    exit
                 else  ! 出会った場所がパーティション内の場合, 
                    isplt=j  ! 分割された後方パーティションの先頭番号
                    n=j-1
                    splitflag(j)=.true.
                    exit
                 end if
              else if(j<itmp)then  ! 前方探索が後方探索と出会う (20 から)
                 isplt=j+1  ! 分割された後方パーティションの先頭番号
                 n=j
                 splitflag(j+1)=.true.
                 exit
              end if
           end do

        else

           splitflag(k)=.false.

        end if

     end do

  end do

  b=c

  if(sig=='r')then  ! 昇べきソート
     do i=1,n
        b(i)=c(n-i+1)
     end do
  end if

end subroutine Quick_Sort_i

!---------------------------------
!---------------------------------

subroutine Quick_Sort_f( a, b, sig )
! Quick ソートを用いて数値データを sig の方向にソートする.
  implicit none
  real, intent(in) :: a(:)  ! ソートする配列
  real, intent(inout) :: b(size(a))  ! ソートした結果を格納する配列
  character(1), intent(in) :: sig  ! ソートの順番
                                   ! 'i' = 要素番号の若いものに小さい値が入る
                                   ! 'r' = 要素番号の若いものに大きい値が入る
  integer :: i, j, k, n, itmp, nt, isplt
  real :: vtmp, refv
  real :: c(size(a))  ! ソートした結果を格納する配列
  logical :: diflag
  logical :: splitflag(size(a))

  nt=size(a)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_1d( n, b ),  &
  &                                     "Bubble_Sort" )
  end if

  if(sig/='i'.and.sig/='r')then
     write(*,*) "### ERROR ###"
     write(*,*) "sig flag is 'r' .or. 'i', STOP."
     stop
  end if

  c=a
  n=nt
  isplt=nt
  splitflag=.false.
  splitflag(1)=.true.

  do k=1,nt

     if(k>1.and.splitflag(k).eqv..true.)then
        do j=k+1,nt
           if(splitflag(j).eqv..true.)then
              isplt=j
              n=isplt-1
              exit
           else if(j==nt)then
              isplt=nt+1
              n=nt
              exit
           end if
        end do
     end if

     ! パーティションフラグがあるかチェック
     do while(splitflag(k).eqv..true.)

        !-- 1 パーティションでの数値比較 (同じ数値のみでまとめられているか)
        diflag=.false.

        if(isplt-1>k)then
           do i=k+1,isplt-1
              if(c(i-1)/=c(i))then
                 diflag=.true.
                 exit
              end if
           end do
        end if

        if(diflag.eqv..true.)then

           itmp=k
           refv=c(itmp)

           !-- 1 パーティションでのソートと 2 分割化
           !-- 最初の前方探索は, refv = パーティション先頭としているので不要.
           !-- k = 前方からの探索 (itmp), n = 後方からの探索 (isplt)
           !-- [パーティションの位置]
           !-- 1. 前方探索が見つからず, 後方探索と出会う場合, 
           !--    出会った地点の 1 つ後ろがパーティション位置.
           !-- 2. 後方探索が見つからず, 前方探索と出会う場合,
           !--    出会った地点の 1 つ前がパーティション位置.
           do j=n,k,-1
              if(j>itmp)then
                 if(refv>c(j))then  ! 後方探索, refv 未満を検出
                    vtmp=c(j)
                    c(j)=c(itmp)
                    c(itmp)=vtmp  ! 該当すれば, c(itmp) と入れ替え

                    if(itmp+1==j)then  ! 入れ替えが隣接で起こる場合, 
                       isplt=j
                       n=j-1
                       splitflag(j)=.true.
                       exit
                    else
                       do i=itmp+1,n  ! 次段階の前方探索, refv 以上を検出
                          if(i<j)then
                             if(refv<=c(i))then  ! 検知すれば, その番号を取得.
                                itmp=i
                                exit
                             end if
                          else  ! 検知しなければ, itmp は j の直前に (go to 20)
                             itmp=j
                             exit
                          end if
                       end do
                    end if
                 end if
              else if(j==itmp)then  ! 後方探索が前方探索と出会う
                 if(itmp==k)then  ! 出会った場所がパーティション先頭の場合,
                    isplt=j+1  ! 分割された後方パーティションの先頭番号
                    n=j
                    splitflag(j+1)=.true.
                    exit
                 else  ! 出会った場所がパーティション内の場合, 
                    isplt=j  ! 分割された後方パーティションの先頭番号
                    n=j-1
                    splitflag(j)=.true.
                    exit
                 end if
              else if(j<itmp)then  ! 前方探索が後方探索と出会う (20 から)
                 isplt=j+1  ! 分割された後方パーティションの先頭番号
                 n=j
                 splitflag(j+1)=.true.
                 exit
              end if
           end do

        else

           splitflag(k)=.false.

        end if

     end do

  end do

  b=c

  if(sig=='r')then  ! 昇べきソート
     do i=1,n
        b(i)=c(n-i+1)
     end do
  end if

end subroutine Quick_Sort_f

!---------------------------------
!---------------------------------

subroutine Quick_Sort_d( a, b, sig )
! Quick ソートを用いて数値データを sig の方向にソートする.
  implicit none
  double precision, intent(in) :: a(:)  ! ソートする配列
  double precision, intent(inout) :: b(size(a))  ! ソートした結果を格納する配列
  character(1), intent(in) :: sig  ! ソートの順番
                                   ! 'i' = 要素番号の若いものに小さい値が入る
                                   ! 'r' = 要素番号の若いものに大きい値が入る
  integer :: i, j, k, n, itmp, nt, isplt
  double precision :: vtmp, refv
  double precision :: c(size(a))  ! ソートした結果を格納する配列
  logical :: diflag
  logical :: splitflag(size(a))

  nt=size(a)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_1d( n, b ),  &
  &                                     "Bubble_Sort" )
  end if

  if(sig/='i'.and.sig/='r')then
     write(*,*) "### ERROR ###"
     write(*,*) "sig flag is 'r' .or. 'i', STOP."
     stop
  end if

  c=a
  n=nt
  isplt=nt
  splitflag=.false.
  splitflag(1)=.true.

  do k=1,nt

     if(k>1.and.splitflag(k).eqv..true.)then
        do j=k+1,nt
           if(splitflag(j).eqv..true.)then
              isplt=j
              n=isplt-1
              exit
           else if(j==nt)then
              isplt=nt+1
              n=nt
              exit
           end if
        end do
     end if

     ! パーティションフラグがあるかチェック
     do while(splitflag(k).eqv..true.)

        !-- 1 パーティションでの数値比較 (同じ数値のみでまとめられているか)
        diflag=.false.

        if(isplt-1>k)then
           do i=k+1,isplt-1
              if(c(i-1)/=c(i))then
                 diflag=.true.
                 exit
              end if
           end do
        end if

        if(diflag.eqv..true.)then

           itmp=k
           refv=c(itmp)

           !-- 1 パーティションでのソートと 2 分割化
           !-- 最初の前方探索は, refv = パーティション先頭としているので不要.
           !-- k = 前方からの探索 (itmp), n = 後方からの探索 (isplt)
           !-- [パーティションの位置]
           !-- 1. 前方探索が見つからず, 後方探索と出会う場合, 
           !--    出会った地点の 1 つ後ろがパーティション位置.
           !-- 2. 後方探索が見つからず, 前方探索と出会う場合,
           !--    出会った地点の 1 つ前がパーティション位置.
           do j=n,k,-1
              if(j>itmp)then
                 if(refv>c(j))then  ! 後方探索, refv 未満を検出
                    vtmp=c(j)
                    c(j)=c(itmp)
                    c(itmp)=vtmp  ! 該当すれば, c(itmp) と入れ替え

                    if(itmp+1==j)then  ! 入れ替えが隣接で起こる場合, 
                       isplt=j
                       n=j-1
                       splitflag(j)=.true.
                       exit
                    else
                       do i=itmp+1,n  ! 次段階の前方探索, refv 以上を検出
                          if(i<j)then
                             if(refv<=c(i))then  ! 検知すれば, その番号を取得.
                                itmp=i
                                exit
                             end if
                          else  ! 検知しなければ, itmp は j の直前に (go to 20)
                             itmp=j
                             exit
                          end if
                       end do
                    end if
                 end if
              else if(j==itmp)then  ! 後方探索が前方探索と出会う
                 if(itmp==k)then  ! 出会った場所がパーティション先頭の場合,
                    isplt=j+1  ! 分割された後方パーティションの先頭番号
                    n=j
                    splitflag(j+1)=.true.
                    exit
                 else  ! 出会った場所がパーティション内の場合, 
                    isplt=j  ! 分割された後方パーティションの先頭番号
                    n=j-1
                    splitflag(j)=.true.
                    exit
                 end if
              else if(j<itmp)then  ! 前方探索が後方探索と出会う (20 から)
                 isplt=j+1  ! 分割された後方パーティションの先頭番号
                 n=j
                 splitflag(j+1)=.true.
                 exit
              end if
           end do

        else

           splitflag(k)=.false.

        end if

     end do

  end do

  b=c

  if(sig=='r')then  ! 昇べきソート
     do i=1,n
        b(i)=c(n-i+1)
     end do
  end if

end subroutine Quick_Sort_d

!---------------------------------
!---------------------------------

subroutine detrend_1df( tax, ival, oval, undef )
! 与えられたデータ ival から平均値とトレンドを除去する.
! 除去する次元要素は ival の最も後ろの次元要素.
! 除去の方法はトレンドの次元要素方向に並べて,
! 1 次の線形回帰を計算する.
! 得られた回帰曲線分, 元のデータから値を差し引く.
! このとき, 元データのトレンド方向の次元については, 回帰曲線の
! 傾きに従って値が変化していく. 
! トレンド除去を行うデータの座標は tax によって指定する.
!
! 未定義値がある場合はその格子点について一切の操作を行わない.

  implicit none

  real, intent(in) :: tax(:)  !  トレンド座標の座標値
  real, intent(in) :: ival(size(tax))  ! 元データ
  real, intent(inout) :: oval(size(tax))  ! トレンド除去されたデータ
  real, intent(in), optional :: undef

  integer :: ii, n1
  real :: yval, dslope, dinterp

  n1=size(ival)

  if(present(undef))then

     call LSM_1df( tax, ival, dslope, dinterp, undef=undef )

     do ii=1,n1
        yval=dinterp+dslope*tax(ii)
        if(ival(ii)/=undef)then
           oval(ii)=ival(ii)-yval
        end if
     end do

  else

     call LSM_1df( tax, ival, dslope, dinterp )

     do ii=1,n1
        yval=dinterp+dslope*tax(ii)
        oval(ii)=ival(ii)-yval
     end do

  end if

end subroutine detrend_1df

!---------------------------------
!---------------------------------

subroutine detrend_1dd( tax, ival, oval, undef )
! 与えられたデータ ival から平均値とトレンドを除去する.
! 除去する次元要素は ival の最も後ろの次元要素.
! 除去の方法はトレンドの次元要素方向に並べて,
! 1 次の線形回帰を計算する.
! 得られた回帰曲線分, 元のデータから値を差し引く.
! このとき, 元データのトレンド方向の次元については, 回帰曲線の
! 傾きに従って値が変化していく. 
! トレンド除去を行うデータの座標は tax によって指定する.
!
! 未定義値がある場合はその格子点について一切の操作を行わない.

  implicit none

  double precision, intent(in) :: tax(:)  !  トレンド座標の座標値
  double precision, intent(in) :: ival(size(tax))  ! 元データ
  double precision, intent(inout) :: oval(size(tax))  ! トレンド除去されたデータ
  double precision, intent(in), optional :: undef

  integer :: ii, n1
  double precision :: yval, dslope, dinterp

  n1=size(ival)

  if(present(undef))then

     call LSM_1dd( tax, ival, dslope, dinterp, undef=undef )

     do ii=1,n1
        yval=dinterp+dslope*tax(ii)
        if(ival(ii)/=undef)then
           oval(ii)=ival(ii)-yval
        end if
     end do

  else

     call LSM_1dd( tax, ival, dslope, dinterp )

     do ii=1,n1
        yval=dinterp+dslope*tax(ii)
        oval(ii)=ival(ii)-yval
     end do

  end if

end subroutine detrend_1dd

!---------------------------------
!---------------------------------

subroutine detrend_2df( tax, ival, oval, undef )
! 与えられたデータ ival から平均値とトレンドを除去する.
! 除去する次元要素は ival の最も後ろの次元要素.
! 除去の方法はトレンドを求める次元要素以外の次元について平均値を
! 計算する. その平均値をトレンドの次元要素方向に並べて,
! 1 次の線形回帰を計算する.
! 得られた回帰曲線分, 元のデータから値を差し引く.
! このとき, 元データのトレンド方向の次元については, 回帰曲線の
! 傾きに従って値が変化していく. それ以外の方向については,
! (回帰曲線の) 平均値が一様に引かれる.
! トレンド除去を行うデータの座標は, tax によって指定する.
!
! 未定義値がある場合はその格子点について一切の操作を行わない.

  implicit none

  real, intent(in), optional :: tax(:)
                         !  トレンド座標が非一様の場合の座標値
  real, intent(in) :: ival(:,:)  ! 元データ
  real, intent(inout) :: oval(size(ival,1),size(ival,2))  ! トレンド除去されたデータ
  real, intent(in), optional :: undef

  integer :: ii, jj, n1, n2
  real :: yval, dslope, dinterp
  real :: dmean(size(ival,2)), lsm_mean(size(ival,2))

  n1=size(ival,1)
  n2=size(ival,2)

  if(present(undef))then
     do ii=1,n2
        call Mean_1df( ival(:,ii), dmean(ii), error=undef )
     end do

     call LSM_1df( tax, dmean, dslope, dinterp, undef=undef )

     do jj=1,n2
        yval=dinterp+dslope*tax(jj)
        do ii=1,n1
           if(ival(ii,jj)/=undef)then
              oval(ii,jj)=ival(ii,jj)-yval
           end if
        end do
     end do

  else

     do ii=1,n2
        call Mean_1df( ival(:,ii), dmean(ii) )
     end do

     call LSM_1df( tax, dmean, dslope, dinterp )

     do jj=1,n2
        yval=dinterp+dslope*tax(jj)
        do ii=1,n1
           oval(ii,jj)=ival(ii,jj)-yval
        end do
     end do

  end if

end subroutine detrend_2df

!---------------------------------
!---------------------------------

subroutine detrend_2dd( tax, ival, oval, undef )
! 与えられたデータ ival から平均値とトレンドを除去する.
! 除去する次元要素は ival の最も後ろの次元要素.
! 除去の方法はトレンドを求める次元要素以外の次元について平均値を
! 計算する. その平均値をトレンドの次元要素方向に並べて,
! 1 次の線形回帰を計算する.
! 得られた回帰曲線分, 元のデータから値を差し引く.
! このとき, 元データのトレンド方向の次元については, 回帰曲線の
! 傾きに従って値が変化していく. それ以外の方向については,
! (回帰曲線の) 平均値が一様に引かれる.
! トレンド除去を行うデータの座標は, tax によって指定する.
!
! 未定義値がある場合はその格子点について一切の操作を行わない.

  implicit none

  double precision, intent(in), optional :: tax(:)
                         !  トレンド座標が非一様の場合の座標値
  double precision, intent(in) :: ival(:,:)  ! 元データ
  double precision, intent(inout) :: oval(size(ival,1),size(ival,2))  ! トレンド除去されたデータ
  double precision, intent(in), optional :: undef

  integer :: ii, jj, n1, n2
  double precision :: yval, dslope, dinterp
  double precision :: dmean(size(ival,2)), lsm_mean(size(ival,2))

  n1=size(ival,1)
  n2=size(ival,2)

  if(present(undef))then
     do ii=1,n2
        call Mean_1dd( ival(:,ii), dmean(ii), error=undef )
     end do

     call LSM_1dd( tax, dmean, dslope, dinterp, undef=undef )

     do jj=1,n2
        yval=dinterp+dslope*tax(jj)
        do ii=1,n1
           if(ival(ii,jj)/=undef)then
              oval(ii,jj)=ival(ii,jj)-yval
           end if
        end do
     end do

  else

     do ii=1,n2
        call Mean_1dd( ival(:,ii), dmean(ii) )
     end do

     call LSM_1dd( tax, dmean, dslope, dinterp )

     do jj=1,n2
        yval=dinterp+dslope*tax(jj)
        do ii=1,n1
           oval(ii,jj)=ival(ii,jj)-yval
        end do
     end do

  end if

end subroutine detrend_2dd

!---------------------------------
!---------------------------------
!---- 以下は, private ルーチン----
!---------------------------------
!---------------------------------

subroutine summf( z, add, undeff, nc )
! undef を除いた総和演算を行う private ルーチン. 外部参照は不可.
  implicit none
  real, intent(in) :: z(:)
  real, intent(inout) :: add
  real, intent(in), optional :: undeff
  integer, intent(inout), optional :: nc
  integer :: i
  integer :: nx, nctmp

  nx=size(z)

  add=0.0
  if(present(undeff))then
     nctmp=0
     do i=1,nx
        if(undeff/=z(i))then
           add=add+z(i)
           nctmp=nctmp+1
        end if
     end do
  else
     do i=1,nx
        add=add+z(i)
     end do
     nctmp=nx
  end if

  if(present(nc))then
     nc=nctmp
  end if

end subroutine summf

subroutine summd( z, add, undeff, nc )
! undef を除いた総和演算を行う private ルーチン. 外部参照は不可.
  implicit none
  double precision, intent(in) :: z(:)
  double precision, intent(inout) :: add
  double precision, intent(in), optional :: undeff
  integer, intent(inout), optional :: nc
  integer :: i
  integer :: nx, nctmp

  nx=size(z)

  add=0.0d0
  if(present(undeff))then
     nctmp=0
     do i=1,nx
        if(undeff/=z(i))then
           add=add+z(i)
           nctmp=nctmp+1
        end if
     end do
  else
     do i=1,nx
        add=add+z(i)
     end do
     nctmp=nx
  end if

  if(present(nc))then
     nc=nctmp
  end if

end subroutine summd

subroutine replace_undef_f( fix, fiy, fox, foy, undef )
! fix, fiy というデータ数の同じ 2 つのデータについて, 各データを比較し,
! 片方に未定義値が入っていて, もう片方が未定義ではない場合, 両方とも
! 未定義として undef を代入する.
  implicit none
  real, intent(in) :: fix(:), fiy(size(fix))
  real, intent(inout) :: fox(size(fix)), foy(size(fix))
  real, intent(in) :: undef
  integer :: i, nx

  nx=size(fix)
  fox=fix
  foy=fiy

  do i=1,nx
     if(fix(i)==undef.or.fiy(i)==undef)then
        fox(i)=undef
        foy(i)=undef
     end if
  end do

end subroutine replace_undef_f

subroutine replace_undef_d( fix, fiy, fox, foy, undef )
! fix, fiy というデータ数の同じ 2 つのデータについて, 各データを比較し,
! 片方に未定義値が入っていて, もう片方が未定義ではない場合, 両方とも
! 未定義として undef を代入する.
  implicit none
  double precision, intent(in) :: fix(:), fiy(size(fix))
  double precision, intent(inout) :: fox(size(fix)), foy(size(fix))
  double precision, intent(in) :: undef
  integer :: i, nx

  nx=size(fix)
  fox=fix
  foy=fiy

  do i=1,nx
     if(fix(i)==undef.or.fiy(i)==undef)then
        fox(i)=undef
        foy(i)=undef
     end if
  end do

end subroutine replace_undef_d

end module
